/* Copyright (C) 2005 Felipe Leme. All rights reserved.
 * 
 * This program and the accompanying materials are made available under
 * the terms of the Common Public License v1.0 which accompanies this distribution,
 * and is available at http://www.eclipse.org/legal/cpl-v10.html
 */
package com.vladium.emma.test;

import junit.framework.TestCase;

import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IWorkspace;
import org.eclipse.core.resources.IWorkspaceRoot;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;

/**
 * This is the base class that provides plumbing for setting up a project 
 * in the workspace.
 * 
 * @author Felipe Leme <emma@felipeal.net>
 *
 */
public abstract class AbstractWorkspaceTestCase extends TestCase {

  /** reference to the workspace */
  protected final IWorkspace fWorkspace;  
    
  /**
   * Default constructor.
   * @param testName name of the testcase being tested
   * @throws Exception never thrown here, but declared so sub-classes can throw
   */
  protected AbstractWorkspaceTestCase( String testName ) throws Exception {
    super( testName );
    this.fWorkspace = createWorkspace();
  }   
  
  // protected methods
  
  /**
   * Calls createProject().
   * @throws Exception propagated from createProject()
   */
  protected void setUp() throws Exception {
    createTestFixture();
  }

  /**
   * Calls dispose().
   * @throws Exception propagated from disposeProject()
   */
  protected void tearDown() throws CoreException {
    disposeTestFixture();
  }
  
  /**
   * Create and open the project fixture.
   */
  protected abstract void createTestFixture() throws CoreException;
  
  /**
   * Dispose a project from the workspace.
   */
  protected abstract void disposeTestFixture() throws CoreException;
  
  /**
   * Create and workspace where the project will reside.
   * @return the workspace;
   */
  private IWorkspace createWorkspace() {
    return ResourcesPlugin.getWorkspace();
  }

  /**
   * Create and open a project.
   * @return an open project.
   * @throws CoreException propagated
   */
  protected IProject createProject( String projectName ) throws CoreException {
    final IWorkspaceRoot root = this.fWorkspace.getRoot();
    final IProject project = root.getProject( projectName );
    project.create( null );
    project.open( null );
    return project;
  }  

}
