/* Copyright (C) 2005 Erik Beijnoff. All rights reserved.
 * 
 * This program and the accompanying materials are made available under
 * the terms of the Common Public License v1.0 which accompanies this distribution,
 * and is available at http://www.eclipse.org/legal/cpl-v10.html
 */
package com.vladium.emma.test.util;

import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URL;

import org.eclipse.core.resources.IFolder;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IProjectDescription;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.Path;
import org.eclipse.core.runtime.Platform;
import org.eclipse.jdt.core.IClasspathEntry;
import org.eclipse.jdt.core.IJavaElement;
import org.eclipse.jdt.core.IJavaProject;
import org.eclipse.jdt.core.IPackageFragment;
import org.eclipse.jdt.core.IPackageFragmentRoot;
import org.eclipse.jdt.core.IType;
import org.eclipse.jdt.core.JavaCore;
import org.eclipse.jdt.core.JavaModelException;
import org.eclipse.jdt.core.search.IJavaSearchConstants;
import org.eclipse.jdt.core.search.ITypeNameRequestor;
import org.eclipse.jdt.core.search.SearchEngine;
import org.eclipse.jdt.core.search.SearchPattern;
import org.eclipse.jdt.launching.JavaRuntime;

/**
 * Used as a convenience java project creator. Creates a java project which has 
 * an output directory and a source folder and adds it to the workspace
 * 
 * @author Erik Beijnoff <erik@beijnoff.com>
 * @since 2006-02-08
 */
public class TestProject{
	private IProject project;
	private IJavaProject javaProject;
	private IPackageFragmentRoot sourceFolder;

	/**
	 * Constructor
	 * 
	 * @param name the name of the project
	 * @param binFolderName the name of the binary output folder
	 * @param sourceFolderName the name of the source folder
	 */
	public TestProject(String name, String binFolderName, String sourceFolderName) throws CoreException{
		project= ResourcesPlugin.getWorkspace().getRoot().getProject(name);
		project.create(null);
		project.open(null);
		javaProject= JavaCore.create(project);

		//Set java nature
		final IProjectDescription description= project.getDescription();
		description.setNatureIds(new String[]{JavaCore.NATURE_ID});
		project.setDescription(description, null);

		javaProject.setRawClasspath(new IClasspathEntry[0], null);

		//Create bin folder
		final IFolder binFolder= project.getFolder(binFolderName);
		binFolder.create(false, true, null);
		final IPath outputLocation= binFolder.getFullPath();
		javaProject.setOutputLocation(outputLocation, null);
		addClassPathEntry(JavaRuntime.getDefaultJREContainerEntry());

		//Create source folder
		final IFolder folder= project.getFolder(sourceFolderName);
		folder.create(false, true, null);
		final IPackageFragmentRoot packageRoot= javaProject.getPackageFragmentRoot(folder);
		addClassPathEntry(JavaCore.newSourceEntry(packageRoot.getPath()));
		sourceFolder= packageRoot;
	}

	private void addClassPathEntry(IClasspathEntry newEntry) throws CoreException{
		final IClasspathEntry[] oldEntries= javaProject.getRawClasspath();
		final IClasspathEntry[] newEntries= new IClasspathEntry[oldEntries.length + 1];
		System.arraycopy(oldEntries, 0, newEntries, 0, oldEntries.length);
		newEntries[oldEntries.length]= newEntry;
		javaProject.setRawClasspath(newEntries, null);
	}

	public IJavaProject getJavaProject(){
		return javaProject;
	}

	/**
	 * Create a package
	 * 
	 * @param name the name of the package
	 */
	public IPackageFragment createPackage(String name) throws CoreException{
		return sourceFolder.createPackageFragment(name, false, null);
	}

	/**
	 * Createa a source file and automatically appends the package name to the top
	 * of the content. Therefore package should not be defined in content.
	 * 
	 * @param pack the package to create the file in
	 * @param name the name of the file
	 * @param content the content of the source file
	 */
	public IType createCompilationUnit(IPackageFragment pack, String name, String content) throws JavaModelException{
		final StringBuffer sb= new StringBuffer();
		if(!pack.getElementName().equals("")){
			sb.append("package " + pack.getElementName() + ";\n\n");
		}
		sb.append(content);
		return pack.createCompilationUnit(name, sb.toString(), false, null).getTypes()[0];
	}

	/**
	 * Add a jar reference to the project
	 * 
	 * @param pluginName the name of the plugin to find the jar in for example "org.junit"
	 * @param jarName the name of the jar for example "junit.jar"
	 */
	public void addJar(String pluginName, String jarName) throws MalformedURLException, IOException, CoreException{
		final URL pluginURL= Platform.getBundle(pluginName).getEntry("/");
		final URL jarURL= new URL(pluginURL, jarName);
		final URL localJarURL= Platform.asLocalURL(jarURL);
		final Path result= new Path(localJarURL.getPath());

		addClassPathEntry(JavaCore.newLibraryEntry(result, null, null));
	}

	/**
	 * Use this to dispose the project once you are finished testing
	 */
	public void dispose() throws CoreException{
		//This makes the execution halt so that the previous setup gets torn down
		//before a new one gets created. If it didn't execution errors would occur.
		//Not quite sure exactly what it does, but I think it's the WAIT flag that
		//halts the execution
		new SearchEngine()
		.searchAllTypeNames(
			null,
			null,
			SearchPattern.R_EXACT_MATCH,
			IJavaSearchConstants.CLASS,
			SearchEngine.createJavaSearchScope(new IJavaElement[0]),
			new ITypeNameRequestor() {
				public void acceptClass(
					char[] packageName,
					char[] simpleTypeName,
					char[][] enclosingTypeNames,
					String path) {
					//Do nothing
				}
				public void acceptInterface(
					char[] packageName,
					char[] simpleTypeName,
					char[][] enclosingTypeNames,
					String path) {
					//Do nothing
				}
			}, 
			IJavaSearchConstants.WAIT_UNTIL_READY_TO_SEARCH, 
			null);
		
		project.delete(true, true, null);
	}
}