/*
 * Decompiled with CFR 0.152.
 */
package eu.cloudnetservice.common.io;

import io.vavr.CheckedConsumer;
import io.vavr.CheckedFunction1;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.file.CopyOption;
import java.nio.file.DirectoryStream;
import java.nio.file.FileSystem;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.PathMatcher;
import java.nio.file.StandardCopyOption;
import java.nio.file.attribute.FileAttribute;
import java.nio.file.spi.FileSystemProvider;
import java.util.Map;
import java.util.UUID;
import java.util.function.BiConsumer;
import lombok.NonNull;
import org.jetbrains.annotations.ApiStatus;
import org.jetbrains.annotations.Nullable;
import org.jetbrains.annotations.UnknownNullability;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@ApiStatus.Internal
public final class FileUtil {
    public static final Path TEMP_DIR = Path.of(System.getProperty("cloudnet.tempDir", "temp"), new String[0]);
    private static final Logger LOGGER = LoggerFactory.getLogger(FileUtil.class);
    private static final DirectoryStream.Filter<Path> ACCEPTING_FILTER = $ -> true;
    private static final FileSystemProvider JAR_FILE_SYSTEM_PROVIDER;
    private static final Map<String, String> ZIP_FILE_SYSTEM_PROPERTIES;

    private FileUtil() {
        throw new UnsupportedOperationException();
    }

    public static void openZipFile(@NonNull Path zip, @NonNull CheckedConsumer<FileSystem> consumer) {
        if (zip == null) {
            throw new NullPointerException("zip is marked non-null but is null");
        }
        if (consumer == null) {
            throw new NullPointerException("consumer is marked non-null but is null");
        }
        try (FileSystem fs = JAR_FILE_SYSTEM_PROVIDER.newFileSystem(zip, ZIP_FILE_SYSTEM_PROPERTIES);){
            consumer.accept((Object)fs);
        }
        catch (Throwable throwable) {
            LOGGER.error("Exception opening zip file system on {}", (Object)zip, (Object)throwable);
        }
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public static <T> @UnknownNullability T mapZipFile(@NonNull Path zip, @NonNull CheckedFunction1<FileSystem, T> mapper, @Nullable T def) {
        if (zip == null) {
            throw new NullPointerException("zip is marked non-null but is null");
        }
        if (mapper == null) {
            throw new NullPointerException("mapper is marked non-null but is null");
        }
        try (FileSystem fs = JAR_FILE_SYSTEM_PROVIDER.newFileSystem(zip, ZIP_FILE_SYSTEM_PROPERTIES);){
            Object object = mapper.apply((Object)fs);
            return (T)object;
        }
        catch (Throwable throwable3) {
            return def;
        }
    }

    public static void move(@NonNull Path from, @NonNull Path to, CopyOption ... options) {
        if (from == null) {
            throw new NullPointerException("from is marked non-null but is null");
        }
        if (to == null) {
            throw new NullPointerException("to is marked non-null but is null");
        }
        if (options == null) {
            throw new NullPointerException("options is marked non-null but is null");
        }
        try {
            Files.move(from, to, options);
        }
        catch (IOException exception) {
            LOGGER.error("Exception moving file from {} to {}", new Object[]{from, to, exception});
        }
    }

    public static void copy(@Nullable InputStream inputStream, @Nullable OutputStream outputStream) {
        if (inputStream != null && outputStream != null) {
            try {
                inputStream.transferTo(outputStream);
            }
            catch (IOException exception) {
                LOGGER.error("Exception copying input stream to output stream", (Throwable)exception);
            }
        }
    }

    public static void copy(@Nullable InputStream inputStream, @Nullable Path target) {
        if (inputStream != null && target != null) {
            FileUtil.createDirectory(target.getParent());
            try (OutputStream out = Files.newOutputStream(target, new OpenOption[0]);){
                FileUtil.copy(inputStream, out);
            }
            catch (IOException exception) {
                LOGGER.error("Exception copying input stream to {}", (Object)target, (Object)exception);
            }
        }
    }

    public static void copy(@NonNull Path from, @NonNull Path to) {
        if (from == null) {
            throw new NullPointerException("from is marked non-null but is null");
        }
        if (to == null) {
            throw new NullPointerException("to is marked non-null but is null");
        }
        try {
            FileUtil.createDirectory(to.getParent());
            Files.copy(from, to, StandardCopyOption.REPLACE_EXISTING);
        }
        catch (IOException exception) {
            LOGGER.error("Exception copying file from {} to {}", new Object[]{from, to, exception});
        }
    }

    public static void copyDirectory(@NonNull Path from, @NonNull Path to) {
        if (from == null) {
            throw new NullPointerException("from is marked non-null but is null");
        }
        if (to == null) {
            throw new NullPointerException("to is marked non-null but is null");
        }
        FileUtil.copyDirectory(from, to, null);
    }

    public static void copyDirectory(@NonNull Path from, @NonNull Path to, @Nullable DirectoryStream.Filter<Path> filter) {
        if (from == null) {
            throw new NullPointerException("from is marked non-null but is null");
        }
        if (to == null) {
            throw new NullPointerException("to is marked non-null but is null");
        }
        FileUtil.walkFileTree(from, (Path $, Path current) -> {
            if (!Files.isDirectory(current, new LinkOption[0])) {
                FileUtil.copy(current, to.resolve(from.relativize((Path)current)));
            }
        }, true, filter == null ? ACCEPTING_FILTER : filter);
    }

    public static void delete(@Nullable Path path) {
        if (path != null && Files.exists(path, new LinkOption[0])) {
            if (Files.isDirectory(path, new LinkOption[0])) {
                FileUtil.walkFileTree(path, ($, current) -> FileUtil.delete(current));
            }
            try {
                Files.delete(path);
            }
            catch (IOException iOException) {
                // empty catch block
            }
        }
    }

    @NonNull
    public static Path createTempFile() {
        FileUtil.createDirectory(TEMP_DIR);
        return TEMP_DIR.resolve(UUID.randomUUID().toString());
    }

    public static void walkFileTree(@NonNull Path root, @NonNull BiConsumer<Path, Path> consumer) {
        if (root == null) {
            throw new NullPointerException("root is marked non-null but is null");
        }
        if (consumer == null) {
            throw new NullPointerException("consumer is marked non-null but is null");
        }
        FileUtil.walkFileTree(root, consumer, true);
    }

    public static void walkFileTree(@NonNull Path root, @NonNull BiConsumer<Path, Path> consumer, boolean visitDirs) {
        if (root == null) {
            throw new NullPointerException("root is marked non-null but is null");
        }
        if (consumer == null) {
            throw new NullPointerException("consumer is marked non-null but is null");
        }
        FileUtil.walkFileTree(root, consumer, visitDirs, "*");
    }

    public static void walkFileTree(@NonNull Path root, @NonNull BiConsumer<Path, Path> consumer, boolean visitDirectories, @NonNull String glob) {
        if (root == null) {
            throw new NullPointerException("root is marked non-null but is null");
        }
        if (consumer == null) {
            throw new NullPointerException("consumer is marked non-null but is null");
        }
        if (glob == null) {
            throw new NullPointerException("glob is marked non-null but is null");
        }
        if (Files.exists(root, new LinkOption[0])) {
            PathMatcher matcher = root.getFileSystem().getPathMatcher("glob:" + glob);
            FileUtil.walkFileTree(root, consumer, visitDirectories, (Path path) -> matcher.matches(path.getFileName()));
        }
    }

    public static void walkFileTree(@NonNull Path root, @NonNull BiConsumer<Path, Path> consumer, boolean visitDirectories, @NonNull DirectoryStream.Filter<Path> filter) {
        if (root == null) {
            throw new NullPointerException("root is marked non-null but is null");
        }
        if (consumer == null) {
            throw new NullPointerException("consumer is marked non-null but is null");
        }
        if (filter == null) {
            throw new NullPointerException("filter is marked non-null but is null");
        }
        if (Files.exists(root, new LinkOption[0])) {
            try (DirectoryStream<Path> stream = Files.newDirectoryStream(root, filter);){
                for (Path path : stream) {
                    if (Files.isDirectory(path, new LinkOption[0])) {
                        if (!visitDirectories) continue;
                        FileUtil.walkFileTree(path, consumer, true, filter);
                    }
                    consumer.accept(root, path);
                }
            }
            catch (IOException exception) {
                LOGGER.error("Exception walking down directory tree starting at {}", (Object)root, (Object)exception);
            }
        }
    }

    public static void createDirectory(@Nullable Path directoryPath) {
        if (directoryPath != null && Files.notExists(directoryPath, new LinkOption[0])) {
            try {
                Files.createDirectories(directoryPath, new FileAttribute[0]);
            }
            catch (IOException exception) {
                LOGGER.error("Exception creating directory at {}", (Object)directoryPath, (Object)exception);
            }
        }
    }

    public static void ensureChild(@NonNull Path root, @NonNull Path child) {
        if (root == null) {
            throw new NullPointerException("root is marked non-null but is null");
        }
        if (child == null) {
            throw new NullPointerException("child is marked non-null but is null");
        }
        Path rootNormal = root.normalize().toAbsolutePath();
        Path childNormal = child.normalize().toAbsolutePath();
        if (childNormal.getNameCount() <= rootNormal.getNameCount() || !childNormal.startsWith(rootNormal)) {
            throw new IllegalStateException("Child " + String.valueOf(childNormal) + " is not in root path " + String.valueOf(rootNormal));
        }
    }

    @NonNull
    public static Path resolve(@NonNull Path base, String ... more) {
        if (base == null) {
            throw new NullPointerException("base is marked non-null but is null");
        }
        if (more == null) {
            throw new NullPointerException("more is marked non-null but is null");
        }
        for (String child : more) {
            base = base.resolve(child);
        }
        return base;
    }

    static {
        ZIP_FILE_SYSTEM_PROPERTIES = Map.of("create", "false", "encoding", "UTF-8");
        JAR_FILE_SYSTEM_PROVIDER = FileSystemProvider.installedProviders().stream().filter(prov -> prov.getScheme().equalsIgnoreCase("jar")).findFirst().orElseThrow(() -> new ExceptionInInitializerError("Unable to find a file system provider supporting jars"));
        FileUtil.createDirectory(TEMP_DIR);
    }
}

