/*
 * Decompiled with CFR 0.152.
 */
package eu.europa.esig.dss.signature;

import eu.europa.esig.dss.model.signature.SignatureCryptographicVerification;
import eu.europa.esig.dss.model.x509.CertificateToken;
import eu.europa.esig.dss.model.x509.Token;
import eu.europa.esig.dss.signature.AbstractSignatureParameters;
import eu.europa.esig.dss.spi.DSSUtils;
import eu.europa.esig.dss.spi.exception.IllegalInputException;
import eu.europa.esig.dss.spi.signature.AdvancedSignature;
import eu.europa.esig.dss.spi.validation.CertificateVerifier;
import eu.europa.esig.dss.spi.validation.SignatureValidationAlerter;
import eu.europa.esig.dss.spi.validation.SignatureValidationContext;
import eu.europa.esig.dss.spi.validation.status.SignatureStatus;
import eu.europa.esig.dss.spi.validation.status.TokenStatus;
import eu.europa.esig.dss.utils.Utils;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.List;
import java.util.stream.Collectors;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SignatureRequirementsChecker {
    private static final Logger LOG = LoggerFactory.getLogger(SignatureRequirementsChecker.class);
    protected final CertificateVerifier certificateVerifier;
    protected final AbstractSignatureParameters<?> signatureParameters;

    public SignatureRequirementsChecker(CertificateVerifier certificateVerifier, AbstractSignatureParameters<?> signatureParameters) {
        this.certificateVerifier = certificateVerifier;
        this.signatureParameters = signatureParameters;
    }

    public void assertSigningCertificateIsValid(CertificateToken certificateToken) {
        this.assertCertificatesAreYetValid(certificateToken);
        this.assertSigningCertificateIsNotExpired(certificateToken);
        this.assertCertificatesAreNotRevoked(certificateToken);
    }

    public void assertSigningCertificateIsValid(AdvancedSignature signature) {
        this.assertSigningCertificateIsValid(Collections.singletonList(signature));
    }

    public void assertSigningCertificateIsValid(Collection<AdvancedSignature> signatures) {
        List signaturesToValidate = signatures.stream().filter(s -> !this.isSignatureGeneratedWithoutCertificate((AdvancedSignature)s)).collect(Collectors.toList());
        if (Utils.isCollectionEmpty(signaturesToValidate)) {
            return;
        }
        List<CertificateToken> signingCertificates = signaturesToValidate.stream().map(AdvancedSignature::getSigningCertificateToken).collect(Collectors.toList());
        this.assertCertificatesAreYetValid(signingCertificates, false);
        this.assertCertificatesAreNotExpired(signingCertificates, false);
        this.assertCertificatesAreNotRevoked(signatures);
    }

    private boolean isSignatureGeneratedWithoutCertificate(AdvancedSignature signature) {
        if (this.signatureParameters.isGenerateTBSWithoutCertificate() && signature.getCertificateSource().getNumberOfCertificates() == 0) {
            LOG.debug("Signature with Id '{}' has been generated without certificate. Validity of the signing-certificate is not checked.", (Object)signature.getId());
            return true;
        }
        return false;
    }

    private void assertCertificatesAreYetValid(CertificateToken certificateToken) {
        this.assertCertificatesAreYetValid(Collections.singletonList(certificateToken), true);
    }

    private void assertCertificatesAreYetValid(Collection<CertificateToken> certificateTokens, boolean signing) {
        if (this.certificateVerifier.getAlertOnNotYetValidCertificate() == null) {
            LOG.trace("The verification of #certificatesAreYetValid has been skipped.");
            return;
        }
        if (Utils.isCollectionEmpty(certificateTokens)) {
            return;
        }
        TokenStatus status = new TokenStatus();
        for (CertificateToken certificateToken : certificateTokens) {
            this.checkCertificateNotYetValid(certificateToken, status);
        }
        if (!status.isEmpty()) {
            if (signing) {
                status.setMessage("Error on signature creation.");
            } else {
                status.setMessage("Error on signature augmentation.");
            }
            this.certificateVerifier.getAlertOnNotYetValidCertificate().alert((Object)status);
        }
    }

    private void checkCertificateNotYetValid(CertificateToken certificateToken, TokenStatus status) {
        if (certificateToken == null) {
            throw new IllegalInputException("Signing-certificate token was not found! Unable to verify its validity range. Provide signing-certificate or use method #setGenerateTBSWithoutCertificate(true) for signature creation without signing-certificate.");
        }
        if (this.isCertificateNotYetValid(certificateToken)) {
            Date notBefore = certificateToken.getNotBefore();
            Date notAfter = certificateToken.getNotAfter();
            Date signingDate = this.signatureParameters.bLevel().getSigningDate();
            status.addRelatedTokenAndErrorMessage((Token)certificateToken, String.format("The signing-certificate (notBefore : %s, notAfter : %s) is not yet valid at signing time %s!", DSSUtils.formatDateToRFC((Date)notBefore), DSSUtils.formatDateToRFC((Date)notAfter), DSSUtils.formatDateToRFC((Date)signingDate)));
        }
    }

    private boolean isCertificateNotYetValid(CertificateToken certificateToken) {
        Date notBefore = certificateToken.getNotBefore();
        Date signingDate = this.signatureParameters.bLevel().getSigningDate();
        return signingDate.before(notBefore);
    }

    private void assertSigningCertificateIsNotExpired(CertificateToken certificateToken) {
        this.assertCertificatesAreNotExpired(Collections.singletonList(certificateToken), true);
    }

    private void assertCertificatesAreNotExpired(Collection<CertificateToken> certificateTokens, boolean signing) {
        if (this.certificateVerifier.getAlertOnExpiredCertificate() == null) {
            LOG.trace("The verification of #certificatesAreNotExpired has been skipped.");
            return;
        }
        if (Utils.isCollectionEmpty(certificateTokens)) {
            return;
        }
        TokenStatus status = new TokenStatus();
        for (CertificateToken certificateToken : certificateTokens) {
            this.checkCertificateExpired(certificateToken, status);
        }
        if (!status.isEmpty()) {
            if (signing) {
                status.setMessage("Error on signature creation.");
            } else {
                status.setMessage("Error on signature augmentation.");
            }
            this.certificateVerifier.getAlertOnExpiredCertificate().alert((Object)status);
        }
    }

    private void checkCertificateExpired(CertificateToken certificateToken, TokenStatus status) {
        if (certificateToken == null) {
            throw new IllegalInputException("Signing-certificate token was not found! Unable to verify its validity range. Provide signing-certificate or use method #setGenerateTBSWithoutCertificate(true) for signature creation without signing-certificate.");
        }
        if (this.isCertificateExpired(certificateToken)) {
            Date notBefore = certificateToken.getNotBefore();
            Date notAfter = certificateToken.getNotAfter();
            Date signingDate = this.signatureParameters.bLevel().getSigningDate();
            status.addRelatedTokenAndErrorMessage((Token)certificateToken, String.format("The signing-certificate (notBefore : %s, notAfter : %s) is expired at signing time %s!", DSSUtils.formatDateToRFC((Date)notBefore), DSSUtils.formatDateToRFC((Date)notAfter), DSSUtils.formatDateToRFC((Date)signingDate)));
        }
    }

    private boolean isCertificateExpired(CertificateToken certificateToken) {
        Date notAfter = certificateToken.getNotAfter();
        Date signingDate = this.signatureParameters.bLevel().getSigningDate();
        return signingDate.after(notAfter);
    }

    private void assertCertificatesAreNotRevoked(CertificateToken certificateToken) {
        if (!this.signatureParameters.isCheckCertificateRevocation()) {
            return;
        }
        if (this.certificateVerifier.getAlertOnMissingRevocationData() == null && this.certificateVerifier.getAlertOnRevokedCertificate() == null) {
            LOG.trace("The verification of #certificatesAreNotRevoked has been skipped.");
            return;
        }
        Date signingDate = this.signatureParameters.bLevel().getSigningDate();
        SignatureValidationContext validationContext = new SignatureValidationContext(signingDate);
        validationContext.initialize(this.certificateVerifier);
        List<CertificateToken> certificateChain = this.signatureParameters.getCertificateChain();
        if (Utils.isCollectionEmpty(certificateChain)) {
            throw new NullPointerException("Certificate chain shall be provided for a revocation check! Please use parameters.setCertificateChain(...) method to provide a certificate chain.");
        }
        validationContext.addCertificateTokenForVerification(certificateToken);
        for (CertificateToken certificate : certificateChain) {
            validationContext.addCertificateTokenForVerification(certificate);
        }
        validationContext.validate();
        SignatureValidationAlerter validationAlerter = new SignatureValidationAlerter(validationContext);
        validationAlerter.assertAllRequiredRevocationDataPresent();
        validationAlerter.assertCertificateNotRevoked(certificateToken);
    }

    private void assertCertificatesAreNotRevoked(Collection<AdvancedSignature> signatures) {
        if (!this.signatureParameters.isCheckCertificateRevocation()) {
            return;
        }
        if (this.certificateVerifier.getAlertOnMissingRevocationData() == null && this.certificateVerifier.getAlertOnRevokedCertificate() == null) {
            LOG.trace("The verification of #certificatesAreNotRevoked has been skipped.");
            return;
        }
        Date signingDate = this.signatureParameters.bLevel().getSigningDate();
        SignatureValidationContext validationContext = new SignatureValidationContext(signingDate);
        validationContext.initialize(this.certificateVerifier);
        for (AdvancedSignature signature : signatures) {
            validationContext.addSignatureForVerification(signature);
        }
        validationContext.validate();
        SignatureValidationAlerter validationAlerter = new SignatureValidationAlerter(validationContext);
        validationAlerter.assertAllRequiredRevocationDataPresent();
        validationAlerter.assertAllSignatureCertificatesNotRevoked();
    }

    public void assertExtendToTLevelPossible(List<AdvancedSignature> signatures) {
        this.assertTLevelIsHighest(signatures);
    }

    protected void assertTLevelIsHighest(List<AdvancedSignature> signatures) {
        if (this.certificateVerifier.getAugmentationAlertOnHigherSignatureLevel() == null) {
            LOG.trace("The verification of #tLevelIsHighest has been skipped.");
            return;
        }
        SignatureStatus status = new SignatureStatus();
        for (AdvancedSignature signature : signatures) {
            this.checkTLevelIsHighest(signature, status);
        }
        if (!status.isEmpty()) {
            status.setMessage("Error on signature augmentation to T-level.");
            this.certificateVerifier.getAugmentationAlertOnHigherSignatureLevel().alert((Object)status);
        }
    }

    protected void checkTLevelIsHighest(AdvancedSignature signature, SignatureStatus status) {
        if (this.hasLTLevelOrHigher(signature)) {
            status.addRelatedTokenAndErrorMessage(signature, "The signature is already extended with a higher level.");
        }
    }

    public boolean hasLTLevelOrHigher(AdvancedSignature signature) {
        return signature.hasLTAProfile() || (signature.hasLTProfile() || signature.hasCProfile()) && !signature.areAllSelfSignedCertificates() && signature.hasTProfile();
    }

    public void assertExtendToLTLevelPossible(List<AdvancedSignature> signatures) {
        this.assertLTLevelIsHighest(signatures);
    }

    protected void assertLTLevelIsHighest(List<AdvancedSignature> signatures) {
        if (this.certificateVerifier.getAugmentationAlertOnHigherSignatureLevel() == null) {
            LOG.trace("The verification of #ltLevelIsHighest has been skipped.");
            return;
        }
        SignatureStatus status = new SignatureStatus();
        for (AdvancedSignature signature : signatures) {
            this.checkLTLevelIsHighest(signature, status);
        }
        if (!status.isEmpty()) {
            status.setMessage("Error on signature augmentation to LT-level.");
            this.certificateVerifier.getAugmentationAlertOnHigherSignatureLevel().alert((Object)status);
        }
    }

    protected void checkLTLevelIsHighest(AdvancedSignature signature, SignatureStatus status) {
        if (this.hasLTALevelOrHigher(signature)) {
            status.addRelatedTokenAndErrorMessage(signature, "The signature is already extended with a higher level.");
        }
    }

    public boolean hasLTALevelOrHigher(AdvancedSignature signature) {
        return signature.hasLTAProfile();
    }

    public void assertCertificateChainValidForLTLevel(List<AdvancedSignature> signatures) {
        this.assertCertificateChainValid(signatures, "LT");
    }

    public void assertCertificateChainValidForCLevel(List<AdvancedSignature> signatures) {
        this.assertCertificateChainValid(signatures, "C");
    }

    public void assertCertificateChainValidForXLLevel(List<AdvancedSignature> signatures) {
        this.assertCertificateChainValid(signatures, "XL");
    }

    private void assertCertificateChainValid(List<AdvancedSignature> signatures, String targetLevel) {
        this.assertCertificatePresent(signatures, targetLevel);
        this.assertCertificatesAreNotSelfSigned(signatures, targetLevel);
    }

    private void assertCertificatePresent(List<AdvancedSignature> signatures, String targetLevel) {
        if (this.certificateVerifier.getAugmentationAlertOnSignatureWithoutCertificates() == null) {
            LOG.trace("The verification of #certificatePresent has been skipped.");
            return;
        }
        SignatureStatus status = new SignatureStatus();
        for (AdvancedSignature signature : signatures) {
            if (signature.getCertificateSource().getNumberOfCertificates() != 0) continue;
            status.addRelatedTokenAndErrorMessage(signature, "The signature does not contain certificates.");
        }
        if (!status.isEmpty()) {
            status.setMessage(String.format("Error on signature augmentation to %s-level.", targetLevel));
            this.certificateVerifier.getAugmentationAlertOnSignatureWithoutCertificates().alert((Object)status);
        }
    }

    private void assertCertificatesAreNotSelfSigned(List<AdvancedSignature> signatures, String targetLevel) {
        if (this.certificateVerifier.getAugmentationAlertOnSelfSignedCertificateChains() == null) {
            LOG.trace("The verification of #certificatesAreNotSelfSigned has been skipped.");
            return;
        }
        SignatureStatus status = new SignatureStatus();
        for (AdvancedSignature signature : signatures) {
            if (!signature.areAllSelfSignedCertificates()) continue;
            status.addRelatedTokenAndErrorMessage(signature, "The signature contains only self-signed certificate chains.");
        }
        if (!status.isEmpty()) {
            status.setMessage(String.format("Error on signature augmentation to %s-level.", targetLevel));
            this.certificateVerifier.getAugmentationAlertOnSelfSignedCertificateChains().alert((Object)status);
        }
    }

    public void assertExtendToCLevelPossible(List<AdvancedSignature> signatures) {
        this.assertCLevelIsHighest(signatures);
    }

    protected void assertCLevelIsHighest(List<AdvancedSignature> signatures) {
        if (this.certificateVerifier.getAugmentationAlertOnHigherSignatureLevel() == null) {
            LOG.trace("The verification of #cLevelIsHighest has been skipped.");
            return;
        }
        SignatureStatus status = new SignatureStatus();
        for (AdvancedSignature signature : signatures) {
            this.checkCLevelIsHighest(signature, status);
        }
        if (!status.isEmpty()) {
            status.setMessage("Error on signature augmentation to C-level.");
            this.certificateVerifier.getAugmentationAlertOnHigherSignatureLevel().alert((Object)status);
        }
    }

    protected void checkCLevelIsHighest(AdvancedSignature signature, SignatureStatus status) {
        if (this.hasXLevelOrHigher(signature)) {
            status.addRelatedTokenAndErrorMessage(signature, "The signature is already extended with a higher level.");
        }
    }

    public boolean hasXLevelOrHigher(AdvancedSignature signature) {
        return signature.hasXProfile() || signature.hasAProfile() || signature.hasXLProfile() && !signature.areAllSelfSignedCertificates() && signature.hasTProfile();
    }

    public void assertExtendToXLevelPossible(List<AdvancedSignature> signatures) {
        this.assertXLevelIsHighest(signatures);
    }

    protected void assertXLevelIsHighest(List<AdvancedSignature> signatures) {
        if (this.certificateVerifier.getAugmentationAlertOnHigherSignatureLevel() == null) {
            LOG.trace("The verification of #xLevelIsHighest has been skipped.");
            return;
        }
        SignatureStatus status = new SignatureStatus();
        for (AdvancedSignature signature : signatures) {
            this.checkXLevelIsHighest(signature, status);
        }
        if (!status.isEmpty()) {
            status.setMessage("Error on signature augmentation to X-level.");
            this.certificateVerifier.getAugmentationAlertOnHigherSignatureLevel().alert((Object)status);
        }
    }

    protected void checkXLevelIsHighest(AdvancedSignature signature, SignatureStatus status) {
        if (this.hasXLLevelOrHigher(signature)) {
            status.addRelatedTokenAndErrorMessage(signature, "The signature is already extended with a higher level.");
        }
    }

    public boolean hasXLLevelOrHigher(AdvancedSignature signature) {
        return signature.hasAProfile() || signature.hasXLProfile() && !signature.areAllSelfSignedCertificates() && signature.hasTProfile() && signature.hasXProfile();
    }

    public void assertExtendToXLLevelPossible(List<AdvancedSignature> signatures) {
        this.assertXLLevelIsHighest(signatures);
    }

    protected void assertXLLevelIsHighest(List<AdvancedSignature> signatures) {
        if (this.certificateVerifier.getAugmentationAlertOnHigherSignatureLevel() == null) {
            LOG.trace("The verification of #xlLevelIsHighest has been skipped.");
            return;
        }
        SignatureStatus status = new SignatureStatus();
        for (AdvancedSignature signature : signatures) {
            this.checkXLLevelIsHighest(signature, status);
        }
        if (!status.isEmpty()) {
            status.setMessage("Error on signature augmentation to XL-level.");
            this.certificateVerifier.getAugmentationAlertOnHigherSignatureLevel().alert((Object)status);
        }
    }

    protected void checkXLLevelIsHighest(AdvancedSignature signature, SignatureStatus status) {
        if (this.hasALevelOrHigher(signature)) {
            status.addRelatedTokenAndErrorMessage(signature, "The signature is already extended with a higher level.");
        }
    }

    public boolean hasALevelOrHigher(AdvancedSignature signature) {
        return this.hasLTALevelOrHigher(signature);
    }

    public void assertSignaturesValid(Collection<AdvancedSignature> signatures) {
        if (this.certificateVerifier.getAlertOnInvalidSignature() == null) {
            LOG.trace("The verification of #signaturesValid has been skipped.");
            return;
        }
        List signaturesToValidate = signatures.stream().filter(s -> !this.isSignatureGeneratedWithoutCertificate((AdvancedSignature)s)).collect(Collectors.toList());
        if (Utils.isCollectionEmpty(signaturesToValidate)) {
            return;
        }
        SignatureStatus status = new SignatureStatus();
        for (AdvancedSignature signature : signaturesToValidate) {
            SignatureCryptographicVerification signatureCryptographicVerification = signature.getSignatureCryptographicVerification();
            if (signatureCryptographicVerification.isSignatureIntact()) continue;
            String errorMessage = signatureCryptographicVerification.getErrorMessage();
            status.addRelatedTokenAndErrorMessage(signature, "Cryptographic signature verification has failed" + (errorMessage.isEmpty() ? "." : " / " + errorMessage));
        }
        if (!status.isEmpty()) {
            status.setMessage("Error on signature augmentation.");
            this.certificateVerifier.getAlertOnInvalidSignature().alert((Object)status);
        }
    }
}

