/*
 * Decompiled with CFR 0.152.
 */
package eu.europa.esig.dss.service.http.commons;

import eu.europa.esig.dss.enumerations.DigestAlgorithm;
import eu.europa.esig.dss.model.DSSDocument;
import eu.europa.esig.dss.model.DSSException;
import eu.europa.esig.dss.model.FileDocument;
import eu.europa.esig.dss.service.http.commons.ResourceLoader;
import eu.europa.esig.dss.spi.DSSUtils;
import eu.europa.esig.dss.spi.client.http.DSSCacheFileLoader;
import eu.europa.esig.dss.spi.client.http.DataLoader;
import eu.europa.esig.dss.spi.client.http.Protocol;
import eu.europa.esig.dss.spi.exception.DSSDataLoaderMultipleException;
import eu.europa.esig.dss.spi.exception.DSSExternalResourceException;
import eu.europa.esig.dss.utils.Utils;
import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Objects;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class FileCacheDataLoader
implements DataLoader,
DSSCacheFileLoader {
    private static final long serialVersionUID = 1028849693098211169L;
    private static final Logger LOG = LoggerFactory.getLogger(FileCacheDataLoader.class);
    private static final String DATA_LOADER_NOT_CONFIGURED = "The DataLoader is not configured";
    private File fileCacheDirectory = new File(System.getProperty("java.io.tmpdir"));
    private transient ResourceLoader resourceLoader = new ResourceLoader();
    private List<String> toBeLoaded;
    private List<String> toIgnored;
    private long cacheExpirationTime = -1L;
    private DataLoader dataLoader;

    public FileCacheDataLoader() {
    }

    public FileCacheDataLoader(DataLoader dataLoader) {
        this.dataLoader = dataLoader;
    }

    public DataLoader getDataLoader() {
        return this.dataLoader;
    }

    public void setDataLoader(DataLoader dataLoader) {
        this.dataLoader = dataLoader;
    }

    public void setFileCacheDirectory(File fileCacheDirectory) {
        Objects.requireNonNull(fileCacheDirectory, "File cache directory cannot be null!");
        this.fileCacheDirectory = fileCacheDirectory;
        if (!this.fileCacheDirectory.exists()) {
            if (this.fileCacheDirectory.mkdirs()) {
                LOG.info("A new directory '{}' has been successfully created.", (Object)fileCacheDirectory.getPath());
            } else {
                throw new IllegalStateException(String.format("Unable to create the directory '%s'!", fileCacheDirectory.getPath()));
            }
        }
    }

    public void setCacheExpirationTime(long cacheExpirationTimeInMilliseconds) {
        this.cacheExpirationTime = cacheExpirationTimeInMilliseconds;
    }

    public void setResourceLoader(ResourceLoader resourceLoader) {
        this.resourceLoader = resourceLoader;
    }

    public void addToBeLoaded(String url) {
        if (this.toBeLoaded == null) {
            this.toBeLoaded = new ArrayList<String>();
        }
        if (Utils.isStringNotBlank((String)url)) {
            this.toBeLoaded.add(url);
        }
    }

    public void addToBeIgnored(String urlString) {
        if (this.toIgnored == null) {
            this.toIgnored = new ArrayList<String>();
        }
        if (Utils.isStringNotBlank((String)urlString)) {
            String normalizedFileName = DSSUtils.getNormalizedString((String)urlString);
            this.toIgnored.add(normalizedFileName);
        }
    }

    public byte[] get(String url, boolean refresh) {
        DSSDocument document = this.getDocument(url, refresh);
        return DSSUtils.toByteArray((DSSDocument)document);
    }

    public byte[] get(String url) throws DSSException {
        return this.get(url, false);
    }

    public DSSDocument getDocument(String url, boolean refresh) {
        Objects.requireNonNull(this.dataLoader, DATA_LOADER_NOT_CONFIGURED);
        if (this.toBeLoaded != null && !this.toBeLoaded.contains(url)) {
            throw new DSSExternalResourceException(String.format("The toBeLoaded list does not contain URL [%s]!", url));
        }
        String fileName = DSSUtils.getNormalizedString((String)url);
        File file = this.getCacheFile(fileName);
        boolean fileExists = file.exists();
        boolean isCacheExpired = this.isCacheExpired(file);
        if (refresh) {
            LOG.trace("The refresh is forced for url '{}'.", (Object)url);
        } else if (!fileExists) {
            LOG.debug("There is no cached file for url '{}'.", (Object)url);
        } else if (isCacheExpired) {
            LOG.debug("Cache expired for url '{}'.", (Object)url);
        } else {
            LOG.debug("Cached file is used for url '{}'.", (Object)url);
            return new FileDocument(file);
        }
        byte[] bytes = !this.isNetworkProtocol(url) ? this.getLocalFileContent(url) : this.dataLoader.get(url);
        if (Utils.isArrayNotEmpty((byte[])bytes)) {
            File out = this.createFile(fileName, bytes);
            return new FileDocument(out);
        }
        throw new DSSExternalResourceException(String.format("Cannot retrieve data from url [%s]. Empty content is obtained!", url));
    }

    public DSSDocument getDocument(String url) {
        return this.getDocument(url, false);
    }

    public boolean remove(String url) {
        String fileName = DSSUtils.getNormalizedString((String)url);
        File file = this.getCacheFile(fileName);
        if (file.exists()) {
            if (LOG.isTraceEnabled()) {
                LOG.trace("Deleting the file corresponding to URL '{}'...", (Object)url);
            }
            try {
                Files.delete(file.toPath());
                return true;
            }
            catch (IOException e) {
                String errorMessage = "Unable to remove the cached file with URL '%s'. Reason : %s";
                if (LOG.isDebugEnabled()) {
                    LOG.warn(String.format(errorMessage, url, e.getMessage()), (Throwable)e);
                } else {
                    LOG.warn(String.format(errorMessage, url, e.getMessage()));
                }
                return false;
            }
        }
        if (LOG.isDebugEnabled()) {
            LOG.debug("Unable to remove the file corresponding to URL '{}'! The file does not exist.", (Object)url);
        }
        return false;
    }

    protected boolean isNetworkProtocol(String urlString) {
        String normalizedUrl = Utils.trim((String)urlString).toLowerCase();
        return Protocol.isHttpUrl((String)normalizedUrl) || Protocol.isLdapUrl((String)normalizedUrl) || Protocol.isFtpUrl((String)normalizedUrl);
    }

    private byte[] getLocalFileContent(String urlString) throws DSSException {
        Objects.requireNonNull(this.dataLoader, DATA_LOADER_NOT_CONFIGURED);
        String resourcePath = this.resourceLoader.getAbsoluteResourceFolder(Utils.trim((String)urlString));
        if (resourcePath != null) {
            File fileResource = new File(resourcePath);
            return DSSUtils.toByteArray((File)fileResource);
        }
        return this.dataLoader.get(urlString);
    }

    private File getCacheFile(String fileName) {
        String trimmedFileName = Utils.trim((String)fileName);
        if (this.toIgnored != null && this.toIgnored.contains(trimmedFileName)) {
            throw new DSSExternalResourceException("Part of urls to ignore.");
        }
        LOG.debug("Cached file: {}/{}", (Object)this.fileCacheDirectory, (Object)trimmedFileName);
        return new File(this.fileCacheDirectory, trimmedFileName);
    }

    public File createFile(String urlString, byte[] bytes) {
        String fileName = DSSUtils.getNormalizedString((String)urlString);
        File file = this.getCacheFile(fileName);
        DSSUtils.saveToFile((byte[])bytes, (File)file);
        return file;
    }

    @Deprecated
    public byte[] loadFileFromCache(String urlString) throws DSSException {
        String fileName = DSSUtils.getNormalizedString((String)urlString);
        File file = this.getCacheFile(fileName);
        if (file.exists()) {
            return DSSUtils.toByteArray((File)file);
        }
        throw new DSSExternalResourceException(String.format("The file with URL [%s] does not exist in the cache!", urlString));
    }

    public DSSDocument getDocumentFromCache(String url) {
        String fileName = DSSUtils.getNormalizedString((String)url);
        File file = this.getCacheFile(fileName);
        if (file.exists()) {
            return new FileDocument(file);
        }
        return null;
    }

    public byte[] post(String urlString, byte[] content) throws DSSException {
        Objects.requireNonNull(this.dataLoader, DATA_LOADER_NOT_CONFIGURED);
        String fileName = DSSUtils.getNormalizedString((String)urlString);
        byte[] digest = DSSUtils.digest((DigestAlgorithm)DigestAlgorithm.MD5, (byte[])content);
        String digestHexEncoded = DSSUtils.toHex((byte[])digest);
        String cacheFileName = fileName + "." + digestHexEncoded;
        File file = this.getCacheFile(cacheFileName);
        boolean fileExists = file.exists();
        boolean isCacheExpired = this.isCacheExpired(file);
        if (fileExists && !isCacheExpired) {
            LOG.debug("Cached file was used");
            return DSSUtils.toByteArray((File)file);
        }
        LOG.debug("There is no cached file!");
        byte[] returnedBytes = null;
        if (this.isNetworkProtocol(urlString)) {
            returnedBytes = this.dataLoader.post(urlString, content);
        }
        if (Utils.isArrayNotEmpty(returnedBytes)) {
            File cacheFile = this.getCacheFile(cacheFileName);
            DSSUtils.saveToFile((byte[])returnedBytes, (File)cacheFile);
            return returnedBytes;
        }
        throw new DSSExternalResourceException(String.format("Cannot retrieve data from URL [%s]", urlString));
    }

    private boolean isCacheExpired(File file) {
        if (this.cacheExpirationTime < 0L) {
            return false;
        }
        if (!file.exists()) {
            return true;
        }
        long currentTime = new Date().getTime();
        if (currentTime - file.lastModified() >= this.cacheExpirationTime) {
            LOG.debug("Cache is expired");
            return true;
        }
        return false;
    }

    public DataLoader.DataAndUrl get(List<String> urlStrings) {
        if (Utils.isCollectionEmpty(urlStrings)) {
            throw new DSSExternalResourceException("Cannot process the GET call. List of URLs is empty!");
        }
        HashMap<String, Exception> exceptions = new HashMap<String, Exception>();
        for (String urlString : urlStrings) {
            LOG.debug("Processing a GET call to URL [{}]...", (Object)urlString);
            try {
                byte[] bytes = this.get(urlString);
                if (Utils.isArrayEmpty((byte[])bytes)) {
                    LOG.debug("The retrieved content from URL [{}] is empty. Continue with other URLs...", (Object)urlString);
                    continue;
                }
                return new DataLoader.DataAndUrl(urlString, bytes);
            }
            catch (Exception e) {
                LOG.warn("Cannot obtain data using '{}' : {}", (Object)urlString, (Object)e.getMessage());
                exceptions.put(urlString, e);
            }
        }
        throw new DSSDataLoaderMultipleException(exceptions);
    }

    public void setContentType(String contentType) {
        Objects.requireNonNull(this.dataLoader, DATA_LOADER_NOT_CONFIGURED);
        this.dataLoader.setContentType(contentType);
    }
}

