/*
 * Decompiled with CFR 0.152.
 */
package eu.fbk.knowledgestore.datastore;

import com.google.common.base.Objects;
import com.google.common.base.Preconditions;
import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import eu.fbk.knowledgestore.data.Record;
import eu.fbk.knowledgestore.data.Stream;
import eu.fbk.knowledgestore.data.XPath;
import eu.fbk.knowledgestore.datastore.DataStore;
import eu.fbk.knowledgestore.datastore.DataTransaction;
import eu.fbk.knowledgestore.datastore.ForwardingDataStore;
import eu.fbk.knowledgestore.datastore.ForwardingDataTransaction;
import eu.fbk.knowledgestore.vocabulary.KS;
import java.io.IOException;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.locks.ReadWriteLock;
import java.util.concurrent.locks.ReentrantReadWriteLock;
import javax.annotation.Nullable;
import org.openrdf.model.URI;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class CachingDataStore
extends ForwardingDataStore {
    private static final Logger LOGGER = LoggerFactory.getLogger(CachingDataStore.class);
    private static final int DEFAULT_MAX_SIZE = 1024;
    private static final int DEFAULT_MAX_CHANGES = 1024;
    private static final int DEFAULT_MAX_BUFFERED_CHANGES = 1024;
    private static final Record NULL = Record.create();
    private final DataStore delegate;
    private final int maxChanges;
    private final int maxBufferedChanges;
    private final ReadWriteLock globalLock;
    private final Map<URI, Cache<URI, Record>> globalCaches;
    private long globalRevision;
    private final AtomicLong globalHitCount;
    private final AtomicLong localHitCount;
    private final AtomicLong fetchCount;
    private final AtomicLong changeCount;
    private final AtomicLong flushCount;

    public CachingDataStore(DataStore delegate, @Nullable Integer maxSize, @Nullable Integer maxChanges, @Nullable Integer maxBufferedChanges) {
        int actualMaxSize = (Integer)Objects.firstNonNull((Object)maxSize, (Object)1024);
        int actualMaxChanges = (Integer)Objects.firstNonNull((Object)maxChanges, (Object)1024);
        int actualMaxBufferedChanges = (Integer)Objects.firstNonNull((Object)maxBufferedChanges, (Object)1024);
        Preconditions.checkArgument((actualMaxSize > 0 ? 1 : 0) != 0);
        Preconditions.checkArgument((actualMaxChanges > 0 ? 1 : 0) != 0);
        this.delegate = (DataStore)Preconditions.checkNotNull((Object)delegate);
        this.maxChanges = actualMaxChanges;
        this.maxBufferedChanges = actualMaxBufferedChanges;
        this.globalLock = new ReentrantReadWriteLock(true);
        this.globalCaches = Maps.newHashMap();
        this.globalRevision = 0L;
        this.globalHitCount = new AtomicLong(0L);
        this.localHitCount = new AtomicLong(0L);
        this.fetchCount = new AtomicLong(0L);
        this.changeCount = new AtomicLong(0L);
        this.flushCount = new AtomicLong(0L);
        for (URI type : DataStore.SUPPORTED_TYPES) {
            this.globalCaches.put(type, (Cache<URI, Record>)CacheBuilder.newBuilder().softValues().maximumSize((long)actualMaxSize).build());
        }
        LOGGER.info("{} configured", (Object)this.getClass().getSimpleName());
    }

    @Override
    protected DataStore delegate() {
        return this.delegate;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public DataTransaction begin(boolean readOnly) throws IOException, IllegalStateException {
        this.globalLock.readLock().lock();
        try {
            long revision = this.globalRevision;
            DataTransaction tx = this.delegate().begin(readOnly);
            CachingDataTransaction cachingDataTransaction = new CachingDataTransaction(tx, readOnly, revision);
            return cachingDataTransaction;
        }
        finally {
            this.globalLock.readLock().unlock();
        }
    }

    @Override
    public void close() {
        try {
            LOGGER.info("{} - {} local cache hits, {} global cache hits, {} fetches, {} changes, {} flushes", new Object[]{this.getClass().getSimpleName(), this.localHitCount, this.globalHitCount, this.fetchCount, this.changeCount, this.flushCount});
        }
        finally {
            super.close();
        }
    }

    private class CachingDataTransaction
    extends ForwardingDataTransaction {
        private final DataTransaction delegate;
        @Nullable
        private final Set<URI> dirty;
        @Nullable
        private final Map<URI, Map<URI, Record>> changes;
        @Nullable
        private final Map<URI, Set<URI>> invalidated;
        private final Map<URI, Cache<URI, Record>> localCaches;
        private final long localRevision;

        CachingDataTransaction(DataTransaction delegate, boolean readOnly, long revision) {
            this.delegate = (DataTransaction)Preconditions.checkNotNull((Object)delegate);
            if (readOnly) {
                this.dirty = null;
                this.changes = null;
                this.invalidated = null;
            } else {
                this.dirty = Sets.newHashSet();
                this.changes = Maps.newHashMap();
                this.invalidated = Maps.newHashMap();
                for (URI type : DataStore.SUPPORTED_TYPES) {
                    this.changes.put(type, Maps.newHashMap());
                    this.invalidated.put(type, Sets.newHashSet());
                }
            }
            this.localRevision = revision;
            this.localCaches = Maps.newHashMap();
            for (URI type : DataStore.SUPPORTED_TYPES) {
                this.localCaches.put(type, (Cache<URI, Record>)CacheBuilder.newBuilder().softValues().build());
            }
        }

        @Override
        protected DataTransaction delegate() {
            return this.delegate;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public Stream<Record> lookup(URI type, Set<? extends URI> ids, Set<? extends URI> properties) throws IOException, IllegalArgumentException, IllegalStateException {
            long globalRevision = CachingDataStore.this.globalRevision;
            Cache globalCache = (Cache)CachingDataStore.this.globalCaches.get(type);
            Cache<URI, Record> localCache = this.localCaches.get(type);
            ArrayList result = Lists.newArrayList();
            boolean mightUseGlobalCache = this.localRevision == globalRevision && (this.dirty == null || !this.dirty.contains(type));
            HashSet missingIDs = Sets.newHashSet();
            for (URI uRI : ids) {
                Record record2 = (Record)localCache.getIfPresent((Object)uRI);
                if (record2 == null) {
                    missingIDs.add(uRI);
                    continue;
                }
                if (record2 == NULL) continue;
                CachingDataStore.this.localHitCount.incrementAndGet();
                result.add(Record.create((Record)record2, (boolean)true));
            }
            if (mightUseGlobalCache) {
                CachingDataStore.this.globalLock.readLock().lock();
                try {
                    if (this.localRevision == globalRevision) {
                        Iterator i = missingIDs.iterator();
                        while (i.hasNext()) {
                            URI uRI = (URI)i.next();
                            Record record = (Record)globalCache.getIfPresent((Object)uRI);
                            if (record == null) continue;
                            CachingDataStore.this.globalHitCount.incrementAndGet();
                            localCache.put((Object)uRI, (Object)record);
                            result.add(Record.create((Record)record, (boolean)true));
                            i.remove();
                        }
                    }
                }
                finally {
                    CachingDataStore.this.globalLock.readLock().unlock();
                }
            }
            ImmutableList fetched = missingIDs.isEmpty() ? ImmutableList.of() : this.delegate().lookup(type, missingIDs, null).toList();
            CachingDataStore.this.fetchCount.addAndGet(missingIDs.size());
            for (Record record : fetched) {
                result.add(Record.create((Record)record, (boolean)true));
                localCache.put((Object)record.getID(), (Object)record);
                missingIDs.remove(record.getID());
            }
            for (URI id : missingIDs) {
                localCache.put((Object)id, (Object)NULL);
            }
            if (mightUseGlobalCache) {
                CachingDataStore.this.globalLock.readLock().lock();
                try {
                    if (this.localRevision == CachingDataStore.this.globalRevision) {
                        for (Record record : fetched) {
                            globalCache.put((Object)record.getID(), (Object)record);
                        }
                    }
                }
                finally {
                    CachingDataStore.this.globalLock.readLock().unlock();
                }
            }
            if (properties != null && !properties.isEmpty()) {
                for (Record record : result) {
                    URI[] projection = properties.toArray(new URI[properties.size()]);
                    record.retain(projection);
                }
            }
            return Stream.create((Iterable)result);
        }

        @Override
        public Stream<Record> retrieve(URI type, XPath condition, Set<? extends URI> properties) throws IOException, IllegalArgumentException, IllegalStateException {
            if (this.changes != null) {
                this.flushChanges(type);
            }
            return this.delegate().retrieve(type, condition, properties);
        }

        @Override
        public long count(URI type, XPath condition) throws IOException, IllegalArgumentException, IllegalStateException {
            if (this.changes != null) {
                this.flushChanges(type);
            }
            return this.delegate().count(type, condition);
        }

        @Override
        public Stream<Record> match(Map<URI, XPath> conditions, Map<URI, Set<URI>> ids, Map<URI, Set<URI>> properties) throws IOException, IllegalStateException {
            if (this.changes != null) {
                this.flushChanges(KS.RESOURCE);
                this.flushChanges(KS.MENTION);
                this.flushChanges(KS.ENTITY);
                this.flushChanges(KS.AXIOM);
            }
            return this.delegate().match(conditions, ids, properties);
        }

        @Override
        public void store(URI type, Record record) throws IOException, IllegalStateException {
            Preconditions.checkState((this.changes != null ? 1 : 0) != 0, (Object)"Read-only DataTransaction");
            this.registerChange(type, record.getID(), record);
        }

        @Override
        public void delete(URI type, URI id) throws IOException, IllegalStateException {
            Preconditions.checkState((this.changes != null ? 1 : 0) != 0, (Object)"Read-only DataTransaction");
            this.registerChange(type, id, NULL);
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public void end(boolean commit) throws IOException, IllegalStateException {
            if (this.changes == null || !commit) {
                this.delegate.end(commit);
                return;
            }
            for (URI type : DataStore.SUPPORTED_TYPES) {
                this.flushChanges(type);
            }
            CachingDataStore.this.globalLock.writeLock().lock();
            try {
                this.delegate().end(true);
                ++CachingDataStore.this.globalRevision;
                for (URI type : DataStore.SUPPORTED_TYPES) {
                    this.synchronizeCaches(this.invalidated.get(type), this.localCaches.get(type), (Cache<URI, Record>)((Cache)CachingDataStore.this.globalCaches.get(type)));
                }
            }
            finally {
                CachingDataStore.this.globalLock.writeLock().unlock();
            }
        }

        private void synchronizeCaches(@Nullable Set<URI> invalidatedIDs, Cache<URI, Record> localCache, Cache<URI, Record> globalCache) {
            if (invalidatedIDs == null) {
                globalCache.invalidateAll();
                return;
            }
            globalCache.invalidateAll(invalidatedIDs);
            for (Map.Entry entry : localCache.asMap().entrySet()) {
                URI id = (URI)entry.getKey();
                Record record = (Record)entry.getValue();
                if (record == NULL) continue;
                globalCache.put((Object)id, (Object)record);
            }
        }

        private void registerChange(URI type, URI id, Record record) throws IOException {
            Set<URI> invalidatedIDs;
            assert (this.changes != null && this.invalidated != null);
            CachingDataStore.this.changeCount.incrementAndGet();
            this.localCaches.get(type).put((Object)id, (Object)record);
            Map<URI, Record> changeMap = this.changes.get(type);
            changeMap.put(id, record);
            if (changeMap.size() > CachingDataStore.this.maxBufferedChanges) {
                this.flushChanges(type);
            }
            if ((invalidatedIDs = this.invalidated.get(type)) != null) {
                invalidatedIDs.add(id);
                if (invalidatedIDs.size() > CachingDataStore.this.maxChanges) {
                    this.invalidated.put(type, null);
                }
            }
        }

        private void flushChanges(URI type) throws IOException {
            assert (this.changes != null && this.invalidated != null);
            Map<URI, Record> map = this.changes.get(type);
            if (map.isEmpty()) {
                return;
            }
            this.dirty.add(type);
            CachingDataStore.this.flushCount.addAndGet(map.size());
            for (Map.Entry<URI, Record> entry : map.entrySet()) {
                URI id = entry.getKey();
                Record record = entry.getValue();
                if (record == NULL) {
                    this.delegate().delete(type, id);
                    continue;
                }
                this.delegate().store(type, record);
            }
            map.clear();
        }
    }
}

