/*
 * Decompiled with CFR 0.152.
 */
package eu.fbk.knowledgestore.filestore;

import com.google.common.base.MoreObjects;
import com.google.common.base.Preconditions;
import com.google.common.base.Throwables;
import com.google.common.collect.AbstractIterator;
import com.google.common.collect.HashMultiset;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.Multiset;
import com.google.common.io.ByteStreams;
import eu.fbk.knowledgestore.data.Data;
import eu.fbk.knowledgestore.data.Stream;
import eu.fbk.knowledgestore.filestore.FileExistsException;
import eu.fbk.knowledgestore.filestore.FileMissingException;
import eu.fbk.knowledgestore.filestore.FileStore;
import java.io.File;
import java.io.FilterInputStream;
import java.io.FilterOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.Map;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.locks.ReadWriteLock;
import java.util.concurrent.locks.ReentrantReadWriteLock;
import java.util.zip.ZipEntry;
import java.util.zip.ZipInputStream;
import java.util.zip.ZipOutputStream;
import javax.annotation.Nullable;
import org.apache.hadoop.fs.FSDataInputStream;
import org.apache.hadoop.fs.FSDataOutputStream;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.lucene.analysis.Analyzer;
import org.apache.lucene.analysis.WhitespaceAnalyzer;
import org.apache.lucene.document.Document;
import org.apache.lucene.document.Field;
import org.apache.lucene.document.Fieldable;
import org.apache.lucene.index.IndexReader;
import org.apache.lucene.index.IndexWriter;
import org.apache.lucene.index.Term;
import org.apache.lucene.index.TermDocs;
import org.apache.lucene.store.Directory;
import org.apache.lucene.store.FSDirectory;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class HadoopMultiFileStore
implements FileStore {
    private static final Logger LOGGER = LoggerFactory.getLogger(HadoopMultiFileStore.class);
    private static final String DEFAULT_ROOT_PATH = "files";
    private static final String DEFAULT_LUCENE_PATH = "./lucene-index";
    private static final int DEFAULT_NUM_SMALL_FILES = 10;
    private static final long DEFAULT_CLEANUP_PERIOD = 10000L;
    private static final String SMALL_FILES_PATH = "_small";
    private final int MAX_LUCENE_ATTEMPTS = 3;
    private static final String KEY_FIELD = "filename";
    private static final String VALUE_FIELD = "zipfilename";
    private static final String DELETED = "__deleted";
    private final FileSystem fileSystem;
    private final Path rootPath;
    private final Path smallFilesPath;
    private final File luceneFolder;
    private final int numSmallFiles;
    private final long cleanupPeriod;
    private final Multiset<String> openedFiles;
    private final ReadWriteLock lock;
    private final AtomicBoolean active;
    private IndexReader luceneReader;
    private IndexWriter luceneWriter;
    private Future<?> cleanupFuture;
    private long zipNameCounter;

    public HadoopMultiFileStore(FileSystem fileSystem, @Nullable String lucenePath, @Nullable String path, @Nullable Integer numSmallFile, @Nullable Long cleanupPeriod) {
        this.fileSystem = (FileSystem)Preconditions.checkNotNull((Object)fileSystem);
        this.luceneFolder = new File((String)MoreObjects.firstNonNull((Object)lucenePath, (Object)DEFAULT_LUCENE_PATH));
        this.rootPath = new Path((String)MoreObjects.firstNonNull((Object)path, (Object)DEFAULT_ROOT_PATH)).makeQualified(this.fileSystem);
        this.smallFilesPath = new Path(this.rootPath.toString() + File.separator + SMALL_FILES_PATH).makeQualified(this.fileSystem);
        this.numSmallFiles = numSmallFile != null ? numSmallFile : 10;
        this.cleanupPeriod = cleanupPeriod != null ? cleanupPeriod : 10000L;
        this.openedFiles = HashMultiset.create();
        this.lock = new ReentrantReadWriteLock(true);
        this.active = new AtomicBoolean(false);
        this.zipNameCounter = System.currentTimeMillis();
        LOGGER.info("{} configured, paths={};{}", new Object[]{this.getClass().getSimpleName(), this.rootPath, this.luceneFolder});
    }

    @Override
    public void init() throws IOException {
        if (!this.fileSystem.exists(this.rootPath)) {
            LOGGER.debug("Creating root folder {}", (Object)this.rootPath);
            if (!this.fileSystem.mkdirs(this.rootPath)) {
                throw new IOException("Cannot create root folter " + this.luceneFolder);
            }
        }
        if (!this.fileSystem.exists(this.smallFilesPath)) {
            LOGGER.debug("Creating small files folder {}", (Object)this.smallFilesPath);
            if (!this.fileSystem.mkdirs(this.smallFilesPath)) {
                throw new IOException("Cannot create small files folter " + this.smallFilesPath);
            }
        }
        if (!this.luceneFolder.exists()) {
            LOGGER.debug("Created lucene folder {}", (Object)this.luceneFolder);
            if (!this.luceneFolder.mkdirs()) {
                throw new IOException("Cannot create lucene folder " + this.luceneFolder);
            }
        }
        this.luceneWriter = new IndexWriter((Directory)FSDirectory.open((File)this.luceneFolder), (Analyzer)new WhitespaceAnalyzer(), IndexWriter.MaxFieldLength.LIMITED);
        this.luceneReader = this.luceneWriter.getReader();
        this.active.set(true);
        this.cleanupFuture = Data.getExecutor().scheduleWithFixedDelay(new Runnable(){

            @Override
            public void run() {
                try {
                    HadoopMultiFileStore.this.merge();
                    HadoopMultiFileStore.this.purge();
                    HadoopMultiFileStore.this.indexOptimize();
                }
                catch (Throwable ex) {
                    LOGGER.warn("Periodic cleanup failed", ex);
                }
            }
        }, this.cleanupPeriod, this.cleanupPeriod, TimeUnit.MILLISECONDS);
    }

    @Override
    public InputStream read(String fileName) throws IOException {
        this.lock.readLock().lock();
        try {
            Preconditions.checkState((boolean)this.active.get());
            String zipName = this.indexGet(fileName);
            if (!DELETED.equals(zipName)) {
                if (zipName != null) {
                    Path zipPath = this.pathForZipFile(zipName);
                    try {
                        ZipEntry entry;
                        ZipInputStream stream = new ZipInputStream(this.openForRead(zipPath));
                        while ((entry = stream.getNextEntry()) != null) {
                            if (!entry.getName().equals(fileName)) continue;
                            LOGGER.debug("Reading {} from ZIP file {}", (Object)fileName, (Object)zipPath);
                            ZipInputStream zipInputStream = stream;
                            return zipInputStream;
                        }
                    }
                    catch (IOException ex) {
                        throw new IOException("Cannot read " + fileName + " from ZIP file" + zipPath, ex);
                    }
                } else {
                    Path smallPath = this.pathForSmallFile(fileName);
                    if (this.fileSystem.exists(smallPath)) {
                        LOGGER.debug("Reading small file {}", (Object)smallPath);
                        InputStream inputStream = this.openForRead(smallPath);
                        return inputStream;
                    }
                }
            }
            throw new FileMissingException(fileName, "The file does not exist");
        }
        finally {
            this.lock.readLock().unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public OutputStream write(String fileName) throws IOException {
        this.lock.writeLock().lock();
        try {
            Preconditions.checkState((boolean)this.active.get());
            String zipName = this.indexGet(fileName);
            Path smallPath = this.pathForSmallFile(fileName);
            boolean fileExists = this.fileSystem.exists(smallPath);
            if (!DELETED.equals(zipName) && (zipName != null || fileExists)) {
                throw new FileExistsException(fileName, "Cannot overwrite file");
            }
            LOGGER.debug("Creating small file {}", (Object)smallPath);
            OutputStream stream = this.openForWrite(smallPath);
            if (fileExists) {
                try {
                    HashMap<String, String> entries = new HashMap<String, String>();
                    entries.put(fileName, null);
                    this.indexPut(entries);
                }
                catch (Throwable ex) {
                    stream.close();
                    Throwables.propagateIfPossible((Throwable)ex, IOException.class);
                    Throwables.propagate((Throwable)ex);
                }
            }
            OutputStream outputStream = stream;
            return outputStream;
        }
        finally {
            this.lock.writeLock().unlock();
        }
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    @Override
    public void delete(String fileName) throws FileMissingException, IOException {
        this.lock.writeLock().lock();
        try {
            Preconditions.checkState((boolean)this.active.get());
            String zipName = this.indexGet(fileName);
            if (DELETED.equals(zipName)) throw new FileMissingException(fileName, "The file does not exist");
            if (zipName != null) {
                HashMap<String, String> entries = new HashMap<String, String>();
                Path zipPath = this.pathForZipFile(zipName);
                LOGGER.debug("Exploding zip file {}", (Object)zipPath);
                try (ZipInputStream zipStream = new ZipInputStream(this.openForRead(zipPath));){
                    ZipEntry entry;
                    while ((entry = zipStream.getNextEntry()) != null) {
                        OutputStream stream;
                        block33: {
                            String smallName = entry.getName();
                            if (smallName.equals(fileName)) continue;
                            entries.put(smallName, null);
                            Path smallPath = this.pathForSmallFile(smallName);
                            stream = this.openForWrite(smallPath);
                            Throwable throwable = null;
                            try {
                                ByteStreams.copy((InputStream)zipStream, (OutputStream)stream);
                                if (stream == null) continue;
                                if (throwable == null) break block33;
                            }
                            catch (Throwable throwable2) {
                                try {
                                    throwable = throwable2;
                                    throw throwable2;
                                }
                                catch (Throwable throwable3) {
                                    if (stream == null) throw throwable3;
                                    if (throwable == null) {
                                        stream.close();
                                        throw throwable3;
                                    }
                                    try {
                                        stream.close();
                                        throw throwable3;
                                    }
                                    catch (Throwable throwable4) {
                                        throwable.addSuppressed(throwable4);
                                        throw throwable3;
                                    }
                                }
                            }
                            try {
                                stream.close();
                                continue;
                            }
                            catch (Throwable throwable5) {
                                throwable.addSuppressed(throwable5);
                                continue;
                            }
                        }
                        stream.close();
                    }
                }
                catch (IOException ex) {
                    Iterator iterator = entries.keySet().iterator();
                    while (iterator.hasNext()) {
                        String smallName = (String)iterator.next();
                        Path smallPath = this.pathForSmallFile(smallName);
                        try {
                            this.fileSystem.delete(smallPath, false);
                        }
                        catch (Throwable ex2) {
                            LOGGER.warn("Could not delete extracted file " + smallPath + " after failed to explod ZIP file " + zipPath, ex2);
                        }
                    }
                    throw new IOException("Cannot explode ZIP file " + zipPath, ex);
                }
                entries.put(fileName, DELETED);
                entries.put(zipName, DELETED);
                this.indexPut(entries);
                return;
            }
            Path smallPath = this.pathForSmallFile(fileName);
            if (!this.fileSystem.exists(smallPath)) throw new FileMissingException(fileName, "The file does not exist");
            LOGGER.debug("Marking small file {} as deleted", (Object)smallPath);
            this.indexPut((Map<String, String>)ImmutableMap.of((Object)fileName, (Object)DELETED));
            return;
        }
        finally {
            this.lock.writeLock().unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public Stream<String> list() throws IOException {
        this.lock.readLock().lock();
        try {
            Preconditions.checkState((boolean)this.active.get());
            ArrayList<String> smallNames = new ArrayList<String>();
            for (FileStatus fs : this.fileSystem.listStatus(this.smallFilesPath)) {
                String smallName = fs.getPath().getName();
                if (this.indexGet(smallName) == null) continue;
                smallNames.add(smallName);
            }
            Iterator<String> zippedNames = this.indexList(false);
            Stream stream = Stream.concat((Iterable[])new Iterable[]{Stream.create(smallNames), Stream.create(zippedNames)});
            return stream;
        }
        finally {
            this.lock.readLock().unlock();
        }
    }

    @Override
    public void close() {
        this.lock.writeLock().lock();
        try {
            try {
                this.cleanupFuture.cancel(false);
            }
            catch (Throwable ex) {
                LOGGER.warn("Unable to stop periodic cleanup task", ex);
            }
            try {
                this.luceneReader.close();
            }
            catch (Throwable ex) {
                LOGGER.warn("Unable to close Lucene reader", ex);
            }
            try {
                this.luceneWriter.optimize();
            }
            catch (Exception ex) {
                LOGGER.warn("Unable to optimize Lucene writer", (Throwable)ex);
            }
            try {
                this.luceneWriter.close();
            }
            catch (Exception ex) {
                LOGGER.warn("Unable to close Lucene writer", (Throwable)ex);
            }
        }
        finally {
            this.active.set(false);
            this.lock.writeLock().unlock();
        }
    }

    public String toString() {
        return this.getClass().getSimpleName();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void purge() throws IOException {
        this.lock.writeLock().lock();
        try {
            HashSet<String> purgableFiles = new HashSet<String>();
            Iterator<String> i = this.indexList(true);
            while (i.hasNext()) {
                purgableFiles.add(i.next());
            }
            Multiset<String> files = this.fileSystem.listStatus(this.smallFilesPath);
            if (files != null) {
                for (Multiset<String> fs : files) {
                    String fileName = fs.getPath().getName();
                    if (this.indexGet(fileName) == null) continue;
                    purgableFiles.add(fileName);
                }
            }
            Multiset<String> multiset = this.openedFiles;
            synchronized (multiset) {
                purgableFiles.removeAll(this.openedFiles.elementSet());
            }
            if (purgableFiles.isEmpty()) {
                return;
            }
            LOGGER.debug("Purging {} files", (Object)purgableFiles.size());
            HashMap<String, String> entries = new HashMap<String, String>();
            for (String file : purgableFiles) {
                try {
                    Path smallPath = this.pathForSmallFile(file);
                    if (this.fileSystem.exists(smallPath)) {
                        this.fileSystem.delete(smallPath, false);
                        LOGGER.debug("Deleted small file {}", (Object)smallPath);
                    } else {
                        Path zipPath = this.pathForZipFile(file);
                        if (this.fileSystem.exists(zipPath)) {
                            this.fileSystem.delete(zipPath, false);
                            LOGGER.debug("Deleted ZIP file {}", (Object)zipPath);
                        } else {
                            LOGGER.warn("Cannot find file " + file);
                        }
                    }
                    entries.put(file, null);
                }
                catch (Throwable ex) {
                    LOGGER.warn("Cannot purge file " + file, ex);
                }
            }
            this.indexPut(entries);
        }
        finally {
            this.lock.writeLock().unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     * WARNING - void declaration
     */
    private void merge() throws IOException {
        this.lock.writeLock().lock();
        try {
            LinkedList<String> mergeableNames = new LinkedList<String>();
            FileStatus[] files = this.fileSystem.listStatus(this.smallFilesPath);
            if (files != null) {
                for (FileStatus fs : files) {
                    String name = fs.getPath().getName();
                    if (fs.isDir() || this.indexGet(name) != null) continue;
                    mergeableNames.add(name);
                }
            }
            while (mergeableNames.size() > this.numSmallFiles) {
                String zipName = Data.hash((Object[])new Object[]{this.zipNameCounter++}) + ".zip";
                Path zipPath = this.pathForZipFile(zipName);
                boolean opened = false;
                HashMap<String, String> entries = new HashMap<String, String>();
                try {
                    ZipOutputStream out = new ZipOutputStream(this.openForWrite(zipPath));
                    Object object = null;
                    try {
                        void var9_14;
                        opened = true;
                        boolean bl = false;
                        while (var9_14 < this.numSmallFiles) {
                            String fileName = (String)mergeableNames.remove(0);
                            out.putNextEntry(new ZipEntry(fileName));
                            try (InputStream in = this.openForRead(this.pathForSmallFile(fileName));){
                                ByteStreams.copy((InputStream)in, (OutputStream)out);
                            }
                            entries.put(fileName, zipName);
                            ++var9_14;
                        }
                    }
                    catch (Throwable throwable) {
                        object = throwable;
                        throw throwable;
                    }
                    finally {
                        if (out != null) {
                            if (object != null) {
                                try {
                                    out.close();
                                }
                                catch (Throwable throwable) {
                                    ((Throwable)object).addSuppressed(throwable);
                                }
                            } else {
                                out.close();
                            }
                        }
                    }
                    this.indexPut(entries);
                }
                catch (Throwable ex) {
                    try {
                        for (Map.Entry entry : entries.entrySet()) {
                            entry.setValue(null);
                        }
                        this.indexPut(entries);
                    }
                    catch (Throwable ex2) {
                        LOGGER.warn("Cannot unindex zip file after failure to generate it", ex2);
                    }
                    try {
                        if (opened) {
                            this.fileSystem.delete(zipPath, false);
                        }
                    }
                    catch (Throwable ex2) {
                        LOGGER.warn("Cannot delete zip file after failure to generate it", ex2);
                    }
                    throw new IOException("Cannot build and index zip file " + zipPath, ex);
                }
                LOGGER.debug("Merged {}/{} small files in ZIP file {}", new Object[]{this.numSmallFiles, this.numSmallFiles + mergeableNames.size(), zipPath});
            }
        }
        finally {
            this.lock.writeLock().unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void indexOptimize() throws IOException {
        IndexWriter indexWriter = this.luceneWriter;
        synchronized (indexWriter) {
            if (!this.luceneReader.isOptimized()) {
                this.luceneWriter.optimize();
                this.luceneWriter.commit();
                this.luceneReader.close();
                this.luceneReader = this.luceneWriter.getReader();
                LOGGER.debug("Index optimized");
            }
        }
    }

    private String indexGet(String key) throws IOException {
        Term s = new Term(KEY_FIELD, key);
        IndexWriter indexWriter = this.luceneWriter;
        synchronized (indexWriter) {
            int attempt = 0;
            while (true) {
                try {
                    TermDocs termDocs = this.luceneReader.termDocs(s);
                    if (termDocs.next()) {
                        Document doc = this.luceneReader.document(termDocs.doc());
                        return doc.get(VALUE_FIELD);
                    }
                    return null;
                }
                catch (Throwable ex) {
                    this.indexError(ex, attempt);
                    ++attempt;
                    continue;
                }
                break;
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void indexPut(Map<String, String> entries) throws IOException {
        try {
            int numDeleted = 0;
            int numUpdated = 0;
            IndexWriter indexWriter = this.luceneWriter;
            synchronized (indexWriter) {
                for (Map.Entry<String, String> entry : entries.entrySet()) {
                    if (entry.getValue() == null) {
                        this.luceneWriter.deleteDocuments(new Term(KEY_FIELD, entry.getKey()));
                        ++numDeleted;
                        continue;
                    }
                    Document doc = new Document();
                    doc.add((Fieldable)new Field(KEY_FIELD, entry.getKey(), Field.Store.YES, Field.Index.NOT_ANALYZED));
                    doc.add((Fieldable)new Field(VALUE_FIELD, entry.getValue(), Field.Store.YES, Field.Index.NOT_ANALYZED));
                    this.luceneWriter.updateDocument(new Term(KEY_FIELD, entry.getKey()), doc);
                    ++numUpdated;
                }
                this.luceneWriter.commit();
                this.luceneReader.close();
                this.luceneReader = this.luceneWriter.getReader();
            }
            LOGGER.debug("Updated Lucene index: {} documents updated, {} documents deleted", (Object)numUpdated, (Object)numDeleted);
        }
        catch (Throwable ex) {
            throw new IOException("Failed to update Lucene index with entries " + entries, ex);
        }
    }

    private Iterator<String> indexList(boolean deleted) throws IOException {
        if (deleted) {
            Term s = new Term(VALUE_FIELD, DELETED);
            ArrayList<String> deletedNames = new ArrayList<String>();
            IndexWriter indexWriter = this.luceneWriter;
            synchronized (indexWriter) {
                int attempt = 0;
                while (true) {
                    try {
                        TermDocs termDocs = this.luceneReader.termDocs(s);
                        while (termDocs.next()) {
                            deletedNames.add(this.luceneReader.document(termDocs.doc()).get(KEY_FIELD));
                        }
                        return deletedNames.iterator();
                    }
                    catch (Throwable ex) {
                        this.indexError(ex, attempt);
                        ++attempt;
                        continue;
                    }
                    break;
                }
            }
        }
        return new AbstractIterator<String>(){
            private int maxIndex = -1;
            private int currentIndex = 0;

            protected String computeNext() {
                try {
                    HadoopMultiFileStore store = HadoopMultiFileStore.this;
                    IndexWriter indexWriter = store.luceneWriter;
                    synchronized (indexWriter) {
                        int attempt = 0;
                        while (true) {
                            try {
                                if (this.maxIndex < 0) {
                                    this.maxIndex = store.luceneReader.maxDoc();
                                }
                                while (this.currentIndex <= this.maxIndex) {
                                    Document document = store.luceneReader.document(this.currentIndex++);
                                    if (document == null || HadoopMultiFileStore.DELETED.equals(document.get(HadoopMultiFileStore.VALUE_FIELD))) continue;
                                    return document.get(HadoopMultiFileStore.KEY_FIELD);
                                }
                                return (String)this.endOfData();
                            }
                            catch (Throwable ex) {
                                HadoopMultiFileStore.this.indexError(ex, attempt);
                                ++attempt;
                                continue;
                            }
                            break;
                        }
                    }
                }
                catch (Throwable ex) {
                    throw new RuntimeException("Error iterating over Lucene index", ex);
                }
            }
        };
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void indexError(Throwable ex, int numAttempt) throws IOException {
        if (numAttempt >= this.MAX_LUCENE_ATTEMPTS) {
            Throwables.propagateIfPossible((Throwable)ex, IOException.class);
            Throwables.propagate((Throwable)ex);
        }
        LOGGER.error("Error accessing Lucene index, will retry", ex);
        IndexWriter indexWriter = this.luceneWriter;
        synchronized (indexWriter) {
            try {
                this.luceneReader.close();
            }
            catch (Throwable ex2) {
                LOGGER.warn("Cannot close lucene reader after failure", ex2);
            }
            this.luceneReader = this.luceneWriter.getReader();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private InputStream openForRead(Path filePath) throws IOException {
        final String fileName = filePath.getName();
        FSDataInputStream stream = this.fileSystem.open(filePath);
        Multiset<String> multiset = this.openedFiles;
        synchronized (multiset) {
            this.openedFiles.add((Object)fileName);
        }
        LOGGER.trace("Opening {} for read", (Object)fileName);
        return new FilterInputStream((InputStream)stream){

            /*
             * WARNING - Removed try catching itself - possible behaviour change.
             */
            @Override
            public void close() throws IOException {
                try {
                    LOGGER.trace("Closing {}", (Object)fileName);
                    super.close();
                }
                finally {
                    Multiset multiset = HadoopMultiFileStore.this.openedFiles;
                    synchronized (multiset) {
                        HadoopMultiFileStore.this.openedFiles.remove((Object)fileName);
                    }
                }
            }
        };
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private OutputStream openForWrite(Path filePath) throws IOException {
        final String fileName = filePath.getName();
        FSDataOutputStream stream = this.fileSystem.create(filePath, true);
        Multiset<String> multiset = this.openedFiles;
        synchronized (multiset) {
            this.openedFiles.add((Object)fileName);
        }
        LOGGER.trace("Opening {} for write", (Object)fileName);
        return new FilterOutputStream((OutputStream)stream){

            /*
             * WARNING - Removed try catching itself - possible behaviour change.
             */
            @Override
            public void close() throws IOException {
                try {
                    LOGGER.trace("Closing {}", (Object)fileName);
                    super.close();
                }
                finally {
                    Multiset multiset = HadoopMultiFileStore.this.openedFiles;
                    synchronized (multiset) {
                        HadoopMultiFileStore.this.openedFiles.remove((Object)fileName);
                    }
                }
            }
        };
    }

    @Nullable
    private Path pathForSmallFile(@Nullable String smallFile) {
        return smallFile == null ? null : new Path(this.smallFilesPath, smallFile);
    }

    @Nullable
    private Path pathForZipFile(@Nullable String zipFile) {
        if (zipFile == null) {
            return null;
        }
        String bucketDirectory = zipFile.substring(0, 2);
        return new Path(this.rootPath, bucketDirectory + File.separator + zipFile);
    }
}

