/*
 * Decompiled with CFR 0.152.
 */
package eu.fbk.knowledgestore.filestore;

import com.google.common.base.MoreObjects;
import com.google.common.base.Preconditions;
import com.google.common.base.Throwables;
import com.google.common.collect.AbstractIterator;
import eu.fbk.knowledgestore.data.Data;
import eu.fbk.knowledgestore.data.Stream;
import eu.fbk.knowledgestore.filestore.FileExistsException;
import eu.fbk.knowledgestore.filestore.FileMissingException;
import eu.fbk.knowledgestore.filestore.FileStore;
import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FilterOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.Collections;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import java.util.zip.ZipEntry;
import java.util.zip.ZipInputStream;
import java.util.zip.ZipOutputStream;
import javax.annotation.Nullable;
import org.apache.hadoop.fs.FSDataInputStream;
import org.apache.hadoop.fs.FSDataOutputStream;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.lucene.analysis.Analyzer;
import org.apache.lucene.analysis.WhitespaceAnalyzer;
import org.apache.lucene.document.Document;
import org.apache.lucene.document.Field;
import org.apache.lucene.document.Fieldable;
import org.apache.lucene.index.IndexReader;
import org.apache.lucene.index.IndexWriter;
import org.apache.lucene.index.Term;
import org.apache.lucene.index.TermDocs;
import org.apache.lucene.store.Directory;
import org.apache.lucene.store.FSDirectory;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class HadoopMultiFileStore
implements FileStore {
    private static final Logger LOGGER = LoggerFactory.getLogger(HadoopMultiFileStore.class);
    private static final String DEFAULT_PATH = "files";
    private static final String DEFAULT_LUCENE_PATH = "./lucene-index";
    private static final String SMALL_FILES_PATH = "_small";
    private final String FILENAME_FIELD_NAME = "filename";
    private final String ZIP_FIELD_NAME = "zipfilename";
    private final FileSystem fileSystem;
    private final int MAX_LUCENE_SEGMENTS = 100;
    private final Path rootPath;
    private final Path smallFilesPath;
    private final File luceneFolder;
    private int MAX_NUM_SMALL_FILES = 10;
    private Set<String> filesInWritingMode = Collections.synchronizedSet(new HashSet());
    private final AtomicBoolean isWritingBigFile = new AtomicBoolean(false);
    private IndexReader luceneReader;
    private IndexWriter luceneWriter;
    private final Lock writeBigFileLock;

    public HadoopMultiFileStore(FileSystem fileSystem, @Nullable String lucenePath, @Nullable String path, @Nullable Integer numSmallFile) {
        if (numSmallFile != null) {
            this.MAX_NUM_SMALL_FILES = numSmallFile;
        }
        this.fileSystem = (FileSystem)Preconditions.checkNotNull((Object)fileSystem);
        this.rootPath = new Path((String)MoreObjects.firstNonNull((Object)path, (Object)DEFAULT_PATH)).makeQualified(this.fileSystem);
        this.luceneFolder = new File((String)MoreObjects.firstNonNull((Object)lucenePath, (Object)DEFAULT_LUCENE_PATH));
        this.smallFilesPath = new Path(this.rootPath.toString() + File.separator + SMALL_FILES_PATH).makeQualified(this.fileSystem);
        this.writeBigFileLock = new ReentrantLock();
        LOGGER.info("{} configured, paths={};{}", new Object[]{this.getClass().getSimpleName(), this.rootPath, this.luceneFolder});
    }

    public InputStream readGenericFile(Path path) throws IOException {
        try {
            FSDataInputStream stream = this.fileSystem.open(path);
            LOGGER.debug("Reading file {}", (Object)path.getName());
            return stream;
        }
        catch (IOException ex) {
            if (!this.fileSystem.exists(path)) {
                throw new FileMissingException(path.getName(), "Cannot read non-existing file");
            }
            throw ex;
        }
    }

    @Override
    public void init() throws IOException {
        if (!this.fileSystem.exists(this.rootPath)) {
            LOGGER.debug("Created root folder");
            this.fileSystem.mkdirs(this.rootPath);
        }
        if (!this.fileSystem.exists(this.smallFilesPath)) {
            LOGGER.debug("Created small files folder");
            this.fileSystem.mkdirs(this.smallFilesPath);
        }
        if (!this.luceneFolder.exists()) {
            LOGGER.debug("Created lucene folder");
            if (!this.luceneFolder.mkdirs()) {
                throw new IOException(String.format("Unable to create dir %s", this.luceneFolder.toString()));
            }
        }
        this.luceneWriter = new IndexWriter((Directory)FSDirectory.open((File)this.luceneFolder), (Analyzer)new WhitespaceAnalyzer(), IndexWriter.MaxFieldLength.LIMITED);
        this.luceneReader = this.luceneWriter.getReader();
    }

    private void deleteFromBigFile(String fileName, Path zipFile) throws IOException {
        ZipEntry entry;
        LOGGER.debug("Deleting zip file {}", (Object)zipFile.getName());
        ZipInputStream zipInputStream = new ZipInputStream(this.readGenericFile(zipFile));
        byte[] buffer = new byte[2048];
        HashSet<String> listOfFiles = new HashSet<String>();
        while ((entry = zipInputStream.getNextEntry()) != null) {
            int len;
            String entryName = entry.getName();
            if (entryName.equals(fileName)) continue;
            Path path = this.getSmallPath(entryName);
            this.filesInWritingMode.add(entryName);
            listOfFiles.add(entryName);
            Object stream = this.fileSystem.create(path, false);
            stream = new InterceptCloseOutputStream((OutputStream)stream, entryName);
            while ((len = zipInputStream.read(buffer)) > 0) {
                ((OutputStream)stream).write(buffer, 0, len);
            }
            ((OutputStream)stream).close();
        }
        zipInputStream.close();
        this.fileSystem.delete(zipFile, false);
        Term s = new Term("zipfilename", zipFile.getName());
        this.luceneWriter.deleteDocuments(s);
        this.luceneWriter.commit();
        this.luceneReader.close();
        this.luceneReader = this.luceneWriter.getReader();
        this.filesInWritingMode.removeAll(listOfFiles);
        LOGGER.debug("Finishing deletion of zip file {}", (Object)zipFile.getName());
        this.checkSmallFilesAndMerge();
    }

    private InputStream loadFromBigFile(String fileName, Path zipFile) throws IOException {
        ZipEntry entry;
        ZipInputStream stream = new ZipInputStream(this.readGenericFile(zipFile));
        while ((entry = stream.getNextEntry()) != null) {
            String entryName = entry.getName();
            if (!entryName.equals(fileName)) continue;
            return stream;
        }
        throw new FileMissingException(fileName, "The file does not exist");
    }

    @Override
    public InputStream read(String fileName) throws IOException {
        this.optimizeOnDemand();
        Path path = this.getSmallPath(fileName);
        if (this.fileSystem.exists(path)) {
            LOGGER.debug("It is a small file");
            return this.readGenericFile(path);
        }
        Term s = new Term("filename", fileName);
        TermDocs termDocs = this.luceneReader.termDocs(s);
        if (termDocs.next()) {
            Document doc = this.luceneReader.document(termDocs.doc());
            String zipFile = doc.get("zipfilename");
            Path inputFile = this.getFolderFromBigFile(zipFile);
            LOGGER.debug("The zip file is {}", (Object)inputFile.getName());
            return this.loadFromBigFile(fileName, inputFile);
        }
        throw new FileMissingException(fileName, "The file does not exist");
    }

    @Override
    public OutputStream write(String fileName) throws IOException {
        Path path = this.getSmallPath(fileName);
        if (this.fileSystem.exists(path)) {
            throw new FileExistsException(path.getName(), "Cannot overwrite file");
        }
        Term s = new Term("filename", fileName);
        TermDocs termDocs = this.luceneReader.termDocs(s);
        if (termDocs.next()) {
            throw new FileExistsException(path.getName(), "Cannot overwrite file");
        }
        this.checkSmallFilesAndMerge();
        LOGGER.debug("Creating file {}", (Object)path.getName());
        FSDataOutputStream stream = this.fileSystem.create(path, false);
        this.filesInWritingMode.add(fileName);
        return new InterceptCloseOutputStream((OutputStream)stream, fileName);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     * Enabled force condition propagation
     * Lifted jumps to return sites
     */
    @Override
    public void delete(String fileName) throws FileMissingException, IOException {
        this.writeBigFileLock.lock();
        try {
            AtomicBoolean atomicBoolean = this.isWritingBigFile;
            synchronized (atomicBoolean) {
                Path path = this.getSmallPath(fileName);
                if (this.fileSystem.exists(path)) {
                    this.fileSystem.delete(path, false);
                    LOGGER.debug("Deleted file {}", (Object)path.getName());
                    return;
                }
                Term s = new Term("filename", fileName);
                TermDocs termDocs = this.luceneReader.termDocs(s);
                if (!termDocs.next()) throw new FileMissingException(fileName, "The file does not exist");
                Document doc = this.luceneReader.document(termDocs.doc());
                String zipFile = doc.get("zipfilename");
                Path inputFile = this.getFolderFromBigFile(zipFile);
                LOGGER.debug("The zip file is {}", (Object)inputFile.getName());
                this.deleteFromBigFile(fileName, inputFile);
                return;
            }
        }
        finally {
            this.writeBigFileLock.unlock();
        }
    }

    private synchronized void optimizeOnDemand() throws IOException {
        if (!this.luceneReader.isOptimized()) {
            LOGGER.debug("Optimizing index");
            this.luceneWriter.optimize(100);
            this.luceneReader.close();
            this.luceneReader = this.luceneWriter.getReader();
        } else {
            LOGGER.debug("Index is optimized");
        }
    }

    @Override
    public Stream<String> list() throws IOException {
        this.optimizeOnDemand();
        return Stream.concat((Iterable[])new Iterable[]{Stream.create((Iterable)new LuceneIterator(this.luceneReader)), Stream.create((Iterator)((Object)new HadoopIterator(this.smallFilesPath, this.fileSystem)))});
    }

    @Override
    public void close() {
        this.writeBigFileLock.lock();
        this.writeBigFileLock.unlock();
        try {
            this.luceneReader.close();
        }
        catch (Exception e) {
            LOGGER.warn("Unable to close Lucene reader");
        }
        try {
            this.luceneWriter.optimize();
        }
        catch (Exception e) {
            LOGGER.warn("Unable to optimize Lucene writer");
        }
        try {
            this.luceneWriter.close();
        }
        catch (Exception e) {
            LOGGER.warn("Unable to close Lucene writer");
        }
    }

    public String toString() {
        return this.getClass().getSimpleName();
    }

    private Path getFolderFromBigFile(String fileName) {
        String bucketDirectory = fileName.substring(0, 2);
        return new Path(this.rootPath, bucketDirectory + File.separator + fileName);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void checkSmallFilesAndMerge() throws IOException {
        FileStatus[] files;
        boolean mustMerge = false;
        LOGGER.debug("Checking small files");
        AtomicBoolean atomicBoolean = this.isWritingBigFile;
        synchronized (atomicBoolean) {
            if (!this.isWritingBigFile.get()) {
                files = this.fileSystem.listStatus(this.smallFilesPath);
                if (files == null) {
                    return;
                }
                LOGGER.debug("Number of files: {}", (Object)files.length);
                if (files.length > this.MAX_NUM_SMALL_FILES) {
                    mustMerge = true;
                    this.isWritingBigFile.set(true);
                    this.writeBigFileLock.lock();
                }
            } else {
                return;
            }
        }
        if (mustMerge) {
            try {
                LOGGER.debug("More than {} small files, building zip", (Object)this.MAX_NUM_SMALL_FILES);
                LOGGER.debug(this.filesInWritingMode.toString());
                FileStatus[] list = new FileStatus[this.MAX_NUM_SMALL_FILES];
                StringBuffer stringBuffer = new StringBuffer();
                int i = 0;
                for (FileStatus fs : files) {
                    if (fs.isDir() || this.filesInWritingMode.contains(fs.getPath().getName())) continue;
                    if (++i > this.MAX_NUM_SMALL_FILES) break;
                    LOGGER.debug("{} - {}", (Object)(i - 1), (Object)fs.getPath().getName());
                    stringBuffer.append(fs.toString());
                    list[i - 1] = fs;
                }
                if (list[list.length - 1] == null) {
                    throw new IOException("Not enough files, skipping");
                }
                String fileName = Data.hash((Object[])new Object[]{stringBuffer.toString()});
                Path outputFile = this.getFolderFromBigFile(fileName);
                Data.getExecutor().schedule((Runnable)new SaveBigFile(this.fileSystem, list, outputFile), 0L, TimeUnit.MILLISECONDS);
            }
            catch (Exception e) {
                this.writeBigFileLock.unlock();
                this.isWritingBigFile.set(false);
                LOGGER.debug(e.getMessage());
            }
        }
    }

    private Path getSmallPath(String fileName) {
        return new Path(this.smallFilesPath, fileName);
    }

    private int numSmallFiles() throws IOException {
        return this.fileSystem.listStatus(this.smallFilesPath).length;
    }

    private class HadoopIterator
    extends AbstractIterator<String> {
        private FileStatus[] files;
        private int fileIndex = 0;

        HadoopIterator(Path path, FileSystem fileSystem) throws IOException {
            this.files = fileSystem.listStatus(path);
        }

        protected String computeNext() {
            try {
                FileStatus file;
                if (this.fileIndex < this.files.length && !(file = this.files[this.fileIndex++]).isDir()) {
                    return file.getPath().getName();
                }
                return (String)this.endOfData();
            }
            catch (Throwable ex) {
                throw Throwables.propagate((Throwable)ex);
            }
        }
    }

    private class LuceneIterator
    implements Iterable<String> {
        private IndexReader reader;

        private LuceneIterator(IndexReader reader) {
            this.reader = reader;
            LOGGER.debug("Reader loaded! Num files: {}", (Object)this.reader.numDocs());
        }

        @Override
        public Iterator<String> iterator() {
            return new Iterator<String>(){
                private int currentIndex = 0;

                @Override
                public boolean hasNext() {
                    try {
                        Document document = LuceneIterator.this.reader.document(this.currentIndex);
                        if (document != null) {
                            return true;
                        }
                    }
                    catch (Exception e) {
                        return false;
                    }
                    return false;
                }

                @Override
                public String next() {
                    try {
                        Document document = LuceneIterator.this.reader.document(this.currentIndex++);
                        if (document != null) {
                            return document.get("filename");
                        }
                    }
                    catch (Exception e) {
                        return null;
                    }
                    return null;
                }

                @Override
                public void remove() {
                }
            };
        }
    }

    private class InterceptCloseOutputStream
    extends FilterOutputStream {
        private String fileName;

        private InterceptCloseOutputStream(OutputStream out, String fileName) {
            super(out);
            this.fileName = fileName;
        }

        @Override
        public void close() throws IOException {
            try {
                LOGGER.debug("Closing {}", (Object)this.fileName);
                super.close();
            }
            finally {
                HadoopMultiFileStore.this.filesInWritingMode.remove(this.fileName);
            }
        }
    }

    private class SaveBigFile
    implements Runnable {
        private FileSystem fileSystem;
        private FileStatus[] files;
        private Path outputFile;

        private SaveBigFile(FileSystem fileSystem, FileStatus[] files, Path outputFile) {
            this.fileSystem = fileSystem;
            this.files = files;
            this.outputFile = outputFile;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public void run() {
            try {
                ZipOutputStream out;
                try {
                    LOGGER.debug("Opening ZIP file {}", (Object)this.outputFile.getName());
                    out = new ZipOutputStream(new BufferedOutputStream((OutputStream)this.fileSystem.create(this.outputFile)));
                }
                catch (Exception e) {
                    LOGGER.debug("Unable to open ZIP file");
                    HadoopMultiFileStore.this.writeBigFileLock.unlock();
                    HadoopMultiFileStore.this.isWritingBigFile.set(false);
                    return;
                }
                byte[] data = new byte[1024];
                for (FileStatus f : this.files) {
                    try {
                        int count;
                        BufferedInputStream in = new BufferedInputStream((InputStream)this.fileSystem.open(f.getPath()));
                        out.putNextEntry(new ZipEntry(f.getPath().getName()));
                        while ((count = in.read(data, 0, 1000)) != -1) {
                            out.write(data, 0, count);
                        }
                        in.close();
                    }
                    catch (Exception e) {
                        LOGGER.warn("Unable to write to ZIP file {}", (Object)this.outputFile.getName(), (Object)e);
                    }
                }
                try {
                    LOGGER.debug("Closing ZIP file {}", (Object)this.outputFile.getName());
                    out.flush();
                    out.close();
                }
                catch (Exception e) {
                    LOGGER.warn("Unable to close ZIP file {}", (Object)this.outputFile.getName(), (Object)e);
                }
                for (FileStatus f : this.files) {
                    LOGGER.debug("Adding file {} to index", (Object)f.getPath().getName());
                    Document doc = new Document();
                    doc.add((Fieldable)new Field("filename", f.getPath().getName(), Field.Store.YES, Field.Index.NOT_ANALYZED));
                    doc.add((Fieldable)new Field("zipfilename", this.outputFile.getName(), Field.Store.YES, Field.Index.NOT_ANALYZED));
                    try {
                        HadoopMultiFileStore.this.luceneWriter.updateDocument(new Term("filename", f.getPath().getName()), doc);
                    }
                    catch (Exception e) {
                        LOGGER.warn("Error in writing file {} to Lucene index", (Object)f.getPath().getName(), (Object)e);
                    }
                }
                try {
                    HadoopMultiFileStore.this.luceneWriter.commit();
                }
                catch (Exception e) {
                    // empty catch block
                }
                for (FileStatus f : this.files) {
                    try {
                        LOGGER.debug("Deleting file {}", (Object)f.getPath().getName());
                        this.fileSystem.delete(f.getPath(), false);
                    }
                    catch (Exception e) {
                        LOGGER.warn("Unable to delete file {}", (Object)f.getPath().getName());
                    }
                }
                try {
                    HadoopMultiFileStore.this.checkSmallFilesAndMerge();
                }
                catch (Exception e) {
                    // empty catch block
                }
            }
            finally {
                HadoopMultiFileStore.this.writeBigFileLock.unlock();
                HadoopMultiFileStore.this.isWritingBigFile.set(false);
            }
        }
    }
}

