/**
 * Copyright (C) 2013 FIspace (info@fispace.eu)
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package eu.fispace.api;

import eu.fispace.api.ag.*;
import eu.limetri.ygg.api.CapabilityType;
import eu.limetri.ygg.api.CapabilityTypeRegistration;
import eu.limetri.ygg.api.Messages;

/**
 *
 * @author Timon Veenstra <timon@limetri.eu>
 */
public class DomainAgriculture {

    public static final String DOMAIN = "ag";
    public static final String SCHEMA = "classpath:/schema/domain/ag/AGMessages.xsd";
    public static final String CONTEXT_PATH = "eu.fispace.api.ag";

    /**
     * Capability to produce phytophthora advice
     *
     * @since 0.5.0
     * @see PhythophthoraAdviceMessage
     */
    @CapabilityTypeRegistration
    public static final CapabilityType PRODUCE_PHYTOPHTHORA_ADVICE = new CapabilityType()
            .withName("phytophthora advice")
            .withSchemaLocation(SCHEMA)
            .withContextPath(CONTEXT_PATH)
            .withResponseMessageType(PhythophthoraAdviceMessage.class.getSimpleName());

    /**
     * Capability to produce cropfield details
     *
     * @since 0.5.0
     * @see CropFieldDetailMessage
     */
    @CapabilityTypeRegistration
    public static final CapabilityType PRODUCE_CROPFIELD_DETAILS = new CapabilityType()
            .withName("cropfield details")
            .withSchemaLocation(SCHEMA)
            .withContextPath(CONTEXT_PATH)
            .withResponseMessageType(CropFieldDetailMessage.class.getSimpleName());

    /**
     * Capability to receive sensor values
     * 
     * Example: FIspace provides this capabilty so that an external system can send sensor values for event detection
     * 
     * @since 0.10.1
     * @see ReceiveSensorValuesRequestMessage
     */
    @CapabilityTypeRegistration
    public static final CapabilityType RECEIVE_SENSOR_VALUES = new CapabilityType()
            .withName("receive sensor values")
            .withSchemaLocation(SCHEMA)
            .withContextPath(CONTEXT_PATH)
            .withRequestMessageType(ReceiveSensorValuesRequestMessage.class.getSimpleName())
            .withResponseMessageType(Messages.Untyped.class.getSimpleName());

    /**
     * Capability to receive an "advice available notification"
     * 
     * Example: a farm management system, that is requesting advices from experts, could provide 
     * this capability so that it can be informed by an expert system as soon as the advice that was 
     * requested by the farmer (via the capability @PROVIDE_ADVICE) is ready.
     * 
     * @since 0.11.0
     * @see AdviceAvailableNotification
     */
    @CapabilityTypeRegistration
    public static final CapabilityType RECEIVE_ADVICE_AVAILABLE_NOTIFICATION = new CapabilityType()
            .withName("RECEIVE_ADVICE_AVAILABLE_NOTIFICATION")
            .withSchemaLocation(SCHEMA)
            .withContextPath(CONTEXT_PATH)
            .withRequestMessageType(AdviceAvailableNotification.class.getSimpleName())
            .withResponseMessageType(Messages.Untyped.class.getSimpleName());

    /**
     *
     * @since 0.7.2
     * @see CropFieldAdviceRequestMessage
     */
    @CapabilityTypeRegistration
    public static final CapabilityType RECEIVE_ADVICE_REQUEST = new CapabilityType()
            .withName("greenhouse advice")
            .withSchemaLocation(SCHEMA)
            .withContextPath(CONTEXT_PATH)
            .withRequestMessageType(CropFieldAdviceRequestMessage.class.getSimpleName())
            .withResponseMessageType(Messages.Untyped.class.getSimpleName());
    
     /**
     * Capability to provide sensor values on request
     * 
     * Example: a farmer could provide this capability, so that business partners 
     * could use this capability to get sensor values from the farmer's system
     *
     * @since 0.10.1
     * @see ProvideSensorValuesRequestMessage
     * @see ProvideSensorValuesResponseMessage
     */
    @CapabilityTypeRegistration
    public static final CapabilityType PROVIDE_SENSOR_VALUES = new CapabilityType()
            .withName("provide sensor values")	
            .withSchemaLocation(SCHEMA)
            .withContextPath(CONTEXT_PATH)
            .withRequestMessageType(ProvideSensorValuesRequestMessage.class.getSimpleName())
            .withResponseMessageType(ProvideSensorValuesResponseMessage.class.getSimpleName());

    
    /**
     * Capability to be implemented by a system that is capable of providing an (expert) advice.
     * 
     * Example: an expert system could provice this capability, then a farm management system could use 
     * this capability to get an advice from the expert system
     *
     * @since 0.10.1
     * @see ProvideAdviceRequestMessage
     * @see ProvideAdviceResponseMessage
     */
    @CapabilityTypeRegistration
    public static final CapabilityType PROVIDE_ADVICE = new CapabilityType()
            .withName("provide advice")	
            .withSchemaLocation(SCHEMA)
            .withContextPath(CONTEXT_PATH)
            .withRequestMessageType(ProvideAdviceRequestMessage.class.getSimpleName())
            .withResponseMessageType(ProvideAdviceResponseMessage.class.getSimpleName());

    @CapabilityTypeRegistration
    public static final CapabilityType PROVIDE_TRACEABILITY = new CapabilityType()
            .withName("provide traceability")
            .withSchemaLocation(SCHEMA)
            .withContextPath(CONTEXT_PATH)
            .withRequestMessageType(BasicTraceabilityInformationRequest.class.getSimpleName())
            .withResponseMessageType(BasicTraceabilityInformationResponse.class.getSimpleName());

    
    /**
     * Capability for providing a weather scenario (a set of weather 
     * predictions for given locations and time interval).
     * The message use the drmCrop message format.
     *
     * @since 0.12.2
     * @see WeatherScenarioRequestMessage
     * @see WeatherScenarioResponseMessage
     */
    @CapabilityTypeRegistration
    public static final CapabilityType PROVIDE_WEATHER_SCENARIO = new CapabilityType()
            .withName("provide weather scenario")	
            .withSchemaLocation(SCHEMA)
            .withContextPath(CONTEXT_PATH)
            .withRequestMessageType(WeatherScenarioRequestMessage.class.getSimpleName())
            .withResponseMessageType(WeatherScenarioResponseMessage.class.getSimpleName());

    /**
     * Capability for providing a weather scenario (a set of weather 
     * predictions for given locations and time interval).
     * The message use the simplified, non-drmCrop message format.
     *
     * @since 0.14.0
     * @see WeatherScenarioSimpleRequestMessage
     * @see WeatherScenarioSimpleResponseMessage
     */
    @CapabilityTypeRegistration
    public static final CapabilityType PROVIDE_WEATHER_SCENARIO_SIMPLE = new CapabilityType()
            .withName("provide weather scenario, simple message format")	
            .withSchemaLocation(SCHEMA)
            .withContextPath(CONTEXT_PATH)
            .withRequestMessageType(WeatherScenarioSimpleRequestMessage.class.getSimpleName())
            .withResponseMessageType(WeatherScenarioSimpleResponseMessage.class.getSimpleName());

    
    /**
     * This capability is to be provided by a "weather conditions provider" to provide weather information to a client.
     * 
     * Expected behaviour of the system implementing this capability:
     * - if a request for weather info cannot be fulfilled at once, the answer should be HTTP 204
     * - if a request for weather info can be fulfilled at once, the answer should be HTTP 303, including a URL where the actual weather info can be fetched
     * - if the URL returned by a previous 303 answer is used to call this capability, the answer should be the actual ProvideWeatherConditionsResponseMessage
     * 
     * @since 0.10.0
     * @see ProvideWeatherConditionsRequestMessage
     * @see ProvideWeatherConditionsResponseMessage
     */
    @CapabilityTypeRegistration
    public static final CapabilityType PROVIDE_WEATHER_CONDITIONS = new CapabilityType()
            .withName("PROVIDE_WEATHER_CONDITIONS")
            .withSchemaLocation(SCHEMA)
            .withContextPath(CONTEXT_PATH)
            .withRequestMessageType(ProvideWeatherConditionsRequestMessage.class.getSimpleName())
            .withResponseMessageType(ProvideWeatherConditionsResponseMessage.class.getSimpleName());
    
    /**
     * This capability is to be provided by a "Bad weather module" to provide workability Data to a client.
     * 
     * Expected behaviour of the system implementing this capability:
     * - if a request for weather info cannot be fulfilled at once, the answer should be HTTP 204
     * - if a request for weather info can be fulfilled at once, the answer should be HTTP 303, including a URL where the actual weather info can be fetched
     * - if the URL returned by a previous 303 answer is used to call this capability, the answer should be the actual ProvideWeatherConditionsResponseMessage
     * 
     * @since 0.10.0
     * @see WorkabilityDataRequestMessage
     * @see WorkabilityDataResponseMessage
     */
    @CapabilityTypeRegistration
    public static final CapabilityType PROVIDE_WORKABILITY_DATA = new CapabilityType()
            .withName("PROVIDE_WORKABILITY_DATA")
            .withSchemaLocation(SCHEMA)
            .withContextPath(CONTEXT_PATH)
            .withRequestMessageType(WorkabilityDataRequestMessage.class.getSimpleName())
            .withResponseMessageType(WorkabilityDataResponseMessage.class.getSimpleName());

    /**
     *
     * Capability to search for farmers to external marketplace of OPEKEPE.This capability
     * is used by Marketplace Operations Service.
     * @see SearchForFarmersRequest
     * @see SearchForFarmersResponse
     */
    @CapabilityTypeRegistration
    public static final CapabilityType SEARCH_FOR_FARMERS = new CapabilityType()
            .withName("SEARCH_FOR_FARMERS")
            .withSchemaLocation(SCHEMA)
            .withContextPath(CONTEXT_PATH)
            .withRequestMessageType(SearchForFarmersRequest.class.getSimpleName())
            .withResponseMessageType(SearchForFarmersResponse.class.getSimpleName());

}
