/**
 * Copyright (C) 2013 FIspace (info@fispace.eu)
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package eu.fispace.api;

import eu.fispace.api.ag.*;
import eu.limetri.ygg.api.CapabilityType;
import eu.limetri.ygg.api.CapabilityTypeRegistration;
import eu.limetri.ygg.api.Messages;

/**
 *
 * @author Timon Veenstra <timon@limetri.eu>
 */
public class DomainAgriculture {

    public static final String DOMAIN = "ag";
    public static final String SCHEMA = "classpath:/schema/domain/ag/AGMessages.xsd";
    public static final String CONTEXT_PATH = "eu.fispace.api.ag";

    /**
     * Capability to produce ActivityField data
     *
     * @since 0.16.0
     * @see ActivityFieldDataRequest and -Response
     */
    @CapabilityTypeRegistration
    public static final CapabilityType PROVIDE_ACTIVITY_FIELD_DATA = new CapabilityType()
            .withName("activity field data")
            .withSchemaLocation(SCHEMA)
            .withContextPath(CONTEXT_PATH)
            .withRequestMessageType(ActivityFieldDataRequest.class.getSimpleName())
            .withResponseMessageType(ActivityFieldDataResponse.class.getSimpleName());

    /**
     * Capability to produce ActivityField data
     *
     * @since 0.16.0
     * @see ActivityFieldLocationRequest and -Response
     */
    @CapabilityTypeRegistration
    public static final CapabilityType PROVIDE_ACTIVITY_FIELD_LOCATION = new CapabilityType()
            .withName("activity field location")
            .withSchemaLocation(SCHEMA)
            .withContextPath(CONTEXT_PATH)
            .withRequestMessageType(ActivityFieldLocationRequest.class.getSimpleName())
            .withResponseMessageType(ActivityFieldLocationResponse.class.getSimpleName());

    /**
     * Capability to be implemented by a system that is capable of providing additional information to an expert system.
     * 
     * Example: an farmer management system could use this capability to send additional information for a plant diagnosis
     * to expert system
     *
     * @since 0.10.1
     * @see AdditionalInformationResponse
     */
    @CapabilityTypeRegistration
    public static final CapabilityType PROVIDE_ADDITIONAL_INFORMATION_RESPONSE = new CapabilityType()
        .withName("PROVIDE_ADDITIONAL_INFORMATION_RESPONSE")
        .withSchemaLocation(SCHEMA)
        .withContextPath(CONTEXT_PATH)
        .withRequestMessageType(AdditionalInformationResponse.class.getSimpleName());
    
    /**
     * This capability is to be provided by a "Additional information request module" to provide additional information data to a client.
     * 
     * Expected behaviour of the system implementing this capability:
     * - if a request for plant diagnosis request data cannot be fulfilled at once, the answer should be HTTP 204
     * - if a request for plant diagnosis request data can be fulfilled at once, the answer should be HTTP 303, including a URL where the actual diagnosis request data can be fetched
     * - if the URL returned by a previous 303 answer is used to call this capability, the answer should be the actual PlantDiagnosisRequestDataResponseMessage
     * 
     * @since 0.10.0
     * @see AdditionalInformationResponseDataRequestInfo
     * @see AdditionalInformationResponseDataResponse
     */
    @CapabilityTypeRegistration
    public static final CapabilityType PROVIDE_ADDITIONAL_INFORMATION_RESPONSE_DATA = new CapabilityType()
            .withName("PROVIDE_ADDITIONAL_INFORMATION_RESPONSE_DATA")
            .withSchemaLocation(SCHEMA)
            .withContextPath(CONTEXT_PATH)
            .withRequestMessageType(AdditionalInformationResponseDataRequest.class.getSimpleName())
            .withResponseMessageType(AdditionalInformationResponseDataResponse.class.getSimpleName());

    /**
     * Capability to be implemented by a system that is capable of providing an (expert) advice.
     * 
     * Example: an expert system could provide this capability, then a farm management system could use 
     * this capability to get an advice from the expert system
     *
     * @since 0.10.1
     * @see AdviceRequest
     * @see AdviceResponse
     */
    @CapabilityTypeRegistration
    public static final CapabilityType PROVIDE_ADVICE = new CapabilityType()
            .withName("provide advice")	
            .withSchemaLocation(SCHEMA)
            .withContextPath(CONTEXT_PATH)
            .withRequestMessageType(AdviceRequest.class.getSimpleName())
            .withResponseMessageType(AdviceResponse.class.getSimpleName());

   @CapabilityTypeRegistration
    public static final CapabilityType PROVIDE_COMPLAINT = new CapabilityType()
            .withName("provide complaint")	
            .withSchemaLocation(SCHEMA)
            .withContextPath(CONTEXT_PATH)
            .withRequestMessageType(Complaint.class.getSimpleName())
            .withResponseMessageType(ComplaintAnalysisResult.class.getSimpleName());

    /**
     * Capability to produce CropField data
     *
     * @since 0.16
     * @see CropFieldDataWFPARequest and -Response
     * */
    @CapabilityTypeRegistration
    public static final CapabilityType PROVIDE_CROPFIELD_DATA_WFPA = new CapabilityType()
            .withName("cropfield data wfpa")
            .withSchemaLocation(SCHEMA)
            .withContextPath(CONTEXT_PATH)
            .withRequestMessageType(CropFieldDataWFPARequest.class.getSimpleName())    
            .withResponseMessageType(CropFieldDataWFPAResponse.class.getSimpleName());     

    /**
     * Capability to be implemented by a system that is capable of asking further advice on an plant diagnosis.
     * 
     * Example: a crop manager management system could use this capability to send a further advice request
     * to consulting firm expert system
     *
     * @since 0.16.0
     * @see FurtherAdviceRequestInfo
     */
    @CapabilityTypeRegistration
    public static final CapabilityType PROVIDE_FURTHER_ADVICE_REQUEST = new CapabilityType()
        .withName("PROVIDE_FURTHER_ADVICE_REQUEST")
        .withSchemaLocation(SCHEMA)
        .withContextPath(CONTEXT_PATH)
        .withRequestMessageType(FurtherAdviceRequest.class.getSimpleName());
    
    /**
     * This capability is to be provided by a "Further advice request module" to provide further advice request data to a client.
     * 
     * Expected behavior of the system implementing this capability:
     * - if a request for plant diagnosis request data cannot be fulfilled at once, the answer should be HTTP 204
     * - if a request for plant diagnosis request data can be fulfilled at once, the answer should be HTTP 303, including a URL where the actual diagnosis request data can be fetched
     * - if the URL returned by a previous 303 answer is used to call this capability, the answer should be the actual PlantDiagnosisRequestDataResponseMessage
     * 
     * @since 0.16.0
     * @see FurtherAdviceRequestDataRequestInfo
     * @see FurtherAdviceRequestDataResponse
     */
    @CapabilityTypeRegistration
    public static final CapabilityType PROVIDE_FURTHER_ADVICE_REQUEST_DATA = new CapabilityType()
        .withName("PROVIDE_FURTHER_ADVICE_REQUEST_DATA")
        .withSchemaLocation(SCHEMA)
        .withContextPath(CONTEXT_PATH)
        .withRequestMessageType(FurtherAdviceRequestDataRequest.class.getSimpleName())
        .withResponseMessageType(FurtherAdviceRequestDataResponse.class.getSimpleName());

    /**
     * Capability to be implemented by a system that is capable of providing an 
     * (expert) plant diagnosis.
     * 
     * Example: a grower management system could use this capability to send an 
     * plant diagnosis to expert system
     *
     * @since 0.16
     * @see PlantDiagnosisRequestInfo
     */
    @CapabilityTypeRegistration
    public static final CapabilityType PLANT_DIAGNOSIS_REQUEST = new CapabilityType()
        .withName("PLANT_DIAGNOSIS_REQUEST")
        .withSchemaLocation(SCHEMA)
        .withContextPath(CONTEXT_PATH)
        .withRequestMessageType(PlantDiagnosisRequest.class.getSimpleName());
    /**
     * This capability is to be provided by a "Plant diagnosis request module" to provide plant diagnosis request data to a client.
     * 
     * Expected behaviour of the system implementing this capability:
     * - if a request for plant diagnosis request data cannot be fulfilled at once, the answer should be HTTP 204
     * - if a request for plant diagnosis request data can be fulfilled at once, the answer should be HTTP 303, including a URL where the actual diagnosis request data can be fetched
     * - if the URL returned by a previous 303 answer is used to call this capability, the answer should be the actual PlantDiagnosisRequestDataResponseMessage
     * 
     * @since 0.16
     * @see PlantDiagnosisRequestDataRequestMessage
     * @see PlantDiagnosisRequestDataResponseMessage
     */
    @CapabilityTypeRegistration
    public static final CapabilityType PROVIDE_PLANT_DIAGNOSIS_REQUEST_DATA = new CapabilityType()
            .withName("PROVIDE_PLANT_DIAGNOSIS_REQUEST_DATA")
            .withSchemaLocation(SCHEMA)
            .withContextPath(CONTEXT_PATH)
            .withRequestMessageType(PlantDiagnosisRequestDataRequest.class.getSimpleName())    
            .withResponseMessageType(PlantDiagnosisRequestDataResponse.class.getSimpleName());     

     /**
     * Capability to provide a schedule for farm operations
     * 
     * @since 0.16
     * @see ScheduleRequest and -Review
     */
    @CapabilityTypeRegistration
    public static final CapabilityType PROVIDE_SCHEDULE = new CapabilityType()
            .withName("schedule")
            .withSchemaLocation(SCHEMA)
            .withContextPath(CONTEXT_PATH)
            .withRequestMessageType(ScheduleRequest.class.getSimpleName())
            .withResponseMessageType(ScheduleResponse.class.getSimpleName());

     /**
     * Capability to receive an "advice available notification"
     * 
     * Example: a farm management system, that is requesting advices from experts, could provide 
     * this capability so that it can be informed by an expert system as soon as the advice that was 
     * requested by the farmer (via the capability @PROVIDE_ADVICE) is ready.
     * 
     * @since 0.11.0
     * @see AdviceAvailableNotification
    
    @CapabilityTypeRegistration
    public static final CapabilityType RECEIVE_ADVICE_AVAILABLE_NOTIFICATION = new CapabilityType()
            .withName("RECEIVE_ADVICE_AVAILABLE_NOTIFICATION")
            .withSchemaLocation(SCHEMA)
            .withContextPath(CONTEXT_PATH)
            .withRequestMessageType(AdviceAvailableNotification.class.getSimpleName())
            .withResponseMessageType(Messages.Untyped.class.getSimpleName()); */

    /**
     *
     * @since 0.7.2
     * @see CropFieldAdviceRequestMessage
    
    @CapabilityTypeRegistration
    public static final CapabilityType RECEIVE_ADVICE_REQUEST = new CapabilityType()
            .withName("greenhouse advice")
            .withSchemaLocation(SCHEMA)
            .withContextPath(CONTEXT_PATH)
            .withRequestMessageType(CropFieldAdviceRequestMessage.class.getSimpleName())
            .withResponseMessageType(Messages.Untyped.class.getSimpleName()); */
    
    
   
    /**
     * This capability is to be provided by a "weather conditions provider" to provide weather information to a client.
     * 
     * Expected behaviour of the system implementing this capability:
     * - if a request for weather info cannot be fulfilled at once, the answer should be HTTP 204
     * - if a request for weather info can be fulfilled at once, the answer should be HTTP 303, including a URL where the actual weather info can be fetched
     * - if the URL returned by a previous 303 answer is used to call this capability, the answer should be the actual ProvideWeatherConditionsResponseMessage
     * 
     * @since 0.10.0
     * @see ProvideWeatherConditionsRequestMessage
     * @see ProvideWeatherConditionsResponseMessage
    
    @CapabilityTypeRegistration
    public static final CapabilityType PROVIDE_WEATHER_CONDITIONS = new CapabilityType()
            .withName("PROVIDE_WEATHER_CONDITIONS")
            .withSchemaLocation(SCHEMA)
            .withContextPath(CONTEXT_PATH)
            .withRequestMessageType(ProvideWeatherConditionsRequestMessage.class.getSimpleName())
            .withResponseMessageType(ProvideWeatherConditionsResponseMessage.class.getSimpleName());  */
    
    /**
     * Capability to receive sensor values
     * 
     * Example: FIspace provides this capabilty so that an external system can send sensor values for event detection
     * 
     * @since 0.10.1
     * @see ReceiveSensorValuesRequestMessage

    @CapabilityTypeRegistration
    public static final CapabilityType RECEIVE_SENSOR_VALUES = new CapabilityType()
            .withName("receive sensor values")
            .withSchemaLocation(SCHEMA)
            .withContextPath(CONTEXT_PATH)
            .withRequestMessageType(SensorValuesRequest.class.getSimpleName())
            .withResponseMessageType(Messages.Untyped.class.getSimpleName());     */

    /**
     *
     * Capability to search for farmers to external marketplace of OPEKEPE.This capability
     * is used by Marketplace Operations Service.
     * @see SearchForFarmersRequest
     * @see SearchForFarmersResponse
     */
    @CapabilityTypeRegistration
    public static final CapabilityType SEARCH_FOR_FARMERS = new CapabilityType()
            .withName("SEARCH_FOR_FARMERS")
            .withSchemaLocation(SCHEMA)
            .withContextPath(CONTEXT_PATH)
            .withRequestMessageType(SearchForFarmersRequest.class.getSimpleName())
            .withResponseMessageType(SearchForFarmersResponse.class.getSimpleName());

     /**
     * Capability to provide sensor data on request
     * 
     * Example: a farmer could provide this capability, so that business partners 
     * could use this capability to get sensor values from the farmer's system
     *
     * @since 0.16
     * @see SensorDataRequest
     * @see SensorDataResponse
     */
    @CapabilityTypeRegistration
    public static final CapabilityType PROVIDE_SENSOR_DATA = new CapabilityType()
            .withName("provide sensor data")	
            .withSchemaLocation(SCHEMA)
            .withContextPath(CONTEXT_PATH)
            .withRequestMessageType(SensorDataRequest.class.getSimpleName())
            .withResponseMessageType(SensorDataResponse.class.getSimpleName());
    /**
     * Capability to provide sensor values on request
     * 
     * Example: a farmer could provide this capability, so that business partners 
     * could use this capability to get sensor values from the farmer's system
     *
     * @since 0.10.1
     * @see SensorValuesRequest
     * @see SensorValuesResponse
     */
    @CapabilityTypeRegistration
    public static final CapabilityType PROVIDE_SENSOR_VALUES = new CapabilityType()
            .withName("provide sensor values")	
            .withSchemaLocation(SCHEMA)
            .withContextPath(CONTEXT_PATH)
            .withRequestMessageType(SensorValuesRequest.class.getSimpleName())
            .withResponseMessageType(SensorValuesResponse.class.getSimpleName());

    /**
     *
     * Capability to provide Task Data by as well a Farm Management Information 
     * System as a Agricultural Activity Server. 
     * This Task data is based on drmCrop and able to handle individual Tasks.
     * To exchange ISO11783 TaskFile.xml files, use PROVIDE_TASK_FILE.
     * @see TaskDataRequest and -Response
     */
    @CapabilityTypeRegistration
    public static final CapabilityType PROVIDE_TASK_DATA = new CapabilityType()
            .withName("provide task data")
            .withSchemaLocation(SCHEMA)
            .withContextPath(CONTEXT_PATH)
            .withRequestMessageType(TaskDataRequest.class.getSimpleName())
            .withResponseMessageType(TaskDataResponse.class.getSimpleName());

    /**
     *
     * Capability to provide an ISO11783 TaskFile by as well a Farm Management  
     * Information System as a Agricultural Activity Server. 
     * @see TaskFileRequest and -Response
     */
    @CapabilityTypeRegistration
    public static final CapabilityType PROVIDE_TASK_FILE = new CapabilityType()
            .withName("provide task file")
            .withSchemaLocation(SCHEMA)
            .withContextPath(CONTEXT_PATH)
            .withRequestMessageType(TaskFileRequest.class.getSimpleName())
            .withResponseMessageType(TaskFileResponse.class.getSimpleName());

    @CapabilityTypeRegistration
    public static final CapabilityType PROVIDE_TRACEABILITY = new CapabilityType()
            .withName("provide traceability")
            .withSchemaLocation(SCHEMA)
            .withContextPath(CONTEXT_PATH)
            .withRequestMessageType(BasicTraceabilityInformationRequest.class.getSimpleName())
            .withResponseMessageType(BasicTraceabilityInformationResponse.class.getSimpleName());

    
    /**
     * Capability for providing a weather scenario (a set of weather 
     * predictions for given locations and time interval).
     * The message use the drmCrop message format.
     *
     * @since 0.12.2
     * @see WeatherScenarioRequest
     * @see WeatherScenarioResponse
     */
    @CapabilityTypeRegistration
    public static final CapabilityType PROVIDE_WEATHER_SCENARIO = new CapabilityType()
            .withName("provide weather scenario")	
            .withSchemaLocation(SCHEMA)
            .withContextPath(CONTEXT_PATH)
            .withRequestMessageType(WeatherScenarioRequest.class.getSimpleName())
            .withResponseMessageType(WeatherScenarioResponse.class.getSimpleName());

     /**
     * Capability for providing a weather scenario (a set of weather 
     * predictions for given locations and time interval).
     * The message use the simplified, non-drmCrop message format.
     *
     * @since 0.14.0
     * @see WeatherScenarioSimpleRequest
     * @see WeatherScenarioSimpleResponse
     */
    @CapabilityTypeRegistration
    public static final CapabilityType PROVIDE_WEATHER_SCENARIO_SIMPLE = new CapabilityType()
            .withName("provide weather scenario, simple message format")	
            .withSchemaLocation(SCHEMA)
            .withContextPath(CONTEXT_PATH)
            .withRequestMessageType(WeatherScenarioSimpleRequest.class.getSimpleName())
            .withResponseMessageType(WeatherScenarioSimpleResponse.class.getSimpleName());

    /**
     * Capability to produce a whole field Phytophthora advice
     *
     * @since 0.16.0
     * @see WFPhytophthoraAdviceRequest
     * @see WFPhytophthoraAdviceResponse
     */
    @CapabilityTypeRegistration
    public static final CapabilityType PROVIDE_WF_PHYTOPHTHORA_ADVICE = new CapabilityType()
            .withName("whole field phytophthora advice")
            .withSchemaLocation(SCHEMA)
            .withContextPath(CONTEXT_PATH)
            .withRequestMessageType(WFPhytophthoraAdviceRequest.class.getSimpleName())
            .withResponseMessageType(WFPhytophthoraAdviceResponse.class.getSimpleName());

    /**
     * This capability is to be provided by a "Farm management Information System" 
     * to provide the criteria for workability criteria for different operation 
     * techniques.
     * 
     * Expected behavior of the system implementing this capability:
     * - if a request for workability criteria cannot be fulfilled at once, the 
     *   answer should be HTTP 204
     * - if a request for workability criteria can be fulfilled at once, the 
     *   answer should be HTTP 303, including a URL where the actual weather 
     *   info can be fetched
     * - if the URL returned by a previous 303 answer is used to call this 
     *   capability, the answer should be the actual WorkabilityCriteriaResponse
     * 
     * @since 0.10.0
     * @see WorkabilityCriteriaRequest
     * @see WorkabilityCriteriaResponse
     */
    @CapabilityTypeRegistration
    public static final CapabilityType PROVIDE_WORKABILITY_CRITERIA = new CapabilityType()
            .withName("workability criteria")
            .withSchemaLocation(SCHEMA)
            .withContextPath(CONTEXT_PATH)
            .withRequestMessageType(WorkabilityCriteriaRequest.class.getSimpleName())
            .withResponseMessageType(WorkabilityCriteriaResponse.class.getSimpleName());

    /**
     * This capability is to be provided by a "Bad weather module" to provide 
     * workability Data to a client.
     * 
     * Expected behavior of the system implementing this capability:
     * - if a request for weather info cannot be fulfilled at once, the answer 
     *   should be HTTP 204
     * - if a request for weather info can be fulfilled at once, the answer 
     *   should be HTTP 303, including a URL where the actual weather info can 
     *   be fetched
     * - if the URL returned by a previous 303 answer is used to call this 
     *   capability, the answer should be the actual WoirkabilityDataResponse
     * 
     * @since 0.10.0
     * @see WorkabilityDataRequest
     * @see WorkabilityDataResponse
     */
    @CapabilityTypeRegistration
    public static final CapabilityType PROVIDE_WORKABILITY_DATA = new CapabilityType()
            .withName("PROVIDE_WORKABILITY_DATA")
            .withSchemaLocation(SCHEMA)
            .withContextPath(CONTEXT_PATH)
            .withRequestMessageType(WorkabilityDataRequest.class.getSimpleName())
            .withResponseMessageType(WorkabilityDataResponse.class.getSimpleName());

    

}
