/*
 * Decompiled with CFR 0.152.
 */
package eu.maveniverse.maven.shared.core.fs;

import java.io.Closeable;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.ByteBuffer;
import java.nio.file.CopyOption;
import java.nio.file.FileVisitResult;
import java.nio.file.FileVisitor;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.StandardCopyOption;
import java.nio.file.attribute.BasicFileAttributes;
import java.nio.file.attribute.FileAttribute;
import java.util.Objects;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.function.Predicate;

public final class FileUtils {
    private static final boolean IS_WINDOWS = System.getProperty("os.name", "unknown").startsWith("Windows");

    private FileUtils() {
    }

    public static Path discoverBaseDirectory(String basedirKey, String defBasedir) {
        Objects.requireNonNull(basedirKey, "basedirKey");
        Objects.requireNonNull(defBasedir, "defBasedir");
        String basedir = System.getProperty(basedirKey);
        if (basedir == null) {
            return FileUtils.canonicalPath(FileUtils.discoverUserHomeDirectory().resolve(defBasedir));
        }
        return FileUtils.canonicalPath(FileUtils.discoverUserCurrentWorkingDirectory().resolve(basedir));
    }

    public static Path discoverUserCurrentWorkingDirectory() {
        String userHome = System.getProperty("user.dir");
        if (userHome == null) {
            throw new IllegalStateException("requires user.dir Java System Property set");
        }
        return FileUtils.canonicalPath(Paths.get(userHome, new String[0]));
    }

    public static Path discoverUserHomeDirectory() {
        String userHome = System.getProperty("user.home");
        if (userHome == null) {
            throw new IllegalStateException("requires user.home Java System Property set");
        }
        return FileUtils.canonicalPath(Paths.get(userHome, new String[0]));
    }

    public static Path canonicalPath(Path path) {
        Objects.requireNonNull(path, "path");
        try {
            return path.toRealPath(new LinkOption[0]);
        }
        catch (IOException e) {
            return FileUtils.canonicalPath(path.getParent()).resolve(path.getFileName());
        }
    }

    public static TempFile newTempFile() throws IOException {
        final Path tempFile = Files.createTempFile("resolver", "tmp", new FileAttribute[0]);
        return new TempFile(){

            @Override
            public Path getPath() {
                return tempFile;
            }

            @Override
            public void close() throws IOException {
                Files.deleteIfExists(tempFile);
            }
        };
    }

    public static CollocatedTempFile newTempFile(final Path file) throws IOException {
        Path parent = Objects.requireNonNull(file.getParent(), "file must have parent");
        Files.createDirectories(parent, new FileAttribute[0]);
        final Path tempFile = parent.resolve(file.getFileName() + "." + Long.toUnsignedString(ThreadLocalRandom.current().nextLong()) + ".tmp");
        return new CollocatedTempFile(){
            private final AtomicBoolean wantsMove = new AtomicBoolean(false);

            @Override
            public Path getPath() {
                return tempFile;
            }

            @Override
            public void move() {
                this.wantsMove.set(true);
            }

            @Override
            public void close() throws IOException {
                if (this.wantsMove.get()) {
                    if (IS_WINDOWS) {
                        FileUtils.winCopy(tempFile, file);
                    } else {
                        Files.move(tempFile, file, StandardCopyOption.ATOMIC_MOVE);
                    }
                }
                Files.deleteIfExists(tempFile);
            }
        };
    }

    private static void winCopy(Path source2, Path target) throws IOException {
        ByteBuffer buffer = ByteBuffer.allocate(32768);
        byte[] array = buffer.array();
        try (InputStream is = Files.newInputStream(source2, new OpenOption[0]);
             OutputStream os = Files.newOutputStream(target, new OpenOption[0]);){
            int bytes;
            while ((bytes = is.read(array)) >= 0) {
                os.write(array, 0, bytes);
            }
        }
    }

    public static void copyOrLink(Path src, Path dst, boolean mayLink) throws IOException {
        if (mayLink && FileUtils.mayLink(src, dst)) {
            FileUtils.link(src, dst);
        } else {
            FileUtils.copy(src, dst);
        }
    }

    public static boolean mayLink(Path src, Path dst) throws IOException {
        if (!Objects.equals(Files.getFileStore(src), Files.getFileStore(dst.getParent()))) {
            return false;
        }
        Path userHome = Paths.get(System.getProperty("user.home"), new String[0]);
        return src.startsWith(userHome) || !dst.startsWith(userHome);
    }

    public static void link(Path src, Path dst) throws IOException {
        Files.createLink(dst, src);
    }

    public static void copy(Path src, Path dst) throws IOException {
        Files.copy(src, dst, new CopyOption[0]);
        Files.setLastModifiedTime(dst, Files.getLastModifiedTime(src, new LinkOption[0]));
    }

    public static void copyRecursively(final Path from, final Path to, final Predicate<Path> predicate, final boolean overwrite) throws IOException {
        Files.walkFileTree(from, (FileVisitor<? super Path>)new SimpleFileVisitor<Path>(){

            @Override
            public FileVisitResult preVisitDirectory(Path dir, BasicFileAttributes attrs) throws IOException {
                if (predicate.test(dir)) {
                    Path target = to.resolve(from.relativize(dir).toString());
                    Files.createDirectories(target, new FileAttribute[0]);
                    return FileVisitResult.CONTINUE;
                }
                return FileVisitResult.SKIP_SUBTREE;
            }

            @Override
            public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) throws IOException {
                if (predicate.test(file)) {
                    Path target = to.resolve(from.relativize(file).toString());
                    if (overwrite) {
                        Files.copy(file, target, StandardCopyOption.REPLACE_EXISTING, StandardCopyOption.COPY_ATTRIBUTES);
                    } else {
                        Files.copy(file, target, StandardCopyOption.COPY_ATTRIBUTES);
                    }
                }
                return FileVisitResult.CONTINUE;
            }
        });
    }

    public static void deleteRecursively(Path path) throws IOException {
        Files.walkFileTree(path, (FileVisitor<? super Path>)new SimpleFileVisitor<Path>(){

            @Override
            public FileVisitResult postVisitDirectory(Path dir, IOException exc) throws IOException {
                Files.delete(dir);
                return FileVisitResult.CONTINUE;
            }

            @Override
            public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) throws IOException {
                Files.delete(file);
                return FileVisitResult.CONTINUE;
            }
        });
    }

    public static void writeFile(Path target, FileWriter writer) throws IOException {
        FileUtils.writeFile(target, writer, false);
    }

    public static void writeFileWithBackup(Path target, FileWriter writer) throws IOException {
        FileUtils.writeFile(target, writer, true);
    }

    private static void writeFile(Path target, FileWriter writer, boolean doBackup) throws IOException {
        Objects.requireNonNull(target, "target is null");
        Objects.requireNonNull(writer, "writer is null");
        Path parent = Objects.requireNonNull(target.getParent(), "target must have parent");
        try (CollocatedTempFile tempFile = FileUtils.newTempFile(target);){
            writer.write(tempFile.getPath());
            if (doBackup && Files.isRegularFile(target, new LinkOption[0])) {
                Files.copy(target, parent.resolve(target.getFileName() + ".bak"), StandardCopyOption.REPLACE_EXISTING);
            }
            tempFile.move();
        }
    }

    @FunctionalInterface
    public static interface FileWriter {
        public void write(Path var1) throws IOException;
    }

    public static interface CollocatedTempFile
    extends TempFile {
        public void move() throws IOException;
    }

    public static interface TempFile
    extends Closeable {
        public Path getPath();
    }
}

