/*! SignatureFieldExtension - v1.0.0 - 2018-01-14
* Copyright (c) 2018 Max Schuster; Licensed Apache-2.0 */
(function (root, factory) {
  if (typeof define === 'function' && define.amd) {
    // AMD. Register as an anonymous module unless amdModuleId is set
    define([], function () {
      return (root['SignaturePad'] = factory());
    });
  } else if (typeof exports === 'object') {
    // Node. Does not work with strict CommonJS, but
    // only CommonJS-like environments that support module.exports,
    // like Node.
    module.exports = factory();
  } else {
    root['SignaturePad'] = factory();
  }
}(this, function () {

/*!
 * Signature Pad v1.5.3
 * https://github.com/szimek/signature_pad
 *
 * Copyright 2016 Szymon Nowak
 * Released under the MIT license
 *
 * The main idea and some parts of the code (e.g. drawing variable width Bézier curve) are taken from:
 * http://corner.squareup.com/2012/07/smoother-signatures.html
 *
 * Implementation of interpolation using cubic Bézier curves is taken from:
 * http://benknowscode.wordpress.com/2012/09/14/path-interpolation-using-cubic-bezier-and-control-point-estimation-in-javascript
 *
 * Algorithm for approximated length of a Bézier curve is taken from:
 * http://www.lemoda.net/maths/bezier-length/index.html
 *
 */
var SignaturePad = (function (document) {
    "use strict";

    var SignaturePad = function (canvas, options) {
        var self = this,
            opts = options || {};

        this.velocityFilterWeight = opts.velocityFilterWeight || 0.7;
        this.minWidth = opts.minWidth || 0.5;
        this.maxWidth = opts.maxWidth || 2.5;
        this.dotSize = opts.dotSize || function () {
            return (this.minWidth + this.maxWidth) / 2;
        };
        this.penColor = opts.penColor || "black";
        this.backgroundColor = opts.backgroundColor || "rgba(0,0,0,0)";
        this.onEnd = opts.onEnd;
        this.onBegin = opts.onBegin;

        this._canvas = canvas;
        this._ctx = canvas.getContext("2d");
        this.clear();

        // we need add these inline so they are available to unbind while still having
        //  access to 'self' we could use _.bind but it's not worth adding a dependency
        this._handleMouseDown = function (event) {
            if (event.which === 1) {
                self._mouseButtonDown = true;
                self._strokeBegin(event);
            }
        };

        this._handleMouseMove = function (event) {
            if (self._mouseButtonDown) {
                self._strokeUpdate(event);
            }
        };

        this._handleMouseUp = function (event) {
            if (event.which === 1 && self._mouseButtonDown) {
                self._mouseButtonDown = false;
                self._strokeEnd(event);
            }
        };

        this._handleTouchStart = function (event) {
            if (event.targetTouches.length == 1) {
                var touch = event.changedTouches[0];
                self._strokeBegin(touch);
             }
        };

        this._handleTouchMove = function (event) {
            // Prevent scrolling.
            event.preventDefault();

            var touch = event.targetTouches[0];
            self._strokeUpdate(touch);
        };

        this._handleTouchEnd = function (event) {
            var wasCanvasTouched = event.target === self._canvas;
            if (wasCanvasTouched) {
                event.preventDefault();
                self._strokeEnd(event);
            }
        };

        this._handleMouseEvents();
        this._handleTouchEvents();
    };

    SignaturePad.prototype.clear = function () {
        var ctx = this._ctx,
            canvas = this._canvas;

        ctx.fillStyle = this.backgroundColor;
        ctx.clearRect(0, 0, canvas.width, canvas.height);
        ctx.fillRect(0, 0, canvas.width, canvas.height);
        this._reset();
    };

    SignaturePad.prototype.toDataURL = function (imageType, quality) {
        var canvas = this._canvas;
        return canvas.toDataURL.apply(canvas, arguments);
    };

    SignaturePad.prototype.fromDataURL = function (dataUrl) {
        var self = this,
            image = new Image(),
            ratio = window.devicePixelRatio || 1,
            width = this._canvas.width / ratio,
            height = this._canvas.height / ratio;

        this._reset();
        image.src = dataUrl;
        image.onload = function () {
            self._ctx.drawImage(image, 0, 0, width, height);
        };
        this._isEmpty = false;
    };

    SignaturePad.prototype._strokeUpdate = function (event) {
        var point = this._createPoint(event);
        this._addPoint(point);
    };

    SignaturePad.prototype._strokeBegin = function (event) {
        this._reset();
        this._strokeUpdate(event);
        if (typeof this.onBegin === 'function') {
            this.onBegin(event);
        }
    };

    SignaturePad.prototype._strokeDraw = function (point) {
        var ctx = this._ctx,
            dotSize = typeof(this.dotSize) === 'function' ? this.dotSize() : this.dotSize;

        ctx.beginPath();
        this._drawPoint(point.x, point.y, dotSize);
        ctx.closePath();
        ctx.fill();
    };

    SignaturePad.prototype._strokeEnd = function (event) {
        var canDrawCurve = this.points.length > 2,
            point = this.points[0];

        if (!canDrawCurve && point) {
            this._strokeDraw(point);
        }
        if (typeof this.onEnd === 'function') {
            this.onEnd(event);
        }
    };

    SignaturePad.prototype._handleMouseEvents = function () {
        this._mouseButtonDown = false;

        this._canvas.addEventListener("mousedown", this._handleMouseDown);
        this._canvas.addEventListener("mousemove", this._handleMouseMove);
        document.addEventListener("mouseup", this._handleMouseUp);
    };

    SignaturePad.prototype._handleTouchEvents = function () {
        // Pass touch events to canvas element on mobile IE11 and Edge.
        this._canvas.style.msTouchAction = 'none';
        this._canvas.style.touchAction = 'none';

        this._canvas.addEventListener("touchstart", this._handleTouchStart);
        this._canvas.addEventListener("touchmove", this._handleTouchMove);
        this._canvas.addEventListener("touchend", this._handleTouchEnd);
    };

    SignaturePad.prototype.on = function () {
        this._handleMouseEvents();
        this._handleTouchEvents();
    };

    SignaturePad.prototype.off = function () {
        this._canvas.removeEventListener("mousedown", this._handleMouseDown);
        this._canvas.removeEventListener("mousemove", this._handleMouseMove);
        document.removeEventListener("mouseup", this._handleMouseUp);

        this._canvas.removeEventListener("touchstart", this._handleTouchStart);
        this._canvas.removeEventListener("touchmove", this._handleTouchMove);
        this._canvas.removeEventListener("touchend", this._handleTouchEnd);
    };

    SignaturePad.prototype.isEmpty = function () {
        return this._isEmpty;
    };

    SignaturePad.prototype._reset = function () {
        this.points = [];
        this._lastVelocity = 0;
        this._lastWidth = (this.minWidth + this.maxWidth) / 2;
        this._isEmpty = true;
        this._ctx.fillStyle = this.penColor;
    };

    SignaturePad.prototype._createPoint = function (event) {
        var rect = this._canvas.getBoundingClientRect();
        return new Point(
            event.clientX - rect.left,
            event.clientY - rect.top
        );
    };

    SignaturePad.prototype._addPoint = function (point) {
        var points = this.points,
            c2, c3,
            curve, tmp;

        points.push(point);

        if (points.length > 2) {
            // To reduce the initial lag make it work with 3 points
            // by copying the first point to the beginning.
            if (points.length === 3) points.unshift(points[0]);

            tmp = this._calculateCurveControlPoints(points[0], points[1], points[2]);
            c2 = tmp.c2;
            tmp = this._calculateCurveControlPoints(points[1], points[2], points[3]);
            c3 = tmp.c1;
            curve = new Bezier(points[1], c2, c3, points[2]);
            this._addCurve(curve);

            // Remove the first element from the list,
            // so that we always have no more than 4 points in points array.
            points.shift();
        }
    };

    SignaturePad.prototype._calculateCurveControlPoints = function (s1, s2, s3) {
        var dx1 = s1.x - s2.x, dy1 = s1.y - s2.y,
            dx2 = s2.x - s3.x, dy2 = s2.y - s3.y,

            m1 = {x: (s1.x + s2.x) / 2.0, y: (s1.y + s2.y) / 2.0},
            m2 = {x: (s2.x + s3.x) / 2.0, y: (s2.y + s3.y) / 2.0},

            l1 = Math.sqrt(dx1*dx1 + dy1*dy1),
            l2 = Math.sqrt(dx2*dx2 + dy2*dy2),

            dxm = (m1.x - m2.x),
            dym = (m1.y - m2.y),

            k = l2 / (l1 + l2),
            cm = {x: m2.x + dxm*k, y: m2.y + dym*k},

            tx = s2.x - cm.x,
            ty = s2.y - cm.y;

        return {
            c1: new Point(m1.x + tx, m1.y + ty),
            c2: new Point(m2.x + tx, m2.y + ty)
        };
    };

    SignaturePad.prototype._addCurve = function (curve) {
        var startPoint = curve.startPoint,
            endPoint = curve.endPoint,
            velocity, newWidth;

        velocity = endPoint.velocityFrom(startPoint);
        velocity = this.velocityFilterWeight * velocity
            + (1 - this.velocityFilterWeight) * this._lastVelocity;

        newWidth = this._strokeWidth(velocity);
        this._drawCurve(curve, this._lastWidth, newWidth);

        this._lastVelocity = velocity;
        this._lastWidth = newWidth;
    };

    SignaturePad.prototype._drawPoint = function (x, y, size) {
        var ctx = this._ctx;

        ctx.moveTo(x, y);
        ctx.arc(x, y, size, 0, 2 * Math.PI, false);
        this._isEmpty = false;
    };

    SignaturePad.prototype._drawCurve = function (curve, startWidth, endWidth) {
        var ctx = this._ctx,
            widthDelta = endWidth - startWidth,
            drawSteps, width, i, t, tt, ttt, u, uu, uuu, x, y;

        drawSteps = Math.floor(curve.length());
        ctx.beginPath();
        for (i = 0; i < drawSteps; i++) {
            // Calculate the Bezier (x, y) coordinate for this step.
            t = i / drawSteps;
            tt = t * t;
            ttt = tt * t;
            u = 1 - t;
            uu = u * u;
            uuu = uu * u;

            x = uuu * curve.startPoint.x;
            x += 3 * uu * t * curve.control1.x;
            x += 3 * u * tt * curve.control2.x;
            x += ttt * curve.endPoint.x;

            y = uuu * curve.startPoint.y;
            y += 3 * uu * t * curve.control1.y;
            y += 3 * u * tt * curve.control2.y;
            y += ttt * curve.endPoint.y;

            width = startWidth + ttt * widthDelta;
            this._drawPoint(x, y, width);
        }
        ctx.closePath();
        ctx.fill();
    };

    SignaturePad.prototype._strokeWidth = function (velocity) {
        return Math.max(this.maxWidth / (velocity + 1), this.minWidth);
    };


    var Point = function (x, y, time) {
        this.x = x;
        this.y = y;
        this.time = time || new Date().getTime();
    };

    Point.prototype.velocityFrom = function (start) {
        return (this.time !== start.time) ? this.distanceTo(start) / (this.time - start.time) : 1;
    };

    Point.prototype.distanceTo = function (start) {
        return Math.sqrt(Math.pow(this.x - start.x, 2) + Math.pow(this.y - start.y, 2));
    };

    var Bezier = function (startPoint, control1, control2, endPoint) {
        this.startPoint = startPoint;
        this.control1 = control1;
        this.control2 = control2;
        this.endPoint = endPoint;
    };

    // Returns approximated length.
    Bezier.prototype.length = function () {
        var steps = 10,
            length = 0,
            i, t, cx, cy, px, py, xdiff, ydiff;

        for (i = 0; i <= steps; i++) {
            t = i / steps;
            cx = this._point(t, this.startPoint.x, this.control1.x, this.control2.x, this.endPoint.x);
            cy = this._point(t, this.startPoint.y, this.control1.y, this.control2.y, this.endPoint.y);
            if (i > 0) {
                xdiff = cx - px;
                ydiff = cy - py;
                length += Math.sqrt(xdiff * xdiff + ydiff * ydiff);
            }
            px = cx;
            py = cy;
        }
        return length;
    };

    Bezier.prototype._point = function (t, start, c1, c2, end) {
        return          start * (1.0 - t) * (1.0 - t)  * (1.0 - t)
               + 3.0 *  c1    * (1.0 - t) * (1.0 - t)  * t
               + 3.0 *  c2    * (1.0 - t) * t          * t
               +        end   * t         * t          * t;
    };

    return SignaturePad;
})(document);

return SignaturePad;

}));

/**
 * Client-side logic of the SignatureField
 * 
 * @property {Element} parent Parent element
 * @property {Element} canvas Canvas
 * @property {Element} clearButton Button that clears the field
 * @property {string} signature Signature data url
 * @property {string} mimeType MIME-Type that determines the format of the signature image
 * @property {SignaturePad} signaturePad SignaturePad instance
 * @property {boolean} immediate Update the signature on the server-side
 * immediately after each pen stroke
 * @property {boolean} readOnly Field is readonly
 * @author Max Schuster
 * @returns {undefined}
 */
function eu_maxschuster_vaadin_signaturefield_SignatureFieldExtension() {

    "use strict";
    
    var defaultMimeType = "image/png",
            defaultDotSize;

    /* jshint validthis:true */
    /* jshint -W087 */ // All 'debugger' statements should be removed

    /**
     * Initializes the extension
     * @returns {undefined}
     */
    this.init = function () {
        var state = this.getState();

        this.parent = this.findAndExtendParent();
        this.canvas = this.createCanvas(this.parent);
        this.clearButton = null;
        this.signature = null;
        this.mimeType = defaultMimeType;
        this.signaturePad = this.createSignaturePad(this.canvas);
        defaultDotSize = this.signaturePad.dotSize;
        this.immediate = state.immediate;
        this.readOnly = state.readOnly;

        this.addResizeListener(this.parent, this.proxy(this.onResize));
    };

    /**
     * Proxy the given function with a context
     * @param {Function} func Function to proxy
     * @param {Object} [ctx=this] Context to apply to the function. Defaults to
     * "this".
     * @param {Array} [args=undefined] arguments that should be passed to the
     * function. Defaults to "arguments"
     * @returns {Function} The function proxy
     */
    this.proxy = function (func, ctx, args) {
        ctx = ctx || this;
        return function () {
            func.apply(ctx, args || arguments);
        };
    };

    /**
     * Imports the given signature dataURL into the canvas. Called from the
     * server-side to update the signature on the client-side
     * @param {string} newSignature Signture to import
     * @returns {undefined}
     */
    this.setSignature = function (newSignature) {
        var signaturePad = this.signaturePad,
                oldSignature = this.signature;
        if (newSignature === oldSignature) {
            return;
        }
        this.signature = newSignature;
        if (newSignature) {
            signaturePad.fromDataURL(newSignature);
        } else {
            signaturePad.clear();
        }
    };

    /**
     * Extends and extends the parent component of this extension.
     * @returns {Element} The parent element.
     */
    this.findAndExtendParent = function () {
        var parent = this.getElement(this.getParentId());
        parent.tabIndex = -1;
        this.addEvent(parent, "focus", this.proxy(this.onFocus));
        this.addEvent(parent, "blur", this.proxy(this.onBlur));
        return parent;
    };

    /**
     * Creates the canvas element and attaches it to the given parent element.
     * @param {Element} parent Parent element of the canvas.
     * @returns {unresolved}
     */
    this.createCanvas = function (parent) {
        var canvas = document.createElement("canvas");
        parent.appendChild(canvas);
        return canvas;
    };

    /**
     * Updates the canvas size to match its parent's size, without clearing it.
     * @returns {undefined}
     */
    this.updateCanvasSize = function () {
        var style = this.getComputedStyle(this.parent),
                canvas = this.canvas,
                signature = this.signature,
                signaturePad = this.signaturePad,
                width = parseInt(style.width),
                height = parseInt(style.height);

        canvas.width = width;
        canvas.height = height;

        if (signature) {
            signaturePad.fromDataURL(signature);
        } else {
            signaturePad.clear();
        }
    };

    /**
     * Creates a signature pad instance with the given canvas element.
     * @param {Element} canvas The canvas to use with the signature pad.
     * @returns {SignaturePad} Signature pad instance.
     */
    this.createSignaturePad = function (canvas) {
        var signaturePad = new SignaturePad(canvas);
        signaturePad.onBegin = this.proxy(this.onStrokeBegin);
        signaturePad.onEnd = this.proxy(this.onStrokeEnd);
        signaturePad.vReadOnly = false;

        // Remove window.devicePixelRatio
        signaturePad.fromDataURL = function (dataUrl) {
            var image = new Image();
            image.src = dataUrl;
            image.onload = function () {
                signaturePad.clear();
                this._isEmpty = false;
                signaturePad._ctx.drawImage(image, 0, 0, canvas.width, canvas.height);
            };
            this._isEmpty = false;
        };
       
        return signaturePad;
    };

    /**
     * Creates the clear button element and adds it to the given parent.
     * @param {Element} parent Parent element to add the clear button to.
     * @returns {Element} Clear button
     */
    this.createClearButton = function (parent) {
        var clearButton = document.createElement("button");
        clearButton.setAttribute("class", "signaturefield-clearbutton");
        parent.appendChild(clearButton);
        this.addEvent(clearButton, "click", this.proxy(this.onClearButtonClick));
        return clearButton;
    };

    /**
     * Gets called when the parent element has changed its size.
     * @returns {undefined}
     */
    this.onResize = function () {
        this.updateCanvasSize();
    };

    /**
     * Gets called when a stroke begins.
     * @returns {undefined}
     */
    this.onStrokeBegin = function () {
        this.parent.focus();
    };

    /**
     * Gets called when a stroke ends.
     * @returns {undefined}
     */
    this.onStrokeEnd = function () {
        if (this.immediate) {
            this.updateSignature();
        }
    };

    /**
     * Gets called when the field becomes focused.
     * @returns {undefined}
     */
    this.onFocus = function () {
        this.addClass(this.parent, "v-focus");
    };

    /**
     * Gets called when the field loses focus.
     * @returns {undefined}
     */
    this.onBlur = function () {
        this.updateSignature();
        this.removeClass(this.parent, "v-focus");
    };

    /**
     * Gets called when the clear button is clicked.
     * @returns {undefined}
     */
    this.onClearButtonClick = function () {
        this.clear();
    };
    
    /**
     * Gets called when the shared state of this extension changes. Updates the
     * parts of the extension with the new values from the state.
     * @returns {undefined}
     */
    this.onStateChange = function () {
        var state = this.getState(),
                parent = this.parent,
                signaturePad = this.signaturePad;

        signaturePad.dotSize = state.dotSize || state.dotSize === 0 ?
                state.dotSize : defaultDotSize;
        signaturePad.minWidth = state.minWidth;
        signaturePad.maxWidth = state.maxWidth;
        signaturePad.backgroundColor = state.backgroundColor;
        signaturePad.penColor = state.penColor;
        signaturePad.velocityFilterWeight = state.velocityFilterWeight;

        this.mimeType = state.mimeType || defaultMimeType;
        this.immediate = state.immediate;

        var readOnly = state.readOnly,
                vReadOnly = signaturePad.vReadOnly;
        if (readOnly && !vReadOnly) {
            signaturePad.off();
            parent.removeAttribute("tabindex");
        } else if (!readOnly && vReadOnly) {
            signaturePad.on();
            parent.tabIndex = -1;
        }
        signaturePad.vReadOnly = readOnly;

        if ((!readOnly && state.clearButtonEnabled) && !this.clearButton) {
            this.clearButton = this.createClearButton(parent);
        } else if ((readOnly || !state.clearButtonEnabled) && this.clearButton) {
            parent.removeChild(this.clearButton);
            this.clearButton = null;
        }
    };

    /**
     * Returns the computed style for the given element
     * @param {Element} el Element to get the styles from
     * @returns {CSSStyleDeclaration}
     */
    this.getComputedStyle = function (el) {
        if (window.getComputedStyle) {
            return window.getComputedStyle(el);
        }
        return el.currentStyle;
    };
    
    /**
     * Gets the current signature as dataURL or null if the signatue pad is
     * empty.
     * @returns {string|null} Signature as dataURL or null.
     */
    this.getCurrentSignature = function () {
        var signaturePad = this.signaturePad;
        return signaturePad.isEmpty() ?
                null : signaturePad.toDataURL(this.mimeType, 1);
    };

    /**
     * Updates the server-side signature value with the given signature.
     * @param {string|null} signature Signature value for the server-side.
     * @returns {undefined}
     */
    this.updateSignature = function (signature) {
        var oldSignature = this.signature,
                newSignature = arguments.length > 0 ?
                signature : this.getCurrentSignature();
        if (newSignature !== oldSignature) {
            this.signature = newSignature;
            this.fireSignatureChange(newSignature);
        }
    };

    /**
     * Clears the signature pad and the signature on the server-side.
     * @returns {undefined}
     */
    this.clear = function () {
        this.signaturePad.clear();
        this.updateSignature(null);
    };

    /**
     * Cross browser add event.
     * Borrowed from https://github.com/samie/Idle
     * 
     * @param {Element} ob
     * @param {string} type
     * @param {Function} fn
     * @returns {Boolean}
     */
    this.addEvent = function (ob, type, fn) {
        if (ob.addEventListener) {
            ob.addEventListener(type, fn, false);
        } else if (ob.attachEvent) {
            ob.attachEvent('on' + type, fn);
        } else {
            type = 'on' + type;
            if (typeof ob[type] === 'function') {
                fn = (function (f1, f2) {
                    return function () {
                        f1.apply(this, arguments);
                        f2.apply(this, arguments);
                    };
                })(ob[type], fn);
            }
            ob[type] = fn;
            return true;
        }
        return false;
    };

    /**
     * Adds a style class to the given element.
     * @param {Element} el Element the class should be added to.
     * @param {string} clazz The class that should be added.
     * @returns {undefined}
     */
    this.addClass = function (el, clazz) {
        var existing = (el.getAttribute("class") || "").split(" ");
        for (var e = 0; e < existing.length; e++) {
            var eClazz = existing[e];
            if (eClazz === clazz) {
                return;
            }
        }
        existing.push(clazz);
        el.setAttribute("class", existing.join(" "));
    };

    /**
     * Removes a style class from the given element.
     * @param {Element} el Element the class should be removed from.
     * @param {string} clazz The class that should be removed.
     * @returns {undefined}
     */
    this.removeClass = function (el, clazz) {
        var existing = (el.getAttribute("class") || "").split(" "),
                length = existing.length;
        for (var e = 0; e < length; e++) {
            var eClazz = existing[e];
            if (eClazz === clazz) {
                existing.splice(e, 1);
                length--;
                e--;
            }
        }
        el.setAttribute("class", existing.join(" "));
    };

    // call the init method
    this.init();

}