/*
 * Copyright 2016-2017 Plumbr OÜ
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package eu.plumbr.api;

/**
 * <p>
 *   An instance of a span. May represent either:
 * </p>
 * <ul>
 *   <li>
 *     A valid span - {@link #isNull()} returns <code>false</code>
 *   </li>
 *   <li>
 *     A null span - {@link #isNull()} returns <code>true</code> (an instance of {@link eu.plumbr.api.noop.NullSpan}).
 *     When Plumbr Agent is not attached or the API version is too new for the agent, all methods returning a span
 *     return a null span instead. When Plumbr Agent is attached and compatible with the API version, the only source
 *     for null spans is the return value of {@link Plumbr#getCurrentSpan()} when there is no currently active span.
 *   </li>
 * </ul>
 * <p>
 *   None of the methods of this interface throw an exception. Methods returning {@link Span} never return
 *   <code>null</code>. Methods returning {@link String} may return null.
 * </p>
 */
@SuppressWarnings("unused")
public interface Span {

  /**
   * A null span is represented by an instance of a {@link Span} where this method returns <code>true</code>. Methods
   * returning a {@link Span} never return <code>null</code>.
   *
   * @return <code>false</code> if this object refers to a valid span, <code>true</code> if it represents a null span.
   */
  boolean isNull();

  /**
   * @return The ID of the current transaction. Never <code>null</code> (refer to {@link eu.plumbr.api.noop.NullSpan}
   *         to see its return value for spans where {@link #isNull()} returns <code>true</code>).
   */
  String getTransactionId();

  /**
   * @deprecated Instead of application name, clusterId and jvmId properties are now used to determine under which API
   *             in the UI the transaction will be listed. Has no effect on latest agent.
   * @param appName application name.
   * @return current span to chain method calls.
   */
  @Deprecated
  Span setAppName(String appName);

  /**
   * @deprecated Instead of application name, clusterId and jvmId properties are now used to determine under which API
   *             in the UI the transaction will be listed. Always returns <code>null</code> on latest agent.
   * @return application name.
   */
  @Deprecated
  String getAppName();

  /**
   * <p>
   *   Set the service name of the current span. If a transaction consists of multiple spans, then any span may have its
   *   own service name. The service name of the transaction is the service name of the root span. For root spans
   *   started with the API, the service name must be set manually. In case the service name is not set for a root span,
   *   the whole transaction is discarded.
   * </p>
   * <p>
   *   For root spans started automatically by the Plumbr Agent, a service name is always present, but may be overridden
   *   using this method.
   * </p>
   *
   * @param serviceName service name.
   * @return This span to chain method calls.
   */
  Span setServiceName(String serviceName);

  /**
   * Get the service name of the current span. In case of a span started with the API, it returns <code>null</code>
   * until the service name is set using {@link #setServiceName(String)}. For spans started automatically by the Plumbr
   * Agent, it returns the service ID automatically assigned by the agent, unless it is overridden by calling
   * {@link #setServiceName(String)}.
   *
   * @return Service name of the this span.
   */
  String getServiceName();

  /**
   * Set the user ID for the current transaction.
   *
   * @param userId user name.
   * @return current span to chain method calls.
   */
  Span setUserId(String userId);

  /**
   * Get the user ID associated with the current span. In case of a span started with the API, it returns
   * <code>null</code> until the user ID has been set using {@link #setUserId(String)}. For spans started automatically
   * by the Plumbr Agent, the user ID may have already been set automatically.
   *
   * @return User ID associated with this span.
   */
  String getUserId();

  /**
   * <p>
   *   Add metadata with the specified name to the span. The metadata of each span can be seen from the single
   *   transaction view for an unhealthy transaction. A span supports up to 32 metadata fields, any new fields added
   *   after that limit is reached will be ignored. A new value for the same field name overrides the previous one.
   * </p>
   * <p>
   *   Metadata is discarded in serialized format and must therefore be set on the same JVM where {@link #start()} is
   *   called.
   * </p>
   *
   * @param name Name of the metadata field.
   * @param value Value to store. Truncated to 1024 characters.
   * @return current span to chain method calls.
   */
  Span addMetadata(String name, String value);

  /**
   * Returns the value associated with the specified metadata field name. Some metadata fields may be automatically set
   * for spans created internally by the Plumbr Agent, for example Url and HTTP Method.
   *
   * @param name Name of the metadata field.
   * @return Current value of the specified metadata field. <code>null</code> if not set.
   */
  String getMetadata(String name);

  /**
   * <p>
   *   Start the span and make it active. The span must be made active for the root causes to be associated with the
   *   span. One thread may have only one active span at a time. When this is called when another span started by the
   *   API is already active, the previously active span will be ended as if {@link #finish()} was called.
   * </p>
   * <p>
   *   If the previously active span was not created via the API, then it will not be finished, but the root causes from
   *   the current thread will be associated with this span until it finished. The internally started span will be
   *   marked as finished at the time that it normally would, but root causes will not be attached to it while an API
   *   span is active.
   * </p>
   * <p>
   *   Calling this method has no effect if the span has already been started. If a span is deserialized twice, then
   *   calling {@link #start()} on both of the deserialized instances may produce two separate spans.
   * </p>
   */
  void start();

  /**
   * Create the child span within the transaction of this span.
   *
   * @return the new created child span, which must be transferred to another thread or another
   *         JVM before invoking {@link #start()} on it.
   */
  Span createChildSpan();

  /**
   * Indicate that the child span is finished. If the child span is marked as finished by the parent span, then the
   * parent decides the status of its subtree of spans in the span tree of the transaction. If the child span is not
   * marked as finished by the parent span, then if the child span fails, it will mark the transaction as failed.
   *
   * @param child Child span to mark as finished.
   */
  void finishChildSpan(Span child);

  /**
   * <p>
   *   Mark the span as successfully finished.
   * </p>
   * <p>
   *   If it is the root span in the transaction or {@link #finishChildSpan(Span)} has been called for all of its
   *   children, then the transaction is finished as well. If there are child spans for which
   *   {@link #finishChildSpan(Span)} has not yet been called, the end of the last of those will determine the end time
   *   of the transaction.
   * </p>
   * <p>
   *   This method has no effect if the span has not been started, has already finished or was started in a different JVM.
   * </p>
   */
  void finish();

  /**
   * Mark current span as finished with failure and associate the throwable as the root cause. This is equivalent to
   * calling {@link #fail(Throwable, boolean)} with arguments <code>fail(t, false)</code>.
   *
   * @param t {@link Throwable} to associate as root cause. Pass <code>null</code> if there is no exception to blame.
   */
  void fail(Throwable t);

  /**
   * Mark current span as finished with failure and associate the throwable as the root cause.
   *
   * @param t {@link Throwable} to associate as root cause. Pass <code>null</code> if there is no exception to blame.
   * @param failEntireTransaction <code>true</code> means that the transaction which contains this span will be marked
   *                              as failed, regardless of how this span is handled by its parent.
   *                              <code>false</code> means that if in the chain from this span to the root span, any
   *                              span is ended with {@link #finishChildSpan(Span)} by its parent, then the failure of
   *                              the current span has no direct effect on the status of the transaction.
   */
  void fail(Throwable t, boolean failEntireTransaction);
}
