/*
 * Copyright 2016-2017 Plumbr OÜ
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package eu.plumbr.api;

import eu.plumbr.api.noop.NullSpan;

/**
 * <p>
 *   Span serializer is intended to be used when a span needs to be serialized before transferring it to another JVM. On
 *   the other end, it is used to deserialize the transferred span.
 * </p>
 * <p>
 *   Span metadata must be set on the same instance of {@link Span} on which {@link Span#start()} is called as it is
 *   discarded in the serialized format. The service name, application name and user identity are also discarded in the
 *   serialized format, however there is no need to set those properties for a child span as they only have an effect
 *   when used on a root span.
 * </p>
 */
@SuppressWarnings("unused")
public class SpanSerializer {
  public static final int API_VERSION = 7;

  /**
   * Deserializes a {@link Span} from a byte array. The contents of the byte array must have been obtained by previously
   * calling {@link #toByteArray(Span)}.
   *
   * @param buffer Byte array representing the serialized span.
   * @return Instance of a valid {@link Span} if the input was valid, Plumbr Agent is attached and is not too old to
   *         support this API version. In any other case, returns a null span ({@link Span#isNull()} returns
   *         <code>true</code>).
   */
  public static Span fromByteArray(byte[] buffer) {
    return new NullSpan();
  }

  /**
   * Deserializes a {@link Span} from a base-64 string representation. The contents of the byte array must have been
   * obtained by previously calling {@link #toBase64(Span)}.
   *
   * @param string Base-64 formatted string representing the serialized span.
   * @return Instance of a valid {@link Span} if the input was valid, Plumbr Agent is attached and is not too old to
   *         support this API version. In any other case, returns a null span ({@link Span#isNull()} returns
   *         <code>true</code>).
   */
  public static Span fromBase64(String string) {
    return new NullSpan();
  }

  /**
   * Serializes a {@link Span} to a byte array. Use {@link #fromByteArray(byte[])} (String)} to deserialize.
   *
   * @param span Span to serialize.
   * @return Serialized format of the specified span if the specified span is non-null, Plumbr Agent is attached and is
   *         not too old to support this API version. In any other case, returns an empty byte array.
   */
  public static byte[] toByteArray(Span span) {
    return new byte[0];
  }

  /**
   * Serializes a {@link Span} to a base-64 string representation. Use {@link #fromBase64(String)} to deserialize.
   *
   * @param span Span to serialize.
   * @return Serialized format of the specified span if the specified span is non-null, Plumbr Agent is attached and is
   *         not too old to support this API version. In any other case, returns an empty string.
   */
  public static String toBase64(Span span) {
    return "";
  }
}
