// $ANTLR 2.7.7 (20060906): "Semantic.g" -> "Semantic.java"$

    package com.sun.jdo.spi.persistence.support.sqlstore.query.jqlc;

    import java.util.Locale;
    import java.util.ResourceBundle;
    import java.util.Collection;

    import org.glassfish.persistence.common.I18NHelper;

    import com.sun.jdo.spi.persistence.support.sqlstore.query.util.type.TypeTable;
    import com.sun.jdo.spi.persistence.support.sqlstore.query.util.type.Type;
    import com.sun.jdo.spi.persistence.support.sqlstore.query.util.type.ClassType;
    import com.sun.jdo.spi.persistence.support.sqlstore.query.util.type.FieldInfo;
    import com.sun.jdo.spi.persistence.support.sqlstore.query.util.type.NumericType;
    import com.sun.jdo.spi.persistence.support.sqlstore.query.util.type.NumericWrapperClassType;

    import com.sun.jdo.spi.persistence.support.sqlstore.query.util.scope.SymbolTable;
    import com.sun.jdo.spi.persistence.support.sqlstore.query.util.scope.Definition;
    import com.sun.jdo.spi.persistence.support.sqlstore.query.util.scope.TypeName;
    import com.sun.jdo.spi.persistence.support.sqlstore.query.util.scope.Variable;
    import com.sun.jdo.spi.persistence.support.sqlstore.query.util.scope.Parameter;
    import com.sun.jdo.spi.persistence.support.sqlstore.query.util.scope.Field;

import antlr.TreeParser;
import antlr.Token;
import antlr.collections.AST;
import antlr.RecognitionException;
import antlr.ANTLRException;
import antlr.NoViableAltException;
import antlr.MismatchedTokenException;
import antlr.SemanticException;
import antlr.collections.impl.BitSet;
import antlr.ASTPair;
import antlr.collections.impl.ASTArray;


/**
 * This class defines the semantic analysis of the JQL compiler.
 * Input of this pass is the AST as produced by the parser,
 * that consists of JQLAST nodes.
 * The result is a typed JQLAST tree.
 *
 * @author  Michael Bouschen
 * @author  Shing Wai Chan
 * @version 0.1
 */
public class Semantic extends antlr.TreeParser       implements SemanticTokenTypes
 {

    /**
     * I18N support
     */
    protected final static ResourceBundle messages =
        I18NHelper.loadBundle(Semantic.class);

    /**
     * symbol table handling names of fields, variables and parameters
     */
    protected SymbolTable symtab;

    /**
     * symbol table handling type names (candidate class and imported names)
     */
    protected SymbolTable typeNames;

    /**
     * type table
     */
    protected TypeTable typetab;

    /**
     * query parameter table
     */
    protected ParameterTable paramtab;

    /**
     * variable table
     */
    protected VariableTable vartab;

    /**
     *
     */
    protected ErrorMsg errorMsg;

    /**
     * Result class for this query. This class is set by setClass.
     */
    protected ClassType candidateClass;

    /**
     *
     */
    public void init(TypeTable typetab, ParameterTable paramtab, ErrorMsg errorMsg)
    {
        this.symtab = new SymbolTable();
        this.typeNames = new SymbolTable();
        this.vartab = new VariableTable(errorMsg);
        this.typetab = typetab;
        this.paramtab = paramtab;
        this.errorMsg = errorMsg;
    }

    /**
     *
     */
    public void reportError(RecognitionException ex) {
        errorMsg.fatal("Error: " + ex); //NOI18N
    }

    /**
     *
     */
    public void reportError(String s) {
        errorMsg.fatal("Error: " + s); //NOI18N
    }

    /**
     * Combines partial ASTs into one query AST.
     */
    public JQLAST createQueryAST(JQLAST candidateClass,
                                 JQLAST importsAST,
                                 JQLAST paramsAST,
                                 JQLAST varsAST,
                                 JQLAST orderingAST,
                                 JQLAST resultAST,
                                 JQLAST filterAST)
    {
        JQLAST query = new JQLAST(QUERY, "query", null); //NOI18N
        if (candidateClass != null)
            query.addChild(candidateClass);
        if (importsAST != null)
            query.addChild(importsAST);
        if (paramsAST != null)
            query.addChild(paramsAST);
        if (varsAST != null)
            query.addChild(varsAST);
        if (orderingAST != null)
            query.addChild(orderingAST);
        if (resultAST != null)
            query.addChild(resultAST);
        if (filterAST != null)
            query.addChild(filterAST);
        return query;
    }

    /**
     * Creates the CLASS_DEF AST that represents the setClass value.
     */
    public JQLAST checkCandidateClass(Class candidateClass)
    {
        Type type = typetab.checkType(candidateClass);
        if (type == null)
        {
            errorMsg.fatal(I18NHelper.getMessage(messages,
                "jqlc.semantic.checkcandidateclass.unknowntype", //NOI18N
                String.valueOf(candidateClass)));
        }
        return new JQLAST(CLASS_DEF, "classDef", type); //NOI18N
    }

    /**
     * This method analyses the expression of a single ordering definition.
     * It checks whether the expression
     * - is valid (see checkValidOrderingExpr)
     * - is of a orderable type
     * @param expr the expression of an ordering definition
     */
    protected void analyseOrderingExpression(JQLAST expr)
    {
        checkValidOrderingExpr(expr);
        Type exprType = expr.getJQLType();
        if (!exprType.isOrderable())
        {
            errorMsg.error(expr.getLine(), expr.getColumn(),
                I18NHelper.getMessage(messages, "jqlc.semantic.analyseorderingexpression.notorderable", //NOI18N
                    exprType.getName()));
            expr.setJQLType(typetab.errorType);
        }
    }

    /**
     * This method checks whether the ordering expression is valid.
     * The following expressions are valid:
     * - field access using the this object
     * - navigation from a field of the this object
     * @param expr the ordering definition
     */
    private void checkValidOrderingExpr(JQLAST expr)
    {
        switch(expr.getType())
        {
        case THIS:
        case VARIABLE:
            //OK;
            break;
        case FIELD_ACCESS:
        case NAVIGATION:
            JQLAST child = (JQLAST)expr.getFirstChild();
            if (child != null)
            {
                // check first part of dot expr
                checkValidOrderingExpr(child);
            }
            break;
        default:
            errorMsg.error(expr.getLine(), expr.getColumn(),
                I18NHelper.getMessage(messages, "jqlc.semantic.checkvalidorderingexpr.invalidordering", //NOI18N
                    expr.getText()));
        }
    }

    /**
     * This method checks whether the result expression is valid.
     * The following expressions are valid:
     * - field access using the this object
     * - navigation from a field of the this object
     * - variable access
     * - distinct expression
     * - aggreagte expression
     * @param expr the result expression
     */
    private void checkValidResultExpr(JQLAST expr)
    {
        switch(expr.getType())
        {
        case THIS:
            //OK;
            break;
        case FIELD_ACCESS:
        case NAVIGATION:
            JQLAST child = (JQLAST)expr.getFirstChild();
            if (child != null)
            {
                // check first part of dot expr
                checkValidResultExpr(child);
            }
            break;
        case VARIABLE:
            // OK
            break;
        case DISTINCT:
            checkValidResultExpr((JQLAST)expr.getFirstChild());
            break;
        case AVG:
        case SUM:
            if (!typetab.isNumberType(expr.getJQLType()) ||
                    typetab.isCharType(expr.getJQLType())) {
                errorMsg.error(expr.getLine(), expr.getColumn(),
                    I18NHelper.getMessage(messages,
                    "jqlc.semantic.checkvalidresultexpr.invalidavgsumexpr", //NOI18N
                    expr.getJQLType().getName(), expr.getText()));
            }
            checkValidResultExpr((JQLAST)expr.getFirstChild());
            break;
        case MAX:
        case MIN:
            if (!expr.getJQLType().isOrderable()) {
                errorMsg.error(expr.getLine(), expr.getColumn(),
                    I18NHelper.getMessage(messages,
                    "jqlc.semantic.checkvalidresultexpr.invalidminmaxexpr", //NOI18N
                    expr.getJQLType().getName(), expr.getText()));
            }
            checkValidResultExpr((JQLAST)expr.getFirstChild());
            break;
        case COUNT:
            checkValidResultExpr((JQLAST)expr.getFirstChild());
            break;
        default:
            errorMsg.error(expr.getLine(), expr.getColumn(),
                I18NHelper.getMessage(messages, "jqlc.semantic.checkvalidresultexpr.invalidresult", //NOI18N
                    expr.getText()));
        }
    }

    /**
     *  Checks that result and ordering are compatible.
     *  If the query result is a field, then it must be the same as ordering
     *  item. If the query is an object, then ordering expression must
     *  have the same navigation prefix of the result expression.
     */
    private void checkResultOrdering(JQLAST result, JQLAST ordering) {
        if (ordering == null) {
            return;
        }

        AST resultReturnAST = result;
        boolean hasResultDistinct = false;
        if (resultReturnAST == null) { // distinct THIS
            resultReturnAST = new JQLAST(THIS, "this", candidateClass);
            hasResultDistinct = true;
        }

        // skip RESULT_DEF node
        if (resultReturnAST.getType() == RESULT_DEF) {
            resultReturnAST = resultReturnAST.getFirstChild();
        }
        // skip DISTINCT node
        if (resultReturnAST.getType() == DISTINCT) {
            resultReturnAST = resultReturnAST.getFirstChild();
            hasResultDistinct = true;
        }

        if (!hasResultDistinct) {
            return;
        }

        if (resultReturnAST.getType() == FIELD_ACCESS) {
            StringBuilder buf = new StringBuilder();
            genPathExpression(resultReturnAST, buf);
            String resultReturnPathExpr = buf.toString();

            for (AST sibling = ordering;
                    sibling != null && sibling.getType() == ORDERING_DEF;
                    sibling = sibling.getNextSibling()) {

                // share buf
                buf.setLength(0);
                genPathExpression(sibling.getFirstChild().getNextSibling(), buf);
                String orderingItemExpr = buf.toString();
                if (!orderingItemExpr.equals(resultReturnPathExpr)) {
                    errorMsg.error(ordering.getLine(), ordering.getColumn(),
                        I18NHelper.getMessage(messages,
                            "jqlc.semantic.checkresultordering.invalidorderingfordistinctresultfield", //NOI18N
                            resultReturnPathExpr, orderingItemExpr));
                }
            }
        } else if (resultReturnAST.getType() == NAVIGATION ||
                resultReturnAST.getType() ==  THIS ) {
            StringBuilder buf = new StringBuilder();
            genPathExpression(resultReturnAST, buf);
            String resultReturnPathExpr = buf.toString();

            for (AST sibling = ordering;
                    sibling != null && sibling.getType() == ORDERING_DEF;
                    sibling = sibling.getNextSibling()) {

                // share buf
                buf.setLength(0);
                genPathExpression(sibling.getFirstChild().getNextSibling().getFirstChild(), buf);
                String orderingRootExpr = buf.toString();
                if (!orderingRootExpr.equals(resultReturnPathExpr)) {
                    buf.setLength(0);
                    genPathExpression(sibling.getFirstChild().getNextSibling(), buf);
                    errorMsg.error(ordering.getLine(), ordering.getColumn(),
                        I18NHelper.getMessage(messages,
                            "jqlc.semantic.checkresultordering.invalidorderingfordistinctresult", //NOI18N
                            resultReturnPathExpr, buf.toString()));
                }
            }
        }
    }

    /**
     * Form a string representation of a dot expression and append to given
     * StringBuilder.
     * @param ast the AST node representing the root the of the expression
     * @param buf the StringBuilder that will have result of path expression
     * append
     */
    private void genPathExpression(AST ast, StringBuilder buf) {
        if (ast == null) {
            return;
        }
        switch (ast.getType()) {
            case FIELD_ACCESS:
            case STATIC_FIELD_ACCESS:
            case NAVIGATION:
                AST left = ast.getFirstChild();
                AST right = left.getNextSibling();
                genPathExpression(left, buf);
                buf.append('.');
                genPathExpression(right, buf);
                break;
            default:
                buf.append(ast.getText());
                break;
        }
    }


    /**
     * This method analyses a dot expression of the form expr.ident or
     * expr.ident(params) where expr itself can again be a dot expression.
     * It checks whether the dot expression is
     * - part of a qualified class name specification
     * - field access,
     * - a method call
     * The method returns a temporary single AST node that is defined with a
     * specific token type (field access, method call, etc.). This node also
     * contains the type of the dot expression.
     * @param expr the left hand side of the dot expression
     * @param ident the right hand side of the dot expression
     * @param args arguments (in the case of a call)
     * @return AST node representing the specialized dot expr
     */
    protected JQLAST analyseDotExpr(JQLAST dot, JQLAST expr, JQLAST ident, JQLAST args)
    {
        Type exprType = expr.getJQLType();
        String name = ident.getText();
        dot.setText(expr.getText() + '.' + name);
        if (exprType instanceof ClassType)
        {
            // left expression is of a class type
            ClassType classType = (ClassType)exprType;
            if (args == null)
            {
                // no paranethesis specified => field access
                FieldInfo fieldInfo = classType.getFieldInfo(name);
                if (fieldInfo == null)
                {
                    errorMsg.error(ident.getLine(), ident.getColumn(),
                                   I18NHelper.getMessage(messages, "jqlc.semantic.generic.unknownfield",  //NOI18N
                                                         ident.getText(), exprType.getName()));
                    dot.setJQLType(typetab.errorType);
                    ident.setJQLType(typetab.errorType);
                    return dot;
                }
                else if (expr.getType() == TYPENAME)
                {
                    // access of the form: className.staticField
                    return analyseStaticFieldAccess(dot, expr, ident, classType, fieldInfo);
                }
                else
                {
                    // access of the form: object.field
                    return analyseFieldAccess(dot, expr, ident, classType, fieldInfo);
                }
            }
            else
            {
                // parenthesis specified => method call
                if (typetab.isCollectionType(exprType))
                {
                    return analyseCollectionCall(dot, expr, ident, args);
                }
                else if (exprType.equals(typetab.stringType))
                {
                    return analyseStringCall(dot, expr, ident, args);
                }
                else if (typetab.isJavaLangMathType(exprType))
                {
                    return analyseMathCall(dot, expr, ident, args);
                }
                errorMsg.error(dot.getLine(), dot.getColumn(),
                               I18NHelper.getMessage(messages, "jqlc.semantic.generic.invalidmethodcall")); //NOI18N
                dot.setJQLType(typetab.errorType);
                return dot;
            }
        }
        else
        {
            errorMsg.error(expr.getLine(), expr.getColumn(),
                           I18NHelper.getMessage(messages, "jqlc.semantic.analysedotexpr.classexprexpected", //NOI18N
                                                 ident.getText(), exprType.getName()));
            dot.setJQLType(typetab.errorType);
            return dot;
        }
    }

    /**
     *
     */
    protected JQLAST analyseFieldAccess(JQLAST access, JQLAST objectExpr, JQLAST ident,
                                        ClassType classType, FieldInfo fieldInfo)
    {
        String name = ident.getText();
        Type fieldType = fieldInfo.getType();
        if (classType.isPersistenceCapable())
        {
            if (!fieldInfo.isPersistent())
            {
                errorMsg.error(ident.getLine(), ident.getColumn(),
                    I18NHelper.getMessage(messages,
                        "jqlc.semantic.analysefieldaccess.nonperistentfield", name, classType.getName())); //NOI18N
            }
            if (typetab.isPersistenceCapableType(fieldType))
            {
                access.setType(NAVIGATION);
            }
            else
            {
                access.setType(FIELD_ACCESS);
            }
        }
        else
        {
            if (!fieldInfo.isPublic())
            {
                errorMsg.error(ident.getLine(), ident.getColumn(),
                    I18NHelper.getMessage(messages,
                        "jqlc.semantic.analysefieldaccess.nonpublicfield", name, classType.getName())); //NOI18N
            }
            access.setType(FIELD_ACCESS);
        }
        access.setText(objectExpr.getText() + '.' + name);
        access.setJQLType(fieldType);
        ident.setJQLType(fieldType);
        access.setFirstChild(objectExpr);
        objectExpr.setNextSibling(ident);
        return access;
    }

    /**
     *
     */
    protected JQLAST analyseStaticFieldAccess(JQLAST access, JQLAST typename, JQLAST ident,
                                              ClassType classType, FieldInfo fieldInfo)
    {
        String name = ident.getText();
        Type fieldType = fieldInfo.getType();
        if (!fieldInfo.isStatic())
        {
            errorMsg.error(ident.getLine(), ident.getColumn(),
                I18NHelper.getMessage(messages, "jqlc.semantic.analysestaticfieldaccess.staticreference", //NOI18N
                    ident.getText(), classType.getName()));
        }
        if (!fieldInfo.isPublic())
        {
            errorMsg.error(ident.getLine(), ident.getColumn(),
                I18NHelper.getMessage(messages,
                    "jqlc.semantic.analysestaticfieldaccess.nonpublicfield", name, classType.getName())); //NOI18N
        }
        access.setType(STATIC_FIELD_ACCESS);
        access.setText(typename.getText() + '.' + name);
        access.setJQLType(fieldType);
        ident.setJQLType(fieldType);
        access.setFirstChild(typename);
        typename.setNextSibling(ident);
        return access;
    }

    /**
     * This method analyses and identifier defined in the current scope:
     * - a field, variable or parameter defined in the symbol table
     * - a type define in a separate symbol table for type names
     * @param ident the identifier AST
     * @param def the entry in the symbol table of the type names tables
     * @return AST node representing a defined identifier
     */
    protected JQLAST analyseDefinedIdentifier(JQLAST ident, Definition def)
    {
        Type type = def.getType();
        if (def instanceof Variable)
        {
            ident.setType(VARIABLE);
        }
        else if (def instanceof Parameter)
        {
            ident.setType(PARAMETER);
        }
        else if (def instanceof Field)
        {
            FieldInfo fieldInfo = ((Field)def).getFieldInfo();
            JQLAST fieldAccessAST = ident;
            JQLAST identAST = new JQLAST(ident);
            if (fieldInfo.isStatic())
            {
                JQLAST typeNameAST = new JQLAST(TYPENAME, candidateClass.getName(), candidateClass);
                ident = analyseStaticFieldAccess(fieldAccessAST, typeNameAST,
                                                 identAST, candidateClass, fieldInfo);
            }
            else
            {
                JQLAST thisAST = new JQLAST(THIS, "this", candidateClass); //NOI18N
                ident = analyseFieldAccess(fieldAccessAST, thisAST,
                                           identAST, candidateClass, fieldInfo);
            }
        }
        else if (def instanceof TypeName)
        {
            ident.setType(TYPENAME);
            ident.setText(((TypeName)def).getQualifiedName());
        }
        else
        {
            type = typetab.errorType;
            errorMsg.fatal(I18NHelper.getMessage(messages,
                "jqlc.semantic.analysedefinedidentifier.illegalident", //NOI18N
                String.valueOf(def)));
        }
        ident.setJQLType(type);
        return ident;
    }

    /**
     * Analyses a call for an object that implements Collection.
     * Currently, contains is the only valid Collection method in a query filter.
     */
    protected JQLAST analyseCollectionCall(JQLAST dot, JQLAST collection, JQLAST method, JQLAST args)
    {
        String methodName = method.getText();
        JQLAST firstArg = (JQLAST)args.getFirstChild();
        if (methodName.equals("contains")) //NOI18N
        {
            checkContainsArgs(collection, method, firstArg);
            dot.setType(CONTAINS);
            dot.setJQLType(typetab.booleanType);
            dot.setFirstChild(collection);
            collection.setNextSibling(firstArg);
            return dot;
        }
        else if (methodName.equals("isEmpty")) //NOI18N
        {
            // isEmpty does not take parameters
            checkNoArgs(method, firstArg);
            dot.setType(IS_EMPTY);
            dot.setJQLType(typetab.booleanType);
            dot.setFirstChild(collection);
            collection.setNextSibling(null);
            return dot;
        }

        errorMsg.error(dot.getLine(), dot.getColumn(),
            I18NHelper.getMessage(messages, "jqlc.semantic.generic.invalidmethodcall"));  //NOI18N
        dot.setJQLType(typetab.errorType);
        return dot;
    }

    /**
     * Analyses a call for an object of type String.
     * Currently startsWith and endsWith are the only valid String methods in a query filter
     */
    protected JQLAST analyseStringCall(JQLAST dot, JQLAST string, JQLAST method, JQLAST args)
    {
        String methodName = method.getText();
        JQLAST firstArg = (JQLAST)args.getFirstChild();
        if (methodName.equals("startsWith")) //NOI18N
        {
            dot.setType(STARTS_WITH);
            checkOneStringArg(method, firstArg);
            dot.setJQLType(typetab.booleanType);
            dot.setFirstChild(string);
            string.setNextSibling(firstArg);
        }
        else if (methodName.equals("endsWith")) //NOI18N
        {
            dot.setType(ENDS_WITH);
            checkOneStringArg(method, firstArg);
            dot.setJQLType(typetab.booleanType);
            dot.setFirstChild(string);
            string.setNextSibling(firstArg);
        }
        else if (methodName.equals("like")) //NOI18N
        {
            checkLikeArgs(method, firstArg);
            dot.setType(LIKE);
            dot.setJQLType(typetab.booleanType);
            dot.setFirstChild(string);
            string.setNextSibling(firstArg);
        }
        else if (methodName.equals("substring")) //NOI18N
        {
            checkTwoIntArgs(method, firstArg);
            dot.setType(SUBSTRING);
            dot.setJQLType(typetab.stringType);
            dot.setFirstChild(string);
            string.setNextSibling(firstArg);
        }
        else if (methodName.equals("indexOf")) //NOI18N
        {
            checkIndexOfArgs(method, firstArg);
            dot.setType(INDEXOF);
            dot.setJQLType(typetab.intType);
            dot.setFirstChild(string);
            string.setNextSibling(firstArg);
        }
        else if (methodName.equals("length")) //NOI18N
        {
            // length does not take parameters
            checkNoArgs(method, firstArg);
            dot.setType(LENGTH);
            dot.setJQLType(typetab.intType);
            dot.setFirstChild(string);
            string.setNextSibling(null);
        }
        else
        {
            errorMsg.error(dot.getLine(), dot.getColumn(),
                I18NHelper.getMessage(messages, "jqlc.semantic.generic.invalidmethodcall"));  //NOI18N
            dot.setJQLType(typetab.errorType);
        }
        return dot;
    }

    /**
     * Analyses a java.lang.Math call.
     */
    protected JQLAST analyseMathCall(JQLAST dot, JQLAST type, JQLAST method, JQLAST args)
    {
        String methodName = method.getText();
        JQLAST firstArg = (JQLAST)args.getFirstChild();
        if (methodName.equals("abs")) //NOI18N
        {
            checkAbsArgs(method, firstArg);
            dot.setType(ABS);
            dot.setJQLType(firstArg.getJQLType());
            dot.setFirstChild(firstArg);
        }
        else if (methodName.equals("sqrt")) //NOI18N
        {
            checkSqrtArgs(method, firstArg);
            dot.setType(SQRT);
            dot.setJQLType(firstArg.getJQLType());
            dot.setFirstChild(firstArg);
        }
        else
        {
            errorMsg.error(dot.getLine(), dot.getColumn(),
                I18NHelper.getMessage(messages, "jqlc.semantic.generic.invalidmethodcall"));  //NOI18N
            dot.setJQLType(typetab.errorType);
        }
        return dot;
    }

    /**
     * This method checks the specified node (args) representing an empty
     * argument list.
     */
    protected void checkNoArgs(JQLAST method, JQLAST firstArg)
    {
        if (firstArg != null)
        {
            errorMsg.error(firstArg.getLine(), firstArg.getColumn(),
                I18NHelper.getMessage(messages,
                    "jqlc.semantic.generic.arguments.numbermismatch")); //NOI18N
        }
    }

    /**
     * This method checks the specified node (args) representing an argument
     * list which consists of a single argument of type String.
     */
    protected void checkOneStringArg(JQLAST method, JQLAST firstArg)
    {
        if (firstArg == null)
        {
            errorMsg.error(method.getLine(), method.getColumn(),
                I18NHelper.getMessage(messages,
                    "jqlc.semantic.generic.arguments.numbermismatch")); //NOI18N
        }
        else if (firstArg.getNextSibling() != null)
        {
            JQLAST nextArg = (JQLAST)firstArg.getNextSibling();
            errorMsg.error(nextArg.getLine(), nextArg.getColumn(),
                I18NHelper.getMessage(messages,
                    "jqlc.semantic.generic.arguments.numbermismatch")); //NOI18N
        }
        else
        {
            Type argType = firstArg.getJQLType();
            if (!argType.equals(typetab.stringType))
            {
                errorMsg.error(firstArg.getLine(), firstArg.getColumn(),
                    I18NHelper.getMessage(messages,
                        "jqlc.semantic.generic.arguments.typemismatch", //NOI18N
                        argType.getName(), typetab.stringType.getName()));
            }
        }
    }

    /**
     * This method checks the specified node (args) representing a valid contains
     * argument list: one argument denoting a variable.
     */
    protected void checkContainsArgs(JQLAST collection, JQLAST method, JQLAST firstArg)
    {
        if (firstArg == null)
        {
            errorMsg.error(method.getLine(), method.getColumn(),
                I18NHelper.getMessage(messages,
                    "jqlc.semantic.generic.arguments.numbermismatch")); //NOI18N
        }
        else if (firstArg.getNextSibling() != null)
        {
            JQLAST nextArg = (JQLAST)firstArg.getNextSibling();
            errorMsg.error(nextArg.getLine(), nextArg.getColumn(),
                I18NHelper.getMessage(messages,
                    "jqlc.semantic.generic.arguments.numbermismatch")); //NOI18N
        }
        else if (firstArg.getType() != VARIABLE)
        {
            errorMsg.unsupported(firstArg.getLine(), firstArg.getColumn(),
                I18NHelper.getMessage(messages,
                    "jqlc.semantic.analysecollectioncall.nonvariable")); //NOI18N
        }
        else
        {
            FieldInfo collectionFieldInfo = getCollectionField(collection);
            if (collectionFieldInfo == null)
            {
                errorMsg.unsupported(collection.getLine(), collection.getColumn(),
                    I18NHelper.getMessage(messages,
                        "jqlc.semantic.analysecollectioncall.unsupportedcollectionexpr", //NOI18N
                    collection.getText()));
            }
            else if (!collectionFieldInfo.isRelationship())
            {
                // check compatibilty of collection element type and type of variable
                errorMsg.error(collection.getLine(), collection.getColumn(),
                    I18NHelper.getMessage(messages,
                        "jqlc.semantic.analysecollectioncall.relationshipexpected", //NOI18N
                        collectionFieldInfo.getName()));
            }
            Type variableType = firstArg.getJQLType();
            Type elementType = collectionFieldInfo.getAssociatedClass();
            if (!elementType.isCompatibleWith(variableType))
            {
                errorMsg.error(collection.getLine(), collection.getColumn(),
                    I18NHelper.getMessage(messages,
                        "jqlc.semantic.analysecollectioncall.typemismatch", //NOI18N
                        elementType.getName(), variableType.getName()));
            }
        }
    }

    /**
     * This method checks the specified node (args) representing a valid like
     * argument list: a string argument plus an optional char argument.
     */
    protected void checkLikeArgs(JQLAST method, JQLAST firstArg)
    {
        if (firstArg == null)
        {
            errorMsg.error(method.getLine(), method.getColumn(),
                I18NHelper.getMessage(messages,
                    "jqlc.semantic.generic.arguments.numbermismatch")); //NOI18N
        }
        else if ((firstArg.getNextSibling() != null) &&
            (firstArg.getNextSibling().getNextSibling() != null))
        {
            JQLAST nextArg = (JQLAST)firstArg.getNextSibling().getNextSibling();
            errorMsg.error(nextArg.getLine(), nextArg.getColumn(),
                I18NHelper.getMessage(messages,
                    "jqlc.semantic.generic.arguments.numbermismatch")); //NOI18N
        }
        else
        {
            // check type of first arg
            Type firstArgType = firstArg.getJQLType();
            if (!firstArgType.equals(typetab.stringType))
            {
                errorMsg.error(firstArg.getLine(), firstArg.getColumn(),
                    I18NHelper.getMessage(messages,
                        "jqlc.semantic.generic.arguments.typemismatch", //NOI18N
                        firstArgType.getName(), typetab.stringType.getName()));
            }
            // check type of second arg (if available)
            JQLAST secondArg = (JQLAST)firstArg.getNextSibling();
            if (secondArg != null)
            {
                Type secondArgType = secondArg.getJQLType();
                if (!typetab.isCharType(secondArgType))
                {
                    errorMsg.error(secondArg.getLine(), secondArg.getColumn(),
                        I18NHelper.getMessage(messages,
                            "jqlc.semantic.generic.arguments.typemismatch", //NOI18N
                        secondArgType.getName(), typetab.charType.getName()));
                }
            }
        }
    }

    /**
     * This method checks the specified node (args) representing an argument
     * list which consists of two integer arguments.
     */
    protected void checkTwoIntArgs(JQLAST method, JQLAST firstArg)
    {
        if (firstArg == null)
        {
            // no args specified
            errorMsg.error(method.getLine(), method.getColumn(),
                I18NHelper.getMessage(messages,
                    "jqlc.semantic.generic.arguments.numbermismatch")); //NOI18N
        }
        else if (firstArg.getNextSibling() == null)
        {
            // one arg specified
            errorMsg.error(method.getLine(), method.getColumn(),
                I18NHelper.getMessage(messages,
                    "jqlc.semantic.generic.arguments.numbermismatch")); //NOI18N
        }
        else if (firstArg.getNextSibling().getNextSibling() != null)
        {
            // more than two args specified
            JQLAST nextArg = (JQLAST)firstArg.getNextSibling().getNextSibling();
            errorMsg.error(nextArg.getLine(), nextArg.getColumn(),
                I18NHelper.getMessage(messages,
                    "jqlc.semantic.generic.arguments.numbermismatch")); //NOI18N
        }
        else
        {
            // specified two args
            // check type of first arg
            Type firstArgType = firstArg.getJQLType();
            if (!typetab.isIntType(firstArgType))
            {
                errorMsg.error(firstArg.getLine(), firstArg.getColumn(),
                    I18NHelper.getMessage(messages,
                        "jqlc.semantic.generic.arguments.typemismatch", //NOI18N
                        firstArgType.getName(), typetab.intType.getName()));
            }
            // check type of second arg
            JQLAST secondArg = (JQLAST)firstArg.getNextSibling();
            Type secondArgType = firstArg.getJQLType();
            if (!typetab.isIntType(secondArgType))
            {
                errorMsg.error(secondArg.getLine(), secondArg.getColumn(),
                    I18NHelper.getMessage(messages,
                        "jqlc.semantic.generic.arguments.typemismatch", //NOI18N
                        secondArgType.getName(), typetab.intType.getName()));
            }
        }
    }

    /**
     * This method checks the specified node (args) representing a valid indexOf
     * argument list: a string argument plus an optional char argument.
     */
    protected void checkIndexOfArgs(JQLAST method, JQLAST firstArg)
    {
        if (firstArg == null)
        {
            errorMsg.error(method.getLine(), method.getColumn(),
                I18NHelper.getMessage(messages,
                    "jqlc.semantic.generic.arguments.numbermismatch")); //NOI18N
        }
        else if ((firstArg.getNextSibling() != null) &&
            (firstArg.getNextSibling().getNextSibling() != null))
        {
            JQLAST nextArg = (JQLAST)firstArg.getNextSibling().getNextSibling();
            errorMsg.error(nextArg.getLine(), nextArg.getColumn(),
                I18NHelper.getMessage(messages,
                    "jqlc.semantic.generic.arguments.numbermismatch")); //NOI18N
        }
        else
        {
            // check type of first arg
            Type firstArgType = firstArg.getJQLType();
            if (!firstArgType.equals(typetab.stringType))
            {
                errorMsg.error(firstArg.getLine(), firstArg.getColumn(),
                    I18NHelper.getMessage(messages,
                        "jqlc.semantic.generic.arguments.typemismatch", //NOI18N
                        firstArgType.getName(), typetab.stringType.getName()));
            }
            // check type of second arg (if available)
            JQLAST secondArg = (JQLAST)firstArg.getNextSibling();
            if (secondArg != null)
            {
                Type secondArgType = secondArg.getJQLType();
                if (!typetab.isIntType(secondArgType))
                {
                    errorMsg.error(secondArg.getLine(), secondArg.getColumn(),
                        I18NHelper.getMessage(messages,
                            "jqlc.semantic.generic.arguments.typemismatch", //NOI18N
                            secondArgType.getName(), typetab.intType.getName()));
                }
            }
        }
    }

    /**
     * This method checks the specified node (args) representing a valid abs
     * argument list: a single number argument.
     */
    protected void checkAbsArgs(JQLAST method, JQLAST firstArg)
    {
        if (firstArg == null)
        {
            errorMsg.error(method.getLine(), method.getColumn(),
                I18NHelper.getMessage(messages,
                    "jqlc.semantic.generic.arguments.numbermismatch")); //NOI18N
        }
        else if (firstArg.getNextSibling() != null)
        {
            JQLAST nextArg = (JQLAST)firstArg.getNextSibling();
                errorMsg.error(nextArg.getLine(), nextArg.getColumn(),
                I18NHelper.getMessage(messages,
                    "jqlc.semantic.generic.arguments.numbermismatch")); //NOI18N
        }
        else
        {
            Type argType = firstArg.getJQLType();
            if (!typetab.isNumberType(argType))
            {
                errorMsg.error(firstArg.getLine(), firstArg.getColumn(),
                    I18NHelper.getMessage(messages,
                        "jqlc.semantic.generic.arguments.typemismatch", //NOI18N
                        argType.getName(), "number type"));
            }
        }
    }

    /**
     * This method checks the specified node (args) representing a valid sqrt
     * argument list: a single argument of type double or Double.
     */
    protected void checkSqrtArgs(JQLAST method, JQLAST firstArg)
    {
        if (firstArg == null)
        {
            errorMsg.error(method.getLine(), method.getColumn(),
                I18NHelper.getMessage(messages,
                    "jqlc.semantic.generic.arguments.numbermismatch")); //NOI18N
        }
        else if (firstArg.getNextSibling() != null)
        {
            JQLAST nextArg = (JQLAST)firstArg.getNextSibling();
                errorMsg.error(nextArg.getLine(), nextArg.getColumn(),
                I18NHelper.getMessage(messages,
                    "jqlc.semantic.generic.arguments.numbermismatch")); //NOI18N
        }
        else
        {
            Type argType = firstArg.getJQLType();
            if (!typetab.isDoubleType(argType))
            {
                errorMsg.error(firstArg.getLine(), firstArg.getColumn(),
                    I18NHelper.getMessage(messages,
                        "jqlc.semantic.generic.arguments.typemismatch", //NOI18N
                        argType.getName(), "double or Double"));
            }
        }
    }

    /**
     *
     */
    protected FieldInfo getCollectionField(JQLAST expr)
    {
        JQLAST child = (JQLAST)expr.getFirstChild();
        switch (expr.getType())
        {
        case FIELD_ACCESS:
        case NAVIGATION:
            if ((child != null) && (child.getNextSibling() != null))
            {
                ClassType classType = (ClassType)child.getJQLType();
                String fieldName = child.getNextSibling().getText();
                return classType.getFieldInfo(fieldName);
            }
            errorMsg.fatal(I18NHelper.getMessage(messages, "jqlc.semantic.getcollectionfield.missingchildren")); //NOI18N
            break;
        case TYPECAST:
            if ((child != null) && (child.getNextSibling() != null))
            {
                return getCollectionField((JQLAST)child.getNextSibling());
            }
            errorMsg.fatal(I18NHelper.getMessage(messages, "jqlc.semantic.getcollectionfield.missingchildren")); //NOI18N
            break;
        }
        return null;
    }

    /**
     * Analyses a bitwise/logical operation (&, |, ^)
     * @param op the bitwise/logical operator
     * @param leftAST left operand
     * @param rightAST right operand
     * @return Type
     */
    protected Type analyseBitwiseExpr(JQLAST op, JQLAST leftAST, JQLAST rightAST)
    {
        Type left = leftAST.getJQLType();
        Type right = rightAST.getJQLType();

        // handle error type
        if (left.equals(typetab.errorType) || right.equals(typetab.errorType))
            return typetab.errorType;

        switch(op.getType())
        {
        case BAND:
        case BOR:
            if (typetab.isBooleanType(left) && typetab.isBooleanType(right))
                return typetab.booleanType;
            else if (typetab.isIntegralType(left) || typetab.isIntegralType(right))
            {
                errorMsg.unsupported(op.getLine(), op.getColumn(),
                    I18NHelper.getMessage(messages, "jqlc.semantic.analysebitwiseexpr.integerbitwiseop", //NOI18N
                        op.getText()));
                return typetab.errorType;
            }
            break;
        case BXOR:
            if (typetab.isBooleanType(left) && typetab.isBooleanType(right))
            {
                errorMsg.unsupported(op.getLine(), op.getColumn(),
                    I18NHelper.getMessage(messages,
                        "jqlc.semantic.analysebitwiseexpr.exclusiveorop")); //NOI18N
                return typetab.errorType;
            }
            else if (typetab.isIntegralType(left) || typetab.isIntegralType(right))
            {
                errorMsg.unsupported(op.getLine(), op.getColumn(),
                    I18NHelper.getMessage(messages, "jqlc.semantic.analysebitwiseexpr.integerbitwiseop",  //NOI18N
                        op.getText()));
                return typetab.errorType;
            }
            break;
        }

        // if this code is reached a bitwise operator was used with invalid arguments
        errorMsg.error(op.getLine(), op.getColumn(),
            I18NHelper.getMessage(messages, "jqlc.semantic.generic.arguments.invalid", //NOI18N
                op.getText()));
        return typetab.errorType;
    }

    /**
     * Analyses a boolean conditional operation (&&, ||)
     * @param op the conditional operator
     * @param leftAST left operand
     * @param rightAST right operand
     * @return Type
     */
    protected Type analyseConditionalExpr(JQLAST op, JQLAST leftAST, JQLAST rightAST)
    {
        Type left = leftAST.getJQLType();
        Type right = rightAST.getJQLType();

        // handle error type
        if (left.equals(typetab.errorType) || right.equals(typetab.errorType))
            return typetab.errorType;

        switch(op.getType())
        {
        case AND:
        case OR:
            if (typetab.isBooleanType(left) && typetab.isBooleanType(right))
                return typetab.booleanType;
            break;
        }

        // if this code is reached a conditional operator was used with invalid arguments
        errorMsg.error(op.getLine(), op.getColumn(),
            I18NHelper.getMessage(messages, "jqlc.semantic.generic.arguments.invalid", //NOI18N
                op.getText()));
        return typetab.errorType;
    }

    /**
     * Analyses a relational operation (<, <=, >, >=, ==, !=)
     * @param op the relational operator
     * @param leftAST left operand
     * @param rightAST right operand
     * @return Type
     */
    protected Type analyseRelationalExpr(JQLAST op, JQLAST leftAST, JQLAST rightAST)
    {
        Type left = leftAST.getJQLType();
        Type right = rightAST.getJQLType();

        // handle error type
        if (left.equals(typetab.errorType) || right.equals(typetab.errorType))
            return typetab.errorType;

        // special check for <, <=, >, >=
        // left and right hand types must be orderable
        switch(op.getType())
        {
        case LT:
        case LE:
        case GT:
        case GE:
            if (!left.isOrderable())
            {
                errorMsg.error(op.getLine(), op.getColumn(),
                    I18NHelper.getMessage(messages, "jqlc.semantic.analyserelationalexpr.notorderable", //NOI18N
                        left.getName(), op.getText()));
                return typetab.errorType;
            }
            if (!right.isOrderable())
            {
                errorMsg.error(op.getLine(), op.getColumn(),
                    I18NHelper.getMessage(messages, "jqlc.semantic.analyserelationalexpr.notorderable", //NOI18N
                        right.getName(), op.getText()));
                return typetab.errorType;
            }
            break;
        case EQUAL:
        case NOT_EQUAL:
            if ((leftAST.getType() == CONTAINS) || (rightAST.getType() == CONTAINS))
            {
                errorMsg.unsupported(op.getLine(), op.getColumn(),
                    I18NHelper.getMessage(messages,
                        "jqlc.semantic.generic.unsupportedconstraintop", op.getText())); //NOI18N
                return typetab.errorType;
            }
            break;
        }

        // check for numeric types, numeric wrapper class types and math class types
        if (typetab.isNumberType(left) && typetab.isNumberType(right))
            return typetab.booleanType;

        // check for boolean and java.lang.Boolean
        if (typetab.isBooleanType(left) && typetab.isBooleanType(right))
            return typetab.booleanType;

        if (left.isCompatibleWith(right) || right.isCompatibleWith(left))
            return typetab.booleanType;

        // if this code is reached a conditional operator was used with invalid arguments
        errorMsg.error(op.getLine(), op.getColumn(),
            I18NHelper.getMessage(messages, "jqlc.semantic.generic.arguments.invalid", //NOI18N
                op.getText()));
        return typetab.errorType;
    }

    /**
     * Analyses a
     * @param op the  operator
     * @param leftAST left operand
     * @param rightAST right operand
     * @return Type
     */
    protected Type analyseBinaryArithmeticExpr(JQLAST op, JQLAST leftAST, JQLAST rightAST)
    {
        Type left = leftAST.getJQLType();
        Type right = rightAST.getJQLType();

        // handle error type
        if (left.equals(typetab.errorType) || right.equals(typetab.errorType))
            return typetab.errorType;

        if (typetab.isNumberType(left) && typetab.isNumberType(right))
        {
            // handle java.math.BigDecimal
            if (left.isCompatibleWith(typetab.bigDecimalType))
                return left;
            if (right.isCompatibleWith(typetab.bigDecimalType))
                return right;

            // handle java.math.BigInteger
            if (left.isCompatibleWith(typetab.bigIntegerType))
            {
                // if right is floating point return BigDecimal,
                // otherwise return BigInteger
                return typetab.isFloatingPointType(right) ?
                       typetab.bigDecimalType : left;
            }
            if (right.isCompatibleWith(typetab.bigIntegerType))
            {
                // if left is floating point return BigDecimal,
                // otherwise return BigInteger
                return typetab.isFloatingPointType(left) ?
                       typetab.bigDecimalType : right;
            }

            boolean wrapper = false;
            if (left instanceof NumericWrapperClassType)
            {
                left = ((NumericWrapperClassType)left).getPrimitiveType();
                wrapper = true;
            }
            if (right instanceof NumericWrapperClassType)
            {
                right = ((NumericWrapperClassType)right).getPrimitiveType();
                wrapper = true;
            }

            // handle numeric types with arbitrary arithmetic operator
            if ((left instanceof NumericType) && (right instanceof NumericType))
            {
                Type promotedType = typetab.binaryNumericPromotion(left, right);
                if (wrapper && (promotedType instanceof NumericType))
                {
                    promotedType =  ((NumericType)promotedType).getWrapper();
                }
                return promotedType;
            }
        }
        else if (op.getType() == PLUS)
        {
            // handle + for strings
            // MBO: note, this if matches char + char (which it should'nt),
            // but this case is already handled above
            if ((left.equals(typetab.stringType) || left.equals(typetab.charType)) &&
                (right.equals(typetab.stringType) || right.equals(typetab.charType)))
            {
                return typetab.stringType;
            }
        }

        // if this code is reached a conditional operator was used with invalid arguments
        errorMsg.error(op.getLine(), op.getColumn(),
            I18NHelper.getMessage(messages, "jqlc.semantic.generic.arguments.invalid", //NOI18N
                op.getText()));
        return typetab.errorType;
    }

    /**
     * Analyses a
     * @param op the operator
     * @param argAST right operand
     * @return Type
     */
    protected Type analyseUnaryArithmeticExpr(JQLAST op, JQLAST argAST)
    {
        Type arg = argAST.getJQLType();

        // handle error type
        if (arg.equals(typetab.errorType))
            return typetab.errorType;

        // handle java.math.BigDecimal and java.math.BigInteger
        if (arg.isCompatibleWith(typetab.bigDecimalType))
            return arg;

        // handle java.math.BigInteger
        if (arg.isCompatibleWith(typetab.bigIntegerType))
            return arg;

        boolean wrapper = false;
        if (arg instanceof NumericWrapperClassType)
        {
            arg = ((NumericWrapperClassType)arg).getPrimitiveType();
            wrapper = true;
        }

        if (arg instanceof NumericType)
        {
            Type promotedType = typetab.unaryNumericPromotion(arg);
            if (wrapper && (promotedType instanceof NumericType))
            {
                promotedType =  ((NumericType)promotedType).getWrapper();
            }
            return promotedType;
        }

        // if this code is reached a conditional operator was used with invalid arguments
        errorMsg.error(op.getLine(), op.getColumn(),
            I18NHelper.getMessage(messages, "jqlc.semantic.generic.arguments.invalid", //NOI18N
                op.getText()));
        return typetab.errorType;
    }
    /**
     * Analyses a
     * @param op the operator
     * @param argAST right operand
     * @return Type
     */
    protected Type analyseComplementExpr(JQLAST op, JQLAST argAST)
    {
        Type arg = argAST.getJQLType();

        // handle error type
        if (arg.equals(typetab.errorType))
            return typetab.errorType;

        switch(op.getType())
        {
        case BNOT:
            if (typetab.isIntegralType(arg))
            {
                return arg;
            }
            break;
        case LNOT:
            if (typetab.isBooleanType(arg))
            {
                if (argAST.getType() == CONTAINS)
                {
                    errorMsg.unsupported(op.getLine(), op.getColumn(),
                        I18NHelper.getMessage(messages,
                            "jqlc.semantic.generic.unsupportedconstraintop", op.getText())); //NOI18N
                    return typetab.errorType;
                }
                return arg;
            }
            break;
        }

        // if this code is reached a conditional operator was used with invalid arguments
        errorMsg.error(op.getLine(), op.getColumn(),
            I18NHelper.getMessage(messages, "jqlc.semantic.generic.arguments.invalid", //NOI18N
                op.getText()));
        return typetab.errorType;
    }

    /**
     *
     */
    protected void checkConstraints(JQLAST ast, VariableTable tab)
    {
        checkConstraints(ast, null, tab);
    }

    /**
     *
     */
    protected void checkConstraints(JQLAST ast, String dependentVariable, VariableTable tab)
    {
        if (ast == null) return;
        switch (ast.getType())
        {
        case VARIABLE:
            tab.markUsed(ast, dependentVariable);
            break;
        case CONTAINS:
            JQLAST expr = (JQLAST)ast.getFirstChild();
            JQLAST var = (JQLAST)expr.getNextSibling();
            checkConstraints(expr, var.getText(), tab);
            tab.markConstraint(var, expr);
            break;
        case BOR:
        case BXOR:
        case OR:
            JQLAST left = (JQLAST)ast.getFirstChild();
            JQLAST right = (JQLAST)left.getNextSibling();
            // prepare tab copy for right hand side and merge the right hand side copy into vartab
            VariableTable copy = new VariableTable(tab);
            checkConstraints(left, dependentVariable, tab);
            checkConstraints(right, dependentVariable, copy);
            tab.merge(copy);
            break;
        default:
            for (JQLAST node = (JQLAST)ast.getFirstChild(); node != null; node = (JQLAST)node.getNextSibling())
            {
                checkConstraints(node, dependentVariable, tab);
            }
            break;
        }
    }

public Semantic() {
	tokenNames = _tokenNames;
}

	public final void query(AST _t) throws RecognitionException {
		
		JQLAST query_AST_in = (_t == ASTNULL) ? null : (JQLAST)_t;
		returnAST = null;
		ASTPair currentAST = new ASTPair();
		JQLAST query_AST = null;
		JQLAST o_AST = null;
		JQLAST o = null;
		JQLAST r_AST = null;
		JQLAST r = null;
		
		AST __t2 = _t;
		JQLAST tmp1_AST = null;
		JQLAST tmp1_AST_in = null;
		tmp1_AST = (JQLAST)astFactory.create((JQLAST)_t);
		tmp1_AST_in = (JQLAST)_t;
		astFactory.addASTChild(currentAST, tmp1_AST);
		ASTPair __currentAST2 = currentAST.copy();
		currentAST.root = currentAST.child;
		currentAST.child = null;
		match(_t,QUERY);
		_t = _t.getFirstChild();
		
		symtab.enterScope();
		typeNames.enterScope();
		
		candidateClass(_t);
		_t = _retTree;
		astFactory.addASTChild(currentAST, returnAST);
		imports(_t);
		_t = _retTree;
		astFactory.addASTChild(currentAST, returnAST);
		
		// enter new scope for variable and parameter names
		symtab.enterScope();
		
		parameters(_t);
		_t = _retTree;
		astFactory.addASTChild(currentAST, returnAST);
		variables(_t);
		_t = _retTree;
		astFactory.addASTChild(currentAST, returnAST);
		o = _t==ASTNULL ? null : (JQLAST)_t;
		ordering(_t);
		_t = _retTree;
		o_AST = (JQLAST)returnAST;
		astFactory.addASTChild(currentAST, returnAST);
		r = _t==ASTNULL ? null : (JQLAST)_t;
		result(_t);
		_t = _retTree;
		r_AST = (JQLAST)returnAST;
		astFactory.addASTChild(currentAST, returnAST);
		filter(_t);
		_t = _retTree;
		astFactory.addASTChild(currentAST, returnAST);
		
		typeNames.leaveScope();
		// leaves variable and parameter name scope
		symtab.leaveScope();
		// leaves global scope
		symtab.leaveScope();
		
		currentAST = __currentAST2;
		_t = __t2;
		_t = _t.getNextSibling();
		
		checkResultOrdering(r_AST, o_AST);
		
		query_AST = (JQLAST)currentAST.root;
		returnAST = query_AST;
		_retTree = _t;
	}
	
	public final void candidateClass(AST _t) throws RecognitionException {
		
		JQLAST candidateClass_AST_in = (_t == ASTNULL) ? null : (JQLAST)_t;
		returnAST = null;
		ASTPair currentAST = new ASTPair();
		JQLAST candidateClass_AST = null;
		JQLAST c = null;
		JQLAST c_AST = null;
		
		errorMsg.setContext("setClass"); //NOI18N
		
		
		c = (JQLAST)_t;
		JQLAST c_AST_in = null;
		c_AST = (JQLAST)astFactory.create(c);
		astFactory.addASTChild(currentAST, c_AST);
		match(_t,CLASS_DEF);
		_t = _t.getNextSibling();
		
		// check persistence capable
		candidateClass = (ClassType)c_AST.getJQLType();
		String className = candidateClass.getName();
		if (!candidateClass.isPersistenceCapable())
		{
		errorMsg.unsupported(c_AST.getLine(), c_AST.getColumn(),
		I18NHelper.getMessage(messages, "jqlc.semantic.candidateclass.nonpc", //NOI18N
		className));
		}
		
		// get base name
		int index = className.lastIndexOf('.');
		String identName = index>0 ? className.substring(index+1) : className;
		typeNames.declare(identName, new TypeName(candidateClass, className));
		
		// init symbol table with field names of the candidate class
		FieldInfo[] fieldInfos = candidateClass.getFieldInfos();
		for (int i = 0; i < fieldInfos.length; i++)
		{
		FieldInfo fieldInfo = fieldInfos[i];
		symtab.declare(fieldInfo.getName(), new Field(fieldInfo));
		}
		
		candidateClass_AST = (JQLAST)currentAST.root;
		returnAST = candidateClass_AST;
		_retTree = _t;
	}
	
	public final void imports(AST _t) throws RecognitionException {
		
		JQLAST imports_AST_in = (_t == ASTNULL) ? null : (JQLAST)_t;
		returnAST = null;
		ASTPair currentAST = new ASTPair();
		JQLAST imports_AST = null;
		
		errorMsg.setContext("declareImports"); //NOI18N
		
		
		{
		_loop6:
		do {
			if (_t==null) _t=ASTNULL;
			if ((_t.getType()==IMPORT_DEF)) {
				declareImport(_t);
				_t = _retTree;
			}
			else {
				break _loop6;
			}
			
		} while (true);
		}
		returnAST = imports_AST;
		_retTree = _t;
	}
	
	public final void parameters(AST _t) throws RecognitionException {
		
		JQLAST parameters_AST_in = (_t == ASTNULL) ? null : (JQLAST)_t;
		returnAST = null;
		ASTPair currentAST = new ASTPair();
		JQLAST parameters_AST = null;
		
		errorMsg.setContext("declareParameters"); //NOI18N
		
		
		{
		_loop11:
		do {
			if (_t==null) _t=ASTNULL;
			if ((_t.getType()==PARAMETER_DEF)) {
				declareParameter(_t);
				_t = _retTree;
				astFactory.addASTChild(currentAST, returnAST);
			}
			else {
				break _loop11;
			}
			
		} while (true);
		}
		parameters_AST = (JQLAST)currentAST.root;
		returnAST = parameters_AST;
		_retTree = _t;
	}
	
	public final void variables(AST _t) throws RecognitionException {
		
		JQLAST variables_AST_in = (_t == ASTNULL) ? null : (JQLAST)_t;
		returnAST = null;
		ASTPair currentAST = new ASTPair();
		JQLAST variables_AST = null;
		
		errorMsg.setContext("declareVariables"); //NOI18N
		
		
		{
		_loop16:
		do {
			if (_t==null) _t=ASTNULL;
			if ((_t.getType()==VARIABLE_DEF)) {
				declareVariable(_t);
				_t = _retTree;
				astFactory.addASTChild(currentAST, returnAST);
			}
			else {
				break _loop16;
			}
			
		} while (true);
		}
		variables_AST = (JQLAST)currentAST.root;
		returnAST = variables_AST;
		_retTree = _t;
	}
	
	public final void ordering(AST _t) throws RecognitionException {
		
		JQLAST ordering_AST_in = (_t == ASTNULL) ? null : (JQLAST)_t;
		returnAST = null;
		ASTPair currentAST = new ASTPair();
		JQLAST ordering_AST = null;
		
		errorMsg.setContext("setOrdering"); //NOI18N
		
		
		{
		_loop21:
		do {
			if (_t==null) _t=ASTNULL;
			if ((_t.getType()==ORDERING_DEF)) {
				orderSpec(_t);
				_t = _retTree;
				astFactory.addASTChild(currentAST, returnAST);
			}
			else {
				break _loop21;
			}
			
		} while (true);
		}
		ordering_AST = (JQLAST)currentAST.root;
		returnAST = ordering_AST;
		_retTree = _t;
	}
	
	public final void result(AST _t) throws RecognitionException {
		
		JQLAST result_AST_in = (_t == ASTNULL) ? null : (JQLAST)_t;
		returnAST = null;
		ASTPair currentAST = new ASTPair();
		JQLAST result_AST = null;
		JQLAST r = null;
		JQLAST r_AST = null;
		JQLAST e_AST = null;
		JQLAST e = null;
		
		errorMsg.setContext("setResult"); //NOI18N
		
		
		if (_t==null) _t=ASTNULL;
		switch ( _t.getType()) {
		case RESULT_DEF:
		{
			AST __t26 = _t;
			r = _t==ASTNULL ? null :(JQLAST)_t;
			JQLAST r_AST_in = null;
			r_AST = (JQLAST)astFactory.create(r);
			astFactory.addASTChild(currentAST, r_AST);
			ASTPair __currentAST26 = currentAST.copy();
			currentAST.root = currentAST.child;
			currentAST.child = null;
			match(_t,RESULT_DEF);
			_t = _t.getFirstChild();
			e = _t==ASTNULL ? null : (JQLAST)_t;
			resultExpr(_t);
			_t = _retTree;
			e_AST = (JQLAST)returnAST;
			astFactory.addASTChild(currentAST, returnAST);
			currentAST = __currentAST26;
			_t = __t26;
			_t = _t.getNextSibling();
			
			checkValidResultExpr(e_AST);
			r_AST.setJQLType(e_AST.getJQLType());
			checkConstraints(e_AST, vartab);
			
			result_AST = (JQLAST)currentAST.root;
			break;
		}
		case FILTER_DEF:
		{
			result_AST = (JQLAST)currentAST.root;
			break;
		}
		default:
		{
			throw new NoViableAltException(_t);
		}
		}
		returnAST = result_AST;
		_retTree = _t;
	}
	
	public final void filter(AST _t) throws RecognitionException {
		
		JQLAST filter_AST_in = (_t == ASTNULL) ? null : (JQLAST)_t;
		returnAST = null;
		ASTPair currentAST = new ASTPair();
		JQLAST filter_AST = null;
		JQLAST e_AST = null;
		JQLAST e = null;
		
		errorMsg.setContext("setFilter"); //NOI18N
		
		
		AST __t35 = _t;
		JQLAST tmp2_AST = null;
		JQLAST tmp2_AST_in = null;
		tmp2_AST = (JQLAST)astFactory.create((JQLAST)_t);
		tmp2_AST_in = (JQLAST)_t;
		astFactory.addASTChild(currentAST, tmp2_AST);
		ASTPair __currentAST35 = currentAST.copy();
		currentAST.root = currentAST.child;
		currentAST.child = null;
		match(_t,FILTER_DEF);
		_t = _t.getFirstChild();
		e = _t==ASTNULL ? null : (JQLAST)_t;
		expression(_t);
		_t = _retTree;
		e_AST = (JQLAST)returnAST;
		astFactory.addASTChild(currentAST, returnAST);
		currentAST = __currentAST35;
		_t = __t35;
		_t = _t.getNextSibling();
		
		Type exprType = e_AST.getJQLType();
		if (!(typetab.isBooleanType(exprType) || exprType.equals(typetab.errorType)))
		{
		// filter expression must have the type boolean or java.lang.Boolean
		errorMsg.error(e_AST.getLine(), e_AST.getColumn(),
		I18NHelper.getMessage(messages, "jqlc.semantic.filter.booleanexpected", exprType)); //NOI18N
		}
		checkConstraints(e_AST, vartab);
		vartab.checkConstraints();
		
		filter_AST = (JQLAST)currentAST.root;
		returnAST = filter_AST;
		_retTree = _t;
	}
	
	public final void declareImport(AST _t) throws RecognitionException {
		
		JQLAST declareImport_AST_in = (_t == ASTNULL) ? null : (JQLAST)_t;
		returnAST = null;
		ASTPair currentAST = new ASTPair();
		JQLAST declareImport_AST = null;
		JQLAST i = null;
		JQLAST i_AST = null;
		String name = null;
		
		AST __t8 = _t;
		i = _t==ASTNULL ? null :(JQLAST)_t;
		JQLAST i_AST_in = null;
		i_AST = (JQLAST)astFactory.create(i);
		astFactory.addASTChild(currentAST, i_AST);
		ASTPair __currentAST8 = currentAST.copy();
		currentAST.root = currentAST.child;
		currentAST.child = null;
		match(_t,IMPORT_DEF);
		_t = _t.getFirstChild();
		name=qualifiedName(_t);
		_t = _retTree;
		astFactory.addASTChild(currentAST, returnAST);
		currentAST = __currentAST8;
		_t = __t8;
		_t = _t.getNextSibling();
		
		Type type = typetab.checkType(name);
		if (type == null)
		{
		errorMsg.error(i_AST.getLine(), i_AST.getColumn(),
		I18NHelper.getMessage(messages, "jqlc.semantic.generic.unknowntype", name)); //NOI18N
		}
		
		// get base name
		int index = name.lastIndexOf('.');
		String identName = index>0 ? name.substring(index+1) : name;
		
		Definition old = typeNames.declare(identName, new TypeName(type, name));
		if (old != null)
		{
		errorMsg.error(i_AST.getLine(), i_AST.getColumn(),
		I18NHelper.getMessage(messages, "jqlc.semantic.generic.alreadydeclared", //NOI18N
		identName, old.getName()));
		}
		
		declareImport_AST = (JQLAST)currentAST.root;
		returnAST = declareImport_AST;
		_retTree = _t;
	}
	
	public final String  qualifiedName(AST _t) throws RecognitionException {
		String name;
		
		JQLAST qualifiedName_AST_in = (_t == ASTNULL) ? null : (JQLAST)_t;
		returnAST = null;
		ASTPair currentAST = new ASTPair();
		JQLAST qualifiedName_AST = null;
		JQLAST id1 = null;
		JQLAST id1_AST = null;
		JQLAST d = null;
		JQLAST d_AST = null;
		JQLAST id2 = null;
		JQLAST id2_AST = null;
		name = null;
		
		if (_t==null) _t=ASTNULL;
		switch ( _t.getType()) {
		case IDENT:
		{
			id1 = (JQLAST)_t;
			JQLAST id1_AST_in = null;
			id1_AST = (JQLAST)astFactory.create(id1);
			astFactory.addASTChild(currentAST, id1_AST);
			match(_t,IDENT);
			_t = _t.getNextSibling();
			
			name = id1_AST.getText();
			
			qualifiedName_AST = (JQLAST)currentAST.root;
			break;
		}
		case DOT:
		{
			AST __t77 = _t;
			d = _t==ASTNULL ? null :(JQLAST)_t;
			JQLAST d_AST_in = null;
			d_AST = (JQLAST)astFactory.create(d);
			astFactory.addASTChild(currentAST, d_AST);
			ASTPair __currentAST77 = currentAST.copy();
			currentAST.root = currentAST.child;
			currentAST.child = null;
			match(_t,DOT);
			_t = _t.getFirstChild();
			name=qualifiedName(_t);
			_t = _retTree;
			astFactory.addASTChild(currentAST, returnAST);
			id2 = (JQLAST)_t;
			JQLAST id2_AST_in = null;
			id2_AST = (JQLAST)astFactory.create(id2);
			astFactory.addASTChild(currentAST, id2_AST);
			match(_t,IDENT);
			_t = _t.getNextSibling();
			
			name += (d_AST.getText() + id2_AST.getText());
			
			currentAST = __currentAST77;
			_t = __t77;
			_t = _t.getNextSibling();
			qualifiedName_AST = (JQLAST)currentAST.root;
			break;
		}
		default:
		{
			throw new NoViableAltException(_t);
		}
		}
		returnAST = qualifiedName_AST;
		_retTree = _t;
		return name;
	}
	
	public final void declareParameter(AST _t) throws RecognitionException {
		
		JQLAST declareParameter_AST_in = (_t == ASTNULL) ? null : (JQLAST)_t;
		returnAST = null;
		ASTPair currentAST = new ASTPair();
		JQLAST declareParameter_AST = null;
		JQLAST t_AST = null;
		JQLAST t = null;
		JQLAST i = null;
		JQLAST i_AST = null;
		
		AST __t13 = _t;
		JQLAST tmp3_AST = null;
		JQLAST tmp3_AST_in = null;
		tmp3_AST = (JQLAST)astFactory.create((JQLAST)_t);
		tmp3_AST_in = (JQLAST)_t;
		astFactory.addASTChild(currentAST, tmp3_AST);
		ASTPair __currentAST13 = currentAST.copy();
		currentAST.root = currentAST.child;
		currentAST.child = null;
		match(_t,PARAMETER_DEF);
		_t = _t.getFirstChild();
		t = _t==ASTNULL ? null : (JQLAST)_t;
		type(_t);
		_t = _retTree;
		t_AST = (JQLAST)returnAST;
		astFactory.addASTChild(currentAST, returnAST);
		i = (JQLAST)_t;
		JQLAST i_AST_in = null;
		i_AST = (JQLAST)astFactory.create(i);
		astFactory.addASTChild(currentAST, i_AST);
		match(_t,IDENT);
		_t = _t.getNextSibling();
		currentAST = __currentAST13;
		_t = __t13;
		_t = _t.getNextSibling();
		
		String name = i_AST.getText();
		Type type = t_AST.getJQLType();
		Definition old = symtab.declare(name, new Parameter(type));
		if (old != null)
		{
		errorMsg.error(i_AST.getLine(), i_AST.getColumn(),
		I18NHelper.getMessage(messages, "jqlc.semantic.generic.alreadydeclared", //NOI18N
		name, old.getName()));
		}
		i_AST.setJQLType(type);
		paramtab.add(name, type);
		
		declareParameter_AST = (JQLAST)currentAST.root;
		returnAST = declareParameter_AST;
		_retTree = _t;
	}
	
	public final void type(AST _t) throws RecognitionException {
		
		JQLAST type_AST_in = (_t == ASTNULL) ? null : (JQLAST)_t;
		returnAST = null;
		ASTPair currentAST = new ASTPair();
		JQLAST type_AST = null;
		JQLAST qn_AST = null;
		JQLAST qn = null;
		JQLAST p_AST = null;
		JQLAST p = null;
		String name = null;
		
		if (_t==null) _t=ASTNULL;
		switch ( _t.getType()) {
		case IDENT:
		case DOT:
		{
			qn = _t==ASTNULL ? null : (JQLAST)_t;
			name=qualifiedName(_t);
			_t = _retTree;
			qn_AST = (JQLAST)returnAST;
			astFactory.addASTChild(currentAST, returnAST);
			
			Type type = null;
			if (typeNames.isDeclared(name))
			{
			Definition def = typeNames.getDefinition(name);
			if (def instanceof TypeName)
			{
			type = def.getType();
			}
			else
			{
			errorMsg.error(qn_AST.getLine(), qn_AST.getColumn(),
			I18NHelper.getMessage(messages, "jqlc.semantic.type.notype", //NOI18N
			name, def.getName()));
			}
			}
			else
			{
			type = typetab.checkType(name);
			if ((type == null) && (name.indexOf('.') == -1))
			{
			// ckeck java.lang class without package name
			type = typetab.checkType("java.lang." + name); //NOI18N
			}
			if (type == null)
			{
			errorMsg.error(qn_AST.getLine(), qn_AST.getColumn(),
			I18NHelper.getMessage(messages, "jqlc.semantic.generic.unknowntype", name)); //NOI18N
			}
			}
			// change AST to a single node that represents the full class name
			qn_AST.setType(TYPENAME);
			qn_AST.setText(name);
			qn_AST.setFirstChild(null);
			qn_AST.setJQLType(type);
			
			type_AST = (JQLAST)currentAST.root;
			break;
		}
		case BOOLEAN:
		case BYTE:
		case CHAR:
		case SHORT:
		case INT:
		case FLOAT:
		case LONG:
		case DOUBLE:
		{
			p = _t==ASTNULL ? null : (JQLAST)_t;
			primitiveType(_t);
			_t = _retTree;
			p_AST = (JQLAST)returnAST;
			astFactory.addASTChild(currentAST, returnAST);
			
			p_AST.setJQLType(typetab.checkType(p_AST.getText()));
			p_AST.setType(TYPENAME);
			
			type_AST = (JQLAST)currentAST.root;
			break;
		}
		default:
		{
			throw new NoViableAltException(_t);
		}
		}
		returnAST = type_AST;
		_retTree = _t;
	}
	
	public final void declareVariable(AST _t) throws RecognitionException {
		
		JQLAST declareVariable_AST_in = (_t == ASTNULL) ? null : (JQLAST)_t;
		returnAST = null;
		ASTPair currentAST = new ASTPair();
		JQLAST declareVariable_AST = null;
		JQLAST t_AST = null;
		JQLAST t = null;
		JQLAST i = null;
		JQLAST i_AST = null;
		
		AST __t18 = _t;
		JQLAST tmp4_AST = null;
		JQLAST tmp4_AST_in = null;
		tmp4_AST = (JQLAST)astFactory.create((JQLAST)_t);
		tmp4_AST_in = (JQLAST)_t;
		astFactory.addASTChild(currentAST, tmp4_AST);
		ASTPair __currentAST18 = currentAST.copy();
		currentAST.root = currentAST.child;
		currentAST.child = null;
		match(_t,VARIABLE_DEF);
		_t = _t.getFirstChild();
		t = _t==ASTNULL ? null : (JQLAST)_t;
		type(_t);
		_t = _retTree;
		t_AST = (JQLAST)returnAST;
		astFactory.addASTChild(currentAST, returnAST);
		i = (JQLAST)_t;
		JQLAST i_AST_in = null;
		i_AST = (JQLAST)astFactory.create(i);
		astFactory.addASTChild(currentAST, i_AST);
		match(_t,IDENT);
		_t = _t.getNextSibling();
		currentAST = __currentAST18;
		_t = __t18;
		_t = _t.getNextSibling();
		
		String name = i_AST.getText();
		Type type = t_AST.getJQLType();
		Definition old = symtab.declare(name, new Variable(type));
		if (old != null)
		{
		errorMsg.error(i_AST.getLine(), i_AST.getColumn(),
		I18NHelper.getMessage(messages, "jqlc.semantic.generic.alreadydeclared", //NOI18N
		name, old.getName()));
		}
		vartab.add(name);
		i_AST.setJQLType(type);
		
		declareVariable_AST = (JQLAST)currentAST.root;
		returnAST = declareVariable_AST;
		_retTree = _t;
	}
	
	public final void orderSpec(AST _t) throws RecognitionException {
		
		JQLAST orderSpec_AST_in = (_t == ASTNULL) ? null : (JQLAST)_t;
		returnAST = null;
		ASTPair currentAST = new ASTPair();
		JQLAST orderSpec_AST = null;
		JQLAST e_AST = null;
		JQLAST e = null;
		
		AST __t23 = _t;
		JQLAST tmp5_AST = null;
		JQLAST tmp5_AST_in = null;
		tmp5_AST = (JQLAST)astFactory.create((JQLAST)_t);
		tmp5_AST_in = (JQLAST)_t;
		astFactory.addASTChild(currentAST, tmp5_AST);
		ASTPair __currentAST23 = currentAST.copy();
		currentAST.root = currentAST.child;
		currentAST.child = null;
		match(_t,ORDERING_DEF);
		_t = _t.getFirstChild();
		{
		if (_t==null) _t=ASTNULL;
		switch ( _t.getType()) {
		case ASCENDING:
		{
			JQLAST tmp6_AST = null;
			JQLAST tmp6_AST_in = null;
			tmp6_AST = (JQLAST)astFactory.create((JQLAST)_t);
			tmp6_AST_in = (JQLAST)_t;
			astFactory.addASTChild(currentAST, tmp6_AST);
			match(_t,ASCENDING);
			_t = _t.getNextSibling();
			break;
		}
		case DESCENDING:
		{
			JQLAST tmp7_AST = null;
			JQLAST tmp7_AST_in = null;
			tmp7_AST = (JQLAST)astFactory.create((JQLAST)_t);
			tmp7_AST_in = (JQLAST)_t;
			astFactory.addASTChild(currentAST, tmp7_AST);
			match(_t,DESCENDING);
			_t = _t.getNextSibling();
			break;
		}
		default:
		{
			throw new NoViableAltException(_t);
		}
		}
		}
		e = _t==ASTNULL ? null : (JQLAST)_t;
		expression(_t);
		_t = _retTree;
		e_AST = (JQLAST)returnAST;
		astFactory.addASTChild(currentAST, returnAST);
		currentAST = __currentAST23;
		_t = __t23;
		_t = _t.getNextSibling();
		
		analyseOrderingExpression(e_AST);
		checkConstraints(e_AST, vartab);
		
		orderSpec_AST = (JQLAST)currentAST.root;
		returnAST = orderSpec_AST;
		_retTree = _t;
	}
	
	public final void expression(AST _t) throws RecognitionException {
		
		JQLAST expression_AST_in = (_t == ASTNULL) ? null : (JQLAST)_t;
		returnAST = null;
		ASTPair currentAST = new ASTPair();
		JQLAST expression_AST = null;
		JQLAST e_AST = null;
		JQLAST e = null;
		String repr;
		
		e = _t==ASTNULL ? null : (JQLAST)_t;
		repr=exprNoCheck(_t,false);
		_t = _retTree;
		e_AST = (JQLAST)returnAST;
		astFactory.addASTChild(currentAST, returnAST);
		
		if (repr != null)
		{
		e_AST.setJQLType(typetab.errorType);
		errorMsg.error(e_AST.getLine(), e_AST.getColumn(),
		I18NHelper.getMessage(messages, "jqlc.semantic.expression.undefined", repr)); //NOI18N
		}
		
		expression_AST = (JQLAST)currentAST.root;
		returnAST = expression_AST;
		_retTree = _t;
	}
	
	public final void resultExpr(AST _t) throws RecognitionException {
		
		JQLAST resultExpr_AST_in = (_t == ASTNULL) ? null : (JQLAST)_t;
		returnAST = null;
		ASTPair currentAST = new ASTPair();
		JQLAST resultExpr_AST = null;
		JQLAST d = null;
		JQLAST d_AST = null;
		JQLAST e0_AST = null;
		JQLAST e0 = null;
		JQLAST a = null;
		JQLAST a_AST = null;
		JQLAST e1_AST = null;
		JQLAST e1 = null;
		JQLAST max = null;
		JQLAST max_AST = null;
		JQLAST e2_AST = null;
		JQLAST e2 = null;
		JQLAST min = null;
		JQLAST min_AST = null;
		JQLAST e3_AST = null;
		JQLAST e3 = null;
		JQLAST s = null;
		JQLAST s_AST = null;
		JQLAST e4_AST = null;
		JQLAST e4 = null;
		JQLAST c = null;
		JQLAST c_AST = null;
		
		if (_t==null) _t=ASTNULL;
		switch ( _t.getType()) {
		case DISTINCT:
		{
			AST __t28 = _t;
			d = _t==ASTNULL ? null :(JQLAST)_t;
			JQLAST d_AST_in = null;
			d_AST = (JQLAST)astFactory.create(d);
			astFactory.addASTChild(currentAST, d_AST);
			ASTPair __currentAST28 = currentAST.copy();
			currentAST.root = currentAST.child;
			currentAST.child = null;
			match(_t,DISTINCT);
			_t = _t.getFirstChild();
			e0 = _t==ASTNULL ? null : (JQLAST)_t;
			resultExpr(_t);
			_t = _retTree;
			e0_AST = (JQLAST)returnAST;
			astFactory.addASTChild(currentAST, returnAST);
			currentAST = __currentAST28;
			_t = __t28;
			_t = _t.getNextSibling();
			
			d_AST.setJQLType(e0_AST.getJQLType());
			
			resultExpr_AST = (JQLAST)currentAST.root;
			break;
		}
		case AVG:
		{
			AST __t29 = _t;
			a = _t==ASTNULL ? null :(JQLAST)_t;
			JQLAST a_AST_in = null;
			a_AST = (JQLAST)astFactory.create(a);
			astFactory.addASTChild(currentAST, a_AST);
			ASTPair __currentAST29 = currentAST.copy();
			currentAST.root = currentAST.child;
			currentAST.child = null;
			match(_t,AVG);
			_t = _t.getFirstChild();
			e1 = _t==ASTNULL ? null : (JQLAST)_t;
			resultExpr(_t);
			_t = _retTree;
			e1_AST = (JQLAST)returnAST;
			astFactory.addASTChild(currentAST, returnAST);
			currentAST = __currentAST29;
			_t = __t29;
			_t = _t.getNextSibling();
			
			a_AST.setJQLType(typetab.getAvgReturnType(e1_AST.getJQLType()));
			
			resultExpr_AST = (JQLAST)currentAST.root;
			break;
		}
		case MAX:
		{
			AST __t30 = _t;
			max = _t==ASTNULL ? null :(JQLAST)_t;
			JQLAST max_AST_in = null;
			max_AST = (JQLAST)astFactory.create(max);
			astFactory.addASTChild(currentAST, max_AST);
			ASTPair __currentAST30 = currentAST.copy();
			currentAST.root = currentAST.child;
			currentAST.child = null;
			match(_t,MAX);
			_t = _t.getFirstChild();
			e2 = _t==ASTNULL ? null : (JQLAST)_t;
			resultExpr(_t);
			_t = _retTree;
			e2_AST = (JQLAST)returnAST;
			astFactory.addASTChild(currentAST, returnAST);
			currentAST = __currentAST30;
			_t = __t30;
			_t = _t.getNextSibling();
			
			max_AST.setJQLType(typetab.getMinMaxReturnType(e2_AST.getJQLType()));
			
			resultExpr_AST = (JQLAST)currentAST.root;
			break;
		}
		case MIN:
		{
			AST __t31 = _t;
			min = _t==ASTNULL ? null :(JQLAST)_t;
			JQLAST min_AST_in = null;
			min_AST = (JQLAST)astFactory.create(min);
			astFactory.addASTChild(currentAST, min_AST);
			ASTPair __currentAST31 = currentAST.copy();
			currentAST.root = currentAST.child;
			currentAST.child = null;
			match(_t,MIN);
			_t = _t.getFirstChild();
			e3 = _t==ASTNULL ? null : (JQLAST)_t;
			resultExpr(_t);
			_t = _retTree;
			e3_AST = (JQLAST)returnAST;
			astFactory.addASTChild(currentAST, returnAST);
			currentAST = __currentAST31;
			_t = __t31;
			_t = _t.getNextSibling();
			
			min_AST.setJQLType(typetab.getMinMaxReturnType(e3_AST.getJQLType()));
			
			resultExpr_AST = (JQLAST)currentAST.root;
			break;
		}
		case SUM:
		{
			AST __t32 = _t;
			s = _t==ASTNULL ? null :(JQLAST)_t;
			JQLAST s_AST_in = null;
			s_AST = (JQLAST)astFactory.create(s);
			astFactory.addASTChild(currentAST, s_AST);
			ASTPair __currentAST32 = currentAST.copy();
			currentAST.root = currentAST.child;
			currentAST.child = null;
			match(_t,SUM);
			_t = _t.getFirstChild();
			e4 = _t==ASTNULL ? null : (JQLAST)_t;
			resultExpr(_t);
			_t = _retTree;
			e4_AST = (JQLAST)returnAST;
			astFactory.addASTChild(currentAST, returnAST);
			currentAST = __currentAST32;
			_t = __t32;
			_t = _t.getNextSibling();
			
			s_AST.setJQLType(typetab.getSumReturnType(e4_AST.getJQLType()));
			
			resultExpr_AST = (JQLAST)currentAST.root;
			break;
		}
		case COUNT:
		{
			AST __t33 = _t;
			c = _t==ASTNULL ? null :(JQLAST)_t;
			JQLAST c_AST_in = null;
			c_AST = (JQLAST)astFactory.create(c);
			astFactory.addASTChild(currentAST, c_AST);
			ASTPair __currentAST33 = currentAST.copy();
			currentAST.root = currentAST.child;
			currentAST.child = null;
			match(_t,COUNT);
			_t = _t.getFirstChild();
			resultExpr(_t);
			_t = _retTree;
			astFactory.addASTChild(currentAST, returnAST);
			currentAST = __currentAST33;
			_t = __t33;
			_t = _t.getNextSibling();
			
			c_AST.setJQLType(typetab.longType);
			
			resultExpr_AST = (JQLAST)currentAST.root;
			break;
		}
		case THIS:
		case NULL:
		case TRUE:
		case FALSE:
		case EQUAL:
		case LNOT:
		case BNOT:
		case NOT_EQUAL:
		case DIV:
		case PLUS:
		case MINUS:
		case STAR:
		case MOD:
		case GE:
		case GT:
		case LE:
		case LT:
		case BXOR:
		case BOR:
		case OR:
		case BAND:
		case AND:
		case CHAR_LITERAL:
		case STRING_LITERAL:
		case INT_LITERAL:
		case IDENT:
		case UNARY_MINUS:
		case UNARY_PLUS:
		case TYPECAST:
		case DOT:
		case LONG_LITERAL:
		case FLOAT_LITERAL:
		case DOUBLE_LITERAL:
		{
			expression(_t);
			_t = _retTree;
			astFactory.addASTChild(currentAST, returnAST);
			resultExpr_AST = (JQLAST)currentAST.root;
			break;
		}
		default:
		{
			throw new NoViableAltException(_t);
		}
		}
		returnAST = resultExpr_AST;
		_retTree = _t;
	}
	
	public final String  exprNoCheck(AST _t,
		boolean insideDotExpr
	) throws RecognitionException {
		String repr;
		
		JQLAST exprNoCheck_AST_in = (_t == ASTNULL) ? null : (JQLAST)_t;
		returnAST = null;
		ASTPair currentAST = new ASTPair();
		JQLAST exprNoCheck_AST = null;
		repr = null;
		
		if (_t==null) _t=ASTNULL;
		switch ( _t.getType()) {
		case BXOR:
		case BOR:
		case BAND:
		{
			bitwiseExpr(_t);
			_t = _retTree;
			astFactory.addASTChild(currentAST, returnAST);
			exprNoCheck_AST = (JQLAST)currentAST.root;
			break;
		}
		case OR:
		case AND:
		{
			conditionalExpr(_t);
			_t = _retTree;
			astFactory.addASTChild(currentAST, returnAST);
			exprNoCheck_AST = (JQLAST)currentAST.root;
			break;
		}
		case EQUAL:
		case NOT_EQUAL:
		case GE:
		case GT:
		case LE:
		case LT:
		{
			relationalExpr(_t);
			_t = _retTree;
			astFactory.addASTChild(currentAST, returnAST);
			exprNoCheck_AST = (JQLAST)currentAST.root;
			break;
		}
		case DIV:
		case PLUS:
		case MINUS:
		case STAR:
		case MOD:
		{
			binaryArithmeticExpr(_t);
			_t = _retTree;
			astFactory.addASTChild(currentAST, returnAST);
			exprNoCheck_AST = (JQLAST)currentAST.root;
			break;
		}
		case UNARY_MINUS:
		case UNARY_PLUS:
		{
			unaryArithmeticExpr(_t);
			_t = _retTree;
			astFactory.addASTChild(currentAST, returnAST);
			exprNoCheck_AST = (JQLAST)currentAST.root;
			break;
		}
		case LNOT:
		case BNOT:
		{
			complementExpr(_t);
			_t = _retTree;
			astFactory.addASTChild(currentAST, returnAST);
			exprNoCheck_AST = (JQLAST)currentAST.root;
			break;
		}
		case THIS:
		case NULL:
		case TRUE:
		case FALSE:
		case CHAR_LITERAL:
		case STRING_LITERAL:
		case INT_LITERAL:
		case IDENT:
		case TYPECAST:
		case DOT:
		case LONG_LITERAL:
		case FLOAT_LITERAL:
		case DOUBLE_LITERAL:
		{
			repr=primary(_t,insideDotExpr);
			_t = _retTree;
			astFactory.addASTChild(currentAST, returnAST);
			exprNoCheck_AST = (JQLAST)currentAST.root;
			break;
		}
		default:
		{
			throw new NoViableAltException(_t);
		}
		}
		returnAST = exprNoCheck_AST;
		_retTree = _t;
		return repr;
	}
	
	public final void bitwiseExpr(AST _t) throws RecognitionException {
		
		JQLAST bitwiseExpr_AST_in = (_t == ASTNULL) ? null : (JQLAST)_t;
		returnAST = null;
		ASTPair currentAST = new ASTPair();
		JQLAST bitwiseExpr_AST = null;
		JQLAST op1 = null;
		JQLAST op1_AST = null;
		JQLAST left1_AST = null;
		JQLAST left1 = null;
		JQLAST right1_AST = null;
		JQLAST right1 = null;
		JQLAST op2 = null;
		JQLAST op2_AST = null;
		JQLAST left2_AST = null;
		JQLAST left2 = null;
		JQLAST right2_AST = null;
		JQLAST right2 = null;
		JQLAST op3 = null;
		JQLAST op3_AST = null;
		JQLAST left3_AST = null;
		JQLAST left3 = null;
		JQLAST right3_AST = null;
		JQLAST right3 = null;
		
		if (_t==null) _t=ASTNULL;
		switch ( _t.getType()) {
		case BAND:
		{
			AST __t39 = _t;
			op1 = _t==ASTNULL ? null :(JQLAST)_t;
			JQLAST op1_AST_in = null;
			op1_AST = (JQLAST)astFactory.create(op1);
			astFactory.addASTChild(currentAST, op1_AST);
			ASTPair __currentAST39 = currentAST.copy();
			currentAST.root = currentAST.child;
			currentAST.child = null;
			match(_t,BAND);
			_t = _t.getFirstChild();
			left1 = _t==ASTNULL ? null : (JQLAST)_t;
			expression(_t);
			_t = _retTree;
			left1_AST = (JQLAST)returnAST;
			astFactory.addASTChild(currentAST, returnAST);
			right1 = _t==ASTNULL ? null : (JQLAST)_t;
			expression(_t);
			_t = _retTree;
			right1_AST = (JQLAST)returnAST;
			astFactory.addASTChild(currentAST, returnAST);
			currentAST = __currentAST39;
			_t = __t39;
			_t = _t.getNextSibling();
			
			op1_AST.setJQLType(analyseBitwiseExpr(op1_AST, left1_AST, right1_AST));
			
			bitwiseExpr_AST = (JQLAST)currentAST.root;
			break;
		}
		case BOR:
		{
			AST __t40 = _t;
			op2 = _t==ASTNULL ? null :(JQLAST)_t;
			JQLAST op2_AST_in = null;
			op2_AST = (JQLAST)astFactory.create(op2);
			astFactory.addASTChild(currentAST, op2_AST);
			ASTPair __currentAST40 = currentAST.copy();
			currentAST.root = currentAST.child;
			currentAST.child = null;
			match(_t,BOR);
			_t = _t.getFirstChild();
			left2 = _t==ASTNULL ? null : (JQLAST)_t;
			expression(_t);
			_t = _retTree;
			left2_AST = (JQLAST)returnAST;
			astFactory.addASTChild(currentAST, returnAST);
			right2 = _t==ASTNULL ? null : (JQLAST)_t;
			expression(_t);
			_t = _retTree;
			right2_AST = (JQLAST)returnAST;
			astFactory.addASTChild(currentAST, returnAST);
			currentAST = __currentAST40;
			_t = __t40;
			_t = _t.getNextSibling();
			
			op2_AST.setJQLType(analyseBitwiseExpr(op2_AST, left2_AST, right2_AST));
			
			bitwiseExpr_AST = (JQLAST)currentAST.root;
			break;
		}
		case BXOR:
		{
			AST __t41 = _t;
			op3 = _t==ASTNULL ? null :(JQLAST)_t;
			JQLAST op3_AST_in = null;
			op3_AST = (JQLAST)astFactory.create(op3);
			astFactory.addASTChild(currentAST, op3_AST);
			ASTPair __currentAST41 = currentAST.copy();
			currentAST.root = currentAST.child;
			currentAST.child = null;
			match(_t,BXOR);
			_t = _t.getFirstChild();
			left3 = _t==ASTNULL ? null : (JQLAST)_t;
			expression(_t);
			_t = _retTree;
			left3_AST = (JQLAST)returnAST;
			astFactory.addASTChild(currentAST, returnAST);
			right3 = _t==ASTNULL ? null : (JQLAST)_t;
			expression(_t);
			_t = _retTree;
			right3_AST = (JQLAST)returnAST;
			astFactory.addASTChild(currentAST, returnAST);
			currentAST = __currentAST41;
			_t = __t41;
			_t = _t.getNextSibling();
			
			op3_AST.setJQLType(analyseBitwiseExpr(op3_AST, left3_AST, right3_AST));
			
			bitwiseExpr_AST = (JQLAST)currentAST.root;
			break;
		}
		default:
		{
			throw new NoViableAltException(_t);
		}
		}
		returnAST = bitwiseExpr_AST;
		_retTree = _t;
	}
	
	public final void conditionalExpr(AST _t) throws RecognitionException {
		
		JQLAST conditionalExpr_AST_in = (_t == ASTNULL) ? null : (JQLAST)_t;
		returnAST = null;
		ASTPair currentAST = new ASTPair();
		JQLAST conditionalExpr_AST = null;
		JQLAST op1 = null;
		JQLAST op1_AST = null;
		JQLAST left1_AST = null;
		JQLAST left1 = null;
		JQLAST right1_AST = null;
		JQLAST right1 = null;
		JQLAST op2 = null;
		JQLAST op2_AST = null;
		JQLAST left2_AST = null;
		JQLAST left2 = null;
		JQLAST right2_AST = null;
		JQLAST right2 = null;
		
		if (_t==null) _t=ASTNULL;
		switch ( _t.getType()) {
		case AND:
		{
			AST __t43 = _t;
			op1 = _t==ASTNULL ? null :(JQLAST)_t;
			JQLAST op1_AST_in = null;
			op1_AST = (JQLAST)astFactory.create(op1);
			astFactory.addASTChild(currentAST, op1_AST);
			ASTPair __currentAST43 = currentAST.copy();
			currentAST.root = currentAST.child;
			currentAST.child = null;
			match(_t,AND);
			_t = _t.getFirstChild();
			left1 = _t==ASTNULL ? null : (JQLAST)_t;
			expression(_t);
			_t = _retTree;
			left1_AST = (JQLAST)returnAST;
			astFactory.addASTChild(currentAST, returnAST);
			right1 = _t==ASTNULL ? null : (JQLAST)_t;
			expression(_t);
			_t = _retTree;
			right1_AST = (JQLAST)returnAST;
			astFactory.addASTChild(currentAST, returnAST);
			currentAST = __currentAST43;
			_t = __t43;
			_t = _t.getNextSibling();
			
			op1_AST.setJQLType(analyseConditionalExpr(op1_AST, left1_AST, right1_AST));
			
			conditionalExpr_AST = (JQLAST)currentAST.root;
			break;
		}
		case OR:
		{
			AST __t44 = _t;
			op2 = _t==ASTNULL ? null :(JQLAST)_t;
			JQLAST op2_AST_in = null;
			op2_AST = (JQLAST)astFactory.create(op2);
			astFactory.addASTChild(currentAST, op2_AST);
			ASTPair __currentAST44 = currentAST.copy();
			currentAST.root = currentAST.child;
			currentAST.child = null;
			match(_t,OR);
			_t = _t.getFirstChild();
			left2 = _t==ASTNULL ? null : (JQLAST)_t;
			expression(_t);
			_t = _retTree;
			left2_AST = (JQLAST)returnAST;
			astFactory.addASTChild(currentAST, returnAST);
			right2 = _t==ASTNULL ? null : (JQLAST)_t;
			expression(_t);
			_t = _retTree;
			right2_AST = (JQLAST)returnAST;
			astFactory.addASTChild(currentAST, returnAST);
			currentAST = __currentAST44;
			_t = __t44;
			_t = _t.getNextSibling();
			
			op2_AST.setJQLType(analyseConditionalExpr(op2_AST, left2_AST, right2_AST));
			
			conditionalExpr_AST = (JQLAST)currentAST.root;
			break;
		}
		default:
		{
			throw new NoViableAltException(_t);
		}
		}
		returnAST = conditionalExpr_AST;
		_retTree = _t;
	}
	
	public final void relationalExpr(AST _t) throws RecognitionException {
		
		JQLAST relationalExpr_AST_in = (_t == ASTNULL) ? null : (JQLAST)_t;
		returnAST = null;
		ASTPair currentAST = new ASTPair();
		JQLAST relationalExpr_AST = null;
		JQLAST op1 = null;
		JQLAST op1_AST = null;
		JQLAST left1_AST = null;
		JQLAST left1 = null;
		JQLAST right1_AST = null;
		JQLAST right1 = null;
		JQLAST op2 = null;
		JQLAST op2_AST = null;
		JQLAST left2_AST = null;
		JQLAST left2 = null;
		JQLAST right2_AST = null;
		JQLAST right2 = null;
		JQLAST op3 = null;
		JQLAST op3_AST = null;
		JQLAST left3_AST = null;
		JQLAST left3 = null;
		JQLAST right3_AST = null;
		JQLAST right3 = null;
		JQLAST op4 = null;
		JQLAST op4_AST = null;
		JQLAST left4_AST = null;
		JQLAST left4 = null;
		JQLAST right4_AST = null;
		JQLAST right4 = null;
		JQLAST op5 = null;
		JQLAST op5_AST = null;
		JQLAST left5_AST = null;
		JQLAST left5 = null;
		JQLAST right5_AST = null;
		JQLAST right5 = null;
		JQLAST op6 = null;
		JQLAST op6_AST = null;
		JQLAST left6_AST = null;
		JQLAST left6 = null;
		JQLAST right6_AST = null;
		JQLAST right6 = null;
		
		Type left = null;
		Type right = null;
		
		
		if (_t==null) _t=ASTNULL;
		switch ( _t.getType()) {
		case EQUAL:
		{
			AST __t46 = _t;
			op1 = _t==ASTNULL ? null :(JQLAST)_t;
			JQLAST op1_AST_in = null;
			op1_AST = (JQLAST)astFactory.create(op1);
			astFactory.addASTChild(currentAST, op1_AST);
			ASTPair __currentAST46 = currentAST.copy();
			currentAST.root = currentAST.child;
			currentAST.child = null;
			match(_t,EQUAL);
			_t = _t.getFirstChild();
			left1 = _t==ASTNULL ? null : (JQLAST)_t;
			expression(_t);
			_t = _retTree;
			left1_AST = (JQLAST)returnAST;
			astFactory.addASTChild(currentAST, returnAST);
			right1 = _t==ASTNULL ? null : (JQLAST)_t;
			expression(_t);
			_t = _retTree;
			right1_AST = (JQLAST)returnAST;
			astFactory.addASTChild(currentAST, returnAST);
			currentAST = __currentAST46;
			_t = __t46;
			_t = _t.getNextSibling();
			
			op1_AST.setJQLType(analyseRelationalExpr(op1_AST, left1_AST, right1_AST));
			left = left1_AST.getJQLType();
			right = right1_AST.getJQLType();
			if (typetab.isPersistenceCapableType(left) || typetab.isPersistenceCapableType(right))
			{
			op1_AST.setType(OBJECT_EQUAL);
			}
			else if (typetab.isCollectionType(left) || typetab.isCollectionType(right))
			{
			op1_AST.setType(COLLECTION_EQUAL);
			}
			
			relationalExpr_AST = (JQLAST)currentAST.root;
			break;
		}
		case NOT_EQUAL:
		{
			AST __t47 = _t;
			op2 = _t==ASTNULL ? null :(JQLAST)_t;
			JQLAST op2_AST_in = null;
			op2_AST = (JQLAST)astFactory.create(op2);
			astFactory.addASTChild(currentAST, op2_AST);
			ASTPair __currentAST47 = currentAST.copy();
			currentAST.root = currentAST.child;
			currentAST.child = null;
			match(_t,NOT_EQUAL);
			_t = _t.getFirstChild();
			left2 = _t==ASTNULL ? null : (JQLAST)_t;
			expression(_t);
			_t = _retTree;
			left2_AST = (JQLAST)returnAST;
			astFactory.addASTChild(currentAST, returnAST);
			right2 = _t==ASTNULL ? null : (JQLAST)_t;
			expression(_t);
			_t = _retTree;
			right2_AST = (JQLAST)returnAST;
			astFactory.addASTChild(currentAST, returnAST);
			currentAST = __currentAST47;
			_t = __t47;
			_t = _t.getNextSibling();
			
			op2_AST.setJQLType(analyseRelationalExpr(op2_AST, left2_AST, right2_AST));
			left = left2_AST.getJQLType();
			right = right2_AST.getJQLType();
			if (typetab.isPersistenceCapableType(left) || typetab.isPersistenceCapableType(right))
			{
			op2_AST.setType(OBJECT_NOT_EQUAL);
			}
			else if (typetab.isCollectionType(left) || typetab.isCollectionType(right))
			{
			op2_AST.setType(COLLECTION_NOT_EQUAL);
			}
			
			relationalExpr_AST = (JQLAST)currentAST.root;
			break;
		}
		case LT:
		{
			AST __t48 = _t;
			op3 = _t==ASTNULL ? null :(JQLAST)_t;
			JQLAST op3_AST_in = null;
			op3_AST = (JQLAST)astFactory.create(op3);
			astFactory.addASTChild(currentAST, op3_AST);
			ASTPair __currentAST48 = currentAST.copy();
			currentAST.root = currentAST.child;
			currentAST.child = null;
			match(_t,LT);
			_t = _t.getFirstChild();
			left3 = _t==ASTNULL ? null : (JQLAST)_t;
			expression(_t);
			_t = _retTree;
			left3_AST = (JQLAST)returnAST;
			astFactory.addASTChild(currentAST, returnAST);
			right3 = _t==ASTNULL ? null : (JQLAST)_t;
			expression(_t);
			_t = _retTree;
			right3_AST = (JQLAST)returnAST;
			astFactory.addASTChild(currentAST, returnAST);
			currentAST = __currentAST48;
			_t = __t48;
			_t = _t.getNextSibling();
			
			op3_AST.setJQLType(analyseRelationalExpr(op3_AST, left3_AST, right3_AST));
			
			relationalExpr_AST = (JQLAST)currentAST.root;
			break;
		}
		case GT:
		{
			AST __t49 = _t;
			op4 = _t==ASTNULL ? null :(JQLAST)_t;
			JQLAST op4_AST_in = null;
			op4_AST = (JQLAST)astFactory.create(op4);
			astFactory.addASTChild(currentAST, op4_AST);
			ASTPair __currentAST49 = currentAST.copy();
			currentAST.root = currentAST.child;
			currentAST.child = null;
			match(_t,GT);
			_t = _t.getFirstChild();
			left4 = _t==ASTNULL ? null : (JQLAST)_t;
			expression(_t);
			_t = _retTree;
			left4_AST = (JQLAST)returnAST;
			astFactory.addASTChild(currentAST, returnAST);
			right4 = _t==ASTNULL ? null : (JQLAST)_t;
			expression(_t);
			_t = _retTree;
			right4_AST = (JQLAST)returnAST;
			astFactory.addASTChild(currentAST, returnAST);
			currentAST = __currentAST49;
			_t = __t49;
			_t = _t.getNextSibling();
			
			op4_AST.setJQLType(analyseRelationalExpr(op4_AST, left4_AST, right4_AST));
			
			relationalExpr_AST = (JQLAST)currentAST.root;
			break;
		}
		case LE:
		{
			AST __t50 = _t;
			op5 = _t==ASTNULL ? null :(JQLAST)_t;
			JQLAST op5_AST_in = null;
			op5_AST = (JQLAST)astFactory.create(op5);
			astFactory.addASTChild(currentAST, op5_AST);
			ASTPair __currentAST50 = currentAST.copy();
			currentAST.root = currentAST.child;
			currentAST.child = null;
			match(_t,LE);
			_t = _t.getFirstChild();
			left5 = _t==ASTNULL ? null : (JQLAST)_t;
			expression(_t);
			_t = _retTree;
			left5_AST = (JQLAST)returnAST;
			astFactory.addASTChild(currentAST, returnAST);
			right5 = _t==ASTNULL ? null : (JQLAST)_t;
			expression(_t);
			_t = _retTree;
			right5_AST = (JQLAST)returnAST;
			astFactory.addASTChild(currentAST, returnAST);
			currentAST = __currentAST50;
			_t = __t50;
			_t = _t.getNextSibling();
			
			op5_AST.setJQLType(analyseRelationalExpr(op5_AST, left5_AST, right5_AST));
			
			relationalExpr_AST = (JQLAST)currentAST.root;
			break;
		}
		case GE:
		{
			AST __t51 = _t;
			op6 = _t==ASTNULL ? null :(JQLAST)_t;
			JQLAST op6_AST_in = null;
			op6_AST = (JQLAST)astFactory.create(op6);
			astFactory.addASTChild(currentAST, op6_AST);
			ASTPair __currentAST51 = currentAST.copy();
			currentAST.root = currentAST.child;
			currentAST.child = null;
			match(_t,GE);
			_t = _t.getFirstChild();
			left6 = _t==ASTNULL ? null : (JQLAST)_t;
			expression(_t);
			_t = _retTree;
			left6_AST = (JQLAST)returnAST;
			astFactory.addASTChild(currentAST, returnAST);
			right6 = _t==ASTNULL ? null : (JQLAST)_t;
			expression(_t);
			_t = _retTree;
			right6_AST = (JQLAST)returnAST;
			astFactory.addASTChild(currentAST, returnAST);
			currentAST = __currentAST51;
			_t = __t51;
			_t = _t.getNextSibling();
			
			op6_AST.setJQLType(analyseRelationalExpr(op6_AST, left6_AST, right6_AST));
			
			relationalExpr_AST = (JQLAST)currentAST.root;
			break;
		}
		default:
		{
			throw new NoViableAltException(_t);
		}
		}
		returnAST = relationalExpr_AST;
		_retTree = _t;
	}
	
	public final void binaryArithmeticExpr(AST _t) throws RecognitionException {
		
		JQLAST binaryArithmeticExpr_AST_in = (_t == ASTNULL) ? null : (JQLAST)_t;
		returnAST = null;
		ASTPair currentAST = new ASTPair();
		JQLAST binaryArithmeticExpr_AST = null;
		JQLAST op1 = null;
		JQLAST op1_AST = null;
		JQLAST left1_AST = null;
		JQLAST left1 = null;
		JQLAST right1_AST = null;
		JQLAST right1 = null;
		JQLAST op2 = null;
		JQLAST op2_AST = null;
		JQLAST left2_AST = null;
		JQLAST left2 = null;
		JQLAST right2_AST = null;
		JQLAST right2 = null;
		JQLAST op3 = null;
		JQLAST op3_AST = null;
		JQLAST left3_AST = null;
		JQLAST left3 = null;
		JQLAST right3_AST = null;
		JQLAST right3 = null;
		JQLAST op4 = null;
		JQLAST op4_AST = null;
		JQLAST left4_AST = null;
		JQLAST left4 = null;
		JQLAST right4_AST = null;
		JQLAST right4 = null;
		JQLAST op5 = null;
		JQLAST op5_AST = null;
		JQLAST left5_AST = null;
		JQLAST left5 = null;
		JQLAST right5_AST = null;
		JQLAST right5 = null;
		
		if (_t==null) _t=ASTNULL;
		switch ( _t.getType()) {
		case PLUS:
		{
			AST __t53 = _t;
			op1 = _t==ASTNULL ? null :(JQLAST)_t;
			JQLAST op1_AST_in = null;
			op1_AST = (JQLAST)astFactory.create(op1);
			astFactory.addASTChild(currentAST, op1_AST);
			ASTPair __currentAST53 = currentAST.copy();
			currentAST.root = currentAST.child;
			currentAST.child = null;
			match(_t,PLUS);
			_t = _t.getFirstChild();
			left1 = _t==ASTNULL ? null : (JQLAST)_t;
			expression(_t);
			_t = _retTree;
			left1_AST = (JQLAST)returnAST;
			astFactory.addASTChild(currentAST, returnAST);
			right1 = _t==ASTNULL ? null : (JQLAST)_t;
			expression(_t);
			_t = _retTree;
			right1_AST = (JQLAST)returnAST;
			astFactory.addASTChild(currentAST, returnAST);
			currentAST = __currentAST53;
			_t = __t53;
			_t = _t.getNextSibling();
			
			op1_AST.setJQLType(analyseBinaryArithmeticExpr(op1_AST, left1_AST, right1_AST));
			if (op1_AST.getJQLType().equals(typetab.stringType))
			{
			// change the operator from PLUS to CONCAT in the case of string concatenation
			op1_AST.setType(CONCAT);
			}
			
			binaryArithmeticExpr_AST = (JQLAST)currentAST.root;
			break;
		}
		case MINUS:
		{
			AST __t54 = _t;
			op2 = _t==ASTNULL ? null :(JQLAST)_t;
			JQLAST op2_AST_in = null;
			op2_AST = (JQLAST)astFactory.create(op2);
			astFactory.addASTChild(currentAST, op2_AST);
			ASTPair __currentAST54 = currentAST.copy();
			currentAST.root = currentAST.child;
			currentAST.child = null;
			match(_t,MINUS);
			_t = _t.getFirstChild();
			left2 = _t==ASTNULL ? null : (JQLAST)_t;
			expression(_t);
			_t = _retTree;
			left2_AST = (JQLAST)returnAST;
			astFactory.addASTChild(currentAST, returnAST);
			right2 = _t==ASTNULL ? null : (JQLAST)_t;
			expression(_t);
			_t = _retTree;
			right2_AST = (JQLAST)returnAST;
			astFactory.addASTChild(currentAST, returnAST);
			currentAST = __currentAST54;
			_t = __t54;
			_t = _t.getNextSibling();
			
			op2_AST.setJQLType(analyseBinaryArithmeticExpr(op2_AST, left2_AST, right2_AST));
			
			binaryArithmeticExpr_AST = (JQLAST)currentAST.root;
			break;
		}
		case STAR:
		{
			AST __t55 = _t;
			op3 = _t==ASTNULL ? null :(JQLAST)_t;
			JQLAST op3_AST_in = null;
			op3_AST = (JQLAST)astFactory.create(op3);
			astFactory.addASTChild(currentAST, op3_AST);
			ASTPair __currentAST55 = currentAST.copy();
			currentAST.root = currentAST.child;
			currentAST.child = null;
			match(_t,STAR);
			_t = _t.getFirstChild();
			left3 = _t==ASTNULL ? null : (JQLAST)_t;
			expression(_t);
			_t = _retTree;
			left3_AST = (JQLAST)returnAST;
			astFactory.addASTChild(currentAST, returnAST);
			right3 = _t==ASTNULL ? null : (JQLAST)_t;
			expression(_t);
			_t = _retTree;
			right3_AST = (JQLAST)returnAST;
			astFactory.addASTChild(currentAST, returnAST);
			currentAST = __currentAST55;
			_t = __t55;
			_t = _t.getNextSibling();
			
			op3_AST.setJQLType(analyseBinaryArithmeticExpr(op3_AST, left3_AST, right3_AST));
			
			binaryArithmeticExpr_AST = (JQLAST)currentAST.root;
			break;
		}
		case DIV:
		{
			AST __t56 = _t;
			op4 = _t==ASTNULL ? null :(JQLAST)_t;
			JQLAST op4_AST_in = null;
			op4_AST = (JQLAST)astFactory.create(op4);
			astFactory.addASTChild(currentAST, op4_AST);
			ASTPair __currentAST56 = currentAST.copy();
			currentAST.root = currentAST.child;
			currentAST.child = null;
			match(_t,DIV);
			_t = _t.getFirstChild();
			left4 = _t==ASTNULL ? null : (JQLAST)_t;
			expression(_t);
			_t = _retTree;
			left4_AST = (JQLAST)returnAST;
			astFactory.addASTChild(currentAST, returnAST);
			right4 = _t==ASTNULL ? null : (JQLAST)_t;
			expression(_t);
			_t = _retTree;
			right4_AST = (JQLAST)returnAST;
			astFactory.addASTChild(currentAST, returnAST);
			currentAST = __currentAST56;
			_t = __t56;
			_t = _t.getNextSibling();
			
			op4_AST.setJQLType(analyseBinaryArithmeticExpr(op4_AST, left4_AST, right4_AST));
			
			binaryArithmeticExpr_AST = (JQLAST)currentAST.root;
			break;
		}
		case MOD:
		{
			AST __t57 = _t;
			op5 = _t==ASTNULL ? null :(JQLAST)_t;
			JQLAST op5_AST_in = null;
			op5_AST = (JQLAST)astFactory.create(op5);
			astFactory.addASTChild(currentAST, op5_AST);
			ASTPair __currentAST57 = currentAST.copy();
			currentAST.root = currentAST.child;
			currentAST.child = null;
			match(_t,MOD);
			_t = _t.getFirstChild();
			left5 = _t==ASTNULL ? null : (JQLAST)_t;
			expression(_t);
			_t = _retTree;
			left5_AST = (JQLAST)returnAST;
			astFactory.addASTChild(currentAST, returnAST);
			right5 = _t==ASTNULL ? null : (JQLAST)_t;
			expression(_t);
			_t = _retTree;
			right5_AST = (JQLAST)returnAST;
			astFactory.addASTChild(currentAST, returnAST);
			currentAST = __currentAST57;
			_t = __t57;
			_t = _t.getNextSibling();
			
			op5_AST.setJQLType(analyseBinaryArithmeticExpr(op5_AST, left5_AST, right5_AST));
			
			binaryArithmeticExpr_AST = (JQLAST)currentAST.root;
			break;
		}
		default:
		{
			throw new NoViableAltException(_t);
		}
		}
		returnAST = binaryArithmeticExpr_AST;
		_retTree = _t;
	}
	
	public final void unaryArithmeticExpr(AST _t) throws RecognitionException {
		
		JQLAST unaryArithmeticExpr_AST_in = (_t == ASTNULL) ? null : (JQLAST)_t;
		returnAST = null;
		ASTPair currentAST = new ASTPair();
		JQLAST unaryArithmeticExpr_AST = null;
		JQLAST op1 = null;
		JQLAST op1_AST = null;
		JQLAST arg1_AST = null;
		JQLAST arg1 = null;
		JQLAST op2 = null;
		JQLAST op2_AST = null;
		JQLAST arg2_AST = null;
		JQLAST arg2 = null;
		
		if (_t==null) _t=ASTNULL;
		switch ( _t.getType()) {
		case UNARY_PLUS:
		{
			AST __t59 = _t;
			op1 = _t==ASTNULL ? null :(JQLAST)_t;
			JQLAST op1_AST_in = null;
			op1_AST = (JQLAST)astFactory.create(op1);
			astFactory.addASTChild(currentAST, op1_AST);
			ASTPair __currentAST59 = currentAST.copy();
			currentAST.root = currentAST.child;
			currentAST.child = null;
			match(_t,UNARY_PLUS);
			_t = _t.getFirstChild();
			arg1 = _t==ASTNULL ? null : (JQLAST)_t;
			expression(_t);
			_t = _retTree;
			arg1_AST = (JQLAST)returnAST;
			astFactory.addASTChild(currentAST, returnAST);
			currentAST = __currentAST59;
			_t = __t59;
			_t = _t.getNextSibling();
			
			op1_AST.setJQLType(analyseUnaryArithmeticExpr(op1_AST, arg1_AST));
			
			unaryArithmeticExpr_AST = (JQLAST)currentAST.root;
			break;
		}
		case UNARY_MINUS:
		{
			AST __t60 = _t;
			op2 = _t==ASTNULL ? null :(JQLAST)_t;
			JQLAST op2_AST_in = null;
			op2_AST = (JQLAST)astFactory.create(op2);
			astFactory.addASTChild(currentAST, op2_AST);
			ASTPair __currentAST60 = currentAST.copy();
			currentAST.root = currentAST.child;
			currentAST.child = null;
			match(_t,UNARY_MINUS);
			_t = _t.getFirstChild();
			arg2 = _t==ASTNULL ? null : (JQLAST)_t;
			expression(_t);
			_t = _retTree;
			arg2_AST = (JQLAST)returnAST;
			astFactory.addASTChild(currentAST, returnAST);
			currentAST = __currentAST60;
			_t = __t60;
			_t = _t.getNextSibling();
			
			op2_AST.setJQLType(analyseUnaryArithmeticExpr(op2_AST, arg2_AST));
			
			unaryArithmeticExpr_AST = (JQLAST)currentAST.root;
			break;
		}
		default:
		{
			throw new NoViableAltException(_t);
		}
		}
		returnAST = unaryArithmeticExpr_AST;
		_retTree = _t;
	}
	
	public final void complementExpr(AST _t) throws RecognitionException {
		
		JQLAST complementExpr_AST_in = (_t == ASTNULL) ? null : (JQLAST)_t;
		returnAST = null;
		ASTPair currentAST = new ASTPair();
		JQLAST complementExpr_AST = null;
		JQLAST op1 = null;
		JQLAST op1_AST = null;
		JQLAST arg1_AST = null;
		JQLAST arg1 = null;
		JQLAST op2 = null;
		JQLAST op2_AST = null;
		JQLAST arg2_AST = null;
		JQLAST arg2 = null;
		
		if (_t==null) _t=ASTNULL;
		switch ( _t.getType()) {
		case BNOT:
		{
			AST __t62 = _t;
			op1 = _t==ASTNULL ? null :(JQLAST)_t;
			JQLAST op1_AST_in = null;
			op1_AST = (JQLAST)astFactory.create(op1);
			astFactory.addASTChild(currentAST, op1_AST);
			ASTPair __currentAST62 = currentAST.copy();
			currentAST.root = currentAST.child;
			currentAST.child = null;
			match(_t,BNOT);
			_t = _t.getFirstChild();
			arg1 = _t==ASTNULL ? null : (JQLAST)_t;
			expression(_t);
			_t = _retTree;
			arg1_AST = (JQLAST)returnAST;
			astFactory.addASTChild(currentAST, returnAST);
			currentAST = __currentAST62;
			_t = __t62;
			_t = _t.getNextSibling();
			
			op1_AST.setJQLType(analyseComplementExpr(op1_AST, arg1_AST));
			
			complementExpr_AST = (JQLAST)currentAST.root;
			break;
		}
		case LNOT:
		{
			AST __t63 = _t;
			op2 = _t==ASTNULL ? null :(JQLAST)_t;
			JQLAST op2_AST_in = null;
			op2_AST = (JQLAST)astFactory.create(op2);
			astFactory.addASTChild(currentAST, op2_AST);
			ASTPair __currentAST63 = currentAST.copy();
			currentAST.root = currentAST.child;
			currentAST.child = null;
			match(_t,LNOT);
			_t = _t.getFirstChild();
			arg2 = _t==ASTNULL ? null : (JQLAST)_t;
			expression(_t);
			_t = _retTree;
			arg2_AST = (JQLAST)returnAST;
			astFactory.addASTChild(currentAST, returnAST);
			currentAST = __currentAST63;
			_t = __t63;
			_t = _t.getNextSibling();
			
			op2_AST.setJQLType(analyseComplementExpr(op2_AST, arg2_AST));
			
			complementExpr_AST = (JQLAST)currentAST.root;
			break;
		}
		default:
		{
			throw new NoViableAltException(_t);
		}
		}
		returnAST = complementExpr_AST;
		_retTree = _t;
	}
	
	public final String  primary(AST _t,
		boolean insideDotExpr
	) throws RecognitionException {
		String repr;
		
		JQLAST primary_AST_in = (_t == ASTNULL) ? null : (JQLAST)_t;
		returnAST = null;
		ASTPair currentAST = new ASTPair();
		JQLAST primary_AST = null;
		JQLAST c = null;
		JQLAST c_AST = null;
		JQLAST t_AST = null;
		JQLAST t = null;
		JQLAST e_AST = null;
		JQLAST e = null;
		JQLAST i = null;
		JQLAST i_AST = null;
		repr = null;
		
		if (_t==null) _t=ASTNULL;
		switch ( _t.getType()) {
		case TYPECAST:
		{
			AST __t65 = _t;
			c = _t==ASTNULL ? null :(JQLAST)_t;
			JQLAST c_AST_in = null;
			c_AST = (JQLAST)astFactory.create(c);
			astFactory.addASTChild(currentAST, c_AST);
			ASTPair __currentAST65 = currentAST.copy();
			currentAST.root = currentAST.child;
			currentAST.child = null;
			match(_t,TYPECAST);
			_t = _t.getFirstChild();
			t = _t==ASTNULL ? null : (JQLAST)_t;
			type(_t);
			_t = _retTree;
			t_AST = (JQLAST)returnAST;
			astFactory.addASTChild(currentAST, returnAST);
			e = _t==ASTNULL ? null : (JQLAST)_t;
			expression(_t);
			_t = _retTree;
			e_AST = (JQLAST)returnAST;
			astFactory.addASTChild(currentAST, returnAST);
			currentAST = __currentAST65;
			_t = __t65;
			_t = _t.getNextSibling();
			
			Type type = t_AST.getJQLType();
			Type exprType = e_AST.getJQLType();
			if (!(type.isCompatibleWith(exprType) || exprType.isCompatibleWith(type)))
			{
			errorMsg.error(c_AST.getLine(), c_AST.getColumn(),
			I18NHelper.getMessage(messages, "jqlc.semantic.primary.invalidcast", //NOI18N
			exprType.getName(), type.getName()));
			type = typetab.errorType;
			}
			c_AST.setJQLType(type);
			
			primary_AST = (JQLAST)currentAST.root;
			break;
		}
		case NULL:
		case TRUE:
		case FALSE:
		case CHAR_LITERAL:
		case STRING_LITERAL:
		case INT_LITERAL:
		case LONG_LITERAL:
		case FLOAT_LITERAL:
		case DOUBLE_LITERAL:
		{
			literal(_t);
			_t = _retTree;
			astFactory.addASTChild(currentAST, returnAST);
			primary_AST = (JQLAST)currentAST.root;
			break;
		}
		case THIS:
		{
			i = (JQLAST)_t;
			JQLAST i_AST_in = null;
			i_AST = (JQLAST)astFactory.create(i);
			astFactory.addASTChild(currentAST, i_AST);
			match(_t,THIS);
			_t = _t.getNextSibling();
			i_AST.setJQLType(candidateClass);
			primary_AST = (JQLAST)currentAST.root;
			break;
		}
		case DOT:
		{
			repr=dotExpr(_t);
			_t = _retTree;
			astFactory.addASTChild(currentAST, returnAST);
			primary_AST = (JQLAST)currentAST.root;
			break;
		}
		case IDENT:
		{
			repr=identifier(_t,insideDotExpr);
			_t = _retTree;
			astFactory.addASTChild(currentAST, returnAST);
			primary_AST = (JQLAST)currentAST.root;
			break;
		}
		default:
		{
			throw new NoViableAltException(_t);
		}
		}
		returnAST = primary_AST;
		_retTree = _t;
		return repr;
	}
	
	public final void literal(AST _t) throws RecognitionException {
		
		JQLAST literal_AST_in = (_t == ASTNULL) ? null : (JQLAST)_t;
		returnAST = null;
		ASTPair currentAST = new ASTPair();
		JQLAST literal_AST = null;
		JQLAST b1 = null;
		JQLAST b1_AST = null;
		JQLAST b2 = null;
		JQLAST b2_AST = null;
		JQLAST i = null;
		JQLAST i_AST = null;
		JQLAST l = null;
		JQLAST l_AST = null;
		JQLAST f = null;
		JQLAST f_AST = null;
		JQLAST d = null;
		JQLAST d_AST = null;
		JQLAST c = null;
		JQLAST c_AST = null;
		JQLAST s = null;
		JQLAST s_AST = null;
		JQLAST n = null;
		JQLAST n_AST = null;
		
		if (_t==null) _t=ASTNULL;
		switch ( _t.getType()) {
		case TRUE:
		{
			b1 = (JQLAST)_t;
			JQLAST b1_AST_in = null;
			b1_AST = (JQLAST)astFactory.create(b1);
			astFactory.addASTChild(currentAST, b1_AST);
			match(_t,TRUE);
			_t = _t.getNextSibling();
			b1_AST.setJQLType(typetab.booleanType);
			literal_AST = (JQLAST)currentAST.root;
			break;
		}
		case FALSE:
		{
			b2 = (JQLAST)_t;
			JQLAST b2_AST_in = null;
			b2_AST = (JQLAST)astFactory.create(b2);
			astFactory.addASTChild(currentAST, b2_AST);
			match(_t,FALSE);
			_t = _t.getNextSibling();
			b2_AST.setJQLType(typetab.booleanType);
			literal_AST = (JQLAST)currentAST.root;
			break;
		}
		case INT_LITERAL:
		{
			i = (JQLAST)_t;
			JQLAST i_AST_in = null;
			i_AST = (JQLAST)astFactory.create(i);
			astFactory.addASTChild(currentAST, i_AST);
			match(_t,INT_LITERAL);
			_t = _t.getNextSibling();
			i_AST.setJQLType(typetab.intType);
			literal_AST = (JQLAST)currentAST.root;
			break;
		}
		case LONG_LITERAL:
		{
			l = (JQLAST)_t;
			JQLAST l_AST_in = null;
			l_AST = (JQLAST)astFactory.create(l);
			astFactory.addASTChild(currentAST, l_AST);
			match(_t,LONG_LITERAL);
			_t = _t.getNextSibling();
			l_AST.setJQLType(typetab.longType);
			literal_AST = (JQLAST)currentAST.root;
			break;
		}
		case FLOAT_LITERAL:
		{
			f = (JQLAST)_t;
			JQLAST f_AST_in = null;
			f_AST = (JQLAST)astFactory.create(f);
			astFactory.addASTChild(currentAST, f_AST);
			match(_t,FLOAT_LITERAL);
			_t = _t.getNextSibling();
			f_AST.setJQLType(typetab.floatType);
			literal_AST = (JQLAST)currentAST.root;
			break;
		}
		case DOUBLE_LITERAL:
		{
			d = (JQLAST)_t;
			JQLAST d_AST_in = null;
			d_AST = (JQLAST)astFactory.create(d);
			astFactory.addASTChild(currentAST, d_AST);
			match(_t,DOUBLE_LITERAL);
			_t = _t.getNextSibling();
			d_AST.setJQLType(typetab.doubleType);
			literal_AST = (JQLAST)currentAST.root;
			break;
		}
		case CHAR_LITERAL:
		{
			c = (JQLAST)_t;
			JQLAST c_AST_in = null;
			c_AST = (JQLAST)astFactory.create(c);
			astFactory.addASTChild(currentAST, c_AST);
			match(_t,CHAR_LITERAL);
			_t = _t.getNextSibling();
			c_AST.setJQLType(typetab.charType);
			literal_AST = (JQLAST)currentAST.root;
			break;
		}
		case STRING_LITERAL:
		{
			s = (JQLAST)_t;
			JQLAST s_AST_in = null;
			s_AST = (JQLAST)astFactory.create(s);
			astFactory.addASTChild(currentAST, s_AST);
			match(_t,STRING_LITERAL);
			_t = _t.getNextSibling();
			s_AST.setJQLType(typetab.stringType);
			literal_AST = (JQLAST)currentAST.root;
			break;
		}
		case NULL:
		{
			n = (JQLAST)_t;
			JQLAST n_AST_in = null;
			n_AST = (JQLAST)astFactory.create(n);
			astFactory.addASTChild(currentAST, n_AST);
			match(_t,NULL);
			_t = _t.getNextSibling();
			n_AST.setJQLType(typetab.nullType);
			literal_AST = (JQLAST)currentAST.root;
			break;
		}
		default:
		{
			throw new NoViableAltException(_t);
		}
		}
		returnAST = literal_AST;
		_retTree = _t;
	}
	
	public final String  dotExpr(AST _t) throws RecognitionException {
		String repr;
		
		JQLAST dotExpr_AST_in = (_t == ASTNULL) ? null : (JQLAST)_t;
		returnAST = null;
		ASTPair currentAST = new ASTPair();
		JQLAST dotExpr_AST = null;
		JQLAST dot = null;
		JQLAST dot_AST = null;
		JQLAST expr_AST = null;
		JQLAST expr = null;
		JQLAST ident = null;
		JQLAST ident_AST = null;
		JQLAST args_AST = null;
		JQLAST args = null;
		
		repr = null;
		
		
		AST __t67 = _t;
		dot = _t==ASTNULL ? null :(JQLAST)_t;
		JQLAST dot_AST_in = null;
		dot_AST = (JQLAST)astFactory.create(dot);
		astFactory.addASTChild(currentAST, dot_AST);
		ASTPair __currentAST67 = currentAST.copy();
		currentAST.root = currentAST.child;
		currentAST.child = null;
		match(_t,DOT);
		_t = _t.getFirstChild();
		expr = _t==ASTNULL ? null : (JQLAST)_t;
		repr=exprNoCheck(_t,true);
		_t = _retTree;
		expr_AST = (JQLAST)returnAST;
		astFactory.addASTChild(currentAST, returnAST);
		ident = (JQLAST)_t;
		JQLAST ident_AST_in = null;
		ident_AST = (JQLAST)astFactory.create(ident);
		astFactory.addASTChild(currentAST, ident_AST);
		match(_t,IDENT);
		_t = _t.getNextSibling();
		{
		if (_t==null) _t=ASTNULL;
		switch ( _t.getType()) {
		case ARG_LIST:
		{
			args = _t==ASTNULL ? null : (JQLAST)_t;
			argList(_t);
			_t = _retTree;
			args_AST = (JQLAST)returnAST;
			astFactory.addASTChild(currentAST, returnAST);
			break;
		}
		case 3:
		{
			break;
		}
		default:
		{
			throw new NoViableAltException(_t);
		}
		}
		}
		currentAST = __currentAST67;
		_t = __t67;
		_t = _t.getNextSibling();
		dotExpr_AST = (JQLAST)currentAST.root;
		
		Type type = null;
		if (repr != null) // possible package name
		{
		String qualifiedName = repr + '.' + ident_AST.getText();
		type = typetab.checkType(qualifiedName);
		if (type == null)
		{
		// name does not define a valid class => return qualifiedName
		repr = qualifiedName;
		}
		else if (args_AST == null)
		{
		// found valid class name and NO arguments specified
		// => use of the class name
		repr = null;
		dot_AST.setType(TYPENAME);
		dot_AST.setText(qualifiedName);
		dot_AST.setFirstChild(null);
		}
		else
		{
		// found valid class name and arguments specified =>
		// looks like constructor call
		repr = null;
		errorMsg.error(dot.getLine(), dot.getColumn(),
		I18NHelper.getMessage(messages, "jqlc.semantic.generic.invalidmethodcall")); //NOI18N
		}
		dot_AST.setJQLType(type);
		dot_AST.setText(expr_AST.getText() + '.' + ident_AST.getText());
		}
		else // no string repr of left hand side => expression is defined
		{
		dotExpr_AST = analyseDotExpr(dot_AST, expr_AST, ident_AST, args_AST);
		}
		
		currentAST.root = dotExpr_AST;
		currentAST.child = dotExpr_AST!=null &&dotExpr_AST.getFirstChild()!=null ?
			dotExpr_AST.getFirstChild() : dotExpr_AST;
		currentAST.advanceChildToEnd();
		dotExpr_AST = (JQLAST)currentAST.root;
		returnAST = dotExpr_AST;
		_retTree = _t;
		return repr;
	}
	
	public final String  identifier(AST _t,
		boolean insideDotExpr
	) throws RecognitionException {
		String repr;
		
		JQLAST identifier_AST_in = (_t == ASTNULL) ? null : (JQLAST)_t;
		returnAST = null;
		ASTPair currentAST = new ASTPair();
		JQLAST identifier_AST = null;
		JQLAST ident = null;
		JQLAST ident_AST = null;
		JQLAST args_AST = null;
		JQLAST args = null;
		
		repr = null;   // repr is set when ident is part of a package name spec
		
		
		ident = (JQLAST)_t;
		JQLAST ident_AST_in = null;
		ident_AST = (JQLAST)astFactory.create(ident);
		astFactory.addASTChild(currentAST, ident_AST);
		match(_t,IDENT);
		_t = _t.getNextSibling();
		{
		if (_t==null) _t=ASTNULL;
		switch ( _t.getType()) {
		case ARG_LIST:
		{
			args = _t==ASTNULL ? null : (JQLAST)_t;
			argList(_t);
			_t = _retTree;
			args_AST = (JQLAST)returnAST;
			astFactory.addASTChild(currentAST, returnAST);
			break;
		}
		case 3:
		case THIS:
		case NULL:
		case TRUE:
		case FALSE:
		case EQUAL:
		case LNOT:
		case BNOT:
		case NOT_EQUAL:
		case DIV:
		case PLUS:
		case MINUS:
		case STAR:
		case MOD:
		case GE:
		case GT:
		case LE:
		case LT:
		case BXOR:
		case BOR:
		case OR:
		case BAND:
		case AND:
		case CHAR_LITERAL:
		case STRING_LITERAL:
		case INT_LITERAL:
		case IDENT:
		case UNARY_MINUS:
		case UNARY_PLUS:
		case TYPECAST:
		case DOT:
		case LONG_LITERAL:
		case FLOAT_LITERAL:
		case DOUBLE_LITERAL:
		{
			break;
		}
		default:
		{
			throw new NoViableAltException(_t);
		}
		}
		}
		
		String name = ident_AST.getText();
		Definition def = symtab.getDefinition(name);
		
		// check args, if defined => invalid method call
		if (args_AST != null)
		{
		ident_AST.setJQLType(typetab.errorType);
		errorMsg.error(ident_AST.getLine(), ident_AST.getColumn(),
		I18NHelper.getMessage(messages, "jqlc.semantic.generic.invalidmethodcall")); //NOI18N
		}
		else if (def != null)
		{
		ident_AST = analyseDefinedIdentifier(ident_AST, def);
		}
		else if (insideDotExpr)
		{
		Definition typedef = typeNames.getDefinition(name);
		if (typedef != null)
		{
		ident_AST = analyseDefinedIdentifier(ident_AST, typedef);
		}
		else
		{
		repr = ident_AST.getText();
		}
		}
		else
		{
		ident_AST.setJQLType(typetab.errorType);
		errorMsg.error(ident.getLine(), ident.getColumn(),
		I18NHelper.getMessage(messages, "jqlc.semantic.identifier.undefined", //NOI18N
		ident.getText()));
		}
		
		identifier_AST = (JQLAST)currentAST.root;
		returnAST = identifier_AST;
		_retTree = _t;
		return repr;
	}
	
	public final void argList(AST _t) throws RecognitionException {
		
		JQLAST argList_AST_in = (_t == ASTNULL) ? null : (JQLAST)_t;
		returnAST = null;
		ASTPair currentAST = new ASTPair();
		JQLAST argList_AST = null;
		
		AST __t70 = _t;
		JQLAST tmp8_AST = null;
		JQLAST tmp8_AST_in = null;
		tmp8_AST = (JQLAST)astFactory.create((JQLAST)_t);
		tmp8_AST_in = (JQLAST)_t;
		astFactory.addASTChild(currentAST, tmp8_AST);
		ASTPair __currentAST70 = currentAST.copy();
		currentAST.root = currentAST.child;
		currentAST.child = null;
		match(_t,ARG_LIST);
		_t = _t.getFirstChild();
		{
		_loop72:
		do {
			if (_t==null) _t=ASTNULL;
			if ((_tokenSet_0.member(_t.getType()))) {
				expression(_t);
				_t = _retTree;
				astFactory.addASTChild(currentAST, returnAST);
			}
			else {
				break _loop72;
			}
			
		} while (true);
		}
		currentAST = __currentAST70;
		_t = __t70;
		_t = _t.getNextSibling();
		argList_AST = (JQLAST)currentAST.root;
		returnAST = argList_AST;
		_retTree = _t;
	}
	
	public final void primitiveType(AST _t) throws RecognitionException {
		
		JQLAST primitiveType_AST_in = (_t == ASTNULL) ? null : (JQLAST)_t;
		returnAST = null;
		ASTPair currentAST = new ASTPair();
		JQLAST primitiveType_AST = null;
		
		if (_t==null) _t=ASTNULL;
		switch ( _t.getType()) {
		case BOOLEAN:
		{
			JQLAST tmp9_AST = null;
			JQLAST tmp9_AST_in = null;
			tmp9_AST = (JQLAST)astFactory.create((JQLAST)_t);
			tmp9_AST_in = (JQLAST)_t;
			astFactory.addASTChild(currentAST, tmp9_AST);
			match(_t,BOOLEAN);
			_t = _t.getNextSibling();
			primitiveType_AST = (JQLAST)currentAST.root;
			break;
		}
		case BYTE:
		{
			JQLAST tmp10_AST = null;
			JQLAST tmp10_AST_in = null;
			tmp10_AST = (JQLAST)astFactory.create((JQLAST)_t);
			tmp10_AST_in = (JQLAST)_t;
			astFactory.addASTChild(currentAST, tmp10_AST);
			match(_t,BYTE);
			_t = _t.getNextSibling();
			primitiveType_AST = (JQLAST)currentAST.root;
			break;
		}
		case CHAR:
		{
			JQLAST tmp11_AST = null;
			JQLAST tmp11_AST_in = null;
			tmp11_AST = (JQLAST)astFactory.create((JQLAST)_t);
			tmp11_AST_in = (JQLAST)_t;
			astFactory.addASTChild(currentAST, tmp11_AST);
			match(_t,CHAR);
			_t = _t.getNextSibling();
			primitiveType_AST = (JQLAST)currentAST.root;
			break;
		}
		case SHORT:
		{
			JQLAST tmp12_AST = null;
			JQLAST tmp12_AST_in = null;
			tmp12_AST = (JQLAST)astFactory.create((JQLAST)_t);
			tmp12_AST_in = (JQLAST)_t;
			astFactory.addASTChild(currentAST, tmp12_AST);
			match(_t,SHORT);
			_t = _t.getNextSibling();
			primitiveType_AST = (JQLAST)currentAST.root;
			break;
		}
		case INT:
		{
			JQLAST tmp13_AST = null;
			JQLAST tmp13_AST_in = null;
			tmp13_AST = (JQLAST)astFactory.create((JQLAST)_t);
			tmp13_AST_in = (JQLAST)_t;
			astFactory.addASTChild(currentAST, tmp13_AST);
			match(_t,INT);
			_t = _t.getNextSibling();
			primitiveType_AST = (JQLAST)currentAST.root;
			break;
		}
		case FLOAT:
		{
			JQLAST tmp14_AST = null;
			JQLAST tmp14_AST_in = null;
			tmp14_AST = (JQLAST)astFactory.create((JQLAST)_t);
			tmp14_AST_in = (JQLAST)_t;
			astFactory.addASTChild(currentAST, tmp14_AST);
			match(_t,FLOAT);
			_t = _t.getNextSibling();
			primitiveType_AST = (JQLAST)currentAST.root;
			break;
		}
		case LONG:
		{
			JQLAST tmp15_AST = null;
			JQLAST tmp15_AST_in = null;
			tmp15_AST = (JQLAST)astFactory.create((JQLAST)_t);
			tmp15_AST_in = (JQLAST)_t;
			astFactory.addASTChild(currentAST, tmp15_AST);
			match(_t,LONG);
			_t = _t.getNextSibling();
			primitiveType_AST = (JQLAST)currentAST.root;
			break;
		}
		case DOUBLE:
		{
			JQLAST tmp16_AST = null;
			JQLAST tmp16_AST_in = null;
			tmp16_AST = (JQLAST)astFactory.create((JQLAST)_t);
			tmp16_AST_in = (JQLAST)_t;
			astFactory.addASTChild(currentAST, tmp16_AST);
			match(_t,DOUBLE);
			_t = _t.getNextSibling();
			primitiveType_AST = (JQLAST)currentAST.root;
			break;
		}
		default:
		{
			throw new NoViableAltException(_t);
		}
		}
		returnAST = primitiveType_AST;
		_retTree = _t;
	}
	
	
	public static final String[] _tokenNames = {
		"<0>",
		"EOF",
		"<2>",
		"NULL_TREE_LOOKAHEAD",
		"\"import\"",
		"\"this\"",
		"\"ascending\"",
		"\"descending\"",
		"\"distinct\"",
		"\"boolean\"",
		"\"byte\"",
		"\"char\"",
		"\"short\"",
		"\"int\"",
		"\"float\"",
		"\"long\"",
		"\"double\"",
		"\"null\"",
		"\"true\"",
		"\"false\"",
		"\"avg\"",
		"\"max\"",
		"\"min\"",
		"\"sum\"",
		"\"count\"",
		"LPAREN",
		"RPAREN",
		"COMMA",
		"EQUAL",
		"LNOT",
		"BNOT",
		"NOT_EQUAL",
		"DIV",
		"PLUS",
		"MINUS",
		"STAR",
		"MOD",
		"GE",
		"GT",
		"LE",
		"LT",
		"BXOR",
		"BOR",
		"OR",
		"BAND",
		"AND",
		"SEMI",
		"WS",
		"NEWLINE",
		"CHAR_LITERAL",
		"STRING_LITERAL",
		"ESC",
		"HEX_DIGIT",
		"INT_LITERAL",
		"EXPONENT",
		"FLOATINGPOINT_SUFFIX",
		"an identifier",
		"UNICODE_ESCAPE",
		"QUERY",
		"CLASS_DEF",
		"IMPORT_DEF",
		"PARAMETER_DEF",
		"VARIABLE_DEF",
		"ORDERING_DEF",
		"FILTER_DEF",
		"ARG_LIST",
		"UNARY_MINUS",
		"UNARY_PLUS",
		"TYPECAST",
		"OBJECT_EQUAL",
		"OBJECT_NOT_EQUAL",
		"COLLECTION_EQUAL",
		"COLLECTION_NOT_EQUAL",
		"CONCAT",
		"FIELD_ACCESS",
		"STATIC_FIELD_ACCESS",
		"CONTAINS",
		"NOT_CONTAINS",
		"NAVIGATION",
		"STARTS_WITH",
		"ENDS_WITH",
		"IS_EMPTY",
		"VARIABLE",
		"PARAMETER",
		"TYPENAME",
		"VALUE",
		"RESULT_DEF",
		"LIKE",
		"SUBSTRING",
		"INDEXOF",
		"LENGTH",
		"ABS",
		"SQRT",
		"NOT_IN",
		"DOT",
		"LONG_LITERAL",
		"FLOAT_LITERAL",
		"DOUBLE_LITERAL"
	};
	
	private static final long[] mk_tokenSet_0() {
		long[] data = { 82824011629592608L, 16106127388L, 0L, 0L};
		return data;
	}
	public static final BitSet _tokenSet_0 = new BitSet(mk_tokenSet_0());
	}
	
