// $ANTLR 2.7.7 (20060906): "Optimizer.g" -> "Optimizer.java"$

    package com.sun.jdo.spi.persistence.support.sqlstore.query.jqlc;
    
    import java.util.*;
    
    import java.math.BigDecimal;
    import java.math.BigInteger;

    import com.sun.jdo.api.persistence.support.JDOFatalUserException;
    import org.glassfish.persistence.common.I18NHelper;
    import com.sun.jdo.spi.persistence.support.sqlstore.query.util.type.TypeTable;
    import com.sun.jdo.spi.persistence.support.sqlstore.query.util.type.Type;
    import com.sun.jdo.spi.persistence.support.sqlstore.query.util.type.ClassType;
    import com.sun.jdo.spi.persistence.support.sqlstore.query.util.type.FieldInfo;
    import com.sun.jdo.spi.persistence.support.sqlstore.query.util.type.NumericType;
    import com.sun.jdo.spi.persistence.support.sqlstore.query.util.type.NumericWrapperClassType;
    import com.sun.jdo.spi.persistence.support.sqlstore.query.util.type.NumberType;
    import com.sun.jdo.spi.persistence.support.sqlstore.query.util.type.StringType;

import antlr.TreeParser;
import antlr.Token;
import antlr.collections.AST;
import antlr.RecognitionException;
import antlr.ANTLRException;
import antlr.NoViableAltException;
import antlr.MismatchedTokenException;
import antlr.SemanticException;
import antlr.collections.impl.BitSet;
import antlr.ASTPair;
import antlr.collections.impl.ASTArray;


/**
 * This class defines the optimizer pass of the JQL compiler.
 * It takes the typed AST as produced by the smenatic analysis and
 * converts it into a simpler but equivalent typed AST.
 * 
 * @author  Michael Bouschen
 * @version 0.1
 */
public class Optimizer extends antlr.TreeParser       implements OptimizerTokenTypes
 {

    /**
     * I18N support
     */
    protected final static ResourceBundle messages = 
        I18NHelper.loadBundle(Optimizer.class);
    
    /**
     * type table 
     */
    protected TypeTable typetab;
    
    /**
     * query parameter table
     */
    protected ParameterTable paramtab;
    
    /**
     *
     */
    protected ErrorMsg errorMsg;

    /**
     *
     */
    public void init(TypeTable typetab, ParameterTable paramtab, 
                     ErrorMsg errorMsg)
    {
        this.typetab = typetab;
        this.paramtab = paramtab;
        this.errorMsg = errorMsg;
    }

    /**
     *
     */
    public void reportError(RecognitionException ex) {
        errorMsg.fatal("Optimizer error", ex); //NOI18N
    }

    /**
     *
     */
    public void reportError(String s) {
        errorMsg.fatal("Optimizer error: " + s); //NOI18N
    }

    /**
     * Converts the string argument into a single char.
     */
    protected static char parseChar(String text)
    {
        char first = text.charAt(0);
        if (first == '\\')
        {
            //found escape => check the next char
            char second = text.charAt(1);
            switch (second)
            {
            case 'n': return '\n';
            case 'r': return '\r';
            case 't': return '\t';
            case 'b': return '\b';
            case 'f': return '\f';
            case 'u': 
                // unicode spec
                return (char)Integer.parseInt(text.substring(2, text.length()), 16);
            case '0':            
            case '1':
            case '2':
            case '3':            
            case '4':
            case '5':            
            case '6':
            case '7': 
                // octal spec 
                return (char)Integer.parseInt(text.substring(1, text.length()), 8);
            default : return second;
            }
        }
        return first;
    }
    

    /**
     * Check an AND operation (BAND, AND) for constant operands 
     * that could be optimized.
     * @param op the AND operator
     * @param left the left operand
     * @param right the right operand
     * @return optimized JQLAST 
     */
    protected JQLAST checkAnd(JQLAST op, JQLAST left, JQLAST right)
    {
        JQLAST ast = op;

        if (isBooleanValueAST(left))
        {
            ast = handleValueAndExpr(op, left.getValue(), right);
        }
        else if (isBooleanValueAST(right))
        {
            ast = handleValueAndExpr(op, right.getValue(), left);
        }
        return ast;
    }

    /**
     * Check an OR operation (BOR, OR) for constant operands 
     * that could be optimized.
     * @param op the OR operator
     * @param left the left operand
     * @param right the right operand
     * @return optimized JQLAST 
     */
    protected JQLAST checkOr(JQLAST op, JQLAST left, JQLAST right)
    {
        JQLAST ast = op;

        if (isBooleanValueAST(left))
        {
            ast = handleValueOrExpr(op, left.getValue(), right);
        }
        else if (isBooleanValueAST(right))
        {
            ast = handleValueOrExpr(op, right.getValue(), left);
        }
        return ast;
    }

    /**
     * Check a equality operation (EQUAL, NOT_EQUAL) for constant operands
     * that could be optimized.
     * @param op the equality operator
     * @param left the left operand
     * @param right the right operand
     * @param negate true for not equal operation, false otherwise
     * @return optimized JQLAST 
     */
    protected JQLAST checkEqualityOp(JQLAST op, JQLAST left, JQLAST right, 
                                     boolean negate)
    {
        JQLAST ast = op;

        // case <VALUE> <op> <VALUE> 
        if ((left.getType() == VALUE) && (right.getType() == VALUE))
        {
            ast = handleValueEqValue(op, left, right, negate);
        }
        // case <boolean VALUE> <op> <expr>
        else if (isBooleanValueAST(left))
        {
            ast = handleBooleanValueEqExpr(op, left.getValue(), right, negate);
        }
        // case <expr> <op> <boolean VALUE>
        else if (isBooleanValueAST(right))
        {
            ast = handleBooleanValueEqExpr(op, right.getValue(), left, negate);
        }
        return ast;
    }

    /**
     * Check a object equality operation (OBJECT_EQUAL, OBJECT_NOT_EQUAL) 
     * for constant operands that could be optimized.
     * @param op the object equality operator
     * @param left the left operand
     * @param right the right operand
     * @param negate true for not equal operation, false otherwise
     * @return optimized JQLAST 
     */
    protected JQLAST checkObjectEqualityOp(JQLAST op, JQLAST left, JQLAST right,
                                           boolean negate)
    {
        JQLAST ast = op;

        if ((left.getType() == VALUE) && (right.getType() == VALUE))
        {
            ast = handleValueEqValue(op, left, right, negate);
        }
        return ast;
    }

    /**
     * Check a collection equality operation (COLLECTION_EQUAL, 
     * COLLECTION_NOT_EQUAL) for constant operands that could be optimized.
     * @param op the collection equality operator
     * @param left the left operand
     * @param right the right operand
     * @param negate true for not equal operation, false otherwise
     * @return optimized JQLAST 
     */
    protected JQLAST checkCollectionEqualityOp(JQLAST op, JQLAST left, 
                                               JQLAST right, boolean negate)
    {
        JQLAST ast = op;
        boolean isLeftConstant = (left.getType() == VALUE);
        boolean isRightConstant = (right.getType() == VALUE);
        
        if (isLeftConstant && isRightConstant)
        {
            ast = handleValueEqValue(op, left, right, negate);
        }
        else if ((isLeftConstant && (left.getValue() == null) && isNonConstantCollection(right)) || 
                 (isRightConstant && (right.getValue() == null) && isNonConstantCollection(left))) 
        {
            // This optimization is datastore dependend. 
            // In TP we know a collection returned by the datastore is never null.
            // null == <collection field> -> false
            // <collection field> == null -> false
            // null != <collection field> -> true
            // <collection field> != null -> true
            ast.setType(VALUE);
            ast.setValue(new Boolean(negate));
            ast.setFirstChild(null);
        }

        return ast;
    }

    /**
     * Check a logical not operation (LNOT) for a constant operand 
     * that could be optimized.
     * @param op the logical not operator
     * @param arg the operand
     * @return optimized JQLAST 
     */
    protected JQLAST checkLogicalNotOp(JQLAST op, JQLAST arg)
    {
        JQLAST ast = op;

        if (arg.getType() == VALUE)
        {
            // !value may be calculated at compile time.
            Object valueObj = arg.getValue();
            boolean value = (valueObj instanceof Boolean) ? 
                            ((Boolean)valueObj).booleanValue() : false;
            arg.setType(VALUE);
            arg.setValue(new Boolean(!value));
            arg.setNextSibling(null);
            ast = arg;
        }
        else
        {
            ast = deMorgan(arg);
        }
        return ast;
    }

    /**
     * Check a binary plus operation (PLUS) for constant operands
     * that could be optimized.
     * @param op the plus operator
     * @param left the left operand
     * @param right the right operand
     * @return optimized JQLAST 
     */
    protected JQLAST checkBinaryPlusOp(JQLAST op, JQLAST left, JQLAST right)
    {
        JQLAST ast = op;

        if ((left.getType() == VALUE) && (right.getType() == VALUE))
        {
            Object leftValue = left.getValue();
            Object rightValue = right.getValue();
            Object value = null;
            if (leftValue == null)
                value = rightValue;
            else if (rightValue == null)
                value = leftValue;
            else
            {
                Type type = op.getJQLType();
                
                if (type instanceof NumericWrapperClassType)
                    type = ((NumericWrapperClassType)type).getPrimitiveType();
                
                if (type.equals(typetab.intType))
                    value = new Integer(((Number)leftValue).intValue() + 
                        ((Number)rightValue).intValue());
                else if (type.equals(typetab.longType))
                    value = new Long(((Number)leftValue).longValue() + 
                        ((Number)rightValue).longValue());
                else if (type.equals(typetab.floatType))
                    value = new Float(((Number)leftValue).floatValue() + 
                        ((Number)rightValue).floatValue());
                else if (type.equals(typetab.doubleType))
                    value = new Double(((Number)leftValue).doubleValue() + 
                        ((Number)rightValue).doubleValue());
                else if (type.equals(typetab.bigDecimalType))
                    value = getBigDecimalValue(leftValue).add(
                       getBigDecimalValue(rightValue));
                else if (type.equals(typetab.bigIntegerType))
                    value = getBigIntegerValue(leftValue).add(
                        getBigIntegerValue(rightValue));
                else 
                    errorMsg.fatal(I18NHelper.getMessage(messages,
                        "jqlc.optimizer.checkbinaryplusop.invalidtype", //NOI18N
                        String.valueOf(type)));
            }
            ast.setType(VALUE);
            ast.setValue(value);
            ast.setFirstChild(null);
        }
        return ast;
    }
    
    /**
     * Check a string concatenation operation (CONCAT) for constant operands
     * that could be optimized.
     * @param op the concat operator
     * @param left the left operand
     * @param right the right operand
     * @return optimized JQLAST 
     */
    protected JQLAST checkConcatOp(JQLAST op, JQLAST left, JQLAST right)
    {
        JQLAST ast = op;

        if ((left.getType() == VALUE) && (right.getType() == VALUE))
        {
            Object leftValue = left.getValue();
            Object rightValue = right.getValue();
            Object value = null;
            if (leftValue == null)
                value = rightValue;
            else if (rightValue == null)
                value = leftValue;
            else 
                value = leftValue.toString() + rightValue.toString();
            ast.setType(VALUE);
            ast.setValue(value);
            ast.setFirstChild(null);
        }
        return ast;
    }

    /**
     * Check a binary minus operation (MINUS) for constant operands
     * that could be optimized.
     * @param op the minus operator
     * @param left the left operand
     * @param right the right operand
     * @return optimized JQLAST 
     */
    protected JQLAST checkBinaryMinusOp(JQLAST op, JQLAST left, JQLAST right)
    {
        JQLAST ast = op;

        if ((left.getType() == VALUE) && (right.getType() == VALUE))
        {
            Object leftValue = left.getValue();
            Object rightValue = right.getValue();
            Object value = null;
            if (rightValue == null)
                value = leftValue;
            else
            {
                if (leftValue == null)
                    leftValue = new Integer(0);
                
                Type type = op.getJQLType();
                
                if (type instanceof NumericWrapperClassType)
                    type = ((NumericWrapperClassType)type).getPrimitiveType();
                
                if (type.equals(typetab.intType))
                    value = new Integer(((Number)leftValue).intValue() -
                        ((Number)rightValue).intValue());
                else if (type.equals(typetab.longType))
                    value = new Long(((Number)leftValue).longValue() -
                        ((Number)rightValue).longValue());
                else if (type.equals(typetab.floatType))
                    value = new Float(((Number)leftValue).floatValue() - 
                        ((Number)rightValue).floatValue());
                else if (type.equals(typetab.doubleType))
                    value = new Double(((Number)leftValue).doubleValue() - 
                        ((Number)rightValue).doubleValue());
                else if (type.equals(typetab.bigDecimalType))
                    value = getBigDecimalValue(leftValue).subtract(
                       getBigDecimalValue(rightValue));
                else if (type.equals(typetab.bigIntegerType))
                    value = getBigIntegerValue(leftValue).subtract(
                        getBigIntegerValue(rightValue));
                else 
                    errorMsg.fatal(I18NHelper.getMessage(messages,
                        "jqlc.optimizer.checkbinaryminusop.invalidtype", //NOI18N
                        String.valueOf(type)));
            }
            ast.setType(VALUE);
            ast.setValue(value);
            ast.setFirstChild(null);
        }
        return ast;
    }
    
    /**
     * Check a binary multiplication operation (STAR) for constant operands
     * that could be optimized.
     * @param op the multiplication operator
     * @param left the left operand
     * @param right the right operand
     * @return optimized JQLAST 
     */
    protected JQLAST checkMultiplicationOp(JQLAST op, JQLAST left, JQLAST right)
    {
        JQLAST ast = op;

        if ((left.getType() == VALUE) && (right.getType() == VALUE))
        {
            Object leftValue = left.getValue();
            Object rightValue = right.getValue();
            Object value = null;
            if (leftValue == null)
                leftValue = new Integer(0);
            if (rightValue == null)
                rightValue = new Integer(0);
            Type type = op.getJQLType();
                
            if (type instanceof NumericWrapperClassType)
                type = ((NumericWrapperClassType)type).getPrimitiveType();
                
            if (type.equals(typetab.intType))
                value = new Integer(((Number)leftValue).intValue() *
                    ((Number)rightValue).intValue());
            else if (type.equals(typetab.longType))
                value = new Long(((Number)leftValue).longValue() *
                    ((Number)rightValue).longValue());
            else if (type.equals(typetab.floatType))
                value = new Float(((Number)leftValue).floatValue() * 
                    ((Number)rightValue).floatValue());
            else if (type.equals(typetab.doubleType))
                value = new Double(((Number)leftValue).doubleValue() * 
                    ((Number)rightValue).doubleValue());
            else if (type.equals(typetab.bigDecimalType))
                value = getBigDecimalValue(leftValue).multiply(
                    getBigDecimalValue(rightValue));
            else if (type.equals(typetab.bigIntegerType))
                value = getBigIntegerValue(leftValue).multiply(
                    getBigIntegerValue(rightValue));
            else 
                errorMsg.fatal(I18NHelper.getMessage(messages,
                    "jqlc.optimizer.checkmultiplicationop.invalidtype", //NOI18N
                    String.valueOf(type)));

            ast.setType(VALUE);
            ast.setValue(value);
            ast.setFirstChild(null);
        }
        return ast;
    }
    
    /**
     * Check a binary division operation (DIV) for constant operands
     * that could be optimized.
     * @param op the division operator
     * @param left the left operand
     * @param right the right operand
     * @return optimized JQLAST 
     */
    protected JQLAST checkDivisionOp(JQLAST op, JQLAST left, JQLAST right)
    {
        JQLAST ast = op;

        if ((left.getType() == VALUE) && (right.getType() == VALUE))
        {
            Object leftValue = left.getValue();
            Object rightValue = right.getValue();
            Object value = null;
            if (leftValue == null)
                leftValue = new Integer(0);
            if (rightValue == null)
                // division by zero!
                rightValue = new Integer(0);

            Type type = op.getJQLType();
            
            if (type instanceof NumericWrapperClassType)
                type = ((NumericWrapperClassType)type).getPrimitiveType();
                
            if (type.equals(typetab.intType))
                value = new Integer(((Number)leftValue).intValue() /
                    ((Number)rightValue).intValue());
            else if (type.equals(typetab.longType))
                value = new Long(((Number)leftValue).longValue() /
                    ((Number)rightValue).longValue());
            else if (type.equals(typetab.floatType))
                value = new Float(((Number)leftValue).floatValue() / 
                    ((Number)rightValue).floatValue());
            else if (type.equals(typetab.doubleType))
                value = new Double(((Number)leftValue).doubleValue() / 
                    ((Number)rightValue).doubleValue());
            else if (type.equals(typetab.bigDecimalType))
                value = getBigDecimalValue(leftValue).divide(
                   getBigDecimalValue(rightValue), BigDecimal.ROUND_HALF_EVEN);
            else if (type.equals(typetab.bigIntegerType))
                value = getBigIntegerValue(leftValue).divide(
                    getBigIntegerValue(rightValue));
            else 
                errorMsg.fatal(I18NHelper.getMessage(messages,
                    "jqlc.optimizer.checkdivisionop.invalidtype", //NOI18N
                    String.valueOf(type)));

            ast.setType(VALUE);
            ast.setValue(value);
            ast.setFirstChild(null);
        }
        return ast;
    }
    
    /**
     * Check a binary modular operation (MOD) for constant operands
     * that could be optimized.
     * @param op the mod operator
     * @param left the left operand
     * @param right the right operand
     * @return optimized JQLAST 
     */
    protected JQLAST checkModOp(JQLAST op, JQLAST left, JQLAST right)
    {
        JQLAST ast = op;

        if ((left.getType() == VALUE) && (right.getType() == VALUE))
        {
            Object leftValue = left.getValue();
            Object rightValue = right.getValue();
            Object value = null;
            if (leftValue == null)
                leftValue = new Integer(0);
            if (rightValue == null)
                // division by zero!
                rightValue = new Integer(0);

            Type type = op.getJQLType();
            
            if (type instanceof NumericWrapperClassType)
                type = ((NumericWrapperClassType)type).getPrimitiveType();
                
            if (type.equals(typetab.intType))
                value = new Integer(((Number)leftValue).intValue() %
                    ((Number)rightValue).intValue());
            else if (type.equals(typetab.longType))
                value = new Long(((Number)leftValue).longValue() %
                    ((Number)rightValue).longValue());
            else if (type.equals(typetab.floatType))
                value = new Float(((Number)leftValue).floatValue() % 
                    ((Number)rightValue).floatValue());
            else if (type.equals(typetab.doubleType))
                value = new Double(((Number)leftValue).doubleValue() % 
                    ((Number)rightValue).doubleValue());
            else if (type.equals(typetab.bigDecimalType))
            {
                BigDecimal leftBigDecimal = getBigDecimalValue(leftValue);
                BigDecimal rightBigDecimal = getBigDecimalValue(rightValue);
                //use ROUND_HALF_EVEN so that it is consistent with div
                BigDecimal quotient = leftBigDecimal.divide(rightBigDecimal,
                    0, BigDecimal.ROUND_HALF_EVEN);
                value = leftBigDecimal.subtract(
                    rightBigDecimal.multiply(quotient));
            }
            else if (type.equals(typetab.bigIntegerType))
                value = getBigIntegerValue(leftValue).remainder(
                    getBigIntegerValue(rightValue));
            else 
                errorMsg.fatal(I18NHelper.getMessage(messages,
                    "jqlc.optimizer.checkmodop.invalidtype", //NOI18N
                    String.valueOf(type)));

            ast.setType(VALUE);
            ast.setValue(value);
            ast.setFirstChild(null);
        }
        return ast;
    }

    /**
     * Check a unary minus operation (UNARY_MINUS) for a constant operand
     * that could be optimized.
     * @param op the unary minus operator
     * @param arg the operand
     * @return optimized JQLAST 
     */
    protected JQLAST checkUnaryMinusOp(JQLAST op, JQLAST arg)
    {
        JQLAST ast = op;
        
        if (arg.getType() == VALUE)
        {
            Object value = arg.getValue();
            Type type = op.getJQLType();
            Object negate = null;
            
            if (type instanceof NumberType)
                negate = ((NumberType)type).negate((Number)value);
            else 
                errorMsg.fatal(I18NHelper.getMessage(messages,
                    "jqlc.optimizer.checkunaryminusop.invalidtype", //NOI18N
                    String.valueOf(type)));
            
            ast.setType(VALUE);
            ast.setValue(negate);
            ast.setFirstChild(null);
        }
        return ast;
    }

    /**
     * Check a cast operation for a constant operand
     * that could be optimized.
     * @param op the cast operator
     * @param castType the cast type
     * @param expr the non constant operand
     * @return optimized JQLAST 
     */
    protected JQLAST checkCastOp(JQLAST op, JQLAST castType, JQLAST expr)
    {
        JQLAST ast = op;
        
        if (expr.getType() == VALUE)
        {
            Object value = expr.getValue();
            Type type = op.getJQLType();
            if (type instanceof NumericWrapperClassType)
                type = ((NumericWrapperClassType)type).getPrimitiveType();
                
            if (type.equals(typetab.intType))
                value = new Integer(((Number)value).intValue());
            else if (type.equals(typetab.longType))
                value = new Long(((Number)value).longValue());
            else if (type.equals(typetab.floatType))
                value = new Float(((Number)value).floatValue());
            else if (type.equals(typetab.doubleType))
                value = new Double(((Number)value).doubleValue());
            else if (type.equals(typetab.bigDecimalType))
                value = getBigDecimalValue(value);
            else if (type.equals(typetab.bigIntegerType))
                value = getBigIntegerValue(value);
            else if (type.equals(typetab.byteType))
                value = new Byte((byte)((Number)value).intValue());
            else if (type.equals(typetab.shortType))
                value = new Short((short)((Number)value).intValue());
            else if (type.equals(typetab.charType))
                value = new Character((char)((Number)value).intValue());
            
            // If non of the above type applies, leave the value as it is

            // convert the TYPECAST op into a VALUE
            ast.setType(VALUE);
            ast.setValue(value);
            ast.setFirstChild(null);
        }
        return ast;
    }

    /**
     * Converts the specified value into a BigDecimal value. 
     * @param value value to be converted
     * @return BigDecimal representation
     */
    protected BigDecimal getBigDecimalValue(Object value)
    {
        BigDecimal ret = null;
        if (value instanceof Number)
            ret = (BigDecimal)typetab.bigDecimalType.getValue((Number)value);
        else
            errorMsg.fatal(I18NHelper.getMessage(messages,
                "jqlc.optimizer.getbigdecimalvalue.notnumber", //NOI18N
                String.valueOf(value)));

        return ret;
    }

    /**
     * Converts the specified value into a BigInteger value. 
     * @param value value to be converted
     * @return BigInteger representation
     */
    protected BigInteger getBigIntegerValue(Object value)
    {
        BigInteger ret = null;

        if (value instanceof Number)
            ret = (BigInteger)typetab.bigIntegerType.getValue((Number)value);
        else
            errorMsg.fatal(I18NHelper.getMessage(messages,
                "jqlc.optimizer.getbigintegervalue.notnumber", //NOI18N
                String.valueOf(value)));

        return ret;
    }
    
    /**
     * This method is called in the case of an equality operation having two 
     * constant operands. It calculates the result of this constant operation 
     * and returns a JQLAST node representing a constant boolean value.
     * @param op the equality operator
     * @param left the left operand
     * @param right the right operand
     * @param negate true for not equal operation, false otherwise
     * @return optimized JQLAST 
     */
    protected JQLAST handleValueEqValue(JQLAST op, JQLAST left, JQLAST right, 
                                        boolean negate)
    {
        Object leftValue = left.getValue();
        Object rightValue = right.getValue();
        boolean value = false;
        
        if ((leftValue == null) && (rightValue == null))
        {
            // both values are null -> true
            value = true;
        }
        else if ((leftValue != null) && (rightValue != null))
        {
            // both values are not null -> use equals
            value = leftValue.equals(rightValue);
        }
        else
        {
            // one value is null, the other is not null -> false
            value = false;
        }
        if (negate) 
        {
            value = !value;
        }
        op.setType(VALUE);
        op.setValue(new Boolean(value));
        op.setFirstChild(null);
        return op;
    }

    /**
     * This method is called in the case of an equality operation having 
     * a boolean constant operand and a non constant operand. 
     * It returns the non constant operand either as it is or inverted, 
     * depending on the equality operation.
     * @param op the equality operator
     * @param value the contant boolean value
     * @param expr the non constant operand
     * @param negate true for not equal operation, false otherwise
     * @return optimized JQLAST 
     */
    private JQLAST handleBooleanValueEqExpr(JQLAST op, Object value, 
                                            JQLAST expr, boolean negate)
    {
        JQLAST ast;
        boolean skip = (value instanceof Boolean) ? 
                       ((Boolean)value).booleanValue() : false;
        if (negate) skip = !skip;

        if (skip)
        {
            // expr == true -> expr
            // expr != false -> expr
            ast = expr;
        }
        else 
        {
            // if expr is a equality op or a not op the invert operation may be "inlined":
            //   (expr1 == expr2) != true -> expr1 != expr2
            //   (expr1 != expr2) != true -> expr1 == expr2
            //   !expr != true -> expr
            //   !expr == false -> expr
            // Otherwise wrap the expr with a not op
            //   expr != true -> !expr
            //   expr == false -> !expr
            switch (expr.getType())
            {
            case EQUAL:
                expr.setType(NOT_EQUAL);
                expr.setText("!="); //NOI18N
                ast = expr;
                break;
            case NOT_EQUAL:
                expr.setType(EQUAL);
                expr.setText("=="); //NOI18N
                ast = expr;
                break;
            case LNOT:
                ast = (JQLAST)expr.getFirstChild();
                break;
            default:
                op.setType(LNOT);
                op.setText("!"); //NOI18N
                op.setFirstChild(expr);
                ast = op;
            }
            expr.setNextSibling(null);
        }
        return ast;
    }

    /**
     * This method is called in the case of an AND operation having at least 
     * one constant operand. If the constant operand evaluates to true it 
     * returns the other operand. If it evaluates to false it returns an AST
     * representing the constant boolean value false.
     * @param op the AND operator
     * @param left the left operand
     * @param right the right operand
     * @return optimized JQLAST 
     */
    private JQLAST handleValueAndExpr(JQLAST op, Object value, JQLAST expr)
    {
        JQLAST ast;

        if ((value instanceof Boolean) && ((Boolean)value).booleanValue())
        {
            // true AND expr -> expr
            // expr AND true -> expr
            expr.setNextSibling(null);
            ast = expr;
        }
        else
        {
            // false AND expr -> false
            // expr AND false -> false
            op.setType(VALUE);
            op.setText("false"); //NOI18N
            op.setValue(new Boolean(false));
            op.setFirstChild(null);
            ast = op;
        }
        return ast;
    }

    /**
     * This method is called in the case of an OR operation having at least 
     * one constant operand. If the constant operand evaluates to false it 
     * returns the other operand. If it evaluates to true it returns an AST
     * representing the constant boolean value true.
     * @param op the AND operator
     * @param left the left operand
     * @param right the right operand
     * @return optimized JQLAST 
     */
    private JQLAST handleValueOrExpr(JQLAST op, Object value, JQLAST expr)
    {
        JQLAST ast;

        if ((value instanceof Boolean) && ((Boolean)value).booleanValue())
        {
            // true OR expr -> true
            // expr OR true -> true
            op.setType(VALUE);
            op.setText("true"); //NOI18N
            op.setValue(new Boolean(true));
            op.setFirstChild(null);
            ast = op;
        }
        else
        {
            // false OR expr -> expr
            // expr OR false -> expr
            expr.setNextSibling(null);
            ast = expr;
        }
        return ast;
    }

    /**
     * Returns true if the specified AST represents a constant boolean value.
     */
    protected boolean isBooleanValueAST(JQLAST ast)
    {
        return (ast.getType() == VALUE) && 
               (typetab.booleanType.equals(ast.getJQLType()));
    }

    /**
     * Returns true if the specified AST represents a datastore value. 
     */
    protected boolean isNonConstantCollection(JQLAST ast)
    {
        switch (ast.getType())
        {
        case FIELD_ACCESS :
        case NAVIGATION :
            return true;
        case TYPECAST :
            JQLAST expr = (JQLAST)ast.getFirstChild().getNextSibling();
            return isNonConstantCollection(expr);
        default:
            return false;
        }
    }

    /**
     * Implements DeMorgans rule: 
     * <br>
     * NOT (a AND b) -> NOT a OR NOT b
     * <br>
     * NOT (a OR b) -> NOT a AND NOT b
     * <br>
     * NOT (NOT a) -> a
     * <br>
     * The method assumes that the tree passed as an argument does not include 
     * the initial NOT. Note, this method checks for contains clauses, because 
     * they require special treatement.
     */
    protected JQLAST deMorgan(JQLAST tree)
    {
        JQLAST result = null;
        JQLAST left = null;
        JQLAST right = null;
        switch (tree.getType()) 
        {
        case AND:
        case BAND:
            left = (JQLAST)tree.getFirstChild();
            right = (JQLAST)left.getNextSibling();
            String leftVar = getVariableFromContainsClause(left);
            String rightVar = getVariableFromContainsClause(right);
            if (leftVar != null)
            {
                // found AND ( CONTAINS, right ), so check right for special
                // variable treatement
                result = buildAST(tree, left, deMorgan(right, leftVar));
            }
            else if (rightVar != null)
            {
                // found AND ( left, CONTAINS, ), so check left for special
                // variable treatement
                result = buildAST(tree, right, deMorgan(left, rightVar));
            }
            else
            {
                invertNode(tree);
                result = buildAST(tree, deMorgan(left), deMorgan(right));
            }
            break;
        case OR:
        case BOR:
            left = (JQLAST)tree.getFirstChild();
            right = (JQLAST)left.getNextSibling();
            invertNode(tree);
            result = buildAST(tree, deMorgan(left), deMorgan(right));
            break;
        case LNOT:
            // This is !(!arg) => return arg
            result = (JQLAST)tree.getFirstChild();
            break;
        default:
            // wrap arg into not operator
            result = buildAST(new JQLAST(LNOT, "!", typetab.booleanType), tree);
            break;
        }       
        return result;
    }

    /**
     * This overloaded deMorgan method implements special treatment of variable 
     * access expressions in the case of !contains. The method keeps an expression
     * accessing the specified variable as it is, but it inverts an expression NOT
     * accessing the variable following regular DeMorgan rules.
     */
    protected JQLAST deMorgan(JQLAST tree, String var)
    {
        JQLAST result = tree;
        switch (tree.getType()) 
        {
        case AND:
        case BAND:
        case OR:
        case BOR:
            JQLAST left = (JQLAST)tree.getFirstChild();
            JQLAST right = (JQLAST)left.getNextSibling();
            if (!includesVariableAccess(left, var) ||
                !includesVariableAccess(right, var))
            {
                invertNode(tree);
            }
            result = buildAST(tree, deMorgan(left, var), deMorgan(right, var));
            break;
        default:
            if (!includesVariableAccess(tree, var))
            {
                result = deMorgan(tree);
            }
            break;
        }
        return result;
    }

    /** 
     * Checks the specified tree being a CONATAINS clause. If yes it returns 
     * the variable used in the contains clause. Otherwise it returns null.
     */
    protected String getVariableFromContainsClause(JQLAST tree)
    {
        switch (tree.getType())
        {
        case CONTAINS:
        case NOT_CONTAINS:
            return tree.getFirstChild().getNextSibling().getText();
        default:
            return null;
        }
    }
    
    /**
     * Checks whether the specified tree accesses the variable with the 
     * specified name. Accessing means either this node or of of the subnodes 
     * has the type VARIABLE. 
     * NOTE, the method is intended to be used in the ! contains case only!
     * If it find a variable access node of the form
     * <br>
     * #(VARIABLE collection)
     * it maps it to
     * #(VARIABLE #(NOT_IN (collection))
     * <br>
     * This incdicates a variable belonging to a !contains clause.
     */
    protected boolean includesVariableAccess(AST tree, String var)
    {
        if ((tree == null) || (var == null))
            return false;

        boolean found = false;
        JQLAST child = (JQLAST)tree.getFirstChild();
        if ((tree.getType() == VARIABLE) && (tree.getText().equals(var)) && 
            (child != null))
        {
            found = true;
            if (child.getType() != NOT_IN)
            {
                tree.setFirstChild(buildAST(
                    new JQLAST(NOT_IN, "notIn", typetab.booleanType), child));
            }
        }
        for (AST node = tree.getFirstChild(); node != null; node = node.getNextSibling())
        {
            if (includesVariableAccess(node, var))
                found = true;
        }
        return found;
    }

    /** 
     * Inverts the specified node: AND -> OR, == -> !=, etc.
     */
    protected void invertNode(JQLAST node)
    {
        switch(node.getType())
        {
        case AND:
            node.setType(OR);
            node.setText("||");
            break;
        case BAND:
            node.setType(BOR);
            node.setText("|");
            break;
        case OR:
            node.setType(AND);
            node.setText("&&");
            break;
        case BOR:
            node.setType(BAND);
            node.setText("&");
            break;
        case EQUAL:
            node.setType(NOT_EQUAL);
            node.setText("!=");
            break;
        case NOT_EQUAL:
            node.setType(EQUAL);
            node.setText("==");
            break;
        case LT:
            node.setType(GE);
            node.setText(">=");
            break;
        case LE:
            node.setType(GT);
            node.setText(">");
            break;
        case GT:
            node.setType(LE);
            node.setText("<=");
            break;
        case GE:
            node.setType(LT);
            node.setText("<");
            break;
        }
    }

    /** Builds a binary tree. */
    protected JQLAST buildAST(JQLAST root, JQLAST left, JQLAST right)
    {
        root.setFirstChild(left);
        left.setNextSibling(right);
        right.setNextSibling(null);
        return root;
    }

    /** */
    protected JQLAST buildAST(JQLAST root, JQLAST arg)
    {
        root.setFirstChild(arg);
        arg.setNextSibling(null);
        return root;
    }
public Optimizer() {
	tokenNames = _tokenNames;
}

	public final void query(AST _t) throws RecognitionException {
		
		JQLAST query_AST_in = (_t == ASTNULL) ? null : (JQLAST)_t;
		returnAST = null;
		ASTPair currentAST = new ASTPair();
		JQLAST query_AST = null;
		JQLAST q = null;
		JQLAST q_AST = null;
		
		AST __t2 = _t;
		q = _t==ASTNULL ? null :(JQLAST)_t;
		JQLAST q_AST_in = null;
		q_AST = (JQLAST)astFactory.create(q);
		astFactory.addASTChild(currentAST, q_AST);
		ASTPair __currentAST2 = currentAST.copy();
		currentAST.root = currentAST.child;
		currentAST.child = null;
		match(_t,QUERY);
		_t = _t.getFirstChild();
		candidateClass(_t);
		_t = _retTree;
		astFactory.addASTChild(currentAST, returnAST);
		parameters(_t);
		_t = _retTree;
		astFactory.addASTChild(currentAST, returnAST);
		variables(_t);
		_t = _retTree;
		astFactory.addASTChild(currentAST, returnAST);
		ordering(_t);
		_t = _retTree;
		astFactory.addASTChild(currentAST, returnAST);
		result(_t);
		_t = _retTree;
		astFactory.addASTChild(currentAST, returnAST);
		filter(_t);
		_t = _retTree;
		astFactory.addASTChild(currentAST, returnAST);
		currentAST = __currentAST2;
		_t = __t2;
		_t = _t.getNextSibling();
		query_AST = (JQLAST)currentAST.root;
		returnAST = query_AST;
		_retTree = _t;
	}
	
	public final void candidateClass(AST _t) throws RecognitionException {
		
		JQLAST candidateClass_AST_in = (_t == ASTNULL) ? null : (JQLAST)_t;
		returnAST = null;
		ASTPair currentAST = new ASTPair();
		JQLAST candidateClass_AST = null;
		
		errorMsg.setContext("setCandidates"); //NOI18N
		
		
		JQLAST tmp1_AST = null;
		JQLAST tmp1_AST_in = null;
		tmp1_AST = (JQLAST)astFactory.create((JQLAST)_t);
		tmp1_AST_in = (JQLAST)_t;
		astFactory.addASTChild(currentAST, tmp1_AST);
		match(_t,CLASS_DEF);
		_t = _t.getNextSibling();
		candidateClass_AST = (JQLAST)currentAST.root;
		returnAST = candidateClass_AST;
		_retTree = _t;
	}
	
	public final void parameters(AST _t) throws RecognitionException {
		
		JQLAST parameters_AST_in = (_t == ASTNULL) ? null : (JQLAST)_t;
		returnAST = null;
		ASTPair currentAST = new ASTPair();
		JQLAST parameters_AST = null;
		
		errorMsg.setContext("declareParameters"); //NOI18N
		
		
		{
		_loop6:
		do {
			if (_t==null) _t=ASTNULL;
			if ((_t.getType()==PARAMETER_DEF)) {
				declareParameter(_t);
				_t = _retTree;
				astFactory.addASTChild(currentAST, returnAST);
			}
			else {
				break _loop6;
			}
			
		} while (true);
		}
		parameters_AST = (JQLAST)currentAST.root;
		returnAST = parameters_AST;
		_retTree = _t;
	}
	
	public final void variables(AST _t) throws RecognitionException {
		
		JQLAST variables_AST_in = (_t == ASTNULL) ? null : (JQLAST)_t;
		returnAST = null;
		ASTPair currentAST = new ASTPair();
		JQLAST variables_AST = null;
		
		errorMsg.setContext("declareVariables"); //NOI18N
		
		
		{
		_loop11:
		do {
			if (_t==null) _t=ASTNULL;
			if ((_t.getType()==VARIABLE_DEF)) {
				declareVariable(_t);
				_t = _retTree;
				astFactory.addASTChild(currentAST, returnAST);
			}
			else {
				break _loop11;
			}
			
		} while (true);
		}
		variables_AST = (JQLAST)currentAST.root;
		returnAST = variables_AST;
		_retTree = _t;
	}
	
	public final void ordering(AST _t) throws RecognitionException {
		
		JQLAST ordering_AST_in = (_t == ASTNULL) ? null : (JQLAST)_t;
		returnAST = null;
		ASTPair currentAST = new ASTPair();
		JQLAST ordering_AST = null;
		
		errorMsg.setContext("setOrdering"); //NOI18N
		
		
		{
		_loop16:
		do {
			if (_t==null) _t=ASTNULL;
			if ((_t.getType()==ORDERING_DEF)) {
				orderSpec(_t);
				_t = _retTree;
				astFactory.addASTChild(currentAST, returnAST);
			}
			else {
				break _loop16;
			}
			
		} while (true);
		}
		ordering_AST = (JQLAST)currentAST.root;
		returnAST = ordering_AST;
		_retTree = _t;
	}
	
	public final void result(AST _t) throws RecognitionException {
		
		JQLAST result_AST_in = (_t == ASTNULL) ? null : (JQLAST)_t;
		returnAST = null;
		ASTPair currentAST = new ASTPair();
		JQLAST result_AST = null;
		JQLAST r = null;
		JQLAST r_AST = null;
		
		errorMsg.setContext("setResult"); //NOI18N
		
		
		if (_t==null) _t=ASTNULL;
		switch ( _t.getType()) {
		case RESULT_DEF:
		{
			AST __t21 = _t;
			r = _t==ASTNULL ? null :(JQLAST)_t;
			JQLAST r_AST_in = null;
			r_AST = (JQLAST)astFactory.create(r);
			astFactory.addASTChild(currentAST, r_AST);
			ASTPair __currentAST21 = currentAST.copy();
			currentAST.root = currentAST.child;
			currentAST.child = null;
			match(_t,RESULT_DEF);
			_t = _t.getFirstChild();
			resultExpr(_t);
			_t = _retTree;
			astFactory.addASTChild(currentAST, returnAST);
			currentAST = __currentAST21;
			_t = __t21;
			_t = _t.getNextSibling();
			result_AST = (JQLAST)currentAST.root;
			break;
		}
		case FILTER_DEF:
		{
			result_AST = (JQLAST)currentAST.root;
			break;
		}
		default:
		{
			throw new NoViableAltException(_t);
		}
		}
		returnAST = result_AST;
		_retTree = _t;
	}
	
	public final void filter(AST _t) throws RecognitionException {
		
		JQLAST filter_AST_in = (_t == ASTNULL) ? null : (JQLAST)_t;
		returnAST = null;
		ASTPair currentAST = new ASTPair();
		JQLAST filter_AST = null;
		
		errorMsg.setContext("setFilter"); //NOI18N
		
		
		AST __t30 = _t;
		JQLAST tmp2_AST = null;
		JQLAST tmp2_AST_in = null;
		tmp2_AST = (JQLAST)astFactory.create((JQLAST)_t);
		tmp2_AST_in = (JQLAST)_t;
		astFactory.addASTChild(currentAST, tmp2_AST);
		ASTPair __currentAST30 = currentAST.copy();
		currentAST.root = currentAST.child;
		currentAST.child = null;
		match(_t,FILTER_DEF);
		_t = _t.getFirstChild();
		expression(_t);
		_t = _retTree;
		astFactory.addASTChild(currentAST, returnAST);
		currentAST = __currentAST30;
		_t = __t30;
		_t = _t.getNextSibling();
		filter_AST = (JQLAST)currentAST.root;
		returnAST = filter_AST;
		_retTree = _t;
	}
	
	public final void declareParameter(AST _t) throws RecognitionException {
		
		JQLAST declareParameter_AST_in = (_t == ASTNULL) ? null : (JQLAST)_t;
		returnAST = null;
		ASTPair currentAST = new ASTPair();
		JQLAST declareParameter_AST = null;
		
		AST __t8 = _t;
		JQLAST tmp3_AST = null;
		JQLAST tmp3_AST_in = null;
		tmp3_AST = (JQLAST)astFactory.create((JQLAST)_t);
		tmp3_AST_in = (JQLAST)_t;
		astFactory.addASTChild(currentAST, tmp3_AST);
		ASTPair __currentAST8 = currentAST.copy();
		currentAST.root = currentAST.child;
		currentAST.child = null;
		match(_t,PARAMETER_DEF);
		_t = _t.getFirstChild();
		type(_t);
		_t = _retTree;
		astFactory.addASTChild(currentAST, returnAST);
		JQLAST tmp4_AST = null;
		JQLAST tmp4_AST_in = null;
		tmp4_AST = (JQLAST)astFactory.create((JQLAST)_t);
		tmp4_AST_in = (JQLAST)_t;
		astFactory.addASTChild(currentAST, tmp4_AST);
		match(_t,IDENT);
		_t = _t.getNextSibling();
		currentAST = __currentAST8;
		_t = __t8;
		_t = _t.getNextSibling();
		declareParameter_AST = (JQLAST)currentAST.root;
		returnAST = declareParameter_AST;
		_retTree = _t;
	}
	
	public final void type(AST _t) throws RecognitionException {
		
		JQLAST type_AST_in = (_t == ASTNULL) ? null : (JQLAST)_t;
		returnAST = null;
		ASTPair currentAST = new ASTPair();
		JQLAST type_AST = null;
		
		if (_t==null) _t=ASTNULL;
		switch ( _t.getType()) {
		case TYPENAME:
		{
			JQLAST tmp5_AST = null;
			JQLAST tmp5_AST_in = null;
			tmp5_AST = (JQLAST)astFactory.create((JQLAST)_t);
			tmp5_AST_in = (JQLAST)_t;
			astFactory.addASTChild(currentAST, tmp5_AST);
			match(_t,TYPENAME);
			_t = _t.getNextSibling();
			type_AST = (JQLAST)currentAST.root;
			break;
		}
		case BOOLEAN:
		case BYTE:
		case CHAR:
		case SHORT:
		case INT:
		case FLOAT:
		case LONG:
		case DOUBLE:
		{
			primitiveType(_t);
			_t = _retTree;
			astFactory.addASTChild(currentAST, returnAST);
			type_AST = (JQLAST)currentAST.root;
			break;
		}
		default:
		{
			throw new NoViableAltException(_t);
		}
		}
		returnAST = type_AST;
		_retTree = _t;
	}
	
	public final void declareVariable(AST _t) throws RecognitionException {
		
		JQLAST declareVariable_AST_in = (_t == ASTNULL) ? null : (JQLAST)_t;
		returnAST = null;
		ASTPair currentAST = new ASTPair();
		JQLAST declareVariable_AST = null;
		
		AST __t13 = _t;
		JQLAST tmp6_AST = null;
		JQLAST tmp6_AST_in = null;
		tmp6_AST = (JQLAST)astFactory.create((JQLAST)_t);
		tmp6_AST_in = (JQLAST)_t;
		astFactory.addASTChild(currentAST, tmp6_AST);
		ASTPair __currentAST13 = currentAST.copy();
		currentAST.root = currentAST.child;
		currentAST.child = null;
		match(_t,VARIABLE_DEF);
		_t = _t.getFirstChild();
		type(_t);
		_t = _retTree;
		astFactory.addASTChild(currentAST, returnAST);
		JQLAST tmp7_AST = null;
		JQLAST tmp7_AST_in = null;
		tmp7_AST = (JQLAST)astFactory.create((JQLAST)_t);
		tmp7_AST_in = (JQLAST)_t;
		astFactory.addASTChild(currentAST, tmp7_AST);
		match(_t,IDENT);
		_t = _t.getNextSibling();
		currentAST = __currentAST13;
		_t = __t13;
		_t = _t.getNextSibling();
		declareVariable_AST = (JQLAST)currentAST.root;
		returnAST = declareVariable_AST;
		_retTree = _t;
	}
	
	public final void orderSpec(AST _t) throws RecognitionException {
		
		JQLAST orderSpec_AST_in = (_t == ASTNULL) ? null : (JQLAST)_t;
		returnAST = null;
		ASTPair currentAST = new ASTPair();
		JQLAST orderSpec_AST = null;
		
		AST __t18 = _t;
		JQLAST tmp8_AST = null;
		JQLAST tmp8_AST_in = null;
		tmp8_AST = (JQLAST)astFactory.create((JQLAST)_t);
		tmp8_AST_in = (JQLAST)_t;
		astFactory.addASTChild(currentAST, tmp8_AST);
		ASTPair __currentAST18 = currentAST.copy();
		currentAST.root = currentAST.child;
		currentAST.child = null;
		match(_t,ORDERING_DEF);
		_t = _t.getFirstChild();
		{
		if (_t==null) _t=ASTNULL;
		switch ( _t.getType()) {
		case ASCENDING:
		{
			JQLAST tmp9_AST = null;
			JQLAST tmp9_AST_in = null;
			tmp9_AST = (JQLAST)astFactory.create((JQLAST)_t);
			tmp9_AST_in = (JQLAST)_t;
			astFactory.addASTChild(currentAST, tmp9_AST);
			match(_t,ASCENDING);
			_t = _t.getNextSibling();
			break;
		}
		case DESCENDING:
		{
			JQLAST tmp10_AST = null;
			JQLAST tmp10_AST_in = null;
			tmp10_AST = (JQLAST)astFactory.create((JQLAST)_t);
			tmp10_AST_in = (JQLAST)_t;
			astFactory.addASTChild(currentAST, tmp10_AST);
			match(_t,DESCENDING);
			_t = _t.getNextSibling();
			break;
		}
		default:
		{
			throw new NoViableAltException(_t);
		}
		}
		}
		expression(_t);
		_t = _retTree;
		astFactory.addASTChild(currentAST, returnAST);
		currentAST = __currentAST18;
		_t = __t18;
		_t = _t.getNextSibling();
		orderSpec_AST = (JQLAST)currentAST.root;
		returnAST = orderSpec_AST;
		_retTree = _t;
	}
	
	public final void expression(AST _t) throws RecognitionException {
		
		JQLAST expression_AST_in = (_t == ASTNULL) ? null : (JQLAST)_t;
		returnAST = null;
		ASTPair currentAST = new ASTPair();
		JQLAST expression_AST = null;
		
		if (_t==null) _t=ASTNULL;
		switch ( _t.getType()) {
		case THIS:
		case NULL:
		case TRUE:
		case FALSE:
		case CHAR_LITERAL:
		case STRING_LITERAL:
		case INT_LITERAL:
		case TYPECAST:
		case FIELD_ACCESS:
		case STATIC_FIELD_ACCESS:
		case CONTAINS:
		case NOT_CONTAINS:
		case NAVIGATION:
		case STARTS_WITH:
		case ENDS_WITH:
		case IS_EMPTY:
		case VARIABLE:
		case PARAMETER:
		case VALUE:
		case LIKE:
		case SUBSTRING:
		case INDEXOF:
		case LENGTH:
		case ABS:
		case SQRT:
		case LONG_LITERAL:
		case FLOAT_LITERAL:
		case DOUBLE_LITERAL:
		{
			primary(_t);
			_t = _retTree;
			astFactory.addASTChild(currentAST, returnAST);
			expression_AST = (JQLAST)currentAST.root;
			break;
		}
		case BXOR:
		case BOR:
		case BAND:
		{
			bitwiseExpr(_t);
			_t = _retTree;
			astFactory.addASTChild(currentAST, returnAST);
			expression_AST = (JQLAST)currentAST.root;
			break;
		}
		case OR:
		case AND:
		{
			conditionalExpr(_t);
			_t = _retTree;
			astFactory.addASTChild(currentAST, returnAST);
			expression_AST = (JQLAST)currentAST.root;
			break;
		}
		case EQUAL:
		case NOT_EQUAL:
		case GE:
		case GT:
		case LE:
		case LT:
		case OBJECT_EQUAL:
		case OBJECT_NOT_EQUAL:
		case COLLECTION_EQUAL:
		case COLLECTION_NOT_EQUAL:
		{
			relationalExpr(_t);
			_t = _retTree;
			astFactory.addASTChild(currentAST, returnAST);
			expression_AST = (JQLAST)currentAST.root;
			break;
		}
		case DIV:
		case PLUS:
		case MINUS:
		case STAR:
		case MOD:
		case CONCAT:
		{
			binaryArithmeticExpr(_t);
			_t = _retTree;
			astFactory.addASTChild(currentAST, returnAST);
			expression_AST = (JQLAST)currentAST.root;
			break;
		}
		case UNARY_MINUS:
		case UNARY_PLUS:
		{
			unaryArithmeticExpr(_t);
			_t = _retTree;
			astFactory.addASTChild(currentAST, returnAST);
			expression_AST = (JQLAST)currentAST.root;
			break;
		}
		case LNOT:
		case BNOT:
		{
			complementExpr(_t);
			_t = _retTree;
			astFactory.addASTChild(currentAST, returnAST);
			expression_AST = (JQLAST)currentAST.root;
			break;
		}
		default:
		{
			throw new NoViableAltException(_t);
		}
		}
		returnAST = expression_AST;
		_retTree = _t;
	}
	
	public final void resultExpr(AST _t) throws RecognitionException {
		
		JQLAST resultExpr_AST_in = (_t == ASTNULL) ? null : (JQLAST)_t;
		returnAST = null;
		ASTPair currentAST = new ASTPair();
		JQLAST resultExpr_AST = null;
		
		if (_t==null) _t=ASTNULL;
		switch ( _t.getType()) {
		case DISTINCT:
		{
			AST __t23 = _t;
			JQLAST tmp11_AST = null;
			JQLAST tmp11_AST_in = null;
			tmp11_AST = (JQLAST)astFactory.create((JQLAST)_t);
			tmp11_AST_in = (JQLAST)_t;
			astFactory.addASTChild(currentAST, tmp11_AST);
			ASTPair __currentAST23 = currentAST.copy();
			currentAST.root = currentAST.child;
			currentAST.child = null;
			match(_t,DISTINCT);
			_t = _t.getFirstChild();
			resultExpr(_t);
			_t = _retTree;
			astFactory.addASTChild(currentAST, returnAST);
			currentAST = __currentAST23;
			_t = __t23;
			_t = _t.getNextSibling();
			resultExpr_AST = (JQLAST)currentAST.root;
			break;
		}
		case AVG:
		{
			AST __t24 = _t;
			JQLAST tmp12_AST = null;
			JQLAST tmp12_AST_in = null;
			tmp12_AST = (JQLAST)astFactory.create((JQLAST)_t);
			tmp12_AST_in = (JQLAST)_t;
			astFactory.addASTChild(currentAST, tmp12_AST);
			ASTPair __currentAST24 = currentAST.copy();
			currentAST.root = currentAST.child;
			currentAST.child = null;
			match(_t,AVG);
			_t = _t.getFirstChild();
			resultExpr(_t);
			_t = _retTree;
			astFactory.addASTChild(currentAST, returnAST);
			currentAST = __currentAST24;
			_t = __t24;
			_t = _t.getNextSibling();
			resultExpr_AST = (JQLAST)currentAST.root;
			break;
		}
		case MAX:
		{
			AST __t25 = _t;
			JQLAST tmp13_AST = null;
			JQLAST tmp13_AST_in = null;
			tmp13_AST = (JQLAST)astFactory.create((JQLAST)_t);
			tmp13_AST_in = (JQLAST)_t;
			astFactory.addASTChild(currentAST, tmp13_AST);
			ASTPair __currentAST25 = currentAST.copy();
			currentAST.root = currentAST.child;
			currentAST.child = null;
			match(_t,MAX);
			_t = _t.getFirstChild();
			resultExpr(_t);
			_t = _retTree;
			astFactory.addASTChild(currentAST, returnAST);
			currentAST = __currentAST25;
			_t = __t25;
			_t = _t.getNextSibling();
			resultExpr_AST = (JQLAST)currentAST.root;
			break;
		}
		case MIN:
		{
			AST __t26 = _t;
			JQLAST tmp14_AST = null;
			JQLAST tmp14_AST_in = null;
			tmp14_AST = (JQLAST)astFactory.create((JQLAST)_t);
			tmp14_AST_in = (JQLAST)_t;
			astFactory.addASTChild(currentAST, tmp14_AST);
			ASTPair __currentAST26 = currentAST.copy();
			currentAST.root = currentAST.child;
			currentAST.child = null;
			match(_t,MIN);
			_t = _t.getFirstChild();
			resultExpr(_t);
			_t = _retTree;
			astFactory.addASTChild(currentAST, returnAST);
			currentAST = __currentAST26;
			_t = __t26;
			_t = _t.getNextSibling();
			resultExpr_AST = (JQLAST)currentAST.root;
			break;
		}
		case SUM:
		{
			AST __t27 = _t;
			JQLAST tmp15_AST = null;
			JQLAST tmp15_AST_in = null;
			tmp15_AST = (JQLAST)astFactory.create((JQLAST)_t);
			tmp15_AST_in = (JQLAST)_t;
			astFactory.addASTChild(currentAST, tmp15_AST);
			ASTPair __currentAST27 = currentAST.copy();
			currentAST.root = currentAST.child;
			currentAST.child = null;
			match(_t,SUM);
			_t = _t.getFirstChild();
			resultExpr(_t);
			_t = _retTree;
			astFactory.addASTChild(currentAST, returnAST);
			currentAST = __currentAST27;
			_t = __t27;
			_t = _t.getNextSibling();
			resultExpr_AST = (JQLAST)currentAST.root;
			break;
		}
		case COUNT:
		{
			AST __t28 = _t;
			JQLAST tmp16_AST = null;
			JQLAST tmp16_AST_in = null;
			tmp16_AST = (JQLAST)astFactory.create((JQLAST)_t);
			tmp16_AST_in = (JQLAST)_t;
			astFactory.addASTChild(currentAST, tmp16_AST);
			ASTPair __currentAST28 = currentAST.copy();
			currentAST.root = currentAST.child;
			currentAST.child = null;
			match(_t,COUNT);
			_t = _t.getFirstChild();
			resultExpr(_t);
			_t = _retTree;
			astFactory.addASTChild(currentAST, returnAST);
			currentAST = __currentAST28;
			_t = __t28;
			_t = _t.getNextSibling();
			resultExpr_AST = (JQLAST)currentAST.root;
			break;
		}
		case THIS:
		case NULL:
		case TRUE:
		case FALSE:
		case EQUAL:
		case LNOT:
		case BNOT:
		case NOT_EQUAL:
		case DIV:
		case PLUS:
		case MINUS:
		case STAR:
		case MOD:
		case GE:
		case GT:
		case LE:
		case LT:
		case BXOR:
		case BOR:
		case OR:
		case BAND:
		case AND:
		case CHAR_LITERAL:
		case STRING_LITERAL:
		case INT_LITERAL:
		case UNARY_MINUS:
		case UNARY_PLUS:
		case TYPECAST:
		case OBJECT_EQUAL:
		case OBJECT_NOT_EQUAL:
		case COLLECTION_EQUAL:
		case COLLECTION_NOT_EQUAL:
		case CONCAT:
		case FIELD_ACCESS:
		case STATIC_FIELD_ACCESS:
		case CONTAINS:
		case NOT_CONTAINS:
		case NAVIGATION:
		case STARTS_WITH:
		case ENDS_WITH:
		case IS_EMPTY:
		case VARIABLE:
		case PARAMETER:
		case VALUE:
		case LIKE:
		case SUBSTRING:
		case INDEXOF:
		case LENGTH:
		case ABS:
		case SQRT:
		case LONG_LITERAL:
		case FLOAT_LITERAL:
		case DOUBLE_LITERAL:
		{
			expression(_t);
			_t = _retTree;
			astFactory.addASTChild(currentAST, returnAST);
			resultExpr_AST = (JQLAST)currentAST.root;
			break;
		}
		default:
		{
			throw new NoViableAltException(_t);
		}
		}
		returnAST = resultExpr_AST;
		_retTree = _t;
	}
	
	public final void primary(AST _t) throws RecognitionException {
		
		JQLAST primary_AST_in = (_t == ASTNULL) ? null : (JQLAST)_t;
		returnAST = null;
		ASTPair currentAST = new ASTPair();
		JQLAST primary_AST = null;
		
		if (_t==null) _t=ASTNULL;
		switch ( _t.getType()) {
		case TYPECAST:
		{
			castExpr(_t);
			_t = _retTree;
			astFactory.addASTChild(currentAST, returnAST);
			primary_AST = (JQLAST)currentAST.root;
			break;
		}
		case NULL:
		case TRUE:
		case FALSE:
		case CHAR_LITERAL:
		case STRING_LITERAL:
		case INT_LITERAL:
		case LONG_LITERAL:
		case FLOAT_LITERAL:
		case DOUBLE_LITERAL:
		{
			literal(_t);
			_t = _retTree;
			astFactory.addASTChild(currentAST, returnAST);
			primary_AST = (JQLAST)currentAST.root;
			break;
		}
		case VALUE:
		{
			JQLAST tmp17_AST = null;
			JQLAST tmp17_AST_in = null;
			tmp17_AST = (JQLAST)astFactory.create((JQLAST)_t);
			tmp17_AST_in = (JQLAST)_t;
			astFactory.addASTChild(currentAST, tmp17_AST);
			match(_t,VALUE);
			_t = _t.getNextSibling();
			primary_AST = (JQLAST)currentAST.root;
			break;
		}
		case THIS:
		{
			JQLAST tmp18_AST = null;
			JQLAST tmp18_AST_in = null;
			tmp18_AST = (JQLAST)astFactory.create((JQLAST)_t);
			tmp18_AST_in = (JQLAST)_t;
			astFactory.addASTChild(currentAST, tmp18_AST);
			match(_t,THIS);
			_t = _t.getNextSibling();
			primary_AST = (JQLAST)currentAST.root;
			break;
		}
		case PARAMETER:
		{
			parameter(_t);
			_t = _retTree;
			astFactory.addASTChild(currentAST, returnAST);
			primary_AST = (JQLAST)currentAST.root;
			break;
		}
		case STATIC_FIELD_ACCESS:
		{
			staticFieldAccess(_t);
			_t = _retTree;
			astFactory.addASTChild(currentAST, returnAST);
			primary_AST = (JQLAST)currentAST.root;
			break;
		}
		case FIELD_ACCESS:
		{
			fieldAccess(_t);
			_t = _retTree;
			astFactory.addASTChild(currentAST, returnAST);
			primary_AST = (JQLAST)currentAST.root;
			break;
		}
		case NAVIGATION:
		{
			navigation(_t);
			_t = _retTree;
			astFactory.addASTChild(currentAST, returnAST);
			primary_AST = (JQLAST)currentAST.root;
			break;
		}
		case VARIABLE:
		{
			variableAccess(_t);
			_t = _retTree;
			astFactory.addASTChild(currentAST, returnAST);
			primary_AST = (JQLAST)currentAST.root;
			break;
		}
		case CONTAINS:
		{
			AST __t69 = _t;
			JQLAST tmp19_AST = null;
			JQLAST tmp19_AST_in = null;
			tmp19_AST = (JQLAST)astFactory.create((JQLAST)_t);
			tmp19_AST_in = (JQLAST)_t;
			astFactory.addASTChild(currentAST, tmp19_AST);
			ASTPair __currentAST69 = currentAST.copy();
			currentAST.root = currentAST.child;
			currentAST.child = null;
			match(_t,CONTAINS);
			_t = _t.getFirstChild();
			expression(_t);
			_t = _retTree;
			astFactory.addASTChild(currentAST, returnAST);
			JQLAST tmp20_AST = null;
			JQLAST tmp20_AST_in = null;
			tmp20_AST = (JQLAST)astFactory.create((JQLAST)_t);
			tmp20_AST_in = (JQLAST)_t;
			astFactory.addASTChild(currentAST, tmp20_AST);
			match(_t,VARIABLE);
			_t = _t.getNextSibling();
			currentAST = __currentAST69;
			_t = __t69;
			_t = _t.getNextSibling();
			primary_AST = (JQLAST)currentAST.root;
			break;
		}
		case NOT_CONTAINS:
		{
			AST __t70 = _t;
			JQLAST tmp21_AST = null;
			JQLAST tmp21_AST_in = null;
			tmp21_AST = (JQLAST)astFactory.create((JQLAST)_t);
			tmp21_AST_in = (JQLAST)_t;
			astFactory.addASTChild(currentAST, tmp21_AST);
			ASTPair __currentAST70 = currentAST.copy();
			currentAST.root = currentAST.child;
			currentAST.child = null;
			match(_t,NOT_CONTAINS);
			_t = _t.getFirstChild();
			expression(_t);
			_t = _retTree;
			astFactory.addASTChild(currentAST, returnAST);
			JQLAST tmp22_AST = null;
			JQLAST tmp22_AST_in = null;
			tmp22_AST = (JQLAST)astFactory.create((JQLAST)_t);
			tmp22_AST_in = (JQLAST)_t;
			astFactory.addASTChild(currentAST, tmp22_AST);
			match(_t,VARIABLE);
			_t = _t.getNextSibling();
			currentAST = __currentAST70;
			_t = __t70;
			_t = _t.getNextSibling();
			primary_AST = (JQLAST)currentAST.root;
			break;
		}
		case STARTS_WITH:
		{
			startsWith(_t);
			_t = _retTree;
			astFactory.addASTChild(currentAST, returnAST);
			primary_AST = (JQLAST)currentAST.root;
			break;
		}
		case ENDS_WITH:
		{
			endsWith(_t);
			_t = _retTree;
			astFactory.addASTChild(currentAST, returnAST);
			primary_AST = (JQLAST)currentAST.root;
			break;
		}
		case IS_EMPTY:
		{
			isEmpty(_t);
			_t = _retTree;
			astFactory.addASTChild(currentAST, returnAST);
			primary_AST = (JQLAST)currentAST.root;
			break;
		}
		case LIKE:
		{
			like(_t);
			_t = _retTree;
			astFactory.addASTChild(currentAST, returnAST);
			primary_AST = (JQLAST)currentAST.root;
			break;
		}
		case SUBSTRING:
		{
			substring(_t);
			_t = _retTree;
			astFactory.addASTChild(currentAST, returnAST);
			primary_AST = (JQLAST)currentAST.root;
			break;
		}
		case INDEXOF:
		{
			indexOf(_t);
			_t = _retTree;
			astFactory.addASTChild(currentAST, returnAST);
			primary_AST = (JQLAST)currentAST.root;
			break;
		}
		case LENGTH:
		{
			length(_t);
			_t = _retTree;
			astFactory.addASTChild(currentAST, returnAST);
			primary_AST = (JQLAST)currentAST.root;
			break;
		}
		case ABS:
		{
			abs(_t);
			_t = _retTree;
			astFactory.addASTChild(currentAST, returnAST);
			primary_AST = (JQLAST)currentAST.root;
			break;
		}
		case SQRT:
		{
			sqrt(_t);
			_t = _retTree;
			astFactory.addASTChild(currentAST, returnAST);
			primary_AST = (JQLAST)currentAST.root;
			break;
		}
		default:
		{
			throw new NoViableAltException(_t);
		}
		}
		returnAST = primary_AST;
		_retTree = _t;
	}
	
	public final void bitwiseExpr(AST _t) throws RecognitionException {
		
		JQLAST bitwiseExpr_AST_in = (_t == ASTNULL) ? null : (JQLAST)_t;
		returnAST = null;
		ASTPair currentAST = new ASTPair();
		JQLAST bitwiseExpr_AST = null;
		JQLAST op1 = null;
		JQLAST op1_AST = null;
		JQLAST left1_AST = null;
		JQLAST left1 = null;
		JQLAST right1_AST = null;
		JQLAST right1 = null;
		JQLAST op2 = null;
		JQLAST op2_AST = null;
		JQLAST left2_AST = null;
		JQLAST left2 = null;
		JQLAST right2_AST = null;
		JQLAST right2 = null;
		JQLAST op3 = null;
		JQLAST op3_AST = null;
		JQLAST left3_AST = null;
		JQLAST left3 = null;
		JQLAST right3_AST = null;
		JQLAST right3 = null;
		
		if (_t==null) _t=ASTNULL;
		switch ( _t.getType()) {
		case BAND:
		{
			AST __t33 = _t;
			op1 = _t==ASTNULL ? null :(JQLAST)_t;
			JQLAST op1_AST_in = null;
			op1_AST = (JQLAST)astFactory.create(op1);
			astFactory.addASTChild(currentAST, op1_AST);
			ASTPair __currentAST33 = currentAST.copy();
			currentAST.root = currentAST.child;
			currentAST.child = null;
			match(_t,BAND);
			_t = _t.getFirstChild();
			left1 = _t==ASTNULL ? null : (JQLAST)_t;
			expression(_t);
			_t = _retTree;
			left1_AST = (JQLAST)returnAST;
			astFactory.addASTChild(currentAST, returnAST);
			right1 = _t==ASTNULL ? null : (JQLAST)_t;
			expression(_t);
			_t = _retTree;
			right1_AST = (JQLAST)returnAST;
			astFactory.addASTChild(currentAST, returnAST);
			currentAST = __currentAST33;
			_t = __t33;
			_t = _t.getNextSibling();
			if ( inputState.guessing==0 ) {
				bitwiseExpr_AST = (JQLAST)currentAST.root;
				
				bitwiseExpr_AST = checkAnd(op1_AST, left1_AST, right1_AST);
				
				currentAST.root = bitwiseExpr_AST;
				currentAST.child = bitwiseExpr_AST!=null &&bitwiseExpr_AST.getFirstChild()!=null ?
					bitwiseExpr_AST.getFirstChild() : bitwiseExpr_AST;
				currentAST.advanceChildToEnd();
			}
			bitwiseExpr_AST = (JQLAST)currentAST.root;
			break;
		}
		case BOR:
		{
			AST __t34 = _t;
			op2 = _t==ASTNULL ? null :(JQLAST)_t;
			JQLAST op2_AST_in = null;
			op2_AST = (JQLAST)astFactory.create(op2);
			astFactory.addASTChild(currentAST, op2_AST);
			ASTPair __currentAST34 = currentAST.copy();
			currentAST.root = currentAST.child;
			currentAST.child = null;
			match(_t,BOR);
			_t = _t.getFirstChild();
			left2 = _t==ASTNULL ? null : (JQLAST)_t;
			expression(_t);
			_t = _retTree;
			left2_AST = (JQLAST)returnAST;
			astFactory.addASTChild(currentAST, returnAST);
			right2 = _t==ASTNULL ? null : (JQLAST)_t;
			expression(_t);
			_t = _retTree;
			right2_AST = (JQLAST)returnAST;
			astFactory.addASTChild(currentAST, returnAST);
			currentAST = __currentAST34;
			_t = __t34;
			_t = _t.getNextSibling();
			if ( inputState.guessing==0 ) {
				bitwiseExpr_AST = (JQLAST)currentAST.root;
				
				bitwiseExpr_AST = checkOr(op2_AST, left2_AST, right2_AST);
				
				currentAST.root = bitwiseExpr_AST;
				currentAST.child = bitwiseExpr_AST!=null &&bitwiseExpr_AST.getFirstChild()!=null ?
					bitwiseExpr_AST.getFirstChild() : bitwiseExpr_AST;
				currentAST.advanceChildToEnd();
			}
			bitwiseExpr_AST = (JQLAST)currentAST.root;
			break;
		}
		case BXOR:
		{
			AST __t35 = _t;
			op3 = _t==ASTNULL ? null :(JQLAST)_t;
			JQLAST op3_AST_in = null;
			op3_AST = (JQLAST)astFactory.create(op3);
			astFactory.addASTChild(currentAST, op3_AST);
			ASTPair __currentAST35 = currentAST.copy();
			currentAST.root = currentAST.child;
			currentAST.child = null;
			match(_t,BXOR);
			_t = _t.getFirstChild();
			left3 = _t==ASTNULL ? null : (JQLAST)_t;
			expression(_t);
			_t = _retTree;
			left3_AST = (JQLAST)returnAST;
			astFactory.addASTChild(currentAST, returnAST);
			right3 = _t==ASTNULL ? null : (JQLAST)_t;
			expression(_t);
			_t = _retTree;
			right3_AST = (JQLAST)returnAST;
			astFactory.addASTChild(currentAST, returnAST);
			currentAST = __currentAST35;
			_t = __t35;
			_t = _t.getNextSibling();
			bitwiseExpr_AST = (JQLAST)currentAST.root;
			break;
		}
		default:
		{
			throw new NoViableAltException(_t);
		}
		}
		returnAST = bitwiseExpr_AST;
		_retTree = _t;
	}
	
	public final void conditionalExpr(AST _t) throws RecognitionException {
		
		JQLAST conditionalExpr_AST_in = (_t == ASTNULL) ? null : (JQLAST)_t;
		returnAST = null;
		ASTPair currentAST = new ASTPair();
		JQLAST conditionalExpr_AST = null;
		JQLAST op1 = null;
		JQLAST op1_AST = null;
		JQLAST left1_AST = null;
		JQLAST left1 = null;
		JQLAST right1_AST = null;
		JQLAST right1 = null;
		JQLAST op2 = null;
		JQLAST op2_AST = null;
		JQLAST left2_AST = null;
		JQLAST left2 = null;
		JQLAST right2_AST = null;
		JQLAST right2 = null;
		
		if (_t==null) _t=ASTNULL;
		switch ( _t.getType()) {
		case AND:
		{
			AST __t37 = _t;
			op1 = _t==ASTNULL ? null :(JQLAST)_t;
			JQLAST op1_AST_in = null;
			op1_AST = (JQLAST)astFactory.create(op1);
			astFactory.addASTChild(currentAST, op1_AST);
			ASTPair __currentAST37 = currentAST.copy();
			currentAST.root = currentAST.child;
			currentAST.child = null;
			match(_t,AND);
			_t = _t.getFirstChild();
			left1 = _t==ASTNULL ? null : (JQLAST)_t;
			expression(_t);
			_t = _retTree;
			left1_AST = (JQLAST)returnAST;
			astFactory.addASTChild(currentAST, returnAST);
			right1 = _t==ASTNULL ? null : (JQLAST)_t;
			expression(_t);
			_t = _retTree;
			right1_AST = (JQLAST)returnAST;
			astFactory.addASTChild(currentAST, returnAST);
			currentAST = __currentAST37;
			_t = __t37;
			_t = _t.getNextSibling();
			if ( inputState.guessing==0 ) {
				conditionalExpr_AST = (JQLAST)currentAST.root;
				
				conditionalExpr_AST = checkAnd(op1_AST, left1_AST, right1_AST);
				
				currentAST.root = conditionalExpr_AST;
				currentAST.child = conditionalExpr_AST!=null &&conditionalExpr_AST.getFirstChild()!=null ?
					conditionalExpr_AST.getFirstChild() : conditionalExpr_AST;
				currentAST.advanceChildToEnd();
			}
			conditionalExpr_AST = (JQLAST)currentAST.root;
			break;
		}
		case OR:
		{
			AST __t38 = _t;
			op2 = _t==ASTNULL ? null :(JQLAST)_t;
			JQLAST op2_AST_in = null;
			op2_AST = (JQLAST)astFactory.create(op2);
			astFactory.addASTChild(currentAST, op2_AST);
			ASTPair __currentAST38 = currentAST.copy();
			currentAST.root = currentAST.child;
			currentAST.child = null;
			match(_t,OR);
			_t = _t.getFirstChild();
			left2 = _t==ASTNULL ? null : (JQLAST)_t;
			expression(_t);
			_t = _retTree;
			left2_AST = (JQLAST)returnAST;
			astFactory.addASTChild(currentAST, returnAST);
			right2 = _t==ASTNULL ? null : (JQLAST)_t;
			expression(_t);
			_t = _retTree;
			right2_AST = (JQLAST)returnAST;
			astFactory.addASTChild(currentAST, returnAST);
			currentAST = __currentAST38;
			_t = __t38;
			_t = _t.getNextSibling();
			if ( inputState.guessing==0 ) {
				conditionalExpr_AST = (JQLAST)currentAST.root;
				
				conditionalExpr_AST = checkOr(op2_AST, left2_AST, right2_AST);
				
				currentAST.root = conditionalExpr_AST;
				currentAST.child = conditionalExpr_AST!=null &&conditionalExpr_AST.getFirstChild()!=null ?
					conditionalExpr_AST.getFirstChild() : conditionalExpr_AST;
				currentAST.advanceChildToEnd();
			}
			conditionalExpr_AST = (JQLAST)currentAST.root;
			break;
		}
		default:
		{
			throw new NoViableAltException(_t);
		}
		}
		returnAST = conditionalExpr_AST;
		_retTree = _t;
	}
	
	public final void relationalExpr(AST _t) throws RecognitionException {
		
		JQLAST relationalExpr_AST_in = (_t == ASTNULL) ? null : (JQLAST)_t;
		returnAST = null;
		ASTPair currentAST = new ASTPair();
		JQLAST relationalExpr_AST = null;
		JQLAST op1 = null;
		JQLAST op1_AST = null;
		JQLAST left1_AST = null;
		JQLAST left1 = null;
		JQLAST right1_AST = null;
		JQLAST right1 = null;
		JQLAST op2 = null;
		JQLAST op2_AST = null;
		JQLAST left2_AST = null;
		JQLAST left2 = null;
		JQLAST right2_AST = null;
		JQLAST right2 = null;
		JQLAST op3 = null;
		JQLAST op3_AST = null;
		JQLAST left3_AST = null;
		JQLAST left3 = null;
		JQLAST right3_AST = null;
		JQLAST right3 = null;
		JQLAST op4 = null;
		JQLAST op4_AST = null;
		JQLAST left4_AST = null;
		JQLAST left4 = null;
		JQLAST right4_AST = null;
		JQLAST right4 = null;
		JQLAST op5 = null;
		JQLAST op5_AST = null;
		JQLAST left5_AST = null;
		JQLAST left5 = null;
		JQLAST right5_AST = null;
		JQLAST right5 = null;
		JQLAST op6 = null;
		JQLAST op6_AST = null;
		JQLAST left6_AST = null;
		JQLAST left6 = null;
		JQLAST right6_AST = null;
		JQLAST right6 = null;
		
		if (_t==null) _t=ASTNULL;
		switch ( _t.getType()) {
		case EQUAL:
		{
			AST __t40 = _t;
			op1 = _t==ASTNULL ? null :(JQLAST)_t;
			JQLAST op1_AST_in = null;
			op1_AST = (JQLAST)astFactory.create(op1);
			astFactory.addASTChild(currentAST, op1_AST);
			ASTPair __currentAST40 = currentAST.copy();
			currentAST.root = currentAST.child;
			currentAST.child = null;
			match(_t,EQUAL);
			_t = _t.getFirstChild();
			left1 = _t==ASTNULL ? null : (JQLAST)_t;
			expression(_t);
			_t = _retTree;
			left1_AST = (JQLAST)returnAST;
			astFactory.addASTChild(currentAST, returnAST);
			right1 = _t==ASTNULL ? null : (JQLAST)_t;
			expression(_t);
			_t = _retTree;
			right1_AST = (JQLAST)returnAST;
			astFactory.addASTChild(currentAST, returnAST);
			currentAST = __currentAST40;
			_t = __t40;
			_t = _t.getNextSibling();
			if ( inputState.guessing==0 ) {
				relationalExpr_AST = (JQLAST)currentAST.root;
				
				relationalExpr_AST = checkEqualityOp(op1_AST, left1_AST, right1_AST, false);
				
				currentAST.root = relationalExpr_AST;
				currentAST.child = relationalExpr_AST!=null &&relationalExpr_AST.getFirstChild()!=null ?
					relationalExpr_AST.getFirstChild() : relationalExpr_AST;
				currentAST.advanceChildToEnd();
			}
			relationalExpr_AST = (JQLAST)currentAST.root;
			break;
		}
		case NOT_EQUAL:
		{
			AST __t41 = _t;
			op2 = _t==ASTNULL ? null :(JQLAST)_t;
			JQLAST op2_AST_in = null;
			op2_AST = (JQLAST)astFactory.create(op2);
			astFactory.addASTChild(currentAST, op2_AST);
			ASTPair __currentAST41 = currentAST.copy();
			currentAST.root = currentAST.child;
			currentAST.child = null;
			match(_t,NOT_EQUAL);
			_t = _t.getFirstChild();
			left2 = _t==ASTNULL ? null : (JQLAST)_t;
			expression(_t);
			_t = _retTree;
			left2_AST = (JQLAST)returnAST;
			astFactory.addASTChild(currentAST, returnAST);
			right2 = _t==ASTNULL ? null : (JQLAST)_t;
			expression(_t);
			_t = _retTree;
			right2_AST = (JQLAST)returnAST;
			astFactory.addASTChild(currentAST, returnAST);
			currentAST = __currentAST41;
			_t = __t41;
			_t = _t.getNextSibling();
			if ( inputState.guessing==0 ) {
				relationalExpr_AST = (JQLAST)currentAST.root;
				
				relationalExpr_AST = checkEqualityOp(op2_AST, left2_AST, right2_AST, true);
				
				currentAST.root = relationalExpr_AST;
				currentAST.child = relationalExpr_AST!=null &&relationalExpr_AST.getFirstChild()!=null ?
					relationalExpr_AST.getFirstChild() : relationalExpr_AST;
				currentAST.advanceChildToEnd();
			}
			relationalExpr_AST = (JQLAST)currentAST.root;
			break;
		}
		case OBJECT_EQUAL:
		{
			AST __t42 = _t;
			op3 = _t==ASTNULL ? null :(JQLAST)_t;
			JQLAST op3_AST_in = null;
			op3_AST = (JQLAST)astFactory.create(op3);
			astFactory.addASTChild(currentAST, op3_AST);
			ASTPair __currentAST42 = currentAST.copy();
			currentAST.root = currentAST.child;
			currentAST.child = null;
			match(_t,OBJECT_EQUAL);
			_t = _t.getFirstChild();
			left3 = _t==ASTNULL ? null : (JQLAST)_t;
			expression(_t);
			_t = _retTree;
			left3_AST = (JQLAST)returnAST;
			astFactory.addASTChild(currentAST, returnAST);
			right3 = _t==ASTNULL ? null : (JQLAST)_t;
			expression(_t);
			_t = _retTree;
			right3_AST = (JQLAST)returnAST;
			astFactory.addASTChild(currentAST, returnAST);
			currentAST = __currentAST42;
			_t = __t42;
			_t = _t.getNextSibling();
			if ( inputState.guessing==0 ) {
				relationalExpr_AST = (JQLAST)currentAST.root;
				
				relationalExpr_AST = checkObjectEqualityOp(op3_AST, left3_AST, right3_AST, false);
				
				currentAST.root = relationalExpr_AST;
				currentAST.child = relationalExpr_AST!=null &&relationalExpr_AST.getFirstChild()!=null ?
					relationalExpr_AST.getFirstChild() : relationalExpr_AST;
				currentAST.advanceChildToEnd();
			}
			relationalExpr_AST = (JQLAST)currentAST.root;
			break;
		}
		case OBJECT_NOT_EQUAL:
		{
			AST __t43 = _t;
			op4 = _t==ASTNULL ? null :(JQLAST)_t;
			JQLAST op4_AST_in = null;
			op4_AST = (JQLAST)astFactory.create(op4);
			astFactory.addASTChild(currentAST, op4_AST);
			ASTPair __currentAST43 = currentAST.copy();
			currentAST.root = currentAST.child;
			currentAST.child = null;
			match(_t,OBJECT_NOT_EQUAL);
			_t = _t.getFirstChild();
			left4 = _t==ASTNULL ? null : (JQLAST)_t;
			expression(_t);
			_t = _retTree;
			left4_AST = (JQLAST)returnAST;
			astFactory.addASTChild(currentAST, returnAST);
			right4 = _t==ASTNULL ? null : (JQLAST)_t;
			expression(_t);
			_t = _retTree;
			right4_AST = (JQLAST)returnAST;
			astFactory.addASTChild(currentAST, returnAST);
			currentAST = __currentAST43;
			_t = __t43;
			_t = _t.getNextSibling();
			if ( inputState.guessing==0 ) {
				relationalExpr_AST = (JQLAST)currentAST.root;
				
				relationalExpr_AST = checkObjectEqualityOp(op4_AST, left4_AST, right4_AST, true);
				
				currentAST.root = relationalExpr_AST;
				currentAST.child = relationalExpr_AST!=null &&relationalExpr_AST.getFirstChild()!=null ?
					relationalExpr_AST.getFirstChild() : relationalExpr_AST;
				currentAST.advanceChildToEnd();
			}
			relationalExpr_AST = (JQLAST)currentAST.root;
			break;
		}
		case COLLECTION_EQUAL:
		{
			AST __t44 = _t;
			op5 = _t==ASTNULL ? null :(JQLAST)_t;
			JQLAST op5_AST_in = null;
			op5_AST = (JQLAST)astFactory.create(op5);
			astFactory.addASTChild(currentAST, op5_AST);
			ASTPair __currentAST44 = currentAST.copy();
			currentAST.root = currentAST.child;
			currentAST.child = null;
			match(_t,COLLECTION_EQUAL);
			_t = _t.getFirstChild();
			left5 = _t==ASTNULL ? null : (JQLAST)_t;
			expression(_t);
			_t = _retTree;
			left5_AST = (JQLAST)returnAST;
			astFactory.addASTChild(currentAST, returnAST);
			right5 = _t==ASTNULL ? null : (JQLAST)_t;
			expression(_t);
			_t = _retTree;
			right5_AST = (JQLAST)returnAST;
			astFactory.addASTChild(currentAST, returnAST);
			currentAST = __currentAST44;
			_t = __t44;
			_t = _t.getNextSibling();
			if ( inputState.guessing==0 ) {
				relationalExpr_AST = (JQLAST)currentAST.root;
				
				relationalExpr_AST = checkCollectionEqualityOp(op5_AST, left5_AST, right5_AST, false);
				
				currentAST.root = relationalExpr_AST;
				currentAST.child = relationalExpr_AST!=null &&relationalExpr_AST.getFirstChild()!=null ?
					relationalExpr_AST.getFirstChild() : relationalExpr_AST;
				currentAST.advanceChildToEnd();
			}
			relationalExpr_AST = (JQLAST)currentAST.root;
			break;
		}
		case COLLECTION_NOT_EQUAL:
		{
			AST __t45 = _t;
			op6 = _t==ASTNULL ? null :(JQLAST)_t;
			JQLAST op6_AST_in = null;
			op6_AST = (JQLAST)astFactory.create(op6);
			astFactory.addASTChild(currentAST, op6_AST);
			ASTPair __currentAST45 = currentAST.copy();
			currentAST.root = currentAST.child;
			currentAST.child = null;
			match(_t,COLLECTION_NOT_EQUAL);
			_t = _t.getFirstChild();
			left6 = _t==ASTNULL ? null : (JQLAST)_t;
			expression(_t);
			_t = _retTree;
			left6_AST = (JQLAST)returnAST;
			astFactory.addASTChild(currentAST, returnAST);
			right6 = _t==ASTNULL ? null : (JQLAST)_t;
			expression(_t);
			_t = _retTree;
			right6_AST = (JQLAST)returnAST;
			astFactory.addASTChild(currentAST, returnAST);
			currentAST = __currentAST45;
			_t = __t45;
			_t = _t.getNextSibling();
			if ( inputState.guessing==0 ) {
				relationalExpr_AST = (JQLAST)currentAST.root;
				
				relationalExpr_AST = checkCollectionEqualityOp(op6_AST, left6_AST, right6_AST, true);
				
				currentAST.root = relationalExpr_AST;
				currentAST.child = relationalExpr_AST!=null &&relationalExpr_AST.getFirstChild()!=null ?
					relationalExpr_AST.getFirstChild() : relationalExpr_AST;
				currentAST.advanceChildToEnd();
			}
			relationalExpr_AST = (JQLAST)currentAST.root;
			break;
		}
		case LT:
		{
			AST __t46 = _t;
			JQLAST tmp23_AST = null;
			JQLAST tmp23_AST_in = null;
			tmp23_AST = (JQLAST)astFactory.create((JQLAST)_t);
			tmp23_AST_in = (JQLAST)_t;
			astFactory.addASTChild(currentAST, tmp23_AST);
			ASTPair __currentAST46 = currentAST.copy();
			currentAST.root = currentAST.child;
			currentAST.child = null;
			match(_t,LT);
			_t = _t.getFirstChild();
			expression(_t);
			_t = _retTree;
			astFactory.addASTChild(currentAST, returnAST);
			expression(_t);
			_t = _retTree;
			astFactory.addASTChild(currentAST, returnAST);
			currentAST = __currentAST46;
			_t = __t46;
			_t = _t.getNextSibling();
			relationalExpr_AST = (JQLAST)currentAST.root;
			break;
		}
		case GT:
		{
			AST __t47 = _t;
			JQLAST tmp24_AST = null;
			JQLAST tmp24_AST_in = null;
			tmp24_AST = (JQLAST)astFactory.create((JQLAST)_t);
			tmp24_AST_in = (JQLAST)_t;
			astFactory.addASTChild(currentAST, tmp24_AST);
			ASTPair __currentAST47 = currentAST.copy();
			currentAST.root = currentAST.child;
			currentAST.child = null;
			match(_t,GT);
			_t = _t.getFirstChild();
			expression(_t);
			_t = _retTree;
			astFactory.addASTChild(currentAST, returnAST);
			expression(_t);
			_t = _retTree;
			astFactory.addASTChild(currentAST, returnAST);
			currentAST = __currentAST47;
			_t = __t47;
			_t = _t.getNextSibling();
			relationalExpr_AST = (JQLAST)currentAST.root;
			break;
		}
		case LE:
		{
			AST __t48 = _t;
			JQLAST tmp25_AST = null;
			JQLAST tmp25_AST_in = null;
			tmp25_AST = (JQLAST)astFactory.create((JQLAST)_t);
			tmp25_AST_in = (JQLAST)_t;
			astFactory.addASTChild(currentAST, tmp25_AST);
			ASTPair __currentAST48 = currentAST.copy();
			currentAST.root = currentAST.child;
			currentAST.child = null;
			match(_t,LE);
			_t = _t.getFirstChild();
			expression(_t);
			_t = _retTree;
			astFactory.addASTChild(currentAST, returnAST);
			expression(_t);
			_t = _retTree;
			astFactory.addASTChild(currentAST, returnAST);
			currentAST = __currentAST48;
			_t = __t48;
			_t = _t.getNextSibling();
			relationalExpr_AST = (JQLAST)currentAST.root;
			break;
		}
		case GE:
		{
			AST __t49 = _t;
			JQLAST tmp26_AST = null;
			JQLAST tmp26_AST_in = null;
			tmp26_AST = (JQLAST)astFactory.create((JQLAST)_t);
			tmp26_AST_in = (JQLAST)_t;
			astFactory.addASTChild(currentAST, tmp26_AST);
			ASTPair __currentAST49 = currentAST.copy();
			currentAST.root = currentAST.child;
			currentAST.child = null;
			match(_t,GE);
			_t = _t.getFirstChild();
			expression(_t);
			_t = _retTree;
			astFactory.addASTChild(currentAST, returnAST);
			expression(_t);
			_t = _retTree;
			astFactory.addASTChild(currentAST, returnAST);
			currentAST = __currentAST49;
			_t = __t49;
			_t = _t.getNextSibling();
			relationalExpr_AST = (JQLAST)currentAST.root;
			break;
		}
		default:
		{
			throw new NoViableAltException(_t);
		}
		}
		returnAST = relationalExpr_AST;
		_retTree = _t;
	}
	
	public final void binaryArithmeticExpr(AST _t) throws RecognitionException {
		
		JQLAST binaryArithmeticExpr_AST_in = (_t == ASTNULL) ? null : (JQLAST)_t;
		returnAST = null;
		ASTPair currentAST = new ASTPair();
		JQLAST binaryArithmeticExpr_AST = null;
		JQLAST op1 = null;
		JQLAST op1_AST = null;
		JQLAST left1_AST = null;
		JQLAST left1 = null;
		JQLAST right1_AST = null;
		JQLAST right1 = null;
		JQLAST op2 = null;
		JQLAST op2_AST = null;
		JQLAST left2_AST = null;
		JQLAST left2 = null;
		JQLAST right2_AST = null;
		JQLAST right2 = null;
		JQLAST op3 = null;
		JQLAST op3_AST = null;
		JQLAST left3_AST = null;
		JQLAST left3 = null;
		JQLAST right3_AST = null;
		JQLAST right3 = null;
		JQLAST op4 = null;
		JQLAST op4_AST = null;
		JQLAST left4_AST = null;
		JQLAST left4 = null;
		JQLAST right4_AST = null;
		JQLAST right4 = null;
		JQLAST op5 = null;
		JQLAST op5_AST = null;
		JQLAST left5_AST = null;
		JQLAST left5 = null;
		JQLAST right5_AST = null;
		JQLAST right5 = null;
		JQLAST op6 = null;
		JQLAST op6_AST = null;
		JQLAST left6_AST = null;
		JQLAST left6 = null;
		JQLAST right6_AST = null;
		JQLAST right6 = null;
		
		if (_t==null) _t=ASTNULL;
		switch ( _t.getType()) {
		case PLUS:
		{
			AST __t51 = _t;
			op1 = _t==ASTNULL ? null :(JQLAST)_t;
			JQLAST op1_AST_in = null;
			op1_AST = (JQLAST)astFactory.create(op1);
			astFactory.addASTChild(currentAST, op1_AST);
			ASTPair __currentAST51 = currentAST.copy();
			currentAST.root = currentAST.child;
			currentAST.child = null;
			match(_t,PLUS);
			_t = _t.getFirstChild();
			left1 = _t==ASTNULL ? null : (JQLAST)_t;
			expression(_t);
			_t = _retTree;
			left1_AST = (JQLAST)returnAST;
			astFactory.addASTChild(currentAST, returnAST);
			right1 = _t==ASTNULL ? null : (JQLAST)_t;
			expression(_t);
			_t = _retTree;
			right1_AST = (JQLAST)returnAST;
			astFactory.addASTChild(currentAST, returnAST);
			currentAST = __currentAST51;
			_t = __t51;
			_t = _t.getNextSibling();
			if ( inputState.guessing==0 ) {
				binaryArithmeticExpr_AST = (JQLAST)currentAST.root;
				
				binaryArithmeticExpr_AST = checkBinaryPlusOp(op1_AST, left1_AST, right1_AST);
				
				currentAST.root = binaryArithmeticExpr_AST;
				currentAST.child = binaryArithmeticExpr_AST!=null &&binaryArithmeticExpr_AST.getFirstChild()!=null ?
					binaryArithmeticExpr_AST.getFirstChild() : binaryArithmeticExpr_AST;
				currentAST.advanceChildToEnd();
			}
			binaryArithmeticExpr_AST = (JQLAST)currentAST.root;
			break;
		}
		case CONCAT:
		{
			AST __t52 = _t;
			op2 = _t==ASTNULL ? null :(JQLAST)_t;
			JQLAST op2_AST_in = null;
			op2_AST = (JQLAST)astFactory.create(op2);
			astFactory.addASTChild(currentAST, op2_AST);
			ASTPair __currentAST52 = currentAST.copy();
			currentAST.root = currentAST.child;
			currentAST.child = null;
			match(_t,CONCAT);
			_t = _t.getFirstChild();
			left2 = _t==ASTNULL ? null : (JQLAST)_t;
			expression(_t);
			_t = _retTree;
			left2_AST = (JQLAST)returnAST;
			astFactory.addASTChild(currentAST, returnAST);
			right2 = _t==ASTNULL ? null : (JQLAST)_t;
			expression(_t);
			_t = _retTree;
			right2_AST = (JQLAST)returnAST;
			astFactory.addASTChild(currentAST, returnAST);
			currentAST = __currentAST52;
			_t = __t52;
			_t = _t.getNextSibling();
			if ( inputState.guessing==0 ) {
				binaryArithmeticExpr_AST = (JQLAST)currentAST.root;
				
				binaryArithmeticExpr_AST = checkConcatOp(op2_AST, left2_AST, right2_AST);
				
				currentAST.root = binaryArithmeticExpr_AST;
				currentAST.child = binaryArithmeticExpr_AST!=null &&binaryArithmeticExpr_AST.getFirstChild()!=null ?
					binaryArithmeticExpr_AST.getFirstChild() : binaryArithmeticExpr_AST;
				currentAST.advanceChildToEnd();
			}
			binaryArithmeticExpr_AST = (JQLAST)currentAST.root;
			break;
		}
		case MINUS:
		{
			AST __t53 = _t;
			op3 = _t==ASTNULL ? null :(JQLAST)_t;
			JQLAST op3_AST_in = null;
			op3_AST = (JQLAST)astFactory.create(op3);
			astFactory.addASTChild(currentAST, op3_AST);
			ASTPair __currentAST53 = currentAST.copy();
			currentAST.root = currentAST.child;
			currentAST.child = null;
			match(_t,MINUS);
			_t = _t.getFirstChild();
			left3 = _t==ASTNULL ? null : (JQLAST)_t;
			expression(_t);
			_t = _retTree;
			left3_AST = (JQLAST)returnAST;
			astFactory.addASTChild(currentAST, returnAST);
			right3 = _t==ASTNULL ? null : (JQLAST)_t;
			expression(_t);
			_t = _retTree;
			right3_AST = (JQLAST)returnAST;
			astFactory.addASTChild(currentAST, returnAST);
			currentAST = __currentAST53;
			_t = __t53;
			_t = _t.getNextSibling();
			if ( inputState.guessing==0 ) {
				binaryArithmeticExpr_AST = (JQLAST)currentAST.root;
				
				binaryArithmeticExpr_AST = checkBinaryMinusOp(op3_AST, left3_AST, right3_AST);
				
				currentAST.root = binaryArithmeticExpr_AST;
				currentAST.child = binaryArithmeticExpr_AST!=null &&binaryArithmeticExpr_AST.getFirstChild()!=null ?
					binaryArithmeticExpr_AST.getFirstChild() : binaryArithmeticExpr_AST;
				currentAST.advanceChildToEnd();
			}
			binaryArithmeticExpr_AST = (JQLAST)currentAST.root;
			break;
		}
		case STAR:
		{
			AST __t54 = _t;
			op4 = _t==ASTNULL ? null :(JQLAST)_t;
			JQLAST op4_AST_in = null;
			op4_AST = (JQLAST)astFactory.create(op4);
			astFactory.addASTChild(currentAST, op4_AST);
			ASTPair __currentAST54 = currentAST.copy();
			currentAST.root = currentAST.child;
			currentAST.child = null;
			match(_t,STAR);
			_t = _t.getFirstChild();
			left4 = _t==ASTNULL ? null : (JQLAST)_t;
			expression(_t);
			_t = _retTree;
			left4_AST = (JQLAST)returnAST;
			astFactory.addASTChild(currentAST, returnAST);
			right4 = _t==ASTNULL ? null : (JQLAST)_t;
			expression(_t);
			_t = _retTree;
			right4_AST = (JQLAST)returnAST;
			astFactory.addASTChild(currentAST, returnAST);
			currentAST = __currentAST54;
			_t = __t54;
			_t = _t.getNextSibling();
			if ( inputState.guessing==0 ) {
				binaryArithmeticExpr_AST = (JQLAST)currentAST.root;
				
				binaryArithmeticExpr_AST = checkMultiplicationOp(op4_AST, left4_AST, right4_AST);
				
				currentAST.root = binaryArithmeticExpr_AST;
				currentAST.child = binaryArithmeticExpr_AST!=null &&binaryArithmeticExpr_AST.getFirstChild()!=null ?
					binaryArithmeticExpr_AST.getFirstChild() : binaryArithmeticExpr_AST;
				currentAST.advanceChildToEnd();
			}
			binaryArithmeticExpr_AST = (JQLAST)currentAST.root;
			break;
		}
		case DIV:
		{
			AST __t55 = _t;
			op5 = _t==ASTNULL ? null :(JQLAST)_t;
			JQLAST op5_AST_in = null;
			op5_AST = (JQLAST)astFactory.create(op5);
			astFactory.addASTChild(currentAST, op5_AST);
			ASTPair __currentAST55 = currentAST.copy();
			currentAST.root = currentAST.child;
			currentAST.child = null;
			match(_t,DIV);
			_t = _t.getFirstChild();
			left5 = _t==ASTNULL ? null : (JQLAST)_t;
			expression(_t);
			_t = _retTree;
			left5_AST = (JQLAST)returnAST;
			astFactory.addASTChild(currentAST, returnAST);
			right5 = _t==ASTNULL ? null : (JQLAST)_t;
			expression(_t);
			_t = _retTree;
			right5_AST = (JQLAST)returnAST;
			astFactory.addASTChild(currentAST, returnAST);
			currentAST = __currentAST55;
			_t = __t55;
			_t = _t.getNextSibling();
			if ( inputState.guessing==0 ) {
				binaryArithmeticExpr_AST = (JQLAST)currentAST.root;
				
				binaryArithmeticExpr_AST = checkDivisionOp(op5_AST, left5_AST, right5_AST);
				
				currentAST.root = binaryArithmeticExpr_AST;
				currentAST.child = binaryArithmeticExpr_AST!=null &&binaryArithmeticExpr_AST.getFirstChild()!=null ?
					binaryArithmeticExpr_AST.getFirstChild() : binaryArithmeticExpr_AST;
				currentAST.advanceChildToEnd();
			}
			binaryArithmeticExpr_AST = (JQLAST)currentAST.root;
			break;
		}
		case MOD:
		{
			AST __t56 = _t;
			op6 = _t==ASTNULL ? null :(JQLAST)_t;
			JQLAST op6_AST_in = null;
			op6_AST = (JQLAST)astFactory.create(op6);
			astFactory.addASTChild(currentAST, op6_AST);
			ASTPair __currentAST56 = currentAST.copy();
			currentAST.root = currentAST.child;
			currentAST.child = null;
			match(_t,MOD);
			_t = _t.getFirstChild();
			left6 = _t==ASTNULL ? null : (JQLAST)_t;
			expression(_t);
			_t = _retTree;
			left6_AST = (JQLAST)returnAST;
			astFactory.addASTChild(currentAST, returnAST);
			right6 = _t==ASTNULL ? null : (JQLAST)_t;
			expression(_t);
			_t = _retTree;
			right6_AST = (JQLAST)returnAST;
			astFactory.addASTChild(currentAST, returnAST);
			currentAST = __currentAST56;
			_t = __t56;
			_t = _t.getNextSibling();
			if ( inputState.guessing==0 ) {
				binaryArithmeticExpr_AST = (JQLAST)currentAST.root;
				
				binaryArithmeticExpr_AST = checkModOp(op6_AST, left6_AST, right6_AST);
				
				currentAST.root = binaryArithmeticExpr_AST;
				currentAST.child = binaryArithmeticExpr_AST!=null &&binaryArithmeticExpr_AST.getFirstChild()!=null ?
					binaryArithmeticExpr_AST.getFirstChild() : binaryArithmeticExpr_AST;
				currentAST.advanceChildToEnd();
			}
			binaryArithmeticExpr_AST = (JQLAST)currentAST.root;
			break;
		}
		default:
		{
			throw new NoViableAltException(_t);
		}
		}
		returnAST = binaryArithmeticExpr_AST;
		_retTree = _t;
	}
	
	public final void unaryArithmeticExpr(AST _t) throws RecognitionException {
		
		JQLAST unaryArithmeticExpr_AST_in = (_t == ASTNULL) ? null : (JQLAST)_t;
		returnAST = null;
		ASTPair currentAST = new ASTPair();
		JQLAST unaryArithmeticExpr_AST = null;
		JQLAST op2 = null;
		JQLAST op2_AST = null;
		JQLAST arg2_AST = null;
		JQLAST arg2 = null;
		
		if (_t==null) _t=ASTNULL;
		if ((_t.getType()==UNARY_PLUS)) {
			AST __t58 = _t;
			JQLAST tmp27_AST = null;
			JQLAST tmp27_AST_in = null;
			tmp27_AST = (JQLAST)astFactory.create((JQLAST)_t);
			tmp27_AST_in = (JQLAST)_t;
			astFactory.addASTChild(currentAST, tmp27_AST);
			ASTPair __currentAST58 = currentAST.copy();
			currentAST.root = currentAST.child;
			currentAST.child = null;
			match(_t,UNARY_PLUS);
			_t = _t.getFirstChild();
			expression(_t);
			_t = _retTree;
			astFactory.addASTChild(currentAST, returnAST);
			currentAST = __currentAST58;
			_t = __t58;
			_t = _t.getNextSibling();
			unaryArithmeticExpr_AST = (JQLAST)currentAST.root;
		}
		else {
			boolean synPredMatched60 = false;
			if (_t==null) _t=ASTNULL;
			if (((_t.getType()==UNARY_MINUS))) {
				AST __t60 = _t;
				synPredMatched60 = true;
				inputState.guessing++;
				try {
					{
					unaryMinusLiteralExpr(_t);
					_t = _retTree;
					}
				}
				catch (RecognitionException pe) {
					synPredMatched60 = false;
				}
				_t = __t60;
inputState.guessing--;
			}
			if ( synPredMatched60 ) {
				unaryMinusLiteralExpr(_t);
				_t = _retTree;
				astFactory.addASTChild(currentAST, returnAST);
				unaryArithmeticExpr_AST = (JQLAST)currentAST.root;
			}
			else if ((_t.getType()==UNARY_MINUS)) {
				AST __t61 = _t;
				op2 = _t==ASTNULL ? null :(JQLAST)_t;
				JQLAST op2_AST_in = null;
				op2_AST = (JQLAST)astFactory.create(op2);
				astFactory.addASTChild(currentAST, op2_AST);
				ASTPair __currentAST61 = currentAST.copy();
				currentAST.root = currentAST.child;
				currentAST.child = null;
				match(_t,UNARY_MINUS);
				_t = _t.getFirstChild();
				arg2 = _t==ASTNULL ? null : (JQLAST)_t;
				expression(_t);
				_t = _retTree;
				arg2_AST = (JQLAST)returnAST;
				astFactory.addASTChild(currentAST, returnAST);
				currentAST = __currentAST61;
				_t = __t61;
				_t = _t.getNextSibling();
				if ( inputState.guessing==0 ) {
					unaryArithmeticExpr_AST = (JQLAST)currentAST.root;
					
					unaryArithmeticExpr_AST = checkUnaryMinusOp(op2_AST, arg2_AST);
					
					currentAST.root = unaryArithmeticExpr_AST;
					currentAST.child = unaryArithmeticExpr_AST!=null &&unaryArithmeticExpr_AST.getFirstChild()!=null ?
						unaryArithmeticExpr_AST.getFirstChild() : unaryArithmeticExpr_AST;
					currentAST.advanceChildToEnd();
				}
				unaryArithmeticExpr_AST = (JQLAST)currentAST.root;
			}
			else {
				throw new NoViableAltException(_t);
			}
			}
			returnAST = unaryArithmeticExpr_AST;
			_retTree = _t;
		}
		
	public final void complementExpr(AST _t) throws RecognitionException {
		
		JQLAST complementExpr_AST_in = (_t == ASTNULL) ? null : (JQLAST)_t;
		returnAST = null;
		ASTPair currentAST = new ASTPair();
		JQLAST complementExpr_AST = null;
		JQLAST op1 = null;
		JQLAST op1_AST = null;
		JQLAST arg1_AST = null;
		JQLAST arg1 = null;
		JQLAST op2 = null;
		JQLAST op2_AST = null;
		JQLAST arg2_AST = null;
		JQLAST arg2 = null;
		
		if (_t==null) _t=ASTNULL;
		switch ( _t.getType()) {
		case BNOT:
		{
			AST __t66 = _t;
			op1 = _t==ASTNULL ? null :(JQLAST)_t;
			JQLAST op1_AST_in = null;
			op1_AST = (JQLAST)astFactory.create(op1);
			astFactory.addASTChild(currentAST, op1_AST);
			ASTPair __currentAST66 = currentAST.copy();
			currentAST.root = currentAST.child;
			currentAST.child = null;
			match(_t,BNOT);
			_t = _t.getFirstChild();
			arg1 = _t==ASTNULL ? null : (JQLAST)_t;
			expression(_t);
			_t = _retTree;
			arg1_AST = (JQLAST)returnAST;
			astFactory.addASTChild(currentAST, returnAST);
			currentAST = __currentAST66;
			_t = __t66;
			_t = _t.getNextSibling();
			complementExpr_AST = (JQLAST)currentAST.root;
			break;
		}
		case LNOT:
		{
			AST __t67 = _t;
			op2 = _t==ASTNULL ? null :(JQLAST)_t;
			JQLAST op2_AST_in = null;
			op2_AST = (JQLAST)astFactory.create(op2);
			astFactory.addASTChild(currentAST, op2_AST);
			ASTPair __currentAST67 = currentAST.copy();
			currentAST.root = currentAST.child;
			currentAST.child = null;
			match(_t,LNOT);
			_t = _t.getFirstChild();
			arg2 = _t==ASTNULL ? null : (JQLAST)_t;
			expression(_t);
			_t = _retTree;
			arg2_AST = (JQLAST)returnAST;
			astFactory.addASTChild(currentAST, returnAST);
			currentAST = __currentAST67;
			_t = __t67;
			_t = _t.getNextSibling();
			if ( inputState.guessing==0 ) {
				complementExpr_AST = (JQLAST)currentAST.root;
				
				complementExpr_AST = checkLogicalNotOp(op2_AST, arg2_AST);
				
				currentAST.root = complementExpr_AST;
				currentAST.child = complementExpr_AST!=null &&complementExpr_AST.getFirstChild()!=null ?
					complementExpr_AST.getFirstChild() : complementExpr_AST;
				currentAST.advanceChildToEnd();
			}
			complementExpr_AST = (JQLAST)currentAST.root;
			break;
		}
		default:
		{
			throw new NoViableAltException(_t);
		}
		}
		returnAST = complementExpr_AST;
		_retTree = _t;
	}
	
	public final void unaryMinusLiteralExpr(AST _t) throws RecognitionException {
		
		JQLAST unaryMinusLiteralExpr_AST_in = (_t == ASTNULL) ? null : (JQLAST)_t;
		returnAST = null;
		ASTPair currentAST = new ASTPair();
		JQLAST unaryMinusLiteralExpr_AST = null;
		JQLAST i = null;
		JQLAST i_AST = null;
		JQLAST l = null;
		JQLAST l_AST = null;
		
		AST __t63 = _t;
		JQLAST tmp28_AST = null;
		JQLAST tmp28_AST_in = null;
		tmp28_AST = (JQLAST)astFactory.create((JQLAST)_t);
		tmp28_AST_in = (JQLAST)_t;
		astFactory.addASTChild(currentAST, tmp28_AST);
		ASTPair __currentAST63 = currentAST.copy();
		currentAST.root = currentAST.child;
		currentAST.child = null;
		match(_t,UNARY_MINUS);
		_t = _t.getFirstChild();
		{
		if (_t==null) _t=ASTNULL;
		switch ( _t.getType()) {
		case INT_LITERAL:
		{
			i = (JQLAST)_t;
			JQLAST i_AST_in = null;
			i_AST = (JQLAST)astFactory.create(i);
			astFactory.addASTChild(currentAST, i_AST);
			match(_t,INT_LITERAL);
			_t = _t.getNextSibling();
			break;
		}
		case LONG_LITERAL:
		{
			l = (JQLAST)_t;
			JQLAST l_AST_in = null;
			l_AST = (JQLAST)astFactory.create(l);
			astFactory.addASTChild(currentAST, l_AST);
			match(_t,LONG_LITERAL);
			_t = _t.getNextSibling();
			break;
		}
		default:
		{
			throw new NoViableAltException(_t);
		}
		}
		}
		currentAST = __currentAST63;
		_t = __t63;
		_t = _t.getNextSibling();
		if ( inputState.guessing==0 ) {
			unaryMinusLiteralExpr_AST = (JQLAST)currentAST.root;
			
			JQLAST li = (i_AST != null) ? i_AST : l_AST;
			li.setText("-" + li.getText());
			// calling literal here directly does not work properly
			// the following logic need to be in sync with that of literal
			li.setValue(literalHelper(li));
			li.setType(VALUE);
			unaryMinusLiteralExpr_AST = li;
			
			currentAST.root = unaryMinusLiteralExpr_AST;
			currentAST.child = unaryMinusLiteralExpr_AST!=null &&unaryMinusLiteralExpr_AST.getFirstChild()!=null ?
				unaryMinusLiteralExpr_AST.getFirstChild() : unaryMinusLiteralExpr_AST;
			currentAST.advanceChildToEnd();
		}
		unaryMinusLiteralExpr_AST = (JQLAST)currentAST.root;
		returnAST = unaryMinusLiteralExpr_AST;
		_retTree = _t;
	}
	
	public final void castExpr(AST _t) throws RecognitionException {
		
		JQLAST castExpr_AST_in = (_t == ASTNULL) ? null : (JQLAST)_t;
		returnAST = null;
		ASTPair currentAST = new ASTPair();
		JQLAST castExpr_AST = null;
		JQLAST c = null;
		JQLAST c_AST = null;
		JQLAST t_AST = null;
		JQLAST t = null;
		JQLAST e_AST = null;
		JQLAST e = null;
		
		AST __t72 = _t;
		c = _t==ASTNULL ? null :(JQLAST)_t;
		JQLAST c_AST_in = null;
		c_AST = (JQLAST)astFactory.create(c);
		astFactory.addASTChild(currentAST, c_AST);
		ASTPair __currentAST72 = currentAST.copy();
		currentAST.root = currentAST.child;
		currentAST.child = null;
		match(_t,TYPECAST);
		_t = _t.getFirstChild();
		t = _t==ASTNULL ? null : (JQLAST)_t;
		type(_t);
		_t = _retTree;
		t_AST = (JQLAST)returnAST;
		astFactory.addASTChild(currentAST, returnAST);
		e = _t==ASTNULL ? null : (JQLAST)_t;
		expression(_t);
		_t = _retTree;
		e_AST = (JQLAST)returnAST;
		astFactory.addASTChild(currentAST, returnAST);
		currentAST = __currentAST72;
		_t = __t72;
		_t = _t.getNextSibling();
		if ( inputState.guessing==0 ) {
			castExpr_AST = (JQLAST)currentAST.root;
			
			castExpr_AST = checkCastOp(c_AST, t_AST, e_AST);
			
			currentAST.root = castExpr_AST;
			currentAST.child = castExpr_AST!=null &&castExpr_AST.getFirstChild()!=null ?
				castExpr_AST.getFirstChild() : castExpr_AST;
			currentAST.advanceChildToEnd();
		}
		castExpr_AST = (JQLAST)currentAST.root;
		returnAST = castExpr_AST;
		_retTree = _t;
	}
	
	public final void literal(AST _t) throws RecognitionException {
		
		JQLAST literal_AST_in = (_t == ASTNULL) ? null : (JQLAST)_t;
		returnAST = null;
		ASTPair currentAST = new ASTPair();
		JQLAST literal_AST = null;
		JQLAST l_AST = null;
		JQLAST l = null;
		
		Object value = null; 
		
		
		l = _t==ASTNULL ? null : (JQLAST)_t;
		value=literalHelper(_t);
		_t = _retTree;
		l_AST = (JQLAST)returnAST;
		astFactory.addASTChild(currentAST, returnAST);
		if ( inputState.guessing==0 ) {
			
			l_AST.setType(VALUE);
			l_AST.setValue(value);
			
		}
		literal_AST = (JQLAST)currentAST.root;
		returnAST = literal_AST;
		_retTree = _t;
	}
	
	public final void parameter(AST _t) throws RecognitionException {
		
		JQLAST parameter_AST_in = (_t == ASTNULL) ? null : (JQLAST)_t;
		returnAST = null;
		ASTPair currentAST = new ASTPair();
		JQLAST parameter_AST = null;
		JQLAST p = null;
		JQLAST p_AST = null;
		
		p = (JQLAST)_t;
		JQLAST p_AST_in = null;
		p_AST = (JQLAST)astFactory.create(p);
		astFactory.addASTChild(currentAST, p_AST);
		match(_t,PARAMETER);
		_t = _t.getNextSibling();
		if ( inputState.guessing==0 ) {
			
			if (paramtab.inline(p_AST.getText())) {
			p_AST.setType(VALUE);
			p_AST.setValue(paramtab.getValueByName(p_AST.getText()));     
			}
			
		}
		parameter_AST = (JQLAST)currentAST.root;
		returnAST = parameter_AST;
		_retTree = _t;
	}
	
	public final void staticFieldAccess(AST _t) throws RecognitionException {
		
		JQLAST staticFieldAccess_AST_in = (_t == ASTNULL) ? null : (JQLAST)_t;
		returnAST = null;
		ASTPair currentAST = new ASTPair();
		JQLAST staticFieldAccess_AST = null;
		JQLAST s = null;
		JQLAST s_AST = null;
		JQLAST t = null;
		JQLAST t_AST = null;
		JQLAST i = null;
		JQLAST i_AST = null;
		
		Object value = null; 
		
		
		AST __t77 = _t;
		s = _t==ASTNULL ? null :(JQLAST)_t;
		JQLAST s_AST_in = null;
		s_AST = (JQLAST)astFactory.create(s);
		astFactory.addASTChild(currentAST, s_AST);
		ASTPair __currentAST77 = currentAST.copy();
		currentAST.root = currentAST.child;
		currentAST.child = null;
		match(_t,STATIC_FIELD_ACCESS);
		_t = _t.getFirstChild();
		t = (JQLAST)_t;
		JQLAST t_AST_in = null;
		t_AST = (JQLAST)astFactory.create(t);
		astFactory.addASTChild(currentAST, t_AST);
		match(_t,TYPENAME);
		_t = _t.getNextSibling();
		i = (JQLAST)_t;
		JQLAST i_AST_in = null;
		i_AST = (JQLAST)astFactory.create(i);
		astFactory.addASTChild(currentAST, i_AST);
		match(_t,IDENT);
		_t = _t.getNextSibling();
		currentAST = __currentAST77;
		_t = __t77;
		_t = _t.getNextSibling();
		if ( inputState.guessing==0 ) {
			
			// Calculate the value of the static field at compile time
			// and treat it as constant value.
			ClassType classType = (ClassType)t.getJQLType();
			FieldInfo fieldInfo = classType.getFieldInfo(i.getText());
			try
			{
			value = fieldInfo.getField().get(null);
			s_AST.setType(VALUE);
			s_AST.setValue(value);
			s_AST.setFirstChild(null);
			}
			catch (IllegalAccessException e) 
			{
			throw new JDOFatalUserException(
			I18NHelper.getMessage(messages, 
			"jqlc.optimizer.staticfieldaccess.illegal",  //NOI18N
			i.getText(), classType.getName()), e);
			}
			
		}
		staticFieldAccess_AST = (JQLAST)currentAST.root;
		returnAST = staticFieldAccess_AST;
		_retTree = _t;
	}
	
	public final void fieldAccess(AST _t) throws RecognitionException {
		
		JQLAST fieldAccess_AST_in = (_t == ASTNULL) ? null : (JQLAST)_t;
		returnAST = null;
		ASTPair currentAST = new ASTPair();
		JQLAST fieldAccess_AST = null;
		JQLAST f = null;
		JQLAST f_AST = null;
		JQLAST o_AST = null;
		JQLAST o = null;
		JQLAST name = null;
		JQLAST name_AST = null;
		
		AST __t79 = _t;
		f = _t==ASTNULL ? null :(JQLAST)_t;
		JQLAST f_AST_in = null;
		f_AST = (JQLAST)astFactory.create(f);
		astFactory.addASTChild(currentAST, f_AST);
		ASTPair __currentAST79 = currentAST.copy();
		currentAST.root = currentAST.child;
		currentAST.child = null;
		match(_t,FIELD_ACCESS);
		_t = _t.getFirstChild();
		o = _t==ASTNULL ? null : (JQLAST)_t;
		expression(_t);
		_t = _retTree;
		o_AST = (JQLAST)returnAST;
		astFactory.addASTChild(currentAST, returnAST);
		name = (JQLAST)_t;
		JQLAST name_AST_in = null;
		name_AST = (JQLAST)astFactory.create(name);
		astFactory.addASTChild(currentAST, name_AST);
		match(_t,IDENT);
		_t = _t.getNextSibling();
		currentAST = __currentAST79;
		_t = __t79;
		_t = _t.getNextSibling();
		if ( inputState.guessing==0 ) {
			
			if (o_AST.getType() == VALUE)
			{
			// If the object of the field access is a constant value, 
			// evaluate the field access at compile time and 
			// treat the expression as constant value.
			Object object = o_AST.getValue();
			ClassType classType = (ClassType)o_AST.getJQLType();
			Object value = CodeGeneration.getFieldValue(classType, object, 
			name_AST.getText());
			f_AST.setType(VALUE);
			f_AST.setValue(value);
			f_AST.setFirstChild(null);
			}
			
		}
		fieldAccess_AST = (JQLAST)currentAST.root;
		returnAST = fieldAccess_AST;
		_retTree = _t;
	}
	
	public final void navigation(AST _t) throws RecognitionException {
		
		JQLAST navigation_AST_in = (_t == ASTNULL) ? null : (JQLAST)_t;
		returnAST = null;
		ASTPair currentAST = new ASTPair();
		JQLAST navigation_AST = null;
		JQLAST n = null;
		JQLAST n_AST = null;
		JQLAST o_AST = null;
		JQLAST o = null;
		JQLAST name = null;
		JQLAST name_AST = null;
		
		AST __t81 = _t;
		n = _t==ASTNULL ? null :(JQLAST)_t;
		JQLAST n_AST_in = null;
		n_AST = (JQLAST)astFactory.create(n);
		astFactory.addASTChild(currentAST, n_AST);
		ASTPair __currentAST81 = currentAST.copy();
		currentAST.root = currentAST.child;
		currentAST.child = null;
		match(_t,NAVIGATION);
		_t = _t.getFirstChild();
		o = _t==ASTNULL ? null : (JQLAST)_t;
		expression(_t);
		_t = _retTree;
		o_AST = (JQLAST)returnAST;
		astFactory.addASTChild(currentAST, returnAST);
		name = (JQLAST)_t;
		JQLAST name_AST_in = null;
		name_AST = (JQLAST)astFactory.create(name);
		astFactory.addASTChild(currentAST, name_AST);
		match(_t,IDENT);
		_t = _t.getNextSibling();
		currentAST = __currentAST81;
		_t = __t81;
		_t = _t.getNextSibling();
		if ( inputState.guessing==0 ) {
			
			if (o_AST.getType() == VALUE)
			{
			// If the object of the navigation is a constant value, 
			// evaluate the field access at compile time and 
			// treat the expression as constant value.
			Object object = o_AST.getValue();
			ClassType classType = (ClassType)o_AST.getJQLType();
			Object value = CodeGeneration.getFieldValue(classType, object, 
			name_AST.getText());
			n_AST.setType(VALUE);
			n_AST.setValue(value);
			n_AST.setFirstChild(null);
			}
			
		}
		navigation_AST = (JQLAST)currentAST.root;
		returnAST = navigation_AST;
		_retTree = _t;
	}
	
	public final void variableAccess(AST _t) throws RecognitionException {
		
		JQLAST variableAccess_AST_in = (_t == ASTNULL) ? null : (JQLAST)_t;
		returnAST = null;
		ASTPair currentAST = new ASTPair();
		JQLAST variableAccess_AST = null;
		
		AST __t83 = _t;
		JQLAST tmp29_AST = null;
		JQLAST tmp29_AST_in = null;
		tmp29_AST = (JQLAST)astFactory.create((JQLAST)_t);
		tmp29_AST_in = (JQLAST)_t;
		astFactory.addASTChild(currentAST, tmp29_AST);
		ASTPair __currentAST83 = currentAST.copy();
		currentAST.root = currentAST.child;
		currentAST.child = null;
		match(_t,VARIABLE);
		_t = _t.getFirstChild();
		{
		if (_t==null) _t=ASTNULL;
		switch ( _t.getType()) {
		case THIS:
		case NULL:
		case TRUE:
		case FALSE:
		case EQUAL:
		case LNOT:
		case BNOT:
		case NOT_EQUAL:
		case DIV:
		case PLUS:
		case MINUS:
		case STAR:
		case MOD:
		case GE:
		case GT:
		case LE:
		case LT:
		case BXOR:
		case BOR:
		case OR:
		case BAND:
		case AND:
		case CHAR_LITERAL:
		case STRING_LITERAL:
		case INT_LITERAL:
		case UNARY_MINUS:
		case UNARY_PLUS:
		case TYPECAST:
		case OBJECT_EQUAL:
		case OBJECT_NOT_EQUAL:
		case COLLECTION_EQUAL:
		case COLLECTION_NOT_EQUAL:
		case CONCAT:
		case FIELD_ACCESS:
		case STATIC_FIELD_ACCESS:
		case CONTAINS:
		case NOT_CONTAINS:
		case NAVIGATION:
		case STARTS_WITH:
		case ENDS_WITH:
		case IS_EMPTY:
		case VARIABLE:
		case PARAMETER:
		case VALUE:
		case LIKE:
		case SUBSTRING:
		case INDEXOF:
		case LENGTH:
		case ABS:
		case SQRT:
		case LONG_LITERAL:
		case FLOAT_LITERAL:
		case DOUBLE_LITERAL:
		{
			expression(_t);
			_t = _retTree;
			astFactory.addASTChild(currentAST, returnAST);
			break;
		}
		case 3:
		{
			break;
		}
		default:
		{
			throw new NoViableAltException(_t);
		}
		}
		}
		currentAST = __currentAST83;
		_t = __t83;
		_t = _t.getNextSibling();
		variableAccess_AST = (JQLAST)currentAST.root;
		returnAST = variableAccess_AST;
		_retTree = _t;
	}
	
	public final void startsWith(AST _t) throws RecognitionException {
		
		JQLAST startsWith_AST_in = (_t == ASTNULL) ? null : (JQLAST)_t;
		returnAST = null;
		ASTPair currentAST = new ASTPair();
		JQLAST startsWith_AST = null;
		
		AST __t86 = _t;
		JQLAST tmp30_AST = null;
		JQLAST tmp30_AST_in = null;
		tmp30_AST = (JQLAST)astFactory.create((JQLAST)_t);
		tmp30_AST_in = (JQLAST)_t;
		astFactory.addASTChild(currentAST, tmp30_AST);
		ASTPair __currentAST86 = currentAST.copy();
		currentAST.root = currentAST.child;
		currentAST.child = null;
		match(_t,STARTS_WITH);
		_t = _t.getFirstChild();
		expression(_t);
		_t = _retTree;
		astFactory.addASTChild(currentAST, returnAST);
		expression(_t);
		_t = _retTree;
		astFactory.addASTChild(currentAST, returnAST);
		currentAST = __currentAST86;
		_t = __t86;
		_t = _t.getNextSibling();
		startsWith_AST = (JQLAST)currentAST.root;
		returnAST = startsWith_AST;
		_retTree = _t;
	}
	
	public final void endsWith(AST _t) throws RecognitionException {
		
		JQLAST endsWith_AST_in = (_t == ASTNULL) ? null : (JQLAST)_t;
		returnAST = null;
		ASTPair currentAST = new ASTPair();
		JQLAST endsWith_AST = null;
		
		AST __t88 = _t;
		JQLAST tmp31_AST = null;
		JQLAST tmp31_AST_in = null;
		tmp31_AST = (JQLAST)astFactory.create((JQLAST)_t);
		tmp31_AST_in = (JQLAST)_t;
		astFactory.addASTChild(currentAST, tmp31_AST);
		ASTPair __currentAST88 = currentAST.copy();
		currentAST.root = currentAST.child;
		currentAST.child = null;
		match(_t,ENDS_WITH);
		_t = _t.getFirstChild();
		expression(_t);
		_t = _retTree;
		astFactory.addASTChild(currentAST, returnAST);
		expression(_t);
		_t = _retTree;
		astFactory.addASTChild(currentAST, returnAST);
		currentAST = __currentAST88;
		_t = __t88;
		_t = _t.getNextSibling();
		endsWith_AST = (JQLAST)currentAST.root;
		returnAST = endsWith_AST;
		_retTree = _t;
	}
	
	public final void isEmpty(AST _t) throws RecognitionException {
		
		JQLAST isEmpty_AST_in = (_t == ASTNULL) ? null : (JQLAST)_t;
		returnAST = null;
		ASTPair currentAST = new ASTPair();
		JQLAST isEmpty_AST = null;
		JQLAST op = null;
		JQLAST op_AST = null;
		JQLAST e_AST = null;
		JQLAST e = null;
		
		AST __t90 = _t;
		op = _t==ASTNULL ? null :(JQLAST)_t;
		JQLAST op_AST_in = null;
		op_AST = (JQLAST)astFactory.create(op);
		astFactory.addASTChild(currentAST, op_AST);
		ASTPair __currentAST90 = currentAST.copy();
		currentAST.root = currentAST.child;
		currentAST.child = null;
		match(_t,IS_EMPTY);
		_t = _t.getFirstChild();
		e = _t==ASTNULL ? null : (JQLAST)_t;
		expression(_t);
		_t = _retTree;
		e_AST = (JQLAST)returnAST;
		astFactory.addASTChild(currentAST, returnAST);
		currentAST = __currentAST90;
		_t = __t90;
		_t = _t.getNextSibling();
		if ( inputState.guessing==0 ) {
			
			if (e_AST.getType() == VALUE)
			{
			// If the expression that specifies the collection is a constant value, 
			// evaluate the isEmpty call at compile time and treat the expression 
			// as constant value.
			Object object = e_AST.getValue();
			Object value = null;
			if (object == null)
			{
			value = new Boolean(false);
			}
			else if (object instanceof Collection)
			{
			value = new Boolean(((Collection)object).isEmpty());
			}
			else
			{
			errorMsg.fatal(I18NHelper.getMessage(messages, "jqlc.optimizer.isempty.requirecollection")); //NOI18N
			}
			op_AST.setType(VALUE);
			op_AST.setValue(value);
			op_AST.setFirstChild(null);
			}
			
		}
		isEmpty_AST = (JQLAST)currentAST.root;
		returnAST = isEmpty_AST;
		_retTree = _t;
	}
	
	public final void like(AST _t) throws RecognitionException {
		
		JQLAST like_AST_in = (_t == ASTNULL) ? null : (JQLAST)_t;
		returnAST = null;
		ASTPair currentAST = new ASTPair();
		JQLAST like_AST = null;
		
		AST __t92 = _t;
		JQLAST tmp32_AST = null;
		JQLAST tmp32_AST_in = null;
		tmp32_AST = (JQLAST)astFactory.create((JQLAST)_t);
		tmp32_AST_in = (JQLAST)_t;
		astFactory.addASTChild(currentAST, tmp32_AST);
		ASTPair __currentAST92 = currentAST.copy();
		currentAST.root = currentAST.child;
		currentAST.child = null;
		match(_t,LIKE);
		_t = _t.getFirstChild();
		expression(_t);
		_t = _retTree;
		astFactory.addASTChild(currentAST, returnAST);
		expression(_t);
		_t = _retTree;
		astFactory.addASTChild(currentAST, returnAST);
		{
		if (_t==null) _t=ASTNULL;
		switch ( _t.getType()) {
		case THIS:
		case NULL:
		case TRUE:
		case FALSE:
		case EQUAL:
		case LNOT:
		case BNOT:
		case NOT_EQUAL:
		case DIV:
		case PLUS:
		case MINUS:
		case STAR:
		case MOD:
		case GE:
		case GT:
		case LE:
		case LT:
		case BXOR:
		case BOR:
		case OR:
		case BAND:
		case AND:
		case CHAR_LITERAL:
		case STRING_LITERAL:
		case INT_LITERAL:
		case UNARY_MINUS:
		case UNARY_PLUS:
		case TYPECAST:
		case OBJECT_EQUAL:
		case OBJECT_NOT_EQUAL:
		case COLLECTION_EQUAL:
		case COLLECTION_NOT_EQUAL:
		case CONCAT:
		case FIELD_ACCESS:
		case STATIC_FIELD_ACCESS:
		case CONTAINS:
		case NOT_CONTAINS:
		case NAVIGATION:
		case STARTS_WITH:
		case ENDS_WITH:
		case IS_EMPTY:
		case VARIABLE:
		case PARAMETER:
		case VALUE:
		case LIKE:
		case SUBSTRING:
		case INDEXOF:
		case LENGTH:
		case ABS:
		case SQRT:
		case LONG_LITERAL:
		case FLOAT_LITERAL:
		case DOUBLE_LITERAL:
		{
			expression(_t);
			_t = _retTree;
			astFactory.addASTChild(currentAST, returnAST);
			break;
		}
		case 3:
		{
			break;
		}
		default:
		{
			throw new NoViableAltException(_t);
		}
		}
		}
		currentAST = __currentAST92;
		_t = __t92;
		_t = _t.getNextSibling();
		like_AST = (JQLAST)currentAST.root;
		returnAST = like_AST;
		_retTree = _t;
	}
	
	public final void substring(AST _t) throws RecognitionException {
		
		JQLAST substring_AST_in = (_t == ASTNULL) ? null : (JQLAST)_t;
		returnAST = null;
		ASTPair currentAST = new ASTPair();
		JQLAST substring_AST = null;
		
		AST __t95 = _t;
		JQLAST tmp33_AST = null;
		JQLAST tmp33_AST_in = null;
		tmp33_AST = (JQLAST)astFactory.create((JQLAST)_t);
		tmp33_AST_in = (JQLAST)_t;
		astFactory.addASTChild(currentAST, tmp33_AST);
		ASTPair __currentAST95 = currentAST.copy();
		currentAST.root = currentAST.child;
		currentAST.child = null;
		match(_t,SUBSTRING);
		_t = _t.getFirstChild();
		expression(_t);
		_t = _retTree;
		astFactory.addASTChild(currentAST, returnAST);
		expression(_t);
		_t = _retTree;
		astFactory.addASTChild(currentAST, returnAST);
		expression(_t);
		_t = _retTree;
		astFactory.addASTChild(currentAST, returnAST);
		currentAST = __currentAST95;
		_t = __t95;
		_t = _t.getNextSibling();
		substring_AST = (JQLAST)currentAST.root;
		returnAST = substring_AST;
		_retTree = _t;
	}
	
	public final void indexOf(AST _t) throws RecognitionException {
		
		JQLAST indexOf_AST_in = (_t == ASTNULL) ? null : (JQLAST)_t;
		returnAST = null;
		ASTPair currentAST = new ASTPair();
		JQLAST indexOf_AST = null;
		
		AST __t97 = _t;
		JQLAST tmp34_AST = null;
		JQLAST tmp34_AST_in = null;
		tmp34_AST = (JQLAST)astFactory.create((JQLAST)_t);
		tmp34_AST_in = (JQLAST)_t;
		astFactory.addASTChild(currentAST, tmp34_AST);
		ASTPair __currentAST97 = currentAST.copy();
		currentAST.root = currentAST.child;
		currentAST.child = null;
		match(_t,INDEXOF);
		_t = _t.getFirstChild();
		expression(_t);
		_t = _retTree;
		astFactory.addASTChild(currentAST, returnAST);
		expression(_t);
		_t = _retTree;
		astFactory.addASTChild(currentAST, returnAST);
		{
		if (_t==null) _t=ASTNULL;
		switch ( _t.getType()) {
		case THIS:
		case NULL:
		case TRUE:
		case FALSE:
		case EQUAL:
		case LNOT:
		case BNOT:
		case NOT_EQUAL:
		case DIV:
		case PLUS:
		case MINUS:
		case STAR:
		case MOD:
		case GE:
		case GT:
		case LE:
		case LT:
		case BXOR:
		case BOR:
		case OR:
		case BAND:
		case AND:
		case CHAR_LITERAL:
		case STRING_LITERAL:
		case INT_LITERAL:
		case UNARY_MINUS:
		case UNARY_PLUS:
		case TYPECAST:
		case OBJECT_EQUAL:
		case OBJECT_NOT_EQUAL:
		case COLLECTION_EQUAL:
		case COLLECTION_NOT_EQUAL:
		case CONCAT:
		case FIELD_ACCESS:
		case STATIC_FIELD_ACCESS:
		case CONTAINS:
		case NOT_CONTAINS:
		case NAVIGATION:
		case STARTS_WITH:
		case ENDS_WITH:
		case IS_EMPTY:
		case VARIABLE:
		case PARAMETER:
		case VALUE:
		case LIKE:
		case SUBSTRING:
		case INDEXOF:
		case LENGTH:
		case ABS:
		case SQRT:
		case LONG_LITERAL:
		case FLOAT_LITERAL:
		case DOUBLE_LITERAL:
		{
			expression(_t);
			_t = _retTree;
			astFactory.addASTChild(currentAST, returnAST);
			break;
		}
		case 3:
		{
			break;
		}
		default:
		{
			throw new NoViableAltException(_t);
		}
		}
		}
		currentAST = __currentAST97;
		_t = __t97;
		_t = _t.getNextSibling();
		indexOf_AST = (JQLAST)currentAST.root;
		returnAST = indexOf_AST;
		_retTree = _t;
	}
	
	public final void length(AST _t) throws RecognitionException {
		
		JQLAST length_AST_in = (_t == ASTNULL) ? null : (JQLAST)_t;
		returnAST = null;
		ASTPair currentAST = new ASTPair();
		JQLAST length_AST = null;
		
		AST __t100 = _t;
		JQLAST tmp35_AST = null;
		JQLAST tmp35_AST_in = null;
		tmp35_AST = (JQLAST)astFactory.create((JQLAST)_t);
		tmp35_AST_in = (JQLAST)_t;
		astFactory.addASTChild(currentAST, tmp35_AST);
		ASTPair __currentAST100 = currentAST.copy();
		currentAST.root = currentAST.child;
		currentAST.child = null;
		match(_t,LENGTH);
		_t = _t.getFirstChild();
		expression(_t);
		_t = _retTree;
		astFactory.addASTChild(currentAST, returnAST);
		currentAST = __currentAST100;
		_t = __t100;
		_t = _t.getNextSibling();
		length_AST = (JQLAST)currentAST.root;
		returnAST = length_AST;
		_retTree = _t;
	}
	
	public final void abs(AST _t) throws RecognitionException {
		
		JQLAST abs_AST_in = (_t == ASTNULL) ? null : (JQLAST)_t;
		returnAST = null;
		ASTPair currentAST = new ASTPair();
		JQLAST abs_AST = null;
		
		AST __t102 = _t;
		JQLAST tmp36_AST = null;
		JQLAST tmp36_AST_in = null;
		tmp36_AST = (JQLAST)astFactory.create((JQLAST)_t);
		tmp36_AST_in = (JQLAST)_t;
		astFactory.addASTChild(currentAST, tmp36_AST);
		ASTPair __currentAST102 = currentAST.copy();
		currentAST.root = currentAST.child;
		currentAST.child = null;
		match(_t,ABS);
		_t = _t.getFirstChild();
		expression(_t);
		_t = _retTree;
		astFactory.addASTChild(currentAST, returnAST);
		currentAST = __currentAST102;
		_t = __t102;
		_t = _t.getNextSibling();
		abs_AST = (JQLAST)currentAST.root;
		returnAST = abs_AST;
		_retTree = _t;
	}
	
	public final void sqrt(AST _t) throws RecognitionException {
		
		JQLAST sqrt_AST_in = (_t == ASTNULL) ? null : (JQLAST)_t;
		returnAST = null;
		ASTPair currentAST = new ASTPair();
		JQLAST sqrt_AST = null;
		
		AST __t104 = _t;
		JQLAST tmp37_AST = null;
		JQLAST tmp37_AST_in = null;
		tmp37_AST = (JQLAST)astFactory.create((JQLAST)_t);
		tmp37_AST_in = (JQLAST)_t;
		astFactory.addASTChild(currentAST, tmp37_AST);
		ASTPair __currentAST104 = currentAST.copy();
		currentAST.root = currentAST.child;
		currentAST.child = null;
		match(_t,SQRT);
		_t = _t.getFirstChild();
		expression(_t);
		_t = _retTree;
		astFactory.addASTChild(currentAST, returnAST);
		currentAST = __currentAST104;
		_t = __t104;
		_t = _t.getNextSibling();
		sqrt_AST = (JQLAST)currentAST.root;
		returnAST = sqrt_AST;
		_retTree = _t;
	}
	
	public final Object  literalHelper(AST _t) throws RecognitionException {
		Object value;
		
		JQLAST literalHelper_AST_in = (_t == ASTNULL) ? null : (JQLAST)_t;
		returnAST = null;
		ASTPair currentAST = new ASTPair();
		JQLAST literalHelper_AST = null;
		JQLAST i = null;
		JQLAST i_AST = null;
		JQLAST l = null;
		JQLAST l_AST = null;
		JQLAST f = null;
		JQLAST f_AST = null;
		JQLAST d = null;
		JQLAST d_AST = null;
		JQLAST c = null;
		JQLAST c_AST = null;
		JQLAST s = null;
		JQLAST s_AST = null;
		JQLAST n = null;
		JQLAST n_AST = null;
		
		value = null; 
		
		
		if (_t==null) _t=ASTNULL;
		switch ( _t.getType()) {
		case TRUE:
		{
			JQLAST tmp38_AST = null;
			JQLAST tmp38_AST_in = null;
			tmp38_AST = (JQLAST)astFactory.create((JQLAST)_t);
			tmp38_AST_in = (JQLAST)_t;
			astFactory.addASTChild(currentAST, tmp38_AST);
			match(_t,TRUE);
			_t = _t.getNextSibling();
			if ( inputState.guessing==0 ) {
				value = new Boolean(true);
			}
			literalHelper_AST = (JQLAST)currentAST.root;
			break;
		}
		case FALSE:
		{
			JQLAST tmp39_AST = null;
			JQLAST tmp39_AST_in = null;
			tmp39_AST = (JQLAST)astFactory.create((JQLAST)_t);
			tmp39_AST_in = (JQLAST)_t;
			astFactory.addASTChild(currentAST, tmp39_AST);
			match(_t,FALSE);
			_t = _t.getNextSibling();
			if ( inputState.guessing==0 ) {
				value = new Boolean(false);
			}
			literalHelper_AST = (JQLAST)currentAST.root;
			break;
		}
		case INT_LITERAL:
		{
			i = (JQLAST)_t;
			JQLAST i_AST_in = null;
			i_AST = (JQLAST)astFactory.create(i);
			astFactory.addASTChild(currentAST, i_AST);
			match(_t,INT_LITERAL);
			_t = _t.getNextSibling();
			if ( inputState.guessing==0 ) {
				
				try
				{
				value = Integer.decode(i.getText());
				}
				catch (NumberFormatException ex)
				{
				errorMsg.error(i.getLine(), i.getColumn(),
				I18NHelper.getMessage(messages, 
				"jqlc.optimizer.literal.invalid",  //NOI18N
				i.getJQLType().getName(), i.getText()));
				}
				
			}
			literalHelper_AST = (JQLAST)currentAST.root;
			break;
		}
		case LONG_LITERAL:
		{
			l = (JQLAST)_t;
			JQLAST l_AST_in = null;
			l_AST = (JQLAST)astFactory.create(l);
			astFactory.addASTChild(currentAST, l_AST);
			match(_t,LONG_LITERAL);
			_t = _t.getNextSibling();
			if ( inputState.guessing==0 ) {
				
				String txt = l.getText();
				char last = txt.charAt(txt.length() - 1);
				if ((last == 'l') || (last == 'L'))
				{
				txt = txt.substring(0, txt.length() - 1);
				}
				try
				{
				value = Long.decode(txt);
				}
				catch (NumberFormatException ex)
				{
				errorMsg.error(l.getLine(), l.getColumn(),
				I18NHelper.getMessage(messages, 
				"jqlc.optimizer.literal.invalid",  //NOI18N
				l.getJQLType().getName(), l.getText()));
				}
				
			}
			literalHelper_AST = (JQLAST)currentAST.root;
			break;
		}
		case FLOAT_LITERAL:
		{
			f = (JQLAST)_t;
			JQLAST f_AST_in = null;
			f_AST = (JQLAST)astFactory.create(f);
			astFactory.addASTChild(currentAST, f_AST);
			match(_t,FLOAT_LITERAL);
			_t = _t.getNextSibling();
			if ( inputState.guessing==0 ) {
				
				String txt = f.getText();
				char last = txt.charAt(txt.length() - 1);
				if ((last == 'f') || (last == 'F'))
				{
				txt = txt.substring(0, txt.length() - 1);
				}
				try
				{
				value = new Float(txt);
				}
				catch (NumberFormatException ex)
				{
				errorMsg.error(f.getLine(), f.getColumn(),
				I18NHelper.getMessage(messages, 
				"jqlc.optimizer.literal.invalid",  //NOI18N
				f.getJQLType().getName(), f.getText()));
				}
				
			}
			literalHelper_AST = (JQLAST)currentAST.root;
			break;
		}
		case DOUBLE_LITERAL:
		{
			d = (JQLAST)_t;
			JQLAST d_AST_in = null;
			d_AST = (JQLAST)astFactory.create(d);
			astFactory.addASTChild(currentAST, d_AST);
			match(_t,DOUBLE_LITERAL);
			_t = _t.getNextSibling();
			if ( inputState.guessing==0 ) {
				
				String txt = d.getText();
				char last = txt.charAt(txt.length() - 1);
				if ((last == 'd') || (last == 'd'))
				{
				txt = txt.substring(0, txt.length() - 1);
				}
				try
				{
				value = new Double(txt);
				}
				catch (NumberFormatException ex)
				{
				errorMsg.error(d.getLine(), d.getColumn(),
				I18NHelper.getMessage(messages, 
				"jqlc.optimizer.literal.invalid",  //NOI18N
				d.getJQLType().getName(), d.getText()));
				}
				
			}
			literalHelper_AST = (JQLAST)currentAST.root;
			break;
		}
		case CHAR_LITERAL:
		{
			c = (JQLAST)_t;
			JQLAST c_AST_in = null;
			c_AST = (JQLAST)astFactory.create(c);
			astFactory.addASTChild(currentAST, c_AST);
			match(_t,CHAR_LITERAL);
			_t = _t.getNextSibling();
			if ( inputState.guessing==0 ) {
				value = new Character(parseChar(c.getText()));
			}
			literalHelper_AST = (JQLAST)currentAST.root;
			break;
		}
		case STRING_LITERAL:
		{
			s = (JQLAST)_t;
			JQLAST s_AST_in = null;
			s_AST = (JQLAST)astFactory.create(s);
			astFactory.addASTChild(currentAST, s_AST);
			match(_t,STRING_LITERAL);
			_t = _t.getNextSibling();
			if ( inputState.guessing==0 ) {
				value = s.getText();
			}
			literalHelper_AST = (JQLAST)currentAST.root;
			break;
		}
		case NULL:
		{
			n = (JQLAST)_t;
			JQLAST n_AST_in = null;
			n_AST = (JQLAST)astFactory.create(n);
			astFactory.addASTChild(currentAST, n_AST);
			match(_t,NULL);
			_t = _t.getNextSibling();
			if ( inputState.guessing==0 ) {
				value = null;
			}
			literalHelper_AST = (JQLAST)currentAST.root;
			break;
		}
		default:
		{
			throw new NoViableAltException(_t);
		}
		}
		returnAST = literalHelper_AST;
		_retTree = _t;
		return value;
	}
	
	public final void primitiveType(AST _t) throws RecognitionException {
		
		JQLAST primitiveType_AST_in = (_t == ASTNULL) ? null : (JQLAST)_t;
		returnAST = null;
		ASTPair currentAST = new ASTPair();
		JQLAST primitiveType_AST = null;
		
		if (_t==null) _t=ASTNULL;
		switch ( _t.getType()) {
		case BOOLEAN:
		{
			JQLAST tmp40_AST = null;
			JQLAST tmp40_AST_in = null;
			tmp40_AST = (JQLAST)astFactory.create((JQLAST)_t);
			tmp40_AST_in = (JQLAST)_t;
			astFactory.addASTChild(currentAST, tmp40_AST);
			match(_t,BOOLEAN);
			_t = _t.getNextSibling();
			primitiveType_AST = (JQLAST)currentAST.root;
			break;
		}
		case BYTE:
		{
			JQLAST tmp41_AST = null;
			JQLAST tmp41_AST_in = null;
			tmp41_AST = (JQLAST)astFactory.create((JQLAST)_t);
			tmp41_AST_in = (JQLAST)_t;
			astFactory.addASTChild(currentAST, tmp41_AST);
			match(_t,BYTE);
			_t = _t.getNextSibling();
			primitiveType_AST = (JQLAST)currentAST.root;
			break;
		}
		case CHAR:
		{
			JQLAST tmp42_AST = null;
			JQLAST tmp42_AST_in = null;
			tmp42_AST = (JQLAST)astFactory.create((JQLAST)_t);
			tmp42_AST_in = (JQLAST)_t;
			astFactory.addASTChild(currentAST, tmp42_AST);
			match(_t,CHAR);
			_t = _t.getNextSibling();
			primitiveType_AST = (JQLAST)currentAST.root;
			break;
		}
		case SHORT:
		{
			JQLAST tmp43_AST = null;
			JQLAST tmp43_AST_in = null;
			tmp43_AST = (JQLAST)astFactory.create((JQLAST)_t);
			tmp43_AST_in = (JQLAST)_t;
			astFactory.addASTChild(currentAST, tmp43_AST);
			match(_t,SHORT);
			_t = _t.getNextSibling();
			primitiveType_AST = (JQLAST)currentAST.root;
			break;
		}
		case INT:
		{
			JQLAST tmp44_AST = null;
			JQLAST tmp44_AST_in = null;
			tmp44_AST = (JQLAST)astFactory.create((JQLAST)_t);
			tmp44_AST_in = (JQLAST)_t;
			astFactory.addASTChild(currentAST, tmp44_AST);
			match(_t,INT);
			_t = _t.getNextSibling();
			primitiveType_AST = (JQLAST)currentAST.root;
			break;
		}
		case FLOAT:
		{
			JQLAST tmp45_AST = null;
			JQLAST tmp45_AST_in = null;
			tmp45_AST = (JQLAST)astFactory.create((JQLAST)_t);
			tmp45_AST_in = (JQLAST)_t;
			astFactory.addASTChild(currentAST, tmp45_AST);
			match(_t,FLOAT);
			_t = _t.getNextSibling();
			primitiveType_AST = (JQLAST)currentAST.root;
			break;
		}
		case LONG:
		{
			JQLAST tmp46_AST = null;
			JQLAST tmp46_AST_in = null;
			tmp46_AST = (JQLAST)astFactory.create((JQLAST)_t);
			tmp46_AST_in = (JQLAST)_t;
			astFactory.addASTChild(currentAST, tmp46_AST);
			match(_t,LONG);
			_t = _t.getNextSibling();
			primitiveType_AST = (JQLAST)currentAST.root;
			break;
		}
		case DOUBLE:
		{
			JQLAST tmp47_AST = null;
			JQLAST tmp47_AST_in = null;
			tmp47_AST = (JQLAST)astFactory.create((JQLAST)_t);
			tmp47_AST_in = (JQLAST)_t;
			astFactory.addASTChild(currentAST, tmp47_AST);
			match(_t,DOUBLE);
			_t = _t.getNextSibling();
			primitiveType_AST = (JQLAST)currentAST.root;
			break;
		}
		default:
		{
			throw new NoViableAltException(_t);
		}
		}
		returnAST = primitiveType_AST;
		_retTree = _t;
	}
	
	
	public static final String[] _tokenNames = {
		"<0>",
		"EOF",
		"<2>",
		"NULL_TREE_LOOKAHEAD",
		"\"import\"",
		"\"this\"",
		"\"ascending\"",
		"\"descending\"",
		"\"distinct\"",
		"\"boolean\"",
		"\"byte\"",
		"\"char\"",
		"\"short\"",
		"\"int\"",
		"\"float\"",
		"\"long\"",
		"\"double\"",
		"\"null\"",
		"\"true\"",
		"\"false\"",
		"\"avg\"",
		"\"max\"",
		"\"min\"",
		"\"sum\"",
		"\"count\"",
		"LPAREN",
		"RPAREN",
		"COMMA",
		"EQUAL",
		"LNOT",
		"BNOT",
		"NOT_EQUAL",
		"DIV",
		"PLUS",
		"MINUS",
		"STAR",
		"MOD",
		"GE",
		"GT",
		"LE",
		"LT",
		"BXOR",
		"BOR",
		"OR",
		"BAND",
		"AND",
		"SEMI",
		"WS",
		"NEWLINE",
		"CHAR_LITERAL",
		"STRING_LITERAL",
		"ESC",
		"HEX_DIGIT",
		"INT_LITERAL",
		"EXPONENT",
		"FLOATINGPOINT_SUFFIX",
		"an identifier",
		"UNICODE_ESCAPE",
		"QUERY",
		"CLASS_DEF",
		"IMPORT_DEF",
		"PARAMETER_DEF",
		"VARIABLE_DEF",
		"ORDERING_DEF",
		"FILTER_DEF",
		"ARG_LIST",
		"UNARY_MINUS",
		"UNARY_PLUS",
		"TYPECAST",
		"OBJECT_EQUAL",
		"OBJECT_NOT_EQUAL",
		"COLLECTION_EQUAL",
		"COLLECTION_NOT_EQUAL",
		"CONCAT",
		"FIELD_ACCESS",
		"STATIC_FIELD_ACCESS",
		"CONTAINS",
		"NOT_CONTAINS",
		"NAVIGATION",
		"STARTS_WITH",
		"ENDS_WITH",
		"IS_EMPTY",
		"VARIABLE",
		"PARAMETER",
		"TYPENAME",
		"VALUE",
		"RESULT_DEF",
		"LIKE",
		"SUBSTRING",
		"INDEXOF",
		"LENGTH",
		"ABS",
		"SQRT",
		"NOT_IN",
		"DOT",
		"LONG_LITERAL",
		"FLOAT_LITERAL",
		"DOUBLE_LITERAL"
	};
	
	}
	
