package fluximpl;

import com.rabbitmq.client.Channel;
import com.rabbitmq.client.Connection;
import com.rabbitmq.client.ConnectionFactory;
import com.rabbitmq.client.QueueingConsumer;
import flux.*;
import flux.dev.NotTriggeredException;
import flux.logging.Logger;

import java.io.IOException;
import java.util.*;

/**
 * Implementation for RabbitMQTrigger. Uses "polling" based message consumption in RabbitMQ.
 *
 * @author arul@fluxcorp.com
 */
public class RabbitMQTriggerImpl extends RabbitMQActionImpl implements RabbitMQTrigger {

  private static final String TRIGGER_VARIABLE = "RABBITMQ_TRIGGER_VARIABLE";

  private static final String EXCLUSIVE_TYPE = "EXCLUSIVE";

  public RabbitMQTriggerImpl() {
    super(new FlowChartImpl(), "RabbitMQ Trigger");
  }

  public RabbitMQTriggerImpl(FlowChartImpl fc, String name) {
    super(fc, name);
  }

  @Override
  public Set<String> getHiddenVariableNames() {
    // The returned raw type is a Set of String.
    @SuppressWarnings("unchecked")
    Set<String> set = super.getHiddenVariableNames();
    set.add(TRIGGER_VARIABLE);
    return set;
  }

  public String getHost() {
    return getVariable().host;
  }

  @Override
  public void setHost(String host) {
    RabbitMQTriggerVariable var = getVariable();
    var.host = host;
    putVariable(var);
  }

  public int getPort() {
    return getVariable().port;
  }

  @Override
  public void setPort(int port) {
    RabbitMQTriggerVariable var = getVariable();
    var.port = port;
    putVariable(var);
  }

  public String getVirtualHost() {
    return getVariable().virtualHost;
  }

  @Override
  public void setVirtualHost(String virtualHost) {
    RabbitMQTriggerVariable var = getVariable();
    var.virtualHost = virtualHost;
    putVariable(var);
  }

  public String getPollingDelay() {
    return getVariable().timeExpression;
  }

  @Override
  public void setPollingDelay(String timeExpression) {
    RabbitMQTriggerVariable var = getVariable();
    var.timeExpression = timeExpression;
    putVariable(var);
  }

  public String getUsername() {
    return getVariable().username;
  }

  @Override
  public void setUsername(String username) {
    RabbitMQTriggerVariable var = getVariable();
    var.username = username;
    putVariable(var);
  }

  public String getPassword() {
    Password password = getVariable().password;
    if (password != null) {
      return password.getEncryptedPassword();
    }
    return null;
  }

  @Override
  public void setPassword(String password) {
    RabbitMQTriggerVariable var = getVariable();
    if (password != null) {
      var.password = Password.makePassword(password);
    }
    putVariable(var);
  }

  public String getQueueName() {
    return getVariable().queueName;
  }

  @Override
  public void setQueueName(String queueName) {
    RabbitMQTriggerVariable var = getVariable();
    var.queueName = queueName;
    putVariable(var);
  }

  @Override
  public Date getNextPollingDate() {
    try {
      Factory fluxFactory = Factory.makeInstance();
      EngineHelper engineHelper = fluxFactory.makeEngineHelper();
      return engineHelper.applyTimeExpression(getPollingDelay(), null, null);
    } // try
    catch (EngineException e) {
      // Should not occur.
    } // catch

    return null;
  } // getNextPollingDate()

  @Override
  public Object execute(FlowContext flowContext) throws Exception {
    Logger log = flowContext.getLogger();
    RabbitMQTriggerResult result = new RabbitMQTriggerResult();
    ConnectionFactory factory = new ConnectionFactory();
    factory.setHost(getHost());
    factory.setPort(getPort());
    if (!StringUtil.isNullOrEmpty(getVirtualHost())) {
      factory.setVirtualHost(getVirtualHost());
    }
    factory.setUsername(getUsername());
    factory.setPassword(getPassword());
    Connection conn = null;
    Channel channel = null;
    try {
      conn = factory.newConnection();
      // Use channel to send and receive messages
      channel = conn.createChannel();

      boolean autoAck = false;
      QueueingConsumer consumer = new QueueingConsumer(channel);
      log.info("Consuming message.");
      channel.basicConsume(getQueueName(), autoAck, consumer);
      String message = null;
      QueueingConsumer.Delivery delivery = null;
      try {
        delivery = consumer.nextDelivery();
        message = new String(delivery.getBody());
        log.info("Message " + message  + " consumed successfully.");
      } catch (InterruptedException exception) {
        log.severe("Exception during consumption. Reason : " + exception.getMessage());
      }
      log.info("Acknowledging receipt.");
      if (delivery != null)
        channel.basicAck(delivery.getEnvelope().getDeliveryTag(), false);

      if (StringUtil.isNullOrEmpty(message)) {
        throw new NotTriggeredException();
      }
      result.message = message;
    } catch (IOException exception) {
      log.severe("Exception during consume. Reason : " + exception.getMessage());
    } finally {
      if (channel != null) {
        channel.close();
      }
      if (conn != null) {
        conn.close();
      }
    }
    return result;
  }

  @Override
  public void verify() throws EngineException {
    verifyCommonProperties();
    if (StringUtil.isNullOrEmpty(getPollingDelay())) {
      throw new EngineException("Expected \"Time Expression\" to be non-null or non-empty, but it was null or empty.");
    }
  }

  private RabbitMQTriggerVariable getVariable() {
    if (!getVariableManager().contains(TRIGGER_VARIABLE)) {
      getVariableManager().put(TRIGGER_VARIABLE, new RabbitMQTriggerVariable());
    }
    return (RabbitMQTriggerVariable) getVariableManager().get(TRIGGER_VARIABLE);
  }

  private void putVariable(RabbitMQTriggerVariable variable) {
    getVariableManager().put(TRIGGER_VARIABLE, variable);
  }
}
