package fo.nya.leaderboard;

import fo.nya.leaderboard.implementation.SkipListLeaderboard;

import java.util.List;
import java.util.Map;
import java.util.Optional;

/**
 * Created by 0da on 08.06.2023 20:45; (ﾉ◕ヮ◕)ﾉ*:･ﾟ✧
 *
 * <br>
 * Main interface of that "library". Describes required methods for any implementation of leaderboard.
 */
public interface Leaderboard {

    /**
     * Method returns current amounts of entries un that leaderboard.
     *
     * @return current amounts of entries un that leaderboard.
     */
    int size();

    /**
     * Method calculates and returns slice of current situation in leaderboard.
     *
     * @param from  place that will be start of slice inclusive. Minimal meaningful value is {@code 1}, because minimal value of {@link LeaderboardRow#place()} is 1.
     * @param limit maximum amount of entries in result.
     * @param desc  this flags determinants 'direction' of a slice. If this value is {@code true} then result will contain values in descending order {@code [from, f - 1, f - 2, ...]}, else {@code [f, f + 1, from + 2, ...]}.
     * @return new list with result for specified arguments, maximum length is {@code limit} and first value (if not empty) with place {@code from}.
     */
    List<LeaderboardRow> slice(int from, int limit, boolean desc);

    /**
     * Method calculates and returns slice of current situation in leaderboard. Shorthand of method {@link #slice(int, int, boolean)}, alternative code:
     * <pre>{@code
     *  var slice = leaderboard.slice(from, limit, false);
     * }</pre>
     *
     * @param from  place that will be start of slice inclusive. Minimal meaningful value is {@code 1}, because minimal value of {@link LeaderboardRow#place()} is 1.
     * @param limit maximum amount of entries in result.
     * @return new list with result for specified arguments, maximum length is {@code limit} and first value (if not empty) with place {@code from}.
     * Order of returned values is {@code [from, from + 1, from + 2, ...]}.
     * @see #slice(int, int, boolean)
     */
    default List<LeaderboardRow> slice(int from, int limit) {
        return slice(from, limit, false);
    }

    /**
     * Method finds entry row of specified target.
     *
     * @param target id of target that you need to acquire.
     * @return valid row of a target if that target is present into that leaderboard.
     */
    Optional<LeaderboardRow> find(long target);

    /**
     * Method takes slice around position of a specific target.
     *
     * @param target id of the target around which slice will be made.
     * @param up     amount of entries that will be taken before target.
     * @param down   amount of entries that will be taken after target.
     * @param desc   this flags determinants 'direction' of a slice.
     *               If this value is {@code true} then result will contain values in descending order {@code [..., t + 1, t + 2, target, t - 1, t - 2, ...]},
     *               else {@code [..., t - 1, t - 2, target, t + 1, t + 2, ...]}.
     * @return new list with result for specified arguments.
     */
    List<LeaderboardRow> around(long target, int up, int down, boolean desc);

    /**
     * Method to update (or insert new) score for specified target in that leaderboard.
     *
     * @param target id of a target to update.
     * @param score  new score to associate with that target.
     * @see #updateAll(Map)
     */
    void update(long target, long score);

    /**
     * Method for updating multiple entries at once. Default implementation:
     * <pre>{@code
     * for (var entry : entries.entrySet()) {
     *     update(entry.getKey(), entry.getValue());
     * }
     * }</pre>
     *
     * @param entries map of entries where key is target and value - new score.
     * @see #update(long, long)
     */
    default void updateAll(Map<Long, Long> entries) {
        for (var entry : entries.entrySet()) {
            update(entry.getKey(), entry.getValue());
        }
    }

    /**
     * Method removes entry with specified target from leaderboard.  All entries after removed must adjust their positions.
     *
     * @param target target which must be removed.
     */
    void remove(long target);

    /**
     * Method returns current best implementation of leaderboard for average usage.
     *
     * @return current best implementation of leaderboard for average usage. At this moment its {@link SkipListLeaderboard}.
     */
    static Leaderboard create() {
        return new SkipListLeaderboard();
    }
}
