package fo.nya.leaderboard.implementation;

import fo.nya.leaderboard.Leaderboard;
import fo.nya.leaderboard.LeaderboardRow;

import java.util.*;
import java.util.concurrent.locks.ReentrantReadWriteLock;

/**
 * Created by 0da on 09.06.2023 12:20; (ﾉ◕ヮ◕)ﾉ*:･ﾟ✧
 * {@inheritDoc}
 * <br>
 * Attempt to create rating over array, but it turns out expansively slow because on insertion in random place you need to shift all previous values.
 * <br>
 * There a some numbers for that implementation. 1 million entries takes about 150mb of RAM. Relative measure of performance in seconds.
 * <br>
 * Insert of new elements with different score.
 * <pre>
 * |             | random | always at end | always at start |
 * |-------------|--------|---------------|-----------------|
 * | 0 - 100k    | 1.11   | 0.44          | 1.87            |
 * | 100k - 200k | 4.22   | 1.31          | 6.76            |
 * | 200k - 300k | 9.7    | 4.88          | 13.11           |
 * | 300k - 400k | 9.61   | 1.97          | 14.12           |
 * | 400k - 500k | 24.78  | 11.97         | 28.29           |
 * | 500k - 600k | 21.8   | 8.66          | 29.5            |
 * | 600k - 700k | 19.78  | 5.4           | 37.92           |
 * | 700k - 800k | 24.68  | 5.74          | 37.18           |
 * | 800k - 900k | 67.4   | 29.7          | 89.14           |
 * | 900k - 1m   | 65.38  | 23.55         | 91.11           |
 * </pre>
 * <br>
 * Numbers for 100k manipulation:
 * <pre>
 * | random | worst case, update first element that it becomes last | good case, update changes position to +- 10 positions |
 * |--------|-------------------------------------------------------|-------------------------------------------------------|
 * | 56.84  | 106.58                                                | 0.07                                                  |
 * </pre>
 */
class ArrayLeaderboard implements Leaderboard {

    private final static int ENTRY_SIZE = 2;

    private static final int TARGET_OFFSET = 0;
    private static final int SCORE_OFFSET = 1;

    private final ReentrantReadWriteLock lock = new ReentrantReadWriteLock();

    private final HashMap<Long, Long> scores = new HashMap<>();

    // exposed for testing
    long[] data = new long[3];

    private int size = 0;

    @Override public int size() {
        return size;
    }

    @Override public List<LeaderboardRow> slice(int from, int limit, boolean desc) {

        LinkedList<LeaderboardRow> result = new LinkedList<>();

        var lock = this.lock.readLock();
        lock.lock();
        try {
            int f = Math.max(from - 1, 0);

            int dir = desc ? -1 : +1;

            for (int i = 0; Math.abs(i) < limit && (f + i) < size && (f + i) >= 0; i += dir) {
                int place = f + i;
                result.add(new LeaderboardRow(place + 1, data[place * ENTRY_SIZE + TARGET_OFFSET], data[place * ENTRY_SIZE + SCORE_OFFSET]));
            }

        } finally {
            lock.unlock();
        }

        return result;
    }

    @Override public Optional<LeaderboardRow> find(long target) {

        var lock = this.lock.readLock();
        lock.lock();
        try {

            int place = get(target);

            if (place == -1) return Optional.empty();

            return Optional.of(new LeaderboardRow(place + 1, target, data[place * ENTRY_SIZE + SCORE_OFFSET]));

        } finally {
            lock.unlock();
        }
    }

    @Override public void update(long target, long score) {

        var lock = this.lock.writeLock();
        lock.lock();
        try {
            int place = get(target);

            if (place < 0) place = insert(target, score);

            scores.put(target, score);

            data[place * ENTRY_SIZE + SCORE_OFFSET] = score;

            int pointer = place;

            while (true) {
                if (pointer > 0 && data[pointer * ENTRY_SIZE + SCORE_OFFSET] > data[(pointer - 1) * ENTRY_SIZE + SCORE_OFFSET]) {
                    swap(pointer * ENTRY_SIZE + TARGET_OFFSET, (pointer - 1) * ENTRY_SIZE + TARGET_OFFSET);
                    swap(pointer * ENTRY_SIZE + SCORE_OFFSET, (pointer - 1) * ENTRY_SIZE + SCORE_OFFSET);

                    pointer--;

                    continue;
                }

                if (pointer < size - 1 && data[pointer * ENTRY_SIZE + SCORE_OFFSET] <= data[(pointer + 1) * ENTRY_SIZE + SCORE_OFFSET]) {
                    swap(pointer * ENTRY_SIZE + TARGET_OFFSET, (pointer + 1) * ENTRY_SIZE + TARGET_OFFSET);
                    swap(pointer * ENTRY_SIZE + SCORE_OFFSET, (pointer + 1) * ENTRY_SIZE + SCORE_OFFSET);

                    pointer++;

                    continue;
                }

                break;
            }

        } finally {
            lock.unlock();
        }
    }

    @Override public void remove(long target) {
        throw new AbstractMethodError();
    }

    @Override public List<LeaderboardRow> around(long target, int up, int down, boolean desc) {
        throw new AbstractMethodError();
    }

    public int get(long key) {

        Long score = scores.get(key);

        if (score == null) return -1;

        int place = findScorePlace(score);

        do {
            place--;
        } while (data[place * ENTRY_SIZE + TARGET_OFFSET] != key);

        return place;
    }


    private int insert(long target, long score) {

        int place = findScorePlace(score);

        resize();

        System.arraycopy(data, place * ENTRY_SIZE, data, place * ENTRY_SIZE + ENTRY_SIZE, data.length - place * ENTRY_SIZE - ENTRY_SIZE);

        data[place * ENTRY_SIZE + TARGET_OFFSET] = target;

        size++;

        return place;
    }

    private void resize() {
        int required = (size * ENTRY_SIZE + ENTRY_SIZE) - data.length;
        if (required > 0) {

            if (data.length == Integer.MAX_VALUE - 8) {
                throw new IllegalStateException("This leaderboard is already contains maximum amount of entries");
            }
            int newSize = Math.max(required, data.length) + data.length;
            data = Arrays.copyOf(data, Math.min(newSize, Integer.MAX_VALUE - 8));
        }
    }

    private int findScorePlace(long score) {

        double s = score - 0.5d;

        int low = 0;
        int high = size - 1;
        while (low <= high) {
            int mid = (low + high) >>> 1;
            long midVal = data[mid * ENTRY_SIZE + SCORE_OFFSET];

            if (midVal < s) {
                high = mid - 1;
            } else {
                low = mid + 1;
            }
        }

        return low;
    }

    private void swap(int a, int b) {
        long tmp = data[a];
        data[a] = data[b];
        data[b] = tmp;
    }
}
