package fo.nya.leaderboard.implementation;

import fo.nya.leaderboard.Leaderboard;
import fo.nya.leaderboard.LeaderboardRow;

import java.util.*;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentSkipListSet;
import java.util.concurrent.atomic.AtomicLong;

/**
 * Created by 0da on 12.06.2023 15:20; (ﾉ◕ヮ◕)ﾉ*:･ﾟ✧
 * <br>
 * Implementation of leaderboard over {@link ConcurrentSkipListSet} and {@link ConcurrentHashMap} for specific entry lookup.
 * Insertion or update at average requires same time as inserting in HashMap + remove from SkipListSet + insertion in SkipListSet.
 */
public class SkipListLeaderboard implements Leaderboard {

    /**
     * Utility entry for sorting trough score and update order.
     */
    private record Entry(long target, long score, long order) implements Comparable<Entry> {

        /**
         * comparator to compare these entries. First its compares {@link #score()} from in descending order then compares {@link #order()}
         */
        public static final Comparator<Entry> COMPARATOR = Comparator
                .comparingLong(Entry::score)
                .reversed()
                .thenComparing(Entry::order);

        @Override public int compareTo(Entry o) {
            return COMPARATOR.compare(this, o);
        }
    }

    /**
     * Incremental value for {@link Entry#order()}, every value must be uniq or else it may collide with other entry in {@link #data} collection.
     */
    private final AtomicLong order = new AtomicLong();

    /**
     * Lookup table for fast access to a specific entry.
     */
    private final ConcurrentHashMap<Long, Entry> lookup = new ConcurrentHashMap<>();

    /**
     * Sorted set to maintain a 'positions' of entries.
     */
    private final ConcurrentSkipListSet<Entry> data = new ConcurrentSkipListSet<>();

    /**
     * {@inheritDoc}
     */
    @Override public int size() {
        return data.size();
    }

    /**
     * {@inheritDoc}
     */
    @Override public List<LeaderboardRow> slice(int from, int limit, boolean desc) {
        LinkedList<LeaderboardRow> results = new LinkedList<>();

        from = Math.max(from - 1, 0);

        int f = from;

        if (limit == 0 || f > size() || size() == 0) {
            return results;
        }

        Entry target = null;

        for (Entry next : data) {
            if (f <= 0) {
                target = next;
                break;
            } else {
                f--;
            }
        }

        // practically impossible
        if (target == null) {
            return results;
        }

        int p = from + 1;
        int dir = desc ? -1 : +1;

        for (Entry entry : desc
                ? data.headSet(target, true).descendingSet()
                : data.tailSet(target, true)
        ) {
            if (limit-- <= 0) break;
            results.add(new LeaderboardRow(p, entry.target(), entry.score()));
            p += dir;
        }

        return results;
    }

    /**
     * {@inheritDoc}
     */
    @Override public List<LeaderboardRow> around(long target, int up, int down, boolean desc) {
        LeaderboardRow row = find(target).orElse(null);
        Entry entry = lookup.get(target);

        LinkedList<LeaderboardRow> result = new LinkedList<>();

        if (row == null || entry == null) return result;

        result.add(row);

        NavigableSet<Entry> upper;
        NavigableSet<Entry> lower;

        if (desc) {
            upper = data.tailSet(entry, false);
            lower = data.headSet(entry, false).descendingSet();
        } else {
            upper = data.headSet(entry, false).descendingSet();
            lower = data.tailSet(entry, false);
        }

        int dir = desc ? +1 : -1;

        int u = 0;
        for (Entry e : upper) {
            if (u >= up) break;
            u++;
            result.addFirst(new LeaderboardRow(row.place() + u * dir, e.target(), e.score()));
        }

        int d = 0;
        for (Entry e : lower) {
            if (d >= down) break;
            d++;
            result.addLast(new LeaderboardRow(row.place() - d * dir, e.target(), e.score()));
        }

        return result;
    }

    /**
     * {@inheritDoc}
     */
    @Override public Optional<LeaderboardRow> find(long target) {
        return Optional.ofNullable(lookup.get(target))
                .map(e -> new LeaderboardRow(data.headSet(e).size() + 1, e.target(), e.score()));
    }

    /**
     * {@inheritDoc}
     */
    @Override public void update(long target, long score) {
        lookup.compute(target, (__, e) -> {
            if (e != null) {
                data.remove(e);
            }

            e = new Entry(target, score, order.incrementAndGet());
            data.add(e);
            return e;
        });
    }

    /**
     * {@inheritDoc}
     */
    @Override public void remove(long target) {
        lookup.compute(target, (__, e) -> {
            if (e != null) {
                data.remove(e);
            }
            return null;
        });
    }
}
