package fr.avianey.appratedialog;

import android.content.Context;
import android.content.DialogInterface;
import android.content.DialogInterface.OnCancelListener;
import android.content.DialogInterface.OnClickListener;
import android.content.Intent;
import android.content.SharedPreferences;
import android.content.SharedPreferences.Editor;
import android.content.pm.PackageInfo;
import android.content.pm.PackageManager;
import android.net.Uri;
import android.os.Build;
import android.support.annotation.StringRes;
import android.support.v7.app.AlertDialog;
import android.text.TextUtils;

import java.lang.ref.WeakReference;
import java.util.Date;

public class AppRateDialog {

    private static final String TAG = AppRateDialog.class.getSimpleName();
    //
    public static final String PREF_NAME = TAG;
    //
    private static final String KEY_INSTALL_DATE = "ard_install_date";
    private static final String KEY_LAUNCH_TIMES = "ard_launch_times";
    private static final String KEY_OPT_OUT = "ard_opt_out";
    private static final String KEY_ASK_LATER_DATE = "ard_ask_later_date";

    private static Date installDate = new Date();
    private static int launchTimes = 0;
    private static boolean optOut = false;
    private static Date askLaterDate = new Date();

    private static Config config = new Config();
    private static Callback callback = null;

    // Weak ref to avoid leaking the context
    private static WeakReference<AlertDialog> dialogRef = null;

    /**
     * Initialize AppRateDialog configuration.
     * @param config Configuration object.
     */
    public static void init(Config config) {
        AppRateDialog.config = config;
    }

    /**
     * Set callback instance.
     * The callback will receive yes/no/later events.
     * @param callback
     */
    public static void setCallback(Callback callback) {
        AppRateDialog.callback = callback;
    }

    /**
     * Call this API when the launcher activity is launched.<br>
     * It is better to call this API in onStart() of the launcher activity.
     * @param context Context
     */
    public static void onStart(Context context) {
        SharedPreferences pref = context.getSharedPreferences(PREF_NAME, Context.MODE_PRIVATE);
        Editor editor = pref.edit();
        // If it is the first launch, save the date in shared preference.
        if (pref.getLong(KEY_INSTALL_DATE, 0) == 0L) {
            storeInstallDate(context, editor);
        }
        // Increment launch times
        final int launchTimes = pref.getInt(KEY_LAUNCH_TIMES, 0);
        editor.putInt(KEY_LAUNCH_TIMES, launchTimes + 1);
        editor.commit();

        installDate = new Date(pref.getLong(KEY_INSTALL_DATE, 0));
        AppRateDialog.launchTimes = launchTimes;
        optOut = pref.getBoolean(KEY_OPT_OUT, false);
        askLaterDate = new Date(pref.getLong(KEY_ASK_LATER_DATE, 0));
    }

    /**
     * Show the rate dialog if the criteria is satisfied.
     * @param context Context
     * @return true if shown, false otherwise.
     */
    public static boolean showRateDialogIfNeeded(final Context context) {
        if (shouldShowRateDialog()) {
            showRateDialog(context);
            return true;
        } else {
            return false;
        }
    }

    /**
     * Show the rate dialog if the criteria is satisfied.
     * @param context Context
     * @param themeId Theme ID
     * @return true if shown, false otherwise.
     */
    public static boolean showRateDialogIfNeeded(final Context context, int themeId) {
        if (shouldShowRateDialog()) {
            showRateDialog(context, themeId);
            return true;
        } else {
            return false;
        }
    }

    /**
     * Check whether the rate dialog should be shown or not.
     * Developers may call this method directly if they want to show their own view instead of
     * dialog provided by this library.
     * @return
     */
    public static boolean shouldShowRateDialog() {
        if (optOut) {
            return false;
        } else {
            if (config.mShowAtFirstLaunch && launchTimes == 0) {
                return true;
            }
            if (launchTimes >= config.mCriteriaLaunchTimes) {
                return true;
            }
            long threshold = config.mCriteriaInstallDays * 24 * 60 * 60 * 1000L;	// msec
            if (new Date().getTime() - installDate.getTime() >= threshold &&
                new Date().getTime() - askLaterDate.getTime() >= threshold) {
                return true;
            }
            return false;
        }
    }

    /**
     * Show the rate dialog
     * @param context
     */
    public static void showRateDialog(final Context context) {
        AlertDialog.Builder builder = new AlertDialog.Builder(context);
        showRateDialog(context, builder);
    }

    /**
     * Show the rate dialog
     * @param context
     * @param themeId
     */
    public static void showRateDialog(final Context context, int themeId) {
        AlertDialog.Builder builder = new AlertDialog.Builder(context, themeId);
        showRateDialog(context, builder);
    }

    /**
     * Stop showing the rate dialog
     * @param context
     */
    public static void stopRateDialog(final Context context){
        setOptOut(context, true);
    }

    private static void showRateDialog(final Context context, AlertDialog.Builder builder) {
        if (dialogRef != null && dialogRef.get() != null) {
            // Dialog is already present
            return;
        }

        int titleId = config.mTitleId != 0 ? config.mTitleId : R.string.ard_dialog_title;
        int messageId = config.mMessageId != 0 ? config.mMessageId : R.string.ard_dialog_message;
        int cancelButtonID = config.mCancelButton != 0 ? config.mCancelButton : R.string.ard_dialog_cancel;
        int thanksButtonID = config.mNoButtonId != 0 ? config.mNoButtonId : R.string.ard_dialog_no;
        int rateButtonID = config.mYesButtonId != 0 ? config.mYesButtonId : R.string.ard_dialog_ok;
        builder.setTitle(titleId);
        builder.setMessage(messageId);
        builder.setCancelable(config.mCancelable);
        builder.setPositiveButton(rateButtonID, new OnClickListener() {
            @Override
            public void onClick(DialogInterface dialog, int which) {
                if (callback != null) {
                    callback.onYesClicked();
                }
                String appPackage = context.getPackageName();
                String url = "https://play.google.com/store/apps/details?id=" + appPackage;
                if (!TextUtils.isEmpty(config.mUrl)) {
                    url = config.mUrl;
                }
                Intent intent = new Intent(Intent.ACTION_VIEW, Uri.parse(url));
                context.startActivity(intent);
                setOptOut(context, true);
            }
        });
        builder.setNeutralButton(cancelButtonID, new OnClickListener() {
            @Override
            public void onClick(DialogInterface dialog, int which) {
                if (callback != null) {
                    callback.onCancelClicked();
                }
                clearSharedPreferences(context);
                storeAskLaterDate(context);
            }
        });
        builder.setNegativeButton(thanksButtonID, new OnClickListener() {
            @Override
            public void onClick(DialogInterface dialog, int which) {
                if (callback != null) {
                    callback.onNoClicked();
                }
                setOptOut(context, true);
            }
        });
        builder.setOnCancelListener(new OnCancelListener() {
            @Override
            public void onCancel(DialogInterface dialog) {
                if (callback != null) {
                    callback.onCancelClicked();
                }
                clearSharedPreferences(context);
                storeAskLaterDate(context);
            }
        });
        builder.setOnDismissListener(new DialogInterface.OnDismissListener() {
            @Override
            public void onDismiss(DialogInterface dialog) {
                dialogRef.clear();
            }
        });
        dialogRef = new WeakReference<>(builder.show());
    }

    /**
     * Clear data in shared preferences.<br>
     * This API is called when the rate dialog is approved or canceled.
     * @param context
     */
    private static void clearSharedPreferences(Context context) {
        SharedPreferences pref = context.getSharedPreferences(PREF_NAME, Context.MODE_PRIVATE);
        Editor editor = pref.edit();
        editor.remove(KEY_INSTALL_DATE);
        editor.remove(KEY_LAUNCH_TIMES);
        editor.commit();
    }

    /**
     * Set opt out flag. If it is true, the rate dialog will never shown unless app data is cleared.
     * @param context
     * @param optOut
     */
    private static void setOptOut(final Context context, boolean optOut) {
        SharedPreferences pref = context.getSharedPreferences(PREF_NAME, Context.MODE_PRIVATE);
        Editor editor = pref.edit();
        editor.putBoolean(KEY_OPT_OUT, optOut);
        editor.commit();
        AppRateDialog.optOut = optOut;
    }

    /**
     * Store install date.
     * Install date is retrieved from package manager if possible.
     * @param context
     * @param editor
     */
    private static void storeInstallDate(final Context context, Editor editor) {
        Date installDate = new Date();
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.GINGERBREAD) {
            PackageManager packMan = context.getPackageManager();
            try {
                PackageInfo pkgInfo = packMan.getPackageInfo(context.getPackageName(), 0);
                installDate = new Date(pkgInfo.firstInstallTime);
            } catch (PackageManager.NameNotFoundException e) {
                e.printStackTrace();
            }
        }
        editor.putLong(KEY_INSTALL_DATE, installDate.getTime());
    }

    /**
     * Store the date the user asked for being asked again later.
     * @param context
     */
    private static void storeAskLaterDate(final Context context) {
        SharedPreferences pref = context.getSharedPreferences(PREF_NAME, Context.MODE_PRIVATE);
        Editor editor = pref.edit();
        editor.putLong(KEY_ASK_LATER_DATE, System.currentTimeMillis());
        editor.commit();
    }

    /**
     * AppRateDialog configuration.
     */
    public static class Config {
        private String mUrl = null;
        private int mCriteriaInstallDays;
        private int mCriteriaLaunchTimes;
        private int mTitleId = 0;
        private int mMessageId = 0;
        private int mYesButtonId = 0;
        private int mNoButtonId = 0;
        private int mCancelButton = 0;
        private boolean mCancelable = true;

        private boolean mShowAtFirstLaunch = false;

        /**
         * Constructor with default criteria.
         */
        public Config() {
            this(7, 10);
        }

        /**
         * Constructor.
         * @param criteriaInstallDays
         * @param criteriaLaunchTimes
         */
        public Config(int criteriaInstallDays, int criteriaLaunchTimes) {
            this.mCriteriaInstallDays = criteriaInstallDays;
            this.mCriteriaLaunchTimes = criteriaLaunchTimes;
        }

        /**
         * Set title string ID.
         * @param stringId
         */
        public void setTitle(@StringRes int stringId) {
            this.mTitleId = stringId;
        }

        /**
         * Set message string ID.
         * @param stringId
         */
        public void setMessage(@StringRes int stringId) {
            this.mMessageId = stringId;
        }
        
        /**
         * Set rate now string ID.
         * @param stringId
         */
        public void setYesButtonText(@StringRes int stringId) {
            this.mYesButtonId = stringId;
        }
        
        /**
         * Set no thanks string ID.
         * @param stringId
         */
        public void setNoButtonText(@StringRes int stringId) {
            this.mNoButtonId = stringId;
        }
        
        /**
         * Set cancel string ID.
         * @param stringId
         */
        public void setCancelButtonText(@StringRes int stringId) {
            this.mCancelButton = stringId;
        }

        /**
         * Set navigation url when user clicks rate button.
         * Typically, url will be https://play.google.com/store/apps/details?id=PACKAGE_NAME for Google Play.
         * @param url
         */
        public void setUrl(String url) {
            this.mUrl = url;
        }

        public void setShowAtFirstLaunch(boolean showAtFirstLaunch) {
            this.mShowAtFirstLaunch = showAtFirstLaunch;
        }

        public void setCancelable(boolean cancelable) {
            this.mCancelable = cancelable;
        }
    }

    /**
     * Callback of dialog click event
     */
    public interface Callback {
        /**
         * "Rate now" event
         */
        void onYesClicked();

        /**
         * "No, thanks" event
         */
        void onNoClicked();

        /**
         * "Later" event
         */
        void onCancelClicked();
    }
}
