package fr.avianey.appratedialog;

import android.content.Context;
import android.content.DialogInterface;
import android.content.Intent;
import android.content.SharedPreferences;
import android.content.SharedPreferences.Editor;
import android.content.pm.PackageInfo;
import android.content.pm.PackageManager;
import android.net.Uri;
import android.support.v7.app.AlertDialog;
import android.text.TextUtils;
import android.view.LayoutInflater;
import android.view.View;
import android.widget.EditText;
import android.widget.RatingBar;
import android.widget.TextView;

import java.lang.ref.WeakReference;
import java.util.Date;

import static android.content.Context.MODE_PRIVATE;
import static android.content.DialogInterface.BUTTON_POSITIVE;

public final class AppRateDialog {

    private AppRateDialog() {}

    private static final String TAG = AppRateDialog.class.getSimpleName();
    //
    public static final String PREF_NAME = TAG;
    //
    private static final String KEY_INSTALL_DATE = "ard_install_date";
    private static final String KEY_LAUNCH_TIMES = "ard_launch_times";
    private static final String KEY_SHOWN_COUNT = "ard_shown_count";
    private static final String KEY_OPT_OUT = "ard_opt_out";
    private static final String KEY_ASK_LATER_DATE = "ard_ask_later_date";

    private static Date installDate = new Date();
    private static Date askLaterDate = new Date();
    private static int launchTimes = 0;
    private static int shownCount = 0;
    private static boolean optOut = false;
    private static AppRateDialogConfig config = new AppRateDialogConfig();
    private static Callback callback = null;

    // Weak ref to avoid leaking the context
    private static WeakReference<AlertDialog> dialogRef = null;

    /**
     * Initialize AppRateDialog configuration.
     * @param config Configuration object.
     */
    public static void init(AppRateDialogConfig config) {
        AppRateDialog.config = config;
    }

    /**
     * Set callback instance.
     * The callback will receive yes/no/later events.
     * @param callback
     */
    public static void setCallback(Callback callback) {
        AppRateDialog.callback = callback;
    }

    /**
     * Call this API when the launcher activity is launched.<br>
     * It is better to call this API in onStart() of the launcher activity.
     * @param context Context
     */
    public static void onStart(Context context) {
        SharedPreferences pref = context.getSharedPreferences(PREF_NAME, MODE_PRIVATE);
        Editor editor = pref.edit();
        // If it is the first launch, save the date in shared preference.
        if (pref.getLong(KEY_INSTALL_DATE, 0) == 0L) {
            storeInstallDate(context, editor);
        }
        shownCount = pref.getInt(KEY_SHOWN_COUNT, 0);
        // Increment launch times
        final int launchTimes = pref.getInt(KEY_LAUNCH_TIMES, 0);
        editor.putInt(KEY_LAUNCH_TIMES, launchTimes + 1);
        editor.apply();

        installDate = new Date(pref.getLong(KEY_INSTALL_DATE, 0));
        AppRateDialog.launchTimes = launchTimes;
        optOut = pref.getBoolean(KEY_OPT_OUT, false);
        askLaterDate = new Date(pref.getLong(KEY_ASK_LATER_DATE, 0));
    }

    /**
     * Show the rate dialog if the criteria is satisfied.
     * @param context Context
     * @return true if shown, false otherwise.
     */
    public static boolean showRateDialogIfNeeded(final Context context) {
        if (shouldShowRateDialog()) {
            showRateDialog(context);
            return true;
        } else {
            return false;
        }
    }

    /**
     * Show the rate dialog if the criteria is satisfied.
     * @param context Context
     * @param themeId Theme ID
     * @return true if shown, false otherwise.
     */
    public static boolean showRateDialogIfNeeded(final Context context, int themeId) {
        if (shouldShowRateDialog()) {
            showRateDialog(context, themeId);
            return true;
        } else {
            return false;
        }
    }

    /**
     * Check whether the rate dialog should be shown or not.
     * Developers may call this method directly if they want to show their own view instead of
     * dialog provided by this library.
     * @return
     */
    public static boolean shouldShowRateDialog() {
        if (optOut) {
            return false;
        } else {
            if (config.showAtFirstLaunch && shownCount == 0) {
                return true;
            }
            if (launchTimes >= config.criteriaLaunchTimes) {
                return true;
            }
            long threshold = config.criteriaInstallDays * 24 * 60 * 60 * 1000L;	// msec
            if (new Date().getTime() - installDate.getTime() >= threshold &&
                new Date().getTime() - askLaterDate.getTime() >= threshold) {
                return true;
            }
            return false;
        }
    }

    /**
     * Show the rate dialog
     * @param context
     */
    public static void showRateDialog(final Context context) {
        AlertDialog.Builder builder = new AlertDialog.Builder(context);
        showRateDialog(context, builder);
    }

    /**
     * Show the rate dialog
     * @param context
     * @param themeId
     */
    public static void showRateDialog(final Context context, int themeId) {
        AlertDialog.Builder builder = new AlertDialog.Builder(context, themeId);
        showRateDialog(context, builder);
    }

    /**
     * Stop showing the rate dialog
     * @param context
     */
    public static void stopRateDialog(final Context context){
        setOptOut(context, true);
    }

    private static void showRateDialog(final Context context, AlertDialog.Builder builder) {
        if (dialogRef != null && dialogRef.get() != null) {
            // Dialog is already present
            return;
        }
        //
        final int titleId = config.titleId != 0 ? config.titleId : R.string.ard_dialog_title;
        final int messageId = config.messageId != 0 ? config.messageId : R.string.ard_dialog_message;
        final int cancelButtonID = config.cancelButton != 0 ? config.cancelButton : R.string.ard_dialog_cancel;
        final int thanksButtonID = config.noButtonId != 0 ? config.noButtonId : R.string.ard_dialog_no;
        final int rateButtonID = config.yesButtonId != 0 ? config.yesButtonId : R.string.ard_dialog_ok;
        builder.setTitle(titleId);
        if (config.useStarRating) {
            View ratingContent = LayoutInflater.from(builder.getContext()).inflate(R.layout.dialog_star_rating, null);
            ((TextView) ratingContent.findViewById(android.R.id.text1)).setText(messageId);
            builder.setView(ratingContent);
            RatingBar ratingBar = ratingContent.findViewById(android.R.id.input);
            ratingBar.setOnRatingBarChangeListener((rb, r, b) -> {
                if (dialogRef != null && dialogRef.get() != null) {
                    dialogRef.get().getButton(BUTTON_POSITIVE).setEnabled(true);
                }
            });
            builder.setPositiveButton(rateButtonID, (dialog, which) -> {
                if (ratingBar.getRating() >= config.starThreshold) {
                    // redirects to rate app on Google Play
                    rate(context);
                } else {
                    // display feedback dialog
                    // with cancel and no thanks entries
                    final int feedbackTitleId = config.feedbackTitleId != 0 ? config.feedbackTitleId : R.string.ard_dialog_feedback_title;
                    final int feedbackMessageId = config.feedbackMessageId != 0 ? config.feedbackMessageId : R.string.ard_dialog_feedback_message;
                    final int feedbackSendButtonId = config.feedbackSendButtonId != 0 ? config.feedbackSendButtonId : R.string.ard_dialog_feedback_send;
                    View feedbackContent = LayoutInflater.from(builder.getContext()).inflate(R.layout.dialog_feedback, null);
                    ((TextView) feedbackContent.findViewById(android.R.id.text1)).setText(feedbackMessageId);
                    new AlertDialog.Builder(builder.getContext())
                            .setTitle(feedbackTitleId)
                            .setView(feedbackContent)
                            .setPositiveButton(feedbackSendButtonId, (dialogInterface, i) -> {
                                if (callback != null) {
                                    EditText input = feedbackContent.findViewById(android.R.id.input);
                                    callback.onFeedback(input.getText().toString());
                                }
                                revoke(context);
                            })
                            .setNeutralButton(cancelButtonID, (dialogInterface, i) -> dismiss(context))
                            .setNegativeButton(thanksButtonID, (dialogInterface, i) -> revoke(context))
                            .setCancelable(config.cancelable)
                            .setOnCancelListener(dialogInterface -> dismiss(context))
                            .show();

                }
            });
        } else {
            builder.setMessage(messageId);
            // redirects if user wants to rate app
            builder.setPositiveButton(rateButtonID, (dialog, which) -> rate(context));
        }
        builder.setCancelable(config.cancelable);
        builder.setNeutralButton(cancelButtonID, (dialog, which) -> dismiss(context));
        builder.setNegativeButton(thanksButtonID, (dialog, which) -> revoke(context));
        builder.setOnCancelListener(dialog -> dismiss(context));
        builder.setOnDismissListener(dialog -> dialogRef.clear());
        dialogRef = new WeakReference<>(builder.show());
        shownCount++;
        if (config.useStarRating) {
            if (dialogRef != null && dialogRef.get() != null) {
                dialogRef.get().getButton(BUTTON_POSITIVE).setEnabled(false);
            }
        }
        context.getSharedPreferences(PREF_NAME, MODE_PRIVATE)
                .edit()
                .putInt(KEY_SHOWN_COUNT, shownCount + 1)
                .apply();
    }

    /**
     * Clear data in shared preferences.<br>
     * This API is called when the rate dialog is approved or canceled.
     * @param context
     */
    private static void clearSharedPreferences(Context context) {
        context.getSharedPreferences(PREF_NAME, MODE_PRIVATE)
                .edit()
                .remove(KEY_INSTALL_DATE)
                .remove(KEY_LAUNCH_TIMES)
                .apply();
    }

    /**
     * Set opt out flag. If it is true, the rate dialog will never shown unless app data is cleared.
     * @param context
     * @param optOut
     */
    private static void setOptOut(final Context context, boolean optOut) {
        context.getSharedPreferences(PREF_NAME, MODE_PRIVATE)
                .edit()
                .putBoolean(KEY_OPT_OUT, optOut)
                .apply();
        AppRateDialog.optOut = optOut;
    }

    /**
     * Store install date.
     * Install date is retrieved from package manager if possible.
     * @param context
     * @param editor
     */
    private static void storeInstallDate(final Context context, Editor editor) {
        Date installDate = new Date();
        PackageManager packageManager = context.getPackageManager();
        try {
            PackageInfo pkgInfo = packageManager.getPackageInfo(context.getPackageName(), 0);
            installDate = new Date(pkgInfo.firstInstallTime);
        } catch (PackageManager.NameNotFoundException ignore) {}
        editor.putLong(KEY_INSTALL_DATE, installDate.getTime());
    }

    /**
     * Store the date the user asked for being asked again later.
     * @param context
     */
    private static void storeAskLaterDate(final Context context) {
        context.getSharedPreferences(PREF_NAME, MODE_PRIVATE)
                .edit()
                .putLong(KEY_ASK_LATER_DATE, System.currentTimeMillis())
                .apply();
    }

    private static void rate(final Context context) {
        if (callback != null) {
            callback.onYesClicked();
        }
        String appPackage = context.getPackageName();
        String url = "https://play.google.com/store/apps/details?id=" + appPackage;
        if (!TextUtils.isEmpty(config.url)) {
            url = config.url;
        }
        Intent intent = new Intent(Intent.ACTION_VIEW, Uri.parse(url));
        context.startActivity(intent);
        setOptOut(context, true);
    }

    private static void dismiss(final Context context) {
        if (callback != null) {
            callback.onCancelClicked();
        }
        clearSharedPreferences(context);
        storeAskLaterDate(context);
    }

    private static void revoke(final Context context) {
        if (callback != null) {
            callback.onNoClicked();
        }
        setOptOut(context, true);
    }

    /**
     * Callback of dialog click event
     */
    public interface Callback {
        /**
         * "Rate now" event
         */
        void onYesClicked();

        /**
         * "No, thanks" event
         */
        void onNoClicked();

        /**
         * "Later" event
         */
        void onCancelClicked();

        /**
         * Feedback given by user
         * @param feedback
         */
        void onFeedback(String feedback);
    }
}
