package fr.bmartel.bboxapi.manager

import android.content.Context
import android.os.Handler
import android.util.Log
import com.google.protobuf.Message
import com.google.protobuf.util.JsonFormat
import de.mannodermaus.rxbonjour.platforms.android.AndroidPlatform
import fr.bmartel.bboxapi.BboxApiProto
import fr.bmartel.bboxapi.router.BboxApiRouter
import fr.bmartel.bboxapi.stb.BboxApiStb
import fr.bmartel.bboxapi.stb.model.StbServiceEvent
import java.util.*

/**
 * Parse protobuf message & dispatch API call.
 *
 * @author Bertrand Martel
 */
class ApiManager(private val context: Context, appId: String, appSecret: String, private val handler: Handler) {

    companion object {
        private val TAG = ApiManager::class.qualifiedName
    }

    val bboxApiRouter = BboxApiRouter()
    val bboxApiStb = BboxApiStb(appId = appId, appSecret = appSecret)

    /**
     * process bbox protobuf format.
     */
    fun process(json: String): BboxApiProto.BboxApiResponse {
        val builder: Message.Builder = BboxApiProto.BboxApiRequest.newBuilder()
        JsonFormat.parser().ignoringUnknownFields().merge(json, builder)
        val bboxApiRequest: BboxApiProto.BboxApiRequest = builder.build() as BboxApiProto.BboxApiRequest
        val receivedAt = Date().time
        when {
            bboxApiRequest.hasPing() -> {
                return Utils.buildResponseMessage(
                        serverSentAt = bboxApiRequest.timing.serverSentAt,
                        executionTime = 0,
                        builder = BboxApiProto.BboxApiResponse.newBuilder()
                                .setPing(BboxApiProto.PingResponse.newBuilder()))
                        .build()
            }
            bboxApiRequest.hasStartApp() -> {
                val actionResponse = ApiUtils.startApp(
                        context = context,
                        appRequest = bboxApiRequest.startApp)
                //start Android application
                return Utils.buildResponseMessage(
                        serverSentAt = bboxApiRequest.timing.serverSentAt,
                        executionTime = Date().time - receivedAt,
                        builder = BboxApiProto.BboxApiResponse.newBuilder(actionResponse)).build()
            }
            bboxApiRequest.hasSetVolume() -> {
                val actionResponse = ApiUtils.setVolume(
                        context = context,
                        volume = bboxApiRequest.setVolume.value,
                        type = bboxApiRequest.setVolume.type)
                //set volume
                return Utils.buildResponseMessage(
                        serverSentAt = bboxApiRequest.timing.serverSentAt,
                        executionTime = Date().time - receivedAt,
                        builder = BboxApiProto.BboxApiResponse.newBuilder()
                                .setVolume(BboxApiProto.VolumeResponse.newBuilder(actionResponse)))
                        .build()
            }
            bboxApiRequest.hasVolumeUp() -> {
                val actionResponse = ApiUtils.setVolumeUp(
                        context = context,
                        type = bboxApiRequest.volumeUp.type)
                //increment volume
                return Utils.buildResponseMessage(
                        serverSentAt = bboxApiRequest.timing.serverSentAt,
                        executionTime = Date().time - receivedAt,
                        builder = BboxApiProto.BboxApiResponse.newBuilder()
                                .setVolume(BboxApiProto.VolumeResponse.newBuilder(actionResponse)))
                        .build()
            }
            bboxApiRequest.hasVolumeDown() -> {
                val actionResponse = ApiUtils.setVolumeDown(
                        context = context,
                        type = bboxApiRequest.volumeDown.type)
                //decrement volume
                return Utils.buildResponseMessage(
                        serverSentAt = bboxApiRequest.timing.serverSentAt,
                        executionTime = Date().time - receivedAt,
                        builder = BboxApiProto.BboxApiResponse.newBuilder()
                                .setVolume(BboxApiProto.VolumeResponse.newBuilder(actionResponse)))
                        .build()
            }
            bboxApiRequest.hasDisplayToast() -> {
                val actionResponse = ApiUtils.displayToast(
                        context = context,
                        handler = handler,
                        toastRequest = bboxApiRequest.displayToast)
                //display toast
                return Utils.buildResponseMessage(
                        serverSentAt = bboxApiRequest.timing.serverSentAt,
                        executionTime = Date().time - receivedAt,
                        builder = BboxApiProto.BboxApiResponse.newBuilder(actionResponse))
                        .build()
            }
            bboxApiRequest.hasMute() -> {
                val actionResponse = ApiUtils.muteVolume(
                        context = context,
                        type = bboxApiRequest.mute.type,
                        state = bboxApiRequest.mute.state)
                //mute
                return Utils.buildResponseMessage(
                        serverSentAt = bboxApiRequest.timing.serverSentAt,
                        executionTime = Date().time - receivedAt,
                        builder = BboxApiProto.BboxApiResponse.newBuilder()
                                .setVolume(BboxApiProto.VolumeResponse.newBuilder(actionResponse)))
                        .build()
            }
            bboxApiRequest.hasWakeUp() -> {
                val actionResponse = ApiUtils.wakeUp(context = context)
                //wake up device
                return Utils.buildResponseMessage(
                        serverSentAt = bboxApiRequest.timing.serverSentAt,
                        executionTime = Date().time - receivedAt,
                        builder = BboxApiProto.BboxApiResponse.newBuilder(actionResponse))
                        .build()
            }
            bboxApiRequest.hasCallRouterApi() -> {
                val actionResponse = ApiUtils.callRouterApi(
                        bboxapi = bboxApiRouter,
                        request = bboxApiRequest.callRouterApi)
                //call router API
                return Utils.buildResponseMessage(
                        serverSentAt = bboxApiRequest.timing.serverSentAt,
                        executionTime = Date().time - receivedAt,
                        builder = BboxApiProto.BboxApiResponse.newBuilder()
                                .setRouter(BboxApiProto.RouterResponse.newBuilder(actionResponse)))
                        .build()
            }
            bboxApiRequest.hasCallStbApi() -> {
                val actionResponse = ApiUtils.callStbApi(
                        bboxapi = bboxApiStb,
                        request = bboxApiRequest.callStbApi)
                //call STB API
                return Utils.buildResponseMessage(
                        serverSentAt = bboxApiRequest.timing.serverSentAt,
                        executionTime = Date().time - receivedAt,
                        builder = BboxApiProto.BboxApiResponse.newBuilder()
                                .setStb(BboxApiProto.StbResponse.newBuilder(actionResponse)))
                        .build()
            }
            bboxApiRequest.hasListApp() -> {
                val resourceResponse = ApiUtils.buildAppList(context = context)
                //get android application list
                return Utils.buildResponseMessage(
                        serverSentAt = bboxApiRequest.timing.serverSentAt,
                        executionTime = Date().time - receivedAt,
                        builder = BboxApiProto.BboxApiResponse.newBuilder()
                                .setListApp(BboxApiProto.ListAppResponse.newBuilder()
                                        .addAllApp(resourceResponse)
                                ))
                        .build()
            }
            bboxApiRequest.hasGetVolume() -> {
                val resourceResponse = ApiUtils.buildVolumeMap(context = context)
                //get volume
                return Utils.buildResponseMessage(
                        serverSentAt = bboxApiRequest.timing.serverSentAt,
                        executionTime = Date().time - receivedAt,
                        builder = BboxApiProto.BboxApiResponse.newBuilder()
                                .setVolume(BboxApiProto.VolumeResponse.newBuilder()
                                        .putAllStreams(resourceResponse)
                                ))
                        .build()
            }
            else -> {
                return Utils.buildResponseMessage(
                        serverSentAt = bboxApiRequest.timing.serverSentAt,
                        executionTime = 0,
                        builder = BboxApiProto.BboxApiResponse.newBuilder()
                                .setError(Utils.buildErrorResponse(
                                        type = BboxApiProto.ErrorType.BAD_FORMAT,
                                        message = "bad format - invalid action")))
                        .build()
            }
        }
    }

    /**
     * start BboxAPI Miami service discovery. This must be called in a background task.
     * If the service is not discovered, the url prefix won't be set for BboxApiStb,
     * so it may result in UnknownHostException in StbResponse message
     */
    fun startDiscovery() {
        bboxApiStb.startRestDiscovery(findOneAndExit = true, platform = AndroidPlatform.create(context)) { eventType, service, error ->
            when (eventType) {
                StbServiceEvent.SERVICE_FOUND -> Log.v(TAG, "Bbox Miami REST service found : ${service?.ip}:${service?.port}")
                StbServiceEvent.DISCOVERY_STOPPED -> Log.v(TAG, "end of REST discovery")
                StbServiceEvent.DISCOVERY_ERROR -> Log.e(TAG, "discovery error", error)
            }
        }
    }
}