package fr.bmartel.bboxapi.manager

import android.content.Context
import android.util.Log
import com.google.protobuf.Message
import com.google.protobuf.util.JsonFormat
import de.mannodermaus.rxbonjour.platforms.android.AndroidPlatform
import fr.bmartel.bboxapi.BboxApiProto
import fr.bmartel.bboxapi.router.BboxApiRouter
import fr.bmartel.bboxapi.stb.BboxApiStb
import fr.bmartel.bboxapi.stb.model.StbServiceEvent
import java.util.*

/**
 * Parse protobuf message & dispatch API call.
 *
 * @author Bertrand Martel
 */
class ApiManager(private val context: Context, appId: String, appSecret: String) {

    companion object {
        private val TAG = ApiManager::class.qualifiedName
    }

    val bboxApiRouter = BboxApiRouter()
    val bboxApiStb = BboxApiStb(appId = appId, appSecret = appSecret)

    /**
     * process bbox protobuf format.
     */
    fun process(json: String): BboxApiProto.BboxApiResponse {
        val builder: Message.Builder = BboxApiProto.BboxApiRequest.newBuilder()
        JsonFormat.parser().ignoringUnknownFields().merge(json, builder)
        val bboxApiRequest: BboxApiProto.BboxApiRequest = builder.build() as BboxApiProto.BboxApiRequest
        val receivedAt = Date().time
        when {
        //process actions
            bboxApiRequest.hasAction() -> when {
                bboxApiRequest.action.hasStartApp() -> {
                    //start Android application
                    return Utils.buildResponseMessage(
                            serverSentAt = bboxApiRequest.timing.serverSentAt,
                            receivedAt = receivedAt,
                            builder = BboxApiProto.BboxApiResponse.newBuilder()
                                    .setAction(ApiUtils.startApp(
                                            context = context,
                                            appRequest = bboxApiRequest.action.startApp))).build()
                }
                bboxApiRequest.action.hasVolume() -> {
                    //set volume
                    return Utils.buildResponseMessage(
                            serverSentAt = bboxApiRequest.timing.serverSentAt,
                            receivedAt = receivedAt,
                            builder = BboxApiProto.BboxApiResponse.newBuilder()
                                    .setAction(BboxApiProto.ActionResponse.newBuilder()
                                            .setVolume(ApiUtils.setVolume(
                                                    context = context,
                                                    volume = bboxApiRequest.action.volume.value,
                                                    type = bboxApiRequest.action.volume.type))))
                            .build()
                }
                bboxApiRequest.action.hasVolumeUp() -> {
                    //increment volume
                    return Utils.buildResponseMessage(
                            serverSentAt = bboxApiRequest.timing.serverSentAt,
                            receivedAt = receivedAt,
                            builder = BboxApiProto.BboxApiResponse.newBuilder()
                                    .setAction(BboxApiProto.ActionResponse.newBuilder()
                                            .setVolume(ApiUtils.setVolumeUp(
                                                    context = context,
                                                    type = bboxApiRequest.action.volumeUp.type))))
                            .build()
                }
                bboxApiRequest.action.hasVolumeDown() -> {
                    //decrement volume
                    return Utils.buildResponseMessage(
                            serverSentAt = bboxApiRequest.timing.serverSentAt,
                            receivedAt = receivedAt,
                            builder = BboxApiProto.BboxApiResponse.newBuilder()
                                    .setAction(BboxApiProto.ActionResponse.newBuilder()
                                            .setVolume(ApiUtils.setVolumeDown(
                                                    context = context,
                                                    type = bboxApiRequest.action.volumeDown.type))))
                            .build()
                }
                bboxApiRequest.action.hasDisplayToast() -> {
                    //display toast
                    return Utils.buildResponseMessage(
                            serverSentAt = bboxApiRequest.timing.serverSentAt,
                            receivedAt = receivedAt,
                            builder = BboxApiProto.BboxApiResponse.newBuilder()
                                    .setAction(ApiUtils.displayToast(
                                            context,
                                            toastRequest = bboxApiRequest.action.displayToast)))
                            .build()
                }
                bboxApiRequest.action.hasMute() -> {
                    //mute
                    return Utils.buildResponseMessage(
                            serverSentAt = bboxApiRequest.timing.serverSentAt,
                            receivedAt = receivedAt,
                            builder = BboxApiProto.BboxApiResponse.newBuilder()
                                    .setAction(BboxApiProto.ActionResponse.newBuilder()
                                            .setVolume(ApiUtils.muteVolume(
                                                    context = context,
                                                    type = bboxApiRequest.action.mute.type,
                                                    state = bboxApiRequest.action.mute.state))))
                            .build()
                }
                bboxApiRequest.action.hasWakeUp() -> {
                    //wake up device
                    return Utils.buildResponseMessage(
                            serverSentAt = bboxApiRequest.timing.serverSentAt,
                            receivedAt = receivedAt,
                            builder = BboxApiProto.BboxApiResponse.newBuilder()
                                    .setAction(ApiUtils.wakeUp(context = context)))
                            .build()
                }
                bboxApiRequest.action.hasCallRouterApi() -> {
                    //call router API
                    return Utils.buildResponseMessage(
                            serverSentAt = bboxApiRequest.timing.serverSentAt,
                            receivedAt = receivedAt,
                            builder = BboxApiProto.BboxApiResponse.newBuilder()
                                    .setAction(BboxApiProto.ActionResponse.newBuilder()
                                            .setRouter(ApiUtils.callRouterApi(
                                                    bboxapi = bboxApiRouter,
                                                    request = bboxApiRequest.action.callRouterApi))))
                            .build()
                }
                bboxApiRequest.action.hasCallStbApi() -> {
                    //call STB API
                    return Utils.buildResponseMessage(
                            serverSentAt = bboxApiRequest.timing.serverSentAt,
                            receivedAt = receivedAt,
                            builder = BboxApiProto.BboxApiResponse.newBuilder()
                                    .setAction(BboxApiProto.ActionResponse.newBuilder()
                                            .setStb(ApiUtils.callStbApi(
                                                    bboxapi = bboxApiStb,
                                                    request = bboxApiRequest.action.callStbApi))))
                            .build()
                }
                else -> {
                    return Utils.buildResponseMessage(
                            serverSentAt = bboxApiRequest.timing.serverSentAt,
                            receivedAt = receivedAt,
                            builder = BboxApiProto.BboxApiResponse.newBuilder()
                                    .setAction(Utils.buildActionResponseError(
                                            type = BboxApiProto.ErrorType.BAD_FORMAT,
                                            message = "bad format - invalid action")))
                            .build()
                }
            }
        //process resource
            bboxApiRequest.hasResource() -> when {
                bboxApiRequest.resource.hasListApp() -> {
                    //get android application list
                    return Utils.buildResponseMessage(
                            serverSentAt = bboxApiRequest.timing.serverSentAt,
                            receivedAt = receivedAt,
                            builder = BboxApiProto.BboxApiResponse.newBuilder()
                                    .setResource(BboxApiProto.ResourceResponse.newBuilder()
                                            .setListApp(BboxApiProto.ListAppResponse.newBuilder()
                                                    .addAllApp(ApiUtils.buildAppList(context = context))
                                            )))
                            .build()
                }
                bboxApiRequest.resource.hasVolume() -> {
                    //get volume
                    return Utils.buildResponseMessage(
                            serverSentAt = bboxApiRequest.timing.serverSentAt,
                            receivedAt = receivedAt,
                            builder = BboxApiProto.BboxApiResponse.newBuilder()
                                    .setResource(BboxApiProto.ResourceResponse.newBuilder()
                                            .setVolume(BboxApiProto.VolumeResponse.newBuilder()
                                                    .putAllStreams(ApiUtils.buildVolumeMap(context = context))
                                            )))
                            .build()
                }
                else -> {
                    return Utils.buildResponseMessage(
                            serverSentAt = bboxApiRequest.timing.serverSentAt,
                            receivedAt = receivedAt,
                            builder = BboxApiProto.BboxApiResponse.newBuilder()
                                    .setResource(Utils.buildResourceResponseError(
                                            type = BboxApiProto.ErrorType.BAD_FORMAT,
                                            message = "bad format - invalid resource")))
                            .build()

                }
            }
            else -> return Utils.buildResponseMessage(
                    serverSentAt = bboxApiRequest.timing.serverSentAt,
                    receivedAt = receivedAt,
                    builder = BboxApiProto.BboxApiResponse.newBuilder()
                            .setAction(Utils.buildActionResponseError(
                                    type = BboxApiProto.ErrorType.BAD_FORMAT,
                                    message = "bad format - request must have resource or action")))
                    .build()
        }
    }

    /**
     * start BboxAPI Miami service discovery. This must be called in a background task.
     * If the service is not discovered, the url prefix won't be set for BboxApiStb,
     * so it may result in UnknownHostException in StbResponse message
     */
    fun startDiscovery() {
        bboxApiStb.startRestDiscovery(findOneAndExit = true, platform = AndroidPlatform.create(context)) { eventType, service, error ->
            when (eventType) {
                StbServiceEvent.SERVICE_FOUND -> Log.v(TAG, "Bbox Miami REST service found : ${service?.ip}:${service?.port}")
                StbServiceEvent.DISCOVERY_STOPPED -> Log.v(TAG, "end of REST discovery")
                StbServiceEvent.DISCOVERY_ERROR -> Log.e(TAG, "discovery error", error)
            }
        }
    }
}