package fr.bmartel.bboxapi.manager

import android.content.ComponentName
import android.content.Context
import android.content.Intent
import android.graphics.Color
import android.media.AudioManager
import android.net.Uri
import android.os.Build
import android.os.Handler
import android.os.PowerManager
import android.view.Gravity
import android.view.LayoutInflater
import android.widget.TextView
import android.widget.Toast
import com.github.kittinunf.fuel.core.HttpException
import com.github.kittinunf.fuel.core.Method
import com.github.kittinunf.result.Result
import com.google.gson.JsonSyntaxException
import fr.bmartel.bboxapi.BboxApiProto
import fr.bmartel.bboxapi.router.BboxApiRouter
import fr.bmartel.bboxapi.stb.BboxApiStb
import java.net.UnknownHostException
import java.util.*

class ApiUtils {

    companion object {
        fun buildAppList(context: Context): ArrayList<BboxApiProto.AndroidApp> {
            val pm = context.packageManager
            val appList = pm.getInstalledApplications(0)
            val apps: ArrayList<BboxApiProto.AndroidApp> = arrayListOf()
            appList
                    .filter { pm.getLeanbackLaunchIntentForPackage(it.packageName) != null }
                    .forEach {
                        apps.add(BboxApiProto.AndroidApp.newBuilder()
                                .setPackageName(it.packageName)
                                .setAppName(pm.getApplicationLabel(it).toString())
                                .build())
                    }
            apps.sortBy { it -> it.appName }
            return apps
        }

        fun buildVolumeMap(context: Context): Map<Int, Int> {
            val volumeMap = mutableMapOf<Int, Int>()
            val audio = context.getSystemService(Context.AUDIO_SERVICE) as AudioManager
            for (type in BboxApiProto.AudioType.values()) {
                volumeMap[type.number] = audio.getStreamVolume(type.number) * 100 / audio.getStreamMaxVolume(type.number)
            }
            return volumeMap
        }

        fun startApp(context: Context, appRequest: BboxApiProto.StartAppRequest): BboxApiProto.ActionResponse {
            var appIntent: Intent? = Intent(appRequest.action)
            var launchPackage = appRequest.packageName
            val pm = context.packageManager

            if (appRequest.hasComponentName() && appRequest.componentName != "") {
                if (appRequest.componentName.indexOf('/') != -1) {
                    val component = ComponentName.unflattenFromString(appRequest.componentName)
                    launchPackage = component.packageName
                    appIntent?.component = component
                    appIntent?.`package` = launchPackage
                } else {
                    appIntent?.setClassName(launchPackage, appRequest.componentName)
                    appIntent?.`package` = launchPackage
                }
            } else if (appRequest.hasAction() && appRequest.action != "") {
                appIntent?.`package` = launchPackage
            } else {
                appIntent = pm.getLeanbackLaunchIntentForPackage(launchPackage) ?: pm.getLaunchIntentForPackage(launchPackage)
            }
            if (appRequest.hasData() && appRequest.data != "") {
                appIntent?.data = Uri.parse(appRequest.data)
            }

            appIntent?.flags = Intent.FLAG_ACTIVITY_NEW_TASK

            if (appIntent != null && pm.resolveActivity(appIntent, 0) != null) {
                context.startActivity(appIntent)
                return BboxApiProto.ActionResponse.newBuilder()
                        .setStatus(BboxApiProto.ActionStatus.newBuilder()
                                .setSuccess(true)
                                .setModified(false)).build()
            }
            return BboxApiProto.ActionResponse.newBuilder()
                    .setError(BboxApiProto.ErrorResponse.newBuilder()
                            .setType(BboxApiProto.ErrorType.ACTION_FAILED)
                            .setMessage("activity not found")).build()
        }

        fun setVolume(context: Context, volume: Int, type: BboxApiProto.AudioType): BboxApiProto.VolumeResponse {
            val audio = context.getSystemService(Context.AUDIO_SERVICE) as AudioManager
            var vol = volume
            if (vol < 0) {
                vol = 0
            } else if (vol > 100) {
                vol = 100
            }
            vol = vol * audio.getStreamMaxVolume(type.number) / 100
            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.N) {
                vol = if (vol == 0) 1 else vol
            }
            audio.setStreamVolume(type.number, vol, AudioManager.FLAG_SHOW_UI)
            return BboxApiProto.VolumeResponse.newBuilder()
                    .putAllStreams(buildVolumeMap(context)).build()
        }

        fun setVolumeUp(context: Context, type: BboxApiProto.AudioType): BboxApiProto.VolumeResponse {
            (context.getSystemService(Context.AUDIO_SERVICE) as AudioManager).adjustStreamVolume(type.ordinal, AudioManager.ADJUST_RAISE, AudioManager.FLAG_SHOW_UI)
            return BboxApiProto.VolumeResponse.newBuilder()
                    .putAllStreams(buildVolumeMap(context)).build()
        }

        fun setVolumeDown(context: Context, type: BboxApiProto.AudioType): BboxApiProto.VolumeResponse {
            (context.getSystemService(Context.AUDIO_SERVICE) as AudioManager).adjustStreamVolume(type.ordinal, AudioManager.ADJUST_LOWER, AudioManager.FLAG_SHOW_UI)
            return BboxApiProto.VolumeResponse.newBuilder()
                    .putAllStreams(buildVolumeMap(context)).build()
        }

        fun muteVolume(context: Context, type: BboxApiProto.AudioType, state: Boolean): BboxApiProto.VolumeResponse {
            val audio = context.getSystemService(Context.AUDIO_SERVICE) as AudioManager
            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.M) {
                audio.adjustStreamVolume(type.ordinal, if (state) AudioManager.ADJUST_MUTE else AudioManager.ADJUST_UNMUTE, AudioManager.FLAG_SHOW_UI)
            } else {
                audio.setStreamMute(type.ordinal, state)
            }
            return BboxApiProto.VolumeResponse.newBuilder()
                    .putAllStreams(buildVolumeMap(context)).build()
        }

        fun displayToast(context: Context, handler: Handler, toastRequest: BboxApiProto.ToastRequest): BboxApiProto.ActionResponse {
            handler.post({
                val toast = Toast.makeText(context, toastRequest.message, toastRequest.duration.number)
                val inflater = context.getSystemService(Context.LAYOUT_INFLATER_SERVICE) as LayoutInflater
                toast.view = inflater.inflate(R.layout.toast, null)

                val v = toast.view.findViewById(R.id.message) as TextView
                if (toastRequest.hasColor()) {
                    val color = Color.parseColor(toastRequest.color)
                    v.setTextColor(color)
                    v.setBackgroundColor(color)
                    toast.view.setBackgroundColor(color)
                }

                val x: Int
                val gx: Int
                val y: Int
                val gy: Int

                if (!toastRequest.hasPosX()) {
                    x = 0
                    gx = Gravity.CENTER_HORIZONTAL
                } else {
                    x = toastRequest.posX
                    gx = Gravity.LEFT
                }

                if (!toastRequest.hasPosY()) {
                    y = toast.yOffset
                    gy = Gravity.BOTTOM
                } else {
                    y = toastRequest.posY
                    gy = Gravity.TOP
                }
                toast.setGravity(gx or gy, x, y)

                toast.show()
            })
            return BboxApiProto.ActionResponse.newBuilder()
                    .setStatus(BboxApiProto.ActionStatus.newBuilder()
                            .setModified(false)
                            .setSuccess(true)).build()
        }

        fun wakeUp(context: Context): BboxApiProto.ActionResponse {
            val pm = context.getSystemService(Context.POWER_SERVICE) as PowerManager
            if (pm.isInteractive) {
                //already wakeup
                return BboxApiProto.ActionResponse.newBuilder()
                        .setStatus(BboxApiProto.ActionStatus.newBuilder()
                                .setModified(false)
                                .setSuccess(true)).build()
            } else {
                val wakeLock = pm.newWakeLock(PowerManager.SCREEN_BRIGHT_WAKE_LOCK or PowerManager.FULL_WAKE_LOCK or PowerManager.ACQUIRE_CAUSES_WAKEUP, "TAG")
                wakeLock.acquire()
                return BboxApiProto.ActionResponse.newBuilder()
                        .setStatus(BboxApiProto.ActionStatus.newBuilder()
                                .setModified(true)
                                .setSuccess(true)).build()
            }
        }

        fun callRouterApi(bboxapi: BboxApiRouter, request: BboxApiProto.HttpRequest): BboxApiProto.RouterResponse {
            val routerResponse = BboxApiProto.RouterResponse.newBuilder().setRequest(request)

            if (request.auth && bboxapi.blocked) {
                //bboxapi is blocked don't send more request
                return routerResponse
                        .setError(BboxApiProto.ErrorResponse.newBuilder()
                                .setType(BboxApiProto.ErrorType.USER_BLOCKED)
                                .setMessage("user is blocked, wait until ${bboxapi.blockedUntil}"))
                        .setAuthStatus(BboxApiProto.AuthStatus.newBuilder()
                                .setAttempts(bboxapi.attempts)
                                .setAuthenticated(bboxapi.authenticated)
                                .setBlocked(bboxapi.blocked)
                                .setBlockedUntil(bboxapi.blockedUntil.time)).build()
            }

            try {
                val (_, response, result) = bboxapi.createCustomRequestSync(
                        request = bboxapi.manager.request(method = Method.valueOf(request.method.name), path = request.url),
                        auth = request.auth)

                when (result) {
                    is Result.Failure -> {
                        val ex = result.getException()
                        var type: BboxApiProto.ErrorType = BboxApiProto.ErrorType.OTHER

                        when {
                            ex.exception is UnknownHostException -> type = BboxApiProto.ErrorType.UNKNOWN_HOST
                            ex.exception is HttpException -> type = BboxApiProto.ErrorType.HTTP_ERROR
                            ex.exception is JsonSyntaxException -> type = BboxApiProto.ErrorType.JSON_SYNTAX
                        }
                        routerResponse
                                .setError(BboxApiProto.ErrorResponse.newBuilder()
                                        .setMessage("router request failed")
                                        .setType(type)
                                        .setStacktrace(Utils.exceptionToString(ex)))
                                .setAuthStatus(BboxApiProto.AuthStatus.newBuilder()
                                        .setAttempts(bboxapi.attempts)
                                        .setAuthenticated(bboxapi.authenticated)
                                        .setBlocked(bboxapi.blocked)
                                        .setBlockedUntil(bboxapi.blockedUntil.time))

                        if (BboxApiProto.HttpStatus.forNumber(response.statusCode) != null) {
                            routerResponse.setResponse(BboxApiProto.HttpResponse.newBuilder()
                                    .setBody(String(response.data))
                                    .putAllHeaders(Utils.convertHeaders(response.headers))
                                    .setStatus(BboxApiProto.HttpStatus.forNumber(response.statusCode)))
                        }
                        return routerResponse.build()
                    }
                    is Result.Success -> {
                        return routerResponse
                                .setResponse(BboxApiProto.HttpResponse.newBuilder()
                                        .setBody(String(response.data))
                                        .putAllHeaders(Utils.convertHeaders(response.headers))
                                        .setStatus(BboxApiProto.HttpStatus.forNumber(response.statusCode)))
                                .setAuthStatus(BboxApiProto.AuthStatus.newBuilder()
                                        .setAttempts(bboxapi.attempts)
                                        .setAuthenticated(bboxapi.authenticated)
                                        .setBlocked(bboxapi.blocked)
                                        .setBlockedUntil(bboxapi.blockedUntil.time)).build()
                    }
                }
            } catch (e: IllegalArgumentException) {
                //method is not supported
                return routerResponse
                        .setError(BboxApiProto.ErrorResponse.newBuilder()
                                .setType(BboxApiProto.ErrorType.METHOD_NOT_SUPPORTED)
                                .setMessage("method ${request.method} is not supported"))
                        .setAuthStatus(BboxApiProto.AuthStatus.newBuilder()
                                .setAttempts(bboxapi.attempts)
                                .setAuthenticated(bboxapi.authenticated)
                                .setBlocked(bboxapi.blocked)
                                .setBlockedUntil(bboxapi.blockedUntil.time)).build()
            }
        }

        fun callStbApi(bboxapi: BboxApiStb, request: BboxApiProto.HttpRequest): BboxApiProto.StbResponse {
            val stbResponse = BboxApiProto.StbResponse.newBuilder().setRequest(request)
            val stbServiceStatus = BboxApiProto.StbServiceStatus.newBuilder()

            //set service status composed of rest service selected & list of stb services found on local network
            if (bboxapi.restService != null) {
                stbServiceStatus
                        .setSelectedService(BboxApiProto.StbService.newBuilder()
                                .setIp(bboxapi.restService?.ip)
                                .setPort(bboxapi.restService?.port ?: -1))
            }
            for (it in bboxapi.restServiceList) {
                stbServiceStatus
                        .addServices(BboxApiProto.StbService.newBuilder()
                                .setIp(it.ip)
                                .setPort(it.port))
            }
            stbResponse.setServiceStatus(stbServiceStatus)

            try {
                val (_, response, result) = bboxapi.createCustomRequestSync(
                        request = bboxapi.manager.request(method = Method.valueOf(request.method.name), path = request.url)
                )

                when (result) {
                    is Result.Failure -> {
                        val ex = result.getException()
                        var type: BboxApiProto.ErrorType = BboxApiProto.ErrorType.OTHER
                        when {
                            ex.exception is UnknownHostException -> type = BboxApiProto.ErrorType.UNKNOWN_HOST
                            ex.exception is HttpException -> type = BboxApiProto.ErrorType.HTTP_ERROR
                            ex.exception is JsonSyntaxException -> type = BboxApiProto.ErrorType.JSON_SYNTAX
                        }
                        stbResponse
                                .setError(BboxApiProto.ErrorResponse.newBuilder()
                                        .setMessage("stb request failed")
                                        .setType(type)
                                        .setStacktrace(Utils.exceptionToString(ex)))

                        if (BboxApiProto.HttpStatus.forNumber(response.statusCode) != null) {
                            stbResponse.setResponse(BboxApiProto.HttpResponse.newBuilder()
                                    .setBody(String(response.data))
                                    .putAllHeaders(Utils.convertHeaders(response.headers))
                                    .setStatus(BboxApiProto.HttpStatus.forNumber(response.statusCode)))
                        }
                        return stbResponse.build()
                    }
                    is Result.Success -> {
                        return stbResponse
                                .setResponse(BboxApiProto.HttpResponse.newBuilder()
                                        .setBody(String(response.data))
                                        .putAllHeaders(Utils.convertHeaders(response.headers))
                                        .setStatus(BboxApiProto.HttpStatus.forNumber(response.statusCode)))
                                .build()
                    }
                }
            } catch (e: IllegalArgumentException) {
                //method is not supported
                return stbResponse
                        .setError(BboxApiProto.ErrorResponse.newBuilder()
                                .setType(BboxApiProto.ErrorType.METHOD_NOT_SUPPORTED)
                                .setMessage("method ${request.method} is not supported"))
                        .build()
            }
        }
    }
}
