package fr.bmartel.bboxapi.manager

import android.content.Context
import android.os.Handler
import com.google.protobuf.InvalidProtocolBufferException
import com.google.protobuf.Message
import com.google.protobuf.util.JsonFormat
import fr.bmartel.bboxapi.BboxApiProto
import fr.bmartel.bboxapi.router.BboxApiRouter
import fr.bmartel.bboxapi.stb.BboxApiStb
import java.util.*

/**
 * Parse protobuf message & dispatch API call.
 *
 * @author Bertrand Martel
 */
class ApiManager(private val context: Context, appId: String, appSecret: String, private val handler: Handler) {

    val bboxApiRouter = BboxApiRouter()
    val bboxApiStb = BboxApiStb(appId = appId, appSecret = appSecret)

    /**
     * process bbox protobuf format.
     */
    fun process(json: String): BboxApiProto.BboxApiResponse? {
        var response: BboxApiProto.BboxApiResponse?
        try {
            val builder: Message.Builder = BboxApiProto.BboxApiRequest.newBuilder()
            JsonFormat.parser().ignoringUnknownFields().merge(json, builder)
            val bboxApiRequest: BboxApiProto.BboxApiRequest = builder.build() as BboxApiProto.BboxApiRequest
            val receivedAt = Date().time
            when {
                bboxApiRequest.hasPing() -> {
                    response = Utils.buildResponseMessage(
                            serverSentAt = bboxApiRequest.timing.serverSentAt,
                            executionTime = 0,
                            builder = BboxApiProto.BboxApiResponse.newBuilder()
                                    .setPing(BboxApiProto.PingResponse.newBuilder()))
                            .build()
                }
                bboxApiRequest.hasStartApp() -> {
                    val actionResponse = ApiUtils.startApp(
                            context = context,
                            appRequest = bboxApiRequest.startApp)
                    //start Android application
                    response = Utils.buildResponseMessage(
                            serverSentAt = bboxApiRequest.timing.serverSentAt,
                            executionTime = Date().time - receivedAt,
                            builder = BboxApiProto.BboxApiResponse.newBuilder(actionResponse)).build()
                }
                bboxApiRequest.hasBroadcastMessage() -> {
                    val actionResponse = ApiUtils.sendBroadcastMessage(
                            context = context,
                            broadcastRequest = bboxApiRequest.broadcastMessage)
                    response = Utils.buildResponseMessage(
                            serverSentAt = bboxApiRequest.timing.serverSentAt,
                            executionTime = 0,
                            builder = BboxApiProto.BboxApiResponse.newBuilder(actionResponse))
                            .build()
                }
                bboxApiRequest.hasSetVolume() -> {
                    val actionResponse = ApiUtils.setVolume(
                            context = context,
                            volume = bboxApiRequest.setVolume.value,
                            type = bboxApiRequest.setVolume.type)
                    //set volume
                    response = Utils.buildResponseMessage(
                            serverSentAt = bboxApiRequest.timing.serverSentAt,
                            executionTime = Date().time - receivedAt,
                            builder = BboxApiProto.BboxApiResponse.newBuilder()
                                    .setVolume(BboxApiProto.VolumeResponse.newBuilder(actionResponse)))
                            .build()
                }
                bboxApiRequest.hasVolumeUp() -> {
                    val actionResponse = ApiUtils.setVolumeUp(
                            context = context,
                            type = bboxApiRequest.volumeUp.type)
                    //increment volume
                    response = Utils.buildResponseMessage(
                            serverSentAt = bboxApiRequest.timing.serverSentAt,
                            executionTime = Date().time - receivedAt,
                            builder = BboxApiProto.BboxApiResponse.newBuilder()
                                    .setVolume(BboxApiProto.VolumeResponse.newBuilder(actionResponse)))
                            .build()
                }
                bboxApiRequest.hasVolumeDown() -> {
                    val actionResponse = ApiUtils.setVolumeDown(
                            context = context,
                            type = bboxApiRequest.volumeDown.type)
                    //decrement volume
                    response = Utils.buildResponseMessage(
                            serverSentAt = bboxApiRequest.timing.serverSentAt,
                            executionTime = Date().time - receivedAt,
                            builder = BboxApiProto.BboxApiResponse.newBuilder()
                                    .setVolume(BboxApiProto.VolumeResponse.newBuilder(actionResponse)))
                            .build()
                }
                bboxApiRequest.hasDisplayToast() -> {
                    val actionResponse = ApiUtils.displayToast(
                            context = context,
                            handler = handler,
                            toastRequest = bboxApiRequest.displayToast)
                    //display toast
                    response = Utils.buildResponseMessage(
                            serverSentAt = bboxApiRequest.timing.serverSentAt,
                            executionTime = Date().time - receivedAt,
                            builder = BboxApiProto.BboxApiResponse.newBuilder(actionResponse))
                            .build()
                }
                bboxApiRequest.hasMute() -> {
                    val actionResponse = ApiUtils.muteVolume(
                            context = context,
                            type = bboxApiRequest.mute.type,
                            state = bboxApiRequest.mute.state)
                    //mute
                    response = Utils.buildResponseMessage(
                            serverSentAt = bboxApiRequest.timing.serverSentAt,
                            executionTime = Date().time - receivedAt,
                            builder = BboxApiProto.BboxApiResponse.newBuilder()
                                    .setVolume(BboxApiProto.VolumeResponse.newBuilder(actionResponse)))
                            .build()
                }
                bboxApiRequest.hasWakeUp() -> {
                    val actionResponse = ApiUtils.wakeUp(context = context)
                    //wake up device
                    response = Utils.buildResponseMessage(
                            serverSentAt = bboxApiRequest.timing.serverSentAt,
                            executionTime = Date().time - receivedAt,
                            builder = BboxApiProto.BboxApiResponse.newBuilder(actionResponse))
                            .build()
                }
                bboxApiRequest.hasGetSpideoId() -> {
                    val spideoIdResponse = ApiUtils.getSpideoId()
                    response = Utils.buildResponseMessage(
                            serverSentAt = bboxApiRequest.timing.serverSentAt,
                            executionTime = Date().time - receivedAt,
                            builder = BboxApiProto.BboxApiResponse.newBuilder()
                                    .setSpideo(BboxApiProto.SpideoIdResponse.newBuilder(spideoIdResponse)))
                            .build()
                }
                bboxApiRequest.hasRouterApi() -> {
                    val actionResponse = ApiUtils.callRouterApi(
                            bboxapi = bboxApiRouter,
                            request = bboxApiRequest.routerApi)
                    //call router API
                    response = Utils.buildResponseMessage(
                            serverSentAt = bboxApiRequest.timing.serverSentAt,
                            executionTime = Date().time - receivedAt,
                            builder = BboxApiProto.BboxApiResponse.newBuilder()
                                    .setRouter(BboxApiProto.RouterResponse.newBuilder(actionResponse)))
                            .build()
                }
                bboxApiRequest.hasStbApi() -> {
                    val actionResponse = ApiUtils.callStbApi(
                            bboxapi = bboxApiStb,
                            request = bboxApiRequest.stbApi)
                    //call STB API
                    response = Utils.buildResponseMessage(
                            serverSentAt = bboxApiRequest.timing.serverSentAt,
                            executionTime = Date().time - receivedAt,
                            builder = BboxApiProto.BboxApiResponse.newBuilder()
                                    .setStb(BboxApiProto.StbResponse.newBuilder(actionResponse)))
                            .build()
                }
                bboxApiRequest.hasListApp() -> {
                    val resourceResponse = ApiUtils.buildAppList(context = context)
                    //get android application list
                    response = Utils.buildResponseMessage(
                            serverSentAt = bboxApiRequest.timing.serverSentAt,
                            executionTime = Date().time - receivedAt,
                            builder = BboxApiProto.BboxApiResponse.newBuilder()
                                    .setListApp(BboxApiProto.ListAppResponse.newBuilder()
                                            .addAllApp(resourceResponse)
                                    ))
                            .build()
                }
                bboxApiRequest.hasGetVolume() -> {
                    val resourceResponse = ApiUtils.buildVolumeMap(context = context)
                    //get volume
                    response = Utils.buildResponseMessage(
                            serverSentAt = bboxApiRequest.timing.serverSentAt,
                            executionTime = Date().time - receivedAt,
                            builder = BboxApiProto.BboxApiResponse.newBuilder()
                                    .setVolume(BboxApiProto.VolumeResponse.newBuilder()
                                            .putAllStreams(resourceResponse)
                                    ))
                            .build()
                }
                else -> {
                    response = Utils.buildResponseMessage(
                            serverSentAt = bboxApiRequest.timing.serverSentAt,
                            executionTime = 0,
                            builder = BboxApiProto.BboxApiResponse.newBuilder()
                                    .setError(Utils.buildErrorResponse(
                                            type = BboxApiProto.ErrorType.BAD_FORMAT,
                                            message = "bad format - invalid action")))
                            .build()
                }
            }
            if (bboxApiRequest.ff) {
                return null
            }
        } catch (e: InvalidProtocolBufferException) {
            response = Utils.buildResponseMessage(
                    serverSentAt = 0,
                    executionTime = 0,
                    builder = BboxApiProto.BboxApiResponse.newBuilder()
                            .setError(Utils.buildErrorResponse(
                                    type = BboxApiProto.ErrorType.BAD_FORMAT,
                                    message = "bad format - invalid action")))
                    .build()
        }
        return response
    }
}