package fr.zebasto.spring.identity.defaults.model;

import com.fasterxml.jackson.databind.annotation.JsonDeserialize;
import fr.zebasto.spring.identity.contract.model.Action;
import fr.zebasto.spring.identity.contract.model.Application;
import fr.zebasto.spring.identity.contract.model.Permission;
import fr.zebasto.spring.identity.defaults.configuration.IdentityRoles;
import org.apache.commons.lang3.builder.ToStringBuilder;
import org.apache.commons.lang3.builder.ToStringStyle;
import org.hibernate.annotations.Type;
import org.hibernate.validator.constraints.NotEmpty;

import javax.persistence.*;
import javax.validation.constraints.NotNull;
import java.text.MessageFormat;
import java.util.UUID;

/**
 * Describe a permission.
 */
@Entity
@Table(name = "IDENTITY_PERMISSIONS")
public class DefaultPermission implements Permission<UUID> {

    private UUID id = UUID.randomUUID();
    private String title;
    private String description;
    private Application application;
    private Action action;
    private String code;

    /**
     * Default constructor
     */
    public DefaultPermission() {

    }

    /**
     * Complete constructor
     *
     * @param title       The title
     * @param description The description
     * @param application The application
     * @param action      The action
     */
    public DefaultPermission(String title, String description, Application application, Action action) {
        this.title = title;
        this.description = description;
        this.application = application;
        this.action = action;
    }

    /**
     * Retrieve the id of the permission
     *
     * @return The id
     */
    @Id
    @Override
    @Type(type = "uuid-char")
    public UUID getId() {
        return id;
    }

    /**
     * Set the id of the permission
     *
     * @param id The id
     */
    @Override
    public void setId(UUID id) {
        this.id = id;
    }

    /**
     * Retrieve the code of the permission
     *
     * @return The code
     */
    @Override
    @Column(nullable = false, unique = true)
    public String getCode() {
        return this.code;
    }

    /**
     * Set the code of the permission
     *
     * @param code The code
     */
    @Override
    public void setCode(String code) {
        this.code = code;
    }

    /**
     * @return The action
     */
    @Override
    @JsonDeserialize(as = DefaultAction.class)
    @ManyToOne(fetch = FetchType.EAGER, targetEntity = DefaultAction.class)
    public Action getAction() {
        return action;
    }

    /**
     * @param action The action
     */
    @Override
    public void setAction(Action action) {
        this.action = action;
    }

    /**
     * Retrieve the title of the permission
     *
     * @return The title
     */
    @Column
    @NotNull
    @NotEmpty
    public String getTitle() {
        return title;
    }

    /**
     * Set the title of the permission
     *
     * @param title The title
     */
    public void setTitle(String title) {
        this.title = title;
    }

    /**
     * Retrieve the description of the permission
     *
     * @return The description
     */
    @Column
    @NotNull
    @NotEmpty
    public String getDescription() {
        return description;
    }

    /**
     * Set the description of the permission
     *
     * @param description The description
     */
    public void setDescription(String description) {
        this.description = description;
    }

    /**
     * @return The application
     */
    @Override
    @JsonDeserialize(as = DefaultApplication.class)
    @ManyToOne(fetch = FetchType.EAGER, targetEntity = DefaultApplication.class)
    public Application getApplication() {
        return application;
    }

    /**
     * @param application The application
     */
    @Override
    public void setApplication(Application application) {
        this.application = application;
    }

    /**
     *
     */
    @PostLoad
    @PreUpdate
    @PrePersist
    public void updateCode() {
        this.code = MessageFormat.format("{0}_{1}_{2}", IdentityRoles.PFX, this.action.getName(), this.application.getName());
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }

        if (o == null || getClass() != o.getClass()) {
            return false;
        }

        DefaultPermission that = (DefaultPermission) o;

        if (!id.equals(that.id)) {
            return false;
        }

        return true;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public int hashCode() {
        return id.hashCode();
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public String toString() {
        return ToStringBuilder.reflectionToString(this, ToStringStyle.DEFAULT_STYLE);
    }
}
