package gg.gamerewards.di

import android.content.Context
import android.util.Log
import dagger.Module
import dagger.Provides
import dagger.hilt.InstallIn
import dagger.hilt.components.SingletonComponent
import gg.gamerewards.data.api.ApiRepository
import gg.gamerewards.data.api.ApiService
import gg.gamerewards.data.api.BaseUrlInterceptor
import gg.gamerewards.data.api.DecryptionInterceptor
import gg.gamerewards.data.api.EncryptionInterceptor
import gg.gamerewards.data.api.InternetConnectivityInterceptor
import gg.gamerewards.data.api.ResponseCodeInterceptor
import gg.gamerewards.data.local.PrefManager
import okhttp3.Interceptor
import okhttp3.OkHttpClient
import okhttp3.Request
import okhttp3.RequestBody
import okhttp3.logging.HttpLoggingInterceptor
import okio.Buffer
import retrofit2.Retrofit
import retrofit2.adapter.rxjava2.RxJava2CallAdapterFactory
import retrofit2.converter.gson.GsonConverterFactory
import java.io.IOException
import java.util.Locale
import javax.inject.Singleton


/**
 * Created by Hasan Güler on 18.05.2023.
 */
@Module
@InstallIn(SingletonComponent::class)
internal class ApiModule {
    companion object {
        val TEST_URL: String = "https://api2.richie.games/v1/"
        val BASE_URL : String = "https://testapi.richie.games/v1/"
        val APP_CATEGORY_URL: String = "https://universe.megafortuna.co:9001/"
        val LICENSE_ISENABLED_URL: String = "https://richie.games/app/version.json"
    }

    @Singleton
    @Provides
    fun providesHttpLoggingInterceptor() = HttpLoggingInterceptor()
        .apply {
            level = HttpLoggingInterceptor.Level.BODY
        }

    @Singleton
    @Provides
    fun providesOkHttpClient(
        httpLoggingInterceptor: HttpLoggingInterceptor,
        prefManager: PrefManager,
        context: Context
    ): OkHttpClient {
        val client = OkHttpClient
            .Builder()
            .retryOnConnectionFailure(true)
            .addInterceptor(Interceptor { chain ->
                val requestBuilder: Request.Builder = chain.request().newBuilder()
                requestBuilder.addHeader("Authorization", "Bearer ${prefManager.jwtToken}")
                requestBuilder.addHeader("X-Firebase-AppCheck", "test")
                requestBuilder.addHeader("App-Name", context.packageName)
                requestBuilder.addHeader("Language", Locale.getDefault().language)
                requestBuilder.addHeader("Os-Type", "Android")
                val build = requestBuilder.build()
                Log.d("Richie",bodyToString(build.body) ?: "")
                chain.proceed(build)
            })
            .addInterceptor(InternetConnectivityInterceptor(context))
            .addInterceptor(ResponseCodeInterceptor(context))
            .addInterceptor(BaseUrlInterceptor(prefManager))
            .addInterceptor(httpLoggingInterceptor)
            .addInterceptor(EncryptionInterceptor())
            .addInterceptor(DecryptionInterceptor())
            //.addInterceptor(VpnDetectionInterceptor(context, prefManager))

        /*
                if(!prefManager.isSslCertificated){
                    try {
                        // Trust all certificates
                        val trustAllCerts = arrayOf<TrustManager>(object : X509TrustManager {
                            override fun checkClientTrusted(chain: Array<X509Certificate>, authType: String) {}
                            override fun checkServerTrusted(chain: Array<X509Certificate>, authType: String) {}
                            override fun getAcceptedIssuers(): Array<X509Certificate> = arrayOf()
                        })

                        // Install the all-trusting trust manager
                        val sslContext = SSLContext.getInstance("SSL")
                        sslContext.init(null, trustAllCerts, java.security.SecureRandom())
                        client.sslSocketFactory(sslContext.socketFactory, trustAllCerts[0] as X509TrustManager)
                        client.hostnameVerifier { _, _ -> true }
                    } catch (e: Exception) {
                        throw RuntimeException(e)
                    }
                }*/

        return client.build()
    }

    private fun bodyToString(request: RequestBody?): String? {
        return try {
            val buffer = Buffer()
            request?.writeTo(buffer)
            buffer.readUtf8()
        } catch (e: IOException) {
            "did not work"
        }
    }

    @Singleton
    @Provides
    fun provideRetrofit(okHttpClient: OkHttpClient): Retrofit = Retrofit.Builder()
        .addConverterFactory(GsonConverterFactory.create())
        .addCallAdapterFactory(RxJava2CallAdapterFactory.create())
        .baseUrl(BASE_URL)
        .client(okHttpClient)
        .build()

    @Singleton
    @Provides
    fun provideApiService(retrofit: Retrofit): ApiService = retrofit.create(ApiService::class.java)

    @Singleton
    @Provides
    fun providesRepository(apiService: ApiService) = ApiRepository(apiService)

}