package gg.gamerewards.ui.fragments.games

import android.content.Context
import android.content.Intent
import android.net.Uri
import android.os.Build
import android.view.LayoutInflater
import android.view.View
import android.view.ViewGroup
import androidx.core.view.doOnAttach
import androidx.core.view.doOnDetach
import androidx.core.view.isVisible
import androidx.recyclerview.widget.GridLayoutManager
import androidx.recyclerview.widget.LinearLayoutManager
import androidx.recyclerview.widget.PagerSnapHelper
import androidx.recyclerview.widget.RecyclerView
import androidx.recyclerview.widget.SnapHelper
import com.bumptech.glide.Glide
import gg.gamerewards.R
import gg.gamerewards.data.model.Profile
import gg.gamerewards.data.model.response.GamesFragmentOfferType
import gg.gamerewards.data.model.response.OfferGroup
import gg.gamerewards.databinding.ItemComponentFooterTermsBinding
import gg.gamerewards.databinding.ItemComponentMultiHorizontalAxisHorizontalCardBinding
import gg.gamerewards.databinding.ItemComponentMultiHorizontalAxisProgressCardBinding
import gg.gamerewards.databinding.ItemComponentMultiHorizontalAxisTwoHorizontalCardBinding
import gg.gamerewards.databinding.ItemComponentMultiHorizontalAxisVerticalCardBinding
import gg.gamerewards.databinding.ItemComponentMultiSlimHorizontalAxisSliderBinding
import gg.gamerewards.databinding.ItemComponentMultiVerticalAxisProgressBannerBinding
import gg.gamerewards.databinding.ItemComponentSingleBannerBinding
import gg.gamerewards.databinding.ItemComponentSingleCardTitleInsideBinding
import gg.gamerewards.databinding.ItemComponentSingleCardTitleOutsideBinding
import gg.gamerewards.databinding.ItemEmptyViewHolderBinding
import gg.gamerewards.ui.fragments.games.subAdapter.HorizontalCardWithCardOfferLabelListAdapter
import gg.gamerewards.ui.fragments.games.subAdapter.HorizontalTwoCardWithOfferLabelListAdapter
import gg.gamerewards.ui.fragments.games.subAdapter.MultiHorizontalAxisProgressCardAdapter
import gg.gamerewards.ui.fragments.games.subAdapter.MultiHorizontalViewSliderAdapter
import gg.gamerewards.ui.fragments.games.subAdapter.MultiSlimHorizontalAxisSliderAdapter
import gg.gamerewards.ui.fragments.games.subAdapter.MultiVerticalAxisProgressBannerAdapter
import gg.gamerewards.utils.CustomAutoSliderTimerTask
import gg.gamerewards.utils.PremiumOfferUIHelper
import gg.gamerewards.utils.RichieUtils
import gg.gamerewards.utils.dpToPx
import gg.gamerewards.utils.isLargeScreen
import gg.gamerewards.utils.setMargins
import java.util.Timer


/**
 * Created by Hasan Güler on 23.05.2023.
 */
internal interface GamesAdapterListener {
    fun onAction(action: String?, clickURL: String?, from: String?, fromTitle: String?)
    fun onEventLogRequest(eventName: Any, parameters: Map<Any, Any?>? = null)
}

internal class GamesAdapter(val context: Context) :
    RecyclerView.Adapter<RecyclerView.ViewHolder>() {
    private var items: ArrayList<OfferGroup> = arrayListOf()
    var listener: GamesAdapterListener? = null
    private var timer: Timer? = null
    private var onDestroyListener: (() -> Unit)? = null

    override fun onCreateViewHolder(parent: ViewGroup, viewType: Int): RecyclerView.ViewHolder {
        return when (viewType) {
            GamesFragmentOfferType.slim_slider.order -> {
                ComponentMultiSlimHorizontalAxisSliderHolder(
                    ItemComponentMultiSlimHorizontalAxisSliderBinding.inflate(
                        LayoutInflater.from(
                            parent.context
                        ), parent, false
                    )
                )
            }
            GamesFragmentOfferType.banner_with_title.order -> ComponentSingleBannerHolder(
                ItemComponentSingleBannerBinding.inflate(
                    LayoutInflater.from(
                        parent.context
                    ), parent, false
                )
            )

            GamesFragmentOfferType.banner_inside_label.order -> ComponentSingleCardTitleInsideViewHolder(
                ItemComponentSingleCardTitleInsideBinding.inflate(
                    LayoutInflater.from(
                        parent.context
                    ), parent, false
                )
            )
            GamesFragmentOfferType.horizontal_view_slider.order -> ComponentSliderCardViewHolder(
                ItemComponentMultiHorizontalAxisHorizontalCardBinding.inflate(
                    LayoutInflater.from(
                        parent.context
                    ), parent, false
                )
            )
            GamesFragmentOfferType.banner_outside_label.order -> ComponentSingleCardTitleOutsideViewHolder(
                ItemComponentSingleCardTitleOutsideBinding.inflate(
                    LayoutInflater.from(
                        parent.context
                    ), parent, false
                )
            )
            GamesFragmentOfferType.list_with_progress.order -> ComponentMultiVerticalAxisProgressBannerViewHolder(
                ItemComponentMultiVerticalAxisProgressBannerBinding.inflate(
                    LayoutInflater.from(
                        parent.context
                    ), parent, false
                )
            )
            GamesFragmentOfferType.list_tile.order -> ComponentMultiVerticalAxisProgressBannerViewHolder(
                ItemComponentMultiVerticalAxisProgressBannerBinding.inflate(
                    LayoutInflater.from(
                        parent.context
                    ), parent, false
                )
            )
            GamesFragmentOfferType.slider_with_inside_progress.order -> ComponentMultiHorizontalAxisProgressCardViewHolder(
                ItemComponentMultiHorizontalAxisProgressCardBinding.inflate(
                    LayoutInflater.from(
                        parent.context
                    ), parent, false
                )
            )
            GamesFragmentOfferType.slider_with_outside_progress.order -> ComponentMultiHorizontalAxisProgressCardViewHolder(
                ItemComponentMultiHorizontalAxisProgressCardBinding.inflate(
                    LayoutInflater.from(
                        parent.context
                    ), parent, false
                )
            )
            GamesFragmentOfferType.slider_without_progress.order -> ComponentMultiHorizontalAxisHorizontalCardViewHolder(
                ItemComponentMultiHorizontalAxisHorizontalCardBinding.inflate(
                    LayoutInflater.from(
                        parent.context
                    ), parent, false
                )
            )
            GamesFragmentOfferType.vertical_view_slider.order -> ComponentMultiHorizontalAxisVerticalCardViewHolder(
                ItemComponentMultiHorizontalAxisVerticalCardBinding.inflate(
                    LayoutInflater.from(
                        parent.context
                    ), parent, false
                )
            )
            GamesFragmentOfferType.two_column_list.order -> ComponentMultiVerticalAxisDoubleColumnGridCardViewHolder(
                ItemComponentMultiHorizontalAxisVerticalCardBinding.inflate(
                    LayoutInflater.from(
                        parent.context
                    ), parent, false
                )
            )
            GamesFragmentOfferType.horizontal_two_column_list.order -> ComponentMultiVerticalAxisTwoColumnHorizontalCardViewHolder(
                ItemComponentMultiHorizontalAxisTwoHorizontalCardBinding.inflate(
                    LayoutInflater.from(
                        parent.context
                    ), parent, false
                )
            )

            GamesFragmentOfferType.footer.order -> ComponentFooter(
                ItemComponentFooterTermsBinding.inflate(
                    LayoutInflater.from(
                        parent.context
                    ), parent, false
                )
            )

            else -> EmptyViewHolder(
                ItemEmptyViewHolderBinding.inflate(
                    LayoutInflater.from(
                        parent.context
                    ), parent, false
                )
            )
        }
    }

    override fun onBindViewHolder(holder: RecyclerView.ViewHolder, position: Int) {
        val currentItem = getCurrentItem(position)
        when (holder) {
            is ComponentMultiSlimHorizontalAxisSliderHolder -> holder.bind(currentItem, context, listener)
            is ComponentSingleBannerHolder -> holder.bind(currentItem, context, listener)//TODO
            is ComponentSingleCardTitleInsideViewHolder -> holder.bind(currentItem, context, listener,position)
            is ComponentSliderCardViewHolder -> holder.bind(currentItem, context, listener,position)
            is ComponentSingleCardTitleOutsideViewHolder -> holder.bind(currentItem, context, listener)//TODO
            is ComponentMultiVerticalAxisProgressBannerViewHolder -> holder.bind(currentItem, context, listener)
            is ComponentMultiHorizontalAxisProgressCardViewHolder -> holder.bind(currentItem, context,
                currentItem.viewStyle == GamesFragmentOfferType.slider_with_inside_progress, listener)
            is ComponentMultiHorizontalAxisHorizontalCardViewHolder -> holder.bind(currentItem, context, listener)
            is ComponentMultiHorizontalAxisVerticalCardViewHolder -> holder.bind(currentItem, context, listener)
            is ComponentMultiVerticalAxisDoubleColumnGridCardViewHolder -> holder.bind(currentItem, context, listener)
            is ComponentMultiVerticalAxisTwoColumnHorizontalCardViewHolder -> holder.bind(currentItem,context,listener)
            is ComponentFooter -> holder.bind(currentItem, context, listener)
            is EmptyViewHolder -> {}
        }
    }

    private fun getCurrentItem(position: Int): OfferGroup = items[position]

    override fun getItemViewType(position: Int): Int {
        return items[position].viewStyle?.order ?: -1
    }

    override fun getItemCount(): Int = items.count()

    private fun setAutoSlider(timer: Timer?,recyclerView: RecyclerView){
        timer?.let {
            val task = CustomAutoSliderTimerTask(recyclerView)
            it.schedule(task, 0, 4000)
        }
    }

    internal fun submitList(addedItems: ArrayList<OfferGroup>?) {
        this.items.clear()
        if (addedItems != null) {
            //Do not show adgem offer on Android 9 and below. Because adgem supports android 9 and above
            if (Build.VERSION.SDK_INT < Build.VERSION_CODES.P) {
                val filteredList = addedItems.filter { dashboardComponent ->
                    dashboardComponent.offers!!.any { offer ->
                        offer.action?.contains("adgemOffer") != true
                    }
                }.let { ArrayList(it) }
                this.items = filteredList
            } else
                this.items = ArrayList(addedItems)
        }

        notifyDataSetChanged()
    }

    /******************************* VIEW HOLDERS *******************************/

    internal class EmptyViewHolder(val binding: ItemEmptyViewHolderBinding) :
        RecyclerView.ViewHolder(binding.root)


    internal inner class ComponentMultiSlimHorizontalAxisSliderHolder(
        val binding: ItemComponentMultiSlimHorizontalAxisSliderBinding) :
        RecyclerView.ViewHolder(binding.root) {
        fun bind(
            model: OfferGroup,
            context: Context,
            listener: GamesAdapterListener?,
        ) {
            with(binding.rvSlider)
            {
                val currentAdapter = adapter
                if (currentAdapter is MultiSlimHorizontalAxisSliderAdapter){
                    currentAdapter.submitList(model.offers)
                }
                else {
                    val mAdapter = MultiSlimHorizontalAxisSliderAdapter(context).apply {
                        submitList(model.offers)
                    }
                    mAdapter.listener = listener
                    mAdapter.from = model.viewStyle?.name
                    mAdapter.fromTitle = model.groupTitle
                    adapter = mAdapter

                    val lManager =
                        LinearLayoutManager(context, LinearLayoutManager.HORIZONTAL, false)
                    layoutManager = lManager

                    if(model.offers?.isNotEmpty() == true && (model.offers?.size ?: 0) > 1){
                        val snapHelper: SnapHelper = PagerSnapHelper()
                        snapHelper.attachToRecyclerView(this)
                        binding.recyclerIndicator.attachToRecyclerView(this)
                    }
                    else{
                        binding.recyclerIndicator.visibility = View.GONE
                    }
                }
            }
        }
    }

    internal class ComponentSingleBannerHolder(val binding: ItemComponentSingleBannerBinding) :
        RecyclerView.ViewHolder(binding.root) {

        fun bind(
            model: OfferGroup,
            context: Context,
            listener: GamesAdapterListener?
        ) {
            try {
                val offer = model.offers?.first()
                Glide.with(context)
                    .load(offer?.horizontalCardImageUrl)
                    .into(binding.ivBackground)

                binding.tvTitle.text = offer?.title
                binding.tvSubtitle.text = offer?.subtitle
                binding.cvActionButton.setCardBackgroundColor(RichieUtils.parseColor("#${offer?.actionButtonBackgroundColor}"))
                binding.tvActionButton.text = offer?.actionButtonTitle
                binding.tvActionButton.setTextColor(RichieUtils.parseColor("#${offer?.actionButtonTextColor}"))
                binding.root.setOnClickListener {
                    listener?.onAction(offer?.action, offer?.clickUrl, model.viewStyle?.name, model.groupTitle)
                }
            } catch (ignored: java.lang.Exception) {
            }
        }
    }

    internal class ComponentFooter(val binding: ItemComponentFooterTermsBinding) :
        RecyclerView.ViewHolder(binding.root) {

        fun bind(
            model: OfferGroup,
            context: Context,
            listener: GamesAdapterListener?
        ) {
            try {
                // Set the URL manually
                val urlTermsAndConditions = "https://blog.gamerewards.gg/terms-of-service/";
                val urlPrivacyPolicy = "https://blog.gamerewards.gg/privacy-policy/"

                binding.tvTermsAndConditions.setOnClickListener{
                    val browserIntent = Intent(Intent.ACTION_VIEW, Uri.parse(urlTermsAndConditions));
                    context.startActivity(browserIntent)
                }
                binding.tvPrivacyPolicy.setOnClickListener{
                    val browserIntent = Intent(Intent.ACTION_VIEW, Uri.parse(urlPrivacyPolicy));
                    context.startActivity(browserIntent)
                }

                binding.tvSupportMail.setOnClickListener {
                    val emailIntent = Intent.createChooser(
                        Intent().apply {
                            action = Intent.ACTION_SEND
                            type = "message/rfc822"
                            putExtra(Intent.EXTRA_EMAIL, binding.tvSupportMail.text)
                            putExtra(Intent.EXTRA_SUBJECT, "Support Mail")
                            flags = Intent.FLAG_GRANT_READ_URI_PERMISSION
                        },
                        "Send E-mail"
                    )
                    context.startActivity(emailIntent)
                }

            } catch (ignored: java.lang.Exception) {
            }
        }
    }

    internal inner class ComponentSingleCardTitleInsideViewHolder(
        val binding: ItemComponentSingleCardTitleInsideBinding
    ) :
        RecyclerView.ViewHolder(binding.root) {

        fun bind(
            model: OfferGroup,
            context: Context,
            listener: GamesAdapterListener?,
            position: Int
        ) {
            try {
                binding.tvTitle.isVisible = model.showTitle ?: false
                binding.tvTitle.text = model.groupTitle

                model.offers?.first()?.let { offer ->
                    Glide.with(context)
                        .load(offer.horizontalCardImageUrl)
                        .into(binding.ivBackground)

                    binding.cvCardLabel.isVisible = !offer.cardLabel.isNullOrEmpty()
                    binding.tvCardLabel.text = offer.cardLabel
                    binding.tvCardLabel.setTextColor(RichieUtils.parseColor("#${offer.cardLabelTextColor}"))
                    binding.llCardLabelContainer.setBackgroundColor(RichieUtils.parseColor("#${offer.cardLabelBackgroundColor}"))
                    Glide.with(context)
                        .load(offer.cardLabelIconUrl)
                        .into(binding.ivCardLabelIcon)

                    binding.layoutPremium.apply {
                        PremiumOfferUIHelper().setUIForPremiumOffer(context,false,offer,clPremiumContainer,tvUnlockCoinAmount,flPremiumLock,llPremiumUnlocked,progressBar,premiumOfferLottieView)
                    }

                    val isPremiumCoinEnough =
                        (Profile.profileData?.totalEarnedCoins ?: 0) >= (offer.premiumUnlockCoinAmount ?: 0)
                    binding.root.setOnClickListener {
                        if (offer.isPremiumOffer != true || isPremiumCoinEnough)
                            listener?.onAction("${offer.action}/${offer.isPremiumOffer}", offer.clickUrl, model.viewStyle?.name, model.groupTitle)
                    }
                }
            } catch (e: Exception) {
                e.printStackTrace()
            }
        }
    }

    internal inner class ComponentSliderCardViewHolder(
        val binding: ItemComponentMultiHorizontalAxisHorizontalCardBinding
    ) :
        RecyclerView.ViewHolder(binding.root) {

        fun bind(
            model: OfferGroup,
            context: Context,
            listener: GamesAdapterListener?,
            position: Int
        ) {
            try {
                binding.tvTitle.isVisible = model.showTitle ?: false
                binding.tvTitle.text = model.groupTitle

                with(binding.rvGames)
                {

                    val currentAdapter = adapter
                    binding.rvGames.doOnAttach {
                        timer = Timer()
                        setAutoSlider(timer,this)
                    }
                    binding.rvGames.doOnDetach {
                        timer?.cancel()
                        timer = null
                    }
                    if (currentAdapter is MultiHorizontalViewSliderAdapter)
                        currentAdapter.submitList(model.offers)
                    else {
                        val mAdapter = MultiHorizontalViewSliderAdapter(
                            context,
                            model.viewStyle?.name,
                            model.groupTitle
                        ).apply {
                            submitList(model.offers)
                        }
                        mAdapter.listener = listener
                        adapter = mAdapter

                        layoutManager = object : LinearLayoutManager(context, HORIZONTAL, false) {
                            override fun checkLayoutParams(lp: RecyclerView.LayoutParams): Boolean {
                                lp.width = (width * if (!context.isLargeScreen()) 0.8 else 0.58).toInt()
                                return true
                            }
                        }
                    }

                    onDestroyListener = {
                        timer?.cancel()
                        timer = null
                        if (adapter is MultiHorizontalViewSliderAdapter){
                            (adapter as MultiHorizontalViewSliderAdapter).listener = null
                        }
                    }
                }
            } catch (e: Exception) {
                e.printStackTrace()
            }
        }
    }

    internal class ComponentSingleCardTitleOutsideViewHolder(val binding: ItemComponentSingleCardTitleOutsideBinding) :
        RecyclerView.ViewHolder(binding.root) {

        fun bind(
            model: OfferGroup,
            context: Context,
            listener: GamesAdapterListener?
        ) {
            try {
                binding.tvTitle.isVisible = model.showTitle ?: false
                binding.tvTitle.text = model.groupTitle

                model.offers?.first()?.let { offer ->
                    Glide.with(context)
                        .load(offer.horizontalCardImageUrl)
                        .into(binding.ivBackground)

                    binding.cvCardLabel.isVisible = !offer.cardLabel.isNullOrEmpty()
                    binding.tvCardLabel.text = offer.cardLabel
                    binding.tvCardLabel.setTextColor(RichieUtils.parseColor("#${offer.cardLabelTextColor}"))
                    binding.llCardLabelContainer.setBackgroundColor(RichieUtils.parseColor("#${offer.cardLabelBackgroundColor}"))
                    Glide.with(context)
                        .load(offer.cardLabelIconUrl)
                        .into(binding.ivCardLabelIcon)

                    /*------------------ Premium offer UI --------------*/
                    val isPremiumCoinEnough =
                        (Profile.profileData?.totalEarnedCoins ?: 0) >= (offer.premiumUnlockCoinAmount ?: 0)
                    val isOfferPremium = (offer.isPremiumOffer ?: false) && (!(offer.isPremiumUnlocked ?: true) || !isPremiumCoinEnough)
                    binding.cvLayoutPremiumContainer.isVisible = isOfferPremium
                    binding.layoutPremium.apply {
                        if (isOfferPremium){
                            if (isPremiumCoinEnough)
                                premiumOfferLottieView.setMargins(bottom = context.dpToPx(40))

                            premiumOfferLottieView.setAnimation(
                                if (isPremiumCoinEnough) R.raw.premium_offer_unlock_anim else R.raw.premium_offer_lock_anim)
                            premiumOfferLottieView.playAnimation()
                        }
                        progressBar.progress = offer.premiumCoinViewProgress
                        tvUnlockCoinAmount.text = context.getString(R.string.premium_offer_unlock_text_horizontal,offer.premiumUnlockCoinAmount.toString())
                        isPremiumCoinEnough.apply {
                            flPremiumLock.isVisible = !this
                            llPremiumUnlocked.isVisible = this
                        }
                    }
                    /*--------------------------------------------------*/
                    binding.root.setOnClickListener {
                        if (offer.isPremiumOffer != true || isPremiumCoinEnough)
                            listener?.onAction("${offer.action}/${offer.isPremiumOffer}", offer.clickUrl, model.viewStyle?.name, model.groupTitle)
                    }
                }
            } catch (ignored: Exception) { }
        }
    }

    internal class ComponentMultiVerticalAxisProgressBannerViewHolder(val binding: ItemComponentMultiVerticalAxisProgressBannerBinding) :
        RecyclerView.ViewHolder(binding.root) {

        fun bind(
            model: OfferGroup,
            context: Context,
            listener: GamesAdapterListener?
        ) {
            try {
                binding.tvTitle.isVisible = model.showTitle ?: false
                binding.tvTitle.text = model.groupTitle

                with(binding.rvGames)
                {
                    val currentAdapter = adapter
                    if (currentAdapter is MultiVerticalAxisProgressBannerAdapter)
                        currentAdapter.submitList(model.offers)
                    else {
                        val mAdapter = MultiVerticalAxisProgressBannerAdapter(context).apply {
                            submitList(model.offers)
                        }
                        mAdapter.listener = listener
                        adapter = mAdapter

                        val lManager =
                            LinearLayoutManager(context, LinearLayoutManager.VERTICAL, false)
                        layoutManager = lManager
                    }
                }
            } catch (ignored: Exception) { }
        }
    }

    internal class ComponentMultiHorizontalAxisProgressCardViewHolder(val binding: ItemComponentMultiHorizontalAxisProgressCardBinding) :
        RecyclerView.ViewHolder(binding.root) {

        fun bind(
            model: OfferGroup,
            context: Context,
            isTitleInside: Boolean,
            listener: GamesAdapterListener?
        ) {
            try {
                binding.tvTitle.isVisible = model.showTitle ?: false
                binding.tvTitle.text = model.groupTitle

                with(binding.rvGames)
                {
                    val currentAdapter = adapter
                    if (currentAdapter is MultiHorizontalAxisProgressCardAdapter)
                        currentAdapter.submitList(model.offers)
                    else {
                        val mAdapter = MultiHorizontalAxisProgressCardAdapter(context, isTitleInside,model.viewStyle?.name,model.groupTitle).apply {
                            submitList(model.offers)
                        }
                        mAdapter.listener = listener
                        adapter = mAdapter

                        layoutManager = object : LinearLayoutManager(context, HORIZONTAL, false) {
                            override fun checkLayoutParams(lp: RecyclerView.LayoutParams): Boolean {
                                lp.width = (width * if (!context.isLargeScreen()) 0.38 else 0.27).toInt()
                                return true
                            }
                        }
                    }
                }
            } catch (ignored: Exception) { }
        }
    }

    internal class ComponentMultiHorizontalAxisHorizontalCardViewHolder(val binding: ItemComponentMultiHorizontalAxisHorizontalCardBinding) :
        RecyclerView.ViewHolder(binding.root) {

        fun bind(
            model: OfferGroup,
            context: Context,
            listener: GamesAdapterListener?) {
            try {
                binding.tvTitle.isVisible = model.showTitle ?: false
                binding.tvTitle.text = model.groupTitle

                with(binding.rvGames)
                {
                    val currentAdapter = adapter
                    if (currentAdapter is HorizontalCardWithCardOfferLabelListAdapter)
                        currentAdapter.submitList(model.offers)
                    else {
                        val mAdapter = HorizontalCardWithCardOfferLabelListAdapter(context, true,
                            from = model.viewStyle?.name, fromTitle = model.groupTitle
                        ).apply {
                            submitList(model.offers)
                        }
                        mAdapter.listener = listener
                        adapter = mAdapter
                        layoutManager = object : LinearLayoutManager(context, HORIZONTAL, false) {
                            override fun checkLayoutParams(lp: RecyclerView.LayoutParams): Boolean {
                                lp.width = (width * if (!context.isLargeScreen()) 0.75 else 0.55).toInt()
                                return true
                            }
                        }
                    }
                }
            } catch (ignored: Exception) { }
        }
    }

    internal class ComponentMultiHorizontalAxisVerticalCardViewHolder(val binding: ItemComponentMultiHorizontalAxisVerticalCardBinding) :
        RecyclerView.ViewHolder(binding.root) {

        fun bind(
            model: OfferGroup,
            context: Context,
            listener: GamesAdapterListener?) {
            try {
                binding.tvTitle.isVisible = model.showTitle ?: false
                binding.tvTitle.text = model.groupTitle

                with(binding.rvGames)
                {
                    val currentAdapter = adapter
                    if (currentAdapter is HorizontalCardWithCardOfferLabelListAdapter)
                        currentAdapter.submitList(model.offers)
                    else {
                        val mAdapter = HorizontalCardWithCardOfferLabelListAdapter(context, false,
                            from = model.viewStyle?.name, fromTitle = model.groupTitle).apply {
                            submitList(model.offers)
                        }
                        mAdapter.listener = listener
                        adapter = mAdapter
                        layoutManager = object : LinearLayoutManager(context, HORIZONTAL, false) {
                            override fun checkLayoutParams(lp: RecyclerView.LayoutParams): Boolean {
                                lp.width = (width * if (!context.isLargeScreen()) 0.38 else 0.27).toInt()
                                return true
                            }
                        }
                    }
                }
            } catch (ignored: Exception) { }
        }
    }

    internal class ComponentMultiVerticalAxisDoubleColumnGridCardViewHolder(val binding: ItemComponentMultiHorizontalAxisVerticalCardBinding) :
        RecyclerView.ViewHolder(binding.root) {

        fun bind(
            model: OfferGroup,
            context: Context,
            listener: GamesAdapterListener?) {
            try {
                binding.tvTitle.isVisible = model.showTitle ?: false
                binding.tvTitle.text = model.groupTitle

                with(binding.rvGames)
                {
                    val currentAdapter = adapter
                    if (currentAdapter is HorizontalCardWithCardOfferLabelListAdapter)
                        currentAdapter.submitList(model.offers)
                    else {
                        val mAdapter = HorizontalCardWithCardOfferLabelListAdapter(context,
                            isHorizontalCard = false,
                            isDoubleCard = true,
                            from = model.viewStyle?.name, fromTitle = model.groupTitle
                        ).apply {
                            submitList(model.offers)
                        }
                        mAdapter.listener = listener
                        adapter = mAdapter
                        layoutManager = GridLayoutManager(context, 2)
                    }
                }
            } catch (ignored: Exception) { }
        }
    }

    internal class ComponentMultiVerticalAxisTwoColumnHorizontalCardViewHolder(val binding: ItemComponentMultiHorizontalAxisTwoHorizontalCardBinding) :
        RecyclerView.ViewHolder(binding.root) {

        fun bind(
            model: OfferGroup,
            context: Context,
            listener: GamesAdapterListener?) {
            try {
                with(binding.rvGames)
                {
                    val currentAdapter = adapter
                    if (currentAdapter is HorizontalTwoCardWithOfferLabelListAdapter)
                        currentAdapter.submitList(model.offers)
                    else {
                        val mAdapter = HorizontalTwoCardWithOfferLabelListAdapter(context,
                            from = model.viewStyle?.name, fromTitle = model.groupTitle
                        ).apply {
                            submitList(model.offers)
                        }
                        mAdapter.listener = listener
                        adapter = mAdapter
                        layoutManager = GridLayoutManager(context, 2)

                    }
                }
            } catch (ignored: Exception) { }
        }
    }
}
