package gg.gamerewards.utils

import android.content.Context
import android.content.res.Configuration
import android.net.ConnectivityManager
import android.net.NetworkCapabilities
import android.os.Handler
import android.os.Looper
import android.util.Log
import android.view.View
import android.view.ViewGroup
import androidx.core.view.marginBottom
import androidx.core.view.marginLeft
import androidx.core.view.marginRight
import androidx.core.view.marginTop
import com.scottyab.rootbeer.RootBeer
import gg.gamerewards.data.api.NoInternetConnectivityException
import gg.gamerewards.data.local.PrefManager
import gg.gamerewards.ui.base.BaseNavigator
import gg.gamerewards.ui.base.BaseViewModel
import kotlinx.coroutines.CoroutineExceptionHandler
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.SupervisorJob
import kotlinx.coroutines.launch
import kotlin.contracts.ExperimentalContracts
import kotlin.reflect.KClass

internal fun Context.isInternetConnected(): Boolean {
    val connectivityManager = getSystemService(Context.CONNECTIVITY_SERVICE) as ConnectivityManager
    val networkCapabilities = connectivityManager.activeNetwork ?: return false
    val activeNetwork = connectivityManager.getNetworkCapabilities(networkCapabilities) ?: return false

    return when {
        activeNetwork.hasTransport(NetworkCapabilities.TRANSPORT_WIFI) -> true
        activeNetwork.hasTransport(NetworkCapabilities.TRANSPORT_CELLULAR) -> true
        // For other devices which can connect through Ethernet or VPN
        activeNetwork.hasTransport(NetworkCapabilities.TRANSPORT_ETHERNET) -> true
        else -> false
    }
}

internal fun Context.isVpnDetected(prefManager: PrefManager): Boolean {
    val connectivityManager = this.getSystemService(ConnectivityManager::class.java) as ConnectivityManager
    val activeNetwork = connectivityManager.activeNetwork
    val networkCapabilities = connectivityManager.getNetworkCapabilities(activeNetwork)
    return (networkCapabilities?.hasTransport(NetworkCapabilities.TRANSPORT_VPN) ?: false)
}

internal fun Context.isRooted(prefManager: PrefManager): Boolean{
    val rootBeer = RootBeer(applicationContext)
    rootBeer.setLogging(false)
    return rootBeer.isRooted
}

internal fun BaseViewModel<*>.getIOScopeForAPICall(onError: (() -> Unit)? = null, handleAPIExceptions: Boolean = true, execute: suspend () -> Unit): CoroutineScope {
    return CoroutineScope(Dispatchers.IO + SupervisorJob() + CoroutineExceptionHandler { _, throwable ->
        Handler(Looper.getMainLooper()).post {
            Log.e("APICall", throwable.message ?: "")
            (navigator as? BaseNavigator)?.hideLoading() //Hide loading on any exception
            onError?.invoke()
            if (handleAPIExceptions) {
                if (throwable is NoInternetConnectivityException) {
                    (navigator as? BaseNavigator)?.showConnectionWarning(true, onRetry = {
                        //RECALL
                        getIOScopeForAPICall(onError, handleAPIExceptions, execute)
                    })
                } else {
                    (navigator as? BaseNavigator)?.handleAPIException(throwable)
                }
            }
        }
    }).apply {
        launch {
            execute()
        }
    }
}

internal fun View.setMargins(
    left: Int = this.marginLeft,
    top: Int = this.marginTop,
    right: Int = this.marginRight,
    bottom: Int = this.marginBottom,
) {
    layoutParams = (layoutParams as ViewGroup.MarginLayoutParams).apply {
        setMargins(left, top, right, bottom)
    }
}
internal fun Context.dpToPx(dp: Int): Int {
    return (dp * resources.displayMetrics.density).toInt()
}


internal fun Context.isLargeScreen(): Boolean{
    return try {
        resources.configuration.screenLayout and Configuration.SCREENLAYOUT_SIZE_MASK >= Configuration.SCREENLAYOUT_SIZE_LARGE
    }catch (_:Exception) {
        false
    }
}

@OptIn(ExperimentalContracts::class)
inline fun <T : Any, reified R : T> T.asInstance(clazz: KClass<R>): R? {
    return if (this is R)
        this
    else
        null
}