/*
 * Copyright (C) 2012 United States Government as represented by the Administrator of the
 * National Aeronautics and Space Administration.
 * All Rights Reserved.
 */
package gov.nasa.worldwind.symbology.milstd2525;

import gov.nasa.worldwind.Configuration;
import gov.nasa.worldwind.avlist.AVKey;
import gov.nasa.worldwind.geom.Position;
import gov.nasa.worldwind.layers.*;
import gov.nasa.worldwind.render.Renderable;
import gov.nasa.worldwind.symbology.*;
import gov.nasa.worldwind.util.WWUtil;
import gov.nasa.worldwindx.examples.ApplicationTemplate;

import javax.swing.*;
import java.awt.*;
import java.awt.event.*;
import java.util.*;
import java.util.List;

/**
 * @author dcollins
 * @version $Id: AllTacticalSymbols.java 1171 2013-02-11 21:45:02Z dcollins $
 */
public class AllTacticalSymbols extends ApplicationTemplate
{
    public static class AppFrame extends ApplicationTemplate.AppFrame
    {
        public AppFrame()
        {
            super(true, true, false);

            this.addSymbols();
            this.addSymbolControls();

            // Size the World Window to provide enough screen space for the symbols and center the World Window on the
            // screen.
            Dimension size = new Dimension(1800, 1000);
            this.setPreferredSize(size);
            this.pack();
            WWUtil.alignComponent(null, this, AVKey.CENTER);
        }

        protected void addSymbols()
        {
            this.addSymbolLayer(emsSymbolIds, "EMS Symbols");
            this.addSymbolLayer(sigintSymbolIds, "SIGINT Symbols");
            this.addSymbolLayer(stbopsSymbolIds, "STBOPS Symbols");
            this.addSymbolLayer(warSymbolIds, "WAR Symbols");
        }

        protected void addSymbolLayer(List<String> nameList, String layerName)
        {
            RenderableLayer layer = new RenderableLayer();
            layer.setEnabled(false);
            layer.setName(layerName);

            double startLat = 40;
            double startLon = -120;
            double dLat = -0.1;
            double dLon = 0.1;
            double lat = startLat;
            double lon = startLon;

            for (String name : nameList)
            {
                for (String symbolId : this.makeSymbolIds(name))
                {
                    Position pos = Position.fromDegrees(lat, lon, 2000);
                    MilStd2525TacticalSymbol symbol = new MilStd2525TacticalSymbol(symbolId, pos);
                    symbol.setValue(AVKey.DISPLAY_NAME, symbolId);
                    symbol.setShowLocation(false);
                    layer.addRenderable(symbol);
                    lon += dLon;
                }

                lat += dLat;
                lon = startLon;
            }

            // Add the symbol layer to the World Wind model.
            this.getWwd().getModel().getLayers().add(layer);

            // Update the layer panel to display the symbol layer.
            this.getLayerPanel().update(this.getWwd());
        }

        protected List<String> makeSymbolIds(String name)
        {
            ArrayList<String> symbolIds = new ArrayList<String>();
            StringBuilder sb = new StringBuilder(name);

            for (String si : standardIdentities)
            {
                sb.setCharAt(1, si.charAt(0));
                sb.setCharAt(3, 'p'); // Present
                symbolIds.add(sb.toString().toUpperCase());
            }

            return symbolIds;
        }

        protected void addSymbolControls()
        {
            Box box = Box.createVerticalBox();
            box.setBorder(BorderFactory.createEmptyBorder(20, 20, 20, 20));

            // Create a check box that toggles the frame visibility for all symbols.
            JCheckBox cb = new JCheckBox("Show Frame", true);
            cb.addActionListener(new ActionListener()
            {
                public void actionPerformed(ActionEvent actionEvent)
                {
                    boolean tf = ((JCheckBox) actionEvent.getSource()).isSelected();

                    for (Layer layer : getWwd().getModel().getLayers())
                    {
                        if (!(layer instanceof RenderableLayer))
                            continue;

                        for (Renderable r : ((RenderableLayer) layer).getRenderables())
                        {
                            if (r instanceof TacticalSymbol)
                                ((MilStd2525TacticalSymbol) r).setShowFrame(tf);
                            getWwd().redraw(); // Cause the World Window to refresh in order to make these changes visible.
                        }
                    }
                }
            });
            cb.setAlignmentX(JComponent.LEFT_ALIGNMENT);
            box.add(Box.createVerticalStrut(10));
            box.add(cb);

            // Create a check box that toggles the fill visibility for all symbols.
            cb = new JCheckBox("Show Fill", true);
            cb.addActionListener(new ActionListener()
            {
                public void actionPerformed(ActionEvent actionEvent)
                {
                    boolean tf = ((JCheckBox) actionEvent.getSource()).isSelected();

                    for (Layer layer : getWwd().getModel().getLayers())
                    {
                        if (!(layer instanceof RenderableLayer))
                            continue;

                        for (Renderable r : ((RenderableLayer) layer).getRenderables())
                        {
                            if (r instanceof TacticalSymbol)
                                ((MilStd2525TacticalSymbol) r).setShowFill(tf);
                            getWwd().redraw(); // Cause the World Window to refresh in order to make these changes visible.
                        }
                    }
                }
            });
            cb.setAlignmentX(JComponent.LEFT_ALIGNMENT);
            box.add(Box.createVerticalStrut(10));
            box.add(cb);

            // Create a check box that toggles the icon visibility for all symbols.
            cb = new JCheckBox("Show Icon", true);
            cb.addActionListener(new ActionListener()
            {
                public void actionPerformed(ActionEvent actionEvent)
                {
                    boolean tf = ((JCheckBox) actionEvent.getSource()).isSelected();

                    for (Layer layer : getWwd().getModel().getLayers())
                    {
                        if (!(layer instanceof RenderableLayer))
                            continue;

                        for (Renderable r : ((RenderableLayer) layer).getRenderables())
                        {
                            if (r instanceof TacticalSymbol)
                                ((MilStd2525TacticalSymbol) r).setShowIcon(tf);
                            getWwd().redraw(); // Cause the World Window to refresh in order to make these changes visible.
                        }
                    }
                }
            });
            cb.setAlignmentX(JComponent.LEFT_ALIGNMENT);
            box.add(Box.createVerticalStrut(10));
            box.add(cb);

            // Create a check box that toggles the icon visibility for all symbols.
            cb = new JCheckBox("Present/anticipated", true);
            cb.addActionListener(new ActionListener()
            {
                public void actionPerformed(ActionEvent actionEvent)
                {
                    String status = ((JCheckBox) actionEvent.getSource()).isSelected()
                        ? SymbologyConstants.STATUS_PRESENT : SymbologyConstants.STATUS_ANTICIPATED;

                    for (Layer layer : getWwd().getModel().getLayers())
                    {
                        if (!(layer instanceof RenderableLayer))
                            continue;

                        for (Renderable r : ((RenderableLayer) layer).getRenderables())
                        {
                            if (r instanceof TacticalSymbol)
                                ((MilStd2525TacticalSymbol) r).setStatus(status);
                            getWwd().redraw(); // Cause the World Window to refresh in order to make these changes visible.
                        }
                    }
                }
            });
            cb.setAlignmentX(JComponent.LEFT_ALIGNMENT);
            box.add(Box.createVerticalStrut(10));
            box.add(cb);

            this.getLayerPanel().add(box, BorderLayout.SOUTH);
        }
    }

    public static void main(String[] args)
    {
        Configuration.setValue(AVKey.INITIAL_LATITUDE, 40);
        Configuration.setValue(AVKey.INITIAL_LONGITUDE, -119.85);
        Configuration.setValue(AVKey.INITIAL_ALTITUDE, 77000);

        start("World Wind All MIL-STD-2525 Tactical Symbols", AppFrame.class);
    }

    protected static List<String> standardIdentities = Arrays.asList(
        "u",
        "f",
        "n",
        "h"
    );

    protected static List<String> emsSymbolIds = Arrays.asList(
        "e-f------------",
        "e-f-a-----h----",
        "e-f-aa----h----",
        "e-f-ab----h----",
        "e-f-ac----h----",
        "e-f-ad----h----",
        "e-f-ae----h----",
        "e-f-af----h----",
        "e-f-ag----h----",
        "e-f-b-----h----",
        "e-f-ba---------",
        "e-f-bb----h----",
        "e-f-bc----h----",
        "e-f-bd----h----",
        "e-f-be----h----",
        "e-f-bf----h----",
        "e-f-c-----h----",
        "e-f-ca----h----",
        "e-f-cb----h----",
        "e-f-cc----h----",
        "e-f-cd----h----",
        "e-f-ce----h----",
        "e-f-cf----h----",
        "e-f-cg----h----",
        "e-f-ch----h----",
        "e-f-ci----h----",
        "e-f-cj----h----",
        "e-f-d-----h----",
        "e-f-da----h----",
        "e-f-db----h----",
        "e-f-ea----h----",
        "e-f-eb----h----",
        "e-f-ee----h----",
        "e-f-f-----h----",
        "e-f-g-----h----",
        "e-f-ga----h----",
        "e-f-h-----h----",
        "e-f-ha----h----",
        "e-f-hb----h----",
        "e-f-i-----h----",
        "e-f-ia----h----",
        "e-f-ib----h----",
        "e-f-ic----h----",
        "e-f-id----h----",
        "e-f-j-----h----",
        "e-f-ja----h----",
        "e-f-jb----h----",
        "e-f-jc----h----",
        "e-f-k-----h----",
        "e-f-kb----h----",
        "e-f-la----h----",
        "e-f-ld----h----",
        "e-f-le----h----",
        "e-f-lf----h----",
        "e-f-lh----h----",
        "e-f-lj----h----",
        "e-f-lk----h----",
        "e-f-lm----h----",
        "e-f-lo----h----",
        "e-f-lp----h----",
        "e-f-ma---------",
        "e-f-mb----h----",
        "e-f-mc---------",
        "e-f-md----h----",
        "e-f-me----h----",
        "e-f-mf----h----",
        "e-f-mg----h----",
        "e-f-mh----h----",
        "e-f-mi----h----",
        "e-i------------",
        "e-i-a----------",
        "e-i-ac---------",
        "e-i-b----------",
        "e-i-ba---------",
        "e-i-bc---------",
        "e-i-bd---------",
        "e-i-bf---------",
        "e-i-c----------",
        "e-i-ca---------",
        "e-i-cb---------",
        "e-i-cc---------",
        "e-i-cd---------",
        "e-i-ce---------",
        "e-i-cf---------",
        "e-i-cg---------",
        "e-i-ch---------",
        "e-i-d----------",
        "e-i-da---------",
        "e-i-db---------",
        "e-i-dc---------",
        "e-i-dd---------",
        "e-i-de---------",
        "e-i-df---------",
        "e-i-dg---------",
        "e-i-dh---------",
        "e-i-di---------",
        "e-i-dj---------",
        "e-i-dk---------",
        "e-i-dl---------",
        "e-i-dm---------",
        "e-i-dn---------",
        "e-i-do---------",
        "e-i-e----------",
        "e-i-ea---------",
        "e-i-f----------",
        "e-i-fa---------",
        "e-i-g----------",
        "e-i-ga---------",
        "e-i-gb---------",
        "e-i-h----------",
        "e-i-ha---------",
        "e-o------------",
        "e-o-a----------",
        "e-o-aa---------",
        "e-o-ab---------",
        "e-o-ac----h----",
        "e-o-ad----h----",
        "e-o-ae---------",
        "e-o-af---------",
        "e-o-ag----h----",
        "e-o-aj----h----",
        "e-o-ak----h----",
        "e-o-al----h----",
        "e-o-am----h----",
        "e-o-b----------",
        "e-o-ba---------",
        "e-o-bb---------",
        "e-o-bc----h----",
        "e-o-bd---------",
        "e-o-be----h----",
        "e-o-bf----h----",
        "e-o-bg----h----",
        "e-o-bh----h----",
        "e-o-bi----h----",
        "e-o-bj---------",
        "e-o-bk----h----",
        "e-o-bl----h----",
        "e-o-c----------",
        "e-o-ca---------",
        "e-o-cb---------",
        "e-o-cc---------",
        "e-o-cd----h----",
        "e-o-ce----h----",
        "e-o-d----------",
        "e-o-da---------",
        "e-o-db---------",
        "e-o-dc----h----",
        "e-o-dd---------",
        "e-o-dda--------",
        "e-o-ddb--------",
        "e-o-ddc--------",
        "e-o-de---------",
        "e-o-dea--------",
        "e-o-deb--------",
        "e-o-dec---h----",
        "e-o-df---------",
        "e-o-dfa--------",
        "e-o-dfb--------",
        "e-o-dfc---h----",
        "e-o-dg---------",
        "e-o-dga--------",
        "e-o-dgb--------",
        "e-o-dgc---h----",
        "e-o-dh---------",
        "e-o-dha--------",
        "e-o-dhb--------",
        "e-o-dhc---h----",
        "e-o-di---------",
        "e-o-dia--------",
        "e-o-dib--------",
        "e-o-dic---h----",
        "e-o-dj---------",
        "e-o-djb--------",
        "e-o-djc---h----",
        "e-o-dk----h----",
        "e-o-dl---------",
        "e-o-dla--------",
        "e-o-dlb--------",
        "e-o-dlc---h----",
        "e-o-dm---------",
        "e-o-dma--------",
        "e-o-dmb--------",
        "e-o-dmc---h----",
        "e-o-dn---------",
        "e-o-dna--------",
        "e-o-dnc---h----",
        "e-o-do---------",
        "e-o-doa--------",
        "e-o-dob--------",
        "e-o-doc---h----",
        "e-o-ea---------",
        "e-o-eb---------",
        "e-o-ec---------",
        "e-o-ed---------",
        "e-o-ee---------");

    protected static List<String> sigintSymbolIds = Arrays.asList(
        "i-a------------",
        "i-a-s----------",
        "i-a-sc---------",
        "i-a-scc--------",
        "i-a-sco--------",
        "i-a-scp--------",
        "i-a-scs--------",
        "i-a-sr---------",
        "i-a-srai-------",
        "i-a-sras-------",
        "i-a-src--------",
        "i-a-srd--------",
        "i-a-sre--------",
        "i-a-srf--------",
        "i-a-sri--------",
        "i-a-srma-------",
        "i-a-srmd-------",
        "i-a-srmf-------",
        "i-a-srmg-------",
        "i-a-srmt-------",
        "i-a-srta-------",
        "i-a-srti-------",
        "i-a-srtt-------",
        "i-a-sru--------",
        "i-g------------",
        "i-g-s----------",
        "i-g-sc---------",
        "i-g-scc--------",
        "i-g-sco--------",
        "i-g-scp--------",
        "i-g-scs--------",
        "i-g-sct--------",
        "i-g-sr---------",
        "i-g-sraa-------",
        "i-g-srat-------",
        "i-g-srb--------",
        "i-g-srca-------",
        "i-g-srcs-------",
        "i-g-srd--------",
        "i-g-sre--------",
        "i-g-srf--------",
        "i-g-srh--------",
        "i-g-sri--------",
        "i-g-srma-------",
        "i-g-srmf-------",
        "i-g-srmg-------",
        "i-g-srmm-------",
        "i-g-srmt-------",
        "i-g-srs--------",
        "i-g-srta-------",
        "i-g-srti-------",
        "i-g-srtt-------",
        "i-g-sru--------",
        "i-p------------",
        "i-p-s----------",
        "i-p-sc---------",
        "i-p-scd--------",
        "i-p-sr---------",
        "i-p-srd--------",
        "i-p-sre--------",
        "i-p-sri--------",
        "i-p-srm--------",
        "i-p-srs--------",
        "i-p-srt--------",
        "i-p-sru--------",
        "i-s------------",
        "i-s-s----------",
        "i-s-sc---------",
        "i-s-scc--------",
        "i-s-sco--------",
        "i-s-scp--------",
        "i-s-scs--------",
        "i-s-sr---------",
        "i-s-sraa-------",
        "i-s-srat-------",
        "i-s-srca-------",
        "i-s-srci-------",
        "i-s-srd--------",
        "i-s-sre--------",
        "i-s-srf--------",
        "i-s-srh--------",
        "i-s-sri--------",
        "i-s-srma-------",
        "i-s-srmf-------",
        "i-s-srmg-------",
        "i-s-srmm-------",
        "i-s-srmt-------",
        "i-s-srs--------",
        "i-s-srta-------",
        "i-s-srti-------",
        "i-s-srtt-------",
        "i-s-sru--------",
        "i-u------------",
        "i-u-s----------",
        "i-u-sc---------",
        "i-u-sco--------",
        "i-u-scp--------",
        "i-u-scs--------",
        "i-u-sr---------",
        "i-u-srd--------",
        "i-u-sre--------",
        "i-u-srm--------",
        "i-u-srs--------",
        "i-u-srt--------",
        "i-u-sru--------");

    protected static List<String> stbopsSymbolIds = Arrays.asList(
        "o-g------------",
        "o-g-a----------",
        "o-g-b----------",
        "o-g-c----------",
        "o-g-d----------",
        "o-g-e----------",
        "o-g-f----------",
        "o-i------------",
        "o-i-d----------",
        "o-i-f----------",
        "o-i-g----------",
        "o-i-i----------",
        "o-i-r----------",
        "o-i-s----------",
        "o-i-v----------",
        "o-l------------",
        "o-l-b----------",
        "o-l-g----------",
        "o-l-m----------",
        "o-l-w----------",
        "o-o------------",
        "o-o-a----------",
        "o-o-c----------",
        "o-o-ca---------",
        "o-o-cb---------",
        "o-o-cc---------",
        "o-o-d----------",
        "o-o-e----------",
        "o-o-f----------",
        "o-o-ha---------",
        "o-o-ht---------",
        "o-o-hv---------",
        "o-o-k----------",
        "o-o-ka---------",
        "o-o-m----------",
        "o-o-o----------",
        "o-o-p----------",
        "o-o-rc---------",
        "o-o-rw---------",
        "o-o-s----------",
        "o-o-u----------",
        "o-o-y----------",
        "o-o-yh---------",
        "o-o-yt---------",
        "o-o-yw---------",
        "o-p------------",
        "o-p-a----------",
        "o-p-b----------",
        "o-p-c----------",
        "o-r------------",
        "o-r-a----------",
        "o-v------------",
        "o-v-a----------",
        "o-v-b----------",
        "o-v-d----------",
        "o-v-e----------",
        "o-v-ei---------",
        "o-v-m----------",
        "o-v-ma---------",
        "o-v-mb---------",
        "o-v-mc---------",
        "o-v-p----------",
        "o-v-s----------",
        "o-v-y----------");

    protected static List<String> warSymbolIds = Arrays.asList(
        "s-a------------",
        "s-a-c----------",
        "s-a-cf---------",
        "s-a-ch---------",
        "s-a-cl---------",
        "s-a-m----------",
        "s-a-me---------",
        "s-a-mf---------",
        "s-a-mfa--------",
        "s-a-mfb--------",
        "s-a-mfc--------",
        "s-a-mfch-------",
        "s-a-mfcl-------",
        "s-a-mfcm-------",
        "s-a-mfd--------",
        "s-a-mff--------",
        "s-a-mffi-------",
        "s-a-mfh--------",
        "s-a-mfj--------",
        "s-a-mfk--------",
        "s-a-mfkb-------",
        "s-a-mfkd-------",
        "s-a-mfl--------",
        "s-a-mfm--------",
        "s-a-mfo--------",
        "s-a-mfp--------",
        "s-a-mfpm-------",
        "s-a-mfpn-------",
        "s-a-mfq--------",
        "s-a-mfqa-------",
        "s-a-mfqb-------",
        "s-a-mfqc-------",
        "s-a-mfqd-------",
        "s-a-mfqf-------",
        "s-a-mfqh-------",
        "s-a-mfqi-------",
        "s-a-mfqj-------",
        "s-a-mfqk-------",
        "s-a-mfql-------",
        "s-a-mfqm-------",
        "s-a-mfqn-------",
        "s-a-mfqo-------",
        "s-a-mfqp-------",
        "s-a-mfqr-------",
        "s-a-mfqrw------",
        "s-a-mfqrx------",
        "s-a-mfqrz------",
        "s-a-mfqs-------",
        "s-a-mfqt-------",
        "s-a-mfqu-------",
        "s-a-mfqy-------",
        "s-a-mfr--------",
        "s-a-mfrw-------",
        "s-a-mfrx-------",
        "s-a-mfrz-------",
        "s-a-mfs--------",
        "s-a-mft--------",
        "s-a-mfu--------",
        "s-a-mfuh-------",
        "s-a-mful-------",
        "s-a-mfum-------",
        "s-a-mfy--------",
        "s-a-mh---------",
        "s-a-mha--------",
        "s-a-mhc--------",
        "s-a-mhch-------",
        "s-a-mhcl-------",
        "s-a-mhcm-------",
        "s-a-mhd--------",
        "s-a-mhh--------",
        "s-a-mhi--------",
        "s-a-mhj--------",
        "s-a-mhk--------",
        "s-a-mhm--------",
        "s-a-mho--------",
        "s-a-mhq--------",
        "s-a-mhr--------",
        "s-a-mhs--------",
        "s-a-mht--------",
        "s-a-mhu--------",
        "s-a-mhuh-------",
        "s-a-mhul-------",
        "s-a-mhum-------",
        "s-a-ml---------",
        "s-a-mv---------",
        "s-a-w----------",
        "s-a-wb---------",
        "s-a-wd---------",
        "s-a-wm---------",
        "s-a-wma--------",
        "s-a-wmaa-------",
        "s-a-wmap-------",
        "s-a-wmas-------",
        "s-a-wmb--------",
        "s-a-wmcm-------",
        "s-a-wms--------",
        "s-a-wmsa-------",
        "s-a-wmsb-------",
        "s-a-wmss-------",
        "s-a-wmsu-------",
        "s-a-wmu--------",
        "s-f------------",
        "s-f-a----------",
        "s-f-af---------",
        "s-f-afa--------",
        "s-f-afk--------",
        "s-f-afu--------",
        "s-f-afuh-------",
        "s-f-aful-------",
        "s-f-afum-------",
        "s-f-ah---------",
        "s-f-aha--------",
        "s-f-ahh--------",
        "s-f-ahu--------",
        "s-f-ahuh-------",
        "s-f-ahul-------",
        "s-f-ahum-------",
        "s-f-av---------",
        "s-f-b----------",
        "s-f-g----------",
        "s-f-gc---------",
        "s-f-gp---------",
        "s-f-gpa--------",
        "s-f-gr---------",
        "s-f-gs---------",
        "s-f-n----------",
        "s-f-nb---------",
        "s-f-nn---------",
        "s-f-ns---------",
        "s-f-nu---------",
        "s-g------------",
        "s-g-e----------",
        "s-g-es---------",
        "s-g-ese--------",
        "s-g-esr--------",
        "s-g-ev---------",
        "s-g-eva--------",
        "s-g-evaa-------",
        "s-g-evaar------",
        "s-g-evac-------",
        "s-g-evai-------",
        "s-g-eval-------",
        "s-g-evas-------",
        "s-g-evat-------",
        "s-g-evath------",
        "s-g-evathr-----",
        "s-g-evatl------",
        "s-g-evatlr-----",
        "s-g-evatm------",
        "s-g-evatmr-----",
        "s-g-evc--------",
        "s-g-evca-------",
        "s-g-evcah------",
        "s-g-evcal------",
        "s-g-evcam------",
        "s-g-evcf-------",
        "s-g-evcfh------",
        "s-g-evcfl------",
        "s-g-evcfm------",
        "s-g-evcj-------",
        "s-g-evcjh------",
        "s-g-evcjl------",
        "s-g-evcjm------",
        "s-g-evcm-------",
        "s-g-evcmh------",
        "s-g-evcml------",
        "s-g-evcmm------",
        "s-g-evco-------",
        "s-g-evcoh------",
        "s-g-evcol------",
        "s-g-evcom------",
        "s-g-evct-------",
        "s-g-evcth------",
        "s-g-evctl------",
        "s-g-evctm------",
        "s-g-evcu-------",
        "s-g-evcuh------",
        "s-g-evcul------",
        "s-g-evcum------",
        "s-g-eve--------",
        "s-g-evea-------",
        "s-g-eveaa------",
        "s-g-eveat------",
        "s-g-eveb-------",
        "s-g-evec-------",
        "s-g-eved-------",
        "s-g-eveda------",
        "s-g-evee-------",
        "s-g-evef-------",
        "s-g-eveh-------",
        "s-g-evem-------",
        "s-g-eveml------",
        "s-g-evemv------",
        "s-g-ever-------",
        "s-g-eves-------",
        "s-g-evm--------",
        "s-g-evs--------",
        "s-g-evsc-------",
        "s-g-evsp-------",
        "s-g-evsr-------",
        "s-g-evst-------",
        "s-g-evsw-------",
        "s-g-evt--------",
        "s-g-evu--------",
        "s-g-evua-------",
        "s-g-evuaa------",
        "s-g-evub-------",
        "s-g-evul-------",
        "s-g-evur-------",
        "s-g-evus-------",
        "s-g-evush------",
        "s-g-evusl------",
        "s-g-evusm------",
        "s-g-evut-------",
        "s-g-evuth------",
        "s-g-evutl------",
        "s-g-evux-------",
        "s-g-ewa--------",
        "s-g-ewah-------",
        "s-g-ewal-------",
        "s-g-ewam-------",
        "s-g-ewd--------",
        "s-g-ewdh-------",
        "s-g-ewdhs------",
        "s-g-ewdl-------",
        "s-g-ewdls------",
        "s-g-ewdm-------",
        "s-g-ewdms------",
        "s-g-ewg--------",
        "s-g-ewgh-------",
        "s-g-ewgl-------",
        "s-g-ewgm-------",
        "s-g-ewgr-------",
        "s-g-ewh--------",
        "s-g-ewhh-------",
        "s-g-ewhhs------",
        "s-g-ewhl-------",
        "s-g-ewhls------",
        "s-g-ewhm-------",
        "s-g-ewhms------",
        "s-g-ewm--------",
        "s-g-ewma-------",
        "s-g-ewmai------",
        "s-g-ewmaie-----",
        "s-g-ewmair-----",
        "s-g-ewmal------",
        "s-g-ewmale-----",
        "s-g-ewmalr-----",
        "s-g-ewmas------",
        "s-g-ewmase-----",
        "s-g-ewmasr-----",
        "s-g-ewmat------",
        "s-g-ewmate-----",
        "s-g-ewmatr-----",
        "s-g-ewms-------",
        "s-g-ewmsi------",
        "s-g-ewmsl------",
        "s-g-ewmss------",
        "s-g-ewmt-------",
        "s-g-ewmth------",
        "s-g-ewmtl------",
        "s-g-ewmtm------",
        "s-g-ewo--------",
        "s-g-ewoh-------",
        "s-g-ewol-------",
        "s-g-ewom-------",
        "s-g-ewr--------",
        "s-g-ewrh-------",
        "s-g-ewrl-------",
        "s-g-ewrr-------",
        "s-g-ews--------",
        "s-g-ewsh-------",
        "s-g-ewsl-------",
        "s-g-ewsm-------",
        "s-g-ewt--------",
        "s-g-ewth-------",
        "s-g-ewtl-------",
        "s-g-ewtm-------",
        "s-g-ewx--------",
        "s-g-ewxh-------",
        "s-g-ewxl-------",
        "s-g-ewxm-------",
        "s-g-ewz--------",
        "s-g-ewzh-------",
        "s-g-ewzl-------",
        "s-g-ewzm-------",
        "s-g-exf--------",
        "s-g-exi--------",
        "s-g-exl--------",
        "s-g-exm--------",
        "s-g-exmc-------",
        "s-g-exml-------",
        "s-g-exn--------",
        "s-g-i-----h----",
        "s-g-ib----h----",
        "s-g-iba---h----",
        "s-g-ibn---h----",
        "s-g-ie----h----",
        "s-g-ig----h----",
        "s-g-ima---h----",
        "s-g-imc---h----",
        "s-g-ime---h----",
        "s-g-imf---h----",
        "s-g-imfa--h----",
        "s-g-imfp--h----",
        "s-g-imfpw-h----",
        "s-g-imfs--h----",
        "s-g-img---h----",
        "s-g-imm---h----",
        "s-g-imn---h----",
        "s-g-imnb--h----",
        "s-g-ims---h----",
        "s-g-imv---h----",
        "s-g-ip----h----",
        "s-g-ipd---h----",
        "s-g-ir----h----",
        "s-g-irm---h----",
        "s-g-irn---h----",
        "s-g-irnb--h----",
        "s-g-irnc--h----",
        "s-g-irnn--h----",
        "s-g-irp---h----",
        "s-g-it----h----",
        "s-g-iu----h----",
        "s-g-iue---h----",
        "s-g-iued--h----",
        "s-g-iuef--h----",
        "s-g-iuen--h----",
        "s-g-iup---h----",
        "s-g-iur---h----",
        "s-g-iut---h----",
        "s-g-ix----h----",
        "s-g-ixh---h----",
        "s-g-u----------",
        "s-g-uc---------",
        "s-g-uca--------",
        "s-g-ucaa-------",
        "s-g-ucaaa------",
        "s-g-ucaaas-----",
        "s-g-ucaaat-----",
        "s-g-ucaaaw-----",
        "s-g-ucaac------",
        "s-g-ucaad------",
        "s-g-ucaal------",
        "s-g-ucaam------",
        "s-g-ucaao------",
        "s-g-ucaaos-----",
        "s-g-ucaas------",
        "s-g-ucaau------",
        "s-g-ucat-------",
        "s-g-ucata------",
        "s-g-ucath------",
        "s-g-ucatl------",
        "s-g-ucatm------",
        "s-g-ucatr------",
        "s-g-ucatw------",
        "s-g-ucatwr-----",
        "s-g-ucaw-------",
        "s-g-ucawa------",
        "s-g-ucawh------",
        "s-g-ucawl------",
        "s-g-ucawm------",
        "s-g-ucawr------",
        "s-g-ucaws------",
        "s-g-ucaww------",
        "s-g-ucawwr-----",
        "s-g-ucd--------",
        "s-g-ucdc-------",
        "s-g-ucdg-------",
        "s-g-ucdh-------",
        "s-g-ucdhh------",
        "s-g-ucdhp------",
        "s-g-ucdm-------",
        "s-g-ucdmh------",
        "s-g-ucdml------",
        "s-g-ucdmla-----",
        "s-g-ucdmm------",
        "s-g-ucdo-------",
        "s-g-ucds-------",
        "s-g-ucdsc------",
        "s-g-ucdss------",
        "s-g-ucdsv------",
        "s-g-ucdt-------",
        "s-g-uce--------",
        "s-g-ucec-------",
        "s-g-uceca------",
        "s-g-ucecc------",
        "s-g-ucech------",
        "s-g-ucecl------",
        "s-g-ucecm------",
        "s-g-uceco------",
        "s-g-ucecr------",
        "s-g-ucecs------",
        "s-g-ucect------",
        "s-g-ucecw------",
        "s-g-ucen-------",
        "s-g-ucenn------",
        "s-g-ucf--------",
        "s-g-ucfh-------",
        "s-g-ucfha------",
        "s-g-ucfhc------",
        "s-g-ucfhe------",
        "s-g-ucfhh------",
        "s-g-ucfhl------",
        "s-g-ucfhm------",
        "s-g-ucfho------",
        "s-g-ucfhs------",
        "s-g-ucfhx------",
        "s-g-ucfm-------",
        "s-g-ucfml------",
        "s-g-ucfms------",
        "s-g-ucfmt------",
        "s-g-ucfmta-----",
        "s-g-ucfmtc-----",
        "s-g-ucfmto-----",
        "s-g-ucfmts-----",
        "s-g-ucfmw------",
        "s-g-ucfo-------",
        "s-g-ucfoa------",
        "s-g-ucfol------",
        "s-g-ucfoo------",
        "s-g-ucfos------",
        "s-g-ucfr-------",
        "s-g-ucfrm------",
        "s-g-ucfrmr-----",
        "s-g-ucfrms-----",
        "s-g-ucfrmt-----",
        "s-g-ucfrs------",
        "s-g-ucfrsr-----",
        "s-g-ucfrss-----",
        "s-g-ucfrst-----",
        "s-g-ucfs-------",
        "s-g-ucfsa------",
        "s-g-ucfsl------",
        "s-g-ucfso------",
        "s-g-ucfss------",
        "s-g-ucft-------",
        "s-g-ucfta------",
        "s-g-ucftc------",
        "s-g-ucftcd-----",
        "s-g-ucftcm-----",
        "s-g-ucftf------",
        "s-g-ucftr------",
        "s-g-ucfts------",
        "s-g-uci--------",
        "s-g-ucia-------",
        "s-g-ucic-------",
        "s-g-ucii-------",
        "s-g-ucil-------",
        "s-g-ucim-------",
        "s-g-ucin-------",
        "s-g-ucio-------",
        "s-g-ucis-------",
        "s-g-uciz-------",
        "s-g-ucm--------",
        "s-g-ucms-------",
        "s-g-ucmt-------",
        "s-g-ucr--------",
        "s-g-ucra-------",
        "s-g-ucrc-------",
        "s-g-ucrh-------",
        "s-g-ucrl-------",
        "s-g-ucro-------",
        "s-g-ucrr-------",
        "s-g-ucrrd------",
        "s-g-ucrrf------",
        "s-g-ucrrl------",
        "s-g-ucrs-------",
        "s-g-ucrv-------",
        "s-g-ucrva------",
        "s-g-ucrvg------",
        "s-g-ucrvm------",
        "s-g-ucrvo------",
        "s-g-ucrx-------",
        "s-g-ucs--------",
        "s-g-ucsa-------",
        "s-g-ucsg-------",
        "s-g-ucsga------",
        "s-g-ucsgd------",
        "s-g-ucsgm------",
        "s-g-ucsm-------",
        "s-g-ucsr-------",
        "s-g-ucsw-------",
        "s-g-ucv--------",
        "s-g-ucvc-------",
        "s-g-ucvf-------",
        "s-g-ucvfa------",
        "s-g-ucvfr------",
        "s-g-ucvfu------",
        "s-g-ucvr-------",
        "s-g-ucvra------",
        "s-g-ucvrm------",
        "s-g-ucvrs------",
        "s-g-ucvru------",
        "s-g-ucvruc-----",
        "s-g-ucvrue-----",
        "s-g-ucvruh-----",
        "s-g-ucvrul-----",
        "s-g-ucvrum-----",
        "s-g-ucvrw------",
        "s-g-ucvs-------",
        "s-g-ucvu-------",
        "s-g-ucvuf------",
        "s-g-ucvur------",
        "s-g-ucvv-------",
        "s-g-uh---------",
        "s-g-us---------",
        "s-g-usa--------",
        "s-g-usac-------",
        "s-g-usaf-------",
        "s-g-usafc------",
        "s-g-usaft------",
        "s-g-usaj-------",
        "s-g-usajc------",
        "s-g-usajt------",
        "s-g-usal-------",
        "s-g-usalc------",
        "s-g-usalt------",
        "s-g-usam-------",
        "s-g-usamc------",
        "s-g-usamt------",
        "s-g-usao-------",
        "s-g-usaoc------",
        "s-g-usaot------",
        "s-g-usap-------",
        "s-g-usapb------",
        "s-g-usapbc-----",
        "s-g-usapbt-----",
        "s-g-usapc------",
        "s-g-usapm------",
        "s-g-usapmc-----",
        "s-g-usapmt-----",
        "s-g-usapt------",
        "s-g-usaq-------",
        "s-g-usaqc------",
        "s-g-usaqt------",
        "s-g-usar-------",
        "s-g-usarc------",
        "s-g-usart------",
        "s-g-usas-------",
        "s-g-usasc------",
        "s-g-usast------",
        "s-g-usat-------",
        "s-g-usaw-------",
        "s-g-usawc------",
        "s-g-usawt------",
        "s-g-usax-------",
        "s-g-usaxc------",
        "s-g-usaxt------",
        "s-g-usm--------",
        "s-g-usmc-------",
        "s-g-usmd-------",
        "s-g-usmdc------",
        "s-g-usmdt------",
        "s-g-usmm-------",
        "s-g-usmmc------",
        "s-g-usmmt------",
        "s-g-usmp-------",
        "s-g-usmpc------",
        "s-g-usmpt------",
        "s-g-usmt-------",
        "s-g-usmv-------",
        "s-g-usmvc------",
        "s-g-usmvt------",
        "s-g-uss--------",
        "s-g-uss1-------",
        "s-g-uss1c------",
        "s-g-uss1t------",
        "s-g-uss2-------",
        "s-g-uss2c------",
        "s-g-uss2t------",
        "s-g-uss3-------",
        "s-g-uss3a------",
        "s-g-uss3ac-----",
        "s-g-uss3at-----",
        "s-g-uss3c------",
        "s-g-uss3t------",
        "s-g-uss4-------",
        "s-g-uss4c------",
        "s-g-uss4t------",
        "s-g-uss5-------",
        "s-g-uss5c------",
        "s-g-uss5t------",
        "s-g-uss6-------",
        "s-g-uss6c------",
        "s-g-uss6t------",
        "s-g-uss7-------",
        "s-g-uss7c------",
        "s-g-uss7t------",
        "s-g-uss8-------",
        "s-g-uss8c------",
        "s-g-uss8t------",
        "s-g-uss9-------",
        "s-g-uss9c------",
        "s-g-uss9t------",
        "s-g-ussc-------",
        "s-g-ussl-------",
        "s-g-usslc------",
        "s-g-usslt------",
        "s-g-usst-------",
        "s-g-ussw-------",
        "s-g-usswc------",
        "s-g-usswp------",
        "s-g-usswpc-----",
        "s-g-usswpt-----",
        "s-g-usswt------",
        "s-g-ussx-------",
        "s-g-ussxc------",
        "s-g-ussxt------",
        "s-g-ust--------",
        "s-g-usta-------",
        "s-g-ustac------",
        "s-g-ustat------",
        "s-g-ustc-------",
        "s-g-usti-------",
        "s-g-ustic------",
        "s-g-ustit------",
        "s-g-ustm-------",
        "s-g-ustmc------",
        "s-g-ustmt------",
        "s-g-ustr-------",
        "s-g-ustrc------",
        "s-g-ustrt------",
        "s-g-usts-------",
        "s-g-ustsc------",
        "s-g-ustst------",
        "s-g-ustt-------",
        "s-g-usx--------",
        "s-g-usxc-------",
        "s-g-usxe-------",
        "s-g-usxec------",
        "s-g-usxet------",
        "s-g-usxh-------",
        "s-g-usxhc------",
        "s-g-usxht------",
        "s-g-usxo-------",
        "s-g-usxoc------",
        "s-g-usxom------",
        "s-g-usxomc-----",
        "s-g-usxomt-----",
        "s-g-usxot------",
        "s-g-usxr-------",
        "s-g-usxrc------",
        "s-g-usxrt------",
        "s-g-usxt-------",
        "s-g-uu---------",
        "s-g-uua--------",
        "s-g-uuab-------",
        "s-g-uuabr------",
        "s-g-uuac-------",
        "s-g-uuacc------",
        "s-g-uuacck-----",
        "s-g-uuaccm-----",
        "s-g-uuacr------",
        "s-g-uuacrs-----",
        "s-g-uuacrw-----",
        "s-g-uuacs------",
        "s-g-uuacsa-----",
        "s-g-uuacsm-----",
        "s-g-uuad-------",
        "s-g-uuan-------",
        "s-g-uue--------",
        "s-g-uui--------",
        "s-g-uul--------",
        "s-g-uulc-------",
        "s-g-uuld-------",
        "s-g-uulf-------",
        "s-g-uulm-------",
        "s-g-uuls-------",
        "s-g-uum--------",
        "s-g-uuma-------",
        "s-g-uumc-------",
        "s-g-uumj-------",
        "s-g-uummo------",
        "s-g-uumo-------",
        "s-g-uumq-------",
        "s-g-uumr-------",
        "s-g-uumrg------",
        "s-g-uumrs------",
        "s-g-uumrss-----",
        "s-g-uumrx------",
        "s-g-uums-------",
        "s-g-uumse------",
        "s-g-uumsea-----",
        "s-g-uumsec-----",
        "s-g-uumsed-----",
        "s-g-uumsei-----",
        "s-g-uumsej-----",
        "s-g-uumset-----",
        "s-g-uumt-------",
        "s-g-uup--------",
        "s-g-uus--------",
        "s-g-uusa-------",
        "s-g-uusc-------",
        "s-g-uuscl------",
        "s-g-uusf-------",
        "s-g-uusm-------",
        "s-g-uusml------",
        "s-g-uusmn------",
        "s-g-uusms------",
        "s-g-uuso-------",
        "s-g-uusr-------",
        "s-g-uusrs------",
        "s-g-uusrt------",
        "s-g-uusrw------",
        "s-g-uuss-------",
        "s-g-uusw-------",
        "s-g-uusx-------",
        "s-p------------",
        "s-p-l----------",
        "s-p-s----------",
        "s-p-t----------",
        "s-p-v----------",
        "s-s------------",
        "s-s-c----------",
        "s-s-ca---------",
        "s-s-cala-------",
        "s-s-calc-------",
        "s-s-cals-------",
        "s-s-calsm------",
        "s-s-calst------",
        "s-s-cd---------",
        "s-s-ch---------",
        "s-s-cl---------",
        "s-s-clbb-------",
        "s-s-clcc-------",
        "s-s-clcv-------",
        "s-s-cldd-------",
        "s-s-clff-------",
        "s-s-clll-------",
        "s-s-clllas-----",
        "s-s-clllmi-----",
        "s-s-clllsu-----",
        "s-s-cm---------",
        "s-s-cmma-------",
        "s-s-cmmh-------",
        "s-s-cmml-------",
        "s-s-cmms-------",
        "s-s-cp---------",
        "s-s-cpsb-------",
        "s-s-cpsu-------",
        "s-s-cpsug------",
        "s-s-cpsum------",
        "s-s-cpsut------",
        "s-s-cu---------",
        "s-s-cum--------",
        "s-s-cun--------",
        "s-s-cur--------",
        "s-s-cus--------",
        "s-s-g----------",
        "s-s-gc---------",
        "s-s-gg---------",
        "s-s-gt---------",
        "s-s-gu---------",
        "s-s-n----------",
        "s-s-nf---------",
        "s-s-nh---------",
        "s-s-ni---------",
        "s-s-nm---------",
        "s-s-nr---------",
        "s-s-ns---------",
        "s-s-o----------",
        "s-s-xa---------",
        "s-s-xar--------",
        "s-s-xas--------",
        "s-s-xf---------",
        "s-s-xfdf-------",
        "s-s-xfdr-------",
        "s-s-xftr-------",
        "s-s-xh---------",
        "s-s-xl---------",
        "s-s-xm---------",
        "s-s-xmc--------",
        "s-s-xmf--------",
        "s-s-xmh--------",
        "s-s-xmo--------",
        "s-s-xmp--------",
        "s-s-xmr--------",
        "s-s-xmto-------",
        "s-s-xmtu-------",
        "s-s-xp---------",
        "s-s-xr---------",
        "s-u------------",
        "s-u-e----------",
        "s-u-nd---------",
        "s-u-s----------",
        "s-u-s1---------",
        "s-u-s2---------",
        "s-u-s3---------",
        "s-u-s4---------",
        "s-u-sb---------",
        "s-u-sc---------",
        "s-u-sca--------",
        "s-u-scb--------",
        "s-u-scf--------",
        "s-u-scg--------",
        "s-u-scm--------",
        "s-u-sf---------",
        "s-u-sk---------",
        "s-u-sl---------",
        "s-u-sn---------",
        "s-u-sna--------",
        "s-u-snb--------",
        "s-u-snf--------",
        "s-u-sng--------",
        "s-u-snm--------",
        "s-u-so---------",
        "s-u-sof--------",
        "s-u-sr---------",
        "s-u-su---------",
        "s-u-sum--------",
        "s-u-sun--------",
        "s-u-sus--------",
        "s-u-sx---------",
        "s-u-v----------",
        "s-u-w----------",
        "s-u-wd---------",
        "s-u-wdm--------",
        "s-u-wdmg-------",
        "s-u-wdmm-------",
        "s-u-wm---------",
        "s-u-wma--------",
        "s-u-wmb--------",
        "s-u-wmbd-------",
        "s-u-wmc--------",
        "s-u-wmd--------",
        "s-u-wme--------",
        "s-u-wmf--------",
        "s-u-wmfc-------",
        "s-u-wmfd-------",
        "s-u-wmfe-------",
        "s-u-wmfo-------",
        "s-u-wmfr-------",
        "s-u-wmfx-------",
        "s-u-wmg--------",
        "s-u-wmgc-------",
        "s-u-wmgd-------",
        "s-u-wmge-------",
        "s-u-wmgo-------",
        "s-u-wmgr-------",
        "s-u-wmgx-------",
        "s-u-wmm--------",
        "s-u-wmmc-------",
        "s-u-wmmd-------",
        "s-u-wmme-------",
        "s-u-wmmo-------",
        "s-u-wmmr-------",
        "s-u-wmmx-------",
        "s-u-wmn--------",
        "s-u-wmo--------",
        "s-u-wmod-------",
        "s-u-wmr--------",
        "s-u-wms--------",
        "s-u-wmsd-------",
        "s-u-wmsx-------",
        "s-u-wmx--------",
        "s-u-wt---------",
        "s-u-x----------");
}
