/*
 * The MIT License
 *
 * Copyright 2015 tibo.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package info.debatty.java.lsh;

import java.util.Random;

/**
 *
 * @author tibo
 */
public class RandomProjection {

    /**
     * @param args the command line arguments
     */
    public static void main(String[] args) {
        
        // Initialize hyperplanes for an error of 0.001, and points in R^3
        RandomProjection rp = new RandomProjection(0.01, 4);
        
        double[] v1 = {1, 1, 1, 1};
        double[] v2 = {1, 2, 1, 4};
        

        boolean[] sig1 = rp.hash(v1);
        boolean[] sig2 = rp.hash(v2);
        
        System.out.println("Signature (estimated) similarity: " + rp.similarity(sig1, sig2));
        System.out.println("Real (cosine) similarity: " + cosineSimilarity(v1, v2));
        
    }
    private int size;
    private int n;
    private double[][] hyperplanes;
    
    public RandomProjection(double error, int n) {
        init(MinHash.size(error), n);
    }

    private void init(int size, int n) {        
        this.n = n;
        this.size = size;
        
        Random rand = new Random();
        
        // Build the random hyperplanes
        this.hyperplanes = new double[size][n];
        
        double[] thetas = new double[n];
        for (int i = 0; i < size; i++) {
            // Choose random angles
            for (int j = 0; j < n-1; j++) {
                thetas[j] = (double) rand.nextInt() / Integer.MAX_VALUE * 2 * Math.PI;
            }
            thetas[n-1] = 0;
            
            /* a = r sin (theta_0)
             * b = r cos (theta_0) sin (theta_1)
             * c = r cos (theta_0) cos (theta_1) sin (theta_2)
             * d = r cos (theta_0) cos (theta_1) cos (theta_2) sin (theta_3)
             * e = r cos (theta_0) cos (theta_1) cos (theta_2) cos (theta_3)
             */
            
            double r = 1;
            for (int j = 0; j < n; j++) {
                this.hyperplanes[i][j] = r * Math.sin(thetas[j]);
                r *= Math.cos(thetas[j]);
            }
        }
        
    }
    
    /**
     * 
     * @param v
     * @return 
     */
    public boolean[] hash(double[] v) {
        boolean[] sig = new boolean[this.size];
        
        for (int i = 0; i < this.size; i++) {
            sig[i] = (dotProduct(v, this.hyperplanes[i]) >= 0);
        }
        
        return sig;
        
    }
    
    
    public double similarity(boolean[] sig1, boolean[] sig2) {
        
        double pr = 0;
        for (int i = 0; i < sig1.length; i++) {
            if (sig1[i] == sig2[i]) {
                pr += 1;
            }
        }
        
        pr = pr / sig1.length;
        return pr;
        
        //return Math.cos((1 - pr) * Math.PI);
    }
    
    public static double dotProduct(double[] v1, double[] v2) {
        double agg = 0;
        
        for (int i = 0; i < v1.length; i++) {
            agg += v1[i] * v2[i];
        }
        
        return agg;
    }
    
    /**
     * Returns the norm L2 : sqrt(Sum_i( v_i^2))
     * @param v
     * @return 
     */
    public static double norm(double[] v) {
        double agg = 0;
        
        for (int i = 0; i < v.length; i++) {
            agg += v[i] * v[i];
        }
        
        return Math.sqrt(agg);
    }
    
    /**
     * Computes the cosine similarity, computed as v1 . v2 / (|v1| x |v2|)
     * Cosine similarity of two vectors is the cosine of the angle between them.
     * The cosine of 0° is 1, and it is less than 1 for any other angle. It is 
     * based on the orientation and not on the magnitude of vectors: two vectors with the 
     * same orientation have a Cosine similarity of 1, two vectors at 90° have 
     * a similarity of 0, and two vectors diametrically opposed have a 
     * similarity of -1, independent of their magnitude.
     * 
     * @param v1
     * @param v2
     * @return 
     */
    public static double cosineSimilarity(double[]v1, double[] v2) {
        return 1 - Math.acos(dotProduct(v1, v2) / (norm(v1) * norm(v2))) / Math.PI;
    }
}
