package izumi.distage.model.reflection

import izumi.distage.model.reflection.Provider.{ProviderType, UnsafeProviderCallArgsMismatched}

trait Provider {
  def parameters: Seq[LinkedParameter]
  final def diKeys: Seq[DIKey] = parameters.map(_.key)
  final def argTypes: Seq[SafeType] = parameters.map(_.key.tpe)
  final def arity: Int = parameters.size

  def ret: SafeType
  // either a function or a constant value
  def underlying: AnyRef
  def fun: Seq[Any] => Any
  def providerType: ProviderType

  def unsafeApply(refs: Seq[GenericTypedRef[?]]): Any = {
    val args = verifyArgs(refs)
    fun(args)
  }

  def unsafeMap(newRet: SafeType, f: Any => ?): Provider
  def unsafeZip(newRet: SafeType, that: Provider): Provider
  def addUnused(keys: Iterable[DIKey]): Provider
  def replaceKeys(f: DIKey => DIKey): Provider

  private val comparable: AnyRef = {
    providerType match {
      case ProviderType.Constructor => (ret, diKeys)
      case ProviderType.Function => (underlying, diKeys)
      case ProviderType.Singleton => (underlying, diKeys)
    }
  }

  override final def equals(obj: Any): Boolean = obj match {
    case that: Provider => comparable == that.comparable
    case _ => false
  }
  override final def hashCode(): Int = comparable.hashCode()

  override final def toString: String = s"$funString(${argTypes.mkString(", ")}): $ret"
  final def funString: String = providerType match {
    case ProviderType.Singleton => s"κ:$underlying"
    case ProviderType.Constructor => s"π:$providerType"
    case ProviderType.Function => s"ƒ:$underlying"
  }

  protected def verifyArgs(refs: Seq[GenericTypedRef[?]]): Seq[Any] = {
    val (newArgs, types, typesCmp) = parameters
      .zip(refs).map {
        case (param, ref) =>
          val newArg = ref.asArgument(param.isByName)
          (newArg, ref.tpe, ref.tpe <:< param.key.tpe)
      }.unzip3

    val countOk = refs.size == parameters.size
    val typesOk = !typesCmp.contains(false)

    if (countOk && typesOk) {
      newArgs
    } else {
      throw new UnsafeProviderCallArgsMismatched(
        message = s"""Mismatched arguments for unsafe call:
                     | ${if (!typesOk) "Wrong types!" else ""}
                     | ${if (!countOk) s"Expected number of arguments $arity, but got ${refs.size}" else ""}
                     |Expected types [${argTypes.mkString(",")}], got types [${types.mkString(",")}], values: (${newArgs.mkString(",")})""".stripMargin,
        expected = argTypes,
        actual = types,
        actualValues = newArgs,
      )
    }
  }
}

object Provider {
  class UnsafeProviderCallArgsMismatched(message: String, val expected: Seq[SafeType], val actual: Seq[SafeType], val actualValues: Seq[Any])
    extends RuntimeException(message)

  sealed trait ProviderType
  object ProviderType {
    /** Reference to a singleton object or a literal constant. */
    case object Singleton extends ProviderType

    /** Function generated by any of the "constructor" macros */
    case object Constructor extends ProviderType

    /** Function captured by the Functoid macro any provider transformed by the unsafeMap helper method */
    case object Function extends ProviderType
  }

  final case class ProviderImpl[+A](
    parameters: Seq[LinkedParameter],
    ret: SafeType,
    underlying: AnyRef,
    fun: Seq[Any] => Any,
    providerType: ProviderType,
  ) extends Provider {
    def this(parameters: Seq[LinkedParameter], ret: SafeType, fun: Seq[Any] => Any, providerType: ProviderType) =
      this(parameters, ret, fun, fun, providerType)

    override def unsafeApply(refs: Seq[GenericTypedRef[?]]): A =
      super.unsafeApply(refs).asInstanceOf[A]

    override def unsafeMap(newRet: SafeType, f: Any => ?): ProviderImpl[?] =
      copy(
        ret = newRet,
        underlying = f,
        fun = xs => f.apply(fun(xs)),
        providerType = ProviderType.Function,
      )

    override def addUnused(keys: Iterable[DIKey]): Provider =
      copy(
        parameters = parameters ++ keys.map(key => LinkedParameter(SymbolInfo("<unused>", key.tpe, isByName = false, wasGeneric = false), key))
      )

    override def replaceKeys(f: DIKey => DIKey): Provider = copy(parameters.map(_.replaceKey(f)))

    override def unsafeZip(newRet: SafeType, that: Provider): Provider = {
      new ProviderImpl(
        parameters = parameters ++ that.parameters,
        ret = newRet,
        fun = {
          args0 =>
            val (args1, args2) = args0.splitAt(arity)
            fun(args1) -> that.fun(args2)
        },
        providerType = ProviderType.Function,
      )
    }
  }
  object ProviderImpl {
    @inline def apply[A](parameters: Seq[LinkedParameter], ret: SafeType, fun: Seq[Any] => Any, providerType: ProviderType): ProviderImpl[A] =
      new ProviderImpl(parameters, ret, fun, providerType)
  }

}
