/*
 * Decompiled with CFR 0.152.
 */
package io.aeron.archive;

import io.aeron.archive.ArchiveMarkFile;
import io.aeron.archive.RecordingSummary;
import io.aeron.archive.checksum.Checksum;
import io.aeron.archive.client.ArchiveException;
import io.aeron.archive.codecs.CatalogHeaderDecoder;
import io.aeron.archive.codecs.CatalogHeaderEncoder;
import io.aeron.archive.codecs.RecordingDescriptorDecoder;
import io.aeron.archive.codecs.RecordingDescriptorEncoder;
import io.aeron.archive.codecs.RecordingDescriptorHeaderDecoder;
import io.aeron.archive.codecs.RecordingDescriptorHeaderEncoder;
import io.aeron.logbuffer.FrameDescriptor;
import java.io.File;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.nio.MappedByteBuffer;
import java.nio.channels.FileChannel;
import java.nio.file.OpenOption;
import java.nio.file.StandardOpenOption;
import java.util.function.IntConsumer;
import java.util.function.Predicate;
import org.agrona.AsciiEncoding;
import org.agrona.BitUtil;
import org.agrona.CloseHelper;
import org.agrona.DirectBuffer;
import org.agrona.IoUtil;
import org.agrona.LangUtil;
import org.agrona.SemanticVersion;
import org.agrona.concurrent.EpochClock;
import org.agrona.concurrent.UnsafeBuffer;

class Catalog
implements AutoCloseable {
    static final int PAGE_SIZE = 4096;
    static final int NULL_RECORD_ID = -1;
    static final int DESCRIPTOR_HEADER_LENGTH = 32;
    static final int DEFAULT_RECORD_LENGTH = 1024;
    static final long MAX_CATALOG_LENGTH = Integer.MAX_VALUE;
    static final long MAX_ENTRIES = Catalog.calculateMaxEntries(Integer.MAX_VALUE, 1024L);
    static final long DEFAULT_MAX_ENTRIES = 8192L;
    static final byte VALID = 1;
    static final byte INVALID = 0;
    private final RecordingDescriptorHeaderDecoder descriptorHeaderDecoder = new RecordingDescriptorHeaderDecoder();
    private final RecordingDescriptorHeaderEncoder descriptorHeaderEncoder = new RecordingDescriptorHeaderEncoder();
    private final RecordingDescriptorEncoder descriptorEncoder = new RecordingDescriptorEncoder();
    private final RecordingDescriptorDecoder descriptorDecoder = new RecordingDescriptorDecoder();
    private final int recordLength;
    private final int maxDescriptorStringsCombinedLength;
    private final boolean forceWrites;
    private final boolean forceMetadata;
    private boolean isClosed;
    private final File catalogFile;
    private final File archiveDir;
    private final EpochClock epochClock;
    private FileChannel catalogChannel;
    private MappedByteBuffer catalogByteBuffer;
    private UnsafeBuffer catalogBuffer;
    private UnsafeBuffer fieldAccessBuffer;
    private int maxRecordingId;
    private long nextRecordingId = 0L;

    Catalog(File archiveDir, FileChannel archiveDirChannel, int fileSyncLevel, long maxNumEntries, EpochClock epochClock, Checksum checksum, UnsafeBuffer buffer) {
        this.archiveDir = archiveDir;
        this.forceWrites = fileSyncLevel > 0;
        this.forceMetadata = fileSyncLevel > 1;
        this.epochClock = epochClock;
        Catalog.validateMaxEntries(maxNumEntries);
        this.catalogFile = new File(archiveDir, "archive.catalog");
        try {
            boolean catalogPreExists = this.catalogFile.exists();
            MappedByteBuffer catalogMappedByteBuffer = null;
            FileChannel catalogFileChannel = null;
            long catalogLength = -1L;
            try {
                catalogFileChannel = FileChannel.open(this.catalogFile.toPath(), StandardOpenOption.CREATE, StandardOpenOption.READ, StandardOpenOption.WRITE, StandardOpenOption.SPARSE);
                catalogLength = catalogPreExists ? Math.max(catalogFileChannel.size(), Catalog.calculateCatalogLength(maxNumEntries)) : Catalog.calculateCatalogLength(maxNumEntries);
                catalogMappedByteBuffer = catalogFileChannel.map(FileChannel.MapMode.READ_WRITE, 0L, catalogLength);
            }
            catch (Exception ex) {
                CloseHelper.close(catalogFileChannel);
                LangUtil.rethrowUnchecked(ex);
            }
            this.catalogChannel = catalogFileChannel;
            this.catalogByteBuffer = catalogMappedByteBuffer;
            this.catalogBuffer = new UnsafeBuffer(this.catalogByteBuffer);
            this.fieldAccessBuffer = new UnsafeBuffer(this.catalogByteBuffer);
            CatalogHeaderDecoder catalogHeaderDecoder = new CatalogHeaderDecoder();
            catalogHeaderDecoder.wrap(this.catalogBuffer, 0, 8, 4);
            if (catalogPreExists) {
                int version = catalogHeaderDecoder.version();
                if (SemanticVersion.major(version) != 2) {
                    throw new ArchiveException("invalid version " + SemanticVersion.toString(version) + ", archive is " + SemanticVersion.toString(ArchiveMarkFile.SEMANTIC_VERSION));
                }
                this.recordLength = catalogHeaderDecoder.entryLength();
            } else {
                this.forceWrites(archiveDirChannel);
                this.recordLength = 1024;
                new CatalogHeaderEncoder().wrap(this.catalogBuffer, 0).entryLength(1024).version(ArchiveMarkFile.SEMANTIC_VERSION);
            }
            this.maxDescriptorStringsCombinedLength = this.recordLength - 124;
            this.maxRecordingId = (int)Catalog.calculateMaxEntries(catalogLength, this.recordLength) - 1;
            this.refreshCatalog(true, checksum, buffer);
        }
        catch (Throwable ex) {
            this.close();
            throw ex;
        }
    }

    Catalog(File archiveDir, EpochClock epochClock) {
        this(archiveDir, epochClock, false, null);
    }

    Catalog(File archiveDir, EpochClock epochClock, boolean writable, IntConsumer versionCheck) {
        this.archiveDir = archiveDir;
        this.forceWrites = false;
        this.forceMetadata = false;
        this.epochClock = epochClock;
        this.catalogChannel = null;
        this.catalogFile = new File(archiveDir, "archive.catalog");
        try {
            OpenOption[] openOptionArray;
            MappedByteBuffer catalogMappedByteBuffer = null;
            long catalogLength = -1L;
            if (writable) {
                StandardOpenOption[] standardOpenOptionArray = new StandardOpenOption[3];
                standardOpenOptionArray[0] = StandardOpenOption.READ;
                standardOpenOptionArray[1] = StandardOpenOption.WRITE;
                openOptionArray = standardOpenOptionArray;
                standardOpenOptionArray[2] = StandardOpenOption.SPARSE;
            } else {
                OpenOption[] openOptionArray2 = new StandardOpenOption[1];
                openOptionArray = openOptionArray2;
                openOptionArray2[0] = StandardOpenOption.READ;
            }
            OpenOption[] openOptions = openOptionArray;
            try (FileChannel channel = FileChannel.open(this.catalogFile.toPath(), openOptions);){
                catalogLength = channel.size();
                catalogMappedByteBuffer = channel.map(writable ? FileChannel.MapMode.READ_WRITE : FileChannel.MapMode.READ_ONLY, 0L, catalogLength);
            }
            catch (Exception ex) {
                LangUtil.rethrowUnchecked(ex);
            }
            this.catalogByteBuffer = catalogMappedByteBuffer;
            this.catalogBuffer = new UnsafeBuffer(this.catalogByteBuffer);
            this.fieldAccessBuffer = new UnsafeBuffer(this.catalogByteBuffer);
            CatalogHeaderDecoder catalogHeaderDecoder = new CatalogHeaderDecoder();
            catalogHeaderDecoder.wrap(this.catalogBuffer, 0, 8, 4);
            int version = catalogHeaderDecoder.version();
            if (null == versionCheck) {
                if (SemanticVersion.major(version) != 2) {
                    throw new ArchiveException("invalid version " + SemanticVersion.toString(version) + ", archive is " + SemanticVersion.toString(ArchiveMarkFile.SEMANTIC_VERSION));
                }
            } else {
                versionCheck.accept(version);
            }
            this.recordLength = catalogHeaderDecoder.entryLength();
            this.maxDescriptorStringsCombinedLength = this.recordLength - 124;
            this.maxRecordingId = (int)Catalog.calculateMaxEntries(catalogLength, this.recordLength) - 1;
            this.refreshCatalog(false, null, null);
        }
        catch (Throwable ex) {
            this.close();
            throw ex;
        }
    }

    @Override
    public void close() {
        if (!this.isClosed) {
            this.isClosed = true;
            this.unmapAndCloseChannel();
        }
    }

    int maxEntries() {
        return this.maxRecordingId + 1;
    }

    int countEntries() {
        return (int)this.nextRecordingId;
    }

    int version() {
        UnsafeBuffer buffer = new UnsafeBuffer(this.catalogByteBuffer);
        CatalogHeaderDecoder catalogHeaderDecoder = new CatalogHeaderDecoder().wrap(buffer, 0, 8, 4);
        return catalogHeaderDecoder.version();
    }

    void updateVersion(int version) {
        UnsafeBuffer buffer = new UnsafeBuffer(this.catalogByteBuffer);
        new CatalogHeaderEncoder().wrap(buffer, 0).version(version);
    }

    long addNewRecording(long startPosition, long stopPosition, long startTimestamp, long stopTimestamp, int imageInitialTermId, int segmentFileLength, int termBufferLength, int mtuLength, int sessionId, int streamId, String strippedChannel, String originalChannel, String sourceIdentity) {
        int combinedStringsLen;
        if (this.nextRecordingId > (long)this.maxRecordingId) {
            this.growCatalog(Integer.MAX_VALUE);
        }
        if ((combinedStringsLen = strippedChannel.length() + sourceIdentity.length() + originalChannel.length()) > this.maxDescriptorStringsCombinedLength) {
            throw new ArchiveException("combined length of channel:'" + strippedChannel + "' and sourceIdentity:'" + sourceIdentity + "' and originalChannel:'" + originalChannel + "' exceeds max allowed:" + this.maxDescriptorStringsCombinedLength);
        }
        long recordingId = this.nextRecordingId++;
        this.catalogBuffer.wrap(this.catalogByteBuffer, this.recordingDescriptorOffset(recordingId), this.recordLength);
        this.descriptorEncoder.wrap(this.catalogBuffer, 32).recordingId(recordingId).startTimestamp(startTimestamp).stopTimestamp(stopTimestamp).startPosition(startPosition).stopPosition(stopPosition).initialTermId(imageInitialTermId).segmentFileLength(segmentFileLength).termBufferLength(termBufferLength).mtuLength(mtuLength).sessionId(sessionId).streamId(streamId).strippedChannel(strippedChannel).originalChannel(originalChannel).sourceIdentity(sourceIdentity);
        this.descriptorHeaderEncoder.wrap(this.catalogBuffer, 0).length(this.descriptorEncoder.encodedLength()).valid((byte)1);
        this.forceWrites(this.catalogChannel);
        return recordingId;
    }

    long addNewRecording(long startPosition, long startTimestamp, int imageInitialTermId, int segmentFileLength, int termBufferLength, int mtuLength, int sessionId, int streamId, String strippedChannel, String originalChannel, String sourceIdentity) {
        return this.addNewRecording(startPosition, -1L, startTimestamp, -1L, imageInitialTermId, segmentFileLength, termBufferLength, mtuLength, sessionId, streamId, strippedChannel, originalChannel, sourceIdentity);
    }

    boolean wrapDescriptor(long recordingId, UnsafeBuffer buffer) {
        if (recordingId < 0L || recordingId > (long)this.maxRecordingId) {
            return false;
        }
        buffer.wrap(this.catalogByteBuffer, this.recordingDescriptorOffset(recordingId), this.recordLength);
        return Catalog.descriptorLength(buffer) > 0;
    }

    boolean wrapAndValidateDescriptor(long recordingId, UnsafeBuffer buffer) {
        if (recordingId < 0L || recordingId > (long)this.maxRecordingId) {
            return false;
        }
        buffer.wrap(this.catalogByteBuffer, this.recordingDescriptorOffset(recordingId), this.recordLength);
        return Catalog.descriptorLength(buffer) > 0 && Catalog.isValidDescriptor(buffer);
    }

    boolean hasRecording(long recordingId) {
        return recordingId >= 0L && recordingId < this.nextRecordingId && this.fieldAccessBuffer.getByte(this.recordingDescriptorOffset(recordingId) + RecordingDescriptorHeaderDecoder.validEncodingOffset()) == 1;
    }

    int forEach(CatalogEntryProcessor consumer) {
        int count = 0;
        long recordingId = 0L;
        while (this.wrapDescriptor(recordingId, this.catalogBuffer)) {
            this.descriptorHeaderDecoder.wrap(this.catalogBuffer, 0, 32, 4);
            this.descriptorHeaderEncoder.wrap(this.catalogBuffer, 0);
            this.descriptorDecoder.wrap(this.catalogBuffer, 32, 80, 4);
            this.descriptorEncoder.wrap(this.catalogBuffer, 32);
            consumer.accept(this.descriptorHeaderEncoder, this.descriptorHeaderDecoder, this.descriptorEncoder, this.descriptorDecoder);
            ++recordingId;
            ++count;
        }
        return count;
    }

    boolean forEntry(long recordingId, CatalogEntryProcessor consumer) {
        if (this.wrapDescriptor(recordingId, this.catalogBuffer)) {
            this.descriptorHeaderDecoder.wrap(this.catalogBuffer, 0, 32, 4);
            this.descriptorHeaderEncoder.wrap(this.catalogBuffer, 0);
            this.descriptorDecoder.wrap(this.catalogBuffer, 32, 80, 4);
            this.descriptorEncoder.wrap(this.catalogBuffer, 32);
            consumer.accept(this.descriptorHeaderEncoder, this.descriptorHeaderDecoder, this.descriptorEncoder, this.descriptorDecoder);
            return true;
        }
        return false;
    }

    long findLast(long minRecordingId, int sessionId, int streamId, byte[] channelFragment) {
        long recordingId = this.nextRecordingId;
        while (--recordingId >= minRecordingId) {
            this.catalogBuffer.wrap(this.catalogByteBuffer, this.recordingDescriptorOffset(recordingId), this.recordLength);
            if (!Catalog.isValidDescriptor(this.catalogBuffer)) continue;
            this.descriptorDecoder.wrap(this.catalogBuffer, 32, 80, 4);
            if (sessionId != this.descriptorDecoder.sessionId() || streamId != this.descriptorDecoder.streamId() || !Catalog.originalChannelContains(this.descriptorDecoder, channelFragment)) continue;
            return recordingId;
        }
        return -1L;
    }

    static boolean originalChannelContains(RecordingDescriptorDecoder descriptorDecoder, byte[] channelFragment) {
        int offset;
        int fragmentLength = channelFragment.length;
        if (0 == fragmentLength) {
            return true;
        }
        int limit = descriptorDecoder.limit();
        int strippedChannelLength = descriptorDecoder.strippedChannelLength();
        int originalChannelOffset = limit + RecordingDescriptorDecoder.strippedChannelHeaderLength() + strippedChannelLength;
        descriptorDecoder.limit(originalChannelOffset);
        int channelLength = descriptorDecoder.originalChannelLength();
        descriptorDecoder.limit(limit);
        DirectBuffer buffer = descriptorDecoder.buffer();
        int end = offset + (channelLength - fragmentLength);
        block0: for (offset = descriptorDecoder.offset() + descriptorDecoder.sbeBlockLength() + RecordingDescriptorDecoder.strippedChannelHeaderLength() + strippedChannelLength + RecordingDescriptorDecoder.originalChannelHeaderLength(); offset <= end; ++offset) {
            for (int i = 0; i < fragmentLength; ++i) {
                if (buffer.getByte(offset + i) != channelFragment[i]) continue block0;
            }
            return true;
        }
        return false;
    }

    void recordingStopped(long recordingId, long position, long timestampMs) {
        int offset = this.recordingDescriptorOffset(recordingId) + 32;
        long stopPosition = ByteOrder.nativeOrder() == RecordingDescriptorDecoder.BYTE_ORDER ? position : Long.reverseBytes(position);
        this.fieldAccessBuffer.putLong(offset + RecordingDescriptorDecoder.stopTimestampEncodingOffset(), timestampMs, RecordingDescriptorDecoder.BYTE_ORDER);
        this.fieldAccessBuffer.putLongVolatile(offset + RecordingDescriptorDecoder.stopPositionEncodingOffset(), stopPosition);
        this.forceWrites(this.catalogChannel);
    }

    void stopPosition(long recordingId, long position) {
        int offset = this.recordingDescriptorOffset(recordingId) + 32;
        long stopPosition = ByteOrder.nativeOrder() == RecordingDescriptorDecoder.BYTE_ORDER ? position : Long.reverseBytes(position);
        this.fieldAccessBuffer.putLongVolatile(offset + RecordingDescriptorDecoder.stopPositionEncodingOffset(), stopPosition);
        this.forceWrites(this.catalogChannel);
    }

    void extendRecording(long recordingId, long controlSessionId, long correlationId, int sessionId) {
        int offset = this.recordingDescriptorOffset(recordingId) + 32;
        long stopPosition = ByteOrder.nativeOrder() == RecordingDescriptorDecoder.BYTE_ORDER ? -1L : Long.reverseBytes(-1L);
        this.fieldAccessBuffer.putLong(offset + RecordingDescriptorDecoder.controlSessionIdEncodingOffset(), controlSessionId, RecordingDescriptorDecoder.BYTE_ORDER);
        this.fieldAccessBuffer.putLong(offset + RecordingDescriptorDecoder.correlationIdEncodingOffset(), correlationId, RecordingDescriptorDecoder.BYTE_ORDER);
        this.fieldAccessBuffer.putLong(offset + RecordingDescriptorDecoder.stopTimestampEncodingOffset(), -1L, RecordingDescriptorDecoder.BYTE_ORDER);
        this.fieldAccessBuffer.putInt(offset + RecordingDescriptorDecoder.sessionIdEncodingOffset(), sessionId, RecordingDescriptorDecoder.BYTE_ORDER);
        this.fieldAccessBuffer.putLongVolatile(offset + RecordingDescriptorDecoder.stopPositionEncodingOffset(), stopPosition);
        this.forceWrites(this.catalogChannel);
    }

    long startPosition(long recordingId) {
        int offset = this.recordingDescriptorOffset(recordingId) + 32 + RecordingDescriptorDecoder.startPositionEncodingOffset();
        long startPosition = this.fieldAccessBuffer.getLongVolatile(offset);
        return ByteOrder.nativeOrder() == RecordingDescriptorDecoder.BYTE_ORDER ? startPosition : Long.reverseBytes(startPosition);
    }

    void startPosition(long recordingId, long position) {
        int offset = this.recordingDescriptorOffset(recordingId) + 32 + RecordingDescriptorDecoder.startPositionEncodingOffset();
        this.fieldAccessBuffer.putLong(offset, position, RecordingDescriptorDecoder.BYTE_ORDER);
        this.forceWrites(this.catalogChannel);
    }

    long stopPosition(long recordingId) {
        int offset = this.recordingDescriptorOffset(recordingId) + 32 + RecordingDescriptorDecoder.stopPositionEncodingOffset();
        long stopPosition = this.fieldAccessBuffer.getLongVolatile(offset);
        return ByteOrder.nativeOrder() == RecordingDescriptorDecoder.BYTE_ORDER ? stopPosition : Long.reverseBytes(stopPosition);
    }

    RecordingSummary recordingSummary(long recordingId, RecordingSummary summary) {
        int offset = this.recordingDescriptorOffset(recordingId) + 32;
        summary.recordingId = recordingId;
        summary.startPosition = this.fieldAccessBuffer.getLong(offset + RecordingDescriptorDecoder.startPositionEncodingOffset(), RecordingDescriptorDecoder.BYTE_ORDER);
        summary.stopPosition = this.fieldAccessBuffer.getLong(offset + RecordingDescriptorDecoder.stopPositionEncodingOffset(), RecordingDescriptorDecoder.BYTE_ORDER);
        summary.initialTermId = this.fieldAccessBuffer.getInt(offset + RecordingDescriptorDecoder.initialTermIdEncodingOffset(), RecordingDescriptorDecoder.BYTE_ORDER);
        summary.segmentFileLength = this.fieldAccessBuffer.getInt(offset + RecordingDescriptorDecoder.segmentFileLengthEncodingOffset(), RecordingDescriptorDecoder.BYTE_ORDER);
        summary.termBufferLength = this.fieldAccessBuffer.getInt(offset + RecordingDescriptorDecoder.termBufferLengthEncodingOffset(), RecordingDescriptorDecoder.BYTE_ORDER);
        summary.mtuLength = this.fieldAccessBuffer.getInt(offset + RecordingDescriptorDecoder.mtuLengthEncodingOffset(), RecordingDescriptorDecoder.BYTE_ORDER);
        summary.streamId = this.fieldAccessBuffer.getInt(offset + RecordingDescriptorDecoder.streamIdEncodingOffset(), RecordingDescriptorDecoder.BYTE_ORDER);
        summary.sessionId = this.fieldAccessBuffer.getInt(offset + RecordingDescriptorDecoder.sessionIdEncodingOffset(), RecordingDescriptorDecoder.BYTE_ORDER);
        return summary;
    }

    static int descriptorLength(UnsafeBuffer descriptorBuffer) {
        return descriptorBuffer.getInt(RecordingDescriptorHeaderDecoder.lengthEncodingOffset(), RecordingDescriptorDecoder.BYTE_ORDER);
    }

    static boolean isValidDescriptor(UnsafeBuffer descriptorBuffer) {
        return descriptorBuffer.getByte(RecordingDescriptorHeaderDecoder.validEncodingOffset()) == 1;
    }

    static long calculateCatalogLength(long maxEntries) {
        return Math.min(maxEntries * 1024L + 1024L, Integer.MAX_VALUE);
    }

    static long calculateMaxEntries(long catalogLength, long recordLength) {
        if (Integer.MAX_VALUE == catalogLength) {
            return (Integer.MAX_VALUE - (recordLength - 1L)) / recordLength;
        }
        return catalogLength / recordLength - 1L;
    }

    int recordingDescriptorOffset(long recordingId) {
        return (int)(recordingId * (long)this.recordLength) + this.recordLength;
    }

    static void validateMaxEntries(long maxEntries) {
        if (maxEntries < 1L || maxEntries > MAX_ENTRIES) {
            throw new ArchiveException("Catalog max entries must be between 1 and " + MAX_ENTRIES + ": maxEntries=" + maxEntries);
        }
    }

    void growCatalog(long maxCatalogLength) {
        long catalogLength = this.catalogByteBuffer.capacity();
        long newCatalogLength = Math.min(catalogLength + (catalogLength >> 1), maxCatalogLength);
        if (newCatalogLength - catalogLength < (long)this.recordLength) {
            throw new ArchiveException("catalog is full, max length reached: " + maxCatalogLength);
        }
        try {
            this.unmapAndCloseChannel();
            this.catalogChannel = FileChannel.open(this.catalogFile.toPath(), StandardOpenOption.READ, StandardOpenOption.WRITE, StandardOpenOption.SPARSE);
            this.catalogByteBuffer = this.catalogChannel.map(FileChannel.MapMode.READ_WRITE, 0L, newCatalogLength);
        }
        catch (Exception ex) {
            this.close();
            LangUtil.rethrowUnchecked(ex);
        }
        this.catalogBuffer = new UnsafeBuffer(this.catalogByteBuffer);
        this.fieldAccessBuffer = new UnsafeBuffer(this.catalogByteBuffer);
        int oldMaxEntries = this.maxEntries();
        int newMaxEntries = (int)Catalog.calculateMaxEntries(newCatalogLength, this.recordLength);
        this.maxRecordingId = newMaxEntries - 1;
        this.catalogResized(oldMaxEntries, catalogLength, newMaxEntries, newCatalogLength);
    }

    void catalogResized(int maxEntries, long catalogLength, int newMaxEntries, long newCatalogLength) {
    }

    private void refreshCatalog(boolean fixOnRefresh, Checksum checksum, UnsafeBuffer buffer) {
        if (fixOnRefresh) {
            UnsafeBuffer tmpBuffer = null != buffer ? buffer : new UnsafeBuffer(ByteBuffer.allocateDirect(0x100000));
            this.forEach((headerEncoder, headerDecoder, descriptorEncoder, descriptorDecoder) -> this.refreshAndFixDescriptor(headerDecoder, descriptorEncoder, descriptorDecoder, checksum, tmpBuffer));
        } else {
            this.forEach((headerEncoder, headerDecoder, descriptorEncoder, descriptorDecoder) -> {
                this.nextRecordingId = descriptorDecoder.recordingId() + 1L;
            });
        }
    }

    private void refreshAndFixDescriptor(RecordingDescriptorHeaderDecoder headerDecoder, RecordingDescriptorEncoder encoder, RecordingDescriptorDecoder decoder, Checksum checksum, UnsafeBuffer buffer) {
        long recordingId = decoder.recordingId();
        if (1 == headerDecoder.valid() && -1L == decoder.stopPosition()) {
            String[] segmentFiles = Catalog.listSegmentFiles(this.archiveDir, recordingId);
            String maxSegmentFile = Catalog.findSegmentFileWithHighestPosition(segmentFiles);
            encoder.stopPosition(Catalog.computeStopPosition(this.archiveDir, maxSegmentFile, decoder.startPosition(), decoder.termBufferLength(), decoder.segmentFileLength(), checksum, buffer, segmentFile -> {
                throw new ArchiveException("Found potentially incomplete last fragment straddling page boundary in file: " + segmentFile.getAbsolutePath() + "\nRun `ArchiveTool verify` for corrective action!");
            }));
            encoder.stopTimestamp(this.epochClock.time());
        }
        this.nextRecordingId = recordingId + 1L;
    }

    private void forceWrites(FileChannel channel) {
        if (null != channel && this.forceWrites) {
            try {
                channel.force(this.forceMetadata);
            }
            catch (Exception ex) {
                LangUtil.rethrowUnchecked(ex);
            }
        }
    }

    static String[] listSegmentFiles(File archiveDir, long recordingId) {
        String prefix = recordingId + "-";
        return archiveDir.list((dir, name) -> name.startsWith(prefix) && name.endsWith(".rec"));
    }

    static String findSegmentFileWithHighestPosition(String[] segmentFiles) {
        if (null == segmentFiles || 0 == segmentFiles.length) {
            return null;
        }
        long maxSegmentPosition = -1L;
        String maxFileName = null;
        for (String filename : segmentFiles) {
            long filePosition = Catalog.parseSegmentFilePosition(filename);
            if (filePosition < 0L) {
                throw new ArchiveException("negative position encoded in the file name: " + filename);
            }
            if (filePosition <= maxSegmentPosition) continue;
            maxSegmentPosition = filePosition;
            maxFileName = filename;
        }
        return maxFileName;
    }

    static long parseSegmentFilePosition(String filename) {
        int dashOffset = filename.indexOf(45);
        if (-1 == dashOffset) {
            throw new ArchiveException("invalid filename format: " + filename);
        }
        int positionOffset = dashOffset + 1;
        int positionLength = filename.length() - positionOffset - ".rec".length();
        if (0 >= positionLength) {
            throw new ArchiveException("no position encoded in the segment file: " + filename);
        }
        return AsciiEncoding.parseLongAscii(filename, positionOffset, positionLength);
    }

    static long computeStopPosition(File archiveDir, String maxSegmentFile, long startPosition, int termLength, int segmentLength, Checksum checksum, UnsafeBuffer buffer, Predicate<File> truncateOnPageStraddle) {
        if (null == maxSegmentFile) {
            return startPosition;
        }
        int startTermOffset = (int)(startPosition & (long)(termLength - 1));
        long startTermBasePosition = startPosition - (long)startTermOffset;
        long segmentFileBasePosition = Catalog.parseSegmentFilePosition(maxSegmentFile);
        int fileOffset = segmentFileBasePosition == startTermBasePosition ? startTermOffset : 0;
        int segmentStopOffset = Catalog.recoverStopOffset(archiveDir, maxSegmentFile, fileOffset, segmentLength, truncateOnPageStraddle, checksum, buffer);
        return Math.max(segmentFileBasePosition + (long)segmentStopOffset, startPosition);
    }

    static boolean fragmentStraddlesPageBoundary(int fragmentOffset, int fragmentLength) {
        return fragmentOffset / 4096 != (fragmentOffset + (fragmentLength - 1)) / 4096;
    }

    private void unmapAndCloseChannel() {
        MappedByteBuffer buffer = this.catalogByteBuffer;
        IoUtil.unmap(buffer);
        this.catalogByteBuffer = null;
        CloseHelper.close(this.catalogChannel);
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private static int recoverStopOffset(File archiveDir, String segmentFile, int offset, int segmentFileLength, Predicate<File> truncateOnPageStraddle, Checksum checksum, UnsafeBuffer buffer) {
        File file = new File(archiveDir, segmentFile);
        try (FileChannel segment = FileChannel.open(file.toPath(), StandardOpenOption.READ, StandardOpenOption.WRITE);){
            int n;
            int lastFragmentOffset;
            int offsetLimit = (int)Math.min((long)segmentFileLength, segment.size());
            ByteBuffer byteBuffer = buffer.byteBuffer();
            int nextFragmentOffset = offset;
            int lastFragmentLength = 0;
            int bufferOffset = 0;
            block14: while (nextFragmentOffset < offsetLimit) {
                int bytesRead = Catalog.readNextChunk(segment, byteBuffer, nextFragmentOffset, offsetLimit);
                for (bufferOffset = 0; bufferOffset < bytesRead; nextFragmentOffset += lastFragmentLength, bufferOffset += lastFragmentLength) {
                    int frameLength = FrameDescriptor.frameLength(buffer, bufferOffset);
                    if (frameLength <= 0) break block14;
                    lastFragmentLength = BitUtil.align(frameLength, 32);
                }
            }
            if (Catalog.fragmentStraddlesPageBoundary(lastFragmentOffset = nextFragmentOffset - lastFragmentLength, lastFragmentLength) && !Catalog.isValidFragment(buffer, bufferOffset - lastFragmentLength, lastFragmentLength, checksum) && truncateOnPageStraddle.test(file)) {
                segment.truncate(lastFragmentOffset);
                byteBuffer.put(0, (byte)0).limit(1).position(0);
                segment.write(byteBuffer, segmentFileLength - 1);
                n = lastFragmentOffset;
                return n;
            }
            n = nextFragmentOffset;
            return n;
        }
        catch (IOException ex) {
            LangUtil.rethrowUnchecked(ex);
            return -1;
        }
    }

    private static int readNextChunk(FileChannel segment, ByteBuffer byteBuffer, int offset, int limit) throws IOException {
        int bytesRead;
        int position = offset;
        byteBuffer.clear().limit(Math.min(byteBuffer.capacity(), limit - position));
        while ((bytesRead = segment.read(byteBuffer, position)) >= 0) {
            position += bytesRead;
            if (byteBuffer.remaining() > 0) continue;
        }
        return position - offset;
    }

    private static boolean isValidFragment(UnsafeBuffer buffer, int fragmentOffset, int alignedFragmentLength, Checksum checksum) {
        return null != checksum && Catalog.hasValidChecksum(buffer, fragmentOffset, alignedFragmentLength, checksum) || Catalog.hasDataInAllPagesAfterStraddle(buffer, fragmentOffset, alignedFragmentLength);
    }

    private static boolean hasValidChecksum(UnsafeBuffer buffer, int fragmentOffset, int alignedFragmentLength, Checksum checksum) {
        int computedChecksum = checksum.compute(buffer.addressOffset(), fragmentOffset + 32, alignedFragmentLength - 32);
        int recordedChecksum = FrameDescriptor.frameSessionId(buffer, fragmentOffset);
        return recordedChecksum == computedChecksum;
    }

    private static boolean hasDataInAllPagesAfterStraddle(UnsafeBuffer buffer, int fragmentOffset, int alignedFragmentLength) {
        int endOffset = fragmentOffset + alignedFragmentLength;
        for (int straddleOffset = (fragmentOffset / 4096 + 1) * 4096; straddleOffset < endOffset; straddleOffset += 4096) {
            if (!Catalog.isEmptyPage(buffer, straddleOffset, endOffset)) continue;
            return false;
        }
        return true;
    }

    private static boolean isEmptyPage(UnsafeBuffer buffer, int pageStart, int endOffset) {
        int pageEnd = Math.min(pageStart + 4096, endOffset);
        for (int i = pageStart; i < pageEnd; i += 8) {
            if (0L == buffer.getLong(i)) continue;
            return false;
        }
        return true;
    }

    @FunctionalInterface
    public static interface CatalogEntryProcessor {
        public void accept(RecordingDescriptorHeaderEncoder var1, RecordingDescriptorHeaderDecoder var2, RecordingDescriptorEncoder var3, RecordingDescriptorDecoder var4);
    }
}

