/*
 * Decompiled with CFR 0.152.
 */
package io.aeron.cluster;

import io.aeron.archive.client.AeronArchive;
import io.aeron.cluster.RecordingExtent;
import io.aeron.cluster.client.ClusterException;
import java.io.File;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.nio.channels.FileChannel;
import java.nio.file.OpenOption;
import java.nio.file.StandardOpenOption;
import java.util.ArrayList;
import java.util.List;
import org.agrona.BitUtil;
import org.agrona.CloseHelper;
import org.agrona.LangUtil;
import org.agrona.collections.IntArrayList;
import org.agrona.collections.Long2LongHashMap;
import org.agrona.collections.MutableReference;
import org.agrona.concurrent.UnsafeBuffer;

public final class RecordingLog
implements AutoCloseable {
    public static final String RECORDING_LOG_FILE_NAME = "recording.log";
    public static final int ENTRY_TYPE_TERM = 0;
    public static final int ENTRY_TYPE_SNAPSHOT = 1;
    public static final int ENTRY_TYPE_INVALID_FLAG = Integer.MIN_VALUE;
    public static final int RECORDING_ID_OFFSET = 0;
    public static final int LEADERSHIP_TERM_ID_OFFSET = 8;
    public static final int TERM_BASE_LOG_POSITION_OFFSET = 16;
    public static final int LOG_POSITION_OFFSET = 24;
    public static final int TIMESTAMP_OFFSET = 32;
    public static final int SERVICE_ID_OFFSET = 40;
    public static final int ENTRY_TYPE_OFFSET = 44;
    private static final int ENTRY_LENGTH = BitUtil.align(48, 64);
    private long filePosition = 0L;
    private int nextEntryIndex;
    private final FileChannel fileChannel;
    private final ByteBuffer byteBuffer = ByteBuffer.allocateDirect(4096).order(ByteOrder.LITTLE_ENDIAN);
    private final UnsafeBuffer buffer = new UnsafeBuffer(this.byteBuffer);
    private final ArrayList<Entry> entriesCache = new ArrayList();
    private final Long2LongHashMap cacheIndexByLeadershipTermIdMap = new Long2LongHashMap(-1L);
    private final IntArrayList invalidSnapshots = new IntArrayList();

    public RecordingLog(File parentDir) {
        File logFile = new File(parentDir, RECORDING_LOG_FILE_NAME);
        boolean newFile = !logFile.exists();
        try {
            this.fileChannel = FileChannel.open(logFile.toPath(), StandardOpenOption.CREATE, StandardOpenOption.READ, StandardOpenOption.WRITE);
            if (newFile) {
                RecordingLog.syncDirectory(parentDir);
            } else {
                this.reload();
            }
        }
        catch (IOException ex) {
            throw new ClusterException(ex);
        }
    }

    @Override
    public void close() {
        CloseHelper.close(this.fileChannel);
    }

    public void force(int fileSyncLevel) {
        if (fileSyncLevel > 0) {
            try {
                this.fileChannel.force(fileSyncLevel > 1);
            }
            catch (IOException ex) {
                LangUtil.rethrowUnchecked(ex);
            }
        }
    }

    public List<Entry> entries() {
        return this.entriesCache;
    }

    public int nextEntryIndex() {
        return this.nextEntryIndex;
    }

    public void reload() {
        this.filePosition = 0L;
        this.entriesCache.clear();
        this.cacheIndexByLeadershipTermIdMap.clear();
        this.invalidSnapshots.clear();
        this.cacheIndexByLeadershipTermIdMap.compact();
        this.nextEntryIndex = 0;
        this.byteBuffer.clear();
        try {
            while (true) {
                int bytesRead = this.fileChannel.read(this.byteBuffer, this.filePosition);
                if (this.byteBuffer.remaining() == 0) {
                    this.byteBuffer.flip();
                    this.captureEntriesFromBuffer(this.byteBuffer, this.buffer, this.entriesCache);
                    this.byteBuffer.clear();
                }
                if (bytesRead <= 0) {
                    if (this.byteBuffer.position() > 0) {
                        this.byteBuffer.flip();
                        this.captureEntriesFromBuffer(this.byteBuffer, this.buffer, this.entriesCache);
                        this.byteBuffer.clear();
                    }
                    break;
                }
                this.filePosition += (long)bytesRead;
            }
        }
        catch (IOException ex) {
            LangUtil.rethrowUnchecked(ex);
        }
    }

    public long findLastTermRecordingId() {
        Entry lastTerm = this.findLastTerm();
        return null != lastTerm ? lastTerm.recordingId : -1L;
    }

    public Entry findLastTerm() {
        for (int i = this.entriesCache.size() - 1; i >= 0; --i) {
            Entry entry = this.entriesCache.get(i);
            if (!RecordingLog.isValidTerm(entry)) continue;
            return entry;
        }
        return null;
    }

    public Entry getTermEntry(long leadershipTermId) {
        int index = (int)this.cacheIndexByLeadershipTermIdMap.get(leadershipTermId);
        if (-1 != index) {
            return this.entriesCache.get(index);
        }
        throw new ClusterException("unknown leadershipTermId=" + leadershipTermId);
    }

    public Entry findTermEntry(long leadershipTermId) {
        int index = (int)this.cacheIndexByLeadershipTermIdMap.get(leadershipTermId);
        if (-1 != index) {
            return this.entriesCache.get(index);
        }
        return null;
    }

    public Entry getLatestSnapshot(int serviceId) {
        for (int i = this.entriesCache.size() - 1; i >= 0; --i) {
            Entry snapshot;
            Entry entry = this.entriesCache.get(i);
            if (!RecordingLog.isValidSnapshot(entry) || -1 != entry.serviceId) continue;
            if (-1 == serviceId) {
                return entry;
            }
            int serviceSnapshotIndex = i - (serviceId + 1);
            if (serviceSnapshotIndex < 0 || !RecordingLog.isValidSnapshot(snapshot = this.entriesCache.get(serviceSnapshotIndex)) || serviceId != snapshot.serviceId) continue;
            return snapshot;
        }
        return null;
    }

    public boolean invalidateLatestSnapshot() {
        int index = -1;
        for (int i = this.entriesCache.size() - 1; i >= 0; --i) {
            Entry entry = this.entriesCache.get(i);
            if (!RecordingLog.isValidSnapshot(entry) || -1 != entry.serviceId) continue;
            index = i;
            break;
        }
        if (index >= 0) {
            Entry entry;
            int serviceId = -1;
            for (int i = index; i >= 0 && RecordingLog.isValidSnapshot(entry = this.entriesCache.get(i)) && entry.serviceId == serviceId; ++serviceId, --i) {
                this.invalidateEntry(entry.leadershipTermId, entry.entryIndex);
            }
            return true;
        }
        return false;
    }

    public long getTermTimestamp(long leadershipTermId) {
        int index = (int)this.cacheIndexByLeadershipTermIdMap.get(leadershipTermId);
        if (-1 != index) {
            return this.entriesCache.get((int)index).timestamp;
        }
        return -1L;
    }

    public RecoveryPlan createRecoveryPlan(AeronArchive archive, int serviceCount) {
        ArrayList<Snapshot> snapshots = new ArrayList<Snapshot>();
        MutableReference<Log> logRef = new MutableReference<Log>();
        RecordingLog.planRecovery(snapshots, logRef, this.entriesCache, archive, serviceCount);
        long lastLeadershipTermId = -1L;
        long lastTermBaseLogPosition = 0L;
        long committedLogPosition = 0L;
        long appendedLogPosition = 0L;
        int snapshotStepsSize = snapshots.size();
        if (snapshotStepsSize > 0) {
            Snapshot snapshot = snapshots.get(0);
            lastLeadershipTermId = snapshot.leadershipTermId;
            lastTermBaseLogPosition = snapshot.termBaseLogPosition;
            appendedLogPosition = snapshot.logPosition;
            committedLogPosition = snapshot.logPosition;
        }
        if (logRef.get() != null) {
            Log log = logRef.get();
            lastLeadershipTermId = log.leadershipTermId;
            lastTermBaseLogPosition = log.termBaseLogPosition;
            appendedLogPosition = log.stopPosition;
            committedLogPosition = -1L != log.logPosition ? log.logPosition : committedLogPosition;
        }
        return new RecoveryPlan(lastLeadershipTermId, lastTermBaseLogPosition, appendedLogPosition, committedLogPosition, snapshots, logRef.get());
    }

    public static RecoveryPlan createRecoveryPlan(ArrayList<Snapshot> snapshots) {
        long lastLeadershipTermId = -1L;
        long lastTermBaseLogPosition = 0L;
        long committedLogPosition = 0L;
        long appendedLogPosition = 0L;
        int snapshotStepsSize = snapshots.size();
        if (snapshotStepsSize > 0) {
            Snapshot snapshot = snapshots.get(0);
            lastLeadershipTermId = snapshot.leadershipTermId;
            lastTermBaseLogPosition = snapshot.termBaseLogPosition;
            appendedLogPosition = snapshot.logPosition;
            committedLogPosition = snapshot.logPosition;
        }
        return new RecoveryPlan(lastLeadershipTermId, lastTermBaseLogPosition, appendedLogPosition, committedLogPosition, snapshots, null);
    }

    public boolean isUnknown(long leadershipTermId) {
        return -1L == this.cacheIndexByLeadershipTermIdMap.get(leadershipTermId);
    }

    public void appendTerm(long recordingId, long leadershipTermId, long termBaseLogPosition, long timestamp) {
        int size = this.entriesCache.size();
        if (size > 0) {
            Entry lastEntry = this.entriesCache.get(size - 1);
            if (lastEntry.type != -1 && lastEntry.leadershipTermId >= leadershipTermId) {
                throw new ClusterException("leadershipTermId out of sequence: previous " + lastEntry.leadershipTermId + " this " + leadershipTermId);
            }
            long previousLeadershipTermId = leadershipTermId - 1L;
            if (-1L != this.cacheIndexByLeadershipTermIdMap.get(previousLeadershipTermId)) {
                this.commitLogPosition(previousLeadershipTermId, termBaseLogPosition);
            }
        }
        this.append(0, recordingId, leadershipTermId, termBaseLogPosition, -1L, timestamp, -1);
        this.cacheIndexByLeadershipTermIdMap.put(leadershipTermId, this.entriesCache.size() - 1);
    }

    public void appendSnapshot(long recordingId, long leadershipTermId, long termBaseLogPosition, long logPosition, long timestamp, int serviceId) {
        int size = this.entriesCache.size();
        if (size > 0) {
            if (this.restoreSnapshotMarkedWithInvalid(recordingId, leadershipTermId, termBaseLogPosition, logPosition, timestamp, serviceId)) {
                return;
            }
            Entry entry = this.entriesCache.get(size - 1);
            if (entry.type == 0 && entry.leadershipTermId != leadershipTermId) {
                throw new ClusterException("leadershipTermId out of sequence: previous " + entry.leadershipTermId + " this " + leadershipTermId);
            }
        }
        this.append(1, recordingId, leadershipTermId, termBaseLogPosition, logPosition, timestamp, serviceId);
    }

    public void commitLogPosition(long leadershipTermId, long logPosition) {
        int index = (int)this.cacheIndexByLeadershipTermIdMap.get(leadershipTermId);
        if (-1 == index) {
            throw new ClusterException("unknown leadershipTermId=" + leadershipTermId);
        }
        Entry entry = this.entriesCache.get(index);
        if (entry.logPosition != logPosition) {
            this.commitEntryLogPosition(entry.entryIndex, logPosition);
            this.entriesCache.set(index, new Entry(entry.recordingId, entry.leadershipTermId, entry.termBaseLogPosition, logPosition, entry.timestamp, entry.serviceId, entry.type, entry.isValid, entry.entryIndex));
        }
    }

    public void invalidateEntry(long leadershipTermId, int entryIndex) {
        Entry invalidEntry = null;
        for (int i = this.entriesCache.size() - 1; i >= 0; --i) {
            Entry entry = this.entriesCache.get(i);
            if (entry.leadershipTermId != leadershipTermId || entry.entryIndex != entryIndex) continue;
            invalidEntry = entry.invalidate();
            this.entriesCache.set(i, invalidEntry);
            if (0 == entry.type) {
                this.cacheIndexByLeadershipTermIdMap.remove(leadershipTermId);
                break;
            }
            if (1 != entry.type) break;
            this.invalidSnapshots.add(i);
            break;
        }
        if (null == invalidEntry) {
            throw new ClusterException("unknown entry index: " + entryIndex);
        }
        int invalidEntryType = Integer.MIN_VALUE | invalidEntry.type;
        this.buffer.putInt(0, invalidEntryType, ByteOrder.LITTLE_ENDIAN);
        this.byteBuffer.limit(4).position(0);
        long position = (long)invalidEntry.entryIndex * (long)ENTRY_LENGTH + 44L;
        try {
            if (4 != this.fileChannel.write(this.byteBuffer, position)) {
                throw new ClusterException("failed to write field atomically");
            }
        }
        catch (Exception ex) {
            LangUtil.rethrowUnchecked(ex);
        }
    }

    void removeEntry(long leadershipTermId, int entryIndex) {
        int index = -1;
        for (int i = this.entriesCache.size() - 1; i >= 0; --i) {
            Entry entry = this.entriesCache.get(i);
            if (entry.leadershipTermId != leadershipTermId || entry.entryIndex != entryIndex) continue;
            index = entry.entryIndex;
            break;
        }
        if (-1 == index) {
            throw new ClusterException("unknown entry index: " + entryIndex);
        }
        this.buffer.putInt(0, -1, ByteOrder.LITTLE_ENDIAN);
        this.byteBuffer.limit(4).position(0);
        long position = (long)index * (long)ENTRY_LENGTH + 44L;
        try {
            if (4 != this.fileChannel.write(this.byteBuffer, position)) {
                throw new ClusterException("failed to write field atomically");
            }
            this.reload();
        }
        catch (Exception ex) {
            LangUtil.rethrowUnchecked(ex);
        }
    }

    public String toString() {
        return "RecordingLog{entries=" + this.entriesCache + ", cacheIndex=" + this.cacheIndexByLeadershipTermIdMap + '}';
    }

    static void addSnapshots(ArrayList<Snapshot> snapshots, ArrayList<Entry> entries, int serviceCount, int snapshotIndex) {
        Entry snapshot = entries.get(snapshotIndex);
        snapshots.add(new Snapshot(snapshot.recordingId, snapshot.leadershipTermId, snapshot.termBaseLogPosition, snapshot.logPosition, snapshot.timestamp, snapshot.serviceId));
        for (int i = 1; i <= serviceCount; ++i) {
            if (snapshotIndex - i < 0) {
                throw new ClusterException("snapshot missing for service at index " + i + " in " + entries);
            }
            Entry entry = entries.get(snapshotIndex - i);
            if (1 != entry.type || entry.leadershipTermId != snapshot.leadershipTermId || entry.logPosition != snapshot.logPosition) continue;
            snapshots.add(entry.serviceId + 1, new Snapshot(entry.recordingId, entry.leadershipTermId, entry.termBaseLogPosition, entry.logPosition, entry.timestamp, entry.serviceId));
        }
    }

    private boolean restoreSnapshotMarkedWithInvalid(long recordingId, long leadershipTermId, long termBaseLogPosition, long logPosition, long timestamp, int serviceId) {
        for (int i = this.invalidSnapshots.size() - 1; i >= 0; --i) {
            int entryCacheIndex = this.invalidSnapshots.getInt(i);
            Entry entry = this.entriesCache.get(entryCacheIndex);
            if (!RecordingLog.entryMatches(entry, leadershipTermId, termBaseLogPosition, logPosition, serviceId)) continue;
            Entry validatedEntry = new Entry(recordingId, leadershipTermId, termBaseLogPosition, logPosition, timestamp, serviceId, 1, true, entry.entryIndex);
            this.writeEntryToBuffer(validatedEntry, this.buffer, this.byteBuffer);
            long position = (long)entry.entryIndex * (long)ENTRY_LENGTH;
            try {
                if (ENTRY_LENGTH != this.fileChannel.write(this.byteBuffer, position)) {
                    throw new ClusterException("failed to write entry atomically");
                }
            }
            catch (IOException ex) {
                LangUtil.rethrowUnchecked(ex);
            }
            this.entriesCache.set(entryCacheIndex, validatedEntry);
            this.invalidSnapshots.fastUnorderedRemove(i);
            return true;
        }
        return false;
    }

    private void append(int entryType, long recordingId, long leadershipTermId, long termBaseLogPosition, long logPosition, long timestamp, int serviceId) {
        Entry entry = new Entry(recordingId, leadershipTermId, termBaseLogPosition, logPosition, timestamp, serviceId, entryType, true, this.nextEntryIndex);
        this.writeEntryToBuffer(entry, this.buffer, this.byteBuffer);
        try {
            int bytesWritten = this.fileChannel.write(this.byteBuffer, this.filePosition);
            if (ENTRY_LENGTH != bytesWritten) {
                throw new ClusterException("failed to write entry atomically");
            }
            this.filePosition += (long)bytesWritten;
        }
        catch (IOException ex) {
            LangUtil.rethrowUnchecked(ex);
        }
        ++this.nextEntryIndex;
        this.entriesCache.add(entry);
    }

    private void writeEntryToBuffer(Entry entry, UnsafeBuffer buffer, ByteBuffer byteBuffer) {
        buffer.putLong(0, entry.recordingId, ByteOrder.LITTLE_ENDIAN);
        buffer.putLong(8, entry.leadershipTermId, ByteOrder.LITTLE_ENDIAN);
        buffer.putLong(16, entry.termBaseLogPosition, ByteOrder.LITTLE_ENDIAN);
        buffer.putLong(24, entry.logPosition, ByteOrder.LITTLE_ENDIAN);
        buffer.putLong(32, entry.timestamp, ByteOrder.LITTLE_ENDIAN);
        buffer.putInt(40, entry.serviceId, ByteOrder.LITTLE_ENDIAN);
        buffer.putInt(44, entry.type, ByteOrder.LITTLE_ENDIAN);
        byteBuffer.limit(ENTRY_LENGTH).position(0);
    }

    private void captureEntriesFromBuffer(ByteBuffer byteBuffer, UnsafeBuffer buffer, ArrayList<Entry> entries) {
        int length = byteBuffer.limit();
        for (int i = 0; i < length; i += ENTRY_LENGTH) {
            int entryType = buffer.getInt(i + 44);
            if (-1 != entryType) {
                int type = entryType & Integer.MAX_VALUE;
                boolean isValid = (entryType & Integer.MIN_VALUE) == 0;
                Entry entry = new Entry(buffer.getLong(i + 0, ByteOrder.LITTLE_ENDIAN), buffer.getLong(i + 8, ByteOrder.LITTLE_ENDIAN), buffer.getLong(i + 16, ByteOrder.LITTLE_ENDIAN), buffer.getLong(i + 24, ByteOrder.LITTLE_ENDIAN), buffer.getLong(i + 32, ByteOrder.LITTLE_ENDIAN), buffer.getInt(i + 40, ByteOrder.LITTLE_ENDIAN), type, isValid, this.nextEntryIndex);
                entries.add(entry);
                if (RecordingLog.isValidTerm(entry)) {
                    this.cacheIndexByLeadershipTermIdMap.put(entry.leadershipTermId, entries.size() - 1);
                }
                if (1 == entry.type && !entry.isValid) {
                    this.invalidSnapshots.add(entries.size() - 1);
                }
            }
            ++this.nextEntryIndex;
        }
    }

    private static void syncDirectory(File dir) {
        try (FileChannel fileChannel = FileChannel.open(dir.toPath(), new OpenOption[0]);){
            fileChannel.force(true);
        }
        catch (IOException iOException) {
            // empty catch block
        }
    }

    private void commitEntryLogPosition(int entryIndex, long value) {
        this.buffer.putLong(0, value, ByteOrder.LITTLE_ENDIAN);
        this.byteBuffer.limit(8).position(0);
        long position = (long)entryIndex * (long)ENTRY_LENGTH + 24L;
        try {
            if (8 != this.fileChannel.write(this.byteBuffer, position)) {
                throw new ClusterException("failed to write field atomically");
            }
        }
        catch (IOException ex) {
            LangUtil.rethrowUnchecked(ex);
        }
    }

    private static void planRecovery(ArrayList<Snapshot> snapshots, MutableReference<Log> logRef, ArrayList<Entry> entries, AeronArchive archive, int serviceCount) {
        if (entries.isEmpty()) {
            return;
        }
        int logIndex = -1;
        int snapshotIndex = -1;
        for (int i = entries.size() - 1; i >= 0; --i) {
            Entry entry = entries.get(i);
            if (-1 == snapshotIndex && RecordingLog.isValidSnapshot(entry) && entry.serviceId == -1) {
                snapshotIndex = i;
                continue;
            }
            if (-1 == logIndex && entry.isValid && 0 == entry.type && -1L != entry.recordingId) {
                logIndex = i;
                continue;
            }
            if (-1 != snapshotIndex && -1 != logIndex) break;
        }
        if (-1 != snapshotIndex) {
            RecordingLog.addSnapshots(snapshots, entries, serviceCount, snapshotIndex);
        }
        if (-1 != logIndex) {
            Entry entry = entries.get(logIndex);
            RecordingExtent recordingExtent = new RecordingExtent();
            if (archive.listRecording(entry.recordingId, recordingExtent) == 0) {
                throw new ClusterException("unknown recording id: " + entry.recordingId);
            }
            long startPosition = -1 == snapshotIndex ? recordingExtent.startPosition : snapshots.get((int)0).logPosition;
            logRef.set(new Log(entry.recordingId, entry.leadershipTermId, entry.termBaseLogPosition, entry.logPosition, startPosition, recordingExtent.stopPosition, recordingExtent.initialTermId, recordingExtent.termBufferLength, recordingExtent.mtuLength, recordingExtent.sessionId));
        }
    }

    private static boolean isValidSnapshot(Entry entry) {
        return entry.isValid && 1 == entry.type;
    }

    private static boolean isValidTerm(Entry entry) {
        return 0 == entry.type && entry.isValid;
    }

    private static boolean entryMatches(Entry entry, long leadershipTermId, long termBaseLogPosition, long logPosition, int serviceId) {
        return entry.leadershipTermId == leadershipTermId && entry.termBaseLogPosition == termBaseLogPosition && entry.logPosition == logPosition && entry.serviceId == serviceId;
    }

    public static final class RecoveryPlan {
        public final long lastLeadershipTermId;
        public final long lastTermBaseLogPosition;
        public final long appendedLogPosition;
        public final long committedLogPosition;
        public final ArrayList<Snapshot> snapshots;
        public final Log log;

        RecoveryPlan(long lastLeadershipTermId, long lastTermBaseLogPosition, long appendedLogPosition, long committedLogPosition, ArrayList<Snapshot> snapshots, Log log) {
            this.lastLeadershipTermId = lastLeadershipTermId;
            this.lastTermBaseLogPosition = lastTermBaseLogPosition;
            this.appendedLogPosition = appendedLogPosition;
            this.committedLogPosition = committedLogPosition;
            this.snapshots = snapshots;
            this.log = log;
        }

        public String toString() {
            return "RecoveryPlan{lastLeadershipTermId=" + this.lastLeadershipTermId + ", lastTermBaseLogPosition=" + this.lastTermBaseLogPosition + ", appendedLogPosition=" + this.appendedLogPosition + ", committedLogPosition=" + this.committedLogPosition + ", snapshots=" + this.snapshots + ", log=" + this.log + '}';
        }
    }

    public static final class Log {
        public final long recordingId;
        public final long leadershipTermId;
        public final long termBaseLogPosition;
        public final long logPosition;
        public final long startPosition;
        public final long stopPosition;
        public final int initialTermId;
        public final int termBufferLength;
        public final int mtuLength;
        public final int sessionId;

        Log(long recordingId, long leadershipTermId, long termBaseLogPosition, long logPosition, long startPosition, long stopPosition, int initialTermId, int termBufferLength, int mtuLength, int sessionId) {
            this.recordingId = recordingId;
            this.leadershipTermId = leadershipTermId;
            this.termBaseLogPosition = termBaseLogPosition;
            this.logPosition = logPosition;
            this.startPosition = startPosition;
            this.stopPosition = stopPosition;
            this.initialTermId = initialTermId;
            this.termBufferLength = termBufferLength;
            this.mtuLength = mtuLength;
            this.sessionId = sessionId;
        }

        public String toString() {
            return "Log{recordingId=" + this.recordingId + ", leadershipTermId=" + this.leadershipTermId + ", termBaseLogPosition=" + this.termBaseLogPosition + ", logPosition=" + this.logPosition + ", startPosition=" + this.startPosition + ", stopPosition=" + this.stopPosition + ", initialTermId=" + this.initialTermId + ", termBufferLength=" + this.termBufferLength + ", mtuLength=" + this.mtuLength + ", sessionId=" + this.sessionId + '}';
        }
    }

    public static final class Snapshot {
        public final long recordingId;
        public final long leadershipTermId;
        public final long termBaseLogPosition;
        public final long logPosition;
        public final long timestamp;
        public final int serviceId;

        Snapshot(long recordingId, long leadershipTermId, long termBaseLogPosition, long logPosition, long timestamp, int serviceId) {
            this.recordingId = recordingId;
            this.leadershipTermId = leadershipTermId;
            this.termBaseLogPosition = termBaseLogPosition;
            this.logPosition = logPosition;
            this.timestamp = timestamp;
            this.serviceId = serviceId;
        }

        public String toString() {
            return "Snapshot{recordingId=" + this.recordingId + ", leadershipTermId=" + this.leadershipTermId + ", termBaseLogPosition=" + this.termBaseLogPosition + ", logPosition=" + this.logPosition + ", timestamp=" + this.timestamp + ", serviceId=" + this.serviceId + '}';
        }
    }

    public static final class Entry {
        public final long recordingId;
        public final long leadershipTermId;
        public final long termBaseLogPosition;
        public final long logPosition;
        public final long timestamp;
        public final int serviceId;
        public final int type;
        public final int entryIndex;
        public final boolean isValid;

        Entry(long recordingId, long leadershipTermId, long termBaseLogPosition, long logPosition, long timestamp, int serviceId, int type, boolean isValid, int entryIndex) {
            this.recordingId = recordingId;
            this.leadershipTermId = leadershipTermId;
            this.termBaseLogPosition = termBaseLogPosition;
            this.logPosition = logPosition;
            this.timestamp = timestamp;
            this.serviceId = serviceId;
            this.type = type;
            this.entryIndex = entryIndex;
            this.isValid = isValid;
        }

        Entry invalidate() {
            return new Entry(this.recordingId, this.leadershipTermId, this.termBaseLogPosition, this.logPosition, this.timestamp, this.serviceId, this.type, false, this.entryIndex);
        }

        public String toString() {
            return "Entry{recordingId=" + this.recordingId + ", leadershipTermId=" + this.leadershipTermId + ", termBaseLogPosition=" + this.termBaseLogPosition + ", logPosition=" + this.logPosition + ", timestamp=" + this.timestamp + ", serviceId=" + this.serviceId + ", type=" + this.type + ", isValid=" + this.isValid + ", entryIndex=" + this.entryIndex + '}';
        }
    }
}

