/*
 * Decompiled with CFR 0.152.
 */
package io.aeron.archive;

import io.aeron.Aeron;
import io.aeron.AvailableImageHandler;
import io.aeron.ChannelUri;
import io.aeron.ChannelUriStringBuilder;
import io.aeron.Counter;
import io.aeron.ExclusivePublication;
import io.aeron.Image;
import io.aeron.Publication;
import io.aeron.Subscription;
import io.aeron.UnavailableCounterHandler;
import io.aeron.archive.Archive;
import io.aeron.archive.ArchiveMarkFile;
import io.aeron.archive.Catalog;
import io.aeron.archive.ControlRequestDecoders;
import io.aeron.archive.ControlResponseProxy;
import io.aeron.archive.ControlSession;
import io.aeron.archive.ControlSessionDemuxer;
import io.aeron.archive.ControlSessionProxy;
import io.aeron.archive.DeleteSegmentsSession;
import io.aeron.archive.ListRecordingSubscriptionsSession;
import io.aeron.archive.ListRecordingsForUriSession;
import io.aeron.archive.ListRecordingsSession;
import io.aeron.archive.RecordingEventsProxy;
import io.aeron.archive.RecordingSession;
import io.aeron.archive.RecordingSummary;
import io.aeron.archive.ReplaySession;
import io.aeron.archive.ReplicationSession;
import io.aeron.archive.Session;
import io.aeron.archive.SessionWorker;
import io.aeron.archive.client.AeronArchive;
import io.aeron.archive.client.ArchiveEvent;
import io.aeron.archive.codecs.RecordingDescriptorDecoder;
import io.aeron.archive.codecs.RecordingSignal;
import io.aeron.archive.codecs.SourceLocation;
import io.aeron.archive.status.RecordingPos;
import io.aeron.exceptions.AeronException;
import io.aeron.exceptions.TimeoutException;
import io.aeron.logbuffer.LogBufferDescriptor;
import io.aeron.protocol.DataHeaderFlyweight;
import io.aeron.security.Authenticator;
import java.io.File;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;
import java.nio.file.StandardOpenOption;
import java.nio.file.attribute.FileAttribute;
import java.util.ArrayDeque;
import java.util.EnumSet;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.TimeUnit;
import org.agrona.CloseHelper;
import org.agrona.ErrorHandler;
import org.agrona.LangUtil;
import org.agrona.SemanticVersion;
import org.agrona.Strings;
import org.agrona.SystemUtil;
import org.agrona.collections.Int2ObjectHashMap;
import org.agrona.collections.Long2LongCounterMap;
import org.agrona.collections.Long2ObjectHashMap;
import org.agrona.collections.Object2ObjectHashMap;
import org.agrona.concurrent.AgentInvoker;
import org.agrona.concurrent.AgentTerminationException;
import org.agrona.concurrent.CachedEpochClock;
import org.agrona.concurrent.EpochClock;
import org.agrona.concurrent.UnsafeBuffer;
import org.agrona.concurrent.status.CountersReader;

abstract class ArchiveConductor
extends SessionWorker<Session>
implements AvailableImageHandler,
UnavailableCounterHandler {
    private static final long MARK_FILE_UPDATE_INTERVAL_MS = TimeUnit.SECONDS.toMillis(1L);
    private static final EnumSet<StandardOpenOption> FILE_OPTIONS = EnumSet.of(StandardOpenOption.READ, StandardOpenOption.WRITE);
    private static final FileAttribute<?>[] NO_ATTRIBUTES = new FileAttribute[0];
    private final long closeHandlerRegistrationId;
    private final long unavailableCounterHandlerRegistrationId;
    private final long connectTimeoutMs;
    private long nextSessionId = ThreadLocalRandom.current().nextInt(Integer.MAX_VALUE);
    private long markFileUpdateDeadlineMs = 0L;
    private int replayId = 1;
    private volatile boolean isAbort;
    private final RecordingSummary recordingSummary = new RecordingSummary();
    private final ControlRequestDecoders decoders = new ControlRequestDecoders();
    private final ArrayDeque<Runnable> taskQueue = new ArrayDeque();
    private final Long2ObjectHashMap<ReplaySession> replaySessionByIdMap = new Long2ObjectHashMap();
    private final Long2ObjectHashMap<RecordingSession> recordingSessionByIdMap = new Long2ObjectHashMap();
    private final Long2ObjectHashMap<ReplicationSession> replicationSessionByIdMap = new Long2ObjectHashMap();
    private final Int2ObjectHashMap<Counter> counterByIdMap = new Int2ObjectHashMap();
    private final Object2ObjectHashMap<String, Subscription> recordingSubscriptionByKeyMap = new Object2ObjectHashMap();
    private final UnsafeBuffer descriptorBuffer = new UnsafeBuffer();
    private final RecordingDescriptorDecoder recordingDescriptorDecoder = new RecordingDescriptorDecoder();
    private final ControlResponseProxy controlResponseProxy = new ControlResponseProxy();
    private final UnsafeBuffer counterMetadataBuffer = new UnsafeBuffer(new byte[512]);
    private final Long2LongCounterMap subscriptionRefCountMap = new Long2LongCounterMap(0L);
    private final Aeron aeron;
    private final AgentInvoker aeronAgentInvoker;
    private final AgentInvoker driverAgentInvoker;
    private final EpochClock epochClock;
    private final CachedEpochClock cachedEpochClock = new CachedEpochClock();
    private final File archiveDir;
    private final Subscription controlSubscription;
    private final Subscription localControlSubscription;
    private final Catalog catalog;
    private final ArchiveMarkFile markFile;
    private final RecordingEventsProxy recordingEventsProxy;
    private final Authenticator authenticator;
    private final ControlSessionProxy controlSessionProxy;
    final Archive.Context ctx;
    SessionWorker<RecordingSession> recorder;
    SessionWorker<ReplaySession> replayer;

    ArchiveConductor(Archive.Context ctx) {
        super("archive-conductor", ctx.countedErrorHandler());
        this.ctx = ctx;
        this.aeron = ctx.aeron();
        this.aeronAgentInvoker = this.aeron.conductorAgentInvoker();
        this.driverAgentInvoker = ctx.mediaDriverAgentInvoker();
        this.epochClock = ctx.epochClock();
        this.archiveDir = ctx.archiveDir();
        this.connectTimeoutMs = TimeUnit.NANOSECONDS.toMillis(ctx.connectTimeoutNs());
        this.unavailableCounterHandlerRegistrationId = this.aeron.addUnavailableCounterHandler((UnavailableCounterHandler)this);
        this.closeHandlerRegistrationId = this.aeron.addCloseHandler(this::abort);
        ChannelUri controlChannelUri = ChannelUri.parse((CharSequence)ctx.controlChannel());
        controlChannelUri.put("sparse", Boolean.toString(ctx.controlTermBufferSparse()));
        this.controlSubscription = this.aeron.addSubscription(controlChannelUri.toString(), ctx.controlStreamId(), (AvailableImageHandler)this, null);
        this.localControlSubscription = this.aeron.addSubscription(ctx.localControlChannel(), ctx.localControlStreamId(), (AvailableImageHandler)this, null);
        this.recordingEventsProxy = ctx.recordingEventsEnabled() ? new RecordingEventsProxy((Publication)this.aeron.addExclusivePublication(ctx.recordingEventsChannel(), ctx.recordingEventsStreamId())) : null;
        this.catalog = ctx.catalog();
        this.markFile = ctx.archiveMarkFile();
        this.cachedEpochClock.update(this.epochClock.time());
        this.authenticator = (Authenticator)ctx.authenticatorSupplier().get();
        this.controlSessionProxy = new ControlSessionProxy(this.controlResponseProxy);
    }

    public void onStart() {
        this.recorder = this.newRecorder();
        this.replayer = this.newReplayer();
    }

    public void onAvailableImage(Image image) {
        this.addSession(new ControlSessionDemuxer(this.decoders, image, this));
    }

    public void onUnavailableCounter(CountersReader countersReader, long registrationId, int counterId) {
        Counter counter = (Counter)this.counterByIdMap.remove(counterId);
        if (null != counter) {
            counter.close();
            for (ReplaySession session : this.replaySessionByIdMap.values()) {
                if (session.limitPosition() != counter) continue;
                session.abort();
            }
        }
    }

    abstract SessionWorker<RecordingSession> newRecorder();

    abstract SessionWorker<ReplaySession> newReplayer();

    @Override
    protected final void preSessionsClose() {
        this.closeSessionWorkers();
    }

    protected abstract void closeSessionWorkers();

    @Override
    protected void postSessionsClose() {
        if (this.isAbort) {
            this.ctx.abortLatch().countDown();
        } else {
            this.aeron.removeCloseHandler(this.closeHandlerRegistrationId);
            this.aeron.removeUnavailableCounterHandler(this.unavailableCounterHandlerRegistrationId);
            if (!this.ctx.ownsAeronClient()) {
                for (Subscription subscription : this.recordingSubscriptionByKeyMap.values()) {
                    subscription.close();
                }
                CloseHelper.close((AutoCloseable)this.localControlSubscription);
                CloseHelper.close((AutoCloseable)this.controlSubscription);
                CloseHelper.close((AutoCloseable)this.recordingEventsProxy);
            }
        }
        this.markFile.updateActivityTimestamp(-1L);
        this.ctx.close();
    }

    @Override
    protected void abort() {
        try {
            this.isAbort = true;
            if (null != this.recorder) {
                this.recorder.abort();
            }
            if (null != this.replayer) {
                this.replayer.abort();
            }
            this.ctx.errorCounter().close();
            if (!this.ctx.abortLatch().await(15000L, TimeUnit.MILLISECONDS)) {
                this.errorHandler.onError((Throwable)new TimeoutException("awaiting abort latch", AeronException.Category.WARN));
            }
        }
        catch (InterruptedException ignore) {
            Thread.currentThread().interrupt();
        }
    }

    @Override
    public int doWork() {
        int workCount = 0;
        if (this.isAbort) {
            throw new AgentTerminationException("unexpected Aeron close");
        }
        long nowMs = this.epochClock.time();
        if (this.cachedEpochClock.time() != nowMs) {
            this.cachedEpochClock.update(nowMs);
            workCount += this.invokeAeronInvoker();
            if (nowMs >= this.markFileUpdateDeadlineMs) {
                this.markFileUpdateDeadlineMs = nowMs + MARK_FILE_UPDATE_INTERVAL_MS;
                this.markFile.updateActivityTimestamp(nowMs);
            }
        }
        workCount += this.invokeDriverConductor();
        return (workCount += this.runTasks(this.taskQueue)) + super.doWork();
    }

    Archive.Context context() {
        return this.ctx;
    }

    final int invokeAeronInvoker() {
        int workCount = 0;
        if (null != this.aeronAgentInvoker) {
            workCount += this.aeronAgentInvoker.invoke();
            if (this.isAbort || this.aeronAgentInvoker.isClosed()) {
                this.isAbort = true;
                throw new AgentTerminationException("unexpected Aeron close");
            }
        }
        return workCount;
    }

    final int invokeDriverConductor() {
        int workCount = 0;
        if (null != this.driverAgentInvoker) {
            workCount += this.driverAgentInvoker.invoke();
            if (this.driverAgentInvoker.isClosed()) {
                throw new AgentTerminationException("unexpected driver close");
            }
        }
        return workCount;
    }

    Catalog catalog() {
        return this.catalog;
    }

    ControlSession newControlSession(long correlationId, int streamId, int version, String channel, byte[] encodedCredentials, ControlSessionDemuxer demuxer) {
        ChannelUri channelUri = ChannelUri.parse((CharSequence)channel);
        String mtuStr = channelUri.get("mtu");
        int mtuLength = null == mtuStr ? this.ctx.controlMtuLength() : (int)SystemUtil.parseSize((String)"mtu", (String)mtuStr);
        String termLengthStr = channelUri.get("term-length");
        int termLength = null == termLengthStr ? this.ctx.controlTermBufferLength() : (int)SystemUtil.parseSize((String)"term-length", (String)termLengthStr);
        String isSparseStr = channelUri.get("sparse");
        boolean isSparse = null == isSparseStr ? this.ctx.controlTermBufferSparse() : Boolean.parseBoolean(isSparseStr);
        String responseChannel = ArchiveConductor.strippedChannelBuilder(channelUri).ttl(channelUri).termLength(Integer.valueOf(termLength)).sparse(Boolean.valueOf(isSparse)).mtu(Integer.valueOf(mtuLength)).build();
        String invalidVersionMessage = null;
        if (SemanticVersion.major((int)version) != 1) {
            invalidVersionMessage = "invalid client version " + SemanticVersion.toString((int)version) + ", archive is " + SemanticVersion.toString((int)AeronArchive.Configuration.PROTOCOL_SEMANTIC_VERSION);
        }
        ControlSession controlSession = new ControlSession(this.nextSessionId++, correlationId, this.connectTimeoutMs, this.aeron.asyncAddExclusivePublication(responseChannel, streamId), invalidVersionMessage, demuxer, this.aeron, this, this.cachedEpochClock, this.controlResponseProxy, this.authenticator, this.controlSessionProxy);
        this.authenticator.onConnectRequest(controlSession.sessionId(), encodedCredentials, this.cachedEpochClock.time());
        this.addSession(controlSession);
        this.ctx.controlSessionsCounter().incrementOrdered();
        return controlSession;
    }

    void startRecording(long correlationId, int streamId, SourceLocation sourceLocation, boolean autoStop, String originalChannel, ControlSession controlSession) {
        if (this.recordingSessionByIdMap.size() >= this.ctx.maxConcurrentRecordings()) {
            String msg = "max concurrent recordings reached " + this.ctx.maxConcurrentRecordings();
            controlSession.sendErrorResponse(correlationId, 8L, msg, this.controlResponseProxy);
            return;
        }
        if (this.isLowStorageSpace(correlationId, controlSession)) {
            return;
        }
        try {
            ChannelUri channelUri = ChannelUri.parse((CharSequence)originalChannel);
            String key = ArchiveConductor.makeKey(streamId, channelUri);
            Subscription oldSubscription = (Subscription)this.recordingSubscriptionByKeyMap.get((Object)key);
            if (null == oldSubscription) {
                String strippedChannel = ArchiveConductor.strippedChannelBuilder(channelUri).build();
                String channel = sourceLocation == SourceLocation.LOCAL && channelUri.isUdp() ? "aeron-spy:" + strippedChannel : strippedChannel;
                AvailableImageHandler handler = image -> this.taskQueue.addLast(() -> this.startRecordingSession(controlSession, correlationId, strippedChannel, originalChannel, image, autoStop));
                Subscription subscription = this.aeron.addSubscription(channel, streamId, handler, null);
                this.recordingSubscriptionByKeyMap.put((Object)key, (Object)subscription);
                this.subscriptionRefCountMap.incrementAndGet(subscription.registrationId());
                controlSession.sendOkResponse(correlationId, subscription.registrationId(), this.controlResponseProxy);
            } else {
                String msg = "recording exists for streamId=" + streamId + " channel=" + originalChannel;
                controlSession.sendErrorResponse(correlationId, 3L, msg, this.controlResponseProxy);
            }
        }
        catch (Exception ex) {
            this.errorHandler.onError((Throwable)ex);
            controlSession.sendErrorResponse(correlationId, ex.getMessage(), this.controlResponseProxy);
        }
    }

    void stopRecording(long correlationId, int streamId, String channel, ControlSession controlSession) {
        try {
            String key = ArchiveConductor.makeKey(streamId, ChannelUri.parse((CharSequence)channel));
            Subscription subscription = (Subscription)this.recordingSubscriptionByKeyMap.remove((Object)key);
            if (null != subscription) {
                for (RecordingSession session : this.recordingSessionByIdMap.values()) {
                    if (subscription != session.subscription()) continue;
                    session.abort();
                }
                if (0L == this.subscriptionRefCountMap.decrementAndGet(subscription.registrationId())) {
                    subscription.close();
                }
                controlSession.sendOkResponse(correlationId, this.controlResponseProxy);
            } else {
                String msg = "no recording found for streamId=" + streamId + " channel=" + channel;
                controlSession.sendErrorResponse(correlationId, 4L, msg, this.controlResponseProxy);
            }
        }
        catch (Exception ex) {
            this.errorHandler.onError((Throwable)ex);
            controlSession.sendErrorResponse(correlationId, ex.getMessage(), this.controlResponseProxy);
        }
    }

    void stopRecordingSubscription(long correlationId, long subscriptionId, ControlSession controlSession) {
        if (this.stopRecordingSubscription(subscriptionId)) {
            controlSession.sendOkResponse(correlationId, this.controlResponseProxy);
        } else {
            String msg = "no recording subscription found for subscriptionId=" + subscriptionId;
            controlSession.sendErrorResponse(correlationId, 4L, msg, this.controlResponseProxy);
        }
    }

    boolean stopRecordingSubscription(long subscriptionId) {
        Subscription subscription = this.removeRecordingSubscription(subscriptionId);
        if (null != subscription) {
            for (RecordingSession session : this.recordingSessionByIdMap.values()) {
                if (subscription != session.subscription()) continue;
                session.abort();
            }
            if (this.subscriptionRefCountMap.decrementAndGet(subscriptionId) <= 0L) {
                CloseHelper.close((ErrorHandler)this.errorHandler, (AutoCloseable)subscription);
            }
            return true;
        }
        return false;
    }

    void newListRecordingsSession(long correlationId, long fromId, int count, ControlSession controlSession) {
        if (controlSession.hasActiveListing()) {
            String msg = "active listing already in progress";
            controlSession.sendErrorResponse(correlationId, 1L, "active listing already in progress", this.controlResponseProxy);
            return;
        }
        ListRecordingsSession session = new ListRecordingsSession(correlationId, fromId, count, this.catalog, this.controlResponseProxy, controlSession, this.descriptorBuffer);
        this.addSession(session);
        controlSession.activeListing(session);
    }

    void newListRecordingsForUriSession(long correlationId, long fromRecordingId, int count, int streamId, byte[] channelFragment, ControlSession controlSession) {
        if (controlSession.hasActiveListing()) {
            String msg = "active listing already in progress";
            controlSession.sendErrorResponse(correlationId, 1L, "active listing already in progress", this.controlResponseProxy);
            return;
        }
        ListRecordingsForUriSession session = new ListRecordingsForUriSession(correlationId, fromRecordingId, count, channelFragment, streamId, this.catalog, this.controlResponseProxy, controlSession, this.descriptorBuffer, this.recordingDescriptorDecoder);
        this.addSession(session);
        controlSession.activeListing(session);
    }

    void listRecording(long correlationId, long recordingId, ControlSession controlSession) {
        if (controlSession.hasActiveListing()) {
            String msg = "active listing already in progress";
            controlSession.sendErrorResponse(correlationId, 1L, "active listing already in progress", this.controlResponseProxy);
        } else if (this.catalog.wrapDescriptor(recordingId, this.descriptorBuffer)) {
            controlSession.sendDescriptor(correlationId, this.descriptorBuffer, this.controlResponseProxy);
        } else {
            controlSession.sendRecordingUnknown(correlationId, recordingId, this.controlResponseProxy);
        }
    }

    void findLastMatchingRecording(long correlationId, long minRecordingId, int sessionId, int streamId, byte[] channelFragment, ControlSession controlSession) {
        if (minRecordingId < 0L || minRecordingId >= this.catalog.nextRecordingId()) {
            String msg = "min recording id outside valid range [0, " + Math.max(0L, this.catalog.nextRecordingId() - 1L) + "]: " + minRecordingId;
            controlSession.sendErrorResponse(correlationId, 5L, msg, this.controlResponseProxy);
        } else {
            long recordingId = this.catalog.findLast(minRecordingId, sessionId, streamId, channelFragment);
            if (-1L == recordingId) {
                String msg = "recording was not found: minRecordingId=" + minRecordingId + ", sessionId=" + sessionId + ", streamId=" + streamId;
                controlSession.sendErrorResponse(correlationId, 5L, msg, this.controlResponseProxy);
            } else {
                controlSession.sendOkResponse(correlationId, recordingId, this.controlResponseProxy);
            }
        }
    }

    void startReplay(long correlationId, long recordingId, long position, long length, int replayStreamId, String replayChannel, Counter limitPosition, ControlSession controlSession) {
        if (this.replaySessionByIdMap.size() >= this.ctx.maxConcurrentReplays()) {
            String msg = "max concurrent replays reached " + this.ctx.maxConcurrentReplays();
            controlSession.sendErrorResponse(correlationId, 7L, msg, this.controlResponseProxy);
            return;
        }
        if (!this.catalog.hasRecording(recordingId)) {
            String msg = "unknown recording id " + recordingId;
            controlSession.sendErrorResponse(correlationId, 5L, msg, this.controlResponseProxy);
            return;
        }
        this.catalog.recordingSummary(recordingId, this.recordingSummary);
        long replayPosition = this.recordingSummary.startPosition;
        if (-1L != position) {
            if (this.isInvalidReplayPosition(correlationId, controlSession, recordingId, position, this.recordingSummary)) {
                return;
            }
            replayPosition = position;
        }
        ExclusivePublication replayPublication = this.newReplayPublication(correlationId, controlSession, replayChannel, replayStreamId, replayPosition, this.recordingSummary);
        long replaySessionId = (long)this.replayId++ << 32 | (long)replayPublication.sessionId() & 0xFFFFFFFFL;
        Counter replayLimitPosition = limitPosition;
        if (null == replayLimitPosition) {
            RecordingSession recordingSession = (RecordingSession)this.recordingSessionByIdMap.get(recordingId);
            replayLimitPosition = null == recordingSession ? null : recordingSession.recordingPosition();
        }
        ReplaySession replaySession = new ReplaySession(replayPosition, length, replaySessionId, this.connectTimeoutMs, correlationId, controlSession, this.controlResponseProxy, this.ctx.replayBuffer(), this.catalog, this.archiveDir, this.cachedEpochClock, replayPublication, this.recordingSummary, replayLimitPosition, this.ctx.replayChecksum());
        this.replaySessionByIdMap.put(replaySessionId, (Object)replaySession);
        this.replayer.addSession(replaySession);
    }

    void startBoundedReplay(long correlationId, long recordingId, long position, long length, int limitCounterId, int replayStreamId, String replayChannel, ControlSession controlSession) {
        Counter replayLimitCounter = (Counter)this.counterByIdMap.get(limitCounterId);
        if (null == replayLimitCounter) {
            try {
                replayLimitCounter = new Counter(this.aeron.countersReader(), -1L, limitCounterId);
            }
            catch (Throwable ex) {
                String msg = "unable to create replay limit counter id= " + limitCounterId + " because of: " + ex.getMessage();
                controlSession.sendErrorResponse(correlationId, 0L, msg, this.controlResponseProxy);
                return;
            }
            this.counterByIdMap.put(limitCounterId, (Object)replayLimitCounter);
        }
        this.startReplay(correlationId, recordingId, position, length, replayStreamId, replayChannel, replayLimitCounter, controlSession);
    }

    void stopReplay(long correlationId, long replaySessionId, ControlSession controlSession) {
        ReplaySession replaySession = (ReplaySession)this.replaySessionByIdMap.get(replaySessionId);
        if (null == replaySession) {
            String errorMessage = "replay session not known for " + replaySessionId;
            controlSession.sendErrorResponse(correlationId, 6L, errorMessage, this.controlResponseProxy);
        } else {
            replaySession.abort();
            controlSession.sendOkResponse(correlationId, this.controlResponseProxy);
        }
    }

    void stopAllReplays(long correlationId, long recordingId, ControlSession controlSession) {
        for (ReplaySession replaySession : this.replaySessionByIdMap.values()) {
            if (-1L != recordingId && replaySession.recordingId() != recordingId) continue;
            replaySession.abort();
        }
        controlSession.sendOkResponse(correlationId, this.controlResponseProxy);
    }

    Subscription extendRecording(long correlationId, long recordingId, int streamId, SourceLocation sourceLocation, boolean autoStop, String originalChannel, ControlSession controlSession) {
        if (this.recordingSessionByIdMap.size() >= this.ctx.maxConcurrentRecordings()) {
            String msg = "max concurrent recordings reached at " + this.ctx.maxConcurrentRecordings();
            controlSession.sendErrorResponse(correlationId, 8L, msg, this.controlResponseProxy);
            return null;
        }
        if (!this.catalog.hasRecording(recordingId)) {
            String msg = "unknown recording " + recordingId;
            controlSession.sendErrorResponse(correlationId, 5L, msg, this.controlResponseProxy);
            return null;
        }
        this.catalog.recordingSummary(recordingId, this.recordingSummary);
        if (streamId != this.recordingSummary.streamId) {
            String msg = "cannot extend recording " + this.recordingSummary.recordingId + " with streamId=" + streamId + " for existing streamId=" + this.recordingSummary.streamId;
            controlSession.sendErrorResponse(correlationId, 5L, msg, this.controlResponseProxy);
            return null;
        }
        if (this.recordingSessionByIdMap.containsKey(recordingId)) {
            String msg = "cannot extend active recording " + recordingId;
            controlSession.sendErrorResponse(correlationId, 2L, msg, this.controlResponseProxy);
            return null;
        }
        if (this.isLowStorageSpace(correlationId, controlSession)) {
            return null;
        }
        try {
            ChannelUri channelUri = ChannelUri.parse((CharSequence)originalChannel);
            String key = ArchiveConductor.makeKey(streamId, channelUri);
            Subscription oldSubscription = (Subscription)this.recordingSubscriptionByKeyMap.get((Object)key);
            if (null == oldSubscription) {
                String strippedChannel = ArchiveConductor.strippedChannelBuilder(channelUri).build();
                String channel = originalChannel.contains("udp") && sourceLocation == SourceLocation.LOCAL ? "aeron-spy:" + strippedChannel : strippedChannel;
                AvailableImageHandler handler = image -> this.taskQueue.addLast(() -> this.extendRecordingSession(controlSession, correlationId, recordingId, strippedChannel, originalChannel, image, autoStop));
                Subscription subscription = this.aeron.addSubscription(channel, streamId, handler, null);
                this.recordingSubscriptionByKeyMap.put((Object)key, (Object)subscription);
                this.subscriptionRefCountMap.incrementAndGet(subscription.registrationId());
                controlSession.sendOkResponse(correlationId, subscription.registrationId(), this.controlResponseProxy);
                return subscription;
            }
            String msg = "recording exists for streamId=" + streamId + " channel=" + originalChannel;
            controlSession.sendErrorResponse(correlationId, 3L, msg, this.controlResponseProxy);
        }
        catch (Exception ex) {
            this.errorHandler.onError((Throwable)ex);
            controlSession.sendErrorResponse(correlationId, ex.getMessage(), this.controlResponseProxy);
        }
        return null;
    }

    void getStartPosition(long correlationId, long recordingId, ControlSession controlSession) {
        if (this.hasRecording(recordingId, correlationId, controlSession)) {
            controlSession.sendOkResponse(correlationId, this.catalog.startPosition(recordingId), this.controlResponseProxy);
        }
    }

    void getRecordingPosition(long correlationId, long recordingId, ControlSession controlSession) {
        if (this.hasRecording(recordingId, correlationId, controlSession)) {
            RecordingSession recordingSession = (RecordingSession)this.recordingSessionByIdMap.get(recordingId);
            long position = null == recordingSession ? -1L : recordingSession.recordingPosition().get();
            controlSession.sendOkResponse(correlationId, position, this.controlResponseProxy);
        }
    }

    void getStopPosition(long correlationId, long recordingId, ControlSession controlSession) {
        if (this.hasRecording(recordingId, correlationId, controlSession)) {
            controlSession.sendOkResponse(correlationId, this.catalog.stopPosition(recordingId), this.controlResponseProxy);
        }
    }

    void truncateRecording(long correlationId, long recordingId, long position, ControlSession controlSession) {
        if (this.hasRecording(recordingId, correlationId, controlSession) && this.isValidTruncate(correlationId, controlSession, recordingId, position)) {
            long stopPosition = this.recordingSummary.stopPosition;
            int segmentLength = this.recordingSummary.segmentFileLength;
            int termLength = this.recordingSummary.termBufferLength;
            long startPosition = this.recordingSummary.startPosition;
            long segmentBasePosition = AeronArchive.segmentFileBasePosition(startPosition, position, termLength, segmentLength);
            int segmentOffset = (int)(position - segmentBasePosition);
            ArrayDeque<String> files = new ArrayDeque<String>();
            if (segmentOffset > 0) {
                File file;
                if (stopPosition != position && !this.eraseRemainingSegment(correlationId, controlSession, position, segmentLength, segmentOffset, termLength, file = new File(this.archiveDir, Archive.segmentFileName(recordingId, segmentBasePosition)))) {
                    return;
                }
            } else {
                files.addLast(Archive.segmentFileName(recordingId, segmentBasePosition));
            }
            this.catalog.stopPosition(recordingId, position);
            for (long p = segmentBasePosition + (long)segmentLength; p <= stopPosition; p += (long)segmentLength) {
                files.addLast(Archive.segmentFileName(recordingId, p));
            }
            controlSession.sendOkResponse(correlationId, this.controlResponseProxy);
            if (!files.isEmpty()) {
                this.addSession(new DeleteSegmentsSession(recordingId, correlationId, files, this.archiveDir, controlSession, this.controlResponseProxy, (ErrorHandler)this.errorHandler));
            } else {
                controlSession.attemptSignal(correlationId, recordingId, -1L, -1L, RecordingSignal.DELETE);
            }
        }
    }

    void purgeRecording(long correlationId, long recordingId, ControlSession controlSession) {
        if (this.hasRecording(recordingId, correlationId, controlSession) && this.isValidPurge(correlationId, controlSession, recordingId)) {
            String[] segmentFiles;
            ArrayDeque<String> files = new ArrayDeque<String>();
            if (this.catalog.invalidateRecording(recordingId) && null != (segmentFiles = Catalog.listSegmentFiles(this.archiveDir, recordingId))) {
                for (String segmentFile : segmentFiles) {
                    files.addLast(segmentFile);
                }
            }
            controlSession.sendOkResponse(correlationId, this.controlResponseProxy);
            if (!files.isEmpty()) {
                this.addSession(new DeleteSegmentsSession(recordingId, correlationId, files, this.archiveDir, controlSession, this.controlResponseProxy, (ErrorHandler)this.errorHandler));
            }
        }
    }

    void listRecordingSubscriptions(long correlationId, int pseudoIndex, int subscriptionCount, boolean applyStreamId, int streamId, String channelFragment, ControlSession controlSession) {
        if (controlSession.hasActiveListing()) {
            String msg = "active listing already in progress";
            controlSession.sendErrorResponse(correlationId, 1L, "active listing already in progress", this.controlResponseProxy);
        } else if (pseudoIndex < 0 || pseudoIndex >= this.recordingSubscriptionByKeyMap.size() || subscriptionCount <= 0) {
            controlSession.sendSubscriptionUnknown(correlationId, this.controlResponseProxy);
        } else {
            ListRecordingSubscriptionsSession session = new ListRecordingSubscriptionsSession(this.recordingSubscriptionByKeyMap, pseudoIndex, subscriptionCount, streamId, applyStreamId, channelFragment, correlationId, controlSession, this.controlResponseProxy);
            this.addSession(session);
            controlSession.activeListing(session);
        }
    }

    void stopRecordingByIdentity(long correlationId, long recordingId, ControlSession controlSession) {
        if (this.hasRecording(recordingId, correlationId, controlSession)) {
            int found = 0;
            RecordingSession recordingSession = (RecordingSession)this.recordingSessionByIdMap.get(recordingId);
            if (null != recordingSession) {
                recordingSession.abort();
                long subscriptionId = recordingSession.subscription().registrationId();
                Subscription subscription = this.removeRecordingSubscription(subscriptionId);
                if (null != subscription) {
                    found = 1;
                    for (RecordingSession session : this.recordingSessionByIdMap.values()) {
                        if (subscription != session.subscription()) continue;
                        session.abort();
                    }
                    this.subscriptionRefCountMap.decrementAndGet(subscriptionId);
                }
            }
            controlSession.sendOkResponse(correlationId, found, this.controlResponseProxy);
        }
    }

    void closeRecordingSession(RecordingSession session) {
        if (this.isAbort) {
            session.abortClose();
        } else {
            Subscription subscription = session.subscription();
            long position = session.recordedPosition();
            long recordingId = session.sessionId();
            long subscriptionId = subscription.registrationId();
            try {
                this.recordingSessionByIdMap.remove(recordingId);
                this.catalog.recordingStopped(recordingId, position, this.epochClock.time());
                session.sendPendingError(this.controlResponseProxy);
                session.controlSession().attemptSignal(session.correlationId(), recordingId, subscriptionId, position, RecordingSignal.STOP);
            }
            catch (Throwable ex) {
                this.errorHandler.onError(ex);
            }
            if (this.subscriptionRefCountMap.decrementAndGet(subscriptionId) <= 0L || session.isAutoStop()) {
                this.closeAndRemoveRecordingSubscription(subscription);
            }
            this.closeSession(session);
        }
    }

    void closeReplaySession(ReplaySession session) {
        if (!this.isAbort) {
            try {
                session.sendPendingError(this.controlResponseProxy);
            }
            catch (Throwable ex) {
                this.errorHandler.onError(ex);
            }
        }
        this.replaySessionByIdMap.remove(session.sessionId());
        this.closeSession(session);
    }

    void replicate(long correlationId, long srcRecordingId, long dstRecordingId, long stopPosition, long channelTagId, long subscriptionTagId, int srcControlStreamId, String srcControlChannel, String liveDestination, String replicationChannel, ControlSession controlSession) {
        boolean hasRecording = this.catalog.hasRecording(dstRecordingId);
        if (-1L != dstRecordingId && !hasRecording) {
            String msg = "unknown destination recording id " + dstRecordingId;
            controlSession.sendErrorResponse(correlationId, 5L, msg, this.controlResponseProxy);
            return;
        }
        if (hasRecording) {
            this.catalog.recordingSummary(dstRecordingId, this.recordingSummary);
            if (-1L == this.recordingSummary.stopPosition || this.recordingSessionByIdMap.containsKey(dstRecordingId)) {
                String msg = "cannot replicate to active recording " + dstRecordingId;
                controlSession.sendErrorResponse(correlationId, 2L, msg, this.controlResponseProxy);
                return;
            }
        }
        AeronArchive.Context remoteArchiveContext = this.ctx.archiveClientContext().clone().controlRequestChannel(srcControlChannel).controlRequestStreamId(srcControlStreamId);
        long replicationId = this.nextSessionId++;
        ReplicationSession replicationSession = new ReplicationSession(srcRecordingId, dstRecordingId, channelTagId, subscriptionTagId, replicationId, stopPosition, liveDestination, Strings.isEmpty((String)replicationChannel) ? this.ctx.replicationChannel() : replicationChannel, hasRecording ? this.recordingSummary : null, remoteArchiveContext, this.cachedEpochClock, this.catalog, this.controlResponseProxy, controlSession);
        this.replicationSessionByIdMap.put(replicationId, (Object)replicationSession);
        this.addSession(replicationSession);
        controlSession.sendOkResponse(correlationId, replicationId, this.controlResponseProxy);
    }

    void stopReplication(long correlationId, long replicationId, ControlSession controlSession) {
        ReplicationSession session = (ReplicationSession)this.replicationSessionByIdMap.remove(replicationId);
        if (null == session) {
            String msg = "unknown replication id " + replicationId;
            controlSession.sendErrorResponse(correlationId, 12L, msg, this.controlResponseProxy);
        } else {
            session.abort();
            controlSession.sendOkResponse(correlationId, this.controlResponseProxy);
        }
    }

    void detachSegments(long correlationId, long recordingId, long newStartPosition, ControlSession controlSession) {
        if (this.hasRecording(recordingId, correlationId, controlSession) && this.isValidDetach(correlationId, controlSession, recordingId, newStartPosition)) {
            this.catalog.startPosition(recordingId, newStartPosition);
            controlSession.sendOkResponse(correlationId, this.controlResponseProxy);
        }
    }

    void deleteDetachedSegments(long correlationId, long recordingId, ControlSession controlSession) {
        if (this.hasRecording(recordingId, correlationId, controlSession)) {
            ArrayDeque<String> files = new ArrayDeque<String>();
            this.findDetachedSegments(recordingId, files);
            int count = files.size();
            if (count > 0) {
                this.addSession(new DeleteSegmentsSession(recordingId, correlationId, files, this.archiveDir, controlSession, this.controlResponseProxy, (ErrorHandler)this.errorHandler));
            }
            controlSession.sendOkResponse(correlationId, count, this.controlResponseProxy);
        }
    }

    void purgeSegments(long correlationId, long recordingId, long newStartPosition, ControlSession controlSession) {
        if (this.hasRecording(recordingId, correlationId, controlSession) && this.isValidDetach(correlationId, controlSession, recordingId, newStartPosition)) {
            this.catalog.startPosition(recordingId, newStartPosition);
            ArrayDeque<String> files = new ArrayDeque<String>();
            this.findDetachedSegments(recordingId, files);
            int count = files.size();
            controlSession.sendOkResponse(correlationId, count, this.controlResponseProxy);
            if (count > 0) {
                this.addSession(new DeleteSegmentsSession(recordingId, correlationId, files, this.archiveDir, controlSession, this.controlResponseProxy, (ErrorHandler)this.errorHandler));
            }
        }
    }

    void attachSegments(long correlationId, long recordingId, ControlSession controlSession) {
        if (this.hasRecording(recordingId, correlationId, controlSession)) {
            File file;
            this.catalog.recordingSummary(recordingId, this.recordingSummary);
            int segmentLength = this.recordingSummary.segmentFileLength;
            int termLength = this.recordingSummary.termBufferLength;
            int bitsToShift = LogBufferDescriptor.positionBitsToShift((int)termLength);
            int streamId = this.recordingSummary.streamId;
            long position = this.recordingSummary.startPosition - (long)segmentLength;
            long count = 0L;
            while (position >= 0L && (file = new File(this.archiveDir, Archive.segmentFileName(recordingId, position))).exists()) {
                long fileLength = file.length();
                if (fileLength != (long)segmentLength) {
                    String msg = "fileLength=" + fileLength + " not equal to segmentLength=" + segmentLength;
                    controlSession.sendErrorResponse(correlationId, msg, this.controlResponseProxy);
                    return;
                }
                try {
                    FileChannel fileChannel = FileChannel.open(file.toPath(), FILE_OPTIONS, NO_ATTRIBUTES);
                    Throwable throwable = null;
                    try {
                        int termCount = (int)(position >> bitsToShift);
                        int termId = this.recordingSummary.initialTermId + termCount;
                        int termOffset = this.findTermOffsetForStart(correlationId, controlSession, file, fileChannel, streamId, termId, termLength);
                        if (termOffset < 0) {
                            return;
                        }
                        if (0 == termOffset) {
                            this.catalog.startPosition(recordingId, position);
                            ++count;
                            position -= (long)segmentLength;
                            continue;
                        }
                        this.catalog.startPosition(recordingId, position + (long)termOffset);
                        ++count;
                        break;
                    }
                    catch (Throwable throwable2) {
                        throwable = throwable2;
                        throw throwable2;
                    }
                    finally {
                        if (fileChannel == null) continue;
                        if (throwable != null) {
                            try {
                                fileChannel.close();
                            }
                            catch (Throwable throwable3) {
                                throwable.addSuppressed(throwable3);
                            }
                            continue;
                        }
                        fileChannel.close();
                    }
                }
                catch (IOException ex) {
                    controlSession.sendErrorResponse(correlationId, ex.getMessage(), this.controlResponseProxy);
                    LangUtil.rethrowUnchecked((Throwable)ex);
                }
            }
            controlSession.sendOkResponse(correlationId, count, this.controlResponseProxy);
        }
    }

    void migrateSegments(long correlationId, long srcRecordingId, long dstRecordingId, ControlSession controlSession) {
        if (this.hasRecording(srcRecordingId, correlationId, controlSession) && this.hasRecording(dstRecordingId, correlationId, controlSession) && this.isValidAttach(correlationId, controlSession, srcRecordingId, dstRecordingId)) {
            long attachedSegmentCount = 0L;
            long position = this.recordingSummary.stopPosition;
            long startPosition = this.recordingSummary.startPosition;
            int segmentLength = this.recordingSummary.segmentFileLength;
            long segmentFileBasePosition = AeronArchive.segmentFileBasePosition(startPosition, startPosition, this.recordingSummary.termBufferLength, segmentLength);
            ArrayDeque<String> files = new ArrayDeque<String>();
            while (position >= segmentFileBasePosition) {
                String segmentFileName = Archive.segmentFileName(srcRecordingId, position);
                File srcFile = new File(this.archiveDir, segmentFileName);
                if (position == this.recordingSummary.stopPosition) {
                    files.addFirst(segmentFileName);
                    position -= (long)segmentLength;
                    continue;
                }
                if (!srcFile.exists()) break;
                String dstFile = Archive.segmentFileName(dstRecordingId, position);
                if (!srcFile.renameTo(new File(this.archiveDir, dstFile))) {
                    String msg = "failed to rename " + srcFile + " to " + dstFile;
                    controlSession.sendErrorResponse(correlationId, msg, this.controlResponseProxy);
                    return;
                }
                ++attachedSegmentCount;
                position -= (long)segmentLength;
            }
            this.catalog.startPosition(dstRecordingId, startPosition);
            this.catalog.stopPosition(srcRecordingId, startPosition);
            controlSession.sendOkResponse(correlationId, attachedSegmentCount, this.controlResponseProxy);
            if (!files.isEmpty()) {
                this.addSession(new DeleteSegmentsSession(srcRecordingId, correlationId, files, this.archiveDir, controlSession, this.controlResponseProxy, (ErrorHandler)this.errorHandler));
            }
        }
    }

    void removeReplicationSession(ReplicationSession replicationSession) {
        this.replicationSessionByIdMap.remove(replicationSession.sessionId());
    }

    private void findDetachedSegments(long recordingId, ArrayDeque<String> files) {
        String segmentFileName;
        File file;
        this.catalog.recordingSummary(recordingId, this.recordingSummary);
        int segmentFile = this.recordingSummary.segmentFileLength;
        for (long filenamePosition = this.recordingSummary.startPosition - (long)segmentFile; filenamePosition >= 0L && (file = new File(this.archiveDir, segmentFileName = Archive.segmentFileName(recordingId, filenamePosition))).exists(); filenamePosition -= (long)segmentFile) {
            files.addFirst(segmentFileName);
        }
    }

    private int findTermOffsetForStart(long correlationId, ControlSession controlSession, File file, FileChannel fileChannel, int streamId, int termId, int termLength) throws IOException {
        int termOffset = 0;
        UnsafeBuffer buffer = this.ctx.dataBuffer();
        ByteBuffer byteBuffer = buffer.byteBuffer();
        byteBuffer.clear().limit(32);
        if (32 != fileChannel.read(byteBuffer, 0L)) {
            String msg = "failed to read segment file";
            controlSession.sendErrorResponse(correlationId, "failed to read segment file", this.controlResponseProxy);
            return termOffset;
        }
        int fragmentLength = DataHeaderFlyweight.fragmentLength((UnsafeBuffer)buffer, (int)termOffset);
        if (fragmentLength <= 0) {
            int offset;
            boolean found = false;
            block0: do {
                byteBuffer.clear().limit(Math.min(termLength - termOffset, byteBuffer.capacity()));
                int bytesRead = fileChannel.read(byteBuffer, termOffset);
                if (bytesRead <= 0) {
                    String msg = "read failed on " + file;
                    controlSession.sendErrorResponse(correlationId, msg, this.controlResponseProxy);
                    return -1;
                }
                int limit = bytesRead - (bytesRead & 0x1F);
                for (offset = 0; offset < limit; offset += 32) {
                    if (DataHeaderFlyweight.fragmentLength((UnsafeBuffer)buffer, (int)offset) <= 0) continue;
                    found = true;
                    continue block0;
                }
            } while ((termOffset += offset) < termLength && !found);
        }
        if (termOffset >= termLength) {
            String msg = "fragment not found in first term of segment " + file;
            controlSession.sendErrorResponse(correlationId, msg, this.controlResponseProxy);
            return -1;
        }
        int fileTermId = DataHeaderFlyweight.termId((UnsafeBuffer)buffer, (int)termOffset);
        if (fileTermId != termId) {
            String msg = "term id does not match: actual=" + fileTermId + " expected=" + termId;
            controlSession.sendErrorResponse(correlationId, msg, this.controlResponseProxy);
            return -1;
        }
        int fileStreamId = DataHeaderFlyweight.streamId((UnsafeBuffer)buffer, (int)termOffset);
        if (fileStreamId != streamId) {
            String msg = "stream id does not match: actual=" + fileStreamId + " expected=" + streamId;
            controlSession.sendErrorResponse(correlationId, msg, this.controlResponseProxy);
            return -1;
        }
        return termOffset;
    }

    private int runTasks(ArrayDeque<Runnable> taskQueue) {
        Runnable runnable;
        int workCount = 0;
        while (null != (runnable = taskQueue.pollFirst())) {
            runnable.run();
            ++workCount;
        }
        return workCount;
    }

    private static ChannelUriStringBuilder strippedChannelBuilder(ChannelUri channelUri) {
        return new ChannelUriStringBuilder().media(channelUri.media()).endpoint(channelUri).networkInterface(channelUri).controlEndpoint(channelUri).controlMode(channelUri).tags(channelUri).rejoin(channelUri).group(channelUri).tether(channelUri).flowControl(channelUri).groupTag(channelUri).congestionControl(channelUri).socketRcvbufLength(channelUri).socketSndbufLength(channelUri).receiverWindowLength(channelUri).sessionId(channelUri).alias(channelUri);
    }

    private static String makeKey(int streamId, ChannelUri channelUri) {
        String tagsStr;
        String sessionIdStr;
        String controlStr;
        String interfaceStr;
        StringBuilder sb = new StringBuilder();
        sb.append(streamId).append(':').append(channelUri.media()).append('?');
        String endpointStr = channelUri.get("endpoint");
        if (null != endpointStr) {
            sb.append("endpoint").append('=').append(endpointStr).append('|');
        }
        if (null != (interfaceStr = channelUri.get("interface"))) {
            sb.append("interface").append('=').append(interfaceStr).append('|');
        }
        if (null != (controlStr = channelUri.get("control"))) {
            sb.append("control").append('=').append(controlStr).append('|');
        }
        if (null != (sessionIdStr = channelUri.get("session-id"))) {
            sb.append("session-id").append('=').append(sessionIdStr).append('|');
        }
        if (null != (tagsStr = channelUri.get("tags"))) {
            sb.append("tags").append('=').append(tagsStr).append('|');
        }
        sb.setLength(sb.length() - 1);
        return sb.toString();
    }

    private boolean hasRecording(long recordingId, long correlationId, ControlSession session) {
        if (!this.catalog.hasRecording(recordingId)) {
            String msg = "unknown recording " + recordingId;
            session.sendErrorResponse(correlationId, 5L, msg, this.controlResponseProxy);
            return false;
        }
        return true;
    }

    private void startRecordingSession(ControlSession controlSession, long correlationId, String strippedChannel, String originalChannel, Image image, boolean autoStop) {
        int sessionId = image.sessionId();
        int streamId = image.subscription().streamId();
        String sourceIdentity = image.sourceIdentity();
        int termBufferLength = image.termBufferLength();
        int mtuLength = image.mtuLength();
        int initialTermId = image.initialTermId();
        long startPosition = image.joinPosition();
        int segmentFileLength = Math.max(this.ctx.segmentFileLength(), termBufferLength);
        long recordingId = this.catalog.addNewRecording(startPosition, this.cachedEpochClock.time(), initialTermId, segmentFileLength, termBufferLength, mtuLength, sessionId, streamId, strippedChannel, originalChannel, sourceIdentity);
        Counter position = RecordingPos.allocate(this.aeron, this.counterMetadataBuffer, recordingId, sessionId, streamId, strippedChannel, sourceIdentity);
        position.setOrdered(startPosition);
        RecordingSession session = new RecordingSession(correlationId, recordingId, startPosition, segmentFileLength, originalChannel, this.recordingEventsProxy, image, position, this.ctx, controlSession, autoStop);
        this.subscriptionRefCountMap.incrementAndGet(image.subscription().registrationId());
        this.recordingSessionByIdMap.put(recordingId, (Object)session);
        this.recorder.addSession(session);
        controlSession.attemptSignal(correlationId, recordingId, image.subscription().registrationId(), image.joinPosition(), RecordingSignal.START);
    }

    private void extendRecordingSession(ControlSession controlSession, long correlationId, long recordingId, String strippedChannel, String originalChannel, Image image, boolean autoStop) {
        block3: {
            long subscriptionId = image.subscription().registrationId();
            try {
                if (this.recordingSessionByIdMap.containsKey(recordingId)) {
                    String msg = "cannot extend active recording " + recordingId + " streamId=" + image.subscription().streamId() + " channel=" + originalChannel;
                    controlSession.attemptErrorResponse(correlationId, 2, msg, this.controlResponseProxy);
                    throw new ArchiveEvent(msg);
                }
                this.catalog.recordingSummary(recordingId, this.recordingSummary);
                this.validateImageForExtendRecording(correlationId, controlSession, image, this.recordingSummary);
                Counter position = RecordingPos.allocate(this.aeron, this.counterMetadataBuffer, recordingId, image.sessionId(), image.subscription().streamId(), strippedChannel, image.sourceIdentity());
                position.setOrdered(image.joinPosition());
                RecordingSession session = new RecordingSession(correlationId, recordingId, this.recordingSummary.startPosition, this.recordingSummary.segmentFileLength, originalChannel, this.recordingEventsProxy, image, position, this.ctx, controlSession, autoStop);
                this.subscriptionRefCountMap.incrementAndGet(subscriptionId);
                this.recordingSessionByIdMap.put(recordingId, (Object)session);
                this.catalog.extendRecording(recordingId, controlSession.sessionId(), correlationId, image.sessionId());
                this.recorder.addSession(session);
                controlSession.attemptSignal(correlationId, recordingId, subscriptionId, image.joinPosition(), RecordingSignal.EXTEND);
            }
            catch (Exception ex) {
                this.errorHandler.onError((Throwable)ex);
                if (!autoStop) break block3;
                this.closeAndRemoveRecordingSubscription(image.subscription());
            }
        }
    }

    private Subscription removeRecordingSubscription(long subscriptionId) {
        Object2ObjectHashMap.ValueIterator iter = this.recordingSubscriptionByKeyMap.values().iterator();
        while (iter.hasNext()) {
            Subscription subscription = (Subscription)iter.next();
            if (subscription.registrationId() != subscriptionId) continue;
            iter.remove();
            return subscription;
        }
        return null;
    }

    private ExclusivePublication newReplayPublication(long correlationId, ControlSession controlSession, String replayChannel, int replayStreamId, long position, RecordingSummary recording) {
        ChannelUri channelUri = ChannelUri.parse((CharSequence)replayChannel);
        ChannelUriStringBuilder channelBuilder = ArchiveConductor.strippedChannelBuilder(channelUri).initialPosition(position, recording.initialTermId, recording.termBufferLength).ttl(channelUri).eos(channelUri).sparse(channelUri).mtu(Integer.valueOf(recording.mtuLength));
        String lingerValue = channelUri.get("linger");
        channelBuilder.linger(Long.valueOf(null != lingerValue ? Long.parseLong(lingerValue) : this.ctx.replayLingerTimeoutNs()));
        try {
            return this.aeron.addExclusivePublication(channelBuilder.build(), replayStreamId);
        }
        catch (Exception ex) {
            String msg = "failed to create replay publication - " + ex;
            controlSession.sendErrorResponse(correlationId, msg, this.controlResponseProxy);
            throw ex;
        }
    }

    private void validateImageForExtendRecording(long correlationId, ControlSession controlSession, Image image, RecordingSummary recordingSummary) {
        if (image.joinPosition() != recordingSummary.stopPosition) {
            String msg = "cannot extend recording " + recordingSummary.recordingId + " image joinPosition=" + image.joinPosition() + " != stopPosition=" + recordingSummary.stopPosition;
            controlSession.attemptErrorResponse(correlationId, 9, msg, this.controlResponseProxy);
            throw new ArchiveEvent(msg);
        }
        if (image.initialTermId() != recordingSummary.initialTermId) {
            String msg = "cannot extend recording " + recordingSummary.recordingId + " image initialTermId=" + image.initialTermId() + " != initialTermId=" + recordingSummary.initialTermId;
            controlSession.attemptErrorResponse(correlationId, 9, msg, this.controlResponseProxy);
            throw new ArchiveEvent(msg);
        }
        if (image.termBufferLength() != recordingSummary.termBufferLength) {
            String msg = "cannot extend recording " + recordingSummary.recordingId + " image termBufferLength=" + image.termBufferLength() + " != termBufferLength=" + recordingSummary.termBufferLength;
            controlSession.attemptErrorResponse(correlationId, 9, msg, this.controlResponseProxy);
            throw new ArchiveEvent(msg);
        }
        if (image.mtuLength() != recordingSummary.mtuLength) {
            String msg = "cannot extend recording " + recordingSummary.recordingId + " image mtuLength=" + image.mtuLength() + " != mtuLength=" + recordingSummary.mtuLength;
            controlSession.attemptErrorResponse(correlationId, 9, msg, this.controlResponseProxy);
            throw new ArchiveEvent(msg);
        }
    }

    private boolean isValidTruncate(long correlationId, ControlSession controlSession, long recordingId, long position) {
        for (ReplaySession replaySession : this.replaySessionByIdMap.values()) {
            if (replaySession.recordingId() != recordingId) continue;
            String msg = "cannot truncate recording with active replay " + recordingId;
            controlSession.sendErrorResponse(correlationId, 2L, msg, this.controlResponseProxy);
            return false;
        }
        this.catalog.recordingSummary(recordingId, this.recordingSummary);
        long stopPosition = this.recordingSummary.stopPosition;
        long startPosition = this.recordingSummary.startPosition;
        if (-1L == stopPosition) {
            String msg = "cannot truncate active recording";
            controlSession.sendErrorResponse(correlationId, 2L, "cannot truncate active recording", this.controlResponseProxy);
            return false;
        }
        if (position < startPosition || position > stopPosition || (position & 0x1FL) != 0L) {
            String msg = "invalid position " + position + ": start=" + startPosition + " stop=" + stopPosition + " alignment=" + 32;
            controlSession.sendErrorResponse(correlationId, msg, this.controlResponseProxy);
            return false;
        }
        return true;
    }

    private boolean isValidPurge(long correlationId, ControlSession controlSession, long recordingId) {
        for (ReplaySession replaySession : this.replaySessionByIdMap.values()) {
            if (replaySession.recordingId() != recordingId) continue;
            String msg = "cannot purge recording with active replay " + recordingId;
            controlSession.sendErrorResponse(correlationId, 2L, msg, this.controlResponseProxy);
            return false;
        }
        this.catalog.recordingSummary(recordingId, this.recordingSummary);
        long stopPosition = this.recordingSummary.stopPosition;
        if (-1L == stopPosition) {
            String msg = "cannot purge active recording " + recordingId;
            controlSession.sendErrorResponse(correlationId, 2L, msg, this.controlResponseProxy);
            return false;
        }
        return true;
    }

    private boolean isInvalidReplayPosition(long correlationId, ControlSession controlSession, long recordingId, long position, RecordingSummary recordingSummary) {
        if ((position & 0x1FL) != 0L) {
            String msg = "requested replay start position=" + position + " is not a multiple of FRAME_ALIGNMENT (" + 32 + ") for recording " + recordingId;
            controlSession.sendErrorResponse(correlationId, msg, this.controlResponseProxy);
            return true;
        }
        long startPosition = recordingSummary.startPosition;
        if (position - startPosition < 0L) {
            String msg = "requested replay start position=" + position + " is less than recording start position=" + startPosition + " for recording " + recordingId;
            controlSession.sendErrorResponse(correlationId, msg, this.controlResponseProxy);
            return true;
        }
        long stopPosition = recordingSummary.stopPosition;
        if (stopPosition != -1L && position >= stopPosition) {
            String msg = "requested replay start position=" + position + " must be less than highest recorded position=" + stopPosition + " for recording " + recordingId;
            controlSession.sendErrorResponse(correlationId, msg, this.controlResponseProxy);
            return true;
        }
        return false;
    }

    private boolean isValidDetach(long correlationId, ControlSession controlSession, long recordingId, long position) {
        this.catalog.recordingSummary(recordingId, this.recordingSummary);
        int segmentLength = this.recordingSummary.segmentFileLength;
        long startPosition = this.recordingSummary.startPosition;
        int termLength = this.recordingSummary.termBufferLength;
        long lowerBound = AeronArchive.segmentFileBasePosition(startPosition, startPosition, termLength, segmentLength) + (long)segmentLength;
        if (position != AeronArchive.segmentFileBasePosition(startPosition, position, termLength, segmentLength)) {
            String msg = "invalid segment start: newStartPosition=" + position;
            controlSession.sendErrorResponse(correlationId, msg, this.controlResponseProxy);
            return false;
        }
        if (position < lowerBound) {
            String msg = "invalid detach: newStartPosition=" + position + " lowerBound=" + lowerBound;
            controlSession.sendErrorResponse(correlationId, msg, this.controlResponseProxy);
            return false;
        }
        long stopPosition = this.recordingSummary.stopPosition;
        long endPosition = -1L == stopPosition ? ((RecordingSession)this.recordingSessionByIdMap.get(recordingId)).recordedPosition() : stopPosition;
        if (position > (endPosition = AeronArchive.segmentFileBasePosition(startPosition, endPosition, termLength, segmentLength))) {
            String msg = "invalid detach: in use, newStartPosition=" + position + " upperBound=" + endPosition;
            controlSession.sendErrorResponse(correlationId, msg, this.controlResponseProxy);
            return false;
        }
        long replayBasePosition = Long.MAX_VALUE;
        for (ReplaySession replaySession : this.replaySessionByIdMap.values()) {
            if (replaySession.recordingId() != recordingId) continue;
            replayBasePosition = Math.min(replayBasePosition, replaySession.segmentFileBasePosition());
        }
        if (position > replayBasePosition) {
            String msg = "invalid detach: replay in progress, newStartPosition=" + position + " upperBound=" + replayBasePosition;
            controlSession.sendErrorResponse(correlationId, msg, this.controlResponseProxy);
            return false;
        }
        return true;
    }

    private boolean isValidAttach(long correlationId, ControlSession controlSession, long srcRecordingId, long dstRecordingId) {
        this.catalog.recordingSummary(dstRecordingId, this.recordingSummary);
        long dstStartPosition = this.recordingSummary.startPosition;
        int dstSegmentFileLength = this.recordingSummary.segmentFileLength;
        int dstTermBufferLength = this.recordingSummary.termBufferLength;
        int dstInitialTermId = this.recordingSummary.initialTermId;
        int dstStreamId = this.recordingSummary.streamId;
        int dstMtuLength = this.recordingSummary.mtuLength;
        this.catalog.recordingSummary(srcRecordingId, this.recordingSummary);
        long srcStopPosition = this.recordingSummary.stopPosition;
        if (-1L == srcStopPosition) {
            String msg = "source recording " + srcRecordingId + " still active";
            controlSession.sendErrorResponse(correlationId, msg, this.controlResponseProxy);
            return false;
        }
        if (dstStartPosition != srcStopPosition) {
            String msg = "invalid migrate: srcStopPosition=" + srcStopPosition + " dstStartPosition=" + dstStartPosition;
            controlSession.sendErrorResponse(correlationId, msg, this.controlResponseProxy);
            return false;
        }
        int srcSegmentFileLength = this.recordingSummary.segmentFileLength;
        if (dstSegmentFileLength != srcSegmentFileLength) {
            String msg = "invalid migrate: srcSegmentFileLength=" + srcSegmentFileLength + " dstSegmentFileLength=" + dstSegmentFileLength;
            controlSession.sendErrorResponse(correlationId, msg, this.controlResponseProxy);
            return false;
        }
        int srcTermBufferLength = this.recordingSummary.termBufferLength;
        if (dstTermBufferLength != srcTermBufferLength) {
            String msg = "invalid migrate: srcTermBufferLength=" + srcTermBufferLength + " dstTermBufferLength=" + dstTermBufferLength;
            controlSession.sendErrorResponse(correlationId, msg, this.controlResponseProxy);
            return false;
        }
        int srcInitialTermId = this.recordingSummary.initialTermId;
        if (dstInitialTermId != srcInitialTermId) {
            String msg = "invalid migrate: srcInitialTermId=" + srcInitialTermId + " dstInitialTermId=" + dstInitialTermId;
            controlSession.sendErrorResponse(correlationId, msg, this.controlResponseProxy);
            return false;
        }
        int srcStreamId = this.recordingSummary.streamId;
        if (dstStreamId != srcStreamId) {
            String msg = "invalid migrate: srcStreamId=" + srcStreamId + " dstStreamId=" + dstStreamId;
            controlSession.sendErrorResponse(correlationId, msg, this.controlResponseProxy);
            return false;
        }
        int srcMtuLength = this.recordingSummary.mtuLength;
        if (dstMtuLength != srcMtuLength) {
            String msg = "invalid migrate: srcMtuLength=" + srcMtuLength + " dstMtuLength=" + dstMtuLength;
            controlSession.sendErrorResponse(correlationId, msg, this.controlResponseProxy);
            return false;
        }
        return true;
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private boolean eraseRemainingSegment(long correlationId, ControlSession controlSession, long position, int segmentLength, int segmentOffset, int termLength, File file) {
        try (FileChannel channel = FileChannel.open(file.toPath(), FILE_OPTIONS, NO_ATTRIBUTES);){
            int termOffset = (int)(position & (long)(termLength - 1));
            int termCount = (int)(position >> LogBufferDescriptor.positionBitsToShift((int)termLength));
            int termId = this.recordingSummary.initialTermId + termCount;
            UnsafeBuffer dataBuffer = this.ctx.dataBuffer();
            if (ReplaySession.notHeaderAligned(channel, dataBuffer, segmentOffset, termOffset, termId, this.recordingSummary.streamId)) {
                String msg = position + " position not aligned to a data header";
                controlSession.sendErrorResponse(correlationId, msg, this.controlResponseProxy);
                boolean bl = false;
                return bl;
            }
            channel.truncate(segmentOffset);
            dataBuffer.byteBuffer().put(0, (byte)0).limit(1).position(0);
            channel.write(dataBuffer.byteBuffer(), segmentLength - 1);
            return true;
        }
        catch (IOException ex) {
            controlSession.sendErrorResponse(correlationId, ex.getMessage(), this.controlResponseProxy);
            LangUtil.rethrowUnchecked((Throwable)ex);
        }
        return true;
    }

    private void closeAndRemoveRecordingSubscription(Subscription subscription) {
        long subscriptionId = subscription.registrationId();
        this.subscriptionRefCountMap.remove(subscriptionId);
        for (RecordingSession session : this.recordingSessionByIdMap.values()) {
            if (subscription != session.subscription()) continue;
            session.abort();
        }
        this.removeRecordingSubscription(subscriptionId);
        CloseHelper.close((ErrorHandler)this.errorHandler, (AutoCloseable)subscription);
    }

    private boolean isLowStorageSpace(long correlationId, ControlSession controlSession) {
        try {
            long usableSpace = this.ctx.archiveFileStore().getUsableSpace();
            long threshold = this.ctx.lowStorageSpaceThreshold();
            if (usableSpace <= threshold) {
                String msg = "low storage threshold=" + threshold + " <= usableSpace=" + usableSpace;
                controlSession.sendErrorResponse(correlationId, 11L, msg, this.controlResponseProxy);
                return true;
            }
        }
        catch (IOException ex) {
            LangUtil.rethrowUnchecked((Throwable)ex);
        }
        return false;
    }
}

