/*
 * Decompiled with CFR 0.152.
 */
package io.aeron.archive;

import io.aeron.archive.ArchiveMarkFile;
import io.aeron.archive.CatalogIndex;
import io.aeron.archive.RecordingSummary;
import io.aeron.archive.checksum.Checksum;
import io.aeron.archive.client.ArchiveException;
import io.aeron.archive.codecs.CatalogHeaderDecoder;
import io.aeron.archive.codecs.CatalogHeaderEncoder;
import io.aeron.archive.codecs.RecordingDescriptorDecoder;
import io.aeron.archive.codecs.RecordingDescriptorEncoder;
import io.aeron.archive.codecs.RecordingDescriptorHeaderDecoder;
import io.aeron.archive.codecs.RecordingDescriptorHeaderEncoder;
import io.aeron.archive.codecs.RecordingState;
import io.aeron.logbuffer.FrameDescriptor;
import java.io.File;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.nio.MappedByteBuffer;
import java.nio.channels.FileChannel;
import java.nio.file.OpenOption;
import java.nio.file.StandardOpenOption;
import java.util.ArrayList;
import java.util.function.IntConsumer;
import java.util.function.Predicate;
import org.agrona.AsciiEncoding;
import org.agrona.BitUtil;
import org.agrona.BufferUtil;
import org.agrona.CloseHelper;
import org.agrona.DirectBuffer;
import org.agrona.LangUtil;
import org.agrona.MutableDirectBuffer;
import org.agrona.SemanticVersion;
import org.agrona.concurrent.EpochClock;
import org.agrona.concurrent.UnsafeBuffer;

final class Catalog
implements AutoCloseable {
    static final int PAGE_SIZE = 4096;
    static final int NULL_RECORD_ID = -1;
    static final int DESCRIPTOR_HEADER_LENGTH = 32;
    @Deprecated
    static final int DEFAULT_RECORD_LENGTH = 1024;
    static final int DEFAULT_ALIGNMENT = 1024;
    static final long MAX_CATALOG_LENGTH = Integer.MAX_VALUE;
    static final long DEFAULT_CAPACITY = 0x100000L;
    static final long MIN_CAPACITY = 32L;
    private final CatalogHeaderDecoder catalogHeaderDecoder = new CatalogHeaderDecoder();
    private final CatalogHeaderEncoder catalogHeaderEncoder = new CatalogHeaderEncoder();
    private final RecordingDescriptorHeaderDecoder descriptorHeaderDecoder = new RecordingDescriptorHeaderDecoder();
    private final RecordingDescriptorHeaderEncoder descriptorHeaderEncoder = new RecordingDescriptorHeaderEncoder();
    private final RecordingDescriptorEncoder descriptorEncoder = new RecordingDescriptorEncoder();
    private final RecordingDescriptorDecoder descriptorDecoder = new RecordingDescriptorDecoder();
    private final boolean forceWrites;
    private final boolean forceMetadata;
    private boolean isClosed;
    private final File catalogFile;
    private final File archiveDir;
    private final EpochClock epochClock;
    private final Checksum checksum;
    private final CatalogIndex catalogIndex = new CatalogIndex();
    private final int alignment;
    private final int firstRecordingDescriptorOffset;
    private FileChannel catalogChannel;
    private MappedByteBuffer catalogByteBuffer;
    private UnsafeBuffer catalogBuffer;
    private UnsafeBuffer fieldAccessBuffer;
    private UnsafeBuffer headerAccessBuffer;
    private long catalogByteBufferAddress;
    private long capacity;
    private long nextRecordingId;
    private int nextRecordingDescriptorOffset;

    Catalog(File archiveDir, FileChannel archiveDirChannel, int fileSyncLevel, long catalogCapacity, EpochClock epochClock, Checksum checksum, UnsafeBuffer buffer) {
        this.archiveDir = archiveDir;
        this.forceWrites = fileSyncLevel > 0;
        this.forceMetadata = fileSyncLevel > 1;
        this.epochClock = epochClock;
        this.checksum = checksum;
        Catalog.validateCapacity(catalogCapacity);
        this.catalogFile = new File(archiveDir, "archive.catalog");
        try {
            boolean catalogExists = this.catalogFile.exists();
            MappedByteBuffer catalogMappedByteBuffer = null;
            FileChannel catalogFileChannel = null;
            try {
                catalogFileChannel = FileChannel.open(this.catalogFile.toPath(), StandardOpenOption.CREATE, StandardOpenOption.READ, StandardOpenOption.WRITE, StandardOpenOption.SPARSE);
                this.capacity = catalogExists ? Math.max(catalogFileChannel.size(), catalogCapacity) : catalogCapacity;
                catalogMappedByteBuffer = catalogFileChannel.map(FileChannel.MapMode.READ_WRITE, 0L, this.capacity);
            }
            catch (Exception ex) {
                CloseHelper.close(catalogFileChannel);
                LangUtil.rethrowUnchecked((Throwable)ex);
            }
            this.catalogChannel = catalogFileChannel;
            this.initBuffers(catalogMappedByteBuffer);
            UnsafeBuffer catalogHeaderBuffer = new UnsafeBuffer((ByteBuffer)this.catalogByteBuffer);
            this.catalogHeaderDecoder.wrap((DirectBuffer)catalogHeaderBuffer, 0, 32, 8);
            this.catalogHeaderEncoder.wrap((MutableDirectBuffer)catalogHeaderBuffer, 0);
            if (catalogExists) {
                int version = this.catalogHeaderDecoder.version();
                if (SemanticVersion.major((int)version) != 3) {
                    throw new ArchiveException("incompatible catalog file version " + SemanticVersion.toString((int)version) + ", archive software is " + SemanticVersion.toString((int)ArchiveMarkFile.SEMANTIC_VERSION));
                }
                this.alignment = this.catalogHeaderDecoder.alignment();
                this.nextRecordingId = this.catalogHeaderDecoder.nextRecordingId();
            } else {
                this.alignment = 64;
                this.catalogHeaderEncoder.version(ArchiveMarkFile.SEMANTIC_VERSION).length(32).nextRecordingId(this.nextRecordingId).alignment(this.alignment);
                this.forceWrites(archiveDirChannel);
            }
            this.firstRecordingDescriptorOffset = 32;
            this.buildIndex(true);
            this.refreshCatalog(true, checksum, buffer);
        }
        catch (Exception ex) {
            this.close();
            throw ex;
        }
    }

    Catalog(File archiveDir, EpochClock epochClock) {
        this(archiveDir, epochClock, 32L, false, null, null);
    }

    Catalog(File archiveDir, EpochClock epochClock, long catalogCapacity, boolean writable, Checksum checksum, IntConsumer versionCheck) {
        this.archiveDir = archiveDir;
        this.forceWrites = false;
        this.forceMetadata = false;
        this.epochClock = epochClock;
        this.catalogChannel = null;
        this.checksum = checksum;
        this.catalogFile = new File(archiveDir, "archive.catalog");
        Catalog.validateCapacity(catalogCapacity);
        try {
            int alignment;
            OpenOption[] openOptionArray;
            MappedByteBuffer catalogMappedByteBuffer = null;
            if (writable) {
                StandardOpenOption[] standardOpenOptionArray = new StandardOpenOption[3];
                standardOpenOptionArray[0] = StandardOpenOption.READ;
                standardOpenOptionArray[1] = StandardOpenOption.WRITE;
                openOptionArray = standardOpenOptionArray;
                standardOpenOptionArray[2] = StandardOpenOption.SPARSE;
            } else {
                OpenOption[] openOptionArray2 = new StandardOpenOption[1];
                openOptionArray = openOptionArray2;
                openOptionArray2[0] = StandardOpenOption.READ;
            }
            OpenOption[] openOptions = openOptionArray;
            try (FileChannel channel = FileChannel.open(this.catalogFile.toPath(), openOptions);){
                this.capacity = Math.max(channel.size(), catalogCapacity);
                catalogMappedByteBuffer = channel.map(writable ? FileChannel.MapMode.READ_WRITE : FileChannel.MapMode.READ_ONLY, 0L, this.capacity);
            }
            catch (Exception ex) {
                LangUtil.rethrowUnchecked((Throwable)ex);
            }
            this.initBuffers(catalogMappedByteBuffer);
            UnsafeBuffer catalogHeaderBuffer = new UnsafeBuffer((ByteBuffer)this.catalogByteBuffer);
            this.catalogHeaderDecoder.wrap((DirectBuffer)catalogHeaderBuffer, 0, 32, 8);
            this.catalogHeaderEncoder.wrap((MutableDirectBuffer)catalogHeaderBuffer, 0);
            int version = this.catalogHeaderDecoder.version();
            if (null == versionCheck) {
                if (SemanticVersion.major((int)version) != 3) {
                    throw new ArchiveException("incompatible catalog file version " + SemanticVersion.toString((int)version) + ", archive software is " + SemanticVersion.toString((int)ArchiveMarkFile.SEMANTIC_VERSION));
                }
            } else {
                versionCheck.accept(version);
            }
            if (0 != (alignment = this.catalogHeaderDecoder.alignment())) {
                this.alignment = alignment;
                this.firstRecordingDescriptorOffset = 32;
                this.nextRecordingId = this.catalogHeaderDecoder.nextRecordingId();
            } else {
                this.alignment = 1024;
                this.firstRecordingDescriptorOffset = 1024;
            }
            this.buildIndex(writable);
            this.refreshCatalog(false, null, null);
        }
        catch (Exception ex) {
            this.close();
            throw ex;
        }
    }

    @Override
    public void close() {
        if (!this.isClosed) {
            this.isClosed = true;
            this.unmapAndCloseChannel();
        }
    }

    boolean isClosed() {
        return this.isClosed;
    }

    long capacity() {
        return this.capacity;
    }

    int alignment() {
        return this.alignment;
    }

    int entryCount() {
        return this.catalogIndex.size();
    }

    long nextRecordingId() {
        return this.nextRecordingId;
    }

    CatalogIndex index() {
        return this.catalogIndex;
    }

    int version() {
        return this.catalogHeaderDecoder.version();
    }

    void updateVersion(int version) {
        this.catalogHeaderEncoder.version(version);
    }

    long addNewRecording(long startPosition, long stopPosition, long startTimestamp, long stopTimestamp, int imageInitialTermId, int segmentFileLength, int termBufferLength, int mtuLength, int sessionId, int streamId, String strippedChannel, String originalChannel, String sourceIdentity) {
        int recordingDescriptorOffset = this.nextRecordingDescriptorOffset;
        int frameLength = this.recordingDescriptorFrameLength(strippedChannel, originalChannel, sourceIdentity);
        if ((long)(recordingDescriptorOffset + frameLength) > this.capacity) {
            this.growCatalog(Integer.MAX_VALUE, frameLength);
        }
        long recordingId = this.nextRecordingId;
        this.catalogBuffer.wrap((ByteBuffer)this.catalogByteBuffer, recordingDescriptorOffset, frameLength);
        this.descriptorEncoder.wrap((MutableDirectBuffer)this.catalogBuffer, 32).recordingId(recordingId).startTimestamp(startTimestamp).stopTimestamp(stopTimestamp).startPosition(startPosition).stopPosition(stopPosition).initialTermId(imageInitialTermId).segmentFileLength(segmentFileLength).termBufferLength(termBufferLength).mtuLength(mtuLength).sessionId(sessionId).streamId(streamId).strippedChannel(strippedChannel).originalChannel(originalChannel).sourceIdentity(sourceIdentity);
        int recordingLength = frameLength - 32;
        this.descriptorHeaderEncoder.wrap((MutableDirectBuffer)this.catalogBuffer, 0).length(recordingLength).checksum(this.computeRecordingDescriptorChecksum(recordingDescriptorOffset, recordingLength)).state(RecordingState.VALID);
        this.catalogHeaderEncoder.nextRecordingId(recordingId + 1L);
        this.forceWrites(this.catalogChannel);
        this.nextRecordingId = recordingId + 1L;
        this.nextRecordingDescriptorOffset = recordingDescriptorOffset + frameLength;
        this.catalogIndex.add(recordingId, recordingDescriptorOffset);
        return recordingId;
    }

    long addNewRecording(long startPosition, long startTimestamp, int imageInitialTermId, int segmentFileLength, int termBufferLength, int mtuLength, int sessionId, int streamId, String strippedChannel, String originalChannel, String sourceIdentity) {
        return this.addNewRecording(startPosition, -1L, startTimestamp, -1L, imageInitialTermId, segmentFileLength, termBufferLength, mtuLength, sessionId, streamId, strippedChannel, originalChannel, sourceIdentity);
    }

    boolean wrapDescriptor(long recordingId, UnsafeBuffer buffer) {
        if (recordingId < 0L) {
            return false;
        }
        int recordingDescriptorOffset = this.recordingDescriptorOffset(recordingId);
        if (recordingDescriptorOffset < 0) {
            return false;
        }
        return this.wrapDescriptorAtOffset(buffer, recordingDescriptorOffset) > 0;
    }

    int wrapDescriptorAtOffset(UnsafeBuffer buffer, int recordingDescriptorOffset) {
        int recordingLength = this.fieldAccessBuffer.getInt(recordingDescriptorOffset + RecordingDescriptorHeaderDecoder.lengthEncodingOffset(), RecordingDescriptorDecoder.BYTE_ORDER);
        if (recordingLength > 0) {
            int frameLength = BitUtil.align((int)(recordingLength + 32), (int)this.alignment);
            buffer.wrap((ByteBuffer)this.catalogByteBuffer, recordingDescriptorOffset, frameLength);
            return frameLength;
        }
        return -1;
    }

    boolean hasRecording(long recordingId) {
        return recordingId >= 0L && this.recordingDescriptorOffset(recordingId) > 0;
    }

    int forEach(CatalogEntryProcessor consumer) {
        int frameLength;
        int count = 0;
        int offset = this.firstRecordingDescriptorOffset;
        while (offset < this.nextRecordingDescriptorOffset && (frameLength = this.wrapDescriptorAtOffset(this.catalogBuffer, offset)) >= 0) {
            this.invokeEntryProcessor(offset, consumer);
            offset += frameLength;
            ++count;
        }
        return count;
    }

    boolean forEntry(long recordingId, CatalogEntryProcessor consumer) {
        if (recordingId < 0L) {
            return false;
        }
        int recordingDescriptorOffset = this.recordingDescriptorOffset(recordingId);
        if (recordingDescriptorOffset < 0) {
            return false;
        }
        if (this.wrapDescriptorAtOffset(this.catalogBuffer, recordingDescriptorOffset) > 0) {
            this.invokeEntryProcessor(recordingDescriptorOffset, consumer);
            return true;
        }
        return false;
    }

    long findLast(long minRecordingId, int sessionId, int streamId, byte[] channelFragment) {
        int lastPosition;
        long recordingId;
        if (minRecordingId < 0L) {
            return -1L;
        }
        long[] index = this.catalogIndex.index();
        for (int i = lastPosition = this.catalogIndex.lastPosition(); i >= 0 && (recordingId = index[i]) >= minRecordingId; i -= 2) {
            this.wrapDescriptorAtOffset(this.catalogBuffer, (int)index[i + 1]);
            this.descriptorDecoder.wrap((DirectBuffer)this.catalogBuffer, 32, 80, 8);
            if (sessionId != this.descriptorDecoder.sessionId() || streamId != this.descriptorDecoder.streamId() || !Catalog.originalChannelContains(this.descriptorDecoder, channelFragment)) continue;
            return recordingId;
        }
        return -1L;
    }

    static boolean originalChannelContains(RecordingDescriptorDecoder descriptorDecoder, byte[] channelFragment) {
        int offset;
        int fragmentLength = channelFragment.length;
        if (0 == fragmentLength) {
            return true;
        }
        int limit = descriptorDecoder.limit();
        int strippedChannelLength = descriptorDecoder.strippedChannelLength();
        int originalChannelOffset = limit + RecordingDescriptorDecoder.strippedChannelHeaderLength() + strippedChannelLength;
        descriptorDecoder.limit(originalChannelOffset);
        int channelLength = descriptorDecoder.originalChannelLength();
        descriptorDecoder.limit(limit);
        DirectBuffer buffer = descriptorDecoder.buffer();
        int end = offset + (channelLength - fragmentLength);
        block0: for (offset = descriptorDecoder.offset() + descriptorDecoder.sbeBlockLength() + RecordingDescriptorDecoder.strippedChannelHeaderLength() + strippedChannelLength + RecordingDescriptorDecoder.originalChannelHeaderLength(); offset <= end; ++offset) {
            for (int i = 0; i < fragmentLength; ++i) {
                if (buffer.getByte(offset + i) != channelFragment[i]) continue block0;
            }
            return true;
        }
        return false;
    }

    void recordingStopped(long recordingId, long position, long timestampMs) {
        int recordingDescriptorOffset = this.recordingDescriptorOffset(recordingId);
        int offset = recordingDescriptorOffset + 32;
        long stopPosition = ByteOrder.nativeOrder() == RecordingDescriptorDecoder.BYTE_ORDER ? position : Long.reverseBytes(position);
        this.fieldAccessBuffer.putLong(offset + RecordingDescriptorDecoder.stopTimestampEncodingOffset(), timestampMs, RecordingDescriptorDecoder.BYTE_ORDER);
        this.fieldAccessBuffer.putLongVolatile(offset + RecordingDescriptorDecoder.stopPositionEncodingOffset(), stopPosition);
        this.updateChecksum(recordingDescriptorOffset);
        this.forceWrites(this.catalogChannel);
    }

    void stopPosition(long recordingId, long position) {
        int recordingDescriptorOffset = this.recordingDescriptorOffset(recordingId);
        int offset = recordingDescriptorOffset + 32;
        long stopPosition = ByteOrder.nativeOrder() == RecordingDescriptorDecoder.BYTE_ORDER ? position : Long.reverseBytes(position);
        this.fieldAccessBuffer.putLongVolatile(offset + RecordingDescriptorDecoder.stopPositionEncodingOffset(), stopPosition);
        this.updateChecksum(recordingDescriptorOffset);
        this.forceWrites(this.catalogChannel);
    }

    void extendRecording(long recordingId, long controlSessionId, long correlationId, int sessionId) {
        int recordingDescriptorOffset = this.recordingDescriptorOffset(recordingId);
        int offset = recordingDescriptorOffset + 32;
        long stopPosition = ByteOrder.nativeOrder() == RecordingDescriptorDecoder.BYTE_ORDER ? -1L : Long.reverseBytes(-1L);
        this.fieldAccessBuffer.putLong(offset + RecordingDescriptorDecoder.controlSessionIdEncodingOffset(), controlSessionId, RecordingDescriptorDecoder.BYTE_ORDER);
        this.fieldAccessBuffer.putLong(offset + RecordingDescriptorDecoder.correlationIdEncodingOffset(), correlationId, RecordingDescriptorDecoder.BYTE_ORDER);
        this.fieldAccessBuffer.putLong(offset + RecordingDescriptorDecoder.stopTimestampEncodingOffset(), -1L, RecordingDescriptorDecoder.BYTE_ORDER);
        this.fieldAccessBuffer.putInt(offset + RecordingDescriptorDecoder.sessionIdEncodingOffset(), sessionId, RecordingDescriptorDecoder.BYTE_ORDER);
        this.fieldAccessBuffer.putLongVolatile(offset + RecordingDescriptorDecoder.stopPositionEncodingOffset(), stopPosition);
        this.updateChecksum(recordingDescriptorOffset);
        this.forceWrites(this.catalogChannel);
    }

    void replaceRecording(long recordingId, long startPosition, long stopPosition, long startTimestamp, long stopTimestamp, int imageInitialTermId, int segmentFileLength, int termBufferLength, int mtuLength, int sessionId, int streamId, String strippedChannel, String originalChannel, String sourceIdentity) {
        int length;
        int checksumLength;
        int recordingOffset = this.recordingDescriptorOffset(recordingId);
        if (-1 == recordingOffset) {
            throw new ArchiveException("unknown recording id: " + recordingId);
        }
        int recordingLength = this.fieldAccessBuffer.getInt(recordingOffset + RecordingDescriptorHeaderDecoder.lengthEncodingOffset(), RecordingDescriptorDecoder.BYTE_ORDER);
        int oldFrameLength = BitUtil.align((int)(recordingLength + 32), (int)this.alignment);
        int newFrameLength = this.recordingDescriptorFrameLength(strippedChannel, originalChannel, sourceIdentity);
        if (newFrameLength > oldFrameLength) {
            int lastPosition;
            length = checksumLength = newFrameLength - 32;
            int endOfLastRecording = this.nextRecordingDescriptorOffset;
            int shiftBytes = newFrameLength - oldFrameLength;
            if ((long)(endOfLastRecording + shiftBytes) > this.capacity) {
                this.growCatalog(Integer.MAX_VALUE, shiftBytes);
            }
            this.nextRecordingDescriptorOffset += shiftBytes;
            long[] index = this.catalogIndex.index();
            if (recordingId != index[lastPosition = this.catalogIndex.lastPosition()]) {
                this.shiftDataToTheRight(recordingOffset, oldFrameLength, newFrameLength, endOfLastRecording);
                Catalog.fixupIndexForShifterRecordings(index, lastPosition, recordingId, shiftBytes);
            }
            this.catalogBuffer.wrap((ByteBuffer)this.catalogByteBuffer, recordingOffset, newFrameLength);
        } else {
            length = oldFrameLength - 32;
            checksumLength = newFrameLength - 32;
            this.catalogBuffer.wrap((ByteBuffer)this.catalogByteBuffer, recordingOffset, oldFrameLength);
            this.catalogBuffer.setMemory(newFrameLength, oldFrameLength - newFrameLength, (byte)0);
        }
        this.descriptorEncoder.wrap((MutableDirectBuffer)this.catalogBuffer, 32).recordingId(recordingId).startTimestamp(startTimestamp).stopTimestamp(stopTimestamp).startPosition(startPosition).stopPosition(stopPosition).initialTermId(imageInitialTermId).segmentFileLength(segmentFileLength).termBufferLength(termBufferLength).mtuLength(mtuLength).sessionId(sessionId).streamId(streamId).strippedChannel(strippedChannel).originalChannel(originalChannel).sourceIdentity(sourceIdentity);
        this.descriptorHeaderEncoder.wrap((MutableDirectBuffer)this.catalogBuffer, 0).length(length).checksum(this.computeRecordingDescriptorChecksum(recordingOffset, checksumLength)).state(RecordingState.VALID);
        this.forceWrites(this.catalogChannel);
    }

    long startPosition(long recordingId) {
        int offset = this.recordingDescriptorOffset(recordingId) + 32 + RecordingDescriptorDecoder.startPositionEncodingOffset();
        long startPosition = this.fieldAccessBuffer.getLongVolatile(offset);
        return ByteOrder.nativeOrder() == RecordingDescriptorDecoder.BYTE_ORDER ? startPosition : Long.reverseBytes(startPosition);
    }

    void startPosition(long recordingId, long position) {
        int recordingDescriptorOffset = this.recordingDescriptorOffset(recordingId);
        int offset = recordingDescriptorOffset + 32 + RecordingDescriptorDecoder.startPositionEncodingOffset();
        this.fieldAccessBuffer.putLong(offset, position, RecordingDescriptorDecoder.BYTE_ORDER);
        this.updateChecksum(recordingDescriptorOffset);
        this.forceWrites(this.catalogChannel);
    }

    long stopPosition(long recordingId) {
        int offset = this.recordingDescriptorOffset(recordingId) + 32 + RecordingDescriptorDecoder.stopPositionEncodingOffset();
        long stopPosition = this.fieldAccessBuffer.getLongVolatile(offset);
        return ByteOrder.nativeOrder() == RecordingDescriptorDecoder.BYTE_ORDER ? stopPosition : Long.reverseBytes(stopPosition);
    }

    boolean invalidateRecording(long recordingId) {
        long offset;
        if (recordingId >= 0L && -1L != (offset = this.catalogIndex.remove(recordingId))) {
            this.fieldAccessBuffer.putInt((int)offset + RecordingDescriptorHeaderEncoder.stateEncodingOffset(), RecordingState.INVALID.value(), RecordingDescriptorDecoder.BYTE_ORDER);
            this.forceWrites(this.catalogChannel);
            return true;
        }
        return false;
    }

    RecordingSummary recordingSummary(long recordingId, RecordingSummary summary) {
        int offset = this.recordingDescriptorOffset(recordingId) + 32;
        summary.recordingId = recordingId;
        summary.startPosition = this.fieldAccessBuffer.getLong(offset + RecordingDescriptorDecoder.startPositionEncodingOffset(), RecordingDescriptorDecoder.BYTE_ORDER);
        summary.stopPosition = this.fieldAccessBuffer.getLong(offset + RecordingDescriptorDecoder.stopPositionEncodingOffset(), RecordingDescriptorDecoder.BYTE_ORDER);
        summary.initialTermId = this.fieldAccessBuffer.getInt(offset + RecordingDescriptorDecoder.initialTermIdEncodingOffset(), RecordingDescriptorDecoder.BYTE_ORDER);
        summary.segmentFileLength = this.fieldAccessBuffer.getInt(offset + RecordingDescriptorDecoder.segmentFileLengthEncodingOffset(), RecordingDescriptorDecoder.BYTE_ORDER);
        summary.termBufferLength = this.fieldAccessBuffer.getInt(offset + RecordingDescriptorDecoder.termBufferLengthEncodingOffset(), RecordingDescriptorDecoder.BYTE_ORDER);
        summary.mtuLength = this.fieldAccessBuffer.getInt(offset + RecordingDescriptorDecoder.mtuLengthEncodingOffset(), RecordingDescriptorDecoder.BYTE_ORDER);
        summary.streamId = this.fieldAccessBuffer.getInt(offset + RecordingDescriptorDecoder.streamIdEncodingOffset(), RecordingDescriptorDecoder.BYTE_ORDER);
        summary.sessionId = this.fieldAccessBuffer.getInt(offset + RecordingDescriptorDecoder.sessionIdEncodingOffset(), RecordingDescriptorDecoder.BYTE_ORDER);
        return summary;
    }

    static int descriptorLength(UnsafeBuffer descriptorBuffer) {
        return descriptorBuffer.getInt(RecordingDescriptorHeaderDecoder.lengthEncodingOffset(), RecordingDescriptorDecoder.BYTE_ORDER);
    }

    static boolean isValidDescriptor(UnsafeBuffer descriptorBuffer) {
        return RecordingState.VALID.value() == descriptorBuffer.getInt(RecordingDescriptorHeaderDecoder.stateEncodingOffset(), RecordingDescriptorDecoder.BYTE_ORDER);
    }

    static long recordingId(UnsafeBuffer descriptorBuffer) {
        return descriptorBuffer.getLong(32 + RecordingDescriptorDecoder.recordingIdEncodingOffset(), RecordingDescriptorDecoder.BYTE_ORDER);
    }

    int recordingDescriptorOffset(long recordingId) {
        long recordingDescriptorOffset = this.catalogIndex.recordingOffset(recordingId);
        if (-1L == recordingDescriptorOffset) {
            return -1;
        }
        return (int)recordingDescriptorOffset;
    }

    void growCatalog(long maxCatalogCapacity, int frameLength) {
        MappedByteBuffer mappedByteBuffer;
        long oldCapacity = this.capacity;
        long recordingOffset = this.nextRecordingDescriptorOffset;
        long targetCapacity = recordingOffset + (long)frameLength;
        if (targetCapacity > maxCatalogCapacity) {
            if (maxCatalogCapacity == oldCapacity) {
                throw new ArchiveException("catalog is full, max capacity reached: " + maxCatalogCapacity);
            }
            throw new ArchiveException("recording is too big: total recording length is " + frameLength + " bytes, available space is " + (maxCatalogCapacity - recordingOffset) + " bytes");
        }
        long newCapacity = oldCapacity;
        while (newCapacity < targetCapacity) {
            newCapacity = Math.min(newCapacity + (newCapacity >> 1), maxCatalogCapacity);
        }
        try {
            this.unmapAndCloseChannel();
            this.catalogChannel = FileChannel.open(this.catalogFile.toPath(), StandardOpenOption.READ, StandardOpenOption.WRITE, StandardOpenOption.SPARSE);
            mappedByteBuffer = this.catalogChannel.map(FileChannel.MapMode.READ_WRITE, 0L, newCapacity);
        }
        catch (Exception ex) {
            this.close();
            LangUtil.rethrowUnchecked((Throwable)ex);
            return;
        }
        this.capacity = newCapacity;
        this.initBuffers(mappedByteBuffer);
        UnsafeBuffer catalogHeaderBuffer = new UnsafeBuffer((ByteBuffer)this.catalogByteBuffer);
        this.catalogHeaderDecoder.wrap((DirectBuffer)catalogHeaderBuffer, 0, 32, 8);
        this.catalogHeaderEncoder.wrap((MutableDirectBuffer)catalogHeaderBuffer, 0);
        this.catalogResized(oldCapacity, newCapacity);
    }

    void updateChecksum(int recordingDescriptorOffset) {
        if (null != this.checksum) {
            UnsafeBuffer headerBuffer = this.headerAccessBuffer;
            int recordingLength = headerBuffer.getInt(recordingDescriptorOffset + RecordingDescriptorHeaderEncoder.lengthEncodingOffset(), RecordingDescriptorDecoder.BYTE_ORDER);
            int checksumValue = this.checksum.compute(this.catalogByteBufferAddress, 32 + recordingDescriptorOffset, recordingLength);
            headerBuffer.putInt(recordingDescriptorOffset + RecordingDescriptorHeaderEncoder.checksumEncodingOffset(), checksumValue, RecordingDescriptorDecoder.BYTE_ORDER);
        }
    }

    int computeRecordingDescriptorChecksum(int recordingDescriptorOffset, int recordingLength) {
        if (null != this.checksum) {
            return this.checksum.compute(this.catalogByteBufferAddress, 32 + recordingDescriptorOffset, recordingLength);
        }
        return 0;
    }

    void catalogResized(long oldCapacity, long newCapacity) {
    }

    private static void validateCapacity(long catalogCapacity) {
        if (catalogCapacity < 32L || catalogCapacity > Integer.MAX_VALUE) {
            throw new IllegalArgumentException("Invalid catalog capacity provided: expected value >= 32, got " + catalogCapacity);
        }
    }

    private void initBuffers(MappedByteBuffer catalogMappedByteBuffer) {
        this.catalogByteBuffer = catalogMappedByteBuffer;
        this.catalogByteBuffer.order(RecordingDescriptorDecoder.BYTE_ORDER);
        this.catalogBuffer = new UnsafeBuffer((ByteBuffer)this.catalogByteBuffer);
        this.catalogByteBufferAddress = this.catalogBuffer.addressOffset();
        this.fieldAccessBuffer = new UnsafeBuffer((ByteBuffer)this.catalogByteBuffer);
        this.headerAccessBuffer = new UnsafeBuffer((ByteBuffer)this.catalogByteBuffer);
    }

    private void buildIndex(boolean writable) {
        int offset;
        int frameLength;
        int endOffset = (int)this.capacity;
        long recordingId = -1L;
        for (offset = this.firstRecordingDescriptorOffset; offset < endOffset && (frameLength = this.wrapDescriptorAtOffset(this.catalogBuffer, offset)) >= 0; offset += frameLength) {
            recordingId = Catalog.recordingId(this.catalogBuffer);
            if (!Catalog.isValidDescriptor(this.catalogBuffer)) continue;
            this.catalogIndex.add(recordingId, offset);
        }
        this.nextRecordingDescriptorOffset = offset;
        if (0L == this.nextRecordingId) {
            this.nextRecordingId = recordingId + 1L;
        } else if (writable && this.nextRecordingId < recordingId + 1L) {
            throw new ArchiveException("invalid nextRecordingId: expected value greater or equal to " + (recordingId + 1L) + ", was " + this.nextRecordingId);
        }
    }

    private void invokeEntryProcessor(int recordingDescriptorOffset, CatalogEntryProcessor consumer) {
        this.descriptorHeaderDecoder.wrap((DirectBuffer)this.catalogBuffer, 0, 32, 8);
        this.descriptorHeaderEncoder.wrap((MutableDirectBuffer)this.catalogBuffer, 0);
        this.descriptorDecoder.wrap((DirectBuffer)this.catalogBuffer, 32, 80, 8);
        this.descriptorEncoder.wrap((MutableDirectBuffer)this.catalogBuffer, 32);
        consumer.accept(recordingDescriptorOffset, this.descriptorHeaderEncoder, this.descriptorHeaderDecoder, this.descriptorEncoder, this.descriptorDecoder);
    }

    private int recordingDescriptorFrameLength(String strippedChannel, String originalChannel, String sourceIdentity) {
        int recordingDescriptorLength = 84 + strippedChannel.length() + 4 + originalChannel.length() + 4 + sourceIdentity.length();
        return BitUtil.align((int)(32 + recordingDescriptorLength), (int)this.alignment);
    }

    private void shiftDataToTheRight(int recordingOffset, int oldFrameLength, int newFrameLength, int endOfLastRecording) {
        this.fieldAccessBuffer.putBytes(recordingOffset + newFrameLength, (DirectBuffer)this.fieldAccessBuffer, recordingOffset + oldFrameLength, endOfLastRecording - (recordingOffset + oldFrameLength));
    }

    private static void fixupIndexForShifterRecordings(long[] index, int lastPosition, long recordingId, int shiftBytes) {
        boolean updateOffset = false;
        for (int i = 0; i <= lastPosition; i += 2) {
            if (updateOffset) {
                int n = i + 1;
                index[n] = index[n] + (long)shiftBytes;
                continue;
            }
            if (recordingId != index[i]) continue;
            updateOffset = true;
        }
    }

    private void refreshCatalog(boolean fixOnRefresh, Checksum checksum, UnsafeBuffer buffer) {
        if (fixOnRefresh) {
            UnsafeBuffer tmpBuffer = null != buffer ? buffer : new UnsafeBuffer(ByteBuffer.allocateDirect(0x100000));
            this.forEach((recordingDescriptorOffset, headerEncoder, headerDecoder, descriptorEncoder, descriptorDecoder) -> this.refreshAndFixDescriptor(headerDecoder, descriptorEncoder, descriptorDecoder, checksum, tmpBuffer));
        }
    }

    private void refreshAndFixDescriptor(RecordingDescriptorHeaderDecoder headerDecoder, RecordingDescriptorEncoder encoder, RecordingDescriptorDecoder decoder, Checksum checksum, UnsafeBuffer buffer) {
        long recordingId = decoder.recordingId();
        if (RecordingState.VALID == headerDecoder.state() && -1L == decoder.stopPosition()) {
            ArrayList<String> segmentFiles = Catalog.listSegmentFiles(this.archiveDir, recordingId);
            String maxSegmentFile = Catalog.findSegmentFileWithHighestPosition(segmentFiles);
            encoder.stopPosition(Catalog.computeStopPosition(this.archiveDir, maxSegmentFile, decoder.startPosition(), decoder.termBufferLength(), decoder.segmentFileLength(), checksum, buffer, segmentFile -> {
                throw new ArchiveException("Found potentially incomplete last fragment straddling page boundary in file: " + segmentFile.getAbsolutePath() + "\nRun `ArchiveTool verify` for corrective action!");
            }));
            encoder.stopTimestamp(this.epochClock.time());
        }
    }

    private void forceWrites(FileChannel channel) {
        if (null != channel && this.forceWrites) {
            try {
                channel.force(this.forceMetadata);
            }
            catch (Exception ex) {
                LangUtil.rethrowUnchecked((Throwable)ex);
            }
        }
    }

    static ArrayList<String> listSegmentFiles(File archiveDir, long recordingId) {
        ArrayList<String> segmentFiles = new ArrayList<String>();
        String[] files = archiveDir.list();
        if (null != files) {
            String prefix = recordingId + "-";
            for (String file : files) {
                if (!file.startsWith(prefix) || !file.endsWith(".rec")) continue;
                segmentFiles.add(file);
            }
        }
        return segmentFiles;
    }

    static String findSegmentFileWithHighestPosition(ArrayList<String> segmentFiles) {
        long maxSegmentPosition = -1L;
        String maxFileName = null;
        for (String filename : segmentFiles) {
            long filePosition = Catalog.parseSegmentFilePosition(filename);
            if (filePosition < 0L) {
                throw new ArchiveException("negative position encoded in the file name: " + filename);
            }
            if (filePosition <= maxSegmentPosition) continue;
            maxSegmentPosition = filePosition;
            maxFileName = filename;
        }
        return maxFileName;
    }

    static long parseSegmentFilePosition(String filename) {
        int dashOffset = filename.indexOf(45);
        if (-1 == dashOffset) {
            throw new ArchiveException("invalid filename format: " + filename);
        }
        int positionOffset = dashOffset + 1;
        int positionLength = filename.length() - positionOffset - ".rec".length();
        if (0 >= positionLength) {
            throw new ArchiveException("no position encoded in the segment file: " + filename);
        }
        return AsciiEncoding.parseLongAscii((CharSequence)filename, (int)positionOffset, (int)positionLength);
    }

    static long computeStopPosition(File archiveDir, String maxSegmentFile, long startPosition, int termLength, int segmentLength, Checksum checksum, UnsafeBuffer buffer, Predicate<File> truncateOnPageStraddle) {
        if (null == maxSegmentFile) {
            return startPosition;
        }
        int startTermOffset = (int)(startPosition & (long)(termLength - 1));
        long startTermBasePosition = startPosition - (long)startTermOffset;
        long segmentFileBasePosition = Catalog.parseSegmentFilePosition(maxSegmentFile);
        int fileOffset = segmentFileBasePosition == startTermBasePosition ? startTermOffset : 0;
        int segmentStopOffset = Catalog.recoverStopOffset(archiveDir, maxSegmentFile, fileOffset, segmentLength, truncateOnPageStraddle, checksum, buffer);
        return Math.max(segmentFileBasePosition + (long)segmentStopOffset, startPosition);
    }

    static boolean fragmentStraddlesPageBoundary(int fragmentOffset, int fragmentLength) {
        return fragmentOffset / 4096 != (fragmentOffset + (fragmentLength - 1)) / 4096;
    }

    private void unmapAndCloseChannel() {
        MappedByteBuffer buffer = this.catalogByteBuffer;
        BufferUtil.free((ByteBuffer)buffer);
        this.catalogByteBuffer = null;
        CloseHelper.close((AutoCloseable)this.catalogChannel);
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private static int recoverStopOffset(File archiveDir, String segmentFile, int offset, int segmentFileLength, Predicate<File> truncateOnPageStraddle, Checksum checksum, UnsafeBuffer buffer) {
        File file = new File(archiveDir, segmentFile);
        try (FileChannel segmentFileChannel = FileChannel.open(file.toPath(), StandardOpenOption.READ, StandardOpenOption.WRITE);){
            int lastFragmentOffset;
            int offsetLimit = (int)Math.min((long)segmentFileLength, segmentFileChannel.size());
            ByteBuffer byteBuffer = buffer.byteBuffer();
            int nextFragmentOffset = offset;
            int lastFragmentLength = 0;
            int bufferOffset = 0;
            block14: while (nextFragmentOffset < offsetLimit) {
                int bytesRead = Catalog.readNextChunk(segmentFileChannel, byteBuffer, nextFragmentOffset, offsetLimit);
                for (bufferOffset = 0; bufferOffset < bytesRead; nextFragmentOffset += lastFragmentLength, bufferOffset += lastFragmentLength) {
                    int frameLength = FrameDescriptor.frameLength((UnsafeBuffer)buffer, (int)bufferOffset);
                    if (frameLength <= 0) break block14;
                    lastFragmentLength = BitUtil.align((int)frameLength, (int)32);
                }
            }
            if (Catalog.fragmentStraddlesPageBoundary(lastFragmentOffset = nextFragmentOffset - lastFragmentLength, lastFragmentLength) && !Catalog.isValidFragment(buffer, bufferOffset - lastFragmentLength, lastFragmentLength, checksum) && truncateOnPageStraddle.test(file)) {
                boolean singleByte = true;
                segmentFileChannel.truncate(lastFragmentOffset);
                byteBuffer.put(0, (byte)0).limit(1).position(0);
                if (1 != segmentFileChannel.write(byteBuffer, segmentFileLength - 1)) {
                    throw new IllegalStateException("Failed to write single byte to set segment file length");
                }
                int n2 = lastFragmentOffset;
                return n2;
            }
            int n = nextFragmentOffset;
            return n;
        }
        catch (IOException ex) {
            LangUtil.rethrowUnchecked((Throwable)ex);
            return -1;
        }
    }

    private static int readNextChunk(FileChannel segment, ByteBuffer byteBuffer, int offset, int limit) throws IOException {
        int bytesRead;
        int position = offset;
        byteBuffer.clear().limit(Math.min(byteBuffer.capacity(), limit - position));
        while ((bytesRead = segment.read(byteBuffer, position)) >= 0) {
            position += bytesRead;
            if (byteBuffer.remaining() > 0) continue;
        }
        return position - offset;
    }

    private static boolean isValidFragment(UnsafeBuffer buffer, int fragmentOffset, int alignedFragmentLength, Checksum checksum) {
        return null != checksum && Catalog.hasValidChecksum(buffer, fragmentOffset, alignedFragmentLength, checksum) || Catalog.hasDataInAllPagesAfterStraddle(buffer, fragmentOffset, alignedFragmentLength);
    }

    private static boolean hasValidChecksum(UnsafeBuffer buffer, int fragmentOffset, int alignedFragmentLength, Checksum checksum) {
        int computedChecksum = checksum.compute(buffer.addressOffset(), fragmentOffset + 32, alignedFragmentLength - 32);
        int recordedChecksum = FrameDescriptor.frameSessionId((UnsafeBuffer)buffer, (int)fragmentOffset);
        return recordedChecksum == computedChecksum;
    }

    private static boolean hasDataInAllPagesAfterStraddle(UnsafeBuffer buffer, int fragmentOffset, int alignedFragmentLength) {
        int endOffset = fragmentOffset + alignedFragmentLength;
        for (int straddleOffset = (fragmentOffset / 4096 + 1) * 4096; straddleOffset < endOffset; straddleOffset += 4096) {
            if (!Catalog.isEmptyPage(buffer, straddleOffset, endOffset)) continue;
            return false;
        }
        return true;
    }

    private static boolean isEmptyPage(UnsafeBuffer buffer, int pageStart, int endOffset) {
        int pageEnd = Math.min(pageStart + 4096, endOffset);
        for (int i = pageStart; i < pageEnd; i += 8) {
            if (0L == buffer.getLong(i)) continue;
            return false;
        }
        return true;
    }

    public String toString() {
        return "Catalog{forceWrites=" + this.forceWrites + ", forceMetadata=" + this.forceMetadata + ", isClosed=" + this.isClosed + ", catalogFile=" + this.catalogFile + ", archiveDir=" + this.archiveDir + ", epochClock=" + this.epochClock + ", checksum=" + this.checksum + ", alignment=" + this.alignment + ", firstRecordingDescriptorOffset=" + this.firstRecordingDescriptorOffset + ", catalogChannel=" + this.catalogChannel + ", capacity=" + this.capacity + ", nextRecordingId=" + this.nextRecordingId + ", nextRecordingDescriptorOffset=" + this.nextRecordingDescriptorOffset + '}';
    }

    @FunctionalInterface
    static interface CatalogEntryProcessor {
        public void accept(int var1, RecordingDescriptorHeaderEncoder var2, RecordingDescriptorHeaderDecoder var3, RecordingDescriptorEncoder var4, RecordingDescriptorDecoder var5);
    }
}

