/*
 *  * EaseMob CONFIDENTIAL
 * __________________
 * Copyright (C) 2017 EaseMob Technologies. All rights reserved.
 *
 * NOTICE: All information contained herein is, and remains
 * the property of EaseMob Technologies.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from EaseMob Technologies.
 */
package io.agora.chat;

import static io.agora.util.EasyUtils.convertToCerts;
import static io.agora.util.EasyUtils.getSystemDefaultTrustManager;

import android.annotation.TargetApi;
import android.app.Activity;
import android.app.Application;
import android.app.Application.ActivityLifecycleCallbacks;
import android.content.BroadcastReceiver;
import android.content.Context;
import android.content.Intent;
import android.content.IntentFilter;
import android.net.ConnectivityManager;
import android.net.http.X509TrustManagerExtensions;
import android.os.Build;
import android.os.Bundle;
import android.os.PowerManager;
import android.os.PowerManager.WakeLock;
import android.support.annotation.NonNull;
import android.text.TextUtils;
import android.util.Log;

import io.agora.CallBack;
import io.agora.ConnectionListener;
import io.agora.Error;
import io.agora.ChatLogListener;
import io.agora.MultiDeviceListener;
import io.agora.ValueCallBack;
import io.agora.chat.adapter.EMAChatClient;
import io.agora.chat.adapter.EMAChatClient.EMANetwork;
import io.agora.chat.adapter.EMAConnectionListener;
import io.agora.chat.adapter.EMADeviceInfo;
import io.agora.chat.adapter.EMAError;
import io.agora.chat.adapter.EMALogCallbackListener;
import io.agora.chat.adapter.EMAMultiDeviceListener;
import io.agora.chat.adapter.EMANetCallback;
import io.agora.chat.core.EMAdvanceDebugManager;
import io.agora.chat.core.EMChatConfigPrivate;
import io.agora.chat.core.EMPreferenceUtils;
import io.agora.cloud.EMHttpClient;
import io.agora.exceptions.ChatException;
import io.agora.monitor.EMNetworkMonitor;
import io.agora.notification.core.EMNotificationHelper;
import io.agora.push.PushConfig;
import io.agora.push.PushHelper;
import io.agora.push.PushType;
import io.agora.util.DeviceUuidFactory;
import io.agora.util.EMLog;
import io.agora.util.NetUtils;
import io.agora.util.PathUtil;
import io.agora.util.Utils;

import org.json.JSONException;
import org.json.JSONObject;

import java.security.cert.X509Certificate;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.UUID;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.regex.Pattern;

import javax.net.ssl.X509TrustManager;

import internal.com.getkeepsafe.relinker.ReLinker;

/**
  * \~english
  * The chat client class, which is the entry of the chat SDK. It defines how to log in to and log out of the chat app and how to manage the connection between the SDK and the chat server.
  *  
  * ```java
  * ChatManager chatManager = ChatClient.getInstance().chatManager();
  * ```
  */
public class ChatClient {
    public final static String TAG = "ChatClient";
    static private ChatClient instance = null;
    static boolean libraryLoaded = false;

    private GroupManager groupManager;
    private ChatRoomManager chatroomManager;
    private ChatManager chatManager;
    private ContactManager contactManager;
    private UserInfoManager userInfoManager;
    private PushManager pushManager;
    private TranslationManager translationManager;
    private volatile PresenceManager presenceManager;
    private ChatThreadManager threadManager;
    private ChatStatisticsManager statisticsManager;

    private EMAChatClient emaObject;
    private Context mContext;
    private ExecutorService executor = null;
    private ExecutorService logQueue = Executors.newSingleThreadExecutor();
    private ExecutorService mainQueue = Executors.newSingleThreadExecutor();
    private ExecutorService sendQueue = Executors.newSingleThreadExecutor();
    private EMEncryptProvider encryptProvider = null;
    private boolean sdkInited = false;
    private EMChatConfigPrivate mChatConfigPrivate;
    private List<ConnectionListener> connectionListeners = Collections.synchronizedList(new ArrayList<ConnectionListener>());
    private List<ChatLogListener> logListeners = Collections.synchronizedList(new ArrayList<ChatLogListener>());
    private ClientLogListener clientLogListener;

	private MyConnectionListener connectionListener;
	private EMSmartHeartBeat smartHeartbeat = null;
    private List<MultiDeviceListener> multiDeviceListeners = Collections.synchronizedList(new ArrayList<MultiDeviceListener>());
    private MyMultiDeviceListener multiDeviceListenerImpl;
	private WakeLock wakeLock;

	private ConnectivityManager connManager;
	private EMANetwork currentNetworkType = EMANetwork.NETWORK_NONE;
    private boolean mIsLoginWithAgoraToken = false;

	public final static String VERSION = "1.2.0";

	/**
     * \~english
	 * Make sure that this ChatClient is initialized by the Chat SDK.
	 *
	 */
	private ChatClient() {
	}

	public static ChatClient getInstance() {
		if(instance == null){
		    synchronized (ChatClient.class) {
		        if(instance == null) {
                    instance = new ChatClient();
		        }
            }
		}
		return instance;
	}


    /**
    * \~english
    * Initializes the SDK.
    * 
    * Make sure to initialize the SDK in the main thread.
    *
    * @param context The context. Make sure to set the parameter.
    * @param options The configuration options. Make sure to set the parameter. See {@link ChatOptions}.
    */
    public void init(Context context, ChatOptions options) {
        if(sdkInited){
            return;
        }

        executor = Executors.newCachedThreadPool();

        final EMTimeTag tag = new EMTimeTag();
        tag.start();

        mContext = context.getApplicationContext();
        connManager = (ConnectivityManager) mContext.getSystemService(Context.CONNECTIVITY_SERVICE);
        //if android sdk >= 14,register callback to know app enter the background or foreground
        registerActivityLifecycleCallbacks();

        loadLibrary();

        mChatConfigPrivate = new EMChatConfigPrivate();
        mChatConfigPrivate.load(mContext, options);
        options.setConfig(mChatConfigPrivate);
        // multi-device is enabled by setDeviceUuid
        // config setDeviceUuid before EMAChatClient.create being called
        DeviceUuidFactory deviceFactory = new DeviceUuidFactory(mContext);
        mChatConfigPrivate.setDeviceUuid(deviceFactory.getDeviceUuid().toString());
        String deviceName= Build.MANUFACTURER + Build.MODEL;
        //The custom device name may override the system device name, and record the actual device with an error level to facilitate bug tracing
        EMLog.e(TAG,"system device name :"+deviceName);
        if(TextUtils.isEmpty(options.getCustomDeviceName())) {
            mChatConfigPrivate.setDeviceName(deviceName);
        }
        mChatConfigPrivate.setDid(getDidInfo());
        mChatConfigPrivate.setServiceId(UUID.randomUUID().toString());

        // Init push relevant.
        PushConfig pushConfig = options.getPushConfig();
        if (pushConfig == null) {
            pushConfig = new PushConfig.Builder(mContext).build();
        }

        PushConfig.Builder builder = new PushConfig.Builder(mContext, pushConfig);
        PushHelper.getInstance().init(mContext, builder.build());

        emaObject = EMAChatClient.create(mChatConfigPrivate.emaObject);
        // sandbox
        // rest 103.241.230.122:31080 im-msync 103.241.230.122:31097
//        mChatConfigPrivate.setChatServer("118.193.28.212");
//        mChatConfigPrivate.setChatPort(31097);
//        mChatConfigPrivate.setRestServer("https://118.193.28.212:31443");
//        mChatConfigPrivate.enableDnsConfig(false);

        connectionListener = new MyConnectionListener();
        emaObject.addConnectionListener(connectionListener);

        multiDeviceListenerImpl = new MyMultiDeviceListener();
        emaObject.addMultiDeviceListener(multiDeviceListenerImpl);

        // init all the managers
        initManagers();

        mIsLoginWithAgoraToken =EMSessionManager.getInstance().getIsLoginWithAgoraToken();

        final String lastLoginUser = EMSessionManager.getInstance().getLastLoginUser();

        EMLog.e(TAG, "is autoLogin : " + options.getAutoLogin());
        EMLog.e(TAG, "lastLoginUser : " + lastLoginUser);
        EMLog.e(TAG, "hyphenate SDK is initialized with version : " + getChatConfigPrivate().getVersion());

        PowerManager pm = (PowerManager) mContext.getSystemService(Context.POWER_SERVICE);
        wakeLock = pm.newWakeLock(PowerManager.PARTIAL_WAKE_LOCK, "emclient");
        EMNotificationHelper.getInstance().onInit(mContext);
        sdkInited = true;

        mContext.registerReceiver(connectivityBroadcastReceiver, new IntentFilter(ConnectivityManager.CONNECTIVITY_ACTION));
        onNetworkChanged();//主动更新一次，上面注册由于异步原因不会及时更新.数据上报需要

        if (options.getAutoLogin() && isLoggedInBefore()) {
                final String lastLoginToken = EMSessionManager.getInstance().getLastLoginToken();
                final String lastLoginPwd = EMSessionManager.getInstance().getLastLoginPwd();

                //sessionManager.login(lastLoginUser, lastLoginPwd,false, null);
                EMSessionManager.getInstance().currentUser = new EMContact(lastLoginUser);

                final CallBack callback = new CallBack() {

                    @Override
                    public void onSuccess() {
                        EMSessionManager.getInstance().currentUser = new EMContact(lastLoginUser);
                        Log.d(TAG, "hyphenate login onSuccess");
                        tag.stop(); EMLog.d(TAG, "[Collector][sdk init]init time is : " + tag.timeStr());
                    }

                    @Override
                    public void onError(int code, String error) {
                        Log.d(TAG, "hyphenate login onError");
                        tag.stop(); EMLog.d(TAG, "[Collector][sdk init]init time is : " + tag.timeStr());
                    }

                    @Override
                    public void onProgress(int progress, String status) {
                    }
                };

                _login(lastLoginUser,
                        EMSessionManager.getInstance().isLastLoginWithToken() ? lastLoginToken : lastLoginPwd,
                        callback,
                        /*autoLogin*/ true,
                        /*loginWithToken*/ EMSessionManager.getInstance().isLastLoginWithToken() ? EMLoginType.LOGIN_TOKEN : EMLoginType.LOGIN_PASSWORD);

        } else {
            tag.stop(); EMLog.d(TAG, "[Collector][sdk init]init time is : " + tag.timeStr());
        }
    }

    private String getDidInfo() {
        String manufacturer =Build.MANUFACTURER;
        String model = Build.MODEL;
        String hardware = Build.HARDWARE;
        int sdkInt = Build.VERSION.SDK_INT;
        String osVersion = Build.VERSION.RELEASE;

        String did=manufacturer+"/"
                +model+"/"
                +hardware+"/"
                +sdkInt+"/"
                +osVersion;
        return did;
    }

    /**
     * \~english
     * Adds a new user account.
     *
     * This method is not recommended and you are advised to call the RESTful API.
     *
     * This is a synchronous method and blocks the current thread.
     *
     * @param username The user ID. The maximum length is 64 characters. Ensure that you set this parameter. The user ID can contain characters of the following types:
   *                 - 26 English letters (a-z).
   *                 - 10 numbers (0-9).
   *                 - "_", "-", "."
   *                 The user ID is case-insensitive, so Aa and aa are the same user ID.
   *                 The email address or the UUID of the user cannot be used as the user ID.
   *                 You can also set this parameter with the regular expression ^[a-zA-Z0-9_-]+$.
     * @param password The password. The maximum length is 64 characters. Ensure that you set this parameter.
     *                 
     *
     * @throws ChatException A description of the issue that caused this call to fail. For example, the user account or password is null, or the account is illegal.
     *                 
     */
    public void createAccount(String username, String password) throws ChatException {
        username = username.toLowerCase();

        Pattern pattern = Pattern.compile("^[a-zA-Z0-9_.-]+$");
        boolean find = pattern.matcher(username).find();
        if (!find) {
            throw new ChatException(Error.USER_ILLEGAL_ARGUMENT, "illegal user name");
        }
        EMAError error = emaObject.createAccount(username, password);
        handleError(error);
    }

    /**
     * \~english
     * Logs in to the chat server with a password.
     * 
     * Also, you can call {@link #loginWithToken(String, String, CallBack)} to log in to the chat server with the user ID and token.
     *
     * This is an asynchronous method.
     *
     * @param id 		The unique chat user ID. Make sure to set the parameter.
     * @param password 	The password. Make sure to set the parameter.
     * @param callback 	The login callback. Make sure to set the parameter. The login result is returned via the callback.
     */
    public void login(String id, String password, @NonNull final CallBack callback) {

        if (id == null || password == null || id.equals("") || password.equals("")) {
            callback.onError(Error.INVALID_PARAM, "username or password is null or empty!");
            return;
        }

        if (TextUtils.isEmpty(getChatConfigPrivate().getAppKey())) {
            callback.onError(Error.INVALID_PARAM, "please setup your App Key  either in AndroidManifest.xml or through the ChatOptions");
            return;
        }

        if (!sdkInited) {
            callback.onError(Error.GENERAL_ERROR, "sdk not initialized");
            return;
        }

        id = id.toLowerCase();
        initLoginWithAgoraData(false,"",0);
        _login(id, password, callback, false, EMLoginType.LOGIN_PASSWORD);
    }

    /**
     * \~english
     * Logs in to the chat server with the user ID and token.
     * 
     * This method supports automatic login.
     *
     * You can also call {@link #login(String, String, CallBack)} to log in to the chat server with the user ID and password.
     *
     * This is an asynchronous method.
     *
     * @param username The user ID. Make sure to set the parameter.
     * @param token    The user token. Make sure to set this parameter.
     * @param callback The login callback. Make sure to set the parameter. Also, this parameter cannot be `null`. The result of login is returned via the callback.
     */
    public void loginWithToken(String username, String token, @NonNull final CallBack callback) {
        if (TextUtils.isEmpty(getChatConfigPrivate().getAppKey())) {
            callback.onError(Error.INVALID_PARAM, "please setup your App Key  either in AndroidManifest.xml or through the ChatOptions");
            return;
        }

        if (username == null || token == null || username.equals("") || token.equals("")) {
            callback.onError(Error.INVALID_PARAM, "username or token is null or empty!");
            return;
        }

        if (!sdkInited) {
            callback.onError(Error.GENERAL_ERROR, "sdk not initialized");
            return;
        }

        username = username.toLowerCase();
        initLoginWithAgoraData(false,"",0);
        _login(username, token, callback, false, EMLoginType.LOGIN_TOKEN);
    }

    /**
     * \~english
     * Logs in to the chat server with the user ID and Agora token.
     * 
     * This method supports automatic login.
     *
     * An app user can also log in to the chat server with the user ID and token. See {@link #login(String, String, CallBack)}.
     *
     * This an asynchronous method.
     *
     * @param username      The user ID. Make sure to set the parameter.
     * @param agoraToken    The Agora token. Make sure to set the parameter.
     * @param callback      The login callback. Make sure to set the parameter. The result of login is returned via the callback.
     */
    public void loginWithAgoraToken(String username, String agoraToken, @NonNull final CallBack callback){
        if (TextUtils.isEmpty(getChatConfigPrivate().getAppKey())) {
            callback.onError(Error.INVALID_PARAM, "please setup your App Key  either in AndroidManifest.xml or through the ChatOptions");
            return;
        }

        if (TextUtils.isEmpty(username) ||TextUtils.isEmpty(agoraToken)) {
            callback.onError(Error.INVALID_PARAM, "username or agoraToken is null or empty!");
            return;
        }

        if (!sdkInited) {
            callback.onError(Error.GENERAL_ERROR, "sdk not initialized");
            return;
        }
        _login(username.toLowerCase(), agoraToken, callback, false, EMLoginType.LOGIN_AGORA_TOKEN);
    }

    private synchronized void getChatToken(String username, String agoraToken, final CallBack callback){
        EMAError error = new EMAError();
        String response = emaObject.getChatTokenbyAgoraToken(agoraToken,error);
        if (error.errCode() == Error.EM_NO_ERROR) {
            EMLog.d(TAG, "getChatTokenbyAgoraToken success");
            if(response != null && response.length() > 0){
                try {
                    JSONObject object = new JSONObject(response);
                    if(object != null){
                        String chatToken = object.optString("access_token");
                        String expireTimestamp = object.optString("expire_timestamp");
                        long tokenAvailablePeriod=Long.valueOf(expireTimestamp)-System.currentTimeMillis();
                        if (TextUtils.isEmpty(chatToken)) {
                            throw new Exception("chatToken  is null or empty!");
                        }

                        initLoginWithAgoraData(true,expireTimestamp,tokenAvailablePeriod);
                        //登录
                        _login(username.toLowerCase(), chatToken, callback, false, EMLoginType.LOGIN_AGORA_TOKEN);
                    }
                }catch (Exception e){
                    EMLog.e(TAG, "getChatTokenbyAgoraToken Exception:"+e.getMessage());
                    callback.onError(Error.GENERAL_ERROR,"getChatTokenbyAgoraToken Exception:"+e.getMessage());
                }
            }else{
                EMLog.e(TAG, "getChatTokenbyAgoraToken response is null");
              callback.onError(Error.GENERAL_ERROR,"getChatTokenbyAgoraToken response is null or empty!");
            }
        } else {
            callback.onError(error.errCode(), error.errMsg());
            EMLog.e(TAG, "getChatTokenbyAgoraToken failed error:" + error.errCode() + "  errorMessage:" + error.errMsg());
        }
    }

    private void initLoginWithAgoraData(boolean isLoginWithAgoraToken, String expireTimestamp, long tokenAvailablePeriod) {
        mIsLoginWithAgoraToken =isLoginWithAgoraToken;
        if(isLoginWithAgoraToken) {
            //存储
            EMSessionManager.getInstance().setLoginWithAgoraData(isLoginWithAgoraToken,expireTimestamp,tokenAvailablePeriod);
        }else{
            EMSessionManager.getInstance().clearLoginWithAgoraTokenData();
        }
    }

    /**
     * \~english
     *
     * Notifies that the token expires.
     *
     * The SDK triggers the token expiry notification callback via connectionListener.
     *
     * @param response The request result, which includes the description of the issue that cause the method fails.
     */
    public void notifyTokenExpired(String response){
        //只在声网 Token 方式登录下才通知。
        if(connectionListener!=null&&mIsLoginWithAgoraToken) {
            //解析
            try {
                JSONObject object = new JSONObject(response);
                if(object!=null) {
                    String errorDescription = object.optString("error_description");
                    EMLog.e(TAG,"notifyTokenExpired--errorDescription:"+errorDescription);
                    if(errorDescription.contains("milliseconds ago")||errorDescription.contains("has expired")||
                            errorDescription.contains("Unable to authenticate due to expired access Token")) {
                        connectionListener.onTokenNotification(401);
                        EMLog.e(TAG,"notifyTokenExpired--onTokenNotification(401) ");
                    }
                }
            } catch (JSONException e) {
                e.printStackTrace();
            }
        }
    }
    /**
     * \~english
     *
     * Renews the Agora token.
     *
     * If you log in with an Agora token and are notified by a callback method {@link ConnectionListener} that the token is to be expired, you can call this method to update the token to avoid unknown issues caused by an invalid token.
     *
     * @param newAgoraToken The new token.
     */
    public void renewToken(String newAgoraToken){
        if(mIsLoginWithAgoraToken&& emaObject.isLoggedIn()) {//只有处于声网登录方式且处于登录状态时才更新Token
            execute(new Runnable() {
                @Override
                public void run() {
                    getChatToken(getCurrentUser(), newAgoraToken, new CallBack() {
                        @Override
                        public void onSuccess() {
                        }

                        @Override
                        public void onError(int code, String error) {
                        }

                        @Override
                        public void onProgress(int progress, String status) {
                        }
                    });
                }
            });
        }else{
           EMLog.e(TAG,"the method  excepted to be called when login by agoraToken and login state is loggeIn");
        }

    }

    /**
     * \~english
     * Logs out of the chat app.
     *
     * This is a synchronous method and blocks the current thread.
     *
     * @param unbindToken Whether to unbind the token upon logout.
     *                    - `true`: Yes.
         *                - `false`: No.
     * @return The logout result.
     *         - If success, the SDK returns {@link Error#EM_NO_ERROR} if the user successfully logs out;
     *         - If a failure occurs, the SDK returns the description of the cause to the failure. See {@link Error}.
     */
    public int logout(boolean unbindToken) {
        if (!emaObject.isLogout()) {
            String pushToken = EMPreferenceUtils.getInstance().getPushToken();
            String pushNotifierName = EMPreferenceUtils.getInstance().getPushNotifierName();
            if(!TextUtils.isEmpty(pushToken) && !TextUtils.isEmpty(pushNotifierName)) {
                try {
                    pushManager().unBindDeviceToken();
                } catch (Exception e) {
                    return Error.USER_UNBIND_DEVICETOKEN_FAILED;
                }
            }else {
                boolean result = PushHelper.getInstance().unregister(unbindToken);
                if (!result) {
                    return Error.USER_UNBIND_DEVICETOKEN_FAILED;
                }
            }
        } else {
            PushHelper.getInstance().unregister(false);
            EMLog.e(TAG, "already logout, skip unbind token");
        }

        logout();

        return Error.EM_NO_ERROR;
    }

    /**
     * \~english
     * Logs out of the chat server.
     * 
     * This is a synchronous method and blocks the current thread.
     * 
     * For the asynchronous method, see {@link ChatClient#logout(CallBack callback)}.
     */
    void logout() {
        EMLog.d(TAG, " SDK Logout");

        try {
            if (connectivityBroadcastReceiver != null) {
                mContext.unregisterReceiver(connectivityBroadcastReceiver);
            }
        } catch (Exception e) {
        }

        EMSessionManager.getInstance().clearLastLoginUser();
        EMSessionManager.getInstance().clearLastLoginToken();
        EMSessionManager.getInstance().clearLoginWithAgoraTokenData();

        // ============ gaode code start ===============
        // EMGDLocation.getInstance().onDestroy();
        // ============ gaode code end ===============

        if (smartHeartbeat != null) {
            smartHeartbeat.stop();
        }

        releaseWakelock();

        // make sure we clear the username and pwd in a sync way
        if (emaObject != null) {
            emaObject.logout();
        }
        if (chatManager != null) {
            chatManager.onLogout();
        }
        if (groupManager != null) {
            groupManager.onLogout();
        }
        if (contactManager != null) {
            contactManager.onLogout();
        }
        if (chatroomManager != null) {
            chatroomManager.onLogout();
        }

        // clear all memory cache
        try {
            EMAdvanceDebugManager.getInstance().onDestroy();
        } catch (Exception e) {
            e.printStackTrace();
        }

        if (EMChatConfigPrivate.isDebugTrafficMode()) {
            EMNetworkMonitor.stopTrafficStat();
        }
    }

    /**
     * \~english
     * Logs out of the chat server.
     *
     * This is an asynchronous method.
     *
     * @param unbindToken  Whether to unbind the token upon logout.
     *                     - `true`: Yes;
     *                     - `false`: No.
     * @param callback     The completion callback, which contains the error message if the method call fails.
     */
    public void logout(final boolean unbindToken, final CallBack callback) {
        new Thread() {
            @Override
            public void run() {
                int error = logout(unbindToken);

                if (error != Error.EM_NO_ERROR) {
                    if (callback != null) {
                        callback.onError(error, "faild to unbind device token");
                    }
                } else {
                    if (callback != null) {
                        callback.onSuccess();
                    }
                }
            }
        }.start();
    }

    /**
     * \~english
     * Logs out of the chat server.
     * 
     * This is an asynchronous method.
     *
     * @param callback     The completion callback, which contains the error message if the method fails.
     */
    void logout(final CallBack callback) {
        Thread logoutThread = new Thread() {
            @Override
            public void run() {
                if (callback != null) {
                    callback.onProgress(0, null);
                }

                logout();

                if (callback != null) {
                    callback.onSuccess();
                }
            }
        };
        logoutThread.setPriority(Thread.MAX_PRIORITY - 1);
        logoutThread.start();
    }

    /**
     * \~english
     * Update the App Key.
     *  
     * **Note**
     *
     * - As this key controls access to the chat service for your app, you can only update the key when the current user logs out.
     *
     * - Updating the App Key means to switch to a new App Key.
     *
     * - You can retrieve the new App Key from the Console.
     * 
     * - You can set App Key by calling {@link ChatOptions#setAppKey(String)} when you log out of the chat service.
     *
     * @param appkey  The App Key, make sure to set the param.
     */
    public void changeAppkey(String appkey) throws ChatException {
        EMAError error = emaObject.changeAppkey(appkey);
        if (error.errCode() == EMAError.EM_NO_ERROR) {
            this.getOptions().updatePath(appkey);
        }
        handleError(error);
    }

    /**
     * \~english
     * Adds a connection status listener.
     *
     * The listener listens for the connection between the chat app and the chat server.
     *
     * @param listener The connection status listener to add.
     *                 - {@link ConnectionListener#onConnected()} indicates a successful connection to the chat server.
     *                 - {@link ConnectionListener#onDisconnected(int)} indicates the connection to the chat server fails. It contains the error code. See {@link Error}.
     */
    public void addConnectionListener(final ConnectionListener listener) {
        if (listener == null) {
            return;
        }

        synchronized (connectionListeners) {
            if (!connectionListeners.contains(listener)) {
                connectionListeners.add(listener);
            }
        }

        execute(new Runnable() {

            @Override
            public void run() {
                if (isConnected()) {
                    listener.onConnected();
                } else {
                    if (isLoggedIn()) {
                        listener.onDisconnected(Error.NETWORK_ERROR);
                    }
                }
            }
        });
    }

    /**
     * \~english
     * Removes the connection status listener.
     *
     * @param listener  The connection status listener to remove.
     */
    public void removeConnectionListener(final ConnectionListener listener) {
        if (listener == null) {
            return;
        }
        synchronized (connectionListeners) {
            connectionListeners.remove(listener);
        }
    }

    /**
     * \~english
     * Adds the log callback listener of SDK.
     *
     * @param listener The log callback listener, {@link ChatLogListener#onLog(String)}.
     */
    public void addLogListener(final ChatLogListener listener) {
        if (listener == null) {
            return;
        }
        synchronized (logListeners) {
            if (!logListeners.contains(listener)) {
                logListeners.add(listener);
            }
        }

        if(clientLogListener==null) {
            clientLogListener = new ClientLogListener();
            emaObject.addLogCallbackListener(clientLogListener);
        }
    }

    /**
     * \~english
     * Removes the log callback listener.
     *
     * @param listener  The log callback listener.
     */
    public void removeLogListener(final ChatLogListener listener) {
        if (listener == null) {
            return;
        }
        synchronized (logListeners) {
            logListeners.remove(listener);
        }
    }

    /**
     * \~english
     * Gets the `GroupManager` class.
     *
     * This method can be called only after the ChatClient is initialized. See {@link #init(Context, ChatOptions)}.
     *
     * @return The `GroupManager` class.
     */
    public GroupManager groupManager() {
        if (groupManager == null) {
            synchronized (ChatClient.class) {
                if(groupManager == null) {
                    groupManager = new GroupManager(this, emaObject.getGroupManager());
                }
            }
        }
        return groupManager;
    }

    /**
     * \~english
     * Gets the `PushManager` class.
     *
     * This method can be called only after the ChatClient is initialized. See {@link #init(Context, ChatOptions)}.
     *
     * @return The `PushManager` class.
     */
    public PushManager pushManager() {
        if (pushManager == null) {
            synchronized (ChatClient.class) {
                if(pushManager == null) {
                    pushManager = new PushManager(this, emaObject.getPushMnager());
                }
            }
        }
        return pushManager;
    }

    /**
     * \~english
     * Gets the `ChatRoomManager` class.
     *
     * This method can be called only after the ChatClient is initialized. See {@link #init(Context, ChatOptions)}.
     *
     * @return The `ChatRoomManager` class.
     */
    public ChatRoomManager chatroomManager() {
        if (chatroomManager == null) {
            synchronized (ChatClient.class) {
                if(chatroomManager == null) {
                    chatroomManager = new ChatRoomManager(this, emaObject.getChatRoomManager());
                }
            }
        }
        return chatroomManager;
    }

    /**
     * \~english
     * Gets the `ChatManager` class.
     *
     * This method can be called only after the ChatClient is initialized. See {@link #init(Context, ChatOptions)}.
     *
     * @return The `ChatManager` class.
     */
    public ChatManager chatManager() {
        if (chatManager == null) {
            synchronized (ChatClient.class) {
                if(chatManager == null) {
                    chatManager = new ChatManager(this, emaObject.getChatManager(), emaObject.getReactionManager());
                }
            }
        }
        return chatManager;
    }


    /**
     * \~english
     * Gets the `UserInfoManager` class.
     *
     * This method can be called only after the ChatClient is initialized. See {@link #init(Context, ChatOptions)}.
     *
     * @return The `UserInfoManager` class.
     */
    public UserInfoManager userInfoManager() {
        if (userInfoManager == null) {
            synchronized (ChatClient.class) {
                if(userInfoManager == null) {
                    userInfoManager = new UserInfoManager(emaObject.getUserInfoManager());
                }
            }
        }
        return userInfoManager;
    }



    /**
     * \~english
     * Gets the `ContactManager` class.
     *
     * This method can be called only after the ChatClient is initialized. See {@link #init(Context, ChatOptions)}.
     *
     * @return The `ContactManager` class.
     */
    public ContactManager contactManager() {
        if (contactManager == null) {
            synchronized (ChatClient.class) {
                if(contactManager == null) {
                    contactManager = new ContactManager(this, emaObject.getContactManager());
                }
            }
        }
        return contactManager;
    }

    public TranslationManager translationManager(){
        if(translationManager == null){
            synchronized (ChatClient.class) {
                if(translationManager == null){
                    translationManager = new TranslationManager(emaObject.getTranslateManager());
                }
            }
        }
        return translationManager;
    }

    public PresenceManager presenceManager(){
        if(presenceManager == null){
            synchronized (ChatClient.class) {
                if(presenceManager == null){
                    presenceManager = new PresenceManager(emaObject.getPresenceManager());
                }
            }
        }
        return presenceManager;
    }

    /**
     * \~english
     * Gets the `ChatThreadManager` class.
     *
     * This method can be called only after the ChatClient is initialized. See {@link #init(Context, ChatOptions)}.
     *
     * @return The `ChatThreadManager` class.
     */
    public ChatThreadManager chatThreadManager(){
        if(threadManager == null){
            synchronized (ChatClient.class) {
                if(threadManager == null){
                    threadManager = new ChatThreadManager(this, emaObject.getThreadManager());
                }
            }
        }
        return threadManager;
    }

    public ChatStatisticsManager statisticsManager(){
        if(statisticsManager == null){
            synchronized (ChatClient.class) {
                if(statisticsManager == null){
                    statisticsManager = new ChatStatisticsManager(this, emaObject.getStatisticsManager());
                }
            }
        }
        return statisticsManager;
    }

    public Context getContext() {
        return mContext;
    }

    /**
     * \~english
     * Gets the user ID of the current logged-in user.
     *
     * @return The user ID of the current logged-in user.
     */
    public synchronized String getCurrentUser() {
        if (EMSessionManager.getInstance().currentUser == null ||
                EMSessionManager.getInstance().currentUser.username == null ||
                EMSessionManager.getInstance().currentUser.username.equals("")) {
            return EMSessionManager.getInstance().getLastLoginUser();
        }
        return EMSessionManager.getInstance().currentUser.username;
    }

    /**
     * \~english
     * Get a token by using the user ID and password.
     *
     * @param username The user ID.
     * @param password The password.
     * 
     * @param callBack The result callback：  
     *                 - If success, the SDK calls {@link ValueCallBack#onSuccess(Object)}, where the parameter is the token;
     *                 - If a failure occurs, the SDK calls {@link ValueCallBack#onError(int, String)}, in which the first parameter is the error code, and the second is the error description.
     */
    public void getUserTokenFromServer(final String username, final String password, final ValueCallBack<String> callBack) {
        execute(new Runnable() {
            @Override
            public void run() {
                EMAError error = new EMAError();
                final String token = emaObject.getUserTokenFromServer(username, password, error);

                if (callBack == null) {
                    return;
                }

                if (error.errCode() == Error.EM_NO_ERROR) {
                    callBack.onSuccess(token);
                } else {
                    callBack.onError(error.errCode(), error.errMsg());
                }
            }
        });
    }

    /**
     * \~english
     * Checks whether the user has logged in before.
     * 
     * This method always returns `true` if you log in successfully and have not called the {@link #logout()} method yet.
     *
     * If you need to check whether the SDK is connected to the server, call {@link #isConnected()}.
     *
     * ```java
     * if(ChatClient.getInstance().isLoggedInBefore()){
     *     // Enter the main activity.
     * }else{
     *     // Enter the login activity.
     * }
     * ```
     *
     * @return Whether the user has logged in before.
     *         - `true`: The user has logged in before;
     *         - `false`: The user has not logged in before or has called the {@link #logout()} method.
     */
    public boolean isLoggedInBefore() {
        EMSessionManager sessionMgr = EMSessionManager.getInstance();
        String user = sessionMgr.getLastLoginUser();
        // no need decrypt password or token when user empty
        if (TextUtils.isEmpty(user)) return false;

        String pwd = sessionMgr.getLastLoginPwd();
        String token = sessionMgr.getLastLoginToken();

        if (user != null  && !user.isEmpty() &&
                ((pwd != null   && !pwd.isEmpty()) ||
                 (token != null && !token.isEmpty())) ) {
            return true;
        }

        return false;
    }


    /**
     * \~english
     * Checks whether the SDK is connected to the chat server.
     *
     * @return Whether the SDK is connected to the chat server.
     *         - `true`: Yes;
     *         - `false`: No.
     */
    public boolean isConnected() {
        return emaObject.isConnected();
    }

    /**
     * \~english
     * Checks whether the user has logged in to the Chat app.
     *
     * @return Whether the user has logged in to the Chat app.
     *         - `true`: Yes;
     *         - `false`: No.
     */
    public boolean isLoggedIn() {
        return emaObject.isLoggedIn();
    }

    /**
     * \~english
     * Sets whether to output the debug information.
     * 
     * This method can be called only after the ChatClient is initialized. See {@link #init(Context, ChatOptions)}.
     *
     * @param debugMode  Whether to output the debug information:
     *                   - `true`: Yes;
     *                   - `false`: No.
     */
    public void setDebugMode(boolean debugMode) {
        if (sdkInited) {
            //if you have set debugmode, use it.
            String mode = EMAdvanceDebugManager.getInstance().getDebugMode();
            if (mode != null)
                debugMode = Boolean.parseBoolean(mode);
        }
        EMLog.debugMode = debugMode;
        getChatConfigPrivate().setDebugMode(debugMode);
    }

    /**
     * \~english
     * Uploads local logs.
     *
     * The debug logs provide reference for our engineers to fix errors and improve system performance.
     *
     * This is a synchronous method and blocks the current thread.
     *
     * @param callback Reserved parameter.
     */
    public void uploadLog(CallBack callback) {
        chatManager().emaObject.uploadLog();
    }


    /**
     * \~english
     * Gets configuration options of the SDK.
     */
    public ChatOptions getOptions() {
        return mChatConfigPrivate.getOptions();
    }


    /**
     * \~english
     * Compresses the debug log file into a gzip archive.
     *
     * It is strongly recommended that you delete this debug archive as soon as this method is called.

     * @return The path of the compressed gz file.
     * @throws ChatException  A description of the cause of the exception if the method fails.
     */
    public String compressLogs() throws ChatException {
        EMAError error = new EMAError();
        String path = emaObject.compressLogs(error);
        handleError(error);
        return path;
    }

    /**
     * \~english
     * Adds the multi-device listener.
     * 
     * @param listener The multi-device listener to add.
     *                See {@link MultiDeviceListener}. {@link MultiDeviceListener#onContactEvent(int, String, String)} is the contact event callback and
     *                 {@link MultiDeviceListener#onGroupEvent(int, String, List)} is the group event callback.
     */
    public void addMultiDeviceListener(MultiDeviceListener listener) {
        multiDeviceListeners.add(listener);
    }

    /**
     * \~english
     * Removes the multi-device listener.
     *
     * @param listener The multi-device listener to remove. See {@link MultiDeviceListener}.
     */
    public void removeMultiDeviceListener(MultiDeviceListener listener) {
        multiDeviceListeners.remove(listener);
    }

    /**
     * \~english
     * Gets the list of online devices to which you have logged in with a specified account.
     *
     * This is a synchronous method and blocks the current thread.
     *
     * @param username The user ID.
     * @param password The password.
     * @return The list of online devices.
     * @throws ChatException A description of the exception, see {@link Error}.
     */
    public List<DeviceInfo> getLoggedInDevicesFromServer(String username, String password) throws ChatException {
        EMAError error = new EMAError();
        List<EMADeviceInfo> devices = emaObject.getLoggedInDevicesFromServer(username, password, error);
        handleError(error);
        List<DeviceInfo> result = new ArrayList<>();
        for (EMADeviceInfo info : devices) {
            result.add(new DeviceInfo(info));
        }
        return result;
    }

    /**
     * \~english
     * Gets the list of online devices to which you have logged in with a specified account.
     *
     * This is a synchronous method and blocks the current thread.
     *
     * @param username The user ID.
     * @param token The token.
     * @return The list of online devices.
     * @throws ChatException A description of the exception, see {@link Error}.
     */
    public List<DeviceInfo> getLoggedInDevicesFromServerWithToken(@NonNull String username, @NonNull String token) throws ChatException {
        EMAError error = new EMAError();
        List<EMADeviceInfo> devices = emaObject.getLoggedInDevicesFromServerWithToken(username, token, error);
        handleError(error);
        List<DeviceInfo> result = new ArrayList<>();
        for (EMADeviceInfo info : devices) {
            result.add(new DeviceInfo(info));
        }
        return result;
    }

    /**
     * \~english
     * Logs out from a specified account on a device.
     *
     * You can call {@link DeviceInfo#getResource()} to get the device ID.
     *
     * This is a synchronous method and blocks the current thread.
     *
     * @param username The user ID.
     * @param password The password.
     * @param resource The device ID. See {@link DeviceInfo#getResource()}.
     * @throws ChatException A description of the exception if the method fails, see {@link Error}.
     */
    public void kickDevice(String username, String password, String resource) throws ChatException {
        EMAError error = new EMAError();
        emaObject.kickDevice(username, password, resource, error);
        handleError(error);
    }

    /**
     * \~english
     * Logs out from a specified account on a device.
     *
     * You can call {@link DeviceInfo#getResource()} to get the device ID.
     *
     * This is a synchronous method and blocks the current thread.
     *
     * @param username The user ID.
     * @param token The token.
     * @param resource The device ID. See {@link DeviceInfo#getResource()}.
     * @throws ChatException A description of the exception if the method fails, see {@link Error}.
     */
    public void kickDeviceWithToken(@NonNull String username,@NonNull String token, String resource) throws ChatException {
        EMAError error = new EMAError();
        emaObject.kickDeviceWithToken(username, token, resource, error);
        handleError(error);
    }

    /**
     * \~english
     * Logs out from a specified account on all devices.
     *
     * This is a synchronous method and blocks the current thread.
     *
     * @param username The user ID.
     * @param password The password.
     * @throws ChatException A description of the exception, see {@link Error}.
     */
    public void kickAllDevices(String username, String password) throws ChatException {
        EMAError error = new EMAError();
        emaObject.kickAllDevices(username, password, error);
        handleError(error);
    }

    /**
     * \~english
     * Logs out from a specified account on all devices.
     *
     * This is a synchronous method and blocks the current thread.
     *
     * @param username The user ID.
     * @param token The token.
     * @throws ChatException A description of the exception, see {@link Error}.
     */
    public void kickAllDevicesWithToken(@NonNull String username, @NonNull String token) throws ChatException {
        EMAError error = new EMAError();
        emaObject.kickAllDevicesWithToken(username, token, error);
        handleError(error);
    }

    /**
     * \~english
     * Uploads the FCM token to the chat server.
     * 
     * The token can be uploaded when the following conditions are met:
     * - The token is not empty;
     * - The user has logged in to the Chat app;
     * - The current device supports Google PUSH service;
     * - The push type is FCM {@link PushType#FCM}.
     * 
     * @param fcmToken The token to upload.
     */
    public void sendFCMTokenToServer(String fcmToken) {
        EMLog.d(TAG, "sendFCMTokenToServer: " + fcmToken);
        if (TextUtils.isEmpty(fcmToken)) {
            return;
        }

        // If no user logged in, stop upload the fcm Token.
        String userName = getCurrentUser();
        if (TextUtils.isEmpty(userName)) {
            EMLog.i(TAG, "No user login currently, stop upload the token.");
            return;
        }
        // let's check if the push service available
        PushType pushType = PushHelper.getInstance().getPushType();
        EMLog.i(TAG, "pushType: " + pushType);
        if (pushType == PushType.FCM) {
            PushHelper.getInstance().onReceiveToken(pushType, fcmToken);
        }
    }

    /**
     * \~english
     * Sends the HUAWEI Push token to the server.
     * @param token    The Huawei Push token.
     */
    public void sendHMSPushTokenToServer(String token){
        if (PushHelper.getInstance().getPushType() == PushType.HMSPUSH) {
            PushHelper.getInstance().onReceiveToken(PushType.HMSPUSH, token);
        }
    }

    /**
     * \~english
     * Sends the Honor Push token to the server.
     * @param token    The Honor Push token.
     */
    public void sendHonorPushTokenToServer(String token){
        if (PushHelper.getInstance().getPushType() == PushType.HONORPUSH) {
            PushHelper.getInstance().onReceiveToken(PushType.HONORPUSH, token);
        }
    }

	//---------------------------private methods ---------------------------------

	private void initManagers(){
		// invoke constructor before login, to listener PB event
		EMHttpClient.getInstance().onInit(mChatConfigPrivate);
		chatManager();
		contactManager();
		groupManager();
		chatroomManager();
        presenceManager();
        chatThreadManager();

		setNatvieNetworkCallback();
		EMSessionManager.getInstance().init(this, emaObject.getSessionManager());
	}

	void _login(final String username, final String code, final CallBack callback, final boolean autoLogin, final EMLoginType loginType) {
        if (getChatConfigPrivate() == null || sdkInited == false) {
            callback.onError(Error.GENERAL_ERROR, "");
            return;
        }

        EMLog.e(TAG, "emchat manager login in process:" + android.os.Process.myPid());

        // convert to lowercase
        execute(new Runnable() {

            @Override
            public void run() {
                EMLog.e(TAG, "emchat manager login in process:" + android.os.Process.myPid() + " threadName:" + Thread.currentThread().getName() + " ID:" + Thread.currentThread().getId());

                if (username == null) {
                    callback.onError(Error.INVALID_USER_NAME, "Invalid user name");
                    return;
                }

                EMAError error = new EMAError();
                emaObject.login(username, code, autoLogin, loginType.ordinal(), error);

                if (error.errCode() == EMAError.EM_NO_ERROR) {
                    EMSessionManager.getInstance().setLastLoginUser(username);
                    if (loginType == EMLoginType.LOGIN_TOKEN || loginType == EMLoginType.LOGIN_AGORA_TOKEN) {
                        saveToken();
                    } else {
                        EMSessionManager.getInstance().setLastLoginPwd(code);
                        EMSessionManager.getInstance().setLastLoginWithToken(false);
                        EMSessionManager.getInstance().clearLastLoginToken();
                    }
                    // Move from before login, do not open db first
                    if(autoLogin) {
                        loadDataFromDb();
                    }

                    onNewLogin();
                    PushHelper.getInstance().register();
                    // set EMConferenceManager parameters, lite jar build need to remove this line
//                    conferenceManager().set(getAccessToken(), getOptions().getAppKey(), getCurrentUser());
                    callback.onSuccess();
                } else {
                    callback.onError(error.errCode(), error.errMsg());
                    if(error.errCode()==EMAError.USER_ALREADY_LOGIN&& mIsLoginWithAgoraToken) {
                        //进程杀死又立马开启，会走到这里，renewToken 也会走这里。
                        //更新 Token
                        emaObject.renewToken(code);
                        EMSessionManager.getInstance().setLastLoginToken(code);
                        //此时计时器可能失效需要重新开启。
                        EMSessionManager.getInstance().startCountDownTokenAvailableTime(connectionListener);
                    }
                }

                if (error.errCode() == EMAError.USER_AUTHENTICATION_FAILED) {
                    EMSessionManager.getInstance().clearLastLoginPwd();
                    EMSessionManager.getInstance().clearLastLoginToken();
                }
            }
        });
    }

    private void loadDataFromDb() {
        execute(()-> {
            groupManager().loadAllGroups();
            chatManager().loadAllConversationsFromDB();
        });
    }

    /**
     * \~english
     * When the login method is Agora token, after the login is successful, the replacement to Chat token may not be returned.
     * For the logic of automatic login, this method needs to be called again in the {@link MyConnectionListener#onReceiveToken(String, long)} method.
     * Currently {@link MyConnectionListener#onReceiveToken(String, long)} will only call back when logging in with Agora token.
     */
    private void saveToken() {
        if(emaObject == null) {
            return;
        }
        EMAError emaError = new EMAError();
        String token = emaObject.getUserToken(false, emaError);
        if (emaError.errCode() == EMAError.EM_NO_ERROR) {
            EMSessionManager.getInstance().setLastLoginToken(token);
            EMSessionManager.getInstance().setLastLoginWithToken(true);
            EMSessionManager.getInstance().clearLastLoginPwd();
            if(mIsLoginWithAgoraToken) {
                EMSessionManager.getInstance().startCountDownTokenAvailableTime(connectionListener);
            }
        }
    }

    /**
     * \~english
     * Checks whether the FCM push is available.
     *
     * @return  Whether the FCM push is available:
     *          - `true`: Yes;
     *          - `false`: No.
     */
    public boolean isFCMAvailable() {
        return PushHelper.getInstance().getPushType() == PushType.FCM;
    }

    void onNewLogin() {
        EMLog.d(TAG, "on new login created");

        String username = EMSessionManager.getInstance().getLastLoginUser();

        PathUtil.getInstance().initDirs(getChatConfigPrivate().getAppKey(), username, mContext);

        EMAdvanceDebugManager.getInstance().onInit(mChatConfigPrivate);

        if (smartHeartbeat == null) {
            smartHeartbeat = EMSmartHeartBeat.create(mContext);
        }

        if (getChatConfigPrivate().emaObject.hasHeartBeatCustomizedParams()) {
            smartHeartbeat.setCustomizedParams(getChatConfigPrivate().emaObject.getWifiHeartBeatCustomizedParams(),
                    getChatConfigPrivate().emaObject.getMobileHeartBeatCustomizedParams());
        }
        smartHeartbeat.onInit();
        if (getOptions().getFixedInterval() != -1) {
            smartHeartbeat.setFixedInterval(getOptions().getFixedInterval());
        }
    }

    /**
     * \~english
     * Gets the access token from the memory.
     * 
     * When uploading or downloading an attachment (a voice, image, or file), you must add the token to the request header. The SDK returns `null` when any exception occurs.
     * 
     * If the token is `null`, you can check the EMLog file for the possible reason.
     *
     * You can also get the token from the server by calling {@link ChatOptions#getAccessToken(boolean)} and passing `true`.
     *
     * @return  The access token.
     */
    public String getAccessToken() {
        return getChatConfigPrivate().getAccessToken();
    }

    /**
     * \~english
     * Checks whether the SDK is initialized.
     *
     * @return   Whether the SDK is initialized:
     *           - `true`: Yes.
     *           - `false`: No.
     */
    public boolean isSdkInited() {
        return sdkInited;
    }

    private boolean _loadLibrary(final String library, boolean trace) {
        try {
            ReLinker.loadLibrary(mContext, library);
            return true;
        } catch (Throwable e) {
            if (trace) {
                e.printStackTrace();
            }
        }
        return false;
    }

    private boolean _loadLibrary(final String library) {
        return _loadLibrary(library, true);
    }

    private void loadLibrary() {
        if (libraryLoaded == false) {
            //if (_loadLibrary("sqlcipher", false) || _loadLibrary("sqlite")) {}
            //_loadLibrary("sqlite");
            _loadLibrary("cipherdb", false);
//            _loadLibrary("hyphenate_av");
//            _loadLibrary("hyphenate_av_recorder");
//            ReLinker.loadLibrary(mContext, "crypto");
//            ReLinker.loadLibrary(mContext, "ssl");
            ReLinker.loadLibrary(mContext, "agora-chat-sdk");
            libraryLoaded = true;
        }
    }

    class MyConnectionListener extends EMAConnectionListener {

        @Override
        public void onConnected() {
            execute(new Runnable() {

                @Override
                public void run() {
                    synchronized (connectionListeners) {
                        try {
                            for (ConnectionListener listener : connectionListeners) {
                                listener.onConnected();
                            }
                        } catch (Exception e) {
                            e.printStackTrace();
                        }
                    }
                }
            });
        }

        @Override
        public void onDisconnected(final int errCode,String info) {
            EMLog.e(TAG,"onDisconnected errcode = "+errCode+",info = "+info);
            execute(new Runnable() {

                @Override
                public void run() {
                    synchronized (connectionListeners) {
                        switch (errCode) {
                            case Error.USER_REMOVED:
                                EMSessionManager.getInstance().clearLastLoginUser();
                                EMSessionManager.getInstance().clearLastLoginToken();
                                EMSessionManager.getInstance().clearLastLoginPwd();
                                break;
                            case Error.USER_LOGIN_ANOTHER_DEVICE:
                            case Error.USER_BIND_ANOTHER_DEVICE:
                            case Error.USER_DEVICE_CHANGED:
                            case Error.SERVER_SERVICE_RESTRICTED:
                            case Error.USER_LOGIN_TOO_MANY_DEVICES:
                            case Error.USER_KICKED_BY_CHANGE_PASSWORD:
                            case Error.USER_KICKED_BY_OTHER_DEVICE:
                            case Error.APP_ACTIVE_NUMBER_REACH_LIMITATION:
                                EMSessionManager.getInstance().clearLastLoginToken();
                                EMSessionManager.getInstance().clearLastLoginPwd();
                                if(isSdkInited()) {
                                    // clear push token info
                                    EMPreferenceUtils.getInstance().setPushNotifierName("");
                                    EMPreferenceUtils.getInstance().setPushToken("");
                                }
                                break;
                        }

                        try {
                            for (ConnectionListener listener : connectionListeners) {
                                listener.onDisconnected(errCode);
                            }
                        } catch (Exception e) {
                            e.printStackTrace();
                        }

                        if ((errCode == Error.USER_LOGIN_ANOTHER_DEVICE) ||
                                (errCode == Error.USER_REMOVED) ||
                                (errCode == Error.USER_BIND_ANOTHER_DEVICE) ||
                                (errCode == Error.USER_DEVICE_CHANGED) ||
                                (errCode == Error.SERVER_SERVICE_RESTRICTED) ||
                                (errCode == Error.USER_LOGIN_TOO_MANY_DEVICES) ||
                                (errCode == Error.USER_KICKED_BY_CHANGE_PASSWORD) ||
                                (errCode == Error.USER_KICKED_BY_OTHER_DEVICE)||
                                (errCode == Error.APP_ACTIVE_NUMBER_REACH_LIMITATION)){
                            for (ConnectionListener listener : connectionListeners) {
                                try {
                                    listener.onLogout(errCode,info);
                                } catch (Exception e) {
                                    e.printStackTrace();
                                }
                            }
                        }
                    }
				}
			});
		}

        @Override
        public boolean verifyServerCert(List<String> certschain,String domain) {
            if (certschain == null) {
                EMLog.d(TAG, "List<String> certschain : null ");
                return false;
            }
            if(TextUtils.isEmpty(domain)) {
                EMLog.d(TAG, "domain is empty or null ");
                return false;
            }
            EMLog.d(TAG, "domain = " + domain);
            X509Certificate[] certsArray = convertToCerts(certschain);
            try {
                X509TrustManager x509TrustManager = getSystemDefaultTrustManager();
                X509TrustManagerExtensions managerExtensions = new X509TrustManagerExtensions(x509TrustManager);
                managerExtensions.checkServerTrusted(certsArray, certsArray[0].getType(),domain);
            } catch (Exception e) {
                e.printStackTrace();
                EMLog.e(TAG, e.getMessage());
                EMLog.d(TAG, "List<String> certschain :" + certschain.toString());

                return false;
            }
            return true;
        }
        public void onTokenNotification(int code) {
            //从EMSessionManager::onDisconnect、io.agora.chat.EMSessionManager.checkTokenAvailability、ChatClient#notifyTokenExpired中通知过来
            execute(new Runnable() {
                @Override
                public void run() {
                    synchronized (connectionListeners) {
                        try {

                            if(code==Error.TOKEN_EXPIRED||code==401) {
                                logout();
                                for (ConnectionListener listener : connectionListeners) {
                                    EMLog.d(TAG,"MyConnectionListener onTokenExpired code: "+code);
                                    listener.onTokenExpired();
                                }
                            }else{
                                for (ConnectionListener listener : connectionListeners) {
                                    EMLog.d(TAG,"MyConnectionListener onTokenWillExpire code: "+code);
                                    listener.onTokenWillExpire();
                                }
                            }
                        } catch (Exception e) {
                            e.printStackTrace();
                            EMLog.e(TAG,"MyConnectionListener onTokenNotification Exception: "+e.getMessage());
                        }
                    }
                }
            });
        }

        /**
         * Come from Linux layer after exchanging with server
         * @param token
         * @param expireTimestamp
         */
        @Override
        public void onReceiveToken(String token, long expireTimestamp) {
            if(TextUtils.isEmpty(token) || expireTimestamp <= 0) {
                EMLog.e(TAG, "onReceiveToken: params received is invalid");
                return;
            }
            long tokenAvailablePeriod = expireTimestamp - System.currentTimeMillis();
            initLoginWithAgoraData(true, String.valueOf(expireTimestamp), tokenAvailablePeriod);
            saveToken();
        }

        @Override
        public void onDatabaseOpened(int errCode) {
            EMLog.e(TAG, "onDatabaseOpened: errCode: "+errCode);
        }
    }

    class MyMultiDeviceListener extends EMAMultiDeviceListener {
        @Override
        public void onContactEvent(final int event, final String target, final String ext) {
            EMLog.d(TAG, "onContactEvent:" + event + " target:" + target + " ext:" + ext);
            execute(new Runnable() {

                @Override
                public void run() {
                    synchronized (multiDeviceListeners) {
                        try {
                            for (MultiDeviceListener listener : multiDeviceListeners) {
                                listener.onContactEvent(event, target, ext);
                            }
                        } catch (Exception e) {
                            e.printStackTrace();
                        }
                    }
                }
            });
        }

        @Override
        public void onGroupEvent(final int event, final String target, final List<String> usernames) {
            EMLog.d(TAG, "onGroupEvent:" + event + " target:" + target + " usernames:" + usernames);
            execute(new Runnable() {

                @Override
                public void run() {
                    synchronized (multiDeviceListeners) {
                        try {
                            for (MultiDeviceListener listener : multiDeviceListeners) {
                                listener.onGroupEvent(event, target, usernames);
                            }
                        } catch (Exception e) {
                            e.printStackTrace();
                        }
                    }
                }
            });
        }

        @Override
        public void onRoamDeleteEvent(String conversation,List<String> msgIdList,String deviceId,long beforeTimeStamp) {
            EMLog.d(TAG, "onRoamDeleteEvent:" + conversation + " " + deviceId);
            if (TextUtils.isEmpty(conversation)) return;
            execute(new Runnable() {
                @Override
                public void run() {
                    synchronized (multiDeviceListeners) {
                        try {
                            if (msgIdList != null && msgIdList.size() > 0){
                                chatManager.clearCaches(conversation,msgIdList);
                            }else {
                                chatManager.clearCaches(conversation,beforeTimeStamp);
                            }
                            for (MultiDeviceListener listener : multiDeviceListeners) {
                                listener.onMessageRemoved(conversation,deviceId);
                            }
                        } catch (Exception e) {
                            e.printStackTrace();
                        }
                    }
                }
            });
        }

        @Override
        public void onThreadEvent(int event, String target, List<String> username) {
            EMLog.d(TAG, "onThreadEvent:" + event + " target:" + target + " usernames:" + username);
            execute(()-> {
                synchronized (multiDeviceListeners) {
                    for (MultiDeviceListener listener : multiDeviceListeners) {
                        try {
                            listener.onChatThreadEvent(event, target, username);
                        } catch (Exception e) {
                            e.printStackTrace();
                        }
                    }
                }
            });
        }

        @Override
        public void onConversationEvent(int event, String conversationId, int type) {
            EMLog.d(TAG, "onConversationEvent: "+event+" conversationId: "+conversationId+" type: "+type);
            final Conversation.ConversationType cType;
            if(type == Conversation.ConversationType.GroupChat.ordinal()) {
                cType = Conversation.ConversationType.GroupChat;
            }else {
                cType = Conversation.ConversationType.Chat;
            }
            execute(()-> {
                synchronized (multiDeviceListeners) {
                    for (MultiDeviceListener listener : multiDeviceListeners) {
                        try {
                            listener.onConversationEvent(event, conversationId, cType);
                        } catch (Exception e) {
                            e.printStackTrace();
                        }
                    }
                }
            });
        }
    }

    class ClientLogListener extends EMALogCallbackListener {
        @Override
        public void onLogCallback(String log) {
            logQueue.submit(()-> {
                synchronized (logListeners) {
                    for (ChatLogListener listener : logListeners) {
                        try {
                            listener.onLog(log);
                        } catch (Exception e) {
                            e.printStackTrace();
                        }
                    }
                }
            });
        }
    }

    void execute(Runnable runnable) {
        executor.execute(runnable);
    }

    void executeOnMainQueue(Runnable runnable) {
        mainQueue.submit(runnable);
    }

    void executeOnSendQueue(Runnable runnable) {
        sendQueue.submit(runnable);
    }

    public EMChatConfigPrivate getChatConfigPrivate() {
        return mChatConfigPrivate;
    }

    void setNatvieNetworkCallback() {
        EMANetCallback callback = new EMANetCallback() {

            @Override
            public int getNetState() {
                if (!NetUtils.hasDataConnection(mContext)) {
                    return EMANetwork.NETWORK_NONE.ordinal();
                } else {
                    if (NetUtils.isWifiConnected(mContext)
                            || NetUtils.isOthersConnected(mContext)) {
                        return EMANetwork.NETWORK_WIFI.ordinal();
                    } else if (NetUtils.isMobileConnected(mContext)) {
                        return EMANetwork.NETWORK_MOBILE.ordinal();
                    } else if (NetUtils.isEthernetConnected(mContext)) {
                        return EMANetwork.NETWORK_CABLE.ordinal();
                    } else {
                        return EMANetwork.NETWORK_NONE.ordinal();
                    }
                }
            }
        };

        mChatConfigPrivate.emaObject.setNetCallback(callback);
    }

    /**
     * \~english
     * Sets the encryption provider.
     *
     * If the encryption provider is not set, the SDK uses the default encryption algorithm.
     *
     * @param provider The encryption provider.
     */
    void setEncryptProvider(EMEncryptProvider provider) {
        this.encryptProvider = provider;
    }

    /**
     *  \~english
     * Gets the encryption provider.
     * 
     * If the encryption provider is not set, the SDK returns the default encryption algorithm.
     *
     * @return The encryption provider.
     */
    EMEncryptProvider getEncryptProvider() {
        if (encryptProvider == null) {
            EMLog.d(TAG, "encrypt provider is not set, create default");
            encryptProvider = new EMEncryptProvider() {

                public byte[] encrypt(byte[] input, String username) {
                    try {
                        String in = new String(input);
                        return emaObject.getSessionManager().encrypt(in).getBytes();
                    } catch (Exception e) {
                        e.printStackTrace();
                        return input;
                    }
                }

                public byte[] decrypt(byte[] input, String username) {
                    try {
                        String in = new String(input);
                        return emaObject.getSessionManager().decrypt(in).getBytes();
                    } catch (Exception e) {
                        e.printStackTrace();
                        return input;
                    }
                }

            };
        }
        return encryptProvider;
    }

    boolean sendPing(boolean waitPong, long timeout) {
        return emaObject.sendPing(waitPong, timeout);
    }

    void checkTokenAvailability(){
        if(mIsLoginWithAgoraToken) {
            EMSessionManager.getInstance().checkTokenAvailability(connectionListener);
        }
    }

    void forceReconnect() {
        EMLog.d(TAG, "forceReconnect");
        disconnect();
        reconnect();
    }

    void reconnect() {
        EMLog.d(TAG, "reconnect");
        try {
            wakeLock.acquire();
        } catch (Exception e) {
            EMLog.e(TAG, e.getMessage());
        }
        emaObject.reconnect();
        releaseWakelock();
    }

    void disconnect() {
        EMLog.d(TAG, "disconnect");
        emaObject.disconnect();
    }

    /**
     * release wake lock
     */
    private void releaseWakelock(){
        // Fix the bug: WakeLock under-locked emclient
        synchronized (this) {
            if (wakeLock != null && wakeLock.isHeld()) {
                try {
                    wakeLock.release();
                } catch (Exception e) {
                    EMLog.e(TAG, e.getMessage());
                }
                EMLog.d(TAG, "released the wake lock");
            }
        }
    }

    private void handleError(EMAError error) throws ChatException {
        if (error.errCode() != EMAError.EM_NO_ERROR) {
            throw new ChatException(error);
        }
    }

    
    /**
     *  \~english 
     * The connectivity change listener.
     * 
     * Only occurs in the reconnection thread and resets the attempts times.
     */
    private BroadcastReceiver connectivityBroadcastReceiver = new BroadcastReceiver() {
        @Override
        public void onReceive(final Context context, Intent intent) {
            String action = intent.getAction();
            if (!action.equals(ConnectivityManager.CONNECTIVITY_ACTION)) {
                EMLog.d(TAG, "skip no connectivity action");
                return;
            }

            EMLog.d(TAG, "connectivity receiver onReceiver");

            EMANetwork networkType;
            NetUtils.Types types = NetUtils.getNetworkTypes(getContext());
            switch (types) {
                case WIFI:
                case OTHERS:
                    networkType = EMANetwork.NETWORK_WIFI;
                    break;
                case MOBILE:
                    networkType = EMANetwork.NETWORK_MOBILE;
                    break;
                case ETHERNET:
                    networkType = EMANetwork.NETWORK_CABLE;
                    break;
                case NONE:
                default:
                    networkType = EMANetwork.NETWORK_NONE;
                    break;
            }

            boolean prevNetworkAvailable = currentNetworkType != EMANetwork.NETWORK_NONE;
            boolean currentNetworkAvailable = networkType != EMANetwork.NETWORK_NONE;
            currentNetworkType = networkType;
            if (prevNetworkAvailable == currentNetworkAvailable && currentNetworkAvailable) {
               execute(new Runnable() {
                    @Override
                    public void run() {
                        if (smartHeartbeat != null) {
                            // Network availability no change, return.
                            EMLog.i(TAG, "Network availability no change, just return. " + currentNetworkType + ", but check ping");
                            smartHeartbeat.sendPingCheckConnection();
                        }
                    }
                });
                return;
            }

            EMLog.i(TAG, "Network availability changed, notify... " + currentNetworkType);

            execute(new Runnable() {

                @Override
                public void run() {
                    emaObject.onNetworkChanged(currentNetworkType);
                }
            });
        }
    };

    void onNetworkChanged() {
        try {
            if (NetUtils.isWifiConnected(mContext)
                    || NetUtils.isOthersConnected(mContext)) {
                EMLog.d(TAG, "has wifi connection");
                currentNetworkType = EMANetwork.NETWORK_WIFI;
                emaObject.onNetworkChanged(EMANetwork.NETWORK_WIFI);
                return;
            }

            if (NetUtils.isMobileConnected(mContext)) {
                EMLog.d(TAG, "has mobile connection");
                currentNetworkType = EMANetwork.NETWORK_MOBILE;
                emaObject.onNetworkChanged(EMANetwork.NETWORK_MOBILE);
                return;
            }

            if (NetUtils.isEthernetConnected(mContext)) {
                EMLog.d(TAG, "has ethernet connection");
                currentNetworkType = EMANetwork.NETWORK_CABLE;
                emaObject.onNetworkChanged(EMANetwork.NETWORK_CABLE);
                return;
            }
            currentNetworkType = EMANetwork.NETWORK_NONE;
            EMLog.d(TAG, "no data connection");
            emaObject.onNetworkChanged(EMANetwork.NETWORK_NONE);
            return;
        } catch (Exception e) {
            e.printStackTrace();
            return;
        }
    }

    void onNetworkChanged(EMANetwork network) {
        emaObject.onNetworkChanged(network);
    }

    private AppStateListener appStateListener;
    private List<Activity> resumeActivityList = new ArrayList<>();

    void setAppStateListener(AppStateListener appStateListener) {
        this.appStateListener = appStateListener;
    }

    @TargetApi(14)
    private void registerActivityLifecycleCallbacks() {
        if (Utils.isSdk14()) {
            Object lifecycleCallbacks = new ActivityLifecycleCallbacks() {

                @Override
                public void onActivityStopped(Activity activity) {
                    resumeActivityList.remove(activity);
                    if (resumeActivityList.isEmpty()) {
                        if (appStateListener != null) appStateListener.onBackground();
                    }
                }

                @Override
                public void onActivityResumed(Activity activity) {
                }

                @Override
                public void onActivityCreated(Activity activity, Bundle savedInstanceState) {

                }

                @Override
                public void onActivityStarted(Activity activity) {
                    if(!resumeActivityList.contains(activity)) {
                        resumeActivityList.add(activity);
                        if(resumeActivityList.size() == 1) {
                            if (appStateListener != null) appStateListener.onForeground();
                        }
                    }
                }

                @Override
                public void onActivitySaveInstanceState(Activity activity, Bundle outState) {

                }

                @Override
                public void onActivityPaused(Activity activity) {

                }

                @Override
                public void onActivityDestroyed(Activity activity) {

                }

            };
            ((Application) mContext).registerActivityLifecycleCallbacks((ActivityLifecycleCallbacks) lifecycleCallbacks);
        }
    }

    interface AppStateListener {
        void onForeground();

        void onBackground();
    }

    // For service check.
    private boolean duringChecking = false;

    /**
     * \~english
     * Checks the Chat service.
     * 
     * The service check process is as follows:
     *
     * 1. Validates the user ID and password.
     * 2. Gets the DNS list from the server.
     * 3. Gets the token from the server.
     * 4. Connects to the IM server.
     * 5. Logs out of the Chat app. (If a user logs in before the check, this step will be ignored.)
     * 
     * If an error occurs during the check, the check process will be interrupted.
     *
     * @param username The user ID for the service check. If a user account has logged in before, this user ID
     *                 will be changed to the logged-in user ID to avoid changes to the information of the current logged-in
     *                 user account.
     * @param password The password. If a user account has logged in before, the password will be changed to the password of the logged-in user.
     * @param listener The service check result callback.
     */
    public void check(String username, String password, final CheckResultListener listener) {
        // Already during checking, just return.
        if (duringChecking) {
            EMLog.i("EMServiceChecker", "During service checking, please hold on...");
            return;
        }

        duringChecking = true;

        // If is logged in before, the username must be current user.
        if (isLoggedInBefore()) {
            username = getCurrentUser();
            EMSessionManager sessionMgr = EMSessionManager.getInstance();
            password = sessionMgr.getLastLoginPwd();
        }

        final String finalUser = username;
        final String finalPwd = password;

        new Thread(new Runnable() {
            @Override
            public void run() {
                /**
                 * Contains account-validation check, get-dns check, get-Token check, login check.
                 * So the {@link EMAChatClient.CheckResultListener#onResult(int, int, String)} callback
                 * will be called four times.
                 */
                emaObject.check(finalUser, finalPwd, new EMAChatClient.CheckResultListener() {
                    @Override
                    public void onResult(final int type, final int result, final String desc) {
                        EMLog.i("EMServiceChecker", "type: " + type + ", result: " + result + ", desc: " + desc);
                        // Notify user once per callback.
                        notifyCheckResult(listener, type, result, desc);

                        // If occur a error during four checks, return.
                        if (result != Error.EM_NO_ERROR) {
                            duringChecking = false;
                            return;
                        }

                        // If login successfully, logout
                        if (type == ChatCheckType.DO_LOGIN) {
                            checkLogout(listener);
                        }
                    }
                });
            }
        }).start();
    }

    private void checkLogout(CheckResultListener listener) {
        // If is not a logged in service check, logout and notify the user.
        if (!isLoggedInBefore()) {
            logout();
            notifyCheckResult(listener, ChatCheckType.DO_LOGOUT, Error.EM_NO_ERROR, "");
        }

        duringChecking = false;
    }

    private void notifyCheckResult(CheckResultListener listener, @ChatCheckType.CheckType int type, int result, String desc) {
        if (listener == null) return;
        listener.onResult(type, result, desc);
    }

    public interface CheckResultListener {
        void onResult(@ChatCheckType.CheckType int type, int result, String desc);
    }

}
