/*
 *  * EaseMob CONFIDENTIAL
 * __________________
 * Copyright (C) 2017 EaseMob Technologies. All rights reserved.
 *
 * NOTICE: All information contained herein is, and remains
 * the property of EaseMob Technologies.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from EaseMob Technologies.
 */
package io.agora.chat;

import io.agora.ValueCallBack;
import io.agora.chat.adapter.EMAChatRoom;

import java.util.List;
import java.util.Map;

/**
 * \~english
 * Chat room types.
 */
public class ChatRoom extends EMBase<EMAChatRoom> {

    enum ChatRoomStyle{
        /**
         * \~english
         * Private chat rooms. Only the owner can invite users to join.
         */
        ChatRoomStylePrivateOnlyOwnerInvite,

        /**
         * \~english
         * Private chat rooms. Both the chat room owner and members can invite users to join.
         */
        ChatRoomStylePrivateMemberCanInvite,

        /**
         * \~english
         * Public chat rooms. Only the chat room owner can invite users to join; a user can join a chat room after the owner approves the user's chat room request.
         */
        ChatRoomStylePublicJoinNeedApproval,

        /**
         * \~english
         *  Public chat rooms. A user can join a chat room without the owner approving the user's chat room request.
         */
        ChatRoomStylePublicOpenJoin
    }

    /**
     *  \~english
     *  Chat room role types.
     */
    public enum ChatRoomPermissionType {
        /**
         * \~english
         * Regular member.
         */
        member(0),

        /**
         * \~english
         * Chat room admin.
         */
        admin(1),

        /**
         * \~english
         * Chat room owner.
         */
        owner(2),

        /**
         * \~english
         * Unknown type.
         */
        none(-1);
        private int permissionType;
        private ChatRoomPermissionType(int permissionType) {
            this.permissionType = permissionType;
        }
    }

	public ChatRoom(){
	    emaObject = new EMAChatRoom();
	}
	
	public ChatRoom(EMAChatRoom room) {
	    emaObject = new EMAChatRoom(room);
	}
	
    public ChatRoom(String roomId){
        emaObject = new EMAChatRoom(roomId);
    }
    
    public ChatRoom(String roomId, String roomName){
        emaObject = new EMAChatRoom(roomId);
//        this.nick = roomName;
    }

    /**
     * \~english
     * Gets the chat room ID.
     *
     * @return The chat room ID.
     */
    public String getId() {
        return emaObject.getId();
     }

    /**
     * \~english
     * Gets the chat room name.
     * @return The chat room name.
     */
     public String getName() {
        return emaObject.getName();
     }

    /**
     * \~english
     * Gets the chat room description.
     * @return The chat room description.
     */
     public String getDescription() {
        return emaObject.getDescription();
    }

    /**
     * \~english
     * Gets the chat room owner.
     * For how to get chat room details, see {@link ChatRoomManager#fetchChatRoomFromServer(String)}.
     * @return The chat room owner ID. If this method returns an empty string, the SDK fails to get chat room details.
     */
    public String getOwner() {
        return emaObject.getOwner();
    }

    /**
     * \~english
     * Gets the chat room admin list.
     * For how to get chat room, see {@link ChatRoomManager#fetchChatRoomFromServer(String)}.
     * @return The chat room admin ID list. If this method returns an empty list, the SDK fails to get the admin list of the chat room.
     */
    public List<String> getAdminList() { return emaObject.getAdministratorList();}

    /**
     * \~english
     * Returns the number of online members.
     * @return The number of online members.
     */
    public int getMemberCount()
    {
        return emaObject.getAffiliationsCount();
    }

    /**
     * \~english
     * Gets the maximum number of members in the chat room which is determined during chat room creation.
     * This method can return a correct value only after you call ChatRoomManager#fetchChatRoomFromServer(String) to get chat room details. 
     * @return The maximum number of members in a chat room. Returns 0 if the SDK fails to get the maximum number.
     */
    public int getMaxUsers() {
        return emaObject.getMaxUserCount();
    }

    /**
     * \~english
     * Returns the member list.
     * You can get the member list in the following ways:
     * - When there are less than 200 members, use {@link ChatRoomManager#fetchChatRoomFromServer(String, boolean)} to get them.
     * If true is passed to the second parameter, you can get up to 200 members.
     * - Use {@link ChatRoomManager#fetchChatRoomMembers(String, String, int)} to get the member list with pagination
     * or use its asynchronous method {@link ChatRoomManager#asyncFetchChatRoomMembers(String, String, int, ValueCallBack)}.
     * 
     * @return The member ID list. If this method returns an empty list, the SDK fails to get the member list.
     */
    public List<String> getMemberList(){
        return emaObject.getMemberList();
    }

    /**
     * \~english
     * Returns the chat room blocklist.
     * To get the blocklist, you can call {@link ChatRoomManager#fetchChatRoomBlackList(String, int, int)}.
     * You can call this method as the chat room owner or admin or call its asynchronous method
     * {@link ChatRoomManager#asyncFetchChatRoomBlackList(String, int, int, ValueCallBack)}.
     * @return The blocklist. If this method returns an empty list, the SDK fails to get the blocklist.
     * @deprecated Deprecated. Please use {@link ChatRoom#getBlacklist()} instead.
     */
    @Deprecated
    public List<String> getBlackList() { return getBlacklist(); }

    /**
     * \~english
     * Returns the chat room blocklist.
     * To get the blocklist, you can call {@link ChatRoomManager#fetchChatRoomBlackList(String, int, int)}.
     * You can call this method as the chat room owner or admin or call its asynchronous method.
     * {@link ChatRoomManager#asyncFetchChatRoomBlackList(String, int, int, ValueCallBack)}.
     * @return The chat room blocklist. If this method returns an empty list, the SDK fails to get the blocklist.
     */
    public List<String> getBlacklist() { return emaObject.getBlockList(); }

    /**
     * \~english
     * Returns the mute list of the chat room.
     * To get the mute list, you can call {@link ChatRoomManager#fetchChatRoomMuteList(String, int, int)}.
     * You can call this method as the chat room owner or admin or call its asynchronous method
     * {@link ChatRoomManager#asyncFetchChatRoomMuteList(String, int, int, ValueCallBack)}.
     *
     * @return The mute list, where Map.entry.key is the ID of the muted member and Map.entry.value is the mute duration in milliseconds.
     *         If this method returns an empty map, the SDK fails to get the mute list.
     */
    public Map<String, Long> getMuteList() { return emaObject.getMuteList(); }

    /**
     * \~english
     * Returns the chat room allowlist.
     * To get the allowlist, you can call {@link ChatRoomManager#fetchChatRoomWhiteList(String, ValueCallBack)}.
     * You can call this method as the chat room owner or admin.
     * @return The allowlist. If this method returns an empty list, the SDK fails to get the allowlist.
     * @deprecated Deprecated. Please use {@link ChatRoom#getWhitelist()} instead.
     */
    @Deprecated
    public List<String> getWhiteList() { return getWhitelist(); }

    /**
     * \~english
     * Returns the chat room allowlist.
     * 
     * To get the allowlist, you can call {@link ChatRoomManager#fetchChatRoomWhiteList(String, ValueCallBack)}.
     * You can call this method as the chat room owner or admin.
     * @return The allowlist. If this method returns an empty list, the SDK fails to get the allowlist.
     */
    public List<String> getWhitelist() { return emaObject.getWhiteList(); }

    /**
     * \~english
     * Checks whether all members are muted.
     * This method has use limitations and should be used with caution.
     * Pay attention to the following when using this method：
     * - Upon your receipt of the callback of one-click mute or unmute after you join the chat room, the status will be updated and the staus obtained using this method is correct.
     * 
     * - After you exit from the chat room before reentering it, the status obtained using this method is not trustworthy.
     * @return Whether all members are muted. 
     * - `true`: All members are muted. 
     * - `false`: Not all members are muted.
     */
    public boolean isAllMemberMuted() { return emaObject.isAllMemberMuted(); }


    /**
     * \~english
     * Gets the chat room announcement.
     * To get the chat room announcement, you can call {@link ChatRoomManager#fetchChatRoomAnnouncement(String)}
     * or its asynchronous method {@link ChatRoomManager#asyncFetchChatRoomAnnouncement(String, ValueCallBack)}.
     * @return The chat room announcement. If this method returns an empty string, the SDK fails to get the chat room announcement.
     */
    public String getAnnouncement() {
        return emaObject.getAnnouncement();
    }

    /**
     * \~english
     * Gets the current user's role in the chat room.
     * @return The current user's role in the chat room, see {@link ChatRoomPermissionType}.
     */
    public ChatRoomPermissionType getChatRoomPermissionType() {
        int permissionType = emaObject.permissionType();
        if(permissionType == ChatRoomPermissionType.member.permissionType) {
            return ChatRoomPermissionType.member;
        }else if(permissionType == ChatRoomPermissionType.admin.permissionType) {
            return ChatRoomPermissionType.admin;
        }else if(permissionType == ChatRoomPermissionType.owner.permissionType) {
            return ChatRoomPermissionType.owner;
        }else {
            return ChatRoomPermissionType.none;
        }
    }

}
