/************************************************************
 *  * hyphenate CONFIDENTIAL 
 * __________________ 
 * Copyright (C) 2013-2014 hyphenate Technologies. All rights reserved. 
 *
 * NOTICE: All information contained herein is, and remains 
 * the property of hyphenate Technologies.
 * Dissemination of this information or reproduction of this material 
 * is strictly forbidden unless prior written permission is obtained
 * from hyphenate Technologies.
 */
package io.agora.chat;

import io.agora.CallBack;
import io.agora.ContactListener;
import io.agora.ValueCallBack;
import io.agora.chat.adapter.EMAContactListener;
import io.agora.chat.adapter.EMAContactManager;
import io.agora.chat.adapter.EMAError;
import io.agora.exceptions.ChatException;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

/**
 * \~english
 * The `ContactManager` is used to record, query, and modify contacts.

 */
public class ContactManager {

    EMAContactManager emaObject;
    ChatClient mClient;
    private List<ContactListener> contactListeners = Collections.synchronizedList(new ArrayList<ContactListener>());
    private EMAContactListenerImpl contactImpl = new EMAContactListenerImpl();

    ContactManager(ChatClient client, EMAContactManager contactManager) {
        mClient = client;
        emaObject = new EMAContactManager(contactManager);
        emaObject.registerContactListener(contactImpl);
    }

    class EMAContactListenerImpl extends EMAContactListener {

        @Override
        public void onContactAdded(String username) {
            synchronized (contactListeners) {
                try {
                    for (ContactListener listener : contactListeners) {
                        listener.onContactAdded(username);
                    }
                } catch (Exception e) {
                    e.printStackTrace();
                }
            }
        }

        @Override
        public void onContactDeleted(String username) {
            ChatClient.getInstance().chatManager().caches.remove(username);

            synchronized (contactListeners) {
                try {
                    for (ContactListener listener : contactListeners) {
                        listener.onContactDeleted(username);
                    }
                } catch (Exception e) {
                    e.printStackTrace();
                }
            }
        }

        @Override
        public void onContactInvited(String username, String reason) {
            synchronized (contactListeners) {
                try {
                    for (ContactListener listener : contactListeners) {
                        listener.onContactInvited(username, reason);
                    }
                } catch (Exception e) {
                    e.printStackTrace();
                }
            }
        }

        @Override
        public void onContactAgreed(String username) {
            synchronized (contactListeners) {
                try {
                    for (ContactListener listener : contactListeners) {
                        listener.onFriendRequestAccepted(username);
                    }
                } catch (Exception e) {
                    e.printStackTrace();
                }
            }
        }

        @Override
        public void onContactRefused(String username) {
            synchronized (contactListeners) {
                try {
                    for (ContactListener listener : contactListeners) {
                        listener.onFriendRequestDeclined(username);
                    }
                } catch (Exception e) {
                    e.printStackTrace();
                }
            }
        }
    }

    private void handleError(EMAError error)  throws ChatException {
        if (error.errCode() != EMAError.EM_NO_ERROR) {
            throw new ChatException(error);
        }
    }

    /**
     * \~english
     * Adds a new contact.
     *
     * This is a synchronous method and blocks the current thread.
     *
     * Reference：
     * For the asynchronous method, see {@link #asyncAddContact(String, String, CallBack)}.
     * 
     * @param username		The user to be added.
     * @param reason     	(optional) The invitation message. Set the parameter as null if you want to ignore the information.
     * @throws ChatException    A description of the issue that caused this exception.
     */
    public void addContact(String username, String reason) throws ChatException {
        EMAError error = new EMAError();
        emaObject.inviteContact(username, reason, error);
        handleError(error);
    }

    /**
     * \~english
     * Adds a new contact.
     * 
     * This is an asynchronous method.
     * 
     * Reference：
     * For the synchronous method, see {@link #addContact(String, String)}
     *
     * @param username  The user ID of the contact to be added.
     * @param reason    The message for adding contact (optional).
     * @param callback  The result of the method, which contains the error information if the method fails.
     */
    public void asyncAddContact(final String username, final String reason, final CallBack callback) {
        ChatClient.getInstance().execute(new Runnable() {

            @Override
            public void run() {
                try {
                    addContact(username, reason);
                    callback.onSuccess();
                } catch (ChatException e) {
                    callback.onError(e.getErrorCode(), e.getDescription());
                }
            }
        });
    }

    /**
     * \~english
     * Deletes a contact and all the conversations associated.
     *
     * This is a synchronous method and blocks the current thread.
     *
     * @param username  The contact to be deleted.
     * @throws ChatException A description of the issue that caused the exception.
     */
    public void deleteContact(String username) throws ChatException {
        deleteContact(username, false);
    }

    /**
     * \~english
     * Deletes a contact.
     *
     * This is a synchronous method and blocks the current thread.
     *
     * @param username  The user ID.
     * @param keepConversation  Whether to keep the associated conversation and messages.
     * @throws ChatException  A description of the issue that caused the exception.
     */
    public void deleteContact(String username, boolean keepConversation) throws ChatException {
        EMAError error = new EMAError();
        emaObject.deleteContact(username, error, keepConversation);
        ChatClient.getInstance().chatManager().caches.remove(username);
        handleError(error);
    }

    /**
     * \~english
     * Deletes a contact.
     * 
     * This is an asynchronous method.
     * 
     * Reference:
     * For the synchronous method, see {@link #deleteContact(String)}.
     *
     * @param username   The contact‘s user ID to be deleted.
     * @param callback   The result of the method, which contains the error information if the method fails.
     */
    public void asyncDeleteContact(final String username,
                                   final CallBack callback) {
        ChatClient.getInstance().execute(new Runnable() {

            @Override
            public void run() {
                try {
                    deleteContact(username);
                    callback.onSuccess();
                } catch (ChatException e) {
                    callback.onError(e.getErrorCode(), e.getDescription());
                }
            }
        });
    }

    /**
     * \~english
     * Get all contacts from the server.
     * 
     * This is a synchronous method and blocks the current thread.
     * 
     * Reference:
     * For the asynchronous method, see {@link #asyncGetAllContactsFromServer(ValueCallBack)}.
     *
     * @return The list of contacts.
     * @throws ChatException A description of the exception.
     */
    public List<String> getAllContactsFromServer() throws ChatException {
        EMAError error = new EMAError();
        List<String> contacts = emaObject.getContactsFromServer(error);
        handleError(error);
        return contacts;
    }

    /**
     * \~english
     * Get all contacts from the server.
     * 
     * Reference:
     * This is a synchronous method see {@link #getAllContactsFromServer()}.
     *
     * @param callback If the method call succeeds, returns the list of contacts; if the method call fails, returns the error information.

     */
    public void asyncGetAllContactsFromServer(final ValueCallBack<List<String>> callback) {
        ChatClient.getInstance().execute(new Runnable() {

            @Override
            public void run() {
                try {
                    List<String> allContacts = getAllContactsFromServer();
                    callback.onSuccess(allContacts);
                } catch (ChatException e) {
                    callback.onError(e.getErrorCode(), e.getDescription());
                }
            }
        });
    }

    //region contactListener
    /**
     * \~english
     * Registers a new contact listener.
     * 
     * Reference:
     * Call {@link #removeContactListener(ContactListener)} to remove the listener.
     *
     * @param contactListener  The contact listener to be registered.

     */
    public void setContactListener(ContactListener contactListener) {
        if (!contactListeners.contains(contactListener)) {
            contactListeners.add(contactListener);
        }
    }

    /**
     * \~english
     * Removes the contact listener.
     * 
     * Reference:
     * Adds the contact listener by calling {@link #setContactListener(ContactListener)}.
     */
    public void removeContactListener(ContactListener contactListener) {
        contactListeners.remove(contactListener);
    }
    //endregion


    /**
     * \~english
     * Adds a user to blocklist.
     * You can send message to the user in blocklist, but you can not receive the message sent by the other.
     * 
     * Reference：
     * For the asynchronous method, see {@link #asyncAddUserToBlackList(String, boolean, CallBack)}.
     * Adds the batch blocklist by calling {@link #saveBlackList(List)}.
     *
     * This is a synchronous method and blocks the current thread.
     *
     * @param username  The user to be blocked.
     * @param both      This parameter is deprecated.
     * @throws ChatException A description of the issue that caused the exception.
     */
    public void addUserToBlackList(String username, boolean both) throws ChatException {
        EMAError error = new EMAError();
        emaObject.addToBlackList(username, both, error);
        handleError(error);
    }

    /**
     * \~english
     * Adds the user to blocklist.
     * You can send message to the user in the blocklist, but you can not receive the message sent by the user in the blocklist.
     * 
     * Reference:
     * For the synchronous method, see {@link #addUserToBlackList(String, boolean)}.
     * If you want to add multiple users in a batch to the blocklist, see {@link #asyncSaveBlackList(List, CallBack)}.
     *
     * @param username  The user to be blocked.
     * @param both      This parameter is deprecated.
     * @param callback  The callback completion.
     * - `Success`: The user is successfully added to the blocklist. 
     * - `Error`: The user fails to be added to the blocklist. In this case, an error message is returned.
     */
    public void asyncAddUserToBlackList(final String username,
                                        final boolean both,
                                        final CallBack callback) {
        ChatClient.getInstance().execute(new Runnable() {

            @Override
            public void run() {
                try {
                    addUserToBlackList(username, both);
                    callback.onSuccess();
                } catch (ChatException e) {
                    callback.onError(e.getErrorCode(), e.getDescription());
                }
            }
        });
    }

    /**
     * \~english
     * Uploads the blocklist to server.
     * 
     * This is a synchronous method and blocks the current thread.
     * 
     * Reference:
     * For the asynchronous method, see {@link #asyncSaveBlackList(List, CallBack)}.
     *
     * @param blackList The blocklist.
     */
    public void saveBlackList(List<String> blackList) throws ChatException {
        EMAError error = new EMAError();
        emaObject.saveBlackList(blackList, error);
        handleError(error);
    }

    /**
     * \~english
     * Uploads the blocklist to server.
     * 
     * Reference:
     * For the synchronous method, see {@link #saveBlackList(List)}.
     *
     * @param blackList  The blocklist.
     * @param callback   The completion callback.  
     * - `Success`: The blocklist is uploaded to the server.
     * - `Error`: The blocklist fails to be uploaded to the server. The reason for the failure will be returned.
     */
    public void asyncSaveBlackList(final List<String> blackList,
                                   final CallBack callback) {
        ChatClient.getInstance().execute(new Runnable() {

            @Override
            public void run() {
                try {
                    saveBlackList(blackList);
                    callback.onSuccess();
                } catch (ChatException e) {
                    callback.onError(e.getErrorCode(), e.getDescription());
                }
            }
        });
    }

    /**
     * \~english
     * Removes the contact from the blocklist.
     * 
     * Reference：
     * For the asynchronous method, see {@link #asyncRemoveUserFromBlackList(String, CallBack)}.
     *
     * This is a synchronous method and blocks the current thread.
     *
     * @param username  The user to be removed from the blocklist.
     * @throws ChatException   A description of the exception.
     */
    public void removeUserFromBlackList(String username) throws ChatException {
        EMAError error = new EMAError();
        emaObject.removeFromBlackList(username, error);
        handleError(error);
    }

    /**
     * \~english
     * Removes the contact from the blocklist.
     * 
     * Reference:
     * For the synchronous method, see {@link #removeUserFromBlackList(String)}.
     *
     * @param username  The user to be removed from the blocklist.
     * @param callback  The completion callback.
     * - `Success`: The user is successfully removed from the blocklist.
     * - `Error`: The user fails to be removed from the blocklist. The reason for the failure will be returned.
     */
    public void asyncRemoveUserFromBlackList(final String username,
                                             final CallBack callback) {
        ChatClient.getInstance().execute(new Runnable() {

            @Override
            public void run() {
                try {
                    removeUserFromBlackList(username);
                    callback.onSuccess();
                } catch (ChatException e) {
                    callback.onError(e.getErrorCode(), e.getDescription());
                }
            }
        });
    }

    /**
     * \~english
     * Gets the local blocklist.
     *
     * @return The blocklist.
     */
    public List<String> getBlackListUsernames() {
        EMAError error = new EMAError();
        return emaObject.getBlackListFromDB(error);
    }

    /**
     * \~english
     * Gets the blocklist from the server.
     * 
     * Reference:
     * For the asynchronous method, see {@link #asyncGetBlackListFromServer(ValueCallBack)}.
     *
     * This is a synchronous method and blocks the current thread.
     *
     * @return  Returns the blocklist if the method succeeds.
     * @throws ChatException A description of the issue that caused the exception.
     */
    public List<String> getBlackListFromServer() throws ChatException {
        EMAError error = new EMAError();
        List<String> blacklist = emaObject.getBlackListFromServer(error);
        handleError(error);
        return blacklist;
    }

    /**
     * \~english
     * Gets the blocklist from the server.
     * 
     * Reference:
     * For the synchronous method, see {@link #getBlackListFromServer()}.
     *
     * @param callback Returns the blocklist if the call succeed and the description of the cause if the call fails.
     */
    public void asyncGetBlackListFromServer(final ValueCallBack<List<String>> callback) {
        ChatClient.getInstance().execute(new Runnable() {

            @Override
            public void run() {
                try {
                    List<String> blackList = getBlackListFromServer();
                    callback.onSuccess(blackList);
                } catch (ChatException e) {
                    callback.onError(e.getErrorCode(), e.getDescription());
                }
            }
        });
    }

    /**
     * \~english
     *
     * Accepts a friend invitation。
     * 
     * This is a synchronous method and blocks the current thread.
     * 
     * Reference：
     * For the asynchronous method, see {@link #asyncAcceptInvitation(String, CallBack)}
     *
     * @param username  The user who initiates the friend request.
     */
    public void acceptInvitation(String username) throws ChatException
    {
        EMAError error = new EMAError();
        emaObject.acceptInvitation(username, error);
        handleError(error);
    }


    /**
     * \~english
     * Accepts a friend invitation.
     *
     * This an asynchronous method.
     * 
     * Reference:
     * For the synchronous method, see {@link #acceptInvitation(String)}.
     *
     * @param username  The user who initiates the friend request.
     * @param callback  Returns `Success` if the call succeed and the description of the cause if the call fails.
     */
    public void asyncAcceptInvitation(final String username,
                                      final CallBack callback) {
        ChatClient.getInstance().execute(new Runnable() {

            @Override
            public void run() {
                try {
                    acceptInvitation(username);
                    callback.onSuccess();
                } catch (ChatException e) {
                    callback.onError(e.getErrorCode(), e.getDescription());
                }
            }
        });
    }

    /**
     * \~english
     * Declines a friend invitation.
     *      
     * This is a synchronous method and blocks the current thread.
     * 
     * Reference:
     * For the asynchronous method, see {@link #asyncDeclineInvitation(String, CallBack)}.
     *
     * @param username The user who initiates the invitation.
     */
    public void declineInvitation(String username) throws ChatException  {
        EMAError error = new EMAError();
        emaObject.declineInvitation(username, error);
        handleError(error);
    }

    /**
     * \~english
     * Declines a friend invitation.
     *      
     * This an asynchronous method.
     * 
     * Reference:
     * For the synchronous method, see {@link #declineInvitation(String)}.
     *
     * @param username The user who initiates the friend request.
     * @param callback  Returns `Success` if the call succeed and the description of the cause if the call fails.
     */
    public void asyncDeclineInvitation(final String username,
                                       final CallBack callback) {
        ChatClient.getInstance().execute(new Runnable() {

            @Override
            public void run() {
                try {
                    declineInvitation(username);
                    callback.onSuccess();
                } catch (ChatException e) {
                    callback.onError(e.getErrorCode(), e.getDescription());
                }
            }
        });
    }

    /**
     * \~english
     * Gets the contact list from the local database.
     *
     * @return The contact list.
     * @throws ChatException A description of the issue that caused the exception.
     */
    public List<String> getContactsFromLocal() throws ChatException {
        EMAError error = new EMAError();
        List<String> contacts = getContactsFromDB(error);
        handleError(error);
        return contacts;
    }

    /**
     * \~english
     * Get the contact list from the local database.
     *
     * @param error  A description of the issue that caused this call to fail.
     * @return  Returns the contact list if the method succeeds.
     */
    List<String> getContactsFromDB(EMAError error) {
        return emaObject.getContactsFromDB(error);
    }

    void onLogout() {
    }

    /**
     * \~english
     * Get the unique IDs of current user on the other devices. The ID is username + "/" + resource.
     *      
     * This is a synchronous method and blocks the current thread.
     * 
     * Reference：
     * For the asynchronous method, see {@link #asyncGetSelfIdsOnOtherPlatform(ValueCallBack)}.
     *
     * @return   Returns the unique device ID list on the other devices if the method succeeds.
     * @throws ChatException A description of the issue that caused this call to fail.
     */
    public List<String> getSelfIdsOnOtherPlatform() throws ChatException {
        EMAError error = new EMAError();
        List<String> selfIds = emaObject.getSelfIdsOnOtherPlatform(error);
        handleError(error);
        return selfIds;
    }

    /**
     * \~english
     * Get the unique selfID list of the current user on the other devices. The selfID contains username + "/" + resource(the device ID).
     *      
     * This is an asynchronous method.
     * 
     * Reference:
     * For the synchronous method, see {@link #getSelfIdsOnOtherPlatform()}.
     *
     * @param callback  - Returns the unique selfID list on the other devices if the method succeeds.
     *                  - Returns the description of the cause of the error if the method fails.

     */
    public void asyncGetSelfIdsOnOtherPlatform(final ValueCallBack<List<String>> callback) {
        ChatClient.getInstance().execute(new Runnable() {

            @Override
            public void run() {
                try {
                    List<String> selfIds = getSelfIdsOnOtherPlatform();
                    callback.onSuccess(selfIds);
                } catch (ChatException e) {
                    callback.onError(e.getErrorCode(), e.getDescription());
                }
            }
        });
    }
}
