/************************************************************
  *  * EaseMob CONFIDENTIAL 
  * __________________ 
  * Copyright (C) 2013-2014 EaseMob Technologies. All rights reserved. 
  *  
  * NOTICE: All information contained herein is, and remains 
  * the property of EaseMob Technologies.
  * Dissemination of this information or reproduction of this material 
  * is strictly forbidden unless prior written permission is obtained
  * from EaseMob Technologies.
  */
package io.agora.chat;

import android.net.Uri;
import android.os.Parcel;
import android.os.Parcelable;

import io.agora.chat.adapter.message.EMAFileMessageBody.EMADownloadStatus;
import io.agora.chat.adapter.message.EMAImageMessageBody;
import io.agora.chat.adapter.message.EMAMessageBody;
import io.agora.util.FileHelper;

import java.io.File;

/**
 * \~english
 * The image message body class.
 *
 * ```java
 *     ImageMessageBody body = new ImageMessageBody(imageFile);
 * ```
 */
public class ImageMessageBody extends FileMessageBody implements Parcelable {
	private static final String TAG = ImageMessageBody.class.getSimpleName();
	
	private boolean sendOriginalImage = false;
    
     /**
     * \~english
     * Creates an image message body with an image file. 
     * 
     * Reference: There are five ways to create an image message body as follows:
	 * 
     *  Creates an image message body with an image file:
     *  - ImageMessageBody(File imageFile)
     *  Creates an image message body with an image URI:
     *  - ImageMessageBody(Uri imageUri)
     *  Creates an image message body with an image file and its thumbnail file:
     *  - ImageMessageBody(File imageFile, File thumbnailFile)
     *  Creates an image message body with an image URI and its thumbnail URI:
     *  - ImageMessageBody(Uri imageLocalPath, Uri thumbnailPath)
     *  Creates an image message body with an EMAImageMessageBody body:
     *  - ImageMessageBody(EMAImageMessageBody body)
     * 
     * @param imageFile The image message body.
     */
    public ImageMessageBody(File imageFile){
    	super(Uri.fromFile(imageFile), EMAMessageBody.EMAMessageBodyType_IMAGE);
    }

    /**
     * \~english
     * Creates an image message body with the image URI.
	 *
     * @param imageUri The image URI.
     */
    public ImageMessageBody(Uri imageUri){
    	super(imageUri, EMAMessageBody.EMAMessageBodyType_IMAGE);
    	setFileName(FileHelper.getInstance().getFilename(imageUri));
    }

	/**
	 * \~english
	 * Creates an image message body with an image and its thumbnail.
	 *
	 * @param imageFile 		The original image file.
	 * @param thumbnailFile		The thumbnail image file.
	 */
    public ImageMessageBody(File imageFile, File thumbnailFile) {
        super(Uri.fromFile(imageFile), EMAMessageBody.EMAMessageBodyType_IMAGE);
        ((EMAImageMessageBody)emaObject).setThumbnailLocalPath(thumbnailFile == null ?
				"" : Uri.fromFile(thumbnailFile).toString());
		((EMAImageMessageBody)emaObject).setThumbnailFileLength(thumbnailFile != null && thumbnailFile.exists() ? thumbnailFile.length() : 0);
    }

	/**
	 * \~english
	 * Creates an image message body with the URI of an image and its thumbnail.
	 *
	 * @param imageLocalPath 	The original image URI.
	 * @param thumbnailPath		The thumbnail image URI.
	 */
    public ImageMessageBody(Uri imageLocalPath, Uri thumbnailPath) {
        super(imageLocalPath, EMAMessageBody.EMAMessageBodyType_IMAGE);
        ((EMAImageMessageBody)emaObject).setThumbnailLocalPath(thumbnailPath != null ?
				thumbnailPath.toString() : "");
        setFileName(FileHelper.getInstance().getFilename(imageLocalPath));
		((EMAImageMessageBody)emaObject).setThumbnailFileLength(FileHelper.getInstance().getFileLength(thumbnailPath));
    }
    
    /**
	 * \~english
	 * Creates an image message body with an `EMAImageMessageBody` instance.
	 *
	 * @param body	The `EMAImageMessageBody` instance.
	 */
    public ImageMessageBody(EMAImageMessageBody body) {
    	super(body);
	}
    
    /**
	 * \~english
     * Creates the image message body from an incoming message, which contains the image file and its thumbnail.
	 *
     * @param fileName       The name of the image file.
     * @param remoteUrl      The image path.
     * @param thumbnailUrl   The thumbnail path.
     * 
     */
    ImageMessageBody(String fileName, String remoteUrl, String thumbnailUrl) {
    	super((Uri) null, EMAMessageBody.EMAMessageBodyType_IMAGE);
    	emaObject = new EMAImageMessageBody((Uri) null, (Uri) null);
    	((EMAImageMessageBody)emaObject).setDisplayName(fileName);
    	((EMAImageMessageBody)emaObject).setRemotePath(remoteUrl);
    	((EMAImageMessageBody)emaObject).setThumbnailRemotePath(thumbnailUrl);
    }

    /**
     * \~english
     * Gets the URL of the thumbnail on the server.
     * 
     * If {@link ChatOptions#setAutoDownloadThumbnail(boolean)} is set, the thumbnail will be downloaded from the path specified with thumbnailRemotePath to a local path during message reception.
     * 
     * @return The URL of the thumbnail on the server.

     */
    public String getThumbnailUrl() {
		return ((EMAImageMessageBody) emaObject).thumbnailRemotePath();
    }
    
    /**
     * \~english
     * Sets the URL of the thumbnail on the server.
     * @param thumbnailUrl The URL of the thumbnail on the server.
     */
    public void setThumbnailUrl(String thumbnailUrl) {
    	((EMAImageMessageBody)emaObject).setThumbnailRemotePath(thumbnailUrl);
    }

    public String toString() {
        return "image: " + ((EMAImageMessageBody)emaObject).displayName() +
				", localurl: " + ((EMAImageMessageBody)emaObject).getLocalUrl() +
				", remoteurl: " + ((EMAImageMessageBody)emaObject).getRemoteUrl() +
				", thumbnail: " + ((EMAImageMessageBody)emaObject).thumbnailRemotePath();
    }

    @Override
    public int describeContents() {
        return 0;
    }

    @Override
    public void writeToParcel(Parcel dest, int flags) {
        //members from FileMessageBody
        dest.writeString(((EMAImageMessageBody)emaObject).displayName());
        dest.writeString(((EMAImageMessageBody)emaObject).getLocalUrl());
        dest.writeString(((EMAImageMessageBody)emaObject).getRemoteUrl());
        dest.writeString(((EMAImageMessageBody)emaObject).thumbnailRemotePath());
        
        dest.writeInt(((EMAImageMessageBody)emaObject).width());
        dest.writeInt(((EMAImageMessageBody)emaObject).height());
        
    }
    
    public static final Parcelable.Creator<ImageMessageBody> CREATOR
            = new Parcelable.Creator<ImageMessageBody>() {
        public ImageMessageBody createFromParcel(Parcel in) {
            return new ImageMessageBody(in);
        }

        public ImageMessageBody[] newArray(int size) {
            return new ImageMessageBody[size];
        }
    };

    private ImageMessageBody(Parcel in) {
    	super((Uri) null, EMAMessageBody.EMAMessageBodyType_IMAGE);
        //members from FileMessageBody
    	((EMAImageMessageBody)emaObject).setDisplayName(in.readString());
    	((EMAImageMessageBody)emaObject).setLocalPath(in.readString());
    	((EMAImageMessageBody)emaObject).setRemotePath(in.readString());
    	((EMAImageMessageBody)emaObject).setThumbnailRemotePath(in.readString());
		int width = in.readInt();
		int height = in.readInt();
		((EMAImageMessageBody) emaObject).setSize(width, height);
    }

	/**
	 * \~english
	 * Sets the secret to access the thumbnail. A secret is required for verification for thumbnail download.
	 * @param secret The secret to access the thumbnail.
	 */
	public void setThumbnailSecret(String secret) {
    	((EMAImageMessageBody)emaObject).setThumbnailSecretKey(secret);
    }

	/**
	 * \~english
	 * Gets the secret to access the thumbnail. A secret is required for verification for thumbnail download.
	 * @return  The secret to access the thumbnail.
	 */
    public String getThumbnailSecret() {
    	return ((EMAImageMessageBody)emaObject).thumbnailSecretKey();
    }
    
    /**
     * \~english
     * Sets whether to send the original image when sending an image.
     * 
     * @param sendOriginalImage Whether to send the original image. 
     * -(Default)`false`: Send the thumbnail(image with size larger than 100k will be compressed);
     * - `true`: Send the original image.
     */
    public void setSendOriginalImage(boolean sendOriginalImage) {
    	this.sendOriginalImage = sendOriginalImage;
    }
    
    /**
     * \~english
     * Gets whether to send the original image.
     * @return Whether to send the original image. 
     * -(Default)`false`: Send the thumbnail(image with size larger than 100k will be compressed);
     * - `true`: Send the original image.
     */
    public boolean isSendOriginalImage(){
    	return sendOriginalImage;
    }
    
    /**
     * \~english
     * Gets the image width.
     * @return The image width.
     */
    public int getWidth() {
		return ((EMAImageMessageBody)emaObject).width();
	}

    /**
     * \~english
     * Gets the image height.
     * @return The image height.
     */
	public int getHeight() {
		return ((EMAImageMessageBody)emaObject).height();
	}
	
	/**
     * \~english
     * Gets the image file name.
     * @return  The image file name.
     */
	public String getFileName() {
		return ((EMAImageMessageBody)emaObject).displayName();
	}
	
	void setSize(int width, int height) {
		((EMAImageMessageBody)emaObject).setSize(width, height);
	}

	/**
	 * \~english
	 * Gets the size of the image file.
	 * @return  The size of the image file.
	 */
	public long getFileSize() {
		return ((EMAImageMessageBody)emaObject).fileLength();
	}
	
	/**
     * \~english
     * Gets the local path or the URI of the thumbnail as a string.
     * @return  The local path or the URI of the thumbnail as a string.
     */
	public String thumbnailLocalPath() {
		return FileHelper.getInstance().formatOutLocalUrl(((EMAImageMessageBody)emaObject).thumbnailLocalPath());
	}

	/**
     * \~english
     * Gets the local path of the thumbnail as a URI.
     * @return The local path of the thumbnail as a URI.
     */
	public Uri thumbnailLocalUri() {
		return FileHelper.getInstance().formatOutUri(((EMAImageMessageBody)emaObject).thumbnailLocalPath());
	}

	/**
     * \~english
     * Sets the local path of the thumbnail as a string.
	 * @param localPath The local path of the thumbnail as a string.
     */
    public void setThumbnailLocalPath(String localPath) {
		((EMAImageMessageBody)emaObject).setThumbnailLocalPath(FileHelper.getInstance().formatInUriToString(localPath));
		((EMAImageMessageBody)emaObject).setThumbnailFileLength(FileHelper.getInstance().getFileLength(localPath));
    }

	/**
     * \~english
     * Sets the local path of the thumbnail as a URI.
	 * @param localPath	 The local path of the thumbnail as a URI.
     */
    public void setThumbnailLocalPath(Uri localPath) {
        ((EMAImageMessageBody)emaObject).setThumbnailLocalPath(FileHelper.getInstance().formatInUriToString(localPath));
		((EMAImageMessageBody)emaObject).setThumbnailFileLength(FileHelper.getInstance().getFileLength(localPath));
    }

	/**
	 * \~english
	 * Sets the width and height of the thumbnail image.
	 * @param width		The width of thumbnail.
	 * @param height	The height of thumbnail.
	 */
    public void setThumbnailSize(int width, int height) {
		((EMAImageMessageBody)emaObject).setThumbnailSize(width, height);
	}

	/**
	 * \~english
	 * Gets the download status of the thumbnail.
	 * @return The download status of the thumbnail.
	 */
	public EMDownloadStatus thumbnailDownloadStatus() {
	    EMADownloadStatus _status = ((EMAImageMessageBody)emaObject).thumbnailDownloadStatus();
	    switch (_status) {
	    case DOWNLOADING:
	        return EMDownloadStatus.DOWNLOADING;
	    case SUCCESSED:
	        return EMDownloadStatus.SUCCESSED;
	    case FAILED:
	        return EMDownloadStatus.FAILED;
	    case PENDING:
	        return EMDownloadStatus.PENDING;
	    }
	    return EMDownloadStatus.SUCCESSED;
	}

	/**
	 * \~english
	 * Sets the download status of the thumbnail in the local database.
	 * @param status The download status of the thumbnail.
	 */
	public void setThumbnailDownloadStatus(EMDownloadStatus status) {
		((EMAImageMessageBody)emaObject).setThumbnailDownloadStatus(EMADownloadStatus.valueOf(status.name()));
    }

	void setOriginalLocalPath(String originalLocalPath) {
		((EMAImageMessageBody)emaObject).setOriginalLocalPath(originalLocalPath);
	}

	String getOriginalLocalPath() {
		return ((EMAImageMessageBody)emaObject).getOriginalLocalPath();
	}
}
