package io.agora.cloud;

import android.content.Context;
import android.net.Uri;
import android.os.NetworkOnMainThreadException;
import android.os.ParcelFileDescriptor;
import android.text.TextUtils;
import android.util.Pair;

import android.support.annotation.NonNull;

import io.agora.Error;
import io.agora.chat.ChatClient;
import io.agora.chat.adapter.EMARHttpCallback;
import io.agora.exceptions.ChatException;
import io.agora.util.FileHelper;
import io.agora.util.EMLog;
import io.agora.util.EMPrivateConstant;
import io.agora.util.NetUtils;

import java.io.DataOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.HttpURLConnection;
import java.util.HashMap;
import java.util.Map;

public class HttpClientManager {
    private static final String TAG = "HttpClientManager";
    public static String Method_GET = "GET";
    public static String Method_POST = "POST";
    public static String Method_PUT = "PUT";
    public static String Method_DELETE = "DELETE";
    public static final int max_retries_times_on_connection_refused = 3;
    private static final int max_retry_times_on_connection_refused = 20;

    private static volatile long retrivedTokenTime = 0;
    private static final int retriveInterval = 2 * 60 * 1000; //2 minute interval
    private static int HIGH_SPEED_DOWNLOAD_BUF_SIZE = 1024*30;
    private static final int REQUEST_FAILED_CODE = 408;
    private static volatile boolean isRetring = false;

    /**
     * get请求，一般用于查询获取数据
     * @param url
     * @param headers
     * @return
     * @throws ChatException
     */
    public static Pair<Integer, String> sendGetRequest(String url, Map<String, String> headers) throws ChatException {
        return sendRequestWithToken(url, null, headers, Method_GET);
    }

    /**
     * post请求，一般用于创建的动作
     * @param url
     * @param body
     * @param headers
     * @return
     * @throws ChatException
     */
    public static Pair<Integer, String> sendPostRequest(String url, String body, Map<String, String> headers) throws ChatException {
        return sendRequestWithToken(url, body, headers, Method_POST);
    }

    /**
     * put请求，一般用于update操作
     * @param url
     * @param body
     * @param headers
     * @return
     * @throws ChatException
     */
    public static Pair<Integer, String> sendPutRequest(String url, String body, Map<String, String> headers) throws ChatException {
        return sendRequestWithToken(url, body, headers, Method_PUT);
    }

    /**
     * 上传单个文件
     * @param localFilePath
     * @param remoteFilePath
     * @param filename
     * @param headers
     * @param responseBuf
     * @param callback
     * @return
     */
    public static int uploadFile(final String localFilePath, final String remoteFilePath,int fpaPort, final String filename,
                          final Map<String, String> headers, StringBuilder responseBuf, final EMARHttpCallback callback) {
        int code = new Builder(ChatClient.getInstance().getContext())
                .uploadFile(localFilePath)
                .setUrl(remoteFilePath)
                .setFpaPort(fpaPort)
                .setFilename(filename)
                .setConnectTimeout(30000)
                .setHeaders(headers)
                .setHeader("app", ChatClient.getInstance().getOptions().getAppKey())
                .setHeader("id", ChatClient.getInstance().getCurrentUser())
                .execute(responseBuf, callback);
        EMLog.d(TAG, "upload code = "+code);
        return code;
    }

    /**
     * 下载文件
     * @param remoteUrl
     * @param localFilePath
     * @param headers
     * @param callback
     * @return
     */
    public static int downloadFile(String remoteUrl,final String localFilePath, final Map<String, String> headers, EMARHttpCallback callback) {
        return new HttpClientManager.Builder(ChatClient.getInstance().getContext())
                .downloadFile(localFilePath)
                .setConnectTimeout(30000)
                .setUrl(remoteUrl)
                .setHeaders(headers)
                .execute(null, callback);

        //return HttpClientConfig.download(remoteUrl, localFilePath, headers, callback);
    }

    public static int downloadFile(String remoteUrl,int  fpaPort, final String localFilePath, final Map<String, String> headers, StringBuilder responseBuf,EMARHttpCallback callback) {
        return new HttpClientManager.Builder(ChatClient.getInstance().getContext())
                .downloadFile(localFilePath)
                .setConnectTimeout(30000)
                .setUrl(remoteUrl)
                .setFpaPort(fpaPort)
                .setHeaders(headers)
                .execute(responseBuf, callback);
    }

    /**
     * delete请求，一般用于执行删除操作
     * @param url
     * @param headers
     * @return
     * @throws ChatException
     */
    public static Pair<Integer, String> sendDeleteRequest(String url, Map<String, String> headers) throws ChatException {
        return sendRequestWithToken(url, null, headers, Method_DELETE);
    }

    public static Pair<Integer, String> sendRequestWithToken(String url, String body, String method) throws ChatException {
        Map<String, String> headers = new HashMap<String, String>();
        headers.put("Authorization", "Bearer " + ChatClient.getInstance().getOptions().getAccessToken());
        try {
            return sendHttpRequestWithRetryToken(url,headers,body,method);
        } catch (IOException e) {
            String errorMsg = " send request : " + url + " failed!";
            if(e != null && e.toString() != null){
                errorMsg = e.toString();
            }
            EMLog.d(TAG, errorMsg);
            throw new ChatException(Error.GENERAL_ERROR,errorMsg);
        }
    }

    static Pair<Integer, String> sendRequestWithToken(String url, String body, Map<String, String> headers, String method) throws ChatException {
        if(headers == null) {
            headers = new HashMap<>();
        }
        //如果没有添加token，则将token加入到请求头中
        if(TextUtils.isEmpty(headers.get("Authorization"))) {
            headers.put("Authorization", "Bearer " + ChatClient.getInstance().getOptions().getAccessToken());
        }
        try {
            return sendHttpRequestWithRetryToken(url,headers,body,method);
        } catch (IOException e) {
            String errorMsg = " send request : " + url + " failed!";
            if(e != null && e.toString() != null){
                errorMsg = e.toString();
            }
            EMLog.d(TAG, errorMsg);
            throw new ChatException(Error.GENERAL_ERROR,errorMsg);
        }
    }

    /**
     * send http request with retry get token if it was overdue
     * @param reqURL
     * @param headers
     * @param body
     * @param method
     * @return return a pair which contains int statusCode and string response content
     * @throws ChatException
     * @throws IOException
     */
    static Pair<Integer, String> sendHttpRequestWithRetryToken(final String reqURL, final Map<String, String> headers, final String body, final String method) throws ChatException, IOException{
        return sendRequest(reqURL, headers, body, method);
    }

    public static Pair<Integer,String> sendRequest(final String reqURL, final Map<String, String> headers, final String body, final String method) throws IOException, ChatException{
        Pair<Integer,String> value = null;
        HttpResponse response = new Builder(ChatClient.getInstance().getContext())
                                            .setRequestMethod(method)
                                            .setUrl(reqURL)
                                            .setHeaders(headers)
                                            .setParams(body)
                                            .execute();

        if (response != null) {
            value = new Pair<Integer, String>(response.code, response.content);
        }
        return value;
    }

    public static HttpResponse httpExecute(String reqURL, Map<String, String> headers, String body, String method, int timeout) throws IOException{
        return new Builder(ChatClient.getInstance().getContext())
                        .setRequestMethod(method)
                        .setUrl(reqURL)
                        .setConnectTimeout(timeout)
                        .setHeaders(headers)
                        .withToken(false)
                        .setParams(body)
                        .execute();
    }

    public static HttpResponse httpExecute(String reqURL,int fpaPort, Map<String, String> headers, String body, String method, int timeout) throws IOException{
        return new Builder(ChatClient.getInstance().getContext())
                        .setRequestMethod(method)
                        .setUrl(reqURL)
                        .setFpaPort(fpaPort)
                        .setConnectTimeout(timeout)
                        .setHeaders(headers)
                        .withToken(false)
                        .setParams(body)
                        .execute();
    }

    public static HttpResponse httpExecute(String reqURL, Map<String, String> headers, String body, String method) throws IOException{
        int timeout = HttpClientConfig.getTimeout(headers);
        return httpExecute(reqURL, headers, body, method, timeout);
    }

    public static HttpResponse httpExecute(String reqURL,int fpaPort, Map<String, String> headers, String body, String method) throws IOException{
        int timeout = HttpClientConfig.getTimeout(headers);
        return httpExecute(reqURL,fpaPort, headers, body, method, timeout);
    }

    static String getNewHost(final String reqUrl, final String baseUrl) {
        return HttpClientConfig.getNewHost(reqUrl, baseUrl);
    }

    public static class Builder {
        private final HttpClientController.HttpParams p;

        public Builder() {
            this(ChatClient.getInstance().getContext());
        }

        public Builder(Context context) {
            p = new HttpClientController.HttpParams(context);
        }

        public Builder get() {
            p.mRequestMethod = "GET";
            return this;
        }

        public Builder post() {
            p.mRequestMethod = "POST";
            return this;
        }

        public Builder put() {
            p.mRequestMethod = "PUT";
            return this;
        }

        public Builder delete() {
            p.mRequestMethod = "DELETE";
            return this;
        }

        public Builder uploadFile(String fileLocalPath) {
            p.mRequestMethod = "POST";
            p.mLocalFileUri = fileLocalPath;
            p.isUploadFile = true;
            return this;
        }

        public Builder uploadFile(String fileLocalPath, String filename, String fileKey) {
            p.mRequestMethod = "POST";
            p.mLocalFileUri = fileLocalPath;
            p.mFilename = filename;
            p.mFileKey = fileKey;
            p.isUploadFile = true;
            return this;
        }

        public Builder downloadFile(String downloadPath) {
            p.mRequestMethod = "GET";
            p.mDownloadPath = downloadPath;
            p.isDownloadFile = true;
            return this;
        }

        public Builder setRequestMethod(@NonNull String requestMethod) {
            p.mRequestMethod = requestMethod;
            return this;
        }

        public Builder setUrl(@NonNull String url) {
            p.mUrl = url;
            return this;
        }

        public Builder setUrl(@NonNull String url, int port) {
            p.mUrl = url;
            p.mPort = port;
            return this;
        }
        public Builder setFpaPort(int fpaPort){
            p.fpaPort=fpaPort;
            return this;
        }

        public Builder setConnectTimeout(int timeout) {
            p.mConnectTimeout = timeout;
            return this;
        }

        public Builder setReadTimeout(int timeout) {
            p.mReadTimeout = timeout;
            return this;
        }

        public Builder setHeader(String key, String value) {
            p.mHeaders.put(key, value);
            return this;
        }

        public Builder setHeaders(Map<String, String> headers) {
            p.mHeaders.putAll(headers);
            return this;
        }

        public Builder setParam(String key, String value) {
            p.mParams.put(key, value);
            return this;
        }

        public Builder setParams(Map<String, String> params) {
            p.mParams.putAll(params);
            return this;
        }

        public Builder setParams(String params) {
            p.mParamsString = params;
            return this;
        }

        public Builder isCanRetry(boolean canRetry) {
            p.canRetry = canRetry;
            return this;
        }

        public Builder setRetryTimes(int retryTimes) {
            p.canRetry = true;
            p.mRetryTimes = retryTimes;
            return this;
        }

        public Builder setLocalFilePath(String fileUri) {
            p.mLocalFileUri = fileUri;
            return this;
        }

        public Builder setFilename(String filename) {
            p.mFilename = filename;
            return this;
        }

        public Builder setFileKey(String fileKey) {
            p.mFileKey = fileKey;
            return this;
        }

        public Builder setDownloadPath(String path) {
            p.mDownloadPath = path;
            return this;
        }

        public Builder checkAndProcessSSL(boolean checkSSL) {
            p.isCheckSSL = checkSSL;
            return this;
        }

        public Builder withToken(boolean withToken) {
            p.isNotUseToken = !withToken;
            return this;
        }

        public Builder followRedirect(boolean followRedirect) {
            p.followRedirect = followRedirect;
            return this;
        }

        public HttpClientController build() throws IOException {
            HttpClientController controller = new HttpClientController(p.mContext);
            p.apply(controller);
            return controller;
        }

        /**
         * 私有的，用于其他执行方法进一步调用
         * @param callback
         * @return
         * @throws IOException
         */
        private HttpResponse executePrivate(HttpCallback callback) throws Exception {
            HttpResponse response = null;
            HttpClientController controller = null;
            try {
                controller = build();
                HttpURLConnection connect = controller.connect();
                boolean isConnectionReset = false;
                if(connect.getDoOutput()) {
                    DataOutputStream out = new DataOutputStream(connect.getOutputStream());
                    controller.addParams(p.mParamsString, out);
                    controller.addParams(p.mParams, out);

                    try {
                        p.addFile(controller, out, callback);
                    } catch (IOException e) {
                        if(!TextUtils.isEmpty(e.getMessage()) && e.getMessage().contains("Connection reset")) {
                            isConnectionReset = true;
                        }else {
                            throw e;
                        }
                    } finally {
                        if(out != null) {
                            out.close();
                        }
                    }
                }
                response = p.getResponse(controller);
                //EMLog.d(TAG, response.toString());

                if(isConnectionReset && response.code != 413) {
                    String message = "Connection reset but not 413";
                    if(callback != null) {
                        callback.onError(response.code, message);
                    }
                    response.content = message;
                }
                if(response.code==401) {
                    ChatClient.getInstance().notifyTokenExpired(response.content);
                }
                if(response.code == HttpURLConnection.HTTP_OK) {
                    if (p.isDownloadFile) {
                        return download(response, callback);
                    }
                    if (callback != null) {
                        callback.onSuccess(response.content);
                    }
                }else {
                    if(callback != null) {
                        callback.onError(response.code, response.content);
                    }
                }
            } catch (NetworkOnMainThreadException | IOException e) {
                EMLog.e(TAG, "exception: "+ e.getClass().getSimpleName() + " error message = "+e.getMessage());
                throw e;
            } catch (IllegalStateException e) {
                EMLog.e(TAG, "error message = "+e.getMessage());
                throw e;
            }
            return response;
        }

        /**
         * 用于{@link io.agora.chat.adapter.EMARHttpAPI#upload(String, String, String, Map, int, StringBuilder, EMARHttpCallback)}调用
         * @param responseBuf
         * @param callback
         * @return
         */
        public int execute(StringBuilder responseBuf, EMARHttpCallback callback) {
            int code = 408;
            HttpResponse response = null;
            try {
                response = executePrivate(new HttpCallback() {
                    @Override
                    public void onSuccess(String result) {

                    }

                    @Override
                    public void onError(int code, String msg) {

                    }

                    @Override
                    public void onProgress(long total, long pos) {
                        if(callback != null) {
                            callback.onProgress(total, pos);
                        }
                    }
                });
                code = response.code;
                try {
                    if(responseBuf != null) {
                        responseBuf.append(response.content);
                    }
                } catch (Exception e) {
                    EMLog.e(TAG, "json parse exception remotefilepath:" + p.mUrl);
                }
                return code;
            } catch (Exception e) {
                //e.printStackTrace();
                String message = (e != null && e.getMessage() != null)?e.getMessage():"failed to upload the files";
                EMLog.e(TAG, "error asyncExecute:" + message);
                if(message.toLowerCase().contains(EMPrivateConstant.CONNECTION_REFUSED) && NetUtils.hasNetwork(p.mContext)) {
                    if(p.canRetry && p.mRetryTimes > 0){
                        String baseUrl = EMHttpClient.getInstance().chatConfig().getNextAvailableBaseUrl();
                        p.mUrl = getNewHost(p.mUrl, baseUrl);
                        --p.mRetryTimes;
                        EMLog.d(TAG, "重试中。。。");
                        return execute(responseBuf, callback);
                    }
                }
                try {
                    if(responseBuf != null) {
                        responseBuf.append(message);
                    }
                } catch (Exception e1) {
                }
                if(response != null) {
                    return response.code;
                }
                EMLog.e(TAG, e.getMessage());
            }
            return code;
        }

        public HttpResponse execute() {
            return execute(null);
        }

        public HttpResponse execute(HttpCallback callback) {
            if(p.isUploadFile || p.isDownloadFile) {
                return executeFile(callback);
            }else {
                return executeNormal(callback);
            }
        }

        public void asyncExecute(HttpCallback callback) {
            if(p.isUploadFile || p.isDownloadFile) {
                asyncExecuteFile(callback);
            }else {
                asyncExecuteNormal(callback);
            }
        }

        private void asyncExecuteNormal(HttpCallback callback) {
            new Thread(){
                public void run(){
                    executeNormal(callback);
                }
            }.start();
        }

        /**
         * 执行一般请求
         * @param callback
         * @return
         */
        private HttpResponse executeNormal(HttpCallback callback) {
            HttpResponse response = null;
            try {
                response = executePrivate(callback);
                if(response.code != HttpURLConnection.HTTP_OK) {
                    if(p.canRetry && p.mRetryTimes > 0) {
                        p.mRetryTimes--;
                        return executeNormal(callback);
                    }
                }
                return response;
            } catch (Exception e) {
                //e.printStackTrace();
                String message = (e != null && e.getMessage() != null)?e.getMessage():"failed to request";
                EMLog.e(TAG, "error execute:" + message);
                if(p.canRetry && p.mRetryTimes > 0) {
                    p.mRetryTimes--;
                    return executeNormal(callback);
                }
                if(response == null) {
                    response = new HttpResponse();
                }
                if(response.code == 0) {
                    response.code = REQUEST_FAILED_CODE;
                }
                response.content = message;
                if(callback != null){
                    callback.onError(response.code, message);
                }
                return response;
            }
        }

        /**
         * 上传或者下载文件
         * @param callback
         * @return
         */
        private HttpResponse executeFile(HttpCallback callback) {
            HttpResponse response = null;
            try {
                response = executePrivate(callback);
                int code = response.code;
                String msg = "";
                switch (code) {
                    case HttpURLConnection.HTTP_UNAUTHORIZED :
                        long tokenSaveTime = EMHttpClient.getInstance().chatConfig().getTokenSaveTime();
                        if ((System.currentTimeMillis() - tokenSaveTime) <= 10 * 60 * 1000) {
                            msg = "unauthorized file";
                            if (callback != null) {
                                callback.onError(code, msg);
                            }
                            response.content = msg;
                        }
                        if(p.canRetry && p.mRetryTimes > 0){
                            String baseUrl = EMHttpClient.getInstance().chatConfig().getNextAvailableBaseUrl();
                            p.mUrl = getNewHost(p.mUrl, baseUrl);
                            p.mRetryTimes--;
                            return executeFile(callback);
                        }
                        return response;
                }
                return response;
            } catch (Exception e) {
                //e.printStackTrace();
                String message = (e != null && e.getMessage() != null)?e.getMessage(): (p.isUploadFile ? "failed to upload the file" : "failed to download file");
                EMLog.e(TAG, "error execute:" + message);
                if(message.toLowerCase().contains(EMPrivateConstant.CONNECTION_REFUSED) && NetUtils.hasNetwork(p.mContext)) {
                    if(!p.isDefaultRetry) {
                        p.isDefaultRetry = true;
                        p.mRetryTimes = max_retry_times_on_connection_refused;
                        p.canRetry = true;
                        String baseUrl = EMHttpClient.getInstance().chatConfig().getNextAvailableBaseUrl();
                        p.mUrl = getNewHost(p.mUrl, baseUrl);
                        return executeFile(callback);
                    }
                    if(p.canRetry && p.mRetryTimes > 0){
                        String baseUrl = EMHttpClient.getInstance().chatConfig().getNextAvailableBaseUrl();
                        p.mUrl = getNewHost(p.mUrl, baseUrl);
                        p.mRetryTimes--;
                        return executeFile(callback);
                    }
                }
                if(response == null) {
                    response = new HttpResponse();
                }
                if(response.code == 0) {
                    response.code = REQUEST_FAILED_CODE;
                }
                response.content = message;
                if (callback != null) {
                    callback.onError(response.code, response.content);
                }
                return response;
            }
        }

        /**
         *
         * @param callback
         */
        private void asyncExecuteFile(HttpCallback callback) {
            new Thread(){
                public void run(){
                    executeFile(callback);
                }
            }.start();
        }

        private HttpResponse download(HttpResponse response, HttpCallback callback) throws IOException, IllegalStateException {
            String filePath = FileHelper.getInstance().getFilePath(p.mDownloadPath);

            if(!TextUtils.isEmpty(filePath)) {
                File file = new File(filePath);
                if (!file.getParentFile().exists()) {
                    file.getParentFile().mkdirs();
                }
            }
            long size = 0;
            String message = "";
            if(p.mDownloadPath.startsWith("content")) {
                size = onDownloadCompleted(response, FileHelper.getInstance().formatInUri(p.mLocalFileUri), callback);
            }else {
                size = onDownloadCompleted(response, filePath, callback);
            }
            if(size <= 0) {
                message = "downloaded content size is zero!";
                if(callback != null) {
                    callback.onError(REQUEST_FAILED_CODE, message);
                }
                response.code = REQUEST_FAILED_CODE;
                response.content = message;
                return response;
            }
            message = "download successfully";
            if (callback != null) {
                EMLog.e(TAG, "download successfully");
                callback.onSuccess(message);
            }
            return response;
        }

    }

    private static long onDownloadCompleted(HttpResponse response, final String localFilePath, HttpCallback callback) throws IOException, IllegalStateException{
        InputStream input = null;
        FileOutputStream output = null;

        int count = 0;
        int current_progress = 0;

        // get file length
        long fileLength = response.contentLength;

        input = response.inputStream;
        int available = input.available();
        EMLog.d(TAG, "inputStream length = "+ available);

        File outputFile = new File(localFilePath);

        try {
            output = new FileOutputStream(localFilePath);
        } catch (IOException e) {
            EMLog.e(TAG, e.getMessage());
            input.close();
            throw e;
        }

        int bufSize = HIGH_SPEED_DOWNLOAD_BUF_SIZE;
        byte[] buffer = new byte[bufSize];
        long total = 0;

        try {
            while ((count = input.read(buffer)) != -1) {
                total += count;
                int progress = (int) ((total * 100) / fileLength);
                EMLog.d(TAG, progress + "");
                if (progress == 100 || progress > current_progress + 5) {
                    current_progress = progress;
                    if (callback != null) {
                        callback.onProgress(fileLength, total);
                    }
                }
                output.write(buffer, 0, count);
            }

            return outputFile.length();
        } catch (IOException e) {
            EMLog.e(TAG, e.getMessage());
            throw e;
        } finally{
            output.close();
            input.close();
        }
    }

    private static long onDownloadCompleted(HttpResponse response, final Uri localFilePath, HttpCallback callback) throws IOException, IllegalStateException{

        InputStream input = null;
        OutputStream output = null;

        int count = 0;
        int current_progress = 0;

        // get file length
        long fileLength = response.contentLength;

        input = response.inputStream;

        ParcelFileDescriptor parcelFileDescriptor = null;
        try {
            parcelFileDescriptor = ChatClient.getInstance().getContext().getContentResolver().openFileDescriptor(localFilePath, "w");
            output = new FileOutputStream(parcelFileDescriptor.getFileDescriptor());
        } catch (Exception e) {
            EMLog.e(TAG, e.getMessage());
            input.close();
            throw e;
        }

        int bufSize = HIGH_SPEED_DOWNLOAD_BUF_SIZE;
        byte[] buffer = new byte[bufSize];
        long total = 0;

        try {
            while ((count = input.read(buffer)) != -1) {
                total += count;
                int progress = (int) ((total * 100) / fileLength);
                EMLog.d(TAG, progress + "");
                if (progress == 100 || progress > current_progress + 5) {
                    current_progress = progress;
                    if (callback != null){
                        callback.onProgress(fileLength, total);
                    }
                }
                output.write(buffer, 0, count);
                EMLog.d(TAG, "执行写入操作 count = "+count);
            }
            //最后判断文件是否存在？
            boolean fileExistByUri = FileHelper.getInstance().isFileExist(localFilePath);
            EMLog.d(TAG, "download by uri fileExistByUri = "+fileExistByUri);
            return input.available();
        } catch (IOException e) {
            EMLog.e(TAG, e.getMessage());
            throw e;
        } finally{
            if(output != null) {
                output.close();
            }
            if(input != null) {
                input.close();
            }
        }
    }

}
