/************************************************************
 *  * EaseMob CONFIDENTIAL 
 * __________________ 
 * Copyright (C) 2013-2014 EaseMob Technologies. All rights reserved. 
 *  
 * NOTICE: All information contained herein is, and remains 
 * the property of EaseMob Technologies.
 * Dissemination of this information or reproduction of this material 
 * is strictly forbidden unless prior written permission is obtained
 * from EaseMob Technologies.
 */
package io.agora.cloud;

import android.content.Context;
import android.text.TextUtils;

import io.agora.chat.ChatManager;
import io.agora.chat.ChatClient;
import io.agora.util.EMLog;

import java.io.File;
import java.util.Map;


/**
 * 此类是内部类，请谨慎调用
 * 相关API可以从ChatManager找到
 * downloadFile ->  ChatManager.downloadFile
 * 
 * Use HTTP POST to upload file to cloud storage. also support download and delete file
 * the operations here are synchronized, user need to run in asynctask if needed
 */
class HttpFileManager extends CloudFileManager {
	private Context appContext;
	private static final int max_retry_times_on_connection_refused = 20;
	
	public HttpFileManager(){
		appContext = ChatClient.getInstance().getContext();
	}
	
	public HttpFileManager(Context appContext, String userServerUrl) {
		this.appContext = appContext.getApplicationContext();
	}
	
	public HttpFileManager(Context context) {
		this.appContext = context.getApplicationContext();
	}

	private void sendFileToServerHttp(final String localFilePath, final String remoteFilePath, final String appId, final String jid,
									  Map<String, String> headers, final CloudOperationCallback listener){
		 sendFileToServerHttpWithCountDown(localFilePath,remoteFilePath,appId,jid,headers,listener,-1,false);
	}
	
	/**
	 * send file using http client
	 */
	private void sendFileToServerHttpWithCountDown(final String localFilePath, final String remoteFilePath, final String appId, final String jid,
												   final Map<String, String> headers, final CloudOperationCallback listener, int countDown, boolean started){

		EMLog.d(TAG, "sendFileToServerHttpWithCountDown .....");
		
		File sourceFile = new File(localFilePath);
		if (!sourceFile.isFile()) {
			EMLog.e(TAG, "Source file doesn't exist");
			listener.onError("Source file doesn't exist");
			return;
		}
		int timeout = HttpClientConfig.getTimeout(headers);

		new HttpClientManager.Builder(appContext)
				.uploadFile(localFilePath)
				.setUrl(remoteFilePath)
				.setConnectTimeout(timeout)
				.setParam("app", appId)
				.setParam("id", jid)
				.setHeaders(headers)
				.setRetryTimes(countDown)
				.execute(new HttpCallback() {
					@Override
					public void onSuccess(String result) {
						if(listener != null) {
						    listener.onSuccess(result);
						}
					}

					@Override
					public void onError(int code, String msg) {
						if(listener != null) {
						    listener.onError(msg);
						}
					}

					@Override
					public void onProgress(long total, long pos) {
						if(listener != null) {
						    listener.onProgress((int) (pos * 100 / total));
						}
					}
				});

	}

	@Override
	public void uploadFileInBackground(final String localFilePath, final String remoteFilePath, final String appId, final String jid,
			final Map<String, String> headers, final CloudOperationCallback callback) {
		new Thread() {
			public void run() {
				try {
					sendFileToServerHttp(localFilePath, remoteFilePath, appId, jid, headers, callback);
				} catch (Exception e) {
					if(e!=null && e.toString() != null){
						EMLog.e(TAG, e.toString());
						
						callback.onError(e.toString());
					}else{
						callback.onError("failed to upload the file : " + localFilePath + " remote path : " + remoteFilePath);   	
					}
				}

			}
		}.start();
	}

	/**
	 * upload file synchronized
	 * 
	 * @param localFilePath
	 * @param remoteFilePath
	 * @param appId
	 * @param jid
	 * @param headers
	 * @param callback
	 */
	public void uploadFile(final String localFilePath, final String remoteFilePath, final String appId, final String jid,
			final Map<String, String> headers, final CloudOperationCallback callback) {
		try {
			sendFileToServerHttp(localFilePath, remoteFilePath, appId, jid, headers, callback);
		} catch (Exception e) {
			EMLog.e(TAG, "uploadFile error:" + e.toString());
			
			callback.onError(e.toString());
		}
	}

	/**
	 * download a remote file to local storage.
	 * 
	 * 由于此类不再对外开放
	 * 
	 * 请用 {@link ChatManager downloadFile}代替
	 * 
	 * @param remoteFilePath
	 *            remote file to download
	 * @param localFilePath
	 *            local file path to store
	 * @param appId
	 * @param headers
	 * @param callback
	 *            operation callback
	 *  @deprecated 已过时,使用 {@link HttpFileManager#downloadFile(String, String, Map, CloudOperationCallback)} 替代
	 *
	 */
	public void downloadFile(final String remoteFilePath, final String localFilePath, final String appId,
			final Map<String, String> headers, final CloudOperationCallback callback) {
		if (TextUtils.isEmpty(remoteFilePath)) {
			if (callback != null)
				callback.onError("remote file path is null or empty");
			EMLog.e(TAG, "remote file path is null or empty");
		}else{
			String remoteUrl = HttpClientConfig.getFileRemoteUrl(remoteFilePath);
			downloadFile(remoteUrl, localFilePath, headers, callback);
		}
	}
	
	/**
	 * 由于此类不再对外开放
	 * 
	 * 请用 {@link ChatManager downloadFile}代替
	 * @param remoteUrl
	 * @param localFilePath
	 * @param headers
	 * @param callback
	 */
	public void downloadFile(String remoteUrl, final String localFilePath, final Map<String, String> headers,
			final CloudOperationCallback callback){
		try{
			downloadFileWithCountDown(remoteUrl,localFilePath,headers,callback,max_retry_times_on_connection_refused);
		}catch(Exception e){
			String errorMsg = "failed to download file : " + remoteUrl;
			
			if(e != null && e.toString() != null){
				errorMsg = e.toString();
			}
			
			if(callback != null){
				callback.onError(errorMsg);
			}
		}
	}
	
	/**
	 * download a remote file to local storage.
	 * @param remoteUrl remote file to download
	 * @param localFilePath local file path to store
	 * @param _headers  http headers
	 * @param callback operation listener
	 */
	private void downloadFileWithCountDown(String remoteUrl, final String localFilePath, final Map<String, String> _headers,
			final CloudOperationCallback callback, int countDown) {
		
		if(remoteUrl == null || remoteUrl.length() <= 0){
			callback.onError("invalid remoteUrl");
			return;
		}
		
		int timeout = HttpClientConfig.getTimeout(_headers);

		remoteUrl = HttpClientConfig.getFileRemoteUrl(remoteUrl);

		EMLog.d(TAG, "remoteUrl:" + remoteUrl + " localFilePath:" + localFilePath);

		new HttpClientManager.Builder(appContext)
				.downloadFile(localFilePath)
				.setUrl(remoteUrl)
				.setConnectTimeout(timeout)
				.setHeaders(_headers)
				.setRetryTimes(countDown)
				.execute(new HttpCallback() {
					@Override
					public void onSuccess(String result) {
						if(callback != null) {
						    callback.onSuccess(result);
						}
					}

					@Override
					public void onError(int code, String msg) {
						if(callback != null) {
						    callback.onError(msg);
						}
					}

					@Override
					public void onProgress(long total, long pos) {
						if(callback != null) {
						    callback.onProgress((int) (pos * 100 / total));
						}
					}
				});
		
	}

}
