package io.agora.util;

import android.content.Context;
import android.net.Uri;
import android.text.TextUtils;

import io.agora.chat.ChatClient;
import io.agora.chat.ChatOptions;

import java.io.File;

/**
 * \~english
 * File help class, to determine whether the file exists, get the absolute path to the file,
 * get the file name, get the file length and get the file type and other methods
 * The following custom logic can be implemented：
 * （1）Determine whether the file exists {@link IFilePresenter#isFileExist(Context, Uri)}
 *  (2) Get the file name {@link IFilePresenter#getFilename(Context, Uri)}
 *  (3) Get the absolute path of file {@link IFilePresenter#getFilePath(Context, Uri)}
 *  (4) Get file length {@link IFilePresenter#getFileLength(Context, Uri)}
 *  (5) Get file type {@link IFilePresenter#getFileMimeType(Context, Uri)}
 * Implement {@link IFilePresenter} by the method of {@link #setFileHelper(IFilePresenter)}
 *
 * Note: Should Calls this after {@link ChatClient#init(Context, ChatOptions)}, otherwise the context may be null
 */
public class FileHelper {
    private static class FileHelperInstance {
        private static final FileHelper instance = new FileHelper();
    }

    private Context mContext;
    private IFilePresenter mHelper;

    private FileHelper() {
        mContext = ChatClient.getInstance().getContext();
        mHelper = new FilePresenterImpl();
    }

    public static FileHelper getInstance() {
        return FileHelperInstance.instance;
    }

    /**
     * \~english
     * Set custom FilePresenter
     * @param presenter
     */
    public void setFileHelper(IFilePresenter presenter) {
        this.mHelper = presenter;
    }

    /**
     * \~english
     * Whether the file exists, call it after {@link ChatClient#init(Context, ChatOptions)}
     * @param fileUri   File's Uri
     * @return  Whether the file exists
     */
    public boolean isFileExist(Uri fileUri) {
        return mHelper.isFileExist(mContext, fileUri);
    }

    /**
     * \~english
     * Whether the file exists, call it after {@link ChatClient#init(Context, ChatOptions)}
     * @param stringUri File's path, which may be the absolute path of file or the string from of Uri
     * @return  Whether the file exists
     */
    public boolean isFileExist(String stringUri) {
        if (TextUtils.isEmpty(stringUri)) {
            return false;
        }
        return isFileExist(Uri.parse(stringUri));
    }

    /**
     * \~english
     * Determine whether the file exists
     * @param context
     * @param fileUri   File's Uri
     * @return  Whether the file exists
     */
    public boolean isFileExist(Context context, Uri fileUri) {
        return mHelper.isFileExist(context, fileUri);
    }

    /**
     * \~english
     * Determine whether the file exists
     * @param context
     * @param stringUri   File's path, which may be the absolute path of file or the string from of Uri
     * @return  Whether the file exists
     */
    public boolean isFileExist(Context context, String stringUri) {
        if (TextUtils.isEmpty(stringUri)) {
            return false;
        }
        return isFileExist(context, Uri.parse(stringUri));
    }

    /**
     * \~english
     * Get file name
     * Note：Calls after {@link ChatClient#init(Context, ChatOptions)}
     * @param fileUri   File's Uri
     * @return  File name
     */
    public String getFilename(Uri fileUri) {
        return mHelper.getFilename(mContext, fileUri);
    }

    /**
     * \~english
     * Get file name
     * Note：Calls after {@link ChatClient#init(Context, ChatOptions)}
     * @param filePath   File's path, which may be the absolute path of file or the string from of Uri
     * @return  File name
     */
    public String getFilename(String filePath) {
        if (TextUtils.isEmpty(filePath)) {
            return "";
        }
        return getFilename(Uri.parse(filePath));
    }

    /**
     * \~english
     * Get file's absolute path
     * Note：Calls after {@link ChatClient#init(Context, ChatOptions)}
     * @param fileUri   File's Uri
     * @return  File's absolute path
     */
    public String getFilePath(Uri fileUri) {
        return mHelper.getFilePath(mContext, fileUri);
    }

    /**
     * \~english
     * Get file's absolute path
     * Note：Calls after {@link ChatClient#init(Context, ChatOptions)}
     * @param filePath   File's path, which may be the absolute path of file or the string from of Uri
     * @return  File's absolute path
     */
    public String getFilePath(String filePath)
    {
        if (TextUtils.isEmpty(filePath)) {
            return filePath;
        }
        return getFilePath(Uri.parse(filePath));
    }

    /**
     * \~english
     * Get file's absolute path
     * @param context
     * @param fileUri   File's Uri
     * @return  File's absolute path
     */
    public String getFilePath(Context context, Uri fileUri) {
        return mHelper.getFilePath(context, fileUri);
    }

    /**
     * \~english
     * Get file's absolute path
     * @param context
     * @param filePath   File's path, which may be the absolute path of file or the string from of Uri
     * @return  File's absolute path
     */
    public String getFilePath(Context context, String filePath)
    {
        if (TextUtils.isEmpty(filePath)) {
            return filePath;
        }
        return getFilePath(context, Uri.parse(filePath));
    }

    /**
     * \~english
     * Get file length
     * Note：Calls after {@link ChatClient#init(Context, ChatOptions)}
     * @param fileUri   File's Uri
     * @return  File length
     */
    public long getFileLength(Uri fileUri) {
        if(fileUri == null) {
            return 0;
        }
        return mHelper.getFileLength(mContext, fileUri);
    }

    /**
     * \~english
     * Get file length
     * Note：Calls after {@link ChatClient#init(Context, ChatOptions)}
     * @param filePath   File's path, which may be the absolute path of file or the string from of Uri
     * @return  File length
     */
    public long getFileLength(String filePath) {
        if (TextUtils.isEmpty(filePath)) {
            return 0;
        }
        return getFileLength(Uri.parse(filePath));
    }

    /**
     * \~english
     * Get file mime type
     * Note：Calls after {@link ChatClient#init(Context, ChatOptions)}
     * @param fileUri   File's Uri
     * @return  File mime type
     */
    public String getFileMimeType(Uri fileUri) {
        return mHelper.getFileMimeType(mContext, fileUri);
    }

    /**
     * 删除文件（可以获取到绝对路径）
     * SDk内部调用，不建议外部使用
     * 注：需要在{@link ChatClient#init(Context, ChatOptions)}后调用
     * @param filePath
     * @return
     */
    public boolean deletePrivateFile(String filePath) {
        if (TextUtils.isEmpty(filePath)) {
            return false;
        }
        if (!isFileExist(filePath)) {
            return false;
        }
        filePath = getFilePath(Uri.parse(filePath));
        if (!TextUtils.isEmpty(filePath))
        {
            File file = new File(filePath);
            if (file.exists()) {
                return file.delete();
            }
        }
        return false;
    }

    /**
     * 格式化输入到SDK内部的文件的Uri，并输出未Uri
     * SDk内部调用，不建议外部使用
     * 注：需要在{@link ChatClient#init(Context, ChatOptions)}后调用
     * @param fileUri
     * @return
     */
    public Uri formatInUri(Uri fileUri) {
        if (fileUri == null) {
            return null;
        }
        if ((VersionUtils.isTargetQ(mContext)) && (UriUtils.uriStartWithContent(fileUri))) {
            return fileUri;
        }
        String path = getFilePath(fileUri);
        if (!TextUtils.isEmpty(path)) {
            fileUri = Uri.parse(path);
        }
        return fileUri;
    }

    /**
     * 格式化输入到SDK的文件，并输出未Uri
     * SDk内部调用，不建议外部使用
     * @param file
     * @return
     */
    public Uri formatInUri(File file) {
        if (file == null) {
            return null;
        }
        return Uri.parse(file.getAbsolutePath());
    }

    /**
     * 格式化输入到SDK内部的文件的Uri，并输出未Uri
     * SDk内部调用，不建议外部使用
     * 注：需要在{@link ChatClient#init(Context, ChatOptions)}后调用
     * @param filePath
     * @return
     */
    public Uri formatInUri(String filePath) {
        if (TextUtils.isEmpty(filePath)) {
            return null;
        }
        return formatInUri(Uri.parse(filePath));
    }

    /**
     * 将格式化后的Uri转为string
     * SDk内部调用，不建议外部使用
     * 注：需要在{@link ChatClient#init(Context, ChatOptions)}后调用
     * @param uri
     * @return
     */
    public String formatInUriToString(Uri uri) {
        uri = formatInUri(uri);
        if (uri == null) {
            return "";
        }
        return uri.toString();
    }

    /**
     * 格式化文件，并最终转为Uri的string样式
     * SDk内部调用，不建议外部使用
     * @param file
     * @return
     */
    public String formatInUriToString(File file) {
        Uri fileUri = formatInUri(file);
        if (fileUri == null) {
            return "";
        }
        return fileUri.toString();
    }

    /**
     * 格式化传入的路径，并最终转为Uri的string样式
     * SDk内部调用，不建议外部使用
     * 注：需要在{@link ChatClient#init(Context, ChatOptions)}后调用
     * @param filePath
     * @return
     */
    public String formatInUriToString(String filePath) {
        if (TextUtils.isEmpty(filePath)) {
            return "";
        }
        return formatInUriToString(Uri.parse(filePath));
    }

    /**
     * 格式化输出的路径，如果可以获取到绝对路径，则优先返回绝对路径
     * SDk内部调用，不建议外部使用
     * 注：需要在{@link ChatClient#init(Context, ChatOptions)}后调用
     * @param filePath
     * @return
     */
    public String formatOutLocalUrl(String filePath) {
        if (TextUtils.isEmpty(filePath)) {
            return filePath;
        }
        String path = getFilePath(filePath);
        if (!TextUtils.isEmpty(path)) {
            return path;
        }
        return filePath;
    }

    /**
     * 格式化输出的Uri
     * SDk内部调用，不建议外部使用
     * 注：需要在{@link ChatClient#init(Context, ChatOptions)}后调用
     * @param filePath
     * @return
     */
    public Uri formatOutUri(String filePath) {
        if (TextUtils.isEmpty(filePath)) {
            return null;
        }
        Uri fileUri = Uri.parse(filePath);
        if ((VersionUtils.isTargetQ(mContext)) && (UriUtils.uriStartWithContent(fileUri))) {
            return fileUri;
        }
        String str = getFilePath(fileUri);
        if (!TextUtils.isEmpty(str)) {
            fileUri = Uri.fromFile(new File(str));
        }
        return fileUri;
    }

    /**
     * \~english
     * Convert the URI to a String
     * @param fileUri   File's Uri
     * @return
     */
    public String uriToString(Uri fileUri) {
        if (fileUri == null) {
            return "";
        }
        return fileUri.toString();
    }

    public static class FilePresenterImpl implements IFilePresenter {

        @Override
        public boolean isFileExist(Context context, Uri fileUri) {
            return UriUtils.isFileExistByUri(context, fileUri);
        }

        @Override
        public String getFilename(Context context, Uri fileUri) {
            return UriUtils.getFileNameByUri(context, fileUri);
        }

        @Override
        public String getFilePath(Context context, Uri fileUri) {
            return UriUtils.getFilePath(context, fileUri);
        }

        @Override
        public long getFileLength(Context context, Uri fileUri) {
            return UriUtils.getFileLength(context, fileUri);
        }

        @Override
        public String getFileMimeType(Context context, Uri fileUri) {
            return UriUtils.getMimeType(context, fileUri);
        }
    }

    /**
     * \~english
     * Operation file interface
     */
    public interface IFilePresenter {

        /**
         * \~english
         * Determine whether file exists
         * @param context
         * @param fileUri   File's Uri
         * @return  Whether file exists
         */
        boolean isFileExist(Context context, Uri fileUri);

        /**
         * \~english
         * Get file name
         * @param context
         * @param fileUri   File's Uri
         * @return  File name
         */
        String getFilename(Context context, Uri fileUri);

        /**
         * \~english
         * Get file's absolute path
         * @param context
         * @param fileUri   File's Uri
         * @return  File's absolute path
         */
        String getFilePath(Context context, Uri fileUri);

        /**
         * \~english
         * Get file length
         * @param context
         * @param fileUri   File's Uri
         * @return  File length
         */
        long getFileLength(Context context, Uri fileUri);

        /**
         * \~english
         * Get file mime type
         * @param context
         * @param fileUri   File's Uri
         * @return  File mime type
         */
        String getFileMimeType(Context context, Uri fileUri);
    }
}

