/*
 * Decompiled with CFR 0.152.
 */
package io.agora.base.internal.video;

import android.graphics.Matrix;
import android.opengl.GLES20;
import android.os.Handler;
import android.os.HandlerThread;
import io.agora.base.TextureBuffer;
import io.agora.base.VideoFrame;
import io.agora.base.internal.ATrace;
import io.agora.base.internal.Logging;
import io.agora.base.internal.ThreadUtils;
import io.agora.base.internal.video.EglBase;
import io.agora.base.internal.video.EglBaseFactory;
import io.agora.base.internal.video.GlRectDrawer;
import io.agora.base.internal.video.GlUtil;
import io.agora.base.internal.video.YuvConverter;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.concurrent.Callable;
import java.util.concurrent.atomic.AtomicInteger;

public class TextureBufferPool {
    private static final boolean VERBOSE = false;
    private static final String TAG = TextureBufferPool.class.getSimpleName();
    private static final AtomicInteger nextSeq = new AtomicInteger(0);
    private final String name;
    private final Handler handler;
    private final boolean ownGlThread;
    private final EglBase eglBase;
    private final EglBase.Context eglContext;
    private final YuvConverter yuvConverter;
    private final GlRectDrawer drawer;
    private static final boolean SHRINK_POOL = true;
    private final int glPixelFormat;
    private final int maxBufferCnt;
    private final int keepBufferCnt;
    private final ArrayList<TextureInfo> textureInfoList = new ArrayList();
    private boolean isQuitting = false;
    private int dropCount = 0;

    public static TextureBufferPool create(final String threadName, final EglBase.Context sharedContext, final int maxBufferCnt) {
        HandlerThread thread = new HandlerThread(threadName);
        thread.start();
        final Handler handler = new Handler(thread.getLooper());
        final TextureBufferPool[] bufferPool = new TextureBufferPool[]{null};
        ThreadUtils.invokeAtFrontUninterruptibly(handler, new Runnable(){

            @Override
            public void run() {
                try {
                    EglBase newEglBase = TextureBufferPool.createEglAndMakeCurrent(sharedContext);
                    bufferPool[0] = new TextureBufferPool(threadName, maxBufferCnt, 6408, handler, true, newEglBase, null);
                }
                catch (RuntimeException e) {
                    Logging.e(TAG, threadName + " failed to initialize egl", e);
                    handler.getLooper().quit();
                }
            }
        });
        return bufferPool[0];
    }

    public static TextureBufferPool createWithinGlThread(String name, int maxBufferCnt, int glPixelFormat, Handler handler, EglBase eglBase, YuvConverter yuvConverter) {
        return new TextureBufferPool(name, maxBufferCnt, glPixelFormat, handler, false, eglBase, yuvConverter);
    }

    private static EglBase createEglAndMakeCurrent(EglBase.Context sharedContext) {
        EglBase newEglBase = EglBaseFactory.create(sharedContext, EglBase.CONFIG_PIXEL_BUFFER);
        try {
            newEglBase.createDummyPbufferSurface();
            newEglBase.makeCurrent();
        }
        catch (RuntimeException e) {
            newEglBase.release();
            throw e;
        }
        return newEglBase;
    }

    private TextureBufferPool(String name, int maxBufferCnt, int glPixelFormat, Handler handler, boolean ownGlThread, EglBase eglBase, YuvConverter yuvConverter) {
        Logging.d(TAG, name + " init buffer pool, ownGlThread: " + ownGlThread + " cnt: " + maxBufferCnt);
        this.name = name;
        this.maxBufferCnt = Math.max(maxBufferCnt, 1);
        this.keepBufferCnt = Math.min(this.maxBufferCnt, 3);
        this.glPixelFormat = glPixelFormat;
        this.handler = handler;
        this.ownGlThread = ownGlThread;
        this.eglBase = eglBase;
        this.yuvConverter = yuvConverter != null ? yuvConverter : new YuvConverter();
        this.eglContext = eglBase.getEglBaseContext();
        this.drawer = new GlRectDrawer();
    }

    public VideoFrame.TextureBuffer textureCopy(VideoFrame.TextureBuffer textureBuffer, Runnable bufferReleaseCallback) {
        if (textureBuffer == null) {
            return null;
        }
        return this.textureCopy(textureBuffer.getTextureId(), textureBuffer.getType(), textureBuffer.getWidth(), textureBuffer.getHeight(), textureBuffer.getTransformMatrix(), bufferReleaseCallback);
    }

    public VideoFrame.TextureBuffer textureCopy(final int textureId, final VideoFrame.TextureBuffer.Type type, final int width, final int height, final Matrix transformMatrix, final Runnable bufferReleaseCallback) {
        return ThreadUtils.invokeAtFrontUninterruptibly(this.handler, new Callable<VideoFrame.TextureBuffer>(){

            @Override
            public VideoFrame.TextureBuffer call() throws Exception {
                return TextureBufferPool.this.doTextureCopy(textureId, type, width, height, transformMatrix, bufferReleaseCallback);
            }
        });
    }

    public void dispose() {
        Logging.d(TAG, this.name + " dispose()");
        ThreadUtils.invokeAtFrontUninterruptibly(this.handler, new Runnable(){

            @Override
            public void run() {
                TextureBufferPool.this.isQuitting = true;
                if (!TextureBufferPool.this.anyTextureInUse()) {
                    Logging.d(TAG, TextureBufferPool.this.name + " ready to release since no more buffer in flight");
                    TextureBufferPool.this.release();
                }
            }
        });
    }

    private VideoFrame.TextureBuffer doTextureCopy(int srcTextureId, VideoFrame.TextureBuffer.Type srcType, int width, int height, Matrix transformMatrix, final Runnable bufferReleaseCallback) {
        TextureBuffer[] ret;
        final TextureInfo texInfo = this.acquireTextureFramebuffer();
        if (texInfo == null) {
            return null;
        }
        ATrace.beginSection("setupTextureFrameBuffer");
        boolean success = this.setupTextureFrameBuffer(texInfo, width, height);
        ATrace.endSection();
        if (!success) {
            this.releaseTextureFramebuffer(texInfo);
            return null;
        }
        GLES20.glBindFramebuffer((int)36160, (int)texInfo.frameBufferId);
        ATrace.beginSection("drawTexture");
        this.drawTexture(srcTextureId, srcType, width, height);
        ATrace.endSection();
        ATrace.beginSection("flush");
        GLES20.glBindFramebuffer((int)36160, (int)0);
        GLES20.glUseProgram((int)0);
        GLES20.glFlush();
        ATrace.endSection();
        ret = new TextureBuffer[]{new TextureBuffer(this.eglContext, width, height, VideoFrame.TextureBuffer.Type.RGB, texInfo.textureId, transformMatrix, this.handler, this.yuvConverter, new Runnable(){

            @Override
            public void run() {
                TextureBufferPool.this.handler.post(new Runnable(){

                    @Override
                    public void run() {
                        TextureBufferPool.this.releaseTextureFramebuffer(texInfo);
                        if (TextureBufferPool.this.isQuitting && !TextureBufferPool.this.anyTextureInUse()) {
                            Logging.d(TAG, TextureBufferPool.this.name + " ready to release since no buffer in flight");
                            TextureBufferPool.this.release();
                        }
                        if (bufferReleaseCallback != null) {
                            bufferReleaseCallback.run();
                        }
                    }
                });
            }
        }).withSequence(nextSeq.getAndIncrement())};
        return ret[0];
    }

    private TextureInfo acquireTextureFramebuffer() {
        TextureInfo texInfo = null;
        for (TextureInfo info : this.textureInfoList) {
            if (info.inUse) continue;
            texInfo = info;
            break;
        }
        if (texInfo == null) {
            if (this.textureInfoList.size() >= this.maxBufferCnt) {
                ++this.dropCount;
                ATrace.traceCounter("Drop@TexPool", this.dropCount);
                return null;
            }
            int textureId = GlUtil.generateTexture(3553);
            int[] frameBuffers = new int[1];
            GLES20.glGenFramebuffers((int)1, (int[])frameBuffers, (int)0);
            texInfo = new TextureInfo(textureId, frameBuffers[0]);
            this.textureInfoList.add(texInfo);
        }
        texInfo.inUse = true;
        return texInfo;
    }

    private void releaseTextureFramebuffer(TextureInfo texInfo) {
        texInfo.inUse = false;
        int index = this.textureInfoList.indexOf(texInfo);
        if (index < 0) {
            Logging.e(TAG, "texture info not found!");
        } else if (index >= this.keepBufferCnt) {
            GLES20.glDeleteTextures((int)1, (int[])new int[]{texInfo.textureId}, (int)0);
            GLES20.glDeleteFramebuffers((int)1, (int[])new int[]{texInfo.frameBufferId}, (int)0);
            this.textureInfoList.remove(index);
        }
    }

    private boolean setupTextureFrameBuffer(TextureInfo texInfo, int width, int height) {
        if (width <= 0 || height <= 0) {
            Logging.w(TAG, "Invalid size: " + width + "x" + height);
            return false;
        }
        if (texInfo.specified && texInfo.width == width && texInfo.height == height) {
            return true;
        }
        GLES20.glActiveTexture((int)33984);
        GLES20.glBindTexture((int)3553, (int)texInfo.textureId);
        ATrace.beginSection("glTexImage2D");
        GLES20.glTexImage2D((int)3553, (int)0, (int)this.glPixelFormat, (int)width, (int)height, (int)0, (int)this.glPixelFormat, (int)5121, null);
        ATrace.endSection();
        GlUtil.checkNoGLES2Error("TextureBufferPool.glTexImage2D");
        GLES20.glBindTexture((int)3553, (int)0);
        GLES20.glBindFramebuffer((int)36160, (int)texInfo.frameBufferId);
        ATrace.beginSection("glFramebufferTexture2D");
        GLES20.glFramebufferTexture2D((int)36160, (int)36064, (int)3553, (int)texInfo.textureId, (int)0);
        ATrace.endSection();
        int status = GLES20.glCheckFramebufferStatus((int)36160);
        GLES20.glBindFramebuffer((int)36160, (int)0);
        if (status != 36053) {
            Logging.w(TAG, "Framebuffer not complete, status: " + status);
            return false;
        }
        texInfo.specified = true;
        texInfo.width = width;
        texInfo.height = height;
        return true;
    }

    private void drawTexture(int srcTexture, VideoFrame.TextureBuffer.Type srcType, int width, int height) {
        GLES20.glClear((int)16384);
        switch (srcType) {
            case OES: {
                this.drawer.drawOes(srcTexture, GlUtil.IDENTITY_MATRIX, width, height, 0, 0, width, height);
                break;
            }
            case RGB: {
                this.drawer.drawRgb(srcTexture, GlUtil.IDENTITY_MATRIX, width, height, 0, 0, width, height);
                break;
            }
            default: {
                throw new IllegalArgumentException("Unknown texture type.");
            }
        }
        GlUtil.checkNoGLES2Error("TextureBufferPool.drawFrameBuffer");
    }

    boolean anyTextureInUse() {
        for (TextureInfo info : this.textureInfoList) {
            if (!info.inUse) continue;
            return true;
        }
        return false;
    }

    private void release() {
        Logging.d(TAG, this.name + " release()");
        if (!this.textureInfoList.isEmpty()) {
            int count = this.textureInfoList.size();
            int[] textures = new int[count];
            int[] framebuffers = new int[count];
            for (int i = 0; i < count; ++i) {
                TextureInfo texInfo = this.textureInfoList.get(i);
                textures[i] = texInfo.textureId;
                framebuffers[i] = texInfo.frameBufferId;
            }
            Logging.d(TAG, this.name + ": delete textures " + Arrays.toString(textures));
            GLES20.glDeleteTextures((int)textures.length, (int[])textures, (int)0);
            GLES20.glDeleteFramebuffers((int)1, (int[])framebuffers, (int)0);
            this.textureInfoList.clear();
        }
        this.drawer.release();
        if (this.ownGlThread) {
            if (this.eglBase != null) {
                this.eglBase.release();
            }
            this.handler.getLooper().quit();
        }
    }

    private static class TextureInfo {
        int textureId;
        int frameBufferId;
        boolean inUse;
        boolean specified;
        int width;
        int height;

        public TextureInfo(int textureId, int frameBufferId) {
            this.textureId = textureId;
            this.frameBufferId = frameBufferId;
            this.inUse = false;
            this.specified = false;
            this.width = 0;
            this.height = 0;
        }

        public String toString() {
            return "TextureInfo{textureId=" + this.textureId + ", frameBufferId=" + this.frameBufferId + ", inUse=" + this.inUse + ", specified=" + this.specified + ", width=" + this.width + ", height=" + this.height + '}';
        }
    }
}

