/*
 * Decompiled with CFR 0.152.
 */
package io.agora.rtc2.internal;

import android.content.Context;
import android.content.res.AssetFileDescriptor;
import android.media.MediaCodec;
import android.media.MediaExtractor;
import android.media.MediaFormat;
import android.media.MediaMetadataRetriever;
import android.net.Uri;
import android.os.Build;
import android.os.Handler;
import android.os.HandlerThread;
import android.os.ParcelFileDescriptor;
import android.text.TextUtils;
import android.util.SparseArray;
import android.webkit.URLUtil;
import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import io.agora.base.internal.CalledByNative;
import io.agora.base.internal.ContextUtils;
import io.agora.base.internal.ThreadUtils;
import io.agora.mediaplayer.Constants;
import io.agora.mediaplayer.data.MediaStreamInfo;
import io.agora.rtc2.internal.Logging;
import io.agora.utils.NetUtil;
import java.io.FileDescriptor;
import java.nio.ByteBuffer;
import java.util.Arrays;
import java.util.List;
import java.util.Random;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;

public class SimpleMediaPlayerSource {
    private static final String TAG = "SMPS";
    private static final List<String> HW_EXCEPTION_MODELS = Arrays.asList("Lenovo S90-u", "CHM-CL00", "CHM-TL00H", "CHM-UL00", "E6533", "HUAWEI CRR-UL00", "HUAWEI MT7-TL00", "HONOR H30-L01", "GN3001", "SCH-I869", "SM-G7509");
    private static final int EXTRACTOR_TIMEOUT_MS = 2000;
    private static final int DEQUEUE_OR_DECODE_TIMEOUT_US = 1000;
    private static final boolean IS_DEBUG = false;
    private static final String PREFIX_ASSETS = "/assets/";
    private static final String PREFIX_DOCUMENT = "content://";
    private static final List<String> HW_UNSUPPORTED_MIMES = Arrays.asList("audio/x-ms-wma");
    private volatile boolean mIsExtractorEndOfStream;
    private MediaExtractorWrapper mExtractor;
    private MediaCodec mAudioDecoder;
    private final SparseArray<MediaStreamInfoWrapper> mMediaStreamInfoMap = new SparseArray();
    private int mBytesPerSample = -1;
    private int mAudioSampleRate = -1;
    private int mAudioChannels = -1;
    private long mDurationMs = -1L;
    private long mPlayPositionInMicroseconds = -1L;
    private final ThreadUtils.ThreadChecker mThreadChecker;
    ByteBuffer mJitterByteBuffer;
    ByteBuffer mByteBuffer;
    final int mAudioFrameSendInterval;

    @CalledByNative
    public SimpleMediaPlayerSource(int audioFrameSendInterval) {
        if (audioFrameSendInterval <= 0) {
            throw new IllegalArgumentException("interval illegal");
        }
        this.mAudioFrameSendInterval = audioFrameSendInterval;
        this.mThreadChecker = new ThreadUtils.ThreadChecker();
        this.mThreadChecker.detachThread();
        this.reset();
    }

    private void reset() {
        Logging.d(TAG, "reset()");
        if (this.mExtractor != null) {
            this.mExtractor.release();
            this.mExtractor = null;
        }
        if (this.mAudioDecoder != null) {
            try {
                this.mAudioDecoder.stop();
            }
            catch (Exception e) {
                Logging.e(TAG, "Media decoder stop failed", e);
            }
            try {
                this.mAudioDecoder.release();
            }
            catch (Exception e) {
                Logging.e(TAG, "Media decoder release failed", e);
            }
            this.mAudioDecoder = null;
        }
        if (this.mJitterByteBuffer != null) {
            this.mJitterByteBuffer.clear();
            this.mJitterByteBuffer = null;
        }
        if (this.mByteBuffer != null) {
            this.mByteBuffer.clear();
            this.mByteBuffer = null;
        }
        this.mMediaStreamInfoMap.clear();
        this.mAudioSampleRate = -1;
        this.mAudioChannels = -1;
        this.mBytesPerSample = -1;
        this.mDurationMs = -1L;
        this.mIsExtractorEndOfStream = false;
    }

    @CalledByNative
    public int open(String url, long startPos) {
        this.mThreadChecker.checkIsOnValidThread();
        Logging.d(TAG, "open() url, startPos: " + startPos);
        if (HW_EXCEPTION_MODELS.contains(Build.MODEL)) {
            Logging.w(TAG, "Not support for " + Build.MODEL);
            return -1;
        }
        if (TextUtils.isEmpty((CharSequence)url)) {
            Logging.w(TAG, "Fail to open, empty url");
            return -1;
        }
        this.reset();
        String urlStr = url;
        if (URLUtil.isNetworkUrl((String)url) && ((urlStr = NetUtil.getEncodedUrl(url)) == null || !NetUtil.testNetworkUrlAvailable(urlStr))) {
            Logging.w(TAG, "Fail to open, 404 for url");
            return -2;
        }
        try {
            return this.openImpl(urlStr, startPos);
        }
        catch (Throwable t) {
            Logging.w(TAG, "Failed to open");
            this.reset();
            return -2;
        }
    }

    private int selectAudioTrackWithPos(int audioTrackIndex, long startPos) {
        this.mThreadChecker.checkIsOnValidThread();
        int trackCount = this.mExtractor.getTrackCount();
        Logging.i(TAG, "selectAudioTrackWithPos audioTrackIndex: " + audioTrackIndex + ", startPos: " + startPos + " ms, TrackCount: " + trackCount);
        MediaStreamInfoWrapper audioTrackInfo = (MediaStreamInfoWrapper)this.mMediaStreamInfoMap.get(audioTrackIndex);
        if (audioTrackInfo == null || audioTrackInfo.getMediaStreamType() != Constants.MediaStreamType.getValue(Constants.MediaStreamType.STREAM_TYPE_AUDIO)) {
            Logging.e(TAG, "Failed to find audio track");
            this.reset();
            return -2;
        }
        for (int i = 0; i < trackCount; ++i) {
            this.mExtractor.unselectTrack(i);
        }
        this.mDurationMs = audioTrackInfo.getDuration();
        this.mAudioChannels = audioTrackInfo.getAudioChannels();
        this.mAudioSampleRate = audioTrackInfo.getAudioSampleRate();
        this.mBytesPerSample = audioTrackInfo.getAudioBytesPerSample();
        Logging.i(TAG, "DurationMs: " + this.mDurationMs + ", AudioChannels: " + this.mAudioChannels + ", AudioSampleRate: " + this.mAudioSampleRate + ", BytesPerSample: " + this.mBytesPerSample);
        if (this.mBytesPerSample == 0) {
            Logging.e(TAG, "Failed to check bytesPerSample");
            this.reset();
            return -2;
        }
        this.mExtractor.selectTrack(audioTrackIndex);
        if (startPos > 0L) {
            this.mExtractor.seekTo(startPos * 1000L, 2);
        }
        return 0;
    }

    private int openImpl(@NonNull String url, long startPos) throws Throwable {
        this.mThreadChecker.checkIsOnValidThread();
        this.mExtractor = new MediaExtractorWrapper();
        if (!this.mExtractor.setDataSource(url) || !this.checkMimeTypeSupported(this.mExtractor)) {
            Logging.w(TAG, "Failed to setDataSource");
            this.reset();
            return -2;
        }
        this.mMediaStreamInfoMap.clear();
        int audioTrackIndex = SimpleMediaPlayerSource.findAudioTrackAndFillStreamInfo(this.mMediaStreamInfoMap, this.mExtractor, url);
        MediaStreamInfoWrapper audioTrackInfo = (MediaStreamInfoWrapper)this.mMediaStreamInfoMap.get(audioTrackIndex);
        if (audioTrackInfo == null) {
            Logging.e(TAG, "Failed to find audio track");
            this.reset();
            return -2;
        }
        MediaFormat audioTrackFormat = audioTrackInfo.getFormat();
        this.mDurationMs = audioTrackInfo.getDuration();
        this.mAudioChannels = audioTrackInfo.getAudioChannels();
        this.mAudioSampleRate = audioTrackInfo.getAudioSampleRate();
        this.mBytesPerSample = audioTrackInfo.getAudioBytesPerSample();
        Logging.d(TAG, "DurationMs: " + this.mDurationMs + ", AudioChannels: " + this.mAudioChannels + ", AudioSampleRate: " + this.mAudioSampleRate + ", BytesPerSample: " + this.mBytesPerSample);
        if (this.mBytesPerSample == 0) {
            Logging.e(TAG, "Failed to check bytesPerSample");
            this.reset();
            return -2;
        }
        this.mExtractor.selectTrack(audioTrackIndex);
        String mime = audioTrackFormat.getString("mime");
        this.mAudioDecoder = MediaCodec.createDecoderByType((String)mime);
        this.mAudioDecoder.configure(audioTrackFormat, null, null, 0);
        this.mAudioDecoder.start();
        this.mJitterByteBuffer = ByteBuffer.allocate(SimpleMediaPlayerSource.getMaxOutputBufferSize(this.mAudioDecoder) * 2);
        this.mJitterByteBuffer.flip();
        int samplesPerChannel = this.mAudioSampleRate / (1000 / this.mAudioFrameSendInterval);
        int bytesPerInterval = samplesPerChannel * this.mBytesPerSample * this.mAudioChannels;
        this.mByteBuffer = ByteBuffer.allocateDirect(bytesPerInterval);
        Logging.d(TAG, " samplesPerChannel: " + samplesPerChannel + ", bytesPerInterval: " + bytesPerInterval);
        if (!this.mByteBuffer.hasArray()) {
            Logging.e(TAG, "ByteBuffer does not have backing array.");
            this.reset();
            return -2;
        }
        if (startPos > 0L) {
            this.mExtractor.seekTo(startPos * 1000L, 2);
        }
        return 0;
    }

    @CalledByNative
    @NonNull
    public SMPSIntervalData acquireIntervalData() {
        this.mThreadChecker.checkIsOnValidThread();
        SMPSIntervalData intervalData = new SMPSIntervalData();
        if (this.mByteBuffer == null) {
            Logging.d(TAG, "Failed to acquireIntervalData, invalidate frame buffer");
            intervalData.isError = true;
            return intervalData;
        }
        try {
            this.mByteBuffer.clear();
            int requestBytesSize = this.mByteBuffer.capacity();
            if (this.mJitterByteBuffer.remaining() < requestBytesSize) {
                this.mJitterByteBuffer.compact();
                intervalData.isEndOfStream = !this.acquireFrames(this.mJitterByteBuffer);
                this.mJitterByteBuffer.flip();
            }
            if (this.mJitterByteBuffer.remaining() >= requestBytesSize) {
                this.mByteBuffer.put(this.mJitterByteBuffer.array(), this.mJitterByteBuffer.arrayOffset() + this.mJitterByteBuffer.position(), requestBytesSize);
                this.mByteBuffer.flip();
                this.mJitterByteBuffer.position(this.mJitterByteBuffer.position() + requestBytesSize);
                intervalData.byteBuffer = this.mByteBuffer;
            }
        }
        catch (Throwable t) {
            Logging.e(TAG, "Failed to acquireIntervalData", t);
            intervalData.isError = true;
        }
        return intervalData;
    }

    private boolean acquireFrames(@NonNull ByteBuffer bufferToBeFilledWith) {
        MediaCodec.BufferInfo info;
        int outIndex;
        this.mThreadChecker.checkIsOnValidThread();
        if (!this.mIsExtractorEndOfStream) {
            int index = this.mAudioDecoder.dequeueInputBuffer(1000L);
            if (index < 0) {
                Logging.w(TAG, "wait for next available input buffer timeout");
            } else {
                int sampleSize = this.mExtractor.readSampleData(this.mAudioDecoder.getInputBuffers()[index], 0);
                if (sampleSize < 0) {
                    Logging.w(TAG, "no more samples are available");
                    this.mAudioDecoder.queueInputBuffer(index, 0, 0, 0L, 4);
                    this.mIsExtractorEndOfStream = true;
                } else {
                    this.mPlayPositionInMicroseconds = this.mExtractor.getSampleTime();
                    this.mAudioDecoder.queueInputBuffer(index, 0, sampleSize, this.mPlayPositionInMicroseconds, 0);
                    this.mExtractor.advance();
                }
            }
        }
        if ((outIndex = this.mAudioDecoder.dequeueOutputBuffer(info = new MediaCodec.BufferInfo(), 1000L)) < 0) {
            if (outIndex == -3) {
                Logging.d(TAG, "output buffers changed");
            } else if (outIndex == -2) {
                Logging.d(TAG, "output format changed");
            } else if (outIndex == -1) {
                Logging.d(TAG, "info try again later");
            }
            return true;
        }
        ByteBuffer buffer = this.mAudioDecoder.getOutputBuffers()[outIndex];
        buffer.position(info.offset);
        buffer.limit(info.offset + info.size);
        bufferToBeFilledWith.put(buffer);
        this.mAudioDecoder.releaseOutputBuffer(outIndex, false);
        if ((info.flags & 4) != 0) {
            Logging.w(TAG, "OutputBuffer BUFFER_FLAG_END_OF_STREAM");
            return false;
        }
        return true;
    }

    @CalledByNative
    public long getDuration() {
        this.mThreadChecker.checkIsOnValidThread();
        return this.mDurationMs;
    }

    @CalledByNative
    public long getPlayPosition() {
        this.mThreadChecker.checkIsOnValidThread();
        if (this.mExtractor != null) {
            return this.mPlayPositionInMicroseconds / 1000L;
        }
        return -1L;
    }

    @CalledByNative
    public int seek(long positionMs) {
        this.mThreadChecker.checkIsOnValidThread();
        Logging.d(TAG, "seek() " + positionMs);
        if (positionMs >= this.getDuration()) {
            positionMs = this.getDuration();
            this.mPlayPositionInMicroseconds = this.getDuration() * 1000L;
        }
        if (this.mExtractor != null) {
            this.mExtractor.seekTo(positionMs * 1000L, 1);
            if (this.mAudioDecoder != null && positionMs == 0L && this.mIsExtractorEndOfStream) {
                this.mIsExtractorEndOfStream = false;
                this.mAudioDecoder.flush();
            }
            return 0;
        }
        return -1;
    }

    @CalledByNative
    public int getStreamCount() {
        this.mThreadChecker.checkIsOnValidThread();
        return this.mMediaStreamInfoMap.size();
    }

    @CalledByNative
    public int selectAudioTrack(int index) {
        if (this.mExtractor == null) {
            Logging.e(TAG, "null extractor");
            return -1;
        }
        long pos_ms = this.mExtractor.getSampleTime() / 1000L;
        int ret = -1;
        try {
            ret = this.selectAudioTrackWithPos(index, pos_ms);
        }
        catch (Exception e) {
            Logging.e(TAG, "selectAudioTrackWithPos failed:", e);
        }
        return ret;
    }

    @CalledByNative
    public MediaStreamInfo getStreamInfo(int index) {
        this.mThreadChecker.checkIsOnValidThread();
        return (MediaStreamInfo)this.mMediaStreamInfoMap.get(index);
    }

    @CalledByNative
    public void dispose() {
        this.mThreadChecker.checkIsOnValidThread();
        this.reset();
    }

    @CalledByNative
    public int getBytesPerSample() {
        this.mThreadChecker.checkIsOnValidThread();
        return this.mBytesPerSample;
    }

    @CalledByNative
    public int getAudioSampleRate() {
        this.mThreadChecker.checkIsOnValidThread();
        return this.mAudioSampleRate;
    }

    @CalledByNative
    public int getAudioChannels() {
        this.mThreadChecker.checkIsOnValidThread();
        return this.mAudioChannels;
    }

    private boolean checkMimeTypeSupported(@NonNull MediaExtractorWrapper extractor) {
        int trackCount = extractor.getTrackCount();
        for (int i = 0; i < trackCount; ++i) {
            MediaFormat format = extractor.getTrackFormat(i);
            String mime = format.getString("mime");
            if (!mime.startsWith("audio") || !HW_UNSUPPORTED_MIMES.contains(mime)) continue;
            return false;
        }
        return true;
    }

    private static int findAudioTrackAndFillStreamInfo(@NonNull SparseArray<MediaStreamInfoWrapper> info, @NonNull MediaExtractorWrapper extractor, @NonNull String url) {
        int audioTrackIndex = -1;
        int trackCount = extractor.getTrackCount();
        Logging.d(TAG, "track count : " + trackCount);
        for (int i = 0; i < trackCount; ++i) {
            MediaStreamInfoWrapper mediaStreamInfo;
            MediaFormat format = extractor.getTrackFormat(i);
            String mime = format.getString("mime");
            Logging.d(TAG, "track : " + i + " type : " + mime);
            if (mime.startsWith("audio")) {
                audioTrackIndex = i;
                mediaStreamInfo = SimpleMediaPlayerSource.getAudioTrackInfo(format);
            } else {
                if (!mime.startsWith("video")) continue;
                mediaStreamInfo = SimpleMediaPlayerSource.getVideoTrackInfo(format);
            }
            if (mediaStreamInfo.getDuration() == 0L && !url.startsWith("http")) {
                mediaStreamInfo.setDuration(SimpleMediaPlayerSource.getLocalFileDuration(url));
            }
            info.put(i, (Object)mediaStreamInfo);
        }
        return audioTrackIndex;
    }

    @NonNull
    protected static MediaStreamInfoWrapper getAudioTrackInfo(@NonNull MediaFormat format) {
        MediaStreamInfoWrapper info = new MediaStreamInfoWrapper(format);
        info.setMediaStreamType(Constants.MediaStreamType.getValue(Constants.MediaStreamType.STREAM_TYPE_AUDIO));
        if (format.containsKey("mime")) {
            info.setCodecName(format.getString("mime"));
        }
        if (Build.VERSION.SDK_INT >= 19 && format.containsKey("language")) {
            info.setLanguage(format.getString("language"));
        }
        if (format.containsKey("sample-rate")) {
            int sampleRate = format.getInteger("sample-rate");
            if (sampleRate == 22050) {
                sampleRate = 22000;
            }
            info.setAudioSampleRate(sampleRate);
        }
        int channelCount = 0;
        if (format.containsKey("channel-count")) {
            channelCount = format.getInteger("channel-count");
            info.setAudioChannels(channelCount);
        }
        if (format.containsKey("durationUs")) {
            info.setDuration(format.getLong("durationUs") / 1000L);
        }
        int pcm_encoding = format.containsKey("pcm-encoding") ? format.getInteger("pcm-encoding") : 2;
        switch (pcm_encoding) {
            case 3: {
                info.setAudioBytesPerSample(1);
                break;
            }
            case 2: {
                info.setAudioBytesPerSample(2);
                break;
            }
            case 4: {
                info.setAudioBytesPerSample(4);
            }
        }
        return info;
    }

    @NonNull
    private static MediaStreamInfoWrapper getVideoTrackInfo(@NonNull MediaFormat format) {
        MediaStreamInfoWrapper info = new MediaStreamInfoWrapper(format);
        info.setMediaStreamType(Constants.MediaStreamType.getValue(Constants.MediaStreamType.STREAM_TYPE_VIDEO));
        if (format.containsKey("mime")) {
            info.setCodecName(format.getString("mime"));
        }
        if (Build.VERSION.SDK_INT >= 19 && format.containsKey("language")) {
            info.setLanguage(format.getString("language"));
        }
        if (format.containsKey("height")) {
            info.setVideoHeight(format.getInteger("height"));
        }
        if (format.containsKey("width")) {
            info.setVideoWidth(format.getInteger("width"));
        }
        if (format.containsKey("durationUs")) {
            info.setDuration(format.getLong("durationUs") / 1000L);
        }
        return info;
    }

    private static long getLocalFileDuration(String url) {
        long duration = 0L;
        try {
            MediaMetadataRetriever mediaMetadataRetriever = new MediaMetadataRetriever();
            mediaMetadataRetriever.setDataSource(url);
            String durationStr = mediaMetadataRetriever.extractMetadata(9);
            duration = Long.parseLong(durationStr);
        }
        catch (Exception e) {
            e.printStackTrace();
        }
        return duration;
    }

    private static int getMaxOutputBufferSize(@NonNull MediaCodec audioDecoder) {
        ByteBuffer[] bb;
        int max = 0;
        for (ByteBuffer byteBuffer : bb = audioDecoder.getOutputBuffers()) {
            int cap = byteBuffer.capacity();
            max = Math.max(cap, max);
        }
        return max;
    }

    private static class MediaExtractorWrapper {
        private final MediaExtractor mMediaExtractor;
        private volatile boolean mPrepared;
        private final Handler mHandler;
        private final String threadName = "MediaExtractor-" + new Random().nextInt();

        MediaExtractorWrapper() {
            HandlerThread handlerThread = new HandlerThread(this.threadName);
            handlerThread.start();
            this.mHandler = new Handler(handlerThread.getLooper());
            this.mMediaExtractor = new MediaExtractor();
        }

        private boolean setDataSource(final String url) throws InterruptedException {
            Logging.d(SimpleMediaPlayerSource.TAG, "setDataSource()");
            final CountDownLatch countDownLatch = new CountDownLatch(1);
            final boolean isAssetsUrl = url.startsWith(SimpleMediaPlayerSource.PREFIX_ASSETS);
            final boolean isDocument = url.startsWith(SimpleMediaPlayerSource.PREFIX_DOCUMENT);
            this.mHandler.post(new Runnable(){

                @Override
                public void run() {
                    Logging.d(SimpleMediaPlayerSource.TAG, "setDataSource in thread " + MediaExtractorWrapper.this.threadName + "  url: " + url);
                    try {
                        Context appContext = ContextUtils.getApplicationContext();
                        if (isAssetsUrl && appContext != null) {
                            AssetFileDescriptor afd = appContext.getAssets().openFd(url.substring(SimpleMediaPlayerSource.PREFIX_ASSETS.length()));
                            MediaExtractorWrapper.this.mMediaExtractor.setDataSource(afd.getFileDescriptor(), afd.getStartOffset(), afd.getLength());
                        } else if (isDocument && appContext != null) {
                            Uri uri = Uri.parse((String)url);
                            ParcelFileDescriptor parcelFileDescriptor = appContext.getContentResolver().openFileDescriptor(uri, "r");
                            FileDescriptor fileDescriptor = parcelFileDescriptor.getFileDescriptor();
                            MediaExtractorWrapper.this.mMediaExtractor.setDataSource(fileDescriptor);
                        } else {
                            MediaExtractorWrapper.this.mMediaExtractor.setDataSource(url);
                        }
                        MediaExtractorWrapper.this.mPrepared = true;
                    }
                    catch (Exception e) {
                        Logging.w(SimpleMediaPlayerSource.TAG, "setDataSource fail: " + e.toString());
                    }
                    countDownLatch.countDown();
                }
            });
            countDownLatch.await(2000L, TimeUnit.MILLISECONDS);
            Logging.d(SimpleMediaPlayerSource.TAG, "setDataSource complete");
            return this.mPrepared;
        }

        void release() {
            Logging.d(SimpleMediaPlayerSource.TAG, "release()");
            this.mHandler.post(new Runnable(){

                @Override
                public void run() {
                    try {
                        if (MediaExtractorWrapper.this.mMediaExtractor != null) {
                            MediaExtractorWrapper.this.mMediaExtractor.release();
                        }
                    }
                    catch (Exception e) {
                        Logging.e(SimpleMediaPlayerSource.TAG, "release media extractor exception.", e);
                    }
                    MediaExtractorWrapper.this.mHandler.getLooper().quit();
                    Logging.d(SimpleMediaPlayerSource.TAG, "mediaExtractor released in thread " + MediaExtractorWrapper.this.threadName);
                }
            });
        }

        int getTrackCount() {
            this.checkPrepared();
            return this.mMediaExtractor.getTrackCount();
        }

        void selectTrack(int index) {
            this.checkPrepared();
            this.mMediaExtractor.selectTrack(index);
        }

        void unselectTrack(int index) {
            this.checkPrepared();
            this.mMediaExtractor.unselectTrack(index);
        }

        void seekTo(long timeUs, int mode) {
            this.checkPrepared();
            this.mMediaExtractor.seekTo(timeUs, mode);
        }

        int readSampleData(ByteBuffer byteBuf, int offset) {
            this.checkPrepared();
            return this.mMediaExtractor.readSampleData(byteBuf, offset);
        }

        long getSampleTime() {
            this.checkPrepared();
            return this.mMediaExtractor.getSampleTime();
        }

        void advance() {
            this.checkPrepared();
            this.mMediaExtractor.advance();
        }

        MediaFormat getTrackFormat(int index) {
            this.checkPrepared();
            return this.mMediaExtractor.getTrackFormat(index);
        }

        void checkPrepared() {
            if (!this.mPrepared) {
                throw new IllegalStateException("mMediaExtractor hasn't prepared");
            }
        }
    }

    protected static class MediaStreamInfoWrapper
    extends MediaStreamInfo {
        @NonNull
        private MediaFormat format;

        public MediaStreamInfoWrapper(@NonNull MediaFormat format) {
            this.format = format;
        }

        @NonNull
        public MediaFormat getFormat() {
            return this.format;
        }
    }

    static class SMPSIntervalData {
        @Nullable
        private ByteBuffer byteBuffer = null;
        private boolean isEndOfStream = false;
        private boolean isError = false;

        SMPSIntervalData() {
        }

        @CalledByNative(value="SMPSIntervalData")
        @Nullable
        public ByteBuffer getByteBuffer() {
            return this.byteBuffer;
        }

        @CalledByNative(value="SMPSIntervalData")
        public boolean isEndOfStream() {
            return this.isEndOfStream;
        }

        @CalledByNative(value="SMPSIntervalData")
        public boolean isError() {
            return this.isError;
        }
    }

    static @interface PlayerError {
        public static final int PLAYER_ERROR_NONE = 0;
        public static final int PLAYER_ERROR_INVALID_ARGUMENTS = -1;
        public static final int PLAYER_ERROR_INTERNAL = -2;
        public static final int PLAYER_ERROR_NO_RESOURCE = -3;
        public static final int PLAYER_ERROR_INVALID_MEDIA_SOURCE = -4;
        public static final int PLAYER_ERROR_UNKNOWN_STREAM_TYPE = -5;
        public static final int PLAYER_ERROR_OBJ_NOT_INITIALIZED = -6;
        public static final int PLAYER_ERROR_CODEC_NOT_SUPPORTED = -7;
        public static final int PLAYER_ERROR_VIDEO_RENDER_FAILED = -8;
        public static final int PLAYER_ERROR_INVALID_STATE = -9;
        public static final int PLAYER_ERROR_URL_NOT_FOUND = -10;
        public static final int PLAYER_ERROR_INVALID_CONNECTION_STATE = -11;
        public static final int PLAYER_ERROR_SRC_BUFFER_UNDERFLOW = -12;
        public static final int PLAYER_ERROR_INTERRUPTED = -13;
    }
}

