/*
 * Decompiled with CFR 0.152.
 */
package io.airlift.http.server;

import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableList;
import com.google.common.hash.HashCode;
import com.google.common.hash.Hashing;
import com.google.common.io.Files;
import io.airlift.http.server.HttpServer;
import io.airlift.http.server.HttpServerConfig;
import io.airlift.log.Logger;
import io.airlift.security.cert.CertificateBuilder;
import io.airlift.security.pem.PemReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.InetAddress;
import java.net.NetworkInterface;
import java.net.SocketException;
import java.nio.charset.StandardCharsets;
import java.security.GeneralSecurityException;
import java.security.KeyPair;
import java.security.KeyPairGenerator;
import java.security.KeyStore;
import java.security.SecureRandom;
import java.security.cert.Certificate;
import java.security.cert.X509Certificate;
import java.time.LocalDate;
import java.time.temporal.ChronoUnit;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import javax.security.auth.x500.X500Principal;
import org.eclipse.jetty.util.ssl.SslContextFactory;

final class ReloadableSslContextFactoryProvider {
    private static final Logger log = Logger.get(ReloadableSslContextFactoryProvider.class);
    private final SslContextFactory.Server sslContextFactory;
    private final Optional<FileWatch> keystoreFile;
    private final String keystorePassword;
    private final String keyManagerPassword;
    private final String automaticHttpsSharedSecret;
    private final String environment;
    private final Optional<FileWatch> trustStoreFile;
    private final String trustStorePassword;

    public ReloadableSslContextFactoryProvider(HttpServerConfig config, ScheduledExecutorService scheduledExecutor, HttpServer.ClientCertificate clientCertificate, String environment) {
        Objects.requireNonNull(config, "config is null");
        Objects.requireNonNull(scheduledExecutor, "scheduledExecutor is null");
        this.keystoreFile = Optional.ofNullable(config.getKeystorePath()).map(File::new).map(FileWatch::new);
        this.keystorePassword = config.getKeystorePassword();
        this.keyManagerPassword = config.getKeyManagerPassword();
        this.automaticHttpsSharedSecret = config.getAutomaticHttpsSharedSecret();
        this.environment = Objects.requireNonNull(environment, "environment is null");
        this.trustStoreFile = Optional.ofNullable(config.getTrustStorePath()).map(File::new).map(FileWatch::new);
        this.trustStorePassword = config.getTrustStorePassword();
        this.sslContextFactory = new SslContextFactory.Server();
        this.sslContextFactory.setIncludeCipherSuites(config.getHttpsIncludedCipherSuites().toArray(new String[0]));
        this.sslContextFactory.setExcludeCipherSuites(config.getHttpsExcludedCipherSuites().toArray(new String[0]));
        this.sslContextFactory.setSecureRandomAlgorithm(config.getSecureRandomAlgorithm());
        switch (clientCertificate) {
            case NONE: {
                break;
            }
            case REQUESTED: {
                this.sslContextFactory.setWantClientAuth(true);
                break;
            }
            case REQUIRED: {
                this.sslContextFactory.setNeedClientAuth(true);
                break;
            }
            default: {
                throw new IllegalArgumentException("Unsupported client certificate value: " + clientCertificate);
            }
        }
        this.sslContextFactory.setSslSessionTimeout(Math.toIntExact(config.getSslSessionTimeout().roundTo(TimeUnit.SECONDS)));
        this.sslContextFactory.setSslSessionCacheSize(config.getSslSessionCacheSize());
        this.loadContextFactory(this.sslContextFactory);
        long refreshTime = config.getSslContextRefreshTime().toMillis();
        scheduledExecutor.scheduleWithFixedDelay(this::reload, refreshTime, refreshTime, TimeUnit.MILLISECONDS);
    }

    private void loadContextFactory(SslContextFactory.Server sslContextFactory) {
        KeyStore keyStore = ReloadableSslContextFactoryProvider.loadKeyStore(this.keystoreFile.map(FileWatch::getFile), this.keystorePassword, this.keyManagerPassword);
        String password = "";
        if (this.keyManagerPassword != null) {
            password = this.keyManagerPassword;
        } else if (this.keystorePassword != null) {
            password = this.keystorePassword;
        }
        if (this.automaticHttpsSharedSecret != null) {
            ReloadableSslContextFactoryProvider.addAutomaticKeyForCurrentNode(this.automaticHttpsSharedSecret, keyStore, this.environment, password);
        }
        sslContextFactory.setKeyStore(keyStore);
        sslContextFactory.setKeyStorePassword(password);
        if (this.trustStoreFile.isPresent()) {
            sslContextFactory.setTrustStore(ReloadableSslContextFactoryProvider.loadTrustStore(this.trustStoreFile.get().getFile(), this.trustStorePassword));
            sslContextFactory.setTrustStorePassword("");
        } else {
            sslContextFactory.setTrustStore(keyStore);
            sslContextFactory.setKeyStorePassword(password);
        }
    }

    public static void addAutomaticKeyForCurrentNode(String sharedSecret, KeyStore keyStore, String commonName, String keyManagerPassword) {
        try {
            byte[] seed = sharedSecret.getBytes(StandardCharsets.UTF_8);
            SecureRandom secureRandom = SecureRandom.getInstance("SHA1PRNG");
            secureRandom.setSeed(seed);
            KeyPairGenerator generator = KeyPairGenerator.getInstance("RSA");
            generator.initialize(2048, secureRandom);
            KeyPair keyPair = generator.generateKeyPair();
            X500Principal subject = new X500Principal("CN=" + commonName);
            LocalDate notBefore = LocalDate.now();
            LocalDate notAfter = notBefore.plus(10L, ChronoUnit.YEARS);
            X509Certificate certificateServer = CertificateBuilder.certificateBuilder().setKeyPair(keyPair).setSerialNumber(System.currentTimeMillis()).setIssuer(subject).setNotBefore(notBefore).setNotAfter(notAfter).setSubject(subject).addSanIpAddresses(ReloadableSslContextFactoryProvider.getAllLocalIpAddresses()).buildSelfSigned();
            char[] password = keyManagerPassword == null ? new char[]{} : keyManagerPassword.toCharArray();
            keyStore.setKeyEntry(commonName, keyPair.getPrivate(), password, new Certificate[]{certificateServer});
        }
        catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    private static List<InetAddress> getAllLocalIpAddresses() throws SocketException {
        ImmutableList.Builder list = ImmutableList.builder();
        for (NetworkInterface networkInterface : Collections.list(NetworkInterface.getNetworkInterfaces())) {
            for (InetAddress address : Collections.list(networkInterface.getInetAddresses())) {
                if (address.isAnyLocalAddress() || address.isLinkLocalAddress() || address.isMulticastAddress()) continue;
                list.add((Object)address);
            }
        }
        return list.build();
    }

    private static KeyStore loadKeyStore(Optional<File> keystoreFile, String keystorePassword, String keyManagerPassword) {
        KeyStore keyStore;
        if (!keystoreFile.isPresent()) {
            try {
                KeyStore keyStore2 = KeyStore.getInstance("JKS");
                keyStore2.load(null, new char[0]);
                return keyStore2;
            }
            catch (IOException | GeneralSecurityException e) {
                throw new RuntimeException(e);
            }
        }
        File file = keystoreFile.get();
        try {
            if (PemReader.isPem((File)file)) {
                Preconditions.checkArgument((keyManagerPassword == null ? 1 : 0) != 0, (Object)"key manager password is not allowed with a PEM keystore");
                return PemReader.loadKeyStore((File)file, (File)file, Optional.ofNullable(keystorePassword), (boolean)true);
            }
        }
        catch (IOException | GeneralSecurityException e) {
            throw new IllegalArgumentException("Error loading PEM key store: " + file, e);
        }
        FileInputStream in = new FileInputStream(file);
        try {
            KeyStore keyStore3 = KeyStore.getInstance("JKS");
            keyStore3.load(in, keystorePassword.toCharArray());
            keyStore = keyStore3;
        }
        catch (Throwable throwable) {
            try {
                try {
                    ((InputStream)in).close();
                }
                catch (Throwable throwable2) {
                    throwable.addSuppressed(throwable2);
                }
                throw throwable;
            }
            catch (IOException | GeneralSecurityException e) {
                throw new IllegalArgumentException("Error loading Java key store: " + file, e);
            }
        }
        ((InputStream)in).close();
        return keyStore;
    }

    private static KeyStore loadTrustStore(File truststoreFile, String truststorePassword) {
        KeyStore keyStore;
        try {
            if (PemReader.isPem((File)truststoreFile)) {
                return PemReader.loadTrustStore((File)truststoreFile);
            }
        }
        catch (IOException | GeneralSecurityException e) {
            throw new IllegalArgumentException("Error loading PEM trust store: " + truststoreFile, e);
        }
        FileInputStream in = new FileInputStream(truststoreFile);
        try {
            KeyStore keyStore2 = KeyStore.getInstance("JKS");
            keyStore2.load(in, truststorePassword == null ? null : truststorePassword.toCharArray());
            keyStore = keyStore2;
        }
        catch (Throwable throwable) {
            try {
                try {
                    ((InputStream)in).close();
                }
                catch (Throwable throwable2) {
                    throwable.addSuppressed(throwable2);
                }
                throw throwable;
            }
            catch (IOException | GeneralSecurityException e) {
                throw new IllegalArgumentException("Error loading Java trust store: " + truststoreFile, e);
            }
        }
        ((InputStream)in).close();
        return keyStore;
    }

    public SslContextFactory.Server getSslContextFactory() {
        return this.sslContextFactory;
    }

    private synchronized void reload() {
        try {
            if (this.keystoreFile.map(FileWatch::updateState).orElse(false).booleanValue() || this.trustStoreFile.map(FileWatch::updateState).orElse(false).booleanValue()) {
                this.sslContextFactory.reload(sslContextFactory -> this.loadContextFactory((SslContextFactory.Server)sslContextFactory));
            }
        }
        catch (Exception e) {
            log.warn((Throwable)e, "Unable to reload SslContext.");
        }
    }

    private static class FileWatch {
        private final File file;
        private long lastModified = -1L;
        private long length = -1L;
        private HashCode hashCode = Hashing.sha256().hashBytes(new byte[0]);

        public FileWatch(File file) {
            this.file = Objects.requireNonNull(file, "file is null");
            this.updateState();
        }

        public File getFile() {
            return this.file;
        }

        public boolean updateState() {
            try {
                long newLastModified = this.file.lastModified();
                long newLength = this.file.length();
                if (this.lastModified == newLastModified && this.length == newLength) {
                    return false;
                }
                this.lastModified = newLastModified;
                this.length = newLength;
                HashCode newHashCode = Files.asByteSource((File)this.file).hash(Hashing.sha256());
                if (Objects.equals(this.hashCode, newHashCode)) {
                    return false;
                }
                this.hashCode = newHashCode;
                return true;
            }
            catch (IOException e) {
                return true;
            }
        }

        public String toString() {
            return this.file.toString();
        }
    }
}

