/*
 * Decompiled with CFR 0.152.
 */
package io.airlift.http.server;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.MoreObjects;
import com.google.common.base.Preconditions;
import com.google.common.base.Verify;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableSet;
import io.airlift.concurrent.Threads;
import io.airlift.http.server.ClassPathResourceFilter;
import io.airlift.http.server.ConnectionStats;
import io.airlift.http.server.HttpServerBinder;
import io.airlift.http.server.HttpServerConfig;
import io.airlift.http.server.HttpServerInfo;
import io.airlift.http.server.HttpsConfig;
import io.airlift.http.server.IgnoreForwardedRequestCustomizer;
import io.airlift.http.server.JettyRequestLog;
import io.airlift.http.server.RejectForwardedRequestCustomizer;
import io.airlift.http.server.ReloadableSslContextFactoryProvider;
import io.airlift.http.server.jetty.MonitoredQueuedThreadPoolMBean;
import io.airlift.log.Logger;
import io.airlift.memory.jetty.ConcurrentRetainableBufferPool;
import io.airlift.node.NodeInfo;
import io.airlift.units.DataSize;
import jakarta.annotation.PostConstruct;
import jakarta.annotation.PreDestroy;
import jakarta.servlet.Filter;
import jakarta.servlet.Servlet;
import java.io.IOException;
import java.nio.channels.ServerSocketChannel;
import java.security.KeyStore;
import java.security.cert.Certificate;
import java.security.cert.X509Certificate;
import java.time.ZoneId;
import java.time.ZonedDateTime;
import java.time.temporal.ChronoUnit;
import java.time.temporal.Temporal;
import java.util.Collections;
import java.util.Comparator;
import java.util.EnumSet;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.OptionalLong;
import java.util.Set;
import java.util.concurrent.Executor;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import javax.management.MBeanServer;
import org.eclipse.jetty.alpn.server.ALPNServerConnectionFactory;
import org.eclipse.jetty.ee10.servlet.FilterHolder;
import org.eclipse.jetty.ee10.servlet.ServletContextHandler;
import org.eclipse.jetty.ee10.servlet.ServletHolder;
import org.eclipse.jetty.http.MimeTypes;
import org.eclipse.jetty.http.UriCompliance;
import org.eclipse.jetty.http2.server.AuthorityCustomizer;
import org.eclipse.jetty.http2.server.HTTP2CServerConnectionFactory;
import org.eclipse.jetty.http2.server.HTTP2ServerConnectionFactory;
import org.eclipse.jetty.io.ArrayByteBufferPool;
import org.eclipse.jetty.io.ByteBufferPool;
import org.eclipse.jetty.io.ConnectionStatistics;
import org.eclipse.jetty.jmx.MBeanContainer;
import org.eclipse.jetty.server.ConnectionFactory;
import org.eclipse.jetty.server.Connector;
import org.eclipse.jetty.server.ForwardedRequestCustomizer;
import org.eclipse.jetty.server.Handler;
import org.eclipse.jetty.server.HostHeaderCustomizer;
import org.eclipse.jetty.server.HttpConfiguration;
import org.eclipse.jetty.server.HttpConnectionFactory;
import org.eclipse.jetty.server.Request;
import org.eclipse.jetty.server.RequestLog;
import org.eclipse.jetty.server.SecureRequestCustomizer;
import org.eclipse.jetty.server.Server;
import org.eclipse.jetty.server.ServerConnector;
import org.eclipse.jetty.server.SslConnectionFactory;
import org.eclipse.jetty.server.handler.ErrorHandler;
import org.eclipse.jetty.server.handler.StatisticsHandler;
import org.eclipse.jetty.server.handler.gzip.GzipHandler;
import org.eclipse.jetty.util.VirtualThreads;
import org.eclipse.jetty.util.ssl.SslContextFactory;
import org.eclipse.jetty.util.thread.MonitoredQueuedThreadPool;
import org.eclipse.jetty.util.thread.ThreadPool;
import org.weakref.jmx.Managed;
import org.weakref.jmx.Nested;

public class HttpServer {
    private static final Logger log = Logger.get(HttpServer.class);
    private final Server server;
    private final MonitoredQueuedThreadPoolMBean monitoredQueuedThreadPoolMBean;
    private ConnectionStats httpConnectionStats;
    private ConnectionStats httpsConnectionStats;
    private ScheduledExecutorService scheduledExecutorService;
    private Optional<SslContextFactory.Server> sslContextFactory;

    public HttpServer(HttpServerInfo httpServerInfo, NodeInfo nodeInfo, HttpServerConfig config, Optional<HttpsConfig> maybeHttpsConfig, Servlet servlet, Set<Filter> filters, Set<HttpServerBinder.HttpResourceBinding> resources, boolean enableVirtualThreads, boolean enableLegacyUriCompliance, boolean enableCaseSensitiveHeaderCache, ClientCertificate clientCertificate, MBeanServer mbeanServer, Optional<SslContextFactory.Server> maybeSslContextFactory, Optional<ByteBufferPool> byteBufferPool) throws IOException {
        Objects.requireNonNull(httpServerInfo, "httpServerInfo is null");
        Objects.requireNonNull(nodeInfo, "nodeInfo is null");
        Objects.requireNonNull(config, "config is null");
        Objects.requireNonNull(maybeHttpsConfig, "httpsConfig is null");
        Objects.requireNonNull(servlet, "servlet is null");
        Objects.requireNonNull(maybeSslContextFactory, "maybeSslContextFactory is null");
        Objects.requireNonNull(clientCertificate, "clientCertificate is null");
        Preconditions.checkArgument((!config.isHttpsEnabled() || maybeHttpsConfig.isPresent() ? 1 : 0) != 0, (Object)"httpsConfig must be present when HTTPS is enabled");
        MonitoredQueuedThreadPool threadPool = new MonitoredQueuedThreadPool(config.getMaxThreads());
        threadPool.setMinThreads(config.getMinThreads());
        threadPool.setIdleTimeout(Math.toIntExact(config.getThreadMaxIdleTime().toMillis()));
        threadPool.setName("http-worker");
        threadPool.setDetailedDump(true);
        if (enableVirtualThreads) {
            Executor executor = VirtualThreads.getNamedVirtualThreadsExecutor((String)"http-worker#v");
            Verify.verify((executor != null ? 1 : 0) != 0, (String)"Could not create virtual threads executor", (Object[])new Object[0]);
            log.info("Virtual threads support is enabled");
            threadPool.setVirtualThreadsExecutor(executor);
        }
        int maxBufferSize = Math.toIntExact(Math.max(Math.max(HttpServer.toSafeBytes(config.getMaxRequestHeaderSize()).orElse(8192L), HttpServer.toSafeBytes(config.getMaxResponseHeaderSize()).orElse(8192L)), HttpServer.toSafeBytes(config.getOutputBufferSize()).orElse(32768L)));
        this.server = new Server((ThreadPool)threadPool, null, byteBufferPool.orElseGet(() -> this.createByteBufferPool(maxBufferSize, config)));
        this.monitoredQueuedThreadPoolMBean = new MonitoredQueuedThreadPoolMBean(threadPool);
        boolean showStackTrace = config.isShowStackTrace();
        boolean enableCompression = config.isCompressionEnabled();
        this.sslContextFactory = maybeSslContextFactory;
        if (mbeanServer != null) {
            MBeanContainer mbeanContainer = new MBeanContainer(mbeanServer);
            this.server.addBean((Object)mbeanContainer);
        }
        HttpConfiguration baseHttpConfiguration = new HttpConfiguration();
        baseHttpConfiguration.setSendServerVersion(false);
        baseHttpConfiguration.setSendXPoweredBy(false);
        baseHttpConfiguration.setNotifyRemoteAsyncErrors(true);
        baseHttpConfiguration.addCustomizer((HttpConfiguration.Customizer)(switch (config.getProcessForwarded()) {
            default -> throw new MatchException(null, null);
            case HttpServerConfig.ProcessForwardedMode.REJECT -> new RejectForwardedRequestCustomizer();
            case HttpServerConfig.ProcessForwardedMode.ACCEPT -> new ForwardedRequestCustomizer();
            case HttpServerConfig.ProcessForwardedMode.IGNORE -> new IgnoreForwardedRequestCustomizer();
        }));
        baseHttpConfiguration.addCustomizer((HttpConfiguration.Customizer)new AuthorityCustomizer());
        baseHttpConfiguration.addCustomizer((HttpConfiguration.Customizer)new HostHeaderCustomizer());
        if (config.getMaxRequestHeaderSize() != null) {
            baseHttpConfiguration.setRequestHeaderSize(Math.toIntExact(config.getMaxRequestHeaderSize().toBytes()));
        }
        if (config.getMaxResponseHeaderSize() != null) {
            baseHttpConfiguration.setResponseHeaderSize(Math.toIntExact(config.getMaxResponseHeaderSize().toBytes()));
        }
        if (config.getOutputBufferSize() != null) {
            baseHttpConfiguration.setOutputBufferSize(Math.toIntExact(config.getOutputBufferSize().toBytes()));
        }
        baseHttpConfiguration.setHeaderCacheCaseSensitive(enableCaseSensitiveHeaderCache);
        if (enableLegacyUriCompliance) {
            UriCompliance uriCompliance = UriCompliance.from(EnumSet.of(UriCompliance.Violation.AMBIGUOUS_PATH_SEPARATOR, UriCompliance.Violation.AMBIGUOUS_PATH_ENCODING, UriCompliance.Violation.SUSPICIOUS_PATH_CHARACTERS));
            baseHttpConfiguration.setUriCompliance(uriCompliance);
        }
        if (config.isHttpEnabled()) {
            HttpConfiguration httpConfiguration = new HttpConfiguration(baseHttpConfiguration);
            if (config.isHttpsEnabled()) {
                httpConfiguration.setSecureScheme("https");
                httpConfiguration.setSecurePort(httpServerInfo.getHttpsUri().getPort());
            }
            Integer acceptors = config.getHttpAcceptorThreads();
            Integer selectors = config.getHttpSelectorThreads();
            ServerConnector httpConnector = HttpServer.createServerConnector(httpServerInfo.getHttpChannel(), this.server, null, (Integer)MoreObjects.firstNonNull((Object)acceptors, (Object)-1), (Integer)MoreObjects.firstNonNull((Object)selectors, (Object)-1), this.insecureFactories(config, httpConfiguration));
            httpConnector.setName("http");
            httpConnector.setPort(httpServerInfo.getHttpUri().getPort());
            httpConnector.setIdleTimeout(config.getNetworkMaxIdleTime().toMillis());
            httpConnector.setHost(nodeInfo.getBindIp().getHostAddress());
            httpConnector.setAcceptQueueSize(config.getHttpAcceptQueueSize());
            ConnectionStatistics connectionStats = new ConnectionStatistics();
            httpConnector.addBean((Object)connectionStats);
            this.httpConnectionStats = new ConnectionStats(connectionStats);
            this.server.addConnector((Connector)httpConnector);
        }
        if (config.isHttpsEnabled()) {
            HttpConfiguration httpsConfiguration = new HttpConfiguration(baseHttpConfiguration);
            HttpServer.setSecureRequestCustomizer(httpsConfiguration);
            HttpsConfig httpsConfig = maybeHttpsConfig.orElseThrow();
            this.sslContextFactory = Optional.of(this.sslContextFactory.orElseGet(() -> this.createReloadingSslContextFactory(httpsConfig, clientCertificate, nodeInfo.getEnvironment())));
            Integer acceptors = config.getHttpsAcceptorThreads();
            Integer selectors = config.getHttpsSelectorThreads();
            ServerConnector httpsConnector = HttpServer.createServerConnector(httpServerInfo.getHttpsChannel(), this.server, null, (Integer)MoreObjects.firstNonNull((Object)acceptors, (Object)-1), (Integer)MoreObjects.firstNonNull((Object)selectors, (Object)-1), this.secureFactories(config, httpsConfiguration, this.sslContextFactory.get()));
            httpsConnector.setName("https");
            httpsConnector.setPort(httpServerInfo.getHttpsUri().getPort());
            httpsConnector.setIdleTimeout(config.getNetworkMaxIdleTime().toMillis());
            httpsConnector.setHost(nodeInfo.getBindIp().getHostAddress());
            httpsConnector.setAcceptQueueSize(config.getHttpAcceptQueueSize());
            ConnectionStatistics connectionStats = new ConnectionStatistics();
            httpsConnector.addBean((Object)connectionStats);
            this.httpsConnectionStats = new ConnectionStats(connectionStats);
            this.server.addConnector((Connector)httpsConnector);
        }
        StatisticsHandler statsHandler = new StatisticsHandler();
        statsHandler.setHandler((Handler)HttpServer.createServletContext(servlet, resources, filters, Set.of("http", "https"), showStackTrace, enableLegacyUriCompliance, enableCompression));
        if (config.isLogEnabled()) {
            this.server.setRequestLog((RequestLog)new JettyRequestLog(config.getLogPath(), config.getLogHistory(), config.getLogQueueSize(), config.getLogMaxFileSize().toBytes(), config.isCompressionEnabled(), config.isLogImmediateFlush()));
        }
        this.server.setHandler((Handler)statsHandler);
        ErrorHandler errorHandler = new ErrorHandler();
        errorHandler.setShowMessageInTitle(showStackTrace);
        errorHandler.setShowStacks(showStackTrace);
        errorHandler.setDefaultResponseMimeType(MimeTypes.Type.TEXT_PLAIN.asString());
        this.server.setErrorHandler((Request.Handler)errorHandler);
    }

    private ByteBufferPool createByteBufferPool(int maxBufferSize, HttpServerConfig config) {
        long maxHeapMemory = config.getMaxHeapMemory().map(DataSize::toBytes).orElse(0L);
        long maxOffHeapMemory = config.getMaxDirectMemory().map(DataSize::toBytes).orElse(0L);
        if (config.getHttpBufferPoolType() == HttpServerConfig.HttpBufferPoolType.FFM) {
            return new ConcurrentRetainableBufferPool(maxHeapMemory, maxOffHeapMemory);
        }
        ArrayByteBufferPool.Quadratic pool = new ArrayByteBufferPool.Quadratic(0, maxBufferSize, Integer.MAX_VALUE, maxHeapMemory, maxOffHeapMemory);
        pool.setStatisticsEnabled(true);
        return pool;
    }

    private ConnectionFactory[] insecureFactories(HttpServerConfig config, HttpConfiguration httpConfiguration) {
        HttpConnectionFactory http1 = new HttpConnectionFactory(httpConfiguration);
        HTTP2CServerConnectionFactory http2c = new HTTP2CServerConnectionFactory(httpConfiguration);
        http2c.setInitialSessionRecvWindow(Math.toIntExact(config.getHttp2InitialSessionReceiveWindowSize().toBytes()));
        http2c.setInitialStreamRecvWindow(Math.toIntExact(config.getHttp2InitialStreamReceiveWindowSize().toBytes()));
        http2c.setMaxConcurrentStreams(config.getHttp2MaxConcurrentStreams());
        http2c.setInputBufferSize(Math.toIntExact(config.getHttp2InputBufferSize().toBytes()));
        http2c.setStreamIdleTimeout(config.getHttp2StreamIdleTimeout().toMillis());
        return new ConnectionFactory[]{http1, http2c};
    }

    private ConnectionFactory[] secureFactories(HttpServerConfig config, HttpConfiguration httpsConfiguration, SslContextFactory.Server server) {
        HttpConnectionFactory http1 = new HttpConnectionFactory(httpsConfiguration);
        ALPNServerConnectionFactory alpn = new ALPNServerConnectionFactory(new String[0]);
        alpn.setDefaultProtocol(http1.getProtocol());
        SslConnectionFactory tls = new SslConnectionFactory(server, alpn.getProtocol());
        HTTP2ServerConnectionFactory http2 = new HTTP2ServerConnectionFactory(httpsConfiguration);
        http2.setInitialSessionRecvWindow(Math.toIntExact(config.getHttp2InitialSessionReceiveWindowSize().toBytes()));
        http2.setInitialStreamRecvWindow(Math.toIntExact(config.getHttp2InitialStreamReceiveWindowSize().toBytes()));
        http2.setMaxConcurrentStreams(config.getHttp2MaxConcurrentStreams());
        http2.setInputBufferSize(Math.toIntExact(config.getHttp2InputBufferSize().toBytes()));
        http2.setStreamIdleTimeout(config.getHttp2StreamIdleTimeout().toMillis());
        return new ConnectionFactory[]{tls, alpn, http2, http1};
    }

    private static void setSecureRequestCustomizer(HttpConfiguration configuration) {
        configuration.setCustomizers((List)ImmutableList.builder().add((Object)new SecureRequestCustomizer(false)).addAll((Iterable)configuration.getCustomizers()).build());
    }

    private static ServletContextHandler createServletContext(Servlet servlet, Set<HttpServerBinder.HttpResourceBinding> resources, Set<Filter> filters, Set<String> connectorNames, boolean showStackTrace, boolean enableLegacyUriCompliance, boolean enableCompression) {
        ServletContextHandler context = new ServletContextHandler(0);
        ErrorHandler handler = new ErrorHandler();
        handler.setShowStacks(showStackTrace);
        handler.setShowMessageInTitle(showStackTrace);
        context.setErrorHandler((Request.Handler)handler);
        if (enableLegacyUriCompliance) {
            context.getServletHandler().setDecodeAmbiguousURIs(true);
        }
        for (Filter filter : filters) {
            context.addFilter(new FilterHolder(filter), "/*", null);
        }
        for (HttpServerBinder.HttpResourceBinding resource : resources) {
            ClassPathResourceFilter filter = new ClassPathResourceFilter(resource.getBaseUri(), resource.getClassPathResourceBase(), resource.getWelcomeFiles());
            context.addFilter(new FilterHolder((Filter)filter), filter.getBaseUri() + "/*", null);
        }
        if (enableCompression) {
            context.insertHandler((Handler.Singleton)new GzipHandler());
        }
        ServletHolder servletHolder = new ServletHolder(servlet);
        context.addServlet(servletHolder, "/*");
        List virtualHosts = (List)connectorNames.stream().map(connectorName -> "@" + connectorName).collect(ImmutableList.toImmutableList());
        context.setVirtualHosts(virtualHosts);
        return context;
    }

    @VisibleForTesting
    Set<X509Certificate> getCertificates() {
        ImmutableSet.Builder certificates = ImmutableSet.builder();
        this.sslContextFactory.ifPresent(factory -> {
            try {
                KeyStore keystore = factory.getKeyStore();
                for (String alias : Collections.list(keystore.aliases())) {
                    Certificate certificate = keystore.getCertificate(alias);
                    if (!(certificate instanceof X509Certificate)) continue;
                    certificates.add((Object)((X509Certificate)certificate));
                }
            }
            catch (Exception e) {
                log.error((Throwable)e, "Error reading certificates");
            }
        });
        return certificates.build();
    }

    @Managed
    public Long getDaysUntilCertificateExpiration() {
        return this.getCertificates().stream().map(X509Certificate::getNotAfter).min(Comparator.naturalOrder()).map(date -> ZonedDateTime.ofInstant(date.toInstant(), ZoneId.systemDefault())).map(date -> ZonedDateTime.now().until((Temporal)date, ChronoUnit.DAYS)).orElse(null);
    }

    @Managed
    @Nested
    public ConnectionStats getHttpConnectionStats() {
        return this.httpConnectionStats;
    }

    @Managed
    @Nested
    public ConnectionStats getHttpsConnectionStats() {
        return this.httpsConnectionStats;
    }

    @Managed
    @Nested
    public MonitoredQueuedThreadPoolMBean getServerThreadPool() {
        return this.monitoredQueuedThreadPoolMBean;
    }

    @PostConstruct
    public void start() throws Exception {
        this.server.start();
        Preconditions.checkState((boolean)this.server.isStarted(), (Object)"server is not started");
    }

    @PreDestroy
    public void stop() throws Exception {
        this.server.setStopTimeout(0L);
        this.server.stop();
        if (this.scheduledExecutorService != null) {
            this.scheduledExecutorService.shutdown();
        }
    }

    @VisibleForTesting
    void join() throws InterruptedException {
        this.server.join();
    }

    private SslContextFactory.Server createReloadingSslContextFactory(HttpsConfig config, ClientCertificate clientCertificate, String environment) {
        if (this.scheduledExecutorService == null) {
            this.scheduledExecutorService = Executors.newSingleThreadScheduledExecutor(Threads.daemonThreadsNamed((String)"HttpServerScheduler"));
        }
        return new ReloadableSslContextFactoryProvider(config, this.scheduledExecutorService, clientCertificate, environment).getSslContextFactory();
    }

    private static ServerConnector createServerConnector(ServerSocketChannel channel, Server server, Executor executor, int acceptors, int selectors, ConnectionFactory ... factories) throws IOException {
        ServerConnector connector = new ServerConnector(server, executor, null, null, acceptors, selectors, factories);
        connector.open(channel);
        return connector;
    }

    private static OptionalLong toSafeBytes(DataSize dataSize) {
        if (dataSize == null) {
            return OptionalLong.empty();
        }
        return OptionalLong.of(dataSize.toBytes());
    }

    public static enum ClientCertificate {
        NONE,
        REQUESTED,
        REQUIRED;

    }
}

