/*
 * Decompiled with CFR 0.152.
 */
package org.webrtc.recorder;

import android.media.AudioRecord;
import android.media.MediaCodec;
import android.media.MediaFormat;
import android.os.Handler;
import android.os.HandlerThread;
import android.os.Looper;
import android.os.Message;
import android.os.SystemClock;
import android.util.Log;
import android.util.SparseLongArray;
import androidx.annotation.RequiresApi;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.util.LinkedList;
import java.util.Locale;
import java.util.Objects;
import java.util.concurrent.atomic.AtomicBoolean;
import org.webrtc.recorder.AudioEncodeConfig;
import org.webrtc.recorder.AudioEncoder;
import org.webrtc.recorder.BaseEncoder;
import org.webrtc.recorder.Encoder;

class MicRecorder
implements Encoder {
    private static final String TAG = "MicRecorder";
    private static final boolean VERBOSE = false;
    private final AudioEncoder mEncoder;
    private final HandlerThread mRecordThread;
    private RecordHandler mRecordHandler;
    private AudioRecord mMic;
    private int mSampleRate;
    private int mChannelConfig;
    private int mFormat = 2;
    private AtomicBoolean mForceStop = new AtomicBoolean(false);
    private BaseEncoder.Callback mCallback;
    private CallbackDelegate mCallbackDelegate;
    private int mChannelsSampleRate;
    private static final int MSG_PREPARE = 0;
    private static final int MSG_FEED_INPUT = 1;
    private static final int MSG_DRAIN_OUTPUT = 2;
    private static final int MSG_RELEASE_OUTPUT = 3;
    private static final int MSG_STOP = 4;
    private static final int MSG_RELEASE = 5;
    private static final int LAST_FRAME_ID = -1;
    private SparseLongArray mFramesUsCache = new SparseLongArray(2);

    MicRecorder(AudioEncodeConfig config) {
        this.mEncoder = new AudioEncoder(config);
        this.mSampleRate = config.sampleRate;
        this.mChannelsSampleRate = this.mSampleRate * config.channelCount;
        this.mChannelConfig = config.channelCount == 2 ? 12 : 16;
        this.mRecordThread = new HandlerThread(TAG);
    }

    @Override
    public void setCallback(Encoder.Callback callback) {
        this.mCallback = (BaseEncoder.Callback)callback;
    }

    public void setCallback(BaseEncoder.Callback callback) {
        this.mCallback = callback;
    }

    @Override
    @RequiresApi(api=19)
    public void prepare() throws IOException {
        Looper myLooper = Objects.requireNonNull(Looper.myLooper(), "Should prepare in HandlerThread");
        this.mCallbackDelegate = new CallbackDelegate(myLooper, this.mCallback);
        this.mRecordThread.start();
        this.mRecordHandler = new RecordHandler(this.mRecordThread.getLooper());
        this.mRecordHandler.sendEmptyMessage(0);
    }

    @Override
    public void stop() {
        if (this.mCallbackDelegate != null) {
            this.mCallbackDelegate.removeCallbacksAndMessages(null);
        }
        this.mForceStop.set(true);
        if (this.mRecordHandler != null) {
            this.mRecordHandler.sendEmptyMessage(4);
        }
    }

    @Override
    public void release() {
        if (this.mRecordHandler != null) {
            this.mRecordHandler.sendEmptyMessage(5);
        }
        this.mRecordThread.quitSafely();
    }

    void releaseOutputBuffer(int index) {
        Message.obtain((Handler)this.mRecordHandler, (int)3, (int)index, (int)0).sendToTarget();
    }

    ByteBuffer getOutputBuffer(int index) {
        return this.mEncoder.getOutputBuffer(index);
    }

    @RequiresApi(api=19)
    private void feedAudioEncoder(int index) {
        if (index < 0 || this.mForceStop.get()) {
            return;
        }
        AudioRecord r = Objects.requireNonNull(this.mMic, "maybe release");
        boolean eos = r.getRecordingState() == 1;
        ByteBuffer frame = this.mEncoder.getInputBuffer(index);
        int offset = frame.position();
        int limit = frame.limit();
        int read = 0;
        if (!eos && (read = r.read(frame, limit)) < 0) {
            read = 0;
        }
        long pstTs = this.calculateFrameTimestamp(read << 3);
        int flags = 1;
        if (eos) {
            flags = 4;
        }
        this.mEncoder.queueInputBuffer(index, offset, read, pstTs, flags);
    }

    private long calculateFrameTimestamp(int totalBits) {
        int samples = totalBits >> 4;
        long frameUs = this.mFramesUsCache.get(samples, -1L);
        if (frameUs == -1L) {
            frameUs = samples * 1000000 / this.mChannelsSampleRate;
            this.mFramesUsCache.put(samples, frameUs);
        }
        long timeUs = SystemClock.elapsedRealtimeNanos() / 1000L;
        long lastFrameUs = this.mFramesUsCache.get(-1, -1L);
        long currentUs = lastFrameUs == -1L ? (timeUs -= frameUs) : lastFrameUs;
        if (timeUs - currentUs >= frameUs << 1) {
            currentUs = timeUs;
        }
        this.mFramesUsCache.put(-1, currentUs + frameUs);
        return currentUs;
    }

    private static AudioRecord createAudioRecord(int sampleRateInHz, int channelConfig, int audioFormat) {
        int minBytes = AudioRecord.getMinBufferSize((int)sampleRateInHz, (int)channelConfig, (int)audioFormat);
        if (minBytes <= 0) {
            Log.e((String)TAG, (String)String.format(Locale.US, "Bad arguments: getMinBufferSize(%d, %d, %d)", sampleRateInHz, channelConfig, audioFormat));
            return null;
        }
        AudioRecord record = new AudioRecord(1, sampleRateInHz, channelConfig, audioFormat, minBytes * 2);
        if (record.getState() == 0) {
            Log.e((String)TAG, (String)String.format(Locale.US, "Bad arguments to new AudioRecord %d, %d, %d", sampleRateInHz, channelConfig, audioFormat));
            return null;
        }
        return record;
    }

    private class RecordHandler
    extends Handler {
        private LinkedList<MediaCodec.BufferInfo> mCachedInfos;
        private LinkedList<Integer> mMuxingOutputBufferIndices;
        private int mPollRate;

        RecordHandler(Looper l) {
            super(l);
            this.mCachedInfos = new LinkedList();
            this.mMuxingOutputBufferIndices = new LinkedList();
            this.mPollRate = 2048000 / MicRecorder.this.mSampleRate;
        }

        @RequiresApi(api=19)
        public void handleMessage(Message msg) {
            switch (msg.what) {
                case 0: {
                    AudioRecord r = MicRecorder.createAudioRecord(MicRecorder.this.mSampleRate, MicRecorder.this.mChannelConfig, MicRecorder.this.mFormat);
                    if (r == null) {
                        Log.e((String)MicRecorder.TAG, (String)"create audio record failure");
                        MicRecorder.this.mCallbackDelegate.onError(MicRecorder.this, new IllegalArgumentException());
                        break;
                    }
                    r.startRecording();
                    MicRecorder.this.mMic = r;
                    try {
                        MicRecorder.this.mEncoder.prepare();
                    }
                    catch (Exception e) {
                        MicRecorder.this.mCallbackDelegate.onError(MicRecorder.this, e);
                        break;
                    }
                }
                case 1: {
                    if (MicRecorder.this.mForceStop.get()) break;
                    int index = this.pollInput();
                    if (index >= 0) {
                        MicRecorder.this.feedAudioEncoder(index);
                        if (MicRecorder.this.mForceStop.get()) break;
                        this.sendEmptyMessage(2);
                        break;
                    }
                    this.sendEmptyMessageDelayed(1, this.mPollRate);
                    break;
                }
                case 2: {
                    this.offerOutput();
                    this.pollInputIfNeed();
                    break;
                }
                case 3: {
                    MicRecorder.this.mEncoder.releaseOutputBuffer(msg.arg1);
                    this.mMuxingOutputBufferIndices.poll();
                    this.pollInputIfNeed();
                    break;
                }
                case 4: {
                    if (MicRecorder.this.mMic != null) {
                        MicRecorder.this.mMic.stop();
                    }
                    MicRecorder.this.mEncoder.stop();
                    break;
                }
                case 5: {
                    if (MicRecorder.this.mMic != null) {
                        MicRecorder.this.mMic.release();
                        MicRecorder.this.mMic = null;
                    }
                    MicRecorder.this.mEncoder.release();
                }
            }
        }

        private void offerOutput() {
            while (!MicRecorder.this.mForceStop.get()) {
                int index;
                MediaCodec.BufferInfo info = this.mCachedInfos.poll();
                if (info == null) {
                    info = new MediaCodec.BufferInfo();
                }
                if ((index = MicRecorder.this.mEncoder.getEncoder().dequeueOutputBuffer(info, 1L)) == -2) {
                    MicRecorder.this.mCallbackDelegate.onOutputFormatChanged(MicRecorder.this.mEncoder, MicRecorder.this.mEncoder.getEncoder().getOutputFormat());
                }
                if (index < 0) {
                    info.set(0, 0, 0L, 0);
                    this.mCachedInfos.offer(info);
                    break;
                }
                this.mMuxingOutputBufferIndices.offer(index);
                MicRecorder.this.mCallbackDelegate.onOutputBufferAvailable(MicRecorder.this.mEncoder, index, info);
            }
        }

        private int pollInput() {
            return MicRecorder.this.mEncoder.getEncoder().dequeueInputBuffer(0L);
        }

        private void pollInputIfNeed() {
            if (this.mMuxingOutputBufferIndices.size() <= 1 && !MicRecorder.this.mForceStop.get()) {
                this.removeMessages(1);
                this.sendEmptyMessageDelayed(1, 0L);
            }
        }
    }

    private static class CallbackDelegate
    extends Handler {
        private BaseEncoder.Callback mCallback;

        CallbackDelegate(Looper l, BaseEncoder.Callback callback) {
            super(l);
            this.mCallback = callback;
        }

        void onError(Encoder encoder, Exception exception) {
            Message.obtain((Handler)this, () -> {
                if (this.mCallback != null) {
                    this.mCallback.onError(encoder, exception);
                }
            }).sendToTarget();
        }

        void onOutputFormatChanged(BaseEncoder encoder, MediaFormat format) {
            Message.obtain((Handler)this, () -> {
                if (this.mCallback != null) {
                    this.mCallback.onOutputFormatChanged(encoder, format);
                }
            }).sendToTarget();
        }

        void onOutputBufferAvailable(BaseEncoder encoder, int index, MediaCodec.BufferInfo info) {
            Message.obtain((Handler)this, () -> {
                if (this.mCallback != null) {
                    this.mCallback.onOutputBufferAvailable(encoder, index, info);
                }
            }).sendToTarget();
        }
    }
}

