/*
 * Copyright 2022 Red Hat Inc
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package io.apicurio.rest.client.handler;

import java.io.IOException;
import java.text.DateFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Date;

import com.fasterxml.jackson.core.JacksonException;
import com.fasterxml.jackson.core.JsonParseException;
import com.fasterxml.jackson.core.JsonParser;
import com.fasterxml.jackson.databind.DeserializationContext;
import com.fasterxml.jackson.databind.JsonDeserializer;
import com.fasterxml.jackson.databind.JsonNode;

/**
 * A custom date deserializer for backwards compatibility.  Handles the date formats from 2.3.1.Final and also
 * 2.4.0.Final (where we fixed the server-side date format returned by the API).
 * @author eric.wittmann@gmail.com
 */
public class RegistryDateDeserializer extends JsonDeserializer<Date> {

    /**
     * thread-safe DateFormat, @see https://www.javacodegeeks.com/2010/07/java-best-practices-dateformat-in.html
     */
    private ThreadLocal<DateFormat> dateFormat = new ThreadLocal<>() {

        @Override
        public DateFormat get() {
            return super.get();
        }

        @Override
        protected DateFormat initialValue() {
            return new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ssZ");
        }

        @Override
        public void remove() {
            super.remove();
        }

        @Override
        public void set(DateFormat value) {
            super.set(value);
        }

    };

    /**
     * @see com.fasterxml.jackson.databind.JsonDeserializer#deserialize(com.fasterxml.jackson.core.JsonParser, com.fasterxml.jackson.databind.DeserializationContext)
     */
    @Override
    public Date deserialize(JsonParser parser, DeserializationContext context) throws IOException, JacksonException {
        JsonNode node = parser.getCodec().readTree(parser);
        String date = node.textValue();
        
        // Handle zulu timezone and convert to RFC 822.
        if (date != null && date.endsWith("Z")) {
            date = date.replace("Z", "+0000");
        }

        try {
            return dateFormat.get().parse(date);
        } catch (ParseException e) {
        }
        throw new JsonParseException(parser, "Failed to parse date (not a supported format).");
    }
}
